/* $Id: config.c,v 1.1.1.1 1996/10/09 11:26:22 davidn Exp $
 * NLMaint Configuration Module
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>

#include "nlmaint.h"
#include "parse.h"
#include "osdep.h"
#include "wfile.h"
#include "mem.h"
#include "log.h"

#undef DEBUGCFG
#ifdef DEBUGCFG
static char *notify_str(unsigned typ);
#endif

#define MAXINCLUDES 8		/* Arbitrary, but sensible */

/* Command line keywords/switches
 */

static struct _verbstr argverbs[] =
{
    {"", 0},
    {"Process", 1},
    {"Test", 1},
    {"Merge", 1},
    {"Name", 1},
    {"Suppress", 1},
    {"Force", 1},
    {"Help", 1},
    {"?", 1},
    {NULL, 0}
};
/* Days of the week
 */

static struct _verbstr dow[] =
{
    {"SUNday", 3},
    {"MONday", 3},
    {"TUEsday", 3},
    {"WEDnesday", 3},
    {"THUrsday", 3},
    {"FRIday", 3},
    {"SATurday", 3},
    {NULL, 0}
};
/* Pvt node disposition
 */

static struct _verbstr pvtverbs[] =
{
    {"OK", 2},
    {"NONe", 3},
    {"HUB", 3},
    {"HOSt", 3},
    {"REGion", 3},
    {"ZONe", 3},
    {NULL, 0}
};


/* Control file keywords themselves, together with the minimum
 * number of characters that must be specified in order to match
 */

static struct _verbstr ctlverbs[] =
{
    /* ---- Directoryes ---- */
    {"MASter", 3},		/* MASter <directory> */
    {"UPDate", 3},		/* UPDate <directory> */
    {"BADfiles", 3},		/* BADfiles <directory */
    {"OUTPath", 4},		/* OUTPath <directory> */
    {"ARCList", 4},		/* ARCList <directory> */
    {"ARCDiff", 4},		/* ARCDiff <directory> */
    /* ---- Files ---- */
    {"LOGFile", 4},		/* LOGFile <explicit_filename> *** */
    {"COPyright", 3},		/* COPyright <explicit_filename> */
    {"PROlog", 3},		/* PROlog <explicit_filename> */
    {"EPIlog", 3},		/* EPIlog <explicit_filename> */
    {"COMPress", 4},		/* COMPress <explicit_filename> */
    {"DATa", 3},		/* DATa (from name) */
    {"COMments", 3},		/* COMments <explicit_filename> */
    {"FLAGs", 4},		/* FLAGs <disposition> [<explicit_filename>] */
    {"UFLAGs", 5},		/* UFLAGs <disposition> [<explicit_filename>] */
    {"MERge", 3},		/* MERge [<nodelist_name>] */
    /* ---- other types of string data ---- */
    {"MESSages", 4},		/* MESsages <directory_path_name> */
    {"OUTFile", 4},		/* OUTFile <filename> */
    {"OUTDiff", 4},		/* OUTDiff <generic_name> */
    {"BAUdrates", 3},		/* BAUdrate <valid_baudrates> */
    {"NAMe", 3},		/* NAMe <network_name> */
    {"PGPPass", 4},		/* PGP Password */
    {"ARC", 3},			/* ARC <method>[,<prefix>] */
    {"DIFFArc", 4},		/* ARCDiff <method>[,<prefix>] */
    {"", 0},
    {"", 0},
    {"LISTDesc", 5},		/* LISTDescription <description_template> */
    {"DIFFDesc", 5},		/* DIFFDescription <description_template> */
    {"BEForeproc", 3},		/* BEForeproc <command> */
    {"AFTerproc", 3},		/* AFTerproc <command> */
    {"EXECSubmit", 5},		/* EXECSubmit <command> */
    {"EXECPost", 5},		/* EXECPost <command> */
    {"PASSword", 4},		/* PASSword <string> */
    {"", 0},

    {"UPLoads", 3},		/* UPLoads <directory> */
    {"MAIlfiles", 3},		/* MAIlfiles <directory> */

    {"MAKe", 3},		/* MAKe <segment_type> [<number>
				 * [<source_file>]] */
    {"PUBlish", 3},		/* PUBlish <day_of_week> */
    {"PROcess", 3},		/* PROcess <day_of_week> */
    {"PRIvate", 3},		/* PRIvate <disposition> */
    {"MINphone", 3},		/* MINphone <minimum_parts> */
    {"THReshold", 3},		/* THReshold <arc_size> [<diff_size>] */
    {"CLEanup", 3},		/* CLEanup */
    {"NETaddress", 3},		/* NETaddress [<zone>:]<net>/<node> */
    {"SUBmit", 3},		/* SUBmit <address> [<flags>] */
    {"NOTify", 3},		/* NOTify <type> [<flags>] */
    {"FILes", 3},		/* FILes */
    /* ---- Extensions ---- */
    {"LOGLevel", 4},		/* LOGLevel <n> */
    {"REDUNdant", 5},		/* Redundant checks */
    {"INCLude", 4},		/* INCLude <filename> */
    {"LISTArc", 4},		/* LISTArc */
    {NULL, 0}
};
/* Nodelist type designators
 */

static struct _verbstr nltverbs[] =
{
    {"HUB", 3},
    {"NETwork", 3},
    {"REGion", 3},
    {"ZONe", 3},
    {"COMposite", 3},
    {"MASter", 3},		/* Synonyms for composite */
    {"GLObal", 3},
    {NULL, 0}
};

static struct _verbstr redverbs[] =
{
    {"NONe", 3},
    {"HUB", 3},
    {"NETwork", 3},
    {"REGion", 3},
    {"ZONe", 3},
    {NULL, 0}
};
/* Notify types
 */

static struct _verbstr nfverbs[] =
{
    {"Nobody", 3},
    {"None", 3},
    {"Receipt", 3},
    {"ErrorsOnly", 3},
    {"WarningsOnly", 4},
    {"All", 3},
    {"Crash", 3},
    {"Hold", 3},
    {"Immediate", 3},
    {"Direct", 3},
    {"Intl", 4},
    {"Xflags", 3},
    {"Pgpsign", 4},
    {"Pgpcrypt", 4},
    {"Nocrc", 3},
    {NULL, 0}
};

static enum _notify nfbits[] =
{
    NF_NONE, NF_NONE, NF_RECEIPT,
    NF_ERRORSONLY, NF_WARNONLY, NF_ALL,
    NF_CRASH, NF_HOLD, NF_INTL,
    NF_XFLAGS, NF_IMM, NF_DIRECT,
    NF_PGPSIGN, NF_PGPCRYPT, NF_NOCRC16
};
/* Flag handling
 */

static struct _verbstr flagaction[] =
{
    {"IGNore", 3},
    {"REMove", 3},
    {"ERROr", 3},
    {"REPortonly", 3},
    {"DEList", 3},
    {"OMIt", 3},
    {NULL, 0}
};
/* controlverb() - short cut to control keyword evaluation
 */

int
controlverb(char const * kwd)
{
    return verbno(ctlverbs, kwd);
}


/* cmdlineverb() - short cut to command line keyword evaluation
 */

int
cmdlineverb(char const * kwd)
{
    return verbno(argverbs, kwd);
}


/* fix_cfg_paths() - resolve paths, tidy up paths and make
 * consistent and insert reasonable default for some values
 */

static void
fixcfg_paths(NLCONFIG * nlc, char const * p, char const * nltdir)
{
    int b;
    ArrayIter I;
    char const *q;
    sofs_t tmp, *so;
    strbuf *sb = &nlc->strings;
    static struct {
	int index;
	int warn;
	char const *name;
    }   defaults[] =
    {
	{
	    V_UPDATE, 1, "Update"
	},
	{
	    V_BADFILES, 1, "Badfiles"
	},
	{
	    V_OUTPATH, 1, ""
	},
	{
	    V_COPYRIGHT, 0, "cpyright.txt"
	},
	{
	    V_PROLOG, 0, "prolog.txt"
	},
	{
	    V_EPILOG, 0, "epilog.txt"
	},
	{
	    -1, 0, NULL
	}
    };

    /*
     * Add some sensible defaults
     */
    for (b = 0; defaults[b].index != -1; b++) {
	enum _verbs idx = (enum _verbs) defaults[b].index;
	if (nlc->str[idx] == 0) {
	    q = resolve_path(NULL, defaults[b].name, p);
	    if (defaults[b].warn == 1) {
		if (array_items(&nlc->Files) == 0)
		    continue;
		logit(LOG_PROGRESS, "No %s path specified, using \"%s\"", ctlverbs[idx].verb, q);
	    }
	    nlc->str[idx] = sb_alloc(sb, q);
	}
    }

    /*
     * Handle as special case
     */

    if (nlc->str[V_COMPRESS] == 0) {
	if ((q = nltdir) == NULL)
	    q = p;
	q = resolve_path(NULL, "compress.cfg", q);
	nlc->str[V_COMPRESS] = sb_alloc(sb, q);
    }
    /*
     * Fix paths which need to be master directory relative
     */
    for (b = V_MASTER + 1; b <= V_LASTPATH; b++) {
	enum _verbs i = (enum _verbs) b;	/* For debugging */
	if ((tmp = nlc->str[i]) != 0) {
	    /*
	     * Resolve path relative to be to master directory
	     */
	    nlc->str[i] = sb_alloc(sb, resolve_path(NULL, sb_string(sb, tmp), p));
	    sb_free(sb, tmp);
	}
    }
    arrayIter_init(&I, &nlc->Inbound);
    while ((so = arrayIter_get(&I, NEXT)) != NULL) {
	tmp = *so;
	*so = sb_alloc(sb, resolve_path(NULL, sb_string(sb, tmp), p));
	sb_free(sb, tmp);
    }
}


/* check_paths() - check that paths exist, and create if they don't
 */

#ifdef DOSISH			/* Case insensitive paths */
#define PATHCMP(a,b) (stricmp((a),(b))==0)
#else
#define PATHCMP(a,b) (strcmp((a),(b))==0)
#endif

static void
check_paths(NLCONFIG * nlc)
{
    int b;
    sofs_t *so;
    ArrayIter I;
    char const *q;
    char const *master = nlstr(nlc, V_MASTER);

    static char const nodir[] = "%s directory \"%s\" is invalid";
    for (b = V_MASTER; b <= V_LASTDIR; b++) {
	enum _verbs i = (enum _verbs) b;	/* For debugging */
	char const *s = nlstring(nlc, i);
	q = (i <= V_LASTDIR) ? s : get_path(NULL, (char *) s);
	if (makesubdir(q) != 0)
	    logit(LOG_WARN, nodir, ctlverbs[i].verb, q);
	else if (i > V_MASTER && i <= V_LASTDIR && PATHCMP(master, q))
	    fatal(EL_CTLERROR, "Master and %s both specify \"%s\"", ctlverbs[i].verb, q);
    }

    arrayIter_init(&I, &nlc->Inbound);
    while ((so = arrayIter_get(&I, NEXT)) != NULL) {
	q = nlstr(nlc, *so);
	if (makesubdir(q) != 0)
	    logit(LOG_WARN, nodir, "MAIlfiles/UPLoads", q);
	else if (PATHCMP(master, q))
	    fatal(EL_CTLERROR, "Master and MAIlfiles/UPLoads both specify \"%s\"", q);
    }

#if 0				/* Disabled fallback, so user can disable
				 * archiving by omitting */
    if (nlsnull(nlc, V_ARCLIST))
	nlc->str[V_ARCLIST] = nlc->str[V_OUTPATH];
    if (nlsnull(nlc, V_ARCDIFF))
	nlc->str[V_ARCDIFF] = nlc->str[V_ARCLIST];
#endif
}


/* include_file() - Primary configuration file parser loop
 * This is called recursively for each level of include file.
 * Current state is maintained outside of the function, so
 * that it may be called for the 'data' section where the
 * local nodelist data is not maintained in the control file.
 */

enum {
    state_default,
    state_data,
    state_files
};

static void
include_file(NLCONFIG * nlc, char const * cfgfile, int *state, enum _nltype * intype)
{
    int len, lineno;
    char const *line;
    strbuf *sb = &nlc->strings;
    WFILE wfp;
    char tmp[MAXLINE];
    char const *filename = resolve_path(tmp, cfgfile, getenv(ENVSTRING));

    static int nested_includes = 0;
    wfile_init(&wfp, filename);
    if ((lineno = wfile_open(&wfp)) == 0) {
	char *p = path_ext(tmp);
	if (!*p) {
	    /*
	     * Try adding a .CTL extension
	     */
	    strcpy(p, ".ctl");
	    wfile_newfile(&wfp, tmp);
	    lineno = wfile_open(&wfp);
	}
	if (lineno == 0)
	    fatal(EL_CTLERROR, "Can't open control file %s: %s", filename, strerror(errno));
    }
    filename = wfp.name;
    lineno = 0;
    while ((line = wfile_ptr(&wfp, &len)) != NULL) {
	/*
	 * Next line number
	 */
	++lineno;
	if (len >= MAXLINE)
	    fatal(EL_CTLERROR, "%s(%d): line too long", filename, lineno);
	else {
	    static char const prmreq[] = "%s(%d): %s requires a parameter";

	    char *g = strncpy(tmp, line, len) + len;
	    char const *kwd;
	    char const *p;
	    for (*g-- = '\0'; g >= tmp && isspace(*g); g--)
		*g = '\0';	/* Shave off trailing spaces */

	    /*
	     * Handle the line according to the current state
	     */
	    if (*state == state_default) {	/* Normal context */
		if ((kwd = parseword(tmp, 0)) != NULL) {
		    enum _verbs vbn = (enum _verbs) controlverb(kwd);
		    int l = 0;
		    switch (vbn) {
		    case V_MAILFILES:	/* MAIlfiles <directory> */
		    case V_UPLOADS:	/* UPLoads <directory> */
			/*
			 * These are synonyms
			 */
			if ((p = parseword(NULL, 0)) == NULL)
			    fatal(EL_CTLERROR, prmreq, filename, lineno, ctlverbs[vbn].verb);
			else {
			    sofs_t *so = array_ptr(&nlc->Inbound, array_add(&nlc->Inbound, NOELEMENT, 1));
			    char const *q = sb_string(sb, nlc->str[V_MASTER]);
			    *so = (*q) ? sb_alloc(sb, resolve_path(NULL, p, q)) : sb_alloc(sb, p);
			}
			break;

		    case V_FLAG:	/* FLAGs <disposition>
					 * [<explicit_filename>] */
		    case V_UFLAG:	/* UFLAGs <disposition>
					 * [<explicit_filename>] */
			if ((p = parseword(NULL, 0)) == NULL)
			    fatal(EL_CTLERROR, prmreq, filename, lineno, ctlverbs[vbn].verb);
			else {
			    unsigned char *fptr = (vbn == V_FLAG) ? &nlc->flag_handling : &nlc->uflag_handling;
			    /*
			     * Zero the default flags setting
			     */
			    *fptr = 0;
			    while (p != NULL) {
				if (strchr(p, '.') != NULL)	/* Assume it is a
								 * filename */
				    goto procstr;
				if ((l = verbno(flagaction, p)) == -1)
				    fatal(EL_CTLERROR, "%s(%d): invalid FLAG disposition \"%s\"", filename, lineno, p);
				*fptr |= (unsigned char) ((0x01 << l) >> 1);
				p = parseword(NULL, 0);
			    }
			}
			break;

		    case V_LOGFILE:	/* LOGFile <explicit_filename> */
			if ((p = parseword(NULL, 0)) == NULL)
			    fatal(EL_CTLERROR, prmreq, filename, lineno, ctlverbs[vbn].verb);
			if (stricmp(p, "FD") == 0)	/* Frontdoor format */
			    nlc->logformat = 1;
			else if (stricmp(p, "BT") == 0)	/* BinkleyTerm format */
			    nlc->logformat = 0;
			else
			    goto procstr;
			setlogformat(nlc->logformat);
			/* Fallthru */
		    case V_MASTER:	/* MASter <directory> */
		    case V_BADFILES:	/* BADfiles <directory */
		    case V_UPDATE:	/* UPDate <directory> */
		    case V_OUTPATH:	/* OUTPath <directory> */
		    case V_COPYRIGHT:	/* COPyright <explicit_filename> */
		    case V_PROLOG:	/* PROlog <explicit_filename> */
		    case V_EPILOG:	/* EPIlog <explicit_filename> */
		    case V_COMPRESS:	/* COMPress <explicit_filename */
		    case V_MERGE:	/* MERge [<nodelist_name>] */
			if ((p = parseword(NULL, 0)) == NULL)
			    fatal(EL_CTLERROR, prmreq, filename, lineno, ctlverbs[vbn].verb);
			else {
		    procstr:
			    if (nlc->str[vbn])
				sb_free(sb, nlc->str[vbn]);
			    if (vbn == V_MASTER) {
				if ((nlc->str[vbn] = sb_alloc(sb, resolve_path(NULL, p, NULL))) == 0)
				    nlc->str[vbn] = sb_alloc(sb, resolve_path(NULL, ".", NULL));
			    } else
				nlc->str[vbn] = sb_alloc(sb, p);
			}
			break;

		    case V_ARCLIST:	/* ARCList <directory> */
		    case V_ARCDIFF:	/* ARCDiff <directory> */
			if ((p = parseword(NULL, 0)) == NULL)
			    fatal(EL_CTLERROR, prmreq, filename, lineno, ctlverbs[vbn].verb);
			else {
			    if (nlc->str[vbn])
				sb_free(sb, nlc->str[vbn]);
			    nlc->str[vbn] = sb_alloc(sb, p);
			}
			break;

		    case V_COMMENTS:	/* COMments <explicit_filename> */
		    case V_OUTFILE:	/* OUTFile <filename> */
		    case V_OUTDIFF:	/* OUTDiff <generic_name> */
		    case V_NAME:	/* NAMe <network_name> */
		    case V_PGPPASS:	/* PGP <password> */
		    case V_PASSWORD:	/* PASSword <password> */
			if ((p = parseword(NULL, 0)) == NULL)
			    fatal(EL_CTLERROR, prmreq, filename, lineno, ctlverbs[vbn].verb);
			else if (nlc->str[vbn] != 0)
			    fatal(EL_CTLERROR, "%s(%d): %s keyword specified twice", filename, lineno, ctlverbs[vbn].verb);
			nlc->str[vbn] = sb_alloc(sb, p);
			break;

		    case V_BEFORE:	/* BEForeproc <command> */
		    case V_AFTER:	/* AFTerproc <command> */
		    case V_EXECSUBMIT:	/* EXECSubmit <command> */
		    case V_EXECPOST:	/* EXECPost <command> */
		    case V_MESSAGES:	/* MESsages <directory_path_name> */
			if ((p = parseline(NULL)) != NULL)
			    nlc->str[vbn] = sb_alloc(sb, p);
			break;

		    case V_LISTARC:
			vbn = V_ARC;	/* Synonym */
		    case V_ARC:/* ARC <method>,<letter>  */
		    case V_DIFFARC:	/* DIFFArc <method>,<letter> */
			if ((p = parseline(NULL)) == NULL)
			    fatal(EL_CTLERROR, prmreq, filename, lineno, ctlverbs[vbn].verb);
			{
			    /*
			     * Extract a prefix
			     */
			    char *q;
			    if (atoi(p) == 5)	/* Backwards compatibility */
				p = "ARC";
			    if ((q = strchr(p, ',')) != NULL)
				*q++ = '\0';
			    else
				q = (char *) p;
			    nlc->str[vbn - V_ARC + V_NLPFX] = sb_salloc(&nlc->strings, q, 1);
			}
			nlc->str[vbn] = sb_alloc(sb, p);
			break;

		    case V_LISTDESC:	/* LISTDescription
					 * <description_template> */
		    case V_DIFFDESC:	/* DIFFDescription
					 * <description_template> */
			if ((p = parseline(NULL)) == NULL)
			    fatal(EL_CTLERROR, prmreq, filename, lineno, ctlverbs[vbn].verb);
			else {
			    if (nlc->str[vbn])
				sb_free(sb, nlc->str[vbn]);
			    nlc->str[vbn] = sb_alloc(sb, p);
			}
			break;

		    case V_MAKE:	/* MAKe <segment_type> [<number>
					 * [<source_file>]] */
			{
			    l = verbno(nltverbs, p = parseword(NULL, 0));
			    if (l == -1)
				fatal(EL_CTLERROR, "%s(%d): unknown MAKe type \"%s\"", filename, lineno, p ? p : "NULL");
			    nlc->type = (enum _nltype) (l > NL_COMPOSITE ? NL_COMPOSITE : l);
			    if ((p = parseword(NULL, 0)) == NULL) {
				if (nlc->type == NL_COMPOSITE)	/* Ok */
				    nlc->number = (node_t) - 1;
				else
				    fatal(EL_CTLERROR, "%s(%d): 'number' is required for MAKe %s", filename, lineno, nltverbs[l].verb);
			    } else {
				if (nlc->type == NL_COMPOSITE)	/* Err */
				    fatal(EL_CTLERROR, "%s(%d): 'number' is not required for MAKe %s", filename, lineno, nltverbs[l].verb);
				else if ((nlc->number = (node_t) atoi(p)) < 1 || nlc->number > 32767)
				    fatal(EL_CTLERROR, "%s(%d): \"%s\" in MAKe is out of range", filename, lineno, p);
				if ((p = parseword(NULL, 0)) != NULL)	/* Node data file */
				    nlc->str[V_DATA] = (nlc->str[V_MASTER])
					    ? sb_alloc(sb, resolve_path(NULL, p, sb_string(sb, nlc->str[V_MASTER])))
					    : sb_alloc(sb, p);
			    }
			}
			break;

		    case V_PUBLISH:	/* PUBlish <day_of_week> */
		    case V_PROCESS:	/* PROcess <day_of_week> */
			l = verbno(dow, p = parseword(NULL, 0));
			if (l == -1)
			    fatal(EL_CTLERROR, "%s(%d): Invalid day of week \"%s\"", filename, lineno, p);
			if (vbn == V_PUBLISH)
			    nlc->publish = (enum _dow) l;
			else
			    nlc->process = (enum _dow) l;
			break;

		    case V_NETADDRESS:	/* NETaddress [<zone>:]<net>/<node> */
			if ((p = parseword(NULL, 0)) == NULL)
			    fatal(EL_CTLERROR, "%s(%d): %s requires an address", filename, lineno, ctlverbs[vbn]);
			else if (!fidoaddr(p, &nlc->myaddr, NULL))
			    fatal(EL_CTLERROR, "%s(%d): address %s is invalid", filename, lineno, p);
			break;

		    case V_PRIVATE:	/* PRIvate <disposition> */
			if ((p = parseword(NULL, 0)) == NULL)
			    fatal(EL_CTLERROR, "%s(%d): %s requires dispisition", filename, lineno, ctlverbs[vbn]);
			else if ((l = verbno(pvtverbs, p)) == -1)
			    fatal(EL_CTLERROR, "%s(%d): invalid PVT disposition \"%s\"", filename, lineno, p);
			else if ((l - 2) > (int) nlc->type)
			    fatal(EL_CTLERROR, "%s(%d): pvt disposition \"%s\" invalid for MAKe %s", filename, lineno, pvtverbs[l].verb, nltverbs[nlc->type].verb);
			nlc->pvtdisp = (enum _pvt) l;
			break;

		    case V_MINPHONE:	/* MINphone <minimum_parts> */
			if ((p = parseword(NULL, 0)) == NULL)
			    fatal(EL_CTLERROR, "%s(%d): %s number of parts required", filename, lineno, ctlverbs[vbn]);
			if ((nlc->phone_parts_min = atol(p)) < 1 || nlc->phone_parts_min > 9)
			    fatal(EL_CTLERROR, "%s(%d): %s arg \"%s\" not between 1 and 9", filename, lineno, ctlverbs[vbn], p);
			if ((p = parseword(NULL, 0)) != NULL) {
			    if ((nlc->phone_parts_max = atol(p)) < 1 || nlc->phone_parts_max > 9)
				fatal(EL_CTLERROR, "%s(%d): %s 'max' arg \"%s\" not between 1 and 9", filename, lineno, ctlverbs[vbn], p);
			}
			break;

		    case V_BAUDRATE:	/* BAUdrate <valid_baudrates> */
			if ((p = parseword(NULL, 0)) == NULL)
			    fatal(EL_CTLERROR, "%s(%d): %s requires a list of baudrates", filename, lineno, ctlverbs[vbn]);
			else {
			    char tmpbuf[128];
			    char *q = tmpbuf;
			    while (p) {
				long lval = atol(p);
				if (lval <= 0)
				    fatal(EL_CTLERROR, "%s(%d): invalid baud rate \"%s\"", filename, lineno, p);
				q += sprintf(q, "|%ld", lval);
				p = parseword(NULL, 0);
				if (q > (tmpbuf + sizeof tmpbuf - 16))
				    fatal(EL_CTLERROR, "%s(%d): too many baud rates specified", filename, lineno);
			    }
			    *q++ = '|';
			    *q = '\0';
			    nlc->str[vbn] = sb_alloc(sb, tmpbuf);
			}
			break;

		    case V_THRESHOLD:	/* THReshold <arc_size> [<diff_size>] */
			logit(LOG_WARN, "%s(%d): %s keyword is obsolete", filename, lineno, ctlverbs[vbn]);
			parseline(NULL);	/* Junk rest of line */
			break;

		    case V_CLEANUP:	/* CLEanup */
			nlc->cleanup = 1;
			break;

		    case V_SUBMIT:	/* SUBmit <address> [<flags>] */
			if ((p = parseword(NULL, 0)) == NULL)
			    fatal(EL_CTLERROR, "%s(%d): %s requires an address and optional flags", filename, lineno, ctlverbs[vbn]);
			else if (!fidoaddr(p, &nlc->submitaddr, NULL))
			    fatal(EL_CTLERROR, "%s(%d): address %s is invalid", filename, lineno, p);
			nlc->submit = NF_NONE;	/* To indicate that we actually
						 * HAVE a submit */
			while ((p = parseword(NULL, 0)) != NULL) {
			    if ((l = verbno(nfverbs, p)) < 4)
				fatal(EL_CTLERROR, "%s(%d): Invalid submit flag \"%s\"", filename, lineno, p);
			    nlc->submit |= nfbits[l];
			}
			break;

		    case V_NOTIFY:	/* NOTify <type> [<flags>] */
			while ((p = parseword(NULL, 0)) != NULL) {
			    if ((l = verbno(nfverbs, p)) == -1)
				fatal(EL_CTLERROR, "%s(%d): Invalid notify type/flag \"%s\"", filename, lineno, p);
			    nlc->notify |= nfbits[l];
			}
			break;

		    case V_LOGLEVEL:	/* LOGLevel <n> */
			if ((p = parseword(NULL, 0)) == NULL)
			    fatal(EL_CTLERROR, "%s(%d): %s requires a number", filename, lineno, ctlverbs[vbn]);
			nlc->loglevel = atoi(p);
			break;

		    case V_DATA:	/* DATa [<data_file>] */
			if ((p = parseword(NULL, 0)) != NULL)
			    nlc->str[V_DATA] = sb_alloc(sb, resolve_path(NULL, p, sb_string(sb, nlc->str[V_MASTER])));
			else
			    *state = state_data;
			break;

		    case V_FILES:	/* FILes */
			*state = state_files;
			break;

		    case V_INCLUDE:	/* Include another file */
			if ((p = parseword(NULL, 0)) == NULL)
			    fatal(EL_CTLERROR, "%s(%d): %s requires a filename to include", filename, lineno, ctlverbs[vbn]);
			if (++nested_includes > MAXINCLUDES)
			    fatal(EL_CTLERROR, "Maximum number of include files exceeded", filename, strerror(errno));
			include_file(nlc, p, state, intype);
			--nested_includes;
			break;

		    case V_REDUNDANT:	/* Set level of redundancies to check */
			if ((p = parseword(NULL, 0)) == NULL)
			    fatal(EL_CTLERROR, "%s(%d): %s requires an argument", filename, lineno, ctlverbs[vbn]);
			if ((l = verbno(redverbs, p)) == -1)
			    fatal(EL_CTLERROR, "%s(%d): Invalid redundance check type \"%s\"", filename, lineno, p);
			nlc->phone_handling = l;
			break;

		    default:
			fatal(EL_CTLERROR, "%s(%d): unknown control verb \"%s\"", filename, lineno, kwd);
			parseline(NULL);
			break;
		    }

		    if ((p = parseword(NULL, 0)) != NULL)
			logit(LOG_WARN, "%s(%d): junk on this line ignored", filename, lineno);

		}
	    } else if (*state == state_data) {
		if (nlc->type == NL_NONE)	/* No can do! */
		    fatal(EL_CTLERROR, "%s(%d): MAKe type is not specified, can't process data", filename, lineno);
		if (nlc->str[V_DATA])
		    fatal(EL_CTLERROR, "%s(%d): Master input file and DATA both specified", filename, lineno);

		for (p = tmp; *p && isspace(*p); ++p);
		if (toupper(*p) == 'F')	/* must be files keyword */
		    *state = state_files;
		else if (*p && *p != ';') {	/* Skip comments & blank lines */
		    if (valid_nl_line(p, nlc->type, intype)) {	/* Add it */
			sofs_t *op = array_ptr(&nlc->Data, array_add(&nlc->Data, NOELEMENT, 1));
			*op = sb_alloc(sb, p);
		    } else
			logit(LOG_WARN, "%s(%d): invalid line in data segment", filename, lineno);
		}
	    } else {		/* *state == state_files */
		if ((kwd = parseword(tmp, 0)) != NULL) {
		    node_t n;
		    int l = verbno(nltverbs, kwd);
		    if (l == -1 || l >= nlc->type)
			fatal(EL_CTLERROR, "%s(%d): file type \"%s\" is invalid", filename, lineno, kwd);
		    if ((p = parseword(NULL, 0)) == NULL)
			fatal(EL_CTLERROR, "%s(%d): %s file requires a number", filename, lineno, kwd);
		    if ((n = (node_t) atol(p)) <= 0 || n > 32767)
			fatal(EL_CTLERROR, "%s(%d): number for %s file is invalid", filename, lineno, kwd);
		    if ((p = parseword(NULL, 0)) == NULL)
			fatal(EL_CTLERROR, "%s(%d): %s file requires a filename", filename, lineno, kwd);
		    else {
			unsigned idx = array_add(&nlc->Files, NOELEMENT, 1);	/* Add one element */
			subfile *sf = array_ptr(&nlc->Files, idx);	/* Get ptr to element */
			memset(sf, 0, sizeof(subfile));
			sf->type = (enum _nltype) l;	/* Type of segment */
			sf->number = n;	/* Segment number */
			sf->name = sb_alloc(sb, p);	/* Save filename */
			/* Now, we need to find an address to send responses to */
			switch (sf->type) {
			case NL_ZONE:	/* x:x/0 */
			    sf->addr.zone = sf->number;
			    sf->addr.net = sf->number;
			    break;
			case NL_REGION:	/* z:x/0 */
			case NL_NET:
			    sf->addr.zone = nlc->myaddr.zone;
			    sf->addr.net = sf->number;
			    break;
			case NL_HUB:	/* z:n/x */
			    sf->addr.zone = nlc->myaddr.zone;
			    sf->addr.net = nlc->myaddr.net;
			    sf->addr.node = sf->number;
			    break;
			case NL_COMPOSITE:	/* Should not do this */
			case NL_NONE:
			    fatal(EL_CTLERROR, "%s(%d): file type \"%s\" is invalid", filename, lineno, kwd);
			}
			/*
			 * On the other hand, we might be given the right
			 * address
			 */
			sf->notify = 0;
			if ((p = parseword(NULL, 0)) != NULL) {
			    ZNNP addr;
			    if (fidoaddr(p, &addr, &nlc->myaddr))
				sf->addr = addr;
			    else {
				l = verbno(nfverbs, p);
				if (l == -1)
				    fatal(EL_CTLERROR, "%s(%d): invalid address \"%s\"", filename, lineno, p);
				sf->notify |= nfbits[l];
			    }
			    while ((p = parseword(NULL, 0)) != NULL) {
				if (strnicmp(p, "pwd=", 4) == 0)
				    sf->password = sb_alloc(&nlc->strings, p + 4);
				else {
				    l = verbno(nfverbs, p);
				    if (l == -1)
					fatal(EL_CTLERROR, "%s(%d): invalid notify type \"%s\"", filename, lineno, p);
				    sf->notify |= nfbits[l];
				}
			    }
			}
		    }
		}
	    }
	}
	/*
	 * Advance to next line
	 */
	wfile_advance(&wfp, len);
    }
    wfile_close(&wfp);
}
/* configure() - start the initial configuration functions,
 * after setting up the config structure and initialising
 * the default set of values.
 */

void
configure(NLCONFIG ** nlcptr, char const * cfgfile)
{
    static char const fn[] = "configure";
    int state = state_default;
    enum _nltype intype = NL_NONE;
    char tmp[_MAX_PATH];
    /*
     * Allocate and initialise config data
     */
    NLCONFIG *nlc = *nlcptr = zmalloc(fn, sizeof(NLCONFIG));
    memset(nlc, 0, sizeof(NLCONFIG));

    sb_init(&nlc->strings, 1024);	/* Strings buffer */
    array_init(&nlc->Files, sizeof(subfile), 0);	/* Submissions */
    array_init(&nlc->Inbound, sizeof(sofs_t), 0);	/* Incoming dirs */
    array_init(&nlc->Data, sizeof(sofs_t), 0);	/* Data lines */

    nlc->publish = V_FRIDAY;	/* Friday is default publication date */
    nlc->process = V_NEVER;	/* Don't auto-process unless day given */
    nlc->type = NL_NONE;	/* Default to none */
    nlc->mode = MODE_TEST;	/* Test mode */
    nlc->flag_handling = (FLAG_REMOVE | FLAG_ERROR);	/* Be strict with these */
    nlc->uflag_handling = FLAG_ERROR;	/* Less strict, flag only */
    nlc->phone_handling = 1;	/* Warn about non-redundant hubs/coords */

    /* First level include */
    include_file(nlc, cfgfile, &state, &intype);

    if (nlc->type == NL_NONE)	/* No can do! */
	fatal(EL_CTLERROR, "No valid MAKe specified - nothing to make");

    if (nlc->str[V_MASTER] == 0) {	/* Use current directory */
	logit(LOG_WARN, "Using current directory as master");
	nlc->str[V_MASTER] = sb_alloc(&nlc->strings, resolve_path(NULL, ".", NULL));
    }
    resolve_path(tmp, cfgfile, getenv(ENVSTRING));
    strcpy(tmp, get_path(NULL, tmp));
    fixcfg_paths(nlc, nlstring(nlc, V_MASTER), tmp);

    /*
     * Read in additional data if a separate filename was given
     */
    if (nlc->str[V_DATA]) {
	state = state_data;	/* Force into data mode */
	include_file(nlc, nlstring(nlc, V_DATA), &state, &intype);
    }
    /*
     * Check paths (existance, collisions)
     */
    check_paths(nlc);

    /*
     * Load the archiver's file
     */
    arcinfo_init(&nlc->Arcs, nlstring(nlc, V_COMPRESS));

    if (nlc->myaddr.net == 0)
	fatal(EL_CTLERROR, "No local \"From\" address for notify/submittals");
    if (nlc->str[V_MESSAGES] == 0)
	fatal(EL_CTLERROR, "No msgbase area specified for notify/submittals");

#ifdef DEBUGCFG
    dump_config(nlc);
#endif
}
/* notify_str() - return the descriptive "notify" type flags set
 */

char *
notify_str(char *dest, unsigned typ)
{
    char *p = dest;
    int i;
    for (i = 0; i < (int) (sizeof(nfbits) / sizeof(nfbits[1])); i++)
	if (typ & nfbits[i])
	    p += sprintf(p, "%s ", nfverbs[i].verb);
    *p = '\0';
    return dest;
}
#ifdef DEBUGCFG

/* dump_config() - dump config data - for debugging
 */

static void
dump_config(NLCONFIG * nlc)
{
    sofs_t i;
    strbuf *sb = &nlc->strings;
    subfile *sf;
    sofs_t *so;
    ArrayIter iter;
    printf("-- Configuration Dump --\n");
    for (i = V_MASTER; i < V_STRINGS; i++)
	printf(" %16s : %s\n", ctlverbs[i].verb, sb_string(sb, nlc->str[i]));
    printf(" %16s : %s\n", "Make type", nltverbs[nlc->type]);
    printf(" %16s : %u\n", "Segment number", nlc->number);
    printf(" %16s : %s\n", "Publication day", dow[nlc->publish].verb);
    printf(" %16s : %s\n", "Processing day", nlc->process == V_NEVER ? "Never" : dow[nlc->process].verb);
    printf(" %16s : %s\n", "Notify type", notify_str(nlc->notify));
    printf(" %16s : %s %s\n", "Submit to", fidofmt(NULL, &nlc->submitaddr, sb_string(sb, nlc->str[V_NAME])), notify_str(nlc->submit & ~NF_NONE));
    printf(" %16s : %s\n", "My address", fidofmt(NULL, &nlc->myaddr, sb_string(sb, nlc->str[V_NAME])));
    printf(" %16s : %s\n", "Cleanup", nlc->cleanup ? "Yes" : "No");
    printf(" %16s : %u\n", "Min log level", nlc->loglevel);
    printf(" %16s : %u\n", "Min phone# parts", nlc->phone_parts_min);
    printf(" %16s : %u\n", "Max phone# parts", nlc->phone_parts_max);
    printf(" %16s : %s\n", "Pvt disposition", pvtverbs[nlc->pvtdisp]);
    i = 0;
    arrayIter_init(&iter, &nlc->Files);
    while ((sf = arrayIter_get(&iter, NEXT)) != NULL)
	printf(" %13s %2d : %s %-4u %s %s %s\n", "File", ++i,
	       nltverbs[sf->type].verb,
	       sf->number,
	       sb_string(sb, sf->name),
	       fidofmt(NULL, &sf->addr, NULL),
	       notify_str(sf->notify));
    printf("-- Local data --\n");
    arrayIter_init(&iter, &nlc->Data);
    while ((so = arrayIter_get(&iter, NEXT)) != NULL)
	printf("%s\n", sb_string(sb, *so));
    printf("-- End --\n");
}
#endif
