/* $Id: merge.c,v 1.1.1.1 1996/10/09 11:24:50 davidn Exp $
 * Merge a nodelist segment into a nodelist
 * All information required can be gathered from the source list and segment;
 * the only provision is that the file to merge is a complete segment, with
 * the first entry of the segment being the effective 'header' of the segment
 * (ie. "Hub" for a hub segment, "Host" for a network and so on.
 * The address passed in Dflt is only used for Region, Net and Hub segment
 * merges, where the default zone or network is required.
 *
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <errno.h>
#include "wfile.h"
#include "cfile.h"
#include "log.h"

int merge_list(char const * dest, char const * src, char const * tomerge, ZNNP addr);

enum {
    ISZONE, ISREGION, ISNET, ISHUB, ISNONE
};
static char const *kwds[] = {"Zone,", "Region,", "Net,", "Hub,"};

static unsigned
settype(char const * p, ZNNP * addr)
{
    unsigned i, l;
    for (i = 0; i < ISNONE && strncmp(kwds[i], p, l = strlen(kwds[i])) != 0; i++);	/* Look for the first
											 * matching keyword */
    switch (i) {
    case ISZONE:
	addr->zone = addr->net = atoi(p + l);
	addr->node = addr->point = 0;
	i = Bf_ZC;
	break;
    case ISREGION:
    case ISNET:
	addr->net = atoi(p + l);
	addr->node = addr->point = 0;
	i = (i == ISNET) ? Bf_NC : Bf_RC;
	break;
    case ISHUB:
	addr->node = atoi(p + l);
	addr->point = 0;
	i = Bf_HUB;
	break;
    case ISNONE:
	i = 0;
	break;
    }
    return i;
}

int
merge_list(char const * dest, char const * src, char const * tomerge, ZNNP addr)
{
    int rc = -1;
    WFILE S;			/* Source nodelist */
    WFILE M;			/* Segment to merge */

    static char const noopen[] = "Can't %s \"%s\": %s (%d)";
    wfile_init(&S, src);
    if (!wfile_open(&S))
	logit(LOG_ERROR, noopen, "open", src, strerror(errno), errno);
    else {
	wfile_init(&M, tomerge);
	if (!wfile_open(&M))
	    logit(LOG_ERROR, noopen, "open", tomerge, strerror(errno), errno);
	else {
	    int len;
	    CFILE *fp;		/* For the output file */
	    unsigned bflags;
	    char const *p;
	    /*
	     * First, let's scan the segment to merge and find out what sort of
	     * segment it is...
	     */
	    while ((p = wfile_ptr(&M, &len)) != NULL && *p == ';')
		wfile_advance(&M, len);
	    /* We're at the first entry, so let's see what it is... */
	    if ((bflags = settype(p, &addr)) == 0)
		logit(LOG_ERROR, "Unable to determine type of segment to merge");
	    /* Create the output file */
	    else if ((fp = cfopen(dest, "w+b")) == NULL)
		logit(LOG_ERROR, noopen, "create", dest, strerror(errno), errno);
	    else {
		off_t crcofs = (off_t) -1;
		char const *q;
		if ((p = wfile_ptr(&S, &len)) == NULL)
		    logit(LOG_ERROR, "Source file appears to be empty!");
		else {
		    int copying = 1;	/* Current mode is copy */
		    ZNNP taddr = {0, 0, 0, 0};
		    /*
		     * If there's a CRC-16 in the source file, remember where
		     * it was
		     */

		    if ((q = memchr(p, ':', len)) != NULL)
			crcofs = (off_t) ((q - p) + 2);

		    /*
		     * Copy the input file's first line, then reset the CRC-16
		     * for the destination file's handle.
		     */
		    rc = (cfwrite(p, 1, len, fp) != (unsigned) len);
		    cfreset(fp);
		    while (rc == 0 && wfile_advance(&S, len)) {
			p = wfile_ptr(&S, &len);
			if (bflags && wfile_analyse(&S, p, len) & bflags) {

			    /*
			     * Is it a contender?
			     */
			    if (copying) {
				taddr.zone = wfile_zone(&S);
				taddr.net = wfile_net(&S);
				taddr.node = wfile_node(&S);
				if (memcmp(&addr, &taddr, sizeof addr) == 0) {	/* Got it! */
				    int nlen;
				    /*
				     * Let's just copy the whole thing in
				     */
				    while (rc == 0 && (q = wfile_ptr(&M, &nlen)) != NULL) {

					/*
					 * Skip non-delimiting comments on the
					 * input file
					 */
					if ((*q != ';' || nlen == 3) && cfwrite(q, 1, nlen, fp) != (unsigned) nlen)
					    rc = -1;	/* Keep only delimiting
							 * comments */
					wfile_advance(&M, nlen);
				    }
				    copying = 0;

				    /*
				     * Change bflags to look for end of this
				     * segment in input list
				     */
				    if (bflags == Bf_HUB)
					bflags = (Bf_HUB | Bf_NC | Bf_RC | Bf_ZC);
				    else if (bflags == Bf_NC)
					bflags = (Bf_NC | Bf_RC | Bf_ZC);
				    else if (bflags == Bf_RC)
					bflags = (Bf_RC | Bf_ZC);
				}
			    } else {
				if (!(bflags & Bf_HUB))
				    cfwrite(";\r\n", 1, 3, fp);	/* Output a delimiter */
				copying = 1;	/* Switch back to copying from
						 * source again */
				bflags = 0;	/* Stop looking */
			    }
			}
			if (copying && cfwrite(p, 1, len, fp) != (unsigned) len)
			    rc = -1;
		    }
		    if (rc != 0)
			logit(LOG_ERROR, "I/O error during merge");
		    else if (bflags != 0) {
			logit(LOG_ERROR, "Did not find point at which to merge segment");
			rc = -1;
		    } else {

			/*
			 * Need to update the crc
			 */
			unsigned short crc16 = cfcrc16(fp);	/* Grab the CRC-16 */
			cfwrite("\1a", 1, 1, fp);
			if (crcofs != (off_t) -1) {	/* Update the CRC-16? */
			    cfseek(fp, crcofs, SEEK_SET);
			    cfprintf(fp, "%05u", crc16);	/* Stuff the CRC-16 in
								 * there */
			}
		    }
		}
		if (cfclose(fp) == EOF && rc == 0) {
		    logit(LOG_ERROR, "Write error on output file");
		    rc = -1;
		}
		if (rc != 0)
		    remove(dest);
	    }
	}
	wfile_destroy(&M);
    }
    wfile_destroy(&S);
    return rc;
}
