/* $Id: checknl.c,v 1.1.1.1 1996/10/09 11:25:08 davidn Exp $
 * Implements a function to check nodelist entries
 *
 */

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include "checknl.h"
#include "strbuf.h"
#include "osdep.h"

static struct {
    char const *keyword;	/* Keyword */
    int kwdlen;			/* Length to match */
    enum _nltype ctx;		/* Valid if context is type or below */
    unsigned long neflags;
}   nlkwds[] =
{
    {
	"Zone,", 5, NL_ZONE, NE_ZONE | NE_REGION | NE_NET | NE_HUB
    },
    {
	"Region,", 7, NL_REGION, NE_REGION | NE_NET | NE_HUB
    },
    {
	"Host,", 5, NL_NET, NE_NET | NE_HUB
    },
    {
	"Hub,", 4, NL_HUB, NE_HUB
    },
    {
	"Ogate,", 6, NL_HUB, NE_OGATE
    },
    {
	"Pvt,", 4, NL_HUB, NE_ISPVT
    },
    {
	"Down,", 5, NL_HUB, NE_ISDOWN
    },
    {
	"Hold,", 5, NL_HUB, NE_ISHOLD
    },
    {
	NULL, 0, NL_NONE, 0
    }
};


static int instringarray(Array * arr, char const * data, int len);
static char *substdata(char *buf, int *buflen, int startindex, int *endindex, char const * data, int datalen);
static int add_node(Array * arr, node_t n);
static int addnodeflag(char const * p, int len);
static int updphone(int atlevel, txtbuf * errs, char const * addr, char const * number, int nlen);


/* valid_nl_line() - determines validity of a nodelist line
 */

int
valid_nl_line(char const * line, enum _nltype listcontext, enum _nltype * context)
{

    /*
     * Valid in all contexts except 'none'
     */
    if (*line == ',')
	return !(*context == NL_NONE);
    else {
	int i = 0;
	while (nlkwds[i].keyword) {
	    if (strncmp(line, nlkwds[i].keyword, nlkwds[i].kwdlen) == 0) {

		/*
		 * Context is stepping up
		 */
		if (nlkwds[i].ctx > *context)
		    return (nlkwds[i].ctx <= listcontext) ? (*context = nlkwds[i].ctx, 1) : 0;
		if (*context == NL_ZONE)
		    return (nlkwds[i].ctx >= NL_REGION) ? (*context = nlkwds[i].ctx, 1) : 0;
		if (*context == NL_REGION)
		    return (nlkwds[i].ctx >= NL_NET) ? (*context = nlkwds[i].ctx, 1) : 0;
		*context = nlkwds[i].ctx;
		return 1;
	    }
	    ++i;
	}
	return 0;
    }
}


/* These flags are explicitly documented in FTS-0005 and are always legal
 * Additional flags may be added using the add_flag() and add_uflag()
 * functions.
 */

static char const *fts5_flags[] =
{
    /* Operating conditions */
    "CM", "LO", "M[NO]",
    /* ITU protocols */
    "V2[129]", "V3[234]", "V32b",
    /* Proprietary protocols */
    "H96", "HST", "H1[46]", "MAX", "PEP", "CSP", "ZYX", "VFC", "V32T",
    /* Error correction flags */
    "MNP", "V42", "V42b",
    /* File request flags */
    "X[ABCPRWX]",
    /* Gateway flags */
    "G*",

    /*
     * Other zone mail hour flags + combos This is not strictly kosher as it
     * will let entries like #29 though
     */
    "\\#[0-2]#",
    "\\#[0-2]#\\#[0-2]#",
    "\\#[0-2]#\\#[0-2]#\\#[0-2]#",
    "\\#[0-2]#\\#[0-2]#\\#[0-2]#\\#[0-2]#",
    "\\#[0-2]#\\#[0-2]#\\#[0-2]#\\#[0-2]#\\#[0-2]#",
    "![0-2]#",
    "![0-2]#![0-2]#",
    "![0-2]#![0-2]#![0-2]#",
    "![0-2]#![0-2]#![0-2]#![0-2]#",
    "![0-2]#![0-2]#![0-2]#![0-2]#![0-2]#",
    NULL
};

static char const *fts5_bauds[] =
{				/* Baud rates allowed by FTS-0005 */
    "300", "1200", "2400", "4800", "9600", "19200", "38400",
    NULL
};

static int init_done = 0;
static strbuf Strings;		/* Strings used by below */
static Array Bauds;		/* List of valid baud rates */
static Array Flags;		/* List of valid flags */
static Array UFlags;		/* List of valid user flags */
static Array nets;		/* List of nets in current zone */
static Array nodes;		/* List of nodes in current network */

static strbuf NodeStrings;	/* Flag strings for each individual node */
static Array NodeFlags;		/* Flags individual node */

static node_t lastzone,		/* Last zone, net, node */
    lastnet, lastnode;
static unsigned long lastresult;/* Result for previous node */
static int phonemin, phonemax, pvtdisp;

static unsigned char flaghandling, uflaghandling, phonehandling, innet;
#define UNPUBLEN  13
static char unpublished[] = "-Unpublished-";
static const char *phtype[] = {"hub", "net", "region", "zone"};
enum {
    p_HUB, p_NET, p_REGION, p_ZONE, p_MAX
};
static int lastlevel = p_HUB;
static struct {
    char number[32];		/* Phone number */
    char addr[32];		/* Address to whom number belongs */
    int used;			/* Number of times it is used */
    int numlen;
}   phones[p_MAX];


/* updphone() - update hub/coord phone number
 */

static int
updphone(int atlevel, txtbuf * errs, char const * addr, char const * number, int nlen)
{
    int i, rc = 0;
    if (atlevel > p_ZONE)
	atlevel = p_ZONE;

    /*
     * First, let's look for non-redundancies admin nodes
     */
    if (phonehandling) {
	for (i = lastlevel; i <= atlevel; i++)
	    if (phones[i].used == 0 && phones[i].number[0] && (i < phonehandling)) {
		++rc;
		tb_printf(errs, "%c%s is a non-redundant %s entry\n", nm_type(NM_WARNING | NM_LOG | NM_FEEDBACK | NM_NOLINE), phones[i].addr, phtype[i]);
	    }
	lastlevel = atlevel;
    }
    /*
     * Then, copy in the new phone number
     */
    if (addr == NULL)
	addr = "";

    for (i = p_HUB; i <= atlevel; i++) {
	memcpy(phones[i].number, number, nlen);
	phones[i].number[phones[i].numlen = nlen] = '\0';
	strncpy(phones[i].addr, addr, sizeof(phones[i].addr));
	phones[i].used = 0;
    }

    return rc;
}


/* init_noinfo() - initialise the flags, baud rate and flag tables
 */

void
init_nlinfo(int pmin, int pmax, int pdisp, char const * bauds, node_t zone, node_t net, unsigned char flagh, unsigned char uflagh, unsigned char phoneh)
{
    int i;
    static char const fn[] = "init_nlinfo";
    /*
     * Init last node status values
     */
    lastzone = lastnet = lastnode = (node_t) - 1;
    lastresult = 0L;
    lastzone = zone;
    lastnet = net;
    flaghandling = flagh;
    uflaghandling = uflagh;
    phonehandling = phoneh;
    innet = 0;

    memset(phones, 0, sizeof(phones));
    for (i = 0; i < p_MAX; i++)
	phones[i].used = -1;

    if (!init_done) {

	phonemin = pmin;
	phonemax = pmax;
	pvtdisp = pdisp;

	sb_init(&NodeStrings, 512);
	array_init(&NodeFlags, sizeof(sofs_t), 16);

	/*
	 * Common strings buffer
	 */
	sb_init(&Strings, 1024);

	/*
	 * Init baud rate array
	 */

	array_init(&Bauds, sizeof(sofs_t), 12);
	if (bauds != NULL && *bauds) {

	    /*
	     * Assume it is a delimited list of baudrates in a string
	     */
	    char *p, *s;
	    char const seps[] = " ,|";
	    p = zstrdup(fn, bauds);
	    if ((s = strtok(p, seps)) != NULL)
		do {
		    sofs_t *so = array_ptr(&Bauds, array_add(&Bauds, NOELEMENT, 1));
		    *so = sb_alloc(&Strings, s);
		} while ((s = strtok(NULL, seps)) != NULL);
	    zfree(fn, p);
	} else {

	    /*
	     * Otherwise, just fill in the defaults
	     */
	    for (i = 0; fts5_bauds[i]; i++) {
		sofs_t *so = array_ptr(&Bauds, array_add(&Bauds, NOELEMENT, 1));
		*so = sb_alloc(&Strings, fts5_bauds[i]);
	    }
	}

	/*
	 * Initialise the flags array
	 */
	array_init(&Flags, sizeof(sofs_t), 64);
	for (i = 0; fts5_flags[i]; i++)
	    add_flag(fts5_flags[i]);

	array_init(&UFlags, sizeof(sofs_t), 32);

	/*
	 * Initialise the net/node arrays
	 */
	array_init(&nets, sizeof(node_t), 1024);	/* Nets in current zone */
	array_init(&nodes, sizeof(node_t), 1024);	/* Nodes in current net */
    }
}


/* add_newflag() - Adds a flag to the 'allowed' flags array
 */

static int
add_newflag(Array * arr, char const * flag)
{
    if (flag && *flag) {
	ArrayIter i;
	sofs_t *sp;
	/*
	 * Check to see if the flag is not already there
	 */
	arrayIter_init(&i, arr);
	while ((sp = arrayIter_get(&i, NEXT)) != NULL)
	    if (strcmp(sb_string(&Strings, *sp), flag) == 0)
		return 0;

	/*
	 * No, then we can add it
	 */
	sp = array_ptr(arr, array_add(arr, NOELEMENT, 1));
	*sp = sb_alloc(&Strings, flag);
	return 1;
    }
    return 0;
}


/* add_flag() - Adds a standard flag
 */

int
add_flag(char const * flag)
{
    return add_newflag(&Flags, flag);
}


/* add_uflag() - Adds a user flag
 */

int
add_uflag(char const * flag)
{
    return add_newflag(&UFlags, flag);
}


/* instringarray() - determines presence of flag in an array
 */

static int
instringarray(Array * arr, char const * data, int len)
{
    int rc = 0;
    sofs_t *so;
    char tmp[128];
    ArrayIter iter;
    /*
     * wildmatch() requires asciiz strings
     */
    strncpy(tmp, data, len);
    tmp[len] = '\0';

    /*
     * Search the array until we find a match
     */
    arrayIter_init(&iter, arr);
    while (rc == 0 && (so = arrayIter_get(&iter, NEXT)) != NULL) {
	char const *patt = sb_string(&Strings, *so);
	rc = wildmatch(tmp, patt, 0);
    }
    return rc;
}


/* substdata() - Substitute data within a nodelist line
 */

static char *
substdata(char *buf, int *buflen, int startindex, int *endindex, char const * data, int datalen)
{
    static char workbuf[MAXLINE];	/* Work buffer */
    int length = *endindex - startindex;
    if (buf != workbuf) {

	/*
	 * Move it to work buffer if we need to do anything with it The
	 * original buffer is logically 'const'
	 */
	buf = memmove(workbuf, buf, *buflen);
	workbuf[*buflen] = '\0';
    }
    if (datalen != length) {
	int blklen = *buflen - *endindex;
	if (blklen > 0)
	    memmove(buf + startindex + datalen, buf + *endindex, blklen);
	blklen = datalen - length;
	*buflen += blklen;
	*endindex += blklen;
    }
    if (datalen > 0)
	memmove(buf + startindex, data, datalen);

    return buf;
}



/* add_node() - Attempt to add a node to an array, but check for dupes
 */

static int
add_node(Array * arr, node_t n)
{
    node_t *np;
    ArrayIter iter;
    /*
     * This could be improved with a binary search and insertion
     */
    arrayIter_init(&iter, arr);
    while ((np = arrayIter_get(&iter, NEXT)) != NULL)
	if (n == *np)
	    return 0;
    /* Not in the array so we can add it */
    np = array_ptr(arr, array_add(arr, NOELEMENT, 1));
    *np = n;
    return 1;
}


/* addnodeflag() - Add flag to a node
 */

static int
addnodeflag(char const * p, int len)
{
    int rc = 0;
    sofs_t *so;
    ArrayIter iter;
    /*
     * Search the array until we find a match
     */
    arrayIter_init(&iter, &NodeFlags);
    while (rc == 0 && (so = arrayIter_get(&iter, NEXT)) != NULL) {
	char const *flag = sb_string(&NodeStrings, *so);
	rc = (strncmp(p, flag, len) == 0 && flag[len] == '\0');
    }

    /*
     * If no match found, then add it
     */
    if (rc == 0) {
	so = array_ptr(&NodeFlags, array_add(&NodeFlags, NOELEMENT, 1));
	*so = sb_salloc(&NodeStrings, p, len);
    }
    return !rc;
}


/* check_nodeent() - Check a nodelist entry
 */

unsigned long
check_nodeent(txtbuf * errs, char *txtaddr, char **line, int *plen, ZNNP * addr)
{
    int i, j, field, len = *plen, uflags = 0;
    char *p;
    node_t n = 0;
    Array *arr = &Flags;
    unsigned long result = NE_OK;
    unsigned char fhandle = flaghandling;
    /*
     * This is used to complete processing only
     */
    if (line == NULL) {
	if (updphone(len, errs, txtaddr, "", 0))
	    result |= NE_NOTRED;
	return result;
    }
    /*
     * Cast away const, but treated that way regardless
     */
    p = (char *) *line;
    strcpy(txtaddr, "??Addr??");

    /*
     * Reset this node's flags array/buffer
     */
    sb_reset(&NodeStrings);
    array_reset(&NodeFlags);

    if (len > MAXLINE) {
	result |= NE_LONG;
	tb_printf(errs, "%cEntry too long\n", nm_type(NM_ERROR | NM_LOG | NM_FEEDBACK));
    }
    if (addr->net == 0 && addr->zone == 0) {
	addr->net = lastnet;
	addr->zone = lastzone;
    }
    for (i = 0; i < len; ++i) {

	/*
	 * Hi-bit set
	 */
	if ((p[i] & 0x80) || p[i] < ' ')
	    result |= NE_8BITCTRL;

	/*
	 * Embedded space
	 */
	else if (isspace(p[i]))
	    result |= NE_SPACE;
    }

    if ((result & NE_8BITCTRL) == NE_8BITCTRL)
	tb_printf(errs, "%cContains illegal characters\n", nm_type(NM_ERROR | NM_LIST | NM_LOG | NM_FEEDBACK));

    if ((result & NE_SPACE) == NE_SPACE)
	tb_printf(errs, "%cContains embedded whitespace\n", nm_type(NM_ERROR | NM_LIST | NM_LOG | NM_FEEDBACK));

    for (i = field = 0; i < len; ++field, ++i) {
	int fl;
	char *q;
	if ((q = memchr(p, ',', len - i)) == NULL) {

	    /*
	     * Not enough required fields
	     */
	    if (field < N_BAUDRATE) {
		result |= NE_NOFIELDS;
		tb_printf(errs, "%cInsufficient number of fields\n", nm_type(NM_ERROR | NM_LIST | NM_LOG | NM_FEEDBACK));
	    }
	    q = p + len - i;
	}
	fl = q - p;
	i += fl;

	if (q != NULL && field > N_BAUDRATE && (i + 1) == len) {

	    /*
	     * Redundant trailing comma after flag
	     */
	    result |= NE_TRCOMMA;
	    --len;
	    tb_printf(errs, "%cTrailing comma removed\n", nm_type(NM_WARNING | NM_LOG | NM_FEEDBACK));
	}
	switch (field) {
	case N_KWD:		/* Keyword */
	    if (fl) {		/* Only check keywords */
		for (j = 0; nlkwds[j].keyword != NULL; j++)
		    if (fl == nlkwds[j].kwdlen - 1 && strncmp(p, nlkwds[j].keyword, nlkwds[j].kwdlen) == 0)
			break;

		/*
		 * Invalid keyword
		 */
		if (nlkwds[j].keyword == NULL) {
		    result |= NE_KEYWORD;
		    tb_printf(errs, "%cInvalid keyword \"%.s\"\n", nm_type(NM_ERROR | NM_FEEDBACK | NM_LOG | NM_FEEDBACK), fl, p);
		} else {
		    result |= nlkwds[j].neflags;
		    innet = !(result & (NE_REGION | NE_ZONE));
		}
	    }
	    break;

	case N_NUMBER:		/* Number */
	    if (!fl)		/* At least one digit */
		result |= NE_NODENUM;
	    else
		for (j = 0; j < fl; j++) {
		    if (!isdigit(p[j]))	/* Must be digit */
			result |= NE_NODENUM;
		    else if ((n = (node_t) (n * 10 + (p[j] - '0'))) == 0)
			result |= NE_NODENUM;	/* Catch leading zeros */
		}
	    if (n > 32767)
		result |= NE_NODENUM;	/* Catch leading zeros */

	    if ((result & NE_NODENUM) == NE_NODENUM)
		tb_printf(errs, "%cInvalid Node number \"%.*s\"\n", nm_type(NM_ERROR | NM_LIST | NM_LOG | NM_FEEDBACK), fl, p);
	    else {
		if (result & NE_NET) {
		    addr->node = 0;
		    addr->net = n;
		    array_reset(&nodes);
		    if (result & NE_ZONE) {
			addr->zone = n;
			array_reset(&nets);
		    }
		    /*
		     * Can't add a duplicate
		     */
		    else if (!add_node(&nets, n)) {

			/*
			 * Ouch - whole net is a dupe
			 */
			result |= NE_NETDUPE;
			tb_printf(errs, "%cDuplicate Network number \"%d\"\n", nm_type(NM_ERROR | NM_LIST | NM_LOG | NM_FEEDBACK), n);
		    }
		    /*
		     * Node is actually /0 for coordinators
		     */
		    n = 0;
		} else {
		    addr->node = n;
		    /* Preserve this until end of net */
		    if ((lastresult & NE_NETDUPE) == NE_NETDUPE)
			result |= NE_NETDUPE;
		}

		/*
		 * Duplicated node
		 */
		if (!add_node(&nodes, n)) {
		    result |= NE_NODEDUPE;
		    tb_printf(errs, "%cDuplicate Node number \"%d\"\n", nm_type(NM_ERROR | NM_LIST | NM_LOG | NM_FEEDBACK), n);
		}
		if (addr->zone)
		    sprintf(txtaddr, "%d:%d/%d", addr->zone, addr->net, addr->node);
		else
		    sprintf(txtaddr, "%d/%d", addr->net, addr->node);
	    }
	    break;

	case N_SYSNAME:	/* System Name */
	case N_LOCATION:	/* System location */
	case N_SYSOP:		/* Name of system operator */
	    if (fl == 0) {
		result |= NE_NOFIELDS;	/* Can't be empty */
		tb_printf(errs, "%cBlank field in entry\n", nm_type(NM_WARNING | NM_LOG | NM_FEEDBACK));
	    }
	    break;

	case N_PHONE:		/* Phone number */
	    if (!fl || fl >= 32) {
		tb_printf(errs, "%cPhone number \"%.*s\" - invalid length\n", nm_type(NM_ERROR | NM_LIST | NM_LOG | NM_FEEDBACK), fl, p);
		result |= NE_LONG;
	    } else if (result & NE_ISPVT) {

		/*
		 * Use as phone number index
		 */
		int pvt = p_HUB;
		/*
		 * Check pvt handling
		 */
		switch (pvtdisp) {
		case PVT_ZONE:
		    ++pvt;
		case PVT_REGION:
		    ++pvt;
		case PVT_NET:
		    ++pvt;
		case PVT_HUB:
		    if (phones[pvt].used > 0) {

			/*
			 * Substitute phone numbers
			 */
			*line = substdata(*line, &len, i - fl, &i, phones[pvt].number, phones[pvt].numlen);
			/* phones[pvt].used++;   / * Use, of a sort */
			q = *line + i;
			memmove(p, p + 3, len - 2);
			i -= 3;
			len -= 3;
			q -= 3;
			break;
		    }
		    result |= NE_PVTALLW;
		    goto nopvt;
		case PVT_OK:
		    if (innet) {
			if (fl != UNPUBLEN || strnicmp(p, unpublished, UNPUBLEN) != 0) {
			    *line = substdata(*line, &len, i - fl, &i, unpublished, UNPUBLEN);
			    q = *line + i;
			}
			break;
		    }
		    result |= NE_PVTERR;
		    goto nopvt;
		    /* Fallthru */
		default:
		case PVT_NONE:
		    result |= NE_PVTALLW;
	    nopvt:
		    tb_printf(errs, "%cPvt node not allowed here\n", nm_type(NM_ERROR | NM_LOG | NM_FEEDBACK));
		    break;
		}
	    } else if ((result & (NE_ISDOWN | NE_ISHOLD)) && fl == UNPUBLEN && strnicmp(p, unpublished, UNPUBLEN) == 0) {

		/*
		 * Normalise all "-Unpublished-" strings
		 */
		*line = substdata(*line, &len, i - fl, &i, unpublished, UNPUBLEN);
		q = *line + i;
	    } else {

		/*
		 * Allow -Unpublished- for down and hold nodes
		 */
		int parts = 1;
		if (p[0] == '-' || p[fl - 1] == '-')
		    result |= NE_PHONECHR;
		for (j = 0; j < fl; j++) {
		    if (p[j] == '-')
			++parts;
		    else if (!isdigit(p[j]))
			result |= NE_PHONECHR;
		}
		if ((result & NE_PHONECHR) == NE_PHONECHR)
		    tb_printf(errs, "%cInvalid characters in phone number \"%.*s\"\n", nm_type(NM_ERROR | NM_LIST | NM_LOG | NM_FEEDBACK), fl, p);

		if ((phonemin && parts < phonemin) || (phonemax && parts > phonemax)) {
		    tb_printf(errs, "%cInvalid number of parts in phone number \"%.*s\"\n", nm_type(NM_ERROR | NM_LIST | NM_LOG | NM_FEEDBACK), fl, p);
		    result |= NE_PHONEPART;
		}
		/*
		 * Save numbers for Pvt phone number substitution
		 */
		else if ((result & NE_PHONECHR) != NE_PHONECHR && (result & NE_HUB)) {
		    int atlevel = (result & NE_ZONE)
		    ? p_ZONE
		    : (result & NE_REGION)
		    ? p_REGION
		    : (result & NE_NET)
		    ? p_NET
		    : p_HUB;
		    updphone(atlevel, errs, txtaddr, p, fl);
		}
		/*
		 * If this is a non-administrative entry, compare this node's
		 * phone number with the admin entries above it and see if it
		 * is the same. If it is, then we know that the admin entry is
		 * (correctly) redundant. This slows things down, but it is
		 * often worth the trouble to look.
		 */
		else
		    for (j = 0; j < p_MAX; j++)
			if (fl == phones[j].numlen &&
			    phones[j].number[fl - 1] == p[fl - 1] &&	/* Might help speed it
									 * up */
			    memcmp(phones[j].number, p, fl) == 0)
			    phones[j].used++;
	    }
	    break;

	case N_BAUDRATE:	/* Nominal baud rate */
	    if (!instringarray(&Bauds, p, fl)) {
		tb_printf(errs, "%cInvalid baudrate \"%.*s\"\n", nm_type(NM_ERROR | NM_LIST | NM_LOG | NM_FEEDBACK), fl, p);
		result |= NE_BAUDRATE;
	    }
	    break;

	default:		/* Flags */
	    if (*p == 'U') {	/* Start of user flags */
		arr = &UFlags;
		fhandle = uflaghandling;
		if (uflags && (fhandle != FLAG_IGNORE)) {
		    tb_printf(errs, "%cRedundant 'U' in user flag \"%.*s\"\n", nm_type(NM_WARNING | NM_LOG | NM_FEEDBACK), fl, p);
		    result |= NE_UREPEAT;
		} else
		    ++uflags;
		++p;

		/*
		 * Sigh, allow for a blank user flag This is a technical
		 * violation of FTS-5, but seems to be encouraged in some
		 * zones...
		 */
		if (!--fl) {

		    /*
		     * However, don't allow a hanging 'U'
		     */
		    if (i == len) {
			result |= NE_LONG;
			tb_printf(errs, "%c'U' specified with no trailing flags - removed\n", nm_type(NM_WARNING | NM_LOG | NM_FEEDBACK));
			len -= 2;	/* Remove ",U" from end */
		    }
		    break;
		}
	    }
	    /*
	     * This is the generic flags handling code Don't worry about it if
	     * we're told to ignore it
	     */
	    if (fhandle != FLAG_IGNORE) {
		char const *msg = NULL;
		if (uflags && instringarray(&Flags, p, fl))
		    tb_printf(errs, "%cflag \"U%.*s\" should not be U-flag\n", nm_type(NM_WARNING | NM_LOG | NM_FEEDBACK), fl, p);

		if (fl > 32) {
		    result |= NE_LONG;
		    msg = "Field too long -";
		} else if (!addnodeflag(p, fl)) {
		    result |= NE_DUPEFLAG;
		    msg = "Duplicate";
		} else if (!instringarray(arr, p, fl)) {
		    result |= NE_ILLFLAG;
		    msg = "Unknown";
		}
		if (msg != NULL) {
		    char const *flagmsg = (fhandle & FLAG_REMOVE) ? " removed" : "";
		    /*
		     * Report the problem
		     */
		    tb_printf(errs, "%c%s %sflag \"%.*s\"%s\n",
			  ((fhandle & FLAG_ERROR) && !(fhandle & FLAG_SKLIST))
			  ? nm_type(NM_ERROR | NM_LOG | NM_LIST | NM_FEEDBACK)
			      : nm_type(NM_WARNING | NM_LOG | NM_FEEDBACK),
			      msg, uflags ? "user" : "", fl, p, flagmsg);

		    /*
		     * Remove errant flag
		     */
		    if (fhandle & FLAG_REMOVE) {
			p = *line;

			/*
			 * Remove leading comma
			 */
			if (p[i - fl - 1] == ',')
			    ++fl;
			*line = substdata(p, &len, i - fl, &i, "", 0);
			q = *line + i;
		    }
		    /*
		     * Skip entry entirely
		     */
		    if (fhandle & FLAG_SKLIST)
			result |= NE_SKIPENT;

		    /*
		     * Output as error
		     */
		    else if (fhandle & FLAG_DELIST)
			result |= NE_ERRORENT;
		}
	    }
	    break;

	}

	/*
	 * advance over comma
	 */
	p = ++q;
    }

    /*
     * Fix length
     */
    *plen = len;
    lastresult = result;
    if ((result & NE_NET) && (result & NE_ERRORENT)) {

	/*
	 * Ouch, problem with coordinator entry, we have to dump the whole net!
	 * Not really a dupe, but it will do what we want
	 */
	lastresult |= NE_NETDUPE;
    }
    return result;
}
