; Low level Sound Blaster interface by White Shadow
; interface format from Tran
; v0.676
; 2b added:
; - special mixes for low step rates (less than .5)
; - Change get_freq to accept EAX as an input
; - More volume control: change input to 0-255 and use 32 tbls
;   giving more resolution to the low volumes
.386p
Ideal
b equ byte
w equ word
d equ dword
ofs equ offset

Happy = 0
Debug = 0
TimingBar = 0

IRQ_Rate = 72                   ;# irq's per second   20 < x < 200
LoopSizeMIN = 128
MixRateMAX = 21739
MixRateMIN = 8000
PlayFreqMIN = 1024
SampleSizeMAX = 8*1024*1024
StepRateMIN = 64

BufferSizeMax = (MixRateMAX / IRQ_Rate)+1
IF BufferSizeMax gt 2000
  Error
EndIf
;
segment code32 para public use32
        assume cs:code32, ds:code32
masm
include pmode.inc
include low_data.inc
ideal
;
public  _sb_data
;
; DATA
;
align 4
Ch_Vol          dd 16 dup(0)    ;Volume table to use (offset / 2)
Ch_StepRate     dw 16 dup(0)    ;16bit StepRate (freq value)
Ch_HighStep     db 16 dup(0)    ;StepRate SHR 12
Ch_LowStep      dd 16 dup(0)    ;StepRate AND 0FFFh SHL 20
Ch_HighPos      dd 16 dup(0)    ;Sample Posistion (memory)
Ch_LowPos       dd 16 dup(0)    ;low 12 bits of pos SHL 20  OR
                                ; 01h sample off  OR
                                ; 02h looping sample
Ch_EndPos       dd 16 dup(0)    ;Last sample to play + 1  OR
                                ;Loop end + 1
Ch_LoopBegin    dd 16 dup(0)    ;Loop start
Ch_MixesLeft    dd 16 dup(0)    ;#Mixes B4 Ch_EndPos is reached
If Happy ne 0                   ;What phun! 
Ch_Happy        db 16 dup(0)
Ch_HappyOld     db 16 dup(0)
EndIf

align 4
MixingBufOfs    dd 0            ;Ofs of 16bit mixing buffer
BufXlatTblOfs   dd 0            ;Table to xlate to 8bits
VolumeTblOfs    dd 16 dup(0)    ;Ofs of all 16 volume tbls
DMABufOfs       dd 3 dup(0)     ;Ofs of playback bufs
BufXlatTblRLE   dw 925,75,52,41,34,31,26,24,23,20,19,18,17,16,15,15
                dw 14,13,13,13,12,11,11,11,11,10,10,10,10,9,9,9
                dw 9,9,8,8,9,8,7,8,8,7,8,7,7,7,7,7
                dw 7,6,7,6,7,6,6,7,6,6,6,6,6,6,5,6
                dw 6,5,6,6,5,6,5,5,6,5,5,5,5,6,5,5
                dw 5,5,5,4,5,5,5,5,5,4,5,5,4,5,5,4
                dw 5,4,5,4,5,4,5,4,5,4,4,5,4,4,5,4
                dw 4,4,5,4,4,4,4,5,4,4,4,4,4,4,4,4
                dw 5,4,4,4,4,4,4,4,5,4,4,4,4,5,4,4
                dw 4,5,4,4,5,4,4,5,4,5,4,5,4,5,4,5
                dw 4,5,5,4,5,5,4,5,5,5,5,5,4,5,5,5
                dw 5,5,6,5,5,5,5,6,5,5,6,5,6,6,5,6
                dw 6,5,6,6,6,6,6,6,7,6,6,7,6,7,6,7
                dw 7,7,7,7,7,8,7,8,8,7,8,9,8,8,9,9
                dw 9,9,9,10,10,10,10,11,11,11,11,12,13,13,13,14
                dw 15,15,16,17,18,19,20,23,24,26,31,34,41,52,75,922

CurrentMixBuf   db 0
CurrentPlayBuf  db 0
NextBufTbl      db 1,2,0

align 4
rmsbirqbuf      db 21 dup(?)    ; buffer for rm SB IRQ callback code
align 4
OldRMIRQvect    dd 0
OldIRQvect      dd 0
OldIRQmask      db 0
ChkSBIRQ        db 10101100b    ;Valid SB IRQs
SBIRQ           db 0

align 2
_sb_port_00     dw 0
_sb_port_06     dw 0
_sb_port_0C     dw 0
_sb_port_0E     dw 0

align 4
TicksB4Beat     dd 0            ;Samples left B4 calling ext routine
TicksPerBeat    dw 0            ;Samples mixed every beat
TicksPerIRQ     dw 0            ;Samples mixed every IRQ
MixFreq         dw 0            ;Samples played in one second
                dw 0            ;To access as a dword
BeatsPerSecond  db 0            ;Beats every second
ChannelsMinus1  db 0            ;#Channels - 1

align 4
_sb_data        dd      _sb_init, _sb_uninit
                dd      _sb_set_bps, _sb_set_mixrate
                dd      _sb_set_voices, _sb_get_freq, _sb_put_data
                dd      _sb_get_data, _ret
                db      00010110b       ; info bitmap
                                        ;  bit 0: 0=system RAM, 1=card RAM
                                        ;  bit 1: port needed
                                        ;  bit 2: IRQ number needed
                                        ;  bit 3: DMA number needed
                                        ;  bit 4: Mixing rate selectable
                db      'Sound Blaster',0,'$'

dummysample     db 128
errorcode       db 0
IRQflag         db 0
codeflags       db 0
_init_f = 01h           ; bit 0: _sb_init
_mixrate_f = 02h        ; bit 1: _sb_set_mixrate
_bps_f = 04h            ; bit 2: _sb_set_bps
_voices_f = 08h         ; bit 3: _sb_set_voices
_mixing_f = 10h         ; bit 4: _sb_mixing
;
; CODE
;
If Debug ne 0
_putdosmsg:
        push ax edx
        add  edx, [_code32a]
        mov  al, dl
        and  ax, 0fh
        shr  edx, 4
        mov  [v86r_ds], dx
        mov  [v86r_dx], ax
        mov  [v86r_ah], 9
        mov  al, 21h
        int  33h
        pop  edx ax
        ret
EndIf
Macro SaveIF
        mov  ax, 900h
        int  31h
        push ax
EndM
Macro RestoreIF
        pop  ax
        int  31h
EndM
;

;
;                                   SB INIT
; In:
;   _low_port  _low_irq  _low_buflen  _low_bufptr
;
; Out:
;   No Carry:   _low_buf - ammount of memory needed
;   Carry:      al       - error code
;
init_er_NotNufMem       = 10h
init_er_MemAbove640k    = 11h
init_er_NoCard          = 12h
init_er_BadIRQ          = 13h
_sb_init:
        pushad
        SaveIF
        test [codeflags], _init_f
        jz  @@00
        call _sb_uninit

        ;-- Set up port addx
@@00:   mov  ax, [_low_port]
        mov  [_sb_port_00], ax
        add  ax, 06h
        mov  [_sb_port_06], ax
        add  ax, 06h
        mov  [_sb_port_0C], ax
        add  ax, 02h
        mov  [_sb_port_0E], ax

        ;-- Reset SB, found it?
        call _sb_reset_card
        jc   @@_no_card

        ;-- Turn on DAC  (pop!)
        mov  al, 0D1h
        call _sb_out_2xC

        ;-- Is there enough mem?
        ;--       DMA Buffers         MixingBuffer          VolTbl   Xlat
        mov  ebx, (BufferSizeMax*3*2)+((BufferSizeMax*2)+1)+(256*17)+4096
        cmp  [_low_buflen], ebx
        jb   @@_not_nuf_mem
        mov  [_low_buf], ebx

        ;-- Do DMA wrap check
        mov  eax, [_low_bufptr]
        add  eax, [_code32a]            ;Convert to absolute address
        add  ebx, eax
        jc   @@_mem_gt_640k
        cmp  ebx, 000A0000h             ;Start+Len >= 640k
        jae  @@_mem_gt_640k
        mov  ebx, eax
        add  ebx, BufferSizeMax*3
        shr  eax, 16
        shr  ebx, 16
        cmp  eax, ebx                   ;<> then crosses page
        mov  eax, [_low_bufptr]         ;move does not touch any flags
        je   @@10
        add  eax, BufferSizeMax*3       ;select next buffer
@@10:   mov  [DMABufOfs], eax           ;Other DMA Buffer ptrs set l8r
        add  eax, BufferSizeMax*3       ;Next Buffer

        ;-- Find and align mixing buffer
        test al, 01h
        jz   @@11
        inc  eax
@@11:   mov  [MixingBufOfs], eax
        add  eax, BufferSizeMax*2       ;Next Buffer

        ;-- Find and align volume tables
        cld
        mov  ebx, eax
        and  ebx, 255
        jz   @@12a
        and  eax, NOT (255)
        add  eax, 256
@@12a:  mov  edi, ofs VolumeTblOfs
        mov  ecx, 16
@@12L:  stosd
        add  eax, 256
        dec  ecx
        jnz  @@12L

        ;-- Find Xlat Tbl
        mov  [BufXlatTblOfs], eax

        ;-- Set SB IRQ
        movzx ecx, [_low_irq]
        mov  al, 01h
        shl  al, cl
        and  al, [ChkSBIRQ]             ;A valid SB IRQ?
        jz   @@_irq_bad
        mov  [SBIRQ], cl
        mov  bl, cl
        call [_getirqvect]              ;Get old IRQ info
        mov  [OldIRQvect], edx
        call _getirqmask
        mov  [OldIRQmask], al
        mov  edx, ofs _DummyIrq         ;Write new IRQ info
        call [_setirqvect]
        xor  al, al
        call _setirqmask
        mov  edi, ofs rmsbirqbuf        ;real mode callback buffer
        call _rmpmirqset
        mov  [OldRMIRQvect],eax

        ;-- Done main init
        or   [codeflags], _init_f       ;Set init'd flag
        RestoreIF
        popad
        clc
        ret
@@_irq_bad:
        mov  [errorcode], init_er_BadIRQ
        jmp  short @@99
@@_not_nuf_mem:
        mov  [errorcode], init_er_NotNufMem
        jmp  short @@99
@@_mem_gt_640k:
        mov  [errorcode], init_er_MemAbove640k
        jmp  short @@99
@@_no_card:
        mov  [errorcode], init_er_NoCard
@@99:   and  [codeflags], NOT(_init_f)
        RestoreIF
        popad
        mov  al, [errorcode]
        stc
        ret

;
;                                  SB UNINIT
;
_sb_uninit:
        test [codeflags], _init_f
        jz   @@_not_init
        pushad
        SaveIF

        call _sb_stopmix
        mov  [codeflags], 0

        ;-- restore IRQ
        mov  bl, [SBIRQ]
        mov  edx, [OldIRQvect]
        call [_setirqvect]
        mov  eax, [OldRMIRQvect]
        call _rmpmirqfree
        mov  al, [OldIRQmask]
        call _setirqmask

        RestoreIF
        popad
@@_not_init:
        ret

;
_sb_startmix:
        pushad
        mov  al, [codeflags]
        test al, _mixing_f
        jnz  @@_mixing                  ;already mixing?
        and  al, 0Fh                    ;init mixrate bps voices
        cmp  al, 0Fh
        jne  @@_not_ready               ;all init routines called?
        SaveIF

        ;-- Clear DMA buffers
        cld
        mov  edi, [DMABufOfs]           ;Relative start of 1st buffer
        movzx ecx, [TicksPerIRQ]        ;Size of one buffer
        lea  ecx, [(ecx*2)+ecx]         ;Clear 3 buffers
        mov  ax, 8080h
        shr  ecx, 1
        rep  stosw
        adc  ecx, ecx                   ;carry held from  shr ecx, 1
        rep  stosb

        ;-- Init Mixer Data
        mov  [CurrentMixBuf], 2
        mov  [CurrentPlayBuf], 0
        mov  [TicksB4Beat], 1

        ;-- Init Channel Data
        mov  edi, ofs Ch_Vol
        mov  eax, [VolumeTblOfs]
        mov  ecx, 16                    ;Volume
        rep  stosd
        xor  eax, eax
        mov  ecx, 8+4+16                ;StepRate HighStep LowStep
        rep  stosd
        mov  eax, ofs dummysample
        mov  ecx, 16                    ;HighPos
        rep  stosd
        mov  eax, 01h                   ;LowPos  (samples off)
        mov  ecx, 16
        rep  stosd
        xor  eax, eax
        mov  ecx, 16+16+16              ;End LoopBegin MixesLeft
        rep  stosd

        ;-- Set MixerIRQ
        mov  [IRQflag], 0
        mov  bl, [SBIRQ]
        mov  edx, ofs _IRQ
        call [_setirqvect]
        mov  edi, ofs rmsbirqbuf        ;real mode callback buffer
        call _rmpmirqset

        ;-- Set DMA channel 1
	mov  al, 5
	out  0Ah, al                    ;MASK chan 1
	xor  al, al
	out  0Ch, al                    ;FF register -Clear MSB/LSB flipflop...
        mov  al, 59h                    ;looping mode
	out  0Bh, al                    ;MODE register
        mov  eax, [DMABufOfs]
        add  eax, [_code32a]            ;absolute addx
        out  02h, al                    ;Addx low then high
        mov  al, ah
        out  02h, al
        shr  eax, 16
        out  083h, al
        movzx eax, [TicksPerIRQ]
        lea  eax, [(eax*2)+eax]
        dec  ax                         ;All DMA buffers' length - 1
        out  03h, al
        mov  al, ah
        out  03h, al
        mov  al, 1
        out  0Ah, al                    ;UnMask chan 1

        ;-- Tell SB to start playing
        mov  al, 014h                   ;Play Cmd
        call _sb_out_2xC
        mov  ax, [TicksPerIRQ]
        dec  ax                         ;One DMA buffer's length - 1
        call _sb_out_2xC
        mov  al, ah
        call _sb_out_2xC

        or   [codeflags], _mixing_f
        RestoreIF
@@_not_ready:
@@_mixing:
        popad
        ret

;
_sb_stopmix:
        push eax ebx edx
        SaveIF
        test [codeflags], _mixing_f
        jz  @@_not_mixing

        ;-- Stop (pause?) SB
        mov  al, 0D0h
        call _sb_out_2xC

        ;-- Mask DMA
        mov  al, 5
        out  0Ah, al

        ;-- Set DumbIRQ
        mov  bl, [SBIRQ]
        mov  edx, ofs _DummyIrq
        call [_setirqvect]
        mov  edi, ofs rmsbirqbuf        ;real mode callback buffer
        call _rmpmirqset

        ;-- IRQ Ack, don't know if it is needed here, but it don't hurt
        mov  dx, [_sb_port_0E]
        in   al, dx

        and  [codeflags], not _mixing_f
@@_not_mixing:
        RestoreIF
        pop  edx ebx eax
        ret

;
; Set number of active voices
; In:
;   AL - number of voices (1-16)
;
align 4
ChMul1 dw 0
ChMul2 dw 0
_sb_set_voices:
        test [codeflags], _init_f
        jz   @@_not_init
        pushad
        call _sb_stopmix

        ;-- Chek range
        cmp  al, 16
        jbe  @@00
        mov  al, 16
@@00:   and  al, al
        jnz  @@01
        mov  al, 1

@@01:   dec  al                         ;0-15
        mov  [ChannelsMinus1], al

        ;-- Uncompress BufXlatTblRLE table to 16 channels
        cld
        mov  esi, ofs BufXlatTblRLE
        mov  edi, [BufXlatTblOfs]
        xor  eax, eax
        xor  ecx, ecx
@@_0:   mov  cx, [esi+eax*2]
        rep  stosb
        inc  al
        jnz  @@_0

        ;-- Modify BufXlatTbl on # channels
        movzx ax, [ChannelsMinus1]
        inc  ax
        mov  [ChMul1], ax
        neg  ax
        add  ax, 16
        mov  [ChMul2], ax
        mov  esi, [BufXlatTblOfs]
        xor  edi, edi
@@05L:  movzx ax, [b ESI+EDI]
        imul ax, [ChMul1]
        mov  ebx, edi
        shr  ebx, 4
        imul bx, [ChMul2]
        add  ax, bx
        shr  ax, 4
        mov  [ESI+EDI], al
        inc  edi
        cmp  edi, 4096
        jne  @@05L


        ;-- Shrink BufXlatTbl to actual # of channels * 255
        movzx ecx, [ChannelsMinus1]
        inc  ecx
        imul ecx, 255
        mov  eax, 4096
        xor  edx, edx
        div  ecx
        mov  ebp, eax
        xor  eax, eax
        div  ecx
        mov  edx, eax
        mov  esi, [BufXlatTblOfs]
        mov  edi, esi
        xor  ebx, ebx
@@__:   mov  al, [esi]
        stosb
        add  ebx, edx
        adc  esi, ebp
        loop @@__

        ;-- Make vol tbls
        mov  edi, [VolumeTblOfs]        ;Low 8 bits MUST be zero
        mov  bp, 15
        xor  ecx, ecx                   ;Vol
        xor  ebx, ebx                   ;Spl
@@10L:  mov  eax, ecx                   ;Vol
        imul bl                         ;Signed Sample
        add  ax, 128*15                 ;MaxVol * 128
        xor  dx, dx
        div  bp                         ;/ MaxVol
        mov  [EDI+EBX], al
        inc  bl
        jnz  @@10L
        add  edi, 256
        inc  cl
        cmp  cl, 16
        jb   @@10L

        or   [codeflags], _voices_f
        call _sb_startmix
        popad
@@_not_init:
        ret

;
; Set beats per second
; In:
;   AL - beats per second (15-255)
; Out:
;   no carry - OK
;   carry    - BPS not set
;
_sb_set_bps:
        test [codeflags], _init_f
        jz   @@_not_init
        push eax
        call _sb_stopmix

        cmp  al, 15                     ;< 15 BPS?
        jnb  @@00
        mov  al, 15
@@00:   mov  [BeatsPerSecond], al

        or   [codeflags], _bps_f
        call _set_TPB
        call _sb_startmix
        pop  eax
@@_not_init:
        ret

;
; In:
;   AX - mixing freq
; Out:
;   AX - real mixing freq
;
_sb_set_mixrate:
        test [codeflags], _init_f
        jz   @@_not_init
        push eax ebx edx
        call _sb_stopmix

        ;-- Force mix freq in range
        cmp  ax, MixRateMAX
        jbe  @@00
        mov  ax, MixRateMAX
@@00:   cmp  ax, MixRateMIN
        jae  @@01
        mov  ax, MixRateMIN

@@01:   movzx ebx, ax
        mov  eax, 1000000
        xor  edx, edx
        div  ebx                        ;1000000/Freq
        and  edx, edx
        jz   @@10
        inc  eax
@@10:   mov  ebx, eax
        mov  eax, 1000000
        xor  edx, edx
        div  ebx
        not  bl                         ;255 - (1000000/Freq)
        mov  [MixFreq], ax

        ;-- Send SB Freq
        mov  al, 40h                    ;Set freq command
        call _sb_out_2xC
        mov  al, bl                     ;sb freq value
        call _sb_out_2xC

        ;-- Find TicksPerIRQ
        mov  ax, [MixFreq]
        xor  dx, dx
        mov  bx, IRQ_Rate
        div  bx
        shr  bx, 1
        cmp  bx, dx                     ;Carry if DX > BX
        adc  ax, 0
        mov  [TicksPerIRQ], ax

        movzx ebx, ax                   ;Ticks per IRQ
        mov  eax, [DMABufOfs]
        add  eax, ebx
        mov  [DMABufOfs+4], eax         ;Start of 2nd buffer
        add  eax, ebx
        mov  [DMABufOfs+8], eax         ;Start of 3rd buffer

        or   [codeflags], _mixrate_f
        call _set_TPB
        call _sb_startmix
        pop  edx ebx eax
        mov  ax, [MixFreq]
@@_not_init:
        ret

;
_set_TPB:
        push ax bx dx
        mov  al, [codeflags]
        and  al, _bps_f OR _mixrate_f
        cmp  al, _bps_f OR _mixrate_f
        jne  @@99

        ;-- MixFreq / BPS = TicksPerBeat
        movzx bx, [BeatsPerSecond]
        mov  ax, [MixFreq]
        xor  dx, dx
        div  bx
        shr  bx, 1
        cmp  bx, dx
        adc  ax, 0
        mov  [TicksPerBeat], ax
@@99:   pop  dx bx ax
        ret

;
; Convert actual frequency to mixer step rate
; In:
;   EAX - frequency
; Out:
;   No Carry    AX      - mixer step rate
;   Carry       none
;
_sb_get_freq:
        test [codeflags], _mixrate_f
        jz   @@99
        push ebx edx
        movzx ebx, [w MixFreq]
        shl  ebx, 4
        cmp  eax, ebx
        jae  @@98                       ;Freq to convert to too high
        shr  ebx, 4
        cmp  eax, PlayFreqMIN
        jae  @@00
        mov  eax, PlayFreqMIN
@@00:   xor  edx, edx
        shl  eax, 12                    ;No bits are lost here..
        div  ebx
        pop  edx ebx
        clc
        ret
@@98:   mov  eax, 0FFFFh                ;Max step rate
        pop  edx ebx
        clc
        ret
@@99:   stc
        ret

;
; Put sample data into sample buffer ram
; In:
;   EBX - addx to put to
;   ECX - length in bytes to put
;   EDX -> sample data to put
;
_sb_put_data:
        push ecx esi edi
        cld

        mov  edi, ebx
        mov  esi, edx
        shr  ecx, 1
        rep  movsw
        adc  ecx, ecx
        rep  movsb

        pop edi esi ecx
        ret

;
; Get sample data from sample buffer ram
; In:
;   EBX - addx to get from
;   ECX - length in bytes to get
;   EDX -> buffer for sample data
;
_sb_get_data:
        push ecx esi edi
        cld

        mov  esi, ebx
        mov  edi, edx
        shr  ecx, 1
        rep  movsw
        adc  ecx, ecx
        rep  movsb

        pop edi esi ecx
        ret


;
;                                   IRQ CODE
;
_IRQreentry:
        call _sb_stopmix
        pop  es ds ebp edi esi ecx ebx edx eax
        iretd
;-------------------------------------------------------------------------------
align 4
_IRQ:   push eax
        mov  al, 20h
        out  20h, al
        sti
        push edx

        mov  dx, [cs:_sb_port_0E]
        in   al, dx
        dec  edx
        dec  edx
@@00L:  in   al, dx
        or   al, al
        js   @@00L
        mov  al, 014h
        out  dx, al
@@01L:  in   al, dx
        or   al, al
        js   @@01L
        mov  ax, [cs:TicksPerIRQ]
        dec  eax
        out  dx, al
@@02L:  in   al, dx
        or   al, al
        js   @@02L
        mov  al, ah
        out  dx, al

        push ebx ecx esi edi ebp ds es
        mov  ds, [cs:_seldata]
        mov  al, -1
        xchg al, [IRQflag]
        and  al, al
        jnz  _IRQreentry
        mov  es, [_seldata]
        cld

      If TimingBar ne 0
        mov  dx, 03C7h
        xor  al, al
        out  dx, al
        inc  dx
        out  dx, al
        inc  dx
        mov  al, 40
        out  dx, al
        out  dx, al
        out  dx, al
      EndIf

        movzx eax, [CurrentPlayBuf]
        mov  al, [eax+NextBufTbl]       ;Next buffer
        mov  [CurrentPlayBuf], al

        mov  al, [CurrentMixBuf]
        mov  edx, [(EAX*4)+DMABufOfs]
        mov  al, [eax+NextBufTbl]
        mov  [CurrentMixBuf], al

        mov  edi, [MixingBufOfs]
        movzx ecx, [TicksPerIRQ]
        call MixBuffer

      If TimingBar ne 0
        mov  dx, 03C7h
        xor  al, al
        out  dx, al
        inc  dx
        out  dx, al
        inc  dx
        out  dx, al
        out  dx, al
        out  dx, al
      EndIf

        mov  [IRQflag], 0
        pop  es ds ebp edi esi ecx ebx edx eax
        iretd
;
;in: edi buffer to mix to
;    ecx length of buffer to mix
align 4
MixBufferLen    dd 0
MixBufferPos    dd 0
MixBufferDMAPos dd 0
MixBuffer:
        mov  [MixBufferPos], edi
        mov  [MixBufferLen], ecx
        mov  [MixBufferDMAPos], edx
align 4
@@00L:  mov  ecx, [MixBufferLen]
        mov  eax, [TicksB4Beat]
        cmp  ecx, eax
        jb   @@_last_mix
        sub  ecx, eax
        mov  [MixBufferLen], ecx
        mov  ecx, eax
        jecxz @@10
        mov  eax, [MixBufferPos]
        mov  edi, edi                   ;MixBufLoc
        add  eax, ecx
        add  eax, ecx
        mov  [MixBufferPos], eax        ;New Pos
        mov  eax, [MixBufferDMAPos]
        mov  edx, eax                   ;MixBufDMA
        add  eax, ecx
        mov  [MixBufferDMAPos], eax     ;New Pos
        call MixChannels
@@10:   movzx eax, [TicksPerBeat]
        mov  [TicksB4Beat], eax
        call [_low_rout]
        push ofs @@00L
        jmp  ProcessChannels
align 4
@@_last_mix:
        jecxz @@11
        sub  eax, ecx
        mov  [TicksB4Beat], eax
        mov  edi, [MixBufferPos]
        mov  edx, [MixBufferDMAPos]
        call MixChannels
@@11:   ret
;
align 4
ProcessChannels:
If Happy ne 0
        movzx esi, [ChannelsMinus1]
@@Happy:mov  edi, esi
        shl  edi, 1
        add  edi, 0B8000h + 24*160
        sub  edi, [_code32a]
        movzx eax, [(esi)+Ch_HappyOld]
        shr  al, 1
        neg  eax
        imul eax, 160
        mov  [w edi+eax], 0Fh SHL 8 + ' '
        movzx eax, [(esi)+Ch_Happy]
        mov  [(esi)+Ch_HappyOld], al
        shr  al, 1
        neg  eax
        imul eax, 160
        mov  [w edi+eax], 0Fh SHL 8 + ''
        mov  al, [(esi)+Ch_Happy]
        dec  al
        js   @@Happy1
        mov  [(esi)+Ch_Happy], al
@@Happy1:
        dec  esi
        jns  @@Happy
EndIf

        movzx esi, [ChannelsMinus1]
@@00L:  xor  cl, cl
        xchg [(esi)+_low_vccmnd], cl
        and  cl, cl
        jz   @@_no_change

        test cl, 08h                    ;Note on?
        jz   @@_chk_vol
        ;-- Set all crap
        ;-- Volume
        movzx eax, [(esi)+_low_vcvol]
        shr  al, 4
;        and  al, 0Fh                    ;0-15 only...
If Happy ne 0
        shl  al, 1
        mov  [(esi)+Ch_Happy], al
        shr  al, 1
EndIf
        mov  eax, [(eax*4)+VolumeTblOfs]
        mov  [(esi*4)+Ch_Vol], eax
        ;-- Chk looping; Zero Ch_LowPos
        movzx eax, [(esi)+_low_vccntrl]
        and  al, 08h                    ;isolate looping bit
        shr  al, 2                      ;shift to bit 1
        mov  [(esi*4)+Ch_LowPos], eax   ;set loop flag  bit 1
                                        ;zero low pos   bits 20-31
        ;-- Sample high posistion
        mov  eax, [(esi*4)+_low_vcsbeg]
        mov  [(esi*4)+Ch_HighPos], eax  ;EAX <- start high
        ;-- Loop pos or sample end pos
        ;EAX <- sample start addx
        mov  ebx, [(esi*4)+_low_vclend] ;last sample played
        inc  ebx                        ;+1
        mov  ecx, ebx
        sub  ebx, eax                   ;end+1 - start = sample length
        jbe  @@_invalid_end             ;end+1 <= start?
        cmp  ebx, SampleSizeMAX         ;8 meg sample max
        ja   @@_invalid_end             ;end+1 > maxsize?
        mov  [(esi*4)+Ch_EndPos], ecx   ;ECX <- EndPos+1
        ;-- Set loop begin
        ;EAX <- sample start addx
        ;ECX <- sample last addx + 1
        mov  ebx, [(esi*4)+_low_vclbeg] ;loop start
        test [b (esi*4)+Ch_LowPos], 02h ;Looping enabled?
        jz   @@_no_looping
        cmp  ebx, eax                   ;loopstart < samplestart?
        jb   @@_invalid_loop
        sub  ecx, ebx                   ;end+1 - loopstart = looplen
        jbe  @@_invalid_loop            ;looplen <= 0?
        cmp  ecx, LoopSizeMIN           ;looplen < LoopSizeMin
        jnb  @@_loop_ok                 ; can also use one signed compare..
@@_invalid_loop:
        and  [b (esi*4)+Ch_LowPos], NOT(02h) ;Disable Looping
        mov  ebx, eax                   ;Set LoopBegin to start of sample
align 4
@@_loop_ok:
@@_no_looping:
        mov  [(esi*4)+Ch_LoopBegin], ebx
        ;-- Set freq
        movzx eax, [(esi*2)+_low_vcfreq]
        cmp  ax, StepRateMIN            ;lowest step 64/4096
        jae  @@10
        mov  ax, StepRateMIN
align 4
@@10:   mov  [(esi*2)+Ch_StepRate], ax
        mov  ebp, eax                   ;EBP <- StepRate
        xor  ebx, ebx
        shrd ebx, eax, 12               ;EBX <- lowstep (bits 20-31)
        shr  eax, 12
        mov  [(esi*4)+Ch_LowStep], ebx
        mov  [(esi)+Ch_HighStep], al
        ;-- Find MixesLeft
        mov  eax, [(esi*4)+Ch_EndPos]
        sub  eax, [(esi*4)+Ch_HighPos]
        xor  edx, edx
        shld edx, eax, 12
        shl  eax, 12
        div  ebp
        and  edx, edx
        jz   @@11
        inc  eax
@@11:   mov  [(esi*4)+Ch_MixesLeft], eax
        jmp  @@_no_change
align 4
@@_invalid_end:
        mov  [Ch_LowPos], 01h           ;Channel Off
        dec  esi
        jns  @@00L
        ret
;-------------
align 4
@@_chk_vol:
        shr  cl, 1
        jnc  @@_chk_bal
        movzx eax, [(esi)+_low_vcvol]
        shr  al, 4                      ;Quick hack 0-255
        mov  eax, [(eax*4)+VolumeTblOfs]
        mov  [(esi*4)+Ch_Vol], eax
;-------------
@@_chk_bal:
        shr  cl, 1
;-------------
@@_chk_freq:
        shr  cl, 1
        jnc  @@_no_change
        movzx eax, [(esi*2)+_low_vcfreq]
        cmp  ax, StepRateMIN
        jae  @@20
        mov  ax, StepRateMIN
@@20:   mov  [(esi*2)+Ch_StepRate], ax
        mov  ebp, eax                   ;StepRate
        xor  ebx, ebx
        shrd ebx, eax, 12
        shr  eax, 12
        mov  [(esi*4)+Ch_LowStep], ebx
        mov  [(esi)+Ch_HighStep], al
        mov  eax, [(esi*4)+Ch_HighPos]
        mov  ebx, [(esi*4)+Ch_LowPos]
        mov  ecx, [(esi*4)+Ch_EndPos]
        call CalcLenToEnd               ;EAX = High     EDX Used
                                        ;EBX = Low
                                        ;ECX = EndPos
                                        ;EBP = StepRate
        mov  [(esi*4)+Ch_MixesLeft], eax
@@_no_change:
        dec  esi
        jns  @@00L
        ret
;
;in: edi buffer to mix to
;    ecx Len
;    edx DMA buffer
align 4
OffCh_128       dw 0                    ;#Off channels * 128
NumMixChannels  db 0
                db 0
MixChannelLen   dd 0
MixChannelPos   dd 0
MixChannelDMAPos dd 0
TempMixLen      dd 0
MixChannels:
        mov  [MixChannelPos], edi
        mov  [MixChannelLen], ecx
        mov  [MixChannelDMAPos], edx

        ;-- Find # of 'off' channels
        mov  esi, ofs Ch_LowPos
        xor  ebx, ebx                   ;#Ch off counter
        movzx edx, [ChannelsMinus1]
        jmp  [(EDX*4)+FindOffCh_Jv]
align 4
Label FindOffCh_Jv dword
irp a,<0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15>
  dd ofs FOC&a
endm
irp a,<15,14,13,12,11,10,9,8,7,6,5,4,3,2,1,0>
FOC&a:  mov  al, [b esi+(a*4)]
        shr  al, 1                      ;ChOff flag in carry
        adc  bl, bh                     ;Inc bl if channel is off
endm
;ebx = # of off channels
;ecx = mix length
;edx = total channels - 1
;edi = mix pos
        sub  edx, ebx                   ;Total Chs - Off Chs = Chs to mix
        jc   NoChToMix
        jz   OneChToMix
        shl  ebx, 7                     ;OffCh * 128
        mov  [OffCh_128], bx
        mov  [NumMixChannels], dl

        ;-- Do first Mix (at least one channel to be mixed..)
;ecx = mix length
;edi = mix pos
        movzx esi, [ChannelsMinus1]     ;scan for first channel
        mov  al, 1
        test [b (esi*4)+Ch_LowPos], al  ;Channel Off?
        jz   @@01                       ;Jmp is channel not off
@@00L:  dec  esi                        ;Next Channel
        test [b (esi*4)+Ch_LowPos], al  ;Channel Off?
        jnz  @@00L
@@01:   mov  eax, [(esi*4)+Ch_MixesLeft]
@@02:   sub  eax, ecx                   ;SMix - ChMix
        jc   @@EndOrLoop
        mov  [(esi*4)+Ch_MixesLeft], eax
        call FMix16
        dec  [NumMixChannels]
        jnz  MiddleMixes
        jmp  LastMix
align 4
@@EndOrLoop:
        mov  ebx, ecx                   ;ChMix
        add  ecx, eax                   ;ChMix + (SMix - ChMix) = SMix
        sub  ebx, ecx                   ;ChMix - SMix = ChMixLeft
        mov  [TempMixLen], ebx
        jecxz @@10                      ;Any SMix?
        call FMix16
@@10:   test [b (esi*4)+Ch_LowPos], 02h ;looping enabled?
        jz   @@EndSample                ;No, go EndSample
        mov  ecx, [(esi*4)+Ch_EndPos]   ;LoopingPos + 1
        mov  eax, [(esi*4)+Ch_HighPos]
        sub  eax, ecx                   ;Current - LoopPos
        add  eax, [(esi*4)+Ch_LoopBegin];Current - LoopPos + LoopBegin
        mov  [(esi*4)+Ch_HighPos], eax  ; = NewPos
        ;-- Find #mixes until EOS
        mov  ebx, [(esi*4)+Ch_LowPos]
        movzx ebp, [(esi*2)+Ch_StepRate]
        sub  ecx, eax                   ;Loop Pos - High Pos
        not  ebx
        add  ebx, 00100000h             ;NEG upper 12 bits
        jc   @@20                       ;Upper 12 bits = 0
        dec  ecx
@@20:   xor  edx, edx
        shld edx, ecx, 12
        shld ecx, ebx, 12
        mov  eax, ecx
        div  ebp
        or   edx, edx
        jz   @@21
        inc  eax
@@21:   mov  [(esi*4)+Ch_MixesLeft], eax
        mov  ecx, [TempMixLen]          ;Remainder of ChMix
        jmp  @@02
align 4
@@EndSample:
        or   [b (esi*4)+Ch_LowPos], 01h ;Set Ch off flag
        mov  ecx, [TempMixLen]
        mov  eax, 00800080h             ;128 in high and low words
        shr  ecx, 1
        rep  stosd
        adc  ecx, ecx
        rep  stosw
        dec  [NumMixChannels]
        jz   LastMix

        ;-- Middle (normal) mixes
align 4
MiddleMixes:
        mov  ecx, [MixChannelLen]
        mov  edi, [MixChannelPos]
        mov  al, 1
@@00L:  dec  esi                        ;Next Channel
        test [b (esi*4)+Ch_LowPos], al  ;Channel Off?
        jnz  @@00L
        mov  eax, [(esi*4)+Ch_MixesLeft]
@@02:   sub  eax, ecx                   ;SMix - ChMix
        jc   @@EndOrLoop
        mov  [(esi*4)+Ch_MixesLeft], eax
        call Mix16
        dec  [NumMixChannels]
        jnz  MiddleMixes
        jmp  LastMix
align 4
@@EndOrLoop:
        mov  ebx, ecx                   ;ChMix
        add  ecx, eax                   ;ChMix + (SMix - ChMix) = SMix
        sub  ebx, ecx                   ;ChMix - SMix = ChMixLeft
        mov  [TempMixLen], ebx
        jecxz @@10                      ;Any SMix?
        call Mix16
@@10:   test [b (esi*4)+Ch_LowPos], 02h ;looping enabled?
        jz   @@EndSample                ;No, go EndSample
        mov  ecx, [(esi*4)+Ch_EndPos]   ;LoopingPos + 1
        mov  eax, [(esi*4)+Ch_HighPos]
        sub  eax, ecx                   ;Current - LoopPos
        add  eax, [(esi*4)+Ch_LoopBegin];Current - LoopPos + LoopBegin
        mov  [(esi*4)+Ch_HighPos], eax  ; = NewPos
        ;-- Find #mixes until EOS
        mov  ebx, [(esi*4)+Ch_LowPos]
        movzx ebp, [(esi*2)+Ch_StepRate]
        sub  ecx, eax                   ;Loop Pos - High Pos
        not  ebx
        add  ebx, 00100000h             ;NEG upper 12 bits
        jc   @@20                       ;Upper 12 bits = 0
        dec  ecx
@@20:   xor  edx, edx
        shld edx, ecx, 12
        shld ecx, ebx, 12
        mov  eax, ecx
        div  ebp
        or   edx, edx
        jz   @@21
        inc  eax
@@21:   mov  [(esi*4)+Ch_MixesLeft], eax
        mov  ecx, [TempMixLen]          ;Remainder of ChMix
        jmp  @@02
align 4
@@EndSample:
        or   [b (esi*4)+Ch_LowPos], 01h ;Set Ch off flag
        mov  ecx, [TempMixLen]
        mov  ax, 128
        call AddAXToBuffer
        dec  [NumMixChannels]
        jnz  MiddleMixes


        ;-- Final mix and xlate to 8bits
align 4
LastMix:
        ;-- Last mix SMC - adds OffCh_128 to output.
        movzx eax, [OffCh_128]
        add  eax, [BufXlatTblOfs]
irp a,<32,31,30,29,28,27,26,25,24,23,22,21,20,19,18,17,16,15,14,13,12,11,10,9,8,7,6,5,4,3,2,1>
        mov  [d LMsmc&a], eax
endm
        mov  ecx, [MixChannelLen]
        mov  edi, [MixChannelPos]
        mov  al, 1
@@00L:  dec  esi                        ;Next Channel
        test [b (esi*4)+Ch_LowPos], al  ;Channel Off?
        jnz  @@00L
        mov  eax, [(esi*4)+Ch_MixesLeft]
@@02:   sub  eax, ecx                   ;SMix - ChMix
        jc   @@EndOrLoop
        mov  [(esi*4)+Ch_MixesLeft], eax
        jmp  LMix16                     ;Call and RET
align 4
@@EndOrLoop:
        mov  ebx, ecx                   ;ChMix
        add  ecx, eax                   ;ChMix + (SMix - ChMix) = SMix
        sub  ebx, ecx                   ;ChMix - SMix = ChMixLeft
        mov  [TempMixLen], ebx
        jecxz @@10                      ;Any SMix?
        call LMix16
@@10:   test [b (esi*4)+Ch_LowPos], 02h ;looping enabled?
        jz   @@EndSample                ;No, go EndSample
        mov  ecx, [(esi*4)+Ch_EndPos]   ;LoopingPos + 1
        mov  eax, [(esi*4)+Ch_HighPos]
        sub  eax, ecx                   ;Current - LoopPos
        add  eax, [(esi*4)+Ch_LoopBegin];Current - LoopPos + LoopBegin
        mov  [(esi*4)+Ch_HighPos], eax  ; = NewPos
        ;-- Find #mixes until EOS
        mov  ebx, [(esi*4)+Ch_LowPos]
        movzx ebp, [(esi*2)+Ch_StepRate]
        sub  ecx, eax                   ;Loop Pos - High Pos
        not  ebx
        add  ebx, 00100000h             ;NEG upper 12 bits
        jc   @@20                       ;Upper 12 bits = 0
        dec  ecx
@@20:   xor  edx, edx
        shld edx, ecx, 12
        shld ecx, ebx, 12
        mov  eax, ecx
        div  ebp
        or   edx, edx
        jz   @@21
        inc  eax
@@21:   mov  [(esi*4)+Ch_MixesLeft], eax
        mov  ecx, [TempMixLen]          ;Remainder of ChMix
        jmp  @@02
align 4
@@EndSample:
        or   [b (esi*4)+Ch_LowPos], 01h ;Set Ch off flag
        xor  eax, eax
        movzx ebx, [OffCh_128]          ;Other off channels
        add  ebx, [BufXlatTblOfs]       ;+Ofs of XlatTbl
        add  ebx, 128                   ;+This channel being off
        mov  ecx, [TempMixLen]
        mov  edx, ecx
        shr  ecx, 5
        and  edx, 1Fh
        jz   @@30
        inc  ecx
@@30:   mov  esi, edi                   ;MixBufPtr
        mov  edi, [MixChannelDMAPos]
        jmp  [(edx*4)+XltJV]
align 4
Label XltJV dword
irp a,<32,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31>
        dd ofs Xlt&a
endm
XltLoop:
irp a,<32,31,30,29,28,27,26,25,24,23,22,21,20,19,18,17,16,15,14,13,12,11,10,9,8,7,6,5,4,3,2,1>
Xlt&a:  lodsw
        mov  al, [EAX+EBX]
        stosb
EndM
        dec  ecx
        jnz  XltLoop
        ret
align 4
OneChToMix:
;ebx = # of off channels
        shl  ebx, 7                     ;OffCh * 128
        mov  [OffCh_128], bx
        xor  eax, eax
        mov  ecx, [MixChannelLen]
        mov  edi, [MixChannelPos]
        shr  ecx, 1
        rep  stosd
        adc  ecx, ecx
        rep  stosw
        movzx esi, [ChannelsMinus1]
        inc  esi
        jmp  LastMix
align 4
NoChToMix:
      If TimingBar ne 0
        mov  dx, 03C7h
        xor  al, al
        out  dx, al
        inc  dx
        out  dx, al
        inc  dx
        mov  al, 60
        out  dx, al
        xor  al, al
        out  dx, al
        out  dx, al
      EndIf
        mov  edi, [MixChannelDMAPos]
        mov  ecx, [MixChannelLen]
        mov  ax, 8080h
        shr  ecx, 1
        rep  stosw
        adc  ecx, ecx
        rep  stosb
        ret
;
; In:
;  EAX - HighPos
;  EBX - LowPos (can have flag bits on..)
;  ECX - EndPos
;  EBP - StepRate (upper 16 bits off)
; Out:
;  EAX - Samples until end
; Uses:
;  EAX EBX ECX EDX EBP
; UnChanged:
;  ESI EDI EBP
;
align 4
CalcLenToEnd:
        sub  ecx, eax                   ;Loop Pos - High Pos
        not  ebx
        add  ebx, 00100000h             ;NEG upper 12 bits
        jc   @@00                       ;Upper 12 bits = 0
        dec  ecx
@@00:   xor  edx, edx
        shld edx, ecx, 12
        shld ecx, ebx, 12
        mov  eax, ecx
        div  ebp
        and  edx, edx
        jz   @@01
        inc  eax
@@01:   ret
;
align 4
LMix_LowStep    dd 0
Label LM16Entries dword
irp a,<32,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31>
        dd ofs LM16&a
endm
LM16_Loop:
irp a,<32,31,30,29,28,27,26,25,24,23,22,21,20,19,18,17,16,15,14,13,12,11,10,9,8,7,6,5,4,3,2,1>
LM16&a: mov  bl, [b esi]                ;Grab sample
        movzx eax, [b ebx]              ;Volume Xlat and clear ah
        add  ax, [edx+(32-a)*2]         ;Add what's already in buffer
        mov  al, [eax+01234567h]        ;16bit -> 8bit conversion + (#ChOff*128)
LMsmc&a = $-4                           ;via SMC
        stosb                           ;store final sample in play buffer
        add  ecx, [LMix_LowStep]        ;add low step
        adc  esi, ebp                   ;add high step with carry from low
EndM
        add  edx, 32*2                  ;Adjust MixBuffer ptr
        dec  cl                         ;Any more rounds?
        jnz  LM16_Loop
        mov  [MixChannelDMAPos], edi    ;DMABufPtr
        mov  edi, edx                   ;MixBufPtr
        pop  eax                        ;Channel#
        shr  ecx, 16
        mov  [w (eax*4)+Ch_LowPos+2], cx;Don't touch flags
        mov  [(eax*4)+Ch_HighPos], esi
        mov  esi, eax
        ret
;ECX - Number of mixes (max = 8191)
;ESI - Channel#
;EDI - 16 bit buffer
align 4
LMix16: push esi
        mov  edx, edi                   ;16 bit buffer pointer
        mov  eax, ecx
        shr  ecx, 5                     ;#mixes / 32
        and  eax, 1Fh                   ;#mixes MOD 32
        jz   @@00                       ;MOD 32=0? -> jmps to top of loop
        inc  cl
        sub  edx, 32*2                  ;Adjust MixBufPtr for jmp to inline code
        add  edx, eax
        add  edx, eax
@@00:   push [(eax*4)+LM16Entries]      ;Addx of JV on stack
        mov  eax, [(esi*4)+Ch_LowPos]
        xor  al, al                     ;Nuke flags...
        or   ecx, eax                   ;LowPos in upper 12 bits, LoopCount in CL
        mov  ebx, [(esi*4)+Ch_Vol]
        mov  eax, [(esi*4)+Ch_LowStep]
        mov  [LMix_LowStep], eax
        movzx ebp,[(esi)+Ch_HighStep]
        mov  esi, [(esi*4)+Ch_HighPos]
        mov  edi, [MixChannelDMAPos]
        ret
;
align 4
Label FM16Entries dword
irp a,<32,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31>
        dd ofs FM16&a
endm
FM16_Loop:
irp a,<32,31,30,29,28,27,26,25,24,23,22,21,20,19,18,17,16,15,14,13,12,11,10,9,8,7,6,5,4,3,2,1>
FM16&a: mov  bl, [b esi]
        mov  al, [b ebx]
        stosw
        add  ecx, edx
        adc  esi, ebp
EndM
        dec  cl
        jnz  FM16_Loop
        pop  eax
        shr  ecx, 16
        mov  [w (eax*4)+Ch_LowPos+2], cx  ;Don't affect flags, write upper 12 bits
        mov  [(eax*4)+Ch_HighPos], esi
        mov  esi, eax
        ret
;ECX - Number of mixes (max = 8191)
;ESI - Channel#
;EDI - 16 bit buffer to mix to
align 4
FMix16: push esi
        mov  eax, ecx
        shr  ecx, 5
        and  eax, 1Fh
        jz   @@00
        inc  cl
@@00:   push [(eax*4)+FM16Entries]
        mov  eax, [(esi*4)+Ch_LowPos]
        xor  al, al
        or   ecx, eax
        mov  ebx, [(esi*4)+Ch_Vol]
        mov  edx, [(esi*4)+Ch_LowStep]
        movzx ebp,[(esi)+Ch_HighStep]
        mov  esi, [(esi*4)+Ch_HighPos]
        ret
;
align 4
Label M16Entries dword
irp a,<32,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31>
        dd ofs M16&a
endm
M16_loop:
irp a,<32,31,30,29,28,27,26,25,24,23,22,21,20,19,18,17,16,15,14,13,12,11,10,9,8,7,6,5,4,3,2,1>
M16&a:  mov  bl, [b esi]          ;Signed sample
        mov  al, [b ebx]
        add  [w edi+(32-a)*2], ax ;Add to unsigned buffer
        add  ecx, edx
        adc  esi, ebp
EndM
        add  edi, 32*2
        dec  cl
        jnz  M16_loop
        pop  eax
        shr  ecx, 16
        mov  [w (eax*4)+Ch_LowPos+2], cx  ;Don't affect flags, write upper 12 bits
        mov  [(eax*4)+Ch_HighPos], esi
        mov  esi, eax
        ret
;ECX - Number of mixes 0 < x < 8160
;ESI - Channel#
;EDI - 16bit mix buffer
align 4
Mix16:  push esi                ;Save channel#
        mov  eax, ecx
        shr  ecx, 5             ;#mixes / 32
        and  eax, 1Fh           ;#mixes MOD 32
        jz   @@00               ;multiple of 32, will jump to top of loop
        inc  cl
        sub  edi, 32*2
        add  edi, eax
        add  edi, eax
@@00:   push [(eax*4)+M16Entries]
        mov  eax, [(esi*4)+Ch_LowPos]
        xor  al, al             ;Nuke flags...
        or   ecx, eax           ;LowPos in upper 12 bits, LoopCount in CL
        mov  ebx, [(esi*4)+Ch_Vol]
      If Debug ne 0
          and  bl, bl
          jz   @@D00
          mov  edx, ofs @@D00m
          call _putdosmsg
          pop  eax
          pop  eax
          ret
        @@D00m db 'BL not zero in Mix16',13,10,'$'
        @@D00:
      EndIf
        mov  edx, [(esi*4)+Ch_LowStep]
        movzx ebp,[(esi)+Ch_HighStep]
        mov  esi, [(esi*4)+Ch_HighPos]
        xor  eax, eax
        ret
;
;- - - - Change to add [esi+x], eax !!! (much faster)
; test  edi, 02h        ;align 4?
; jz    @@00            ;yes
; add   [w edi], ax     ;no
; inc   edi
; inc   edi
; dec   ecx
; jz    done
; @@00:  ...
align 4
Label AddAXEntries dword
irp a,<32,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31>
        dd ofs ADDAX&a
endm
align 4
@AddAXLoop:
irp a,<32,31,30,29,28,27,26,25,24,23,22,21,20,19,18,17,16,15,14,13,12,11,10,9,8,7,6,5,4,3,2,1>
ADDAX&a:add  [edi+(32-a)*2*2], eax
endm
        add  edi, 32*2*2
        dec  ecx
        jnz  @AddAXLoop
        ret
align 4
AddAXToBuffer:
        mov  ebx, eax
        shl  eax, 16
        mov  ax, bx                     ;Get add value in low and high words
        shr  ecx, 1
        jz   @@10                       ;Only one sample to add?
        jnc  @@00
        add  [w edi], ax
        inc  edi
        inc  edi
@@00:   mov  ebx, ecx
        shr  ecx, 5
        and  ebx, 1Fh
        jz   @@01
        inc  ecx
        sub  edi, 32*2*2
        shl  ebx, 2
        add  edi, ebx
        jmp  [(ebx)+AddAXEntries]
align 4
@@01:   jmp  [(ebx*4)+AddAXEntries]
align 4
@@10:   add  [w edi], ax
        inc  edi
        inc  edi
        ret
;
; Low freq mix routine parts


;
align 4
_DummyIrq:
        push ax
        mov  al, 20h
        out  20h, al
        pop  ax
        iretd
;
_sb_reset_card:
        push ax ecx dx
        mov  dx, [_sb_port_06]          ;2x6h
        mov  al, 1
        out  dx, al
        in   al, dx
        in   al, dx
        in   al, dx
        in   al, dx
        xor  al, al
        out  dx, al
        add  dx, 4                      ;2xAh
        mov  ecx, 200
@@00L:  in   al, dx
        cmp  al, 0AAh
        je   @@01
        loop @@00L
        pop  dx ecx ax
        stc
        ret
@@01:   pop  dx ecx ax
        clc
        ret
;
_sb_out_2xC:
        push ax dx
        mov  ah, al
        mov  dx, [_sb_port_0C]
@@00L:  in   al, dx
        or   al, al
        js   @@00L
        mov  al, ah
        out  dx, al
        pop  dx ax
        ret
;
EndS    code32
End
