UNIT StringDateHandling;
{$D-,F+} (* I'am using procedural variables! *)
(* ----------------------------------------------------------------------
   Part of 4DESC - A Simple 4DOS File Description Editor
       and 4FF   - 4DOS File Finder

       David Frey,         & Tom Bowden
       Urdorferstrasse 30    1575 Canberra Drive
       8952 Schlieren ZH     Stone Mountain, GA 30088-3629
       Switzerland           USA

       Code created using Turbo Pascal 7.0, (c) Borland International 1992

   DISCLAIMER: This unit is freeware: you are allowed to use, copy
               and change it free of charge, but you may not sell or hire
               this part of 4DESC. The copyright remains in our hands.

               If you make any (considerable) changes to the source code,
               please let us know. (send a copy or a listing).
               We would like to see what you have done.

               We, David Frey and Tom Bowden, the authors, provide absolutely
               no warranty of any kind. The user of this software takes the
               entire risk of damages, failures, data losses or other
               incidents.


       Code created using Turbo Pascal 6.0 (c) Borland International 1990

   This unit provides the string handling and the date/time handling.

   ----------------------------------------------------------------------- *)

INTERFACE USES Dos;

TYPE  DateStr    = STRING[8];  (* 'mm-dd-yy','dd.mm.yy' or 'yy/mm/dd' *)
      TimeStr    = STRING[6];  (* 'hh:mmp' or 'hh:mm'                 *)

VAR   DateFormat: DateStr; (* 'mm-dd-yy','dd.mm.yy','yy/mm/dd' or 'ddmmmyy' *)
      TimeFormat: TimeStr; (* 'hh:mmp' or 'hh:mm'                           *)

(* String handling routines. The strings can be converted to lower/upper-
   case. National characters will be converted.                           *)

FUNCTION  Chars(c: CHAR; Count: BYTE): STRING;
FUNCTION  DownCase(C: CHAR): CHAR;
FUNCTION  DownStr(s: STRING): STRING;
PROCEDURE DownString(VAR s: STRING);
FUNCTION  UpStr(s: STRING): STRING;
PROCEDURE UpString(VAR s: STRING);

PROCEDURE StripLeadingSpaces(VAR s: STRING);
PROCEDURE StripTrailingSpaces(VAR s: STRING);

(* Date/Time handling routines. Date/Time and Numbers will be formatted
   in accordance with your COUNTRY=-settings in CONFIG.SYS.               *)

TYPE  FormDateFunc = FUNCTION (DateRec: DateTime) : DateStr;
      FormTimeFunc = FUNCTION (DateRec: DateTime) : TimeStr;

VAR   FormDate : FormDateFunc;
      FormTime : FormTimeFunc;


FUNCTION FormattedIntStr(nr: WORD;minlength: BYTE): STRING;
FUNCTION FormattedLongIntStr(nr: LONGINT;minlength: BYTE): STRING;

IMPLEMENTATION USES HandleINIFile;

CONST MonthName: ARRAY[1..12] OF STRING[3] =
                  ('Jan','Feb','Mar','Apr','May','Jun',
                   'Jul','Aug','Sep','Oct','Nov','Dec');

CONST DateSep  : CHAR = '.';
      TimeSep  : CHAR = ':';
      MilleSep : CHAR = '''';

VAR   Buffer: ARRAY[0..15] OF CHAR;
      (* Buffer for country code information.
         This buffer may not be moved into GetCountryInfo,
         since MS-DOS needs the address of this buffer!    *)

(*-------------------------------------------------------- String-Handling *)
FUNCTION Chars(c: CHAR; Count: BYTE): STRING; ASSEMBLER;
(* Concats Count times the character c *)

ASM
 LES DI,@Result
 MOV AL,&Count
 CLD
 STOSB
 MOV CL,AL
 XOR CH,CH
 MOV AL,&c
 REP STOSB
END;

FUNCTION  DownCase(C: CHAR): CHAR; ASSEMBLER;
(* Returns the character c in lower case, national characters will not
   be handled correctly. [we will use this function to lowercase file
   names and DOS doesn't like special characters in filenames anyway] *)

ASM
  MOV AL,&c
  CMP AL,'A'
  JB  @@9                  (* No conversion below 'A'                     *)
  CMP AL,'Z'
  JA  @@9                  (* Conversion between 'A' and 'Z'              *)
  ADD AL,$20
@@9:
END;                       (* finished. *)

FUNCTION  DownStr(s: STRING): STRING; ASSEMBLER;
(* Returns the string s in lower case, national characters will not
   be handled correctly. [we will use this function to lowercase file
   names and DOS doesn't like special characters in filenames anyway] *)

ASM
 PUSH DS
 CLD
 LDS SI,s
 LES DI,@Result
 LODSB
 STOSB
 XOR AH,AH
 XCHG AX,CX
 JCXZ @11
@10:
 LODSB
 CMP AL,'A'
 JB  @@9                  (* No conversion below 'A'                     *)
 CMP AL,'Z'
 JA  @@9                  (* Conversion between 'A' and 'Z'              *)
 ADD AL,$20
@@9:
 STOSB
 LOOP @10
@11:
 POP DS
END;


PROCEDURE DownString(VAR s: STRING);
(* Returns the string s in lower case, national characters will not
   be handled correctly. [we will use this function to lowercase file
   names and DOS doesn't like special characters in filenames anyway] *)

VAR i : BYTE;

BEGIN
 FOR i := 1 TO Length(s) DO s[i] := DownCase(s[i]);
END;


FUNCTION  UpStr(s: STRING): STRING; ASSEMBLER;
(* Returns the string s in upper case, national characters will not
   be handled correctly.                                              *)

ASM
 PUSH DS
 CLD
 LDS SI,s
 LES DI,@Result
 LODSB
 STOSB
 XOR AH,AH
 XCHG AX,CX
 JCXZ @11
@10:
 LODSB
 CMP AL,'a'
 JB @@9
 CMP AL,'z'
 JA @@9
 SUB AL,20H
@@9:
 STOSB
 LOOP @10
@11:
 POP DS
END;

PROCEDURE UpString(VAR s: STRING);
(* Returns the string s in upper case, national characters will not
   be handled correctly.                                              *)

VAR l : BYTE;

BEGIN
 FOR l := 1 TO Length(s) DO s[l] := UpCase(s[l]);
END;

PROCEDURE StripLeadingSpaces(VAR s: STRING);

BEGIN
 WHILE (Length(s) > 0) AND (s[1] = ' ') DO Delete(s,1,1);
END;

PROCEDURE StripTrailingSpaces(VAR s: STRING);

VAR l : BYTE;

BEGIN
 l := Length(s);
 WHILE (l>0) AND (s[l] = ' ') DO BEGIN Delete(s,l,1); l := Length(s); END;
END;

(*-------------------------------------------------------- Date-Handling *)

(* Various Date/Time format utilities to suit national date/time formats *)

FUNCTION FormDateEuropean(DateRec: DateTime): DateStr;

VAR MonStr, DayStr, YearStr : STRING[2];
    res                     : DateStr;

BEGIN
 Str(DateRec.Day:2, DayStr);

 Str(DateRec.Month:2, MonStr);
 IF DateRec.Month < 10 THEN MonStr[1] := '0';

 DateRec.Year := DateRec.Year MOD 100;
 Str(DateRec.Year:2, YearStr);
 IF DateRec.Year < 10 THEN YearStr[1] := '0';

 FormDateEuropean := DayStr + DateSep + MonStr + DateSep + YearStr;
END;

FUNCTION FormDateUS(DateRec: DateTime): DateStr;

VAR MonStr, DayStr, YearStr : STRING[2];
    res                     : DateStr;

BEGIN
 Str(DateRec.Day:2, DayStr);
 IF DateRec.Day < 10 THEN DayStr[1] := '0';

 Str(DateRec.Month:2, MonStr);

 DateRec.Year := DateRec.Year MOD 100;
 Str(DateRec.Year:2, YearStr);
 IF DateRec.Year < 10 THEN YearStr[1] := '0';

 FormDateUS := MonStr + DateSep + DayStr + DateSep + YearStr;
END;

FUNCTION FormDateJapanese(DateRec: DateTime): DateStr;

VAR MonStr, DayStr, YearStr : STRING[2];
    res                     : DateStr;

BEGIN
 Str(DateRec.Day:2, DayStr);
 IF (DateRec.Day < 10) THEN DayStr[1] := '0';

 Str(DateRec.Month:2, MonStr);
 IF (DateRec.Month < 10) THEN MonStr[1] := '0';

 DateRec.Year := DateRec.Year MOD 100;
 Str(DateRec.Year:2, YearStr);
 IF DateRec.Year < 10 THEN YearStr[1] := '0';

 FormDateJapanese := YearStr + DateSep + MonStr + DateSep + DayStr;
END;

FUNCTION FormDateMyOwn(DateRec: DateTime): DateStr;

VAR DayStr, YearStr : STRING[2];
    res             : DateStr;

BEGIN
 Str(DateRec.Day:2, DayStr);

 DateRec.Year := DateRec.Year MOD 100;
 Str(DateRec.Year:2, YearStr);
 IF DateRec.Year < 10 THEN YearStr[1] := '0';

 FormDateMyOwn := DayStr + MonthName[DateRec.Month] + YearStr;
END;

FUNCTION FormTime12(DateRec: DateTime): TimeStr;

VAR HourStr, MinStr, SecStr : STRING[2];
    amflag                  : CHAR;
    res                     : TimeStr;

BEGIN
 IF DateRec.Hour < 12 THEN amflag := 'a'
                      ELSE BEGIN amflag := 'p'; DEC(DateRec.Hour,12); END;
 Str(DateRec.Hour:2,HourStr);
 Str(DateRec.Min :2,MinStr ); IF DateRec.Min < 10 THEN MinStr[1] := '0';
 Str(DateRec.Sec :2,SecStr ); IF DateRec.Sec < 10 THEN SecStr[1] := '0';

 FormTime12 := HourStr + TimeSep + MinStr + amflag;
END;

FUNCTION FormTime24(DateRec: DateTime): TimeStr;

VAR HourStr, MinStr, SecStr : STRING[2];
    res                     : TimeStr;

BEGIN
 Str(DateRec.Hour:2,HourStr);
 Str(DateRec.Min :2,MinStr ); IF DateRec.Min < 10 THEN MinStr[1] := '0';
 Str(DateRec.Sec :2,SecStr ); IF DateRec.Sec < 10 THEN SecStr[1] := '0';

 FormTime24 := HourStr + TimeSep + MinStr;
END;

(*------------------------------------------------ Formatting of numbers *)

FUNCTION FormattedIntStr(nr: WORD;minlength: BYTE): STRING;
(* Converts an integer number into a string of the form xxx'xxx...') *)

VAR helpstr  : STRING;
    millestr : STRING[4];
    n,i      : BYTE;

BEGIN
 IF nr = 0 THEN FormattedIntStr := Chars(' ',minlength-1)+'0'
 ELSE
  BEGIN
   helpstr := '';
   n := nr DIV 1000; nr := nr MOD 1000;
   IF n > 0 THEN
    BEGIN
     Str(n,helpstr);
     helpstr := millestr+helpstr+MilleSep;
    END;

   IF n = 0 THEN Str(nr,millestr)
   ELSE
    BEGIN
     Str(nr:3,millestr);
     FOR i := 1 TO 3 DO IF millestr[i] = ' ' THEN millestr[i] := '0';
    END;
   helpstr:=helpstr+millestr;
   n := Length(helpstr);
   IF n < minlength THEN helpstr := Chars(' ',minlength-n)+helpstr;

   FormattedIntStr := helpstr;
  END;
END;

FUNCTION FormattedLongIntStr(nr: LONGINT;minlength: BYTE): STRING;
(* Converts a long integer number into a string of the form xxx'xxx...') *)

VAR helpstr  : STRING;
    millestr : STRING[4];
    n,i      : WORD;

BEGIN
 IF nr = 0 THEN FormattedLongIntStr := Chars(' ',minlength-1)+'0'
 ELSE
  BEGIN
   helpstr := '';

   n := nr DIV 1000000; nr := nr MOD 1000000;
   IF n > 0 THEN
    BEGIN
     Str(n,millestr); helpstr := millestr+MilleSep;
    END;

   n := nr DIV 1000; nr := nr MOD 1000;
   IF n > 0 THEN
    BEGIN
     Str(n:3,millestr);
     IF helpstr > '' THEN
      BEGIN
       FOR i := 1 TO 3 DO IF millestr[i] = ' ' THEN millestr[i] := '0';
       helpstr := helpstr+millestr+MilleSep;
      END
     ELSE helpstr := millestr+MilleSep;
    END;

   IF n = 0 THEN Str(nr,millestr)
   ELSE
    BEGIN
     Str(nr:3,millestr);
     FOR i := 1 TO 3 DO IF millestr[i] = ' ' THEN millestr[i] := '0';
    END;
   helpstr:=helpstr+millestr;
   n := Length(helpstr);
   IF n < minlength THEN helpstr := Chars(' ',minlength-n)+helpstr;

   FormattedLongIntStr := helpstr;
  END;
END;

(*------------------------------------------------------- Initialisation *)

PROCEDURE GetCountryInfo;

VAR Regs  : Registers;

BEGIN
 WITH Regs DO
  BEGIN
   ah := $38; (* Get / Set Country Data *)
   al := $00;
   ds := Seg(Buffer); dx := Ofs(Buffer); (* Address of Buffer *)
  END;
 MsDos(Regs);

 IF Regs.Flags AND FCarry = 0 THEN
  BEGIN
   MilleSep := Buffer[ 7];
   DateSep  := Buffer[11];
   TimeSep  := Buffer[13];
  END;

 CASE Ord(Buffer[0]) OF
  0 : BEGIN
       FormDate := FormDateUS;       DateFormat := 'mm'+DateSep+'dd'+DateSep+'yy';
       FormTime := FormTime12;       TimeFormat := 'hh'+TimeSep+'mmp';
      END;
  1 : BEGIN
       FormDate := FormDateEuropean; DateFormat := 'dd'+DateSep+'mm'+DateSep+'yy';
       FormTime := FormTime24;       TimeFormat := 'hh'+TimeSep+'mm';
      END;
  2 : BEGIN
       FormDate := FormDateJapanese; DateFormat := 'yy'+DateSep+'mm'+DateSep+'dd';
       FormTime := FormTime24;       TimeFormat := 'hh'+TimeSep+'mm';
      END;
 ELSE
  BEGIN
   FormDate := FormDateEuropean;     DateFormat := 'dd'+DateSep+'mm'+DateSep+'yy';
   FormTime := FormTime24;           TimeFormat := 'hh'+TimeSep+'mm';
  END;
 END; (* CASE *)
END;

PROCEDURE EvaluateINIFileSettings;

VAR s : STRING[7];

BEGIN
 IF INIFileExists THEN
  BEGIN
   MilleSep := ReadSettingsChar('date & time formats','millesep',MilleSep);
   TimeSep  := ReadSettingsChar('date & time formats','timesep' ,TimeSep);
   DateSep  := ReadSettingsChar('date & time formats','datesep' ,DateSep);

   s := ReadSettingsString('date & time formats','dateformat','ddmmmyy');
   IF s = 'ddmmyy' THEN
    BEGIN
     FormDate := FormDateEuropean; DateFormat := 'dd'+DateSep+'mm'+DateSep+'yy';
    END
   ELSE
   IF s = 'mmddyy' THEN
    BEGIN
     FormDate := FormDateUS;       DateFormat := 'mm'+DateSep+'dd'+DateSep+'yy';
    END
   ELSE
   IF s = 'yymmdd' THEN
    BEGIN
     FormDate := FormDateJapanese; DateFormat := 'yy'+DateSep+'mm'+DateSep+'dd';
    END
   ELSE
    BEGIN
     FormDate := FormDateMyOwn;    DateFormat := 'ddmmmyy';
    END;

   s := ReadSettingsString('date & time formats','timeformat','24');
   IF s = '12' THEN
    BEGIN
     FormTime := FormTime12; TimeFormat := 'hh'+TimeSep+'mmp';
    END
   ELSE
    BEGIN
     FormTime := FormTime24; TimeFormat := 'hh'+TimeSep+'mm';
    END;
  END;
END;

BEGIN
 GetCountryInfo;
 IF INIFileExists THEN EvaluateINIFileSettings;
END.
