/*
 *  Settings.c
 *      Source for Pundit (V 2.0)
 *      Settings dialog box, etc.
 *
 *  Author:         Jeff Bienstadt
 *
 *  Environment:
 *
 *      Run-Time:   Microsoft Windows 3.0
 *
 *      Compilers/Tools:
 *                  Microsoft C 6.0
 *                  Microsoft Windows SDK 3.0
 *
 */

#include    <windows.h>     // All the Windows goodies
#include    <stdio.h>       // for sscanf()
#include    <string.h>      // for strlen(), etc.
#include    "pundit.h"      // constants for Pundit
#include    "extras.h"      // for WriteProfileInt(), etc.

static void     HandleColorScroll(WORD, LONG);
static void     InitSettings(HWND);
static void     NewBackground(void);
static void     SaveSettings(HWND);
static void     SaveSizeAndPosition(RECT *);

extern HWND     Pundit;         // a global Window Handle (for main window)
extern HDC      hdc;            // a global Display Context Handle
extern HANDLE   hOrgBrush;      // a global Brush handle
extern SETTINGS settings;       // the Settings storage structure
extern char     szAppName[],    // The Application Name
                szProfName[];   // The name of the Profile file
extern int      category,       // what category are we using now?
                hide_mode;      // when we go away, do we hide or do we iconize?

static HWND     hInScroll,      // Handle for IN scrollbar
                hOutScroll,     // Handle for OUT scrollbar
                hRScroll,       // Handle for RED scrollbar
                hGScroll,       // Handle for GREEN scrollbar
                hBScroll;       // Handle for BLUE scrollbar

static short    fRed,
                fGreen,
                fBlue,
                bRed,
                bGreen,
                bBlue,
                foreground;     // processing foreground or background color?

static SETTINGS s;              // a temporary values for settings

static char     text[15];       // a string to write text into


BOOL FAR PASCAL Settings(hDlg, message, wParam, lParam)
HWND        hDlg;       // Window Handle for our Dialog Box
unsigned    message;    // The Window's Message
WORD        wParam;     // The WORD parameter value
LONG        lParam;     // The LONG parameter value
{
    HWND    hsb;        // Handle for a Horizontal Scroll Bar
    char    temp[10];   // string to write "numbers" into

    switch (message) {      // check for messages...
    case    WM_INITDIALOG:      // If the Dialog is just starting...
        PostMessage(Pundit, PM_PAUSE, 0, 0L);   // shut down the timer
        InitSettings(hDlg);     // Initialize the Settings Dialog
        return TRUE;

    case    WM_PAINT:               // if dialog is painted
        PostMessage(hDlg, PM_DLGPAINT, 0, 0L);
        return FALSE;

    case    PM_DLGPAINT:
        SetTextColor(hdc, RGB(fRed, fGreen, fBlue));
        SetBkColor(hdc, RGB(bRed, bGreen, bBlue));

        TextOut(hdc, 0, 0, (LPSTR)text, strlen(text));
        return TRUE;

    case    WM_HSCROLL:         // We got a Horizontal Scroll Bar Message...
        if (hsb = HIWORD(lParam)) { // if it's not zero
            if (hsb == hRScroll || hsb == hGScroll || hsb == hBScroll)
                HandleColorScroll(wParam, lParam);  // do the colors
            else if (hsb == hInScroll) { // If it's from the IN Scroll Bar
                switch (wParam) {       // find out what happened...
                case    SB_PAGEUP:          // User did a PAGE UP
                    settings.insecs -= 4;   // take off 4 seconds and...
                case    SB_LINEUP:          // User did a LINE UP
                        // take off 1 second or set to 1, whichever is greater
                    settings.insecs = max(1, settings.insecs-1);
                    break;
                case    SB_PAGEDOWN:        // User did a PAGE DOWN
                    settings.insecs += 4;   // add 4 seconds and...
                case    SB_LINEDOWN:        // User did a LINE DOWN
                        // add 1 second or set to 1 minute, whichever is lesser
                    settings.insecs = min(MAX_IN, settings.insecs+1);
                    break;
                case    SB_TOP:             // User went to the TOP
                    settings.insecs = 1;    // set to 1 second
                    break;
                case    SB_BOTTOM:          // User went the to BOTTOM
                    settings.insecs = MAX_IN;   // set to 1 minute
                    break;
                case    SB_THUMBPOSITION:   // User moved...
                case    SB_THUMBTRACK:      // or is moving the THUMB BUTTON
                        // Windows tells us how far...
                    settings.insecs = LOWORD(lParam);
                default:
                    break;
                }
                // Put the new value into a string
                wsprintf(temp, "%4d", settings.insecs);
                // Display it in the box next to the Scroll Bar
                SetDlgItemText(hDlg, IDC_INSECS, temp);
                // Set the new Scroll Bar position
                SetScrollPos(hsb, SB_CTL, settings.insecs, TRUE);
            }
            else if (hsb == hOutScroll) {   // If it's from the OUT Scroll Bar
                switch (wParam) {       // find out what happened...
                case    SB_PAGEUP:          // User did a PAGE UP
                    settings.outsecs -= 559;   // take off 10 min and...
                case    SB_LINEUP:          // User did a LINE UP
                        // take off 1 second or set to 1, whichever is greater
                    settings.outsecs = max(1, settings.outsecs-1);
                    break;
                case    SB_PAGEDOWN:        // User did a PAGE DOWN
                    settings.outsecs += 559;   // add 10 min and...
                case    SB_LINEDOWN:        // User did a LINE DOWN
                        // add 1 second or set to 1 hour, whichever is lesser
                    settings.outsecs = min(MAX_OUT, settings.outsecs+1);
                    break;
                case    SB_TOP:             // User went to the TOP
                    settings.outsecs = 1;    // set to 1 second
                    break;
                case    SB_BOTTOM:          // User went the to BOTTOM
                    settings.outsecs = MAX_IN;   // set to 1 hour
                    break;
                case    SB_THUMBPOSITION:   // User moved...
                case    SB_THUMBTRACK:      // or is moving the THUMB BUTTON
                        // Windows tells us how far...
                    settings.outsecs = LOWORD(lParam);
                default:
                    break;
                }
                // Put the new value into a string
                wsprintf(temp, "%4d", settings.outsecs);
                // Display it in the box next to the Scroll Bar
                SetDlgItemText(hDlg, IDC_OUTSECS, temp);
                // Set the new Scroll Bar position
                SetScrollPos(hsb, SB_CTL, settings.outsecs, TRUE);
            }
        }
        return TRUE;

    case    WM_COMMAND:         // One of the Buttons got pressed
        switch (wParam) {           // which one?
        case    IDC_FORE:
        case    IDC_BACK:
            foreground = (wParam == IDC_FORE) ? TRUE : FALSE;
            SendMessage(GetDlgItem(hDlg, IDC_FORE), BM_SETCHECK,
                        foreground, 0L);
            SendMessage(GetDlgItem(hDlg, IDC_BACK), BM_SETCHECK,
                        !foreground, 0L);
            SetScrollPos(hRScroll, SB_CTL,
                        (foreground ? fRed : bRed), TRUE);
            SetScrollPos(hGScroll, SB_CTL,
                        (foreground ? fGreen : bGreen), TRUE);
            SetScrollPos(hBScroll, SB_CTL,
                        (foreground ? fBlue : bBlue), TRUE);
            break;

        case    IDC_SAVE:           // Save Settings
            SaveSettings(hDlg);     // Save the Settings
            break;

        case    IDCANCEL:           // CANCEL
            settings = s;           // Restore setting to what they were

        case    IDOK:               // OK (or fall-thru from CANCEL)
            if (wParam == IDOK) {
                settings.fore = RGB(fRed, fGreen, fBlue);
                settings.back = RGB(bRed, bGreen, bBlue);
                NewBackground();
                InvalidateRect(Pundit, NULL, TRUE);
            }
            ReleaseDC(hDlg, hdc);
            EndDialog(hDlg, TRUE);  // we're done
            return TRUE;

        default:
            break;
        }
        break;
    }
    return FALSE;       // Say that we did NOT process the message
}


//  Retrieve WIN.INI settings for Pundit
void FetchSettings(void)
{
    BYTE    r,
            g,
            b;
    char    hold[80],
            tmp[80];

    //  Here we get each of the 8 settings values
    //  from PUNDIT.INI
    settings.insecs =  GetPrivateProfileInt((LPSTR)szAppName, (LPSTR)"InSecs",
                                    settings.insecs, (LPSTR)szProfName);
    settings.outsecs = GetPrivateProfileInt((LPSTR)szAppName, (LPSTR)"OutSecs",
                                    settings.outsecs, (LPSTR)szProfName);
    settings.cat =     GetPrivateProfileInt((LPSTR)szAppName, (LPSTR)"Category",
                                    settings.cat, (LPSTR)szProfName);
    settings.hide =    GetPrivateProfileInt((LPSTR)szAppName, (LPSTR)"Hide",
                                    settings.hide, (LPSTR)szProfName);
    settings.x =       GetPrivateProfileInt((LPSTR)szAppName, (LPSTR)"X",
                                    settings.x, (LPSTR)szProfName);
    settings.y =       GetPrivateProfileInt((LPSTR)szAppName, (LPSTR)"Y",
                                    settings.y, (LPSTR)szProfName);
    settings.width =   GetPrivateProfileInt((LPSTR)szAppName, (LPSTR)"Width",
                                    settings.width, (LPSTR)szProfName);
    settings.height =  GetPrivateProfileInt((LPSTR)szAppName, (LPSTR)"Height",
                                    settings.height, (LPSTR)szProfName);

    wsprintf(hold, "%d %d %d", GetRValue(settings.back),
                               GetGValue(settings.back),
                               GetBValue(settings.back));
    GetPrivateProfileString((LPSTR)szAppName, (LPSTR)"Back", (LPSTR)hold,
                            (LPSTR)tmp, 80, (LPSTR)szProfName);
    sscanf(tmp, "%d %d %d", &r, &g, &b);
    settings.back = RGB(r, g, b);

    wsprintf(hold, "%d %d %d", GetRValue(settings.fore),
                               GetGValue(settings.fore),
                               GetBValue(settings.fore));
    GetPrivateProfileString((LPSTR)szAppName, (LPSTR)"Fore", (LPSTR)hold,
                            (LPSTR)tmp, 80, (LPSTR)szProfName);
    sscanf(tmp, "%d %d %d", &r, &g, &b);
    settings.fore = RGB(r, g, b);

    if (settings.cat < IDM_QUOTES || settings.cat > IDM_RANDOM)
        settings.cat = 1;
    category = settings.cat;
    hide_mode = SW_HIDE;
    if (!settings.hide)
        hide_mode = SW_MINIMIZE;
    settings.hide = hide_mode;
}


//  Save the Settings (from Settings Dialog)
static void SaveSettings(hDlg)
HWND        hDlg;
{
    RECT    rect;               // boundaries of entire Window

    char    hold[80];

    //  Save the colors
    settings.fore = RGB(fRed, fGreen, fBlue);
    wsprintf(hold, "%d %d %d", GetRValue(settings.back),
                               GetGValue(settings.back),
                               GetBValue(settings.back));
    WritePrivateProfileString((LPSTR)szAppName, (LPSTR)"Back", (LPSTR)hold,
                              (LPSTR)szProfName);
    settings.back = RGB(bRed, bGreen, bBlue);
    wsprintf(hold, "%d %d %d", GetRValue(settings.back),
                               GetGValue(settings.back),
                               GetBValue(settings.back));
    WritePrivateProfileString((LPSTR)szAppName, (LPSTR)"Back", (LPSTR)hold,
                              (LPSTR)szProfName);

    //  write the IN seconds
    WritePrivateProfileInt((LPSTR)szAppName, (LPSTR)"InSecs",
                              settings.insecs,
                              (LPSTR)szProfName);

    //  write the OUT seconds
    WritePrivateProfileInt((LPSTR)szAppName, (LPSTR)"OutSecs",
                              settings.outsecs,
                              (LPSTR)szProfName);

    wsprintf(hold, "%d %d %d", GetRValue(settings.back),
                               GetGValue(settings.back),
                               GetBValue(settings.back));
    WritePrivateProfileString((LPSTR)szAppName, (LPSTR)"Back", (LPSTR)hold,
                              (LPSTR)szProfName);

    wsprintf(hold, "%d %d %d", GetRValue(settings.fore),
                               GetGValue(settings.fore),
                               GetBValue(settings.fore));
    WritePrivateProfileString((LPSTR)szAppName, (LPSTR)"Fore", (LPSTR)hold,
                              (LPSTR)szProfName);


    // Ask Windows for size and position of entire window
    GetWindowRect(Pundit, (LPRECT)&rect);

    // find out if size or position has changed
    if ((settings.x      != rect.left) ||
        (settings.y      != rect.top) ||
        (settings.width  != rect.right - rect.left) ||
        (settings.height != rect.bottom - rect.top))
    {
        //  Ask user if we should save position and size
        if (MessageBox(hDlg,
                       (LPSTR)"Save Current Window Position and Size?",
                       (LPSTR)"Saving",
                       MB_ICONQUESTION | MB_YESNO) == IDYES)
        {
            SaveSizeAndPosition(&rect);   // yes, save them
        }
    }
}


static void SaveSizeAndPosition(rect)
RECT    *rect;
{
    settings.x      = rect->left;               // X position
    settings.y      = rect->top;                // Y position
    settings.width  = rect->right - rect->left; // Width
    settings.height = rect->bottom - rect->top; // Height

    //  write the X position
    WritePrivateProfileInt((LPSTR)szAppName, (LPSTR)"X", settings.x,
                              (LPSTR)szProfName);

    //  write the Y position
    WritePrivateProfileInt((LPSTR)szAppName, (LPSTR)"Y",
                              settings.y,
                              (LPSTR)szProfName);

    //  write the Width
    WritePrivateProfileInt((LPSTR)szAppName, (LPSTR)"Width",
                              settings.width,
                              (LPSTR)szProfName);

    //  write the Height
    WritePrivateProfileInt((LPSTR)szAppName, (LPSTR)"Height",
                              settings.height,
                              (LPSTR)szProfName);
}


//  Initialize Settings Dialog info
static void InitSettings(hDlg)
HWND    hDlg;
{
    HWND    hwnd;
    char    temp[15];

    s = settings;       // stash the current settings (in case of CANCEL)

    //  Write the current IN seconds in the box next to the Scroll Bar
    wsprintf(temp, "%4d", settings.insecs);
    SetDlgItemText(hDlg, IDC_INSECS, temp);

    //  Write the current OUT seconds in the box next to the Scroll Bar
    wsprintf(temp, "%4d", settings.outsecs);
    SetDlgItemText(hDlg, IDC_OUTSECS, temp);

    //  Get handles to each of the 5 Scroll Bar Controls
    hInScroll  = GetDlgItem(hDlg, IDC_INSCROLL);
    hOutScroll = GetDlgItem(hDlg, IDC_OUTSCROLL);
    hRScroll   = GetDlgItem(hDlg, IDC_RSCROLL);
    hGScroll   = GetDlgItem(hDlg, IDC_GSCROLL);
    hBScroll   = GetDlgItem(hDlg, IDC_BSCROLL);

    //  Separate the color values for Foreground and Background
    fRed    = GetRValue(settings.fore);
    fGreen  = GetGValue(settings.fore);
    fBlue   = GetBValue(settings.fore);
    bRed    = GetRValue(settings.back);
    bGreen  = GetGValue(settings.back);
    bBlue   = GetBValue(settings.back);

    //  Set the Scroll Bar Ranges
    SetScrollRange(hInScroll,  SB_CTL, 1, MAX_IN,  TRUE);
    SetScrollRange(hOutScroll, SB_CTL, 1, MAX_OUT, TRUE);
    SetScrollRange(hRScroll,   SB_CTL, 0, 255,     TRUE);
    SetScrollRange(hGScroll,   SB_CTL, 0, 255,     TRUE);
    SetScrollRange(hBScroll,   SB_CTL, 0, 255,     TRUE);

    //  Set the Scroll Bar Positions
    SetScrollPos(hInScroll,    SB_CTL, settings.insecs,                TRUE);
    SetScrollPos(hOutScroll,   SB_CTL, settings.outsecs,               TRUE);
    SetScrollPos(hRScroll,     SB_CTL, (foreground ? fRed   : bRed),   TRUE);
    SetScrollPos(hGScroll,     SB_CTL, (foreground ? fGreen : bGreen), TRUE);
    SetScrollPos(hBScroll,     SB_CTL, (foreground ? fBlue  : bBlue),  TRUE);

    hdc = GetDC(hwnd = GetDlgItem(hDlg, IDC_DEMO));
    GetDlgItemText(hDlg, IDC_DEMO, (LPSTR)text, sizeof(text) - 1);
    SetClassWord(hwnd, GCW_HBRBACKGROUND, hOrgBrush);

    SetBkMode(hdc, OPAQUE); // has to be OPAQUE for background color to show

    SendMessage(GetDlgItem(hDlg, IDC_FORE), BM_SETCHECK, foreground, 0L);
    SendMessage(GetDlgItem(hDlg, IDC_BACK), BM_SETCHECK, !foreground, 0L);
}


//  Process the Scroll Bars for Color
void HandleColorScroll(wParam, lParam)
WORD    wParam;
LONG    lParam;
{
    HWND    hsb;
    short   *color;

    hsb = HIWORD(lParam);
    if (hsb == hRScroll)
        color = (foreground ? &fRed : &bRed);
    if (hsb == hGScroll)
        color = (foreground ? &fGreen : &bGreen);
    if (hsb == hBScroll)
        color = (foreground ? &fBlue : &bBlue);

    switch (wParam) {
    case    SB_PAGEDOWN:
        *color += 15;       // fall through
    case    SB_LINEDOWN:
        *color = min(255, *color + 1);
        break;
    case    SB_PAGEUP:
        *color -= 15;       // fall through
    case    SB_LINEUP:
        *color = max(0, *color - 1);
        break;
    case    SB_TOP:
        *color = 0;
        break;
    case    SB_BOTTOM:
        *color = 255;
        break;
    case    SB_THUMBPOSITION:
    case    SB_THUMBTRACK:
        *color = LOWORD(lParam);
    default:
        break;
    }
    SetScrollPos(hsb, SB_CTL, *color, TRUE);
    SetTextColor(hdc, RGB(fRed, fGreen, fBlue));
    SetBkColor(hdc, RGB(bRed, bGreen, bBlue));

    TextOut(hdc, 0, 0, (LPSTR)text, strlen(text));
}


//  set up background color for main window
static void NewBackground()
{
    HDC     hdc = GetDC(Pundit);

    DeleteObject(SelectObject(hdc, hOrgBrush = CreateSolidBrush(settings.back)));
    SetClassWord(Pundit, GCW_HBRBACKGROUND, hOrgBrush);
    ReleaseDC(Pundit, hdc);
}

