/*
 *  Supplier Data Validation: The H109 Test for Downward Supplier Bias
 *
 *                   A Quality Assurance Training Tool:
 *            Statistics Committee of the QA Section of the PMA
 *
 *               Bob Obenchain, CompuServe User [72007,467]
 *
 *                    Usage: DOSprompt>  sdvalid
 *
 *      Parameter settings are either...
 *             specified interactively in response to prompts, or
 *             redirected to come from a batch input file.
 */

#include <math.h>
#include <stdio.h>
#include <signal.h>
#include <stdlib.h>
#include <string.h>
#include <process.h>

#define abs(a)      (( (a) >= 0.0 ) ? (a) : (-a))  /* absolute value of a */
#define sgn(a)      (( (a) >= 0.0 ) ?  1  :  -1 )  /* numerical sign of a */
#define ESCK    0x01   /* scancode for Escape key */
#define BKEY    0x30
#define KKEY    0x25   
#define ITMAX   100
#define EPS     3.0e-7

/* externals */

FILE    *inpfile, *outfile;
char    buf[256], inpnam[13], outnam[13];
double  thresh, ratio, siglve, siglvh, fdse, fdce, fdsh, fdch;
long    ns, nc, ds, dc;
double  quans[5], ratios[5];
static  double probs[] = { 0.01, 0.05, 0.10, 0.25, 0.50 };

/* prototypes */

void   main( void );
int    handler( void );
double qbeta( double p, double a, double b );
double qnorms( double pr );
double betai( double a, double b, double x );
double betacf( double a, double b, double x );
double gammln( double xx );
extern int get1key( void );

int handler() {     /* Interrupt Handler */
      int c;        /* key scan code,,char */
      
      signal(SIGINT,handler);
      printf("\n\n...Hit the ESCape Key to Quit, any other key to Continue...\n\n");
      c = get1key() >> 8;
      if( c == ESCK ) {
           exit( 1 );
           }
      /* else...normal return, to continue processing... */
      return( 0 );
      }

double qbeta(p,a,b)
double p, a, b; {
      int right, i;
      double quan, tol, xl, pl, xr, pr, xm, sd, xx, pp, pdif;
      tol = 1.17577E-37;

      quan = p;
      if( a == 1.0 && b == 1.0 )
          return( quan );

      if( p < 0.05 ) {
          xl = 0.0;
          pl = 0.0;
          right = 0;
          }
      else if( p > 0.95 ) {
          xr = 1.0;
          pr = 1.0;
          right = 1;
          }
      else {
          /* normal approximations... */
          xm = a + b;
          sd = sqrt((a*b)/(xm*xm*(xm+1.)));
          xl = qnorms(p) * sd + a / xm;
          if( xl < tol ) {
              xl=0.0;
              pl=0.0;
              right=0;
              }
          else if( xl > (1.0-tol) ) {
              xr=1.0;
              pr=1.0;
              right=1;
              }
          else {
              pl=betai(a,b,xl);
              right=0;
              if( pl > p ) {
                  right=1;
                  pr=pl;
                  xr=xl;
                  }
              }
          }

      if( right == 1 ) {

          /* find left end... */
          while( 1 ) {
              xl = max( xr-0.05, 0.0 );
              if( xl <= 0.0 ) {
                  pl = 0.0;
                  break;
                  }
              pl=betai(a,b,xl);
              quan = xl;
              if( pl == p )
                  return( quan );
              if( pl < p )
                  break;
              xr=xl;
              pr=pl;
              }
          }

      if( right == 0 ) {

          /* find right end... */
          while( 1 ) {
              xr = min( xl+0.05, 1.0 );
              if( xr >= 1.0 ) {
                  pr = 1.0;
                  break;
                  }
              pr=betai(a,b,xr);
              quan = xr;
              if( pr == p )
                  return( quan );
              if( pr > p )
                  break;
              xl=xr;
              pl=pr;
              }
          }

      /* now quantile is between xl and xr; use bisection */
      for( i=1; i<=6; i++ ) {
          xx = (xl+xr)*.5;
          pp = betai(a,b,xx);
          pdif = pp-p;
          quan = xx;
          if( abs(pdif) < tol )
              return( quan );
          if( pdif > 0.0 ) {
              xr=xx;
              pr=pp;
              }
          else {
              xl = xx;
              pl = pp;
              }
          }

      /* next, try secant method... */
      for( i=1; i<=6; i++ ) {
          xx = xl+(p-pl)*(xr-xl)/(pr-pl);
          pp = betai(a,b,xx);
          pdif = pp-p;
          quan =xx;
          if( abs(pdif) < tol)
              return( quan );
          if( pdif > 0.0) {
              xr = xx;
              pr = pp;
              }
          else {
              xl = xx;
              pl = pp;
              }
          }
      quan = xx;
      /* failed to converge... */
      return( quan );
      }

double qnorms( pr )
double pr; {
      double quan, p, eta, f1, f2, f3, f4, f5, f6;
      quan = 0.0;
      if( pr <= 0.0 || pr == 0.5 || pr >= 1.0 )
          return( quan );
      f1 = .010328;
      f2 = .802853;
      f3 = 2.515517;
      f4 = .001308;
      f5 = .189269;
      f6 = 1.432788;
      p = pr;
      if( pr > 0.5 )
          p = 1.0 - pr;
      eta = sqrt( -2.0 * log( p ) );
      quan = eta - ((f1*eta+f2)*eta+f3)/(((f4*eta+f5)*eta+f6)*eta+1.0);
      if( pr <= 0.5 )
          quan *= -1.0;
      return( quan );
      }

/*
 *  The source code for the betai(a,b,x), betacf(a,b,x), and gammln(xx)
 *  functions is not reproduced here because these are copyrighted works
 *  by Press, et.al. (1988). "Numerical Recipes in C," Cambridge Univ.Press.
 *  The source code is also available on floppy disks distributed by
 *  Cambridge Univ.Press.
 */

void main() {

      int kb, i, yesno;

      if( signal(SIGINT,handler) == (int(*)())-1 ) {
             printf("\n\n\tSDVALID: Couldn't set SIGINT...Abort!\n\n");
             exit( 1 );
             }

      printf("\n\n     ***  Supplier Data Validation ...using H109 & Exact  ***" );
      printf(  "\n         Tests for Downward Bias in Reported Defect Rates" );
      printf("\n\n                   SDVALID.EXE....Version 9101");
      printf("\n\n                A Quality Assurance Training Tool:");
      printf(  "\n         Statistics Committee of the QA Section of the PMA");
      printf("\n\n            Bob Obenchain, CompuServe User [72007,467]\n\n" );
      
      printf(      "\tWill Parameter Settings be Input via K = Keyboard ? ...or" );
      printf(    "\n\t                                     B = Batch File ?" );
      printf(  "\n\n\tPress the   K   or   B   key now --> " );
      while( 1 ) {
              if( ( kb = get1key() >> 8 ) == KKEY || kb == BKEY )
                      break;
              printf( "\r\tPress either K  or  B  ...Try Again --> " );
              }

      strcpy( outnam, "sdvalid" );
      strcpy( inpnam, "sdvalid" );

      if( kb == BKEY )
              printf( "\n\n\tBatch Input of Parameter Settings Selected..." );
      else
              printf( "\n\n\tKeyboard Input of Parameter Settings Selected..." );
      
      printf( "\n\n\tAt colon Prompts : ...simply press ENTER to get the [default]." );

      if( kb == BKEY ) {
              printf(  "\n\n\tSpecify filename of Batch Input File [%s] : ",
                  inpnam );
              gets( buf );
              if( buf[0] != '\0' && buf[0] != '\r' && buf[0] != '\n' )
                  sscanf( buf, "%s", inpnam );
              inpnam[ min( 8, (int)strcspn( inpnam, ". " ) ) ] = '\0';
              if( inpnam[0] == '\0' )
                  strcpy( inpnam, "sdvalid" );
              if( stricmp( inpnam, "sdvalid" ) != 0 ) {
                  strcpy( outnam, inpnam );
                  }
              strcat( inpnam, ".inp" );
              printf( "\n\tThe Batch Input file is to be: %s\n", inpnam );
              if( ( inpfile = fopen( inpnam, "r" ) ) == NULL ) {
                      printf(  "\tCannot read Batch Input file: %s\n", inpnam );
                      printf(  "\t...using Keyboard Input from standard Infile, stdin.\n" );
                      kb = KKEY;
                      }
              }
      
      if( kb == KKEY )
              inpfile = stdin;

      printf(  "\n\n\tSpecify filename to save primary SDValid Output [%s] : ",
          outnam );
      fgets( buf, 13, inpfile );
      if( buf[0] != '\0' && buf[0] != '\r' && buf[0] != '\n' )
          sscanf( buf, "%s", outnam );
      outnam[ min( 8, (int)strcspn( outnam, ". " ) ) ] = '\0';
      if( outnam[0] == '\0' )
          strcpy( outnam, "sdvalid" );
      strcat( outnam, ".out" );
      printf(  "\n\tThe SDValid primary Output file is to be: %s\n",
              outnam );
      if( ( outfile = fopen( outnam, "w" ) ) == NULL ) {
              printf(  "\tCannot write to Output filename : %s\n", outnam );
              if( ( outfile = fopen( "sdvalid.out", "w" ) ) == NULL )
                      printf(  "\tCannot write to Output filename : sdvalid.out\n" );
              else {
                      printf(  "\t...using default Outfile name : sdvalid.out\n" );
                      strcpy( outnam, "sdvalid.out" );
                      }
              }

      if( outfile != NULL )
          fprintf( outfile, "Output file for SDValid..." );
     
      printf("\n\n\tThis software performs both Exact and H109 type tests" );
      printf(  "\n\tfor Downward Bias in supplier provided defect rates.");
      
      ns = 50;
      nc =  5;
      ds =  0;
      dc =  1;

      while( 1 ) {

          printf("\n\n\tWhat is the Supplier Sample Size ? [ns=%ld] : ", ns );
          fgets( buf, 8, inpfile );
          if( buf[0] != '\0' && buf[0] != '\r' && buf[0] != '\n' )
              sscanf( buf, "%ld", &ns );
          if( ns < 1 )
              ns = 1;
          if( ns > 2000000000 )
              ns = 2000000000;
          printf("\n\tSupplier Sample Size: ns =%ld", ns);

          printf("\n\n\tWhat is the Customer Sample Size ? [nc=%ld] : ", nc );
          fgets( buf, 8, inpfile );
          if( buf[0] != '\0' && buf[0] != '\r' && buf[0] != '\n' )
              sscanf( buf, "%ld", &nc );
          if( nc < 1 )
              nc = 1;
          if( nc > 2000000000 )
              nc = 2000000000;
          printf("\n\tCustomer Sample Size: nc =%ld", nc);

          printf("\n\n\tHow many Nonconformances in Supplier Sample? [ds=%ld] : ",
              ds );
          fgets( buf, 8, inpfile );
          if( buf[0] != '\0' && buf[0] != '\r' && buf[0] != '\n' )
              sscanf( buf, "%ld", &ds );
          if( ds > ns )
              ds = ns;
          if( ds < 0 )
              ds = 0;
          printf("\n\tSupplier Reported Nonconformances: ds =%ld", ds );

          printf("\n\n\tHow many Nonconformances in Customer Sample? [dc=%ld] : ",
              dc );
          fgets( buf, 8, inpfile );
          if( buf[0] != '\0' && buf[0] != '\r' && buf[0] != '\n' )
              sscanf( buf, "%ld", &dc );
          if( dc > nc )
              dc = nc;
          if( dc < 0 )
              dc = 0;
          printf("\n\tCustomer Observed Nonconformances: dc =%ld", dc );

          ratio = (double)ns / (double)nc;
          thresh = 1.0 / (1.0 + ratio);

          printf( "\n\n\n\tSDValid results for %ld=ns, %ld=nc, %ld=ds, and %ld=dc.",
                  ns, nc, ds, dc );
          printf( "\n\n\tThe observed sample size ratio of ns/nc = %7.3lf",
                  ratio );
          if( outfile != NULL ) {
              fprintf( outfile, "\n\n*****************\n" );
              fprintf( outfile, "\nSDValid results for %ld=ns, %ld=nc, %ld=ds, and %ld=dc.",
                  ns, nc, ds, dc );
              fprintf( outfile, "\n\nThe observed sample size ratio of ns/nc = %7.3lf",
                  ratio );
              }

          fdse = (double)ds + 1.0;
          fdce = (double)dc;
          if( dc > (long)0 ) {
              siglve = betai( fdce, fdse, thresh );
              printf( "\n\t\t...has EXACT observed significance level = %6.3lf",
                  siglve );
              if( outfile != NULL )
                  fprintf( outfile, "\n...has EXACT observed significance level = %6.3lf",
                      siglve );
              }

          fdsh = (double)ds + 0.5;
          fdch = (double)dc + 0.5;
          siglvh = betai( fdch, fdsh, thresh );

          if( dc > (long)0 )
              printf( "\n\t\t...and H109  observed significance level = %6.3lf",
                  siglvh );
          else {
              printf( "\n\t\tH109  observed significance level = %6.3lf",
                  siglvh );
              printf( "\n\tNOTE: This level is actually useless because dc = 0." );
              printf( "\n\t====================================================" );
              printf( "\n\tExact calculations are impossible in this case." );
              printf( "\n\t====================================================" );
              }

          if( outfile != NULL && dc > (long)0 )
              fprintf( outfile,
                  "\n...and H109  observed significance level = %6.3lf",
                  siglvh );
          else if( outfile != NULL ) {
              fprintf( outfile,
                  "\nH109  observed significance level = %6.3lf",
                  siglvh );
              fprintf( outfile,
                  "\nNOTE: This level is actually useless because dc = 0." );
              fprintf( outfile,
                  "\n====================================================" );
              fprintf( outfile,
                  "\nExact calculations are impossible in this case." );
              fprintf( outfile,
                  "\n====================================================" );
              }

          if( siglvh > 0.1 ) {
              printf( "\n\n\tThis result is in the NORMAL range (No warn/action.)" );
              if( outfile != NULL )
                  fprintf( outfile, "\n\nThis result is in the NORMAL range (No warn/action.)" );
              }
          else if( siglvh > 0.05 && siglvh <= 0.1 ) {
              printf( "\n\n\tThis result is in the WARNING range (no action yet.)" );
              if( outfile != NULL )
                  fprintf( outfile, "\n\nThis result is in the WARNING range (no action yet.)" );
              }
          else if( siglvh <= 0.05 ) {
              printf( "\n\n\tThis result is in the ACTION range..." );
              printf( "\n\tSupplier data show significant (5%% level) downward bias." );
              if( outfile != NULL ) {
                  fprintf( outfile, "\n\nThis result is in the ACTION range..." );
                  fprintf( outfile, "\nSupplier data show significant (5%% level) downward bias." );
                  }
              }

          for( i=0; i<5; i++ ) {
              if( dc > (long)0 )
                  quans[i] = qbeta( probs[i], fdce, fdse );
              else
                  quans[i] = 0.0;
              ratios[i] = 9999.999;
              if( quans[i] > 0.0 )
                  ratios[i] = (1.0 - quans[i] ) / quans[i];
              }

          printf( "\n\n\tThresholds" );
          printf(   "\n\tfor ns/nc:   1%%       5%%      10%%      25%%      50%%" );
          printf( "\n\tEXACT: %8.3lf %8.3lf %8.3lf %8.3lf %8.3lf",
              ratios[0], ratios[1], ratios[2], ratios[3], ratios[4] );
          if( outfile != NULL ) {
              fprintf( outfile, "\n\nThresholds" );
              fprintf( outfile,
                  "\nfor ns/nc:   1%%       5%%      10%%      25%%      50%%" );
              fprintf( outfile, "\nEXACT: %8.3lf %8.3lf %8.3lf %8.3lf %8.3lf",
                  ratios[0], ratios[1], ratios[2], ratios[3], ratios[4] );
              }

          for( i=0; i<5; i++ ) {
              quans[i] = qbeta( probs[i], fdch, fdsh );
              ratios[i] = 999.999;
              if( quans[i] > 0.0 )
                  ratios[i] = (1.0 - quans[i] ) / quans[i];
              }

          printf( "\n\tH109 : %8.3lf %8.3lf %8.3lf %8.3lf %8.3lf",
              ratios[0], ratios[1], ratios[2], ratios[3], ratios[4] );
          if( outfile != NULL )
              fprintf( outfile, "\nH109 : %8.3lf %8.3lf %8.3lf %8.3lf %8.3lf",
                  ratios[0], ratios[1], ratios[2], ratios[3], ratios[4] );

          yesno = 'y';
          printf("\n\n\tDo you want to specify additional tests [Y|n] ? ");
          fgets( buf, 2, inpfile );
          if( buf[0] != '\0' && buf[0] != '\r' && buf[0] != '\n' )
              sscanf( buf, "%c", &yesno );
          if( yesno == 'n' || yesno == 'N' )
              break;
          }

      printf( "\n\nExiting SDValid...\n\n" );
      if( outfile != NULL )
          fprintf( outfile, "\n\n*****************\n\nExiting SDValid..." );
      }

