/*
 * OCINTRVL...Simulated Operating Characteristic Curves for 1-, 2- or
 *            3-stage Acceptance Sampling Plans that Reject when
 *            Intervals of the form [Xbar-K*S,Xbar+K*S] fail to fall
 *            entirely within Stated Specification Limits.
 */

#include <stdio.h>
#include <math.h>
#include <string.h>
#include <stdlib.h>
#include <dos.h>
#include <ctype.h>

#define square(x)  ( (x) * (x) )
#define min(x,y)   ( (x) <= (y) ? (x) : (y) )

FILE      *outfile, *csvfile;
char      buf[256], yesno, outnam[13], csvnam[13];
int       repl, replics, rstep, ss, samsiz[3], samtot, stg, stgsiz;
int       inside[6], lfside[6], multips, idum, ndx;
long      stages[2][6];
double    noiseavg, noisevar, noisestd, cyield, vyield;
double    kfact[2][3], noise[500], limlor[3], limupr[3];
double    avgstg[2], component, fratio;
static    double sz[] = { 4.645, 3.480, 2.807, 2.576, 1.960, 1.645 };
static    double az[] = { 4.500, 3.291, 2.576, 2.326, 1.645, 1.281 };
static    double  p[] = { .0000034, .0005, .0050, .0100, .0500, .1000 };

void      main( void );
long      time( long *time );
int       initseed( void );
double    box_mull( int *idum );
float     uniran( int *idum );
void      dblget( double *value );
void      intget( int *value );
void      keyget( char *value );
void      statval( double value[], int size, double *mean, double *var );

extern    double beta( double z, double w );
extern    double betai( double a, double b, double x );
extern    double betacf( double a, double b, double x );
extern    double gammln( double xx );

extern    void exit( int code );

/*
 *      initseed()...initialize pseudo-random sequence using the Microsoft
 *                   long time() function to read system clock.
 */

int initseed() {
    long ltime, time();
    int seed;

    time( &ltime );
    seed = (int)ltime ;
    printf( "\n\tUsing Clock to get Seed...\n");
    if( seed > 0 ) return( -seed );
    else if( seed == 0 ) return( -13 );
    else return( seed );
    }

/*
 * Brute Force Box-Muller Transformation.
 */
double box_mull( idum )
int *idum; {

        static int iset=0;
        static double next;
        double rad,ang;

        if( iset == 0 ) {
            rad = sqrt( -2.0 * log( (double)uniran( idum ) ) );
            ang = 2.0 * 3.1415926536 * (double)uniran( idum );
            next = rad * cos( ang );
            iset = 1;
            return( rad * sin( ang ) );
            }
        else {
            iset=0;
            return next;
            }
        }

/*
 * Portable 32-bit random number generator proposed by Pierre
 * L'Ecuyer(1988), Communications of the ACM 31, 742-749 and 774.
 * Static integer seed values s1 and s2 must not only be >= 1
 * but also s1 <= 2,147,483,562 and s2 <= 2,147,483,398, respectively.
 */

float uniran( int *idum ) {

        static long s1, s2, ldum;
        static int iff=0;

        if( *idum < 0 || iff == 0 ) {
                ldum = (long)abs(*idum);
                if( ldum < (long)1 )
                    ldum = (long)1;
                s1 = s2 = ldum;
                *idum = iff = 1;
                }

        ldum = s1 / (long)53668;
        s1 =  (long)40014 * ( s1 - ldum * (long)53668 ) - ldum * (long)12211;
        if( s1 < 0 )
            s1 += (long)2147483563;

        ldum = s2 / (long)52774;
        s2 =  (long)40692 * ( s2 - ldum * (long)52774 ) - ldum * (long)3791;
        if( s2 < 0 )
            s2 += (long)2147483399;

        ldum = s1 - s2;
        if( ldum < (long)1 )
            ldum += (long)2147483562;

        return (float)( (double)ldum * 4.656613e-10 );
        }

void dblget( value )
double *value; {

        gets( buf );
        if( buf[0] != '\0' && buf[0] != '\r' && buf[0] != '\n' )
                sscanf( buf, "%lf", value );
        }

void intget( value )
int *value; {

        gets( buf );
        if( buf[0] != '\0' && buf[0] != '\r' && buf[0] != '\n' )
                sscanf( buf, "%d", value );
        }

void keyget( value )
char *value; {
        int p;
        gets( buf );
        for( p = 0; p < 256; p++ ) {
            if( buf[p] != ' ' && buf[p] != '\t' && buf[p] != '\r' )
                break;
            }
        if( buf[p] != '\0' && buf[p] != '\n' )
            sscanf( buf+p, "%c", value );
        }

void statval( value, size, avrg, disp )
double value[];
int size;
double *avrg, *disp; {
   int idx;
   *avrg = value[0];
   *disp = 0.0;
   if( size < 2 )
       return;
   for( idx=1; idx<size; idx++ )
       *avrg += value[idx];
   *avrg /= (double)size;
   for( idx=0; idx<size; idx++ )
       *disp += square( value[idx] - *avrg );
   *disp /= (double)size-1.0;
   }

/* OCINTRVL main module */

void main() {

   int idx;
   char c;

   printf( "\n\n\t***  Simulated OC Curve for Sample Mean+/-K*Std.Err. Intervals  ***\n");
   printf( "\n\n\t                    OCINTRVL.EXE...Version 9402" );
   printf( "\n\n\t                A Quality Assurance Training Tool:" );
   printf(   "\n\t         Statistics Committee of the QA Section of the PMA" );
   printf( "\n\n\t            Bob Obenchain, CompuServe User [72007,467]\n\n" );
   
   strcpy( outnam, "ocintrvl" );

   printf( "\n\n\tAt colon Prompts : ...simply press ENTER to get the [default]." );
   
   printf( "\n\n\tSpecify filename for Detailed Output [%s] : ", outnam );
   gets( buf );
   if( buf[0] != '\0' && buf[0] != '\r' && buf[0] != '\n' )
           sscanf( buf, "%s", outnam );
   outnam[ min( 8, (int)strcspn( outnam, ". " ) ) ] = '\0';
   if( outnam[0] == '\0' )
           strcpy( outnam, "ocintrvl" );
   strcpy( csvnam, outnam );
   strcat( outnam, ".out" );
   printf( "\n\tThe OCINTRVL Output Save file is to be: %s\n",
           outnam );
   if( ( outfile = fopen( outnam, "w" ) ) == NULL ) {
           printf( "\tCannot write to Output filename : %s\n", outnam );
           if( ( outfile = fopen( "ocintrvl.out", "w" ) ) == NULL )
                   printf( "\tCannot write to Output filename : ocintrvl.out\n" );
           else {
                   printf( "\t...using default Outfile name : ocintrvl.out\n" );
                   strcpy( outnam, "ocintrvl.out" );
                   }
           }

   strcat( csvnam, ".csv" );
   printf( "\n\tThe OCINTRVL Comma Separated Value file is to be: %s\n",
           csvnam );
   if( ( csvfile = fopen( csvnam, "w" ) ) == NULL ) {
           printf( "\tCannot write to CSV filename : %s\n", csvnam );
           if( ( csvfile = fopen( "ocintrvl.csv", "w" ) ) == NULL )
                  printf( "\tCannot write to CVS filename : ocintrvl.csv\n" );
           else {
                  printf( "\t...using default CSV name : ocintrvl.csv\n" );
                  strcpy( csvnam, "ocintrvl.csv" );
                  }
           }

   printf( "\n\n\t\tEnter a Q key to QUIT now..." );
   printf( "\n\t\tSimply Press Enter to Continue..." );
   keyget( &c );
   if( c == 'Q' || c == 'q' )
       exit( 0 );


   samsiz[0] = 15;
   samsiz[1] = samsiz[2] = 0;
   kfact[0][0] = kfact[1][0] = 3.00;
   replics = 1000;

   while( 1 ) {

       printf( "\n\n\tSpecify Parameters by Responding to the following prompts...\n" );

       printf( "\n\n\tMaximum Sample Size = 500 (total over all stages.)" );
       printf( "\n\n\tMinimum Sample Size =   2.");
       printf( "\n\n\tEnter 1st Stage Sample Size [%d] : ", samsiz[0] );
       intget( &samsiz[0] );
       if( samsiz[0] < 1 || samsiz[0] > 500 ) {
               printf( "\n\t1st Stage Sample Size will be 15, the default.\n");
               samsiz[0] = 15;
               }

       printf( "\n\n\tEnter 2nd Stage Sample Size [%d] : ", samsiz[1] );
       intget( &samsiz[1] );
       if( samsiz[1] < 0 || samsiz[0] + samsiz[1] > 500 ) {
               printf( "\n\t2nd Stage Sample Size will be 0, the default.\n");
               samsiz[1] = 0;
               }

       if( samsiz[1] > 0 ) {
           printf( "\n\n\tEnter 3rd Stage Sample Size [%d] : ", samsiz[2] );
           intget( &samsiz[2] );
           if( samsiz[2] < 0 || samsiz[0] + samsiz[1] + samsiz[2] > 500 ) {
               printf( "\n\t3rd Stage Sample Size will be 0, the default.\n");
               samsiz[2] = 0;
               }
           }
       else
           samsiz[2] = 0;
       samtot = samsiz[0] + samsiz[1] + samsiz[2];

       printf( "\n\n\tSimulate Operating Characteristic Curve for Intervals" );
       printf(   "\n\twith Half-Width set at K Standard Deviations..." );
       printf( "\n\n\tEnter 1st Stage Lower K Multiplier [%5.2lf] : ", kfact[0][0] );
       dblget( &kfact[0][0] );
       if( kfact[0][0] < 1.0 )
           kfact[0][0] = 1.0;
       else if( kfact[0][0] > 10.0 )
           kfact[0][0] = 10.0;

       kfact[1][0] = kfact[0][0];
       printf( "\n\n\tEnter 1st Stage Upper K Multiplier [%5.2lf] : ", kfact[1][0] );
       dblget( &kfact[1][0] );
       if( kfact[1][0] < 1.0 )
           kfact[1][0] = 1.0;
       else if( kfact[1][0] > 10.0 )
           kfact[1][0] = 10.0;

       kfact[0][1] = kfact[0][0];
       if( samsiz[1] > 0 ) {
           printf( "\n\n\tEnter 2nd Stage Lower K Multiplier [%5.2lf] : ", kfact[0][1] );
           dblget( &kfact[0][1] );
           if( kfact[0][1] < 1.0 )
               kfact[0][1] = 1.0;
           else if( kfact[0][1] > 10.0 )
               kfact[0][1] = 10.0;
           }

       kfact[1][1] = kfact[1][0];
       if( samsiz[1] > 0 ) {
           printf( "\n\n\tEnter 2nd Stage Upper K Multiplier [%5.2lf] : ", kfact[1][1] );
           dblget( &kfact[1][1] );
           if( kfact[1][1] < 1.0 )
               kfact[1][1] = 1.0;
           else if( kfact[1][1] > 10.0 )
               kfact[1][1] = 10.0;
           }

       kfact[0][2] = kfact[0][1];
       if( samsiz[2] > 0 ) {
           printf( "\n\n\tEnter 3rd Stage Lower K Multiplier [%5.2lf] : ", kfact[0][2] );
           dblget( &kfact[0][2] );
           if( kfact[0][2] < 1.0 )
               kfact[0][2] = 1.0;
           else if( kfact[0][2] > 10.0 )
               kfact[0][2] = 10.0;
           }

       kfact[1][2] = kfact[1][1];
       if( samsiz[2] > 0 ) {
           printf( "\n\n\tEnter 3rd Stage Upper K Multiplier [%5.2lf] : ", kfact[1][2] );
           dblget( &kfact[1][2] );
           if( kfact[1][2] < 1.0 )
               kfact[1][2] = 1.0;
           else if( kfact[1][2] > 10.0 )
               kfact[1][2] = 10.0;
           }

       printf( "\n\n\tMaximum Simulation Replications = 25,000.");
       printf( "\n\n\tEnter Number of Simulation Replications [%d] : ", replics );
       intget( &replics );
       if( replics < 100 || replics > 25000 ) {
               printf( "\n\tNumber of Simulation Replications will be 1000, the default.\n");
               replics = 1000;
               }

       printf( "\n\n\tSimulated Acceptance Probability for Mean+/-K*Std.Err. Intervals..." );
       printf( "\n\t1-, 2- & 3-Stage Sample Sizes will be %d, %d & %d.",
           samsiz[0], samsiz[1], samsiz[2] );
       printf( "\n\tCorresponding Numerical Values of Upper K = %5.2lf, %5.2lf & %5.2lf.",
           kfact[1][0], kfact[1][1], kfact[1][2] );
       printf( "\n\tCorresponding Numerical Values of Lower K = %5.2lf, %5.2lf & %5.2lf.",
           kfact[0][0], kfact[0][1], kfact[0][2] );
       printf( "\n\tMaximum Number of Simulation Replications will be %d.", replics );

       fprintf( outfile, "\nSimulated Acceptance Probability for Mean+/-K*Std.Err. Intervals..." );
       fprintf( outfile, "\n1-, 2- & 3-Stage Sample Sizes will be %d, %d & %d.",
           samsiz[0], samsiz[1], samsiz[2] );
       fprintf( outfile, "\nCorresponding Numerical Values of Upper K = %5.2lf, %5.2lf & %5.2lf.",
           kfact[1][0], kfact[1][1], kfact[1][2] );
       fprintf( outfile, "\nCorresponding Numerical Values of Lower K = %5.2lf, %5.2lf & %5.2lf.",
           kfact[0][0], kfact[0][1], kfact[0][2] );
       fprintf( outfile, "\nMaximum Number of Simulation Replications will be %d.", replics );

       idum = 0;
       printf( "\n\n\tInput Start-up Seed ( 0 => use Clock ): " );
       intget( &idum );
       if( idum > 0 )
               idum = -idum ;
       if( idum == 0 )
               idum = initseed();

       printf( "\n\tRandom Number Seed  = %d", idum );
       fprintf( outfile, "\nRandom Number Seed  = %d", idum );

       printf( "\n\tSimulation Initiated...\n\n" );

       for( idx = 0; idx < 6; idx++ ) {
            inside[idx] = 0;
            lfside[idx] = 0;
            stages[0][idx] = stages[1][idx] = 0L;
            }

       for( repl = 0; repl < replics; repl++ ) {

           for( ss=0; ss < samtot; ss++ )
               noise[ss] = box_mull( &idum );

           stgsiz = 0;
           for( stg = 0; stg < 3; stg++ ) {
               stgsiz += samsiz[stg];
               if( samsiz[stg] > 0 )
                   statval( noise, stgsiz, &noiseavg, &noisevar );
               noisestd = sqrt( noisevar );
               limlor[stg] = noiseavg - kfact[0][stg] * noisestd;
               limupr[stg] = noiseavg + kfact[1][stg] * noisestd;
               }

           for( idx = 0; idx < 6; idx++ ) {

               for( stg = 0; stg < 3; stg++ ) {
                   if( limupr[stg] <= sz[idx] && limlor[stg] >= -1.0*sz[idx] ) {
                       inside[idx] += 1;
                       break;
                       }
                   }
               if( stg == 0 || samsiz[1] == 0 )
                   stages[0][idx] += 1L;
               else if( stg == 1 || samsiz[2] == 0 )
                   stages[0][idx] += 2L;
               else
                   stages[0][idx] += 3L;

               for( stg = 0; stg < 3; stg++ ) {
                   if( limupr[stg] <= az[idx] ) {
                       lfside[idx] += 1;
                       break;
                       }
                   }
               if( stg == 0 || samsiz[1] == 0 )
                   stages[1][idx] += 1L;
               else if( stg == 1 || samsiz[2] == 0 )
                   stages[1][idx] += 2L;
               else
                   stages[1][idx] += 3L;
               }

           if( (repl+1) % 10 == 0 )
               printf( "\r\tReplication Number: %d", repl+1 );
           }

       if( repl < replics )
           replics = repl + 1;
       fprintf( outfile, "\nActual Number of Simulation Replications = %d.", replics );
       printf( "\n\n\tActual Simulation Replications = %d.", replics );

       for( idx = 0; idx < 6; idx++ ) {
           printf( "\n\tFor Nonconformance Rate = %9.7lf, ", p[idx] );
           avgstg[0] = (double)stages[0][idx] / (double)replics;
           printf( " Avg.Stages(Centering)= %4.2lf", avgstg[0] );

           printf( "\n\tFor Nonconformance Rate = %9.7lf, ", p[idx] );
           avgstg[1] = (double)stages[1][idx] / (double)replics;
           printf( " Avg.Stages(Variance) = %4.2lf", avgstg[1] );

           fprintf( outfile, "\nWhen True Nonconformance Rate = %9.7lf, ", p[idx] );
           fprintf( outfile, "\n   Avg. Stages (Centering) = %4.2lf", avgstg[0] );
           fprintf( outfile, "\n & Avg. Stages (Variance)  = %4.2lf", avgstg[1] );
           }

       printf( "\n\n\tPress a Key to Continue..." );
       keyget( &c );

       printf( "\n\n\tSimulated Acceptance/Yield Percentages:" );
       fprintf( outfile, "\n\nSimulated Acceptance/Yield Percentages:" );
       for( idx = 0; idx < 6; idx++ ) {
            vyield = 100.0 * (double)inside[idx] / (double)replics;
            printf( "\n\tWhen True Nonconformance Rate = %9.7lf, ", p[idx] );
            printf( "V-Yield =%6.2lf%% at +/-%5.3lf", vyield, sz[idx] );
            fprintf( outfile, "\nWhen True Nonconformance Rate = %9.7lf, ", p[idx] );
            fprintf( outfile, "V-Yield =%6.2lf%% at +/-%5.3lf", vyield, sz[idx] );
            cyield = 100.0 * (double)lfside[idx] / (double)replics;
            printf( "\n\tWhen True Nonconformance Rate = %9.7lf, ", p[idx] );
            printf( "C-Yield =%6.2lf%% below %5.3lf", cyield, az[idx] );
            fprintf( outfile, "\nWhen True Nonconformance Rate = %9.7lf, ", p[idx] );
            fprintf( outfile, "C-Yield =%6.2lf%% below %5.3lf", cyield, az[idx] );
            }
       fprintf( outfile, "\n\n" );

       if( csvfile != NULL ) {
            fprintf( csvfile, "1-, 2-, 3-Stage Sample Sizes will be, %d, %d, %d.",
                samsiz[0], samsiz[1], samsiz[2] );
            fprintf( csvfile, "\nCorresponding, Numerical Values, of Upper K =, %5.2lf, %5.2lf, %5.2lf.",
                kfact[1][0], kfact[1][1], kfact[1][2] );
            fprintf( csvfile, "\nCorresponding, Numerical Values, of Lower K =, %5.2lf, %5.2lf, %5.2lf.",
                kfact[0][0], kfact[0][1], kfact[0][2] );
            fprintf( csvfile, "\n\np,PaC,PaV,ZC,ZVlower,ZVupper" );
            for( idx = 0; idx < 6; idx++ ) {
                vyield = 100.0 * (double)inside[idx] / (double)replics;
                cyield = 100.0 * (double)lfside[idx] / (double)replics;
                fprintf( csvfile, "\n%9.7lf,%6.2lf,%6.2lf,%5.3lf,%5.3lf,%5.3lf",
                    p[idx], cyield, vyield, az[idx], -sz[idx], sz[idx] );
                }
            fprintf( csvfile, "\n\n" );
            }

       printf( "\n\n\t\tPress the Q key to QUIT now..." );
       printf( "\n\t\tPress a different Key to Start Another Simulation..." );
       keyget( &c );
       if( c == 'Q' || c == 'q' )
           break;
       }

   if( outfile != NULL ) {
           printf( "\n\nREMINDER:\n" );
           printf( "\nOCINTRVL created an Output File named... %s\n", outnam );
           printf( "\n\tUse the DOS invocation... TYPE %s | MORE", outnam );
           printf( "\n\tto review detailed computational results from" );
           printf( "\n\tthis OCINTRVL run.\n" );
           }

   if( csvfile != NULL ) {
           printf( "\n\nOCINTRVL created a Comma Separated Value file... %s\n", csvnam );
           printf( "\n\tUse Microsoft Excel to import this file and" );
           printf( "\n\tgraph the Operating Characteristic curve.\n" );
       }

   printf( "\n" );
   exit( 0 );
   }

