/* daubwave - This program decomposes a data set by orthogonal wavelets.
	      The results are then filtered by setting to zero a number
	      of iterations from the low end or from the high end.
	      The data is then reconstructed from the modified data.	*/

#include<stdio.h>
#include<stdlib.h>
/* 	stdlib.h is not necessary on all platforms - remove if it causes
	a compiler error and try again					*/
#include<string.h>

#define MAXNUM 4096

void VectorCopy( float *, float *, long, int );
void Wavelet( float *, long, int, double *, double * );
void VectorPermute( float *, float *, long, int );
void VectorShift( float *, long, int, int );
FILE *FileOpen( char *, char * );
long FileLoad( float *, struct opt );
void FileSave( float *, int, long, struct opt );
void *MemAlloc( int, long );
void TranCoef( double *, double *, int );
long SpanData( int *, long );
char HyperHex( int );
void HelpFile( void );

struct opt {   char infile[81];
	       char outfile[81];
	       char root[81];
	       char extension[5];
	       int filter;
	       int compress;
	       char action;
	       int level;
	       int shift;
	       double normal;
	       int unix;
	       int rotate; };

struct opt InitOption( int, char *[] );

double coef[11][20] = {      {	1.000000000000, 0.000000000000,
				0.000000000000, 0.000000000000,
				0.000000000000, 0.000000000000,
				0.000000000000, 0.000000000000,
				0.000000000000, 0.000000000000,
				0.000000000000, 0.000000000000,
				0.000000000000, 0.000000000000,
				0.000000000000, 0.000000000000,
				0.000000000000, 0.000000000000,
				0.000000000000, 0.000000000000  },

			     {	0.707106781187, 0.707106781187,
				0.000000000000, 0.000000000000,
				0.000000000000, 0.000000000000,
				0.000000000000, 0.000000000000,
				0.000000000000, 0.000000000000,
				0.000000000000, 0.000000000000,
				0.000000000000, 0.000000000000,
				0.000000000000, 0.000000000000,
				0.000000000000, 0.000000000000,
				0.000000000000, 0.000000000000  },

			    {	0.482962913145, 0.836516303738,
				0.224143868042, -0.129409522551,
				0.000000000000, 0.000000000000,
				0.000000000000, 0.000000000000,
				0.000000000000, 0.000000000000,
				0.000000000000, 0.000000000000,
				0.000000000000, 0.000000000000,
				0.000000000000, 0.000000000000,
				0.000000000000, 0.000000000000,
				0.000000000000, 0.000000000000  },

			    {	0.332670552950, 0.806891509311,
				0.459877502118, -0.135011020010,
				-0.085441273882, 0.035226291882,
				0.000000000000, 0.000000000000,
				0.000000000000, 0.000000000000,
				0.000000000000, 0.000000000000,
				0.000000000000, 0.000000000000,
				0.000000000000, 0.000000000000,
				0.000000000000, 0.000000000000,
				0.000000000000, 0.000000000000  },

			    {	0.230377813309, 0.714846570553,
				0.630880767930, -0.027983769417,
				-0.187034811719, 0.030841381836,
				0.032883011667, -0.010597401785,
				0.000000000000, 0.000000000000,
				0.000000000000, 0.000000000000,
				0.000000000000, 0.000000000000,
				0.000000000000, 0.000000000000,
				0.000000000000, 0.000000000000,
				0.000000000000, 0.000000000000  },

			    {	0.160102397974, 0.603829269797,
				0.724308528438, 0.138428145901,
				-0.242294887066, -0.032244869585,
				0.077571493840, -0.006241490213,
				-0.012580751999, 0.003335725285,
				0.000000000000, 0.000000000000,
				0.000000000000, 0.000000000000,
				0.000000000000, 0.000000000000,
				0.000000000000, 0.000000000000,
				0.000000000000, 0.000000000000  },

			    {	0.111540743350, 0.494623890398,
				0.751133908021, 0.315250351709,
				-0.226264693965, -0.129766867567,
				0.097501605587, 0.027522865530,
				-0.031582039318, 0.000553842201,
				0.004777257511, -0.001077301085,
				0.000000000000, 0.000000000000,
				0.000000000000, 0.000000000000,
				0.000000000000, 0.000000000000,
				0.000000000000, 0.000000000000  },

			    {	0.077852054085, 0.396539319482,
				0.729132090846, 0.469782287405,
				-0.143906003929, -0.224036184994,
				0.071309219267, 0.080612609151,
				-0.038029936935, -0.016574541631,
				0.012550998556, 0.000429577973,
				-0.001801640704, 0.000353713800,
				0.000000000000, 0.000000000000,
				0.000000000000, 0.000000000000,
				0.000000000000, 0.000000000000  },

			    {	0.054415842243, 0.312871590914,
				0.675630736297, 0.585354683654,
				-0.015829105256, -0.284015542962,
				0.000472484574, 0.128747426620,
				-0.017369301002, -0.044088253931,
				0.013981027917, 0.008746094047,
				-0.004870352993, -0.000391740373,
				0.000675449406, -0.000117476784,
				0.000000000000, 0.000000000000,
				0.000000000000, 0.000000000000  },

			    {	0.038077947364, 0.243834674613,
				0.604823123690, 0.657288078051,
				0.133197385825, -0.293273783279,
				-0.096840783223, 0.148540749338,
				0.030725681479, -0.067632829061,
				0.000250947115, 0.022361662124,
				-0.004723204758, -0.004281503682,
				0.001847646883, 0.000230385764,
				-0.000251963189, 0.000039347320,
				0.000000000000, 0.000000000000  },

			    {	0.026670057901, 0.188176800078,
				0.527201188932, 0.688459039454,
				0.281172343661, -0.249846424327,
				-0.195946274377, 0.127369340336,
				0.093057364604, -0.071394147166,
				-0.029457536822, 0.033212674059,
				0.003606553567, -0.010733175483,
				0.001395351747, 0.001992405295,
				-0.000685856695, -0.000116466855,
				0.000093588670, -0.000013264203  }   };


void main(int argc, char *argv[])
{
   long i;
   int j;
   float *data;
   float *buf;
   long num;
   long nn;
   long width;
   long offset;
   struct opt opt;
   double ecoef[20];
   double ocoef[20];
   int wvshft[11][2] = {     { 0, 0 },
			     { 0, 0 },
			     { 0, 1 },
			     { 0, 2 },
			     { 0, 3 },
			     { 1, 3 },
			     { 1, 4 },
			     { 1, 5 },
			     { 1, 6 },
			     { 1, 7 },
			     { 1, 8 }	};

   /* load in the command line options from the application window */

   if( argc < 2 )
      {
      fprintf( stderr, "\nDAUBWAVE <Input file> -(options)\n" );
      fprintf( stderr, "   Options:  -o <Output file> -d# -n# -t# -i# -s# -l# -h# -b# -k# -r -1 -c -u\n" );
      fprintf( stderr, "   Use -? for a help file.\n" );
      exit( 1 );
      }
   else
      {
      opt = InitOption( argc, argv );
      }

   /* allocate the memory for storing the data and result vectors */

   data = (float *)MemAlloc( sizeof(float), MAXNUM );
   buf = (float *)MemAlloc( sizeof(float), MAXNUM + 2*(opt.filter-1) );

   /* load in the data */

   num = FileLoad( data, opt );

   /* determine the largest power of two that covers the data */

   nn = SpanData( &opt.level, num );

   /* set up the coefficients for the transform */

   TranCoef( ecoef, ocoef, opt.filter );

   /* apply normalization */

   for( j=0; j<opt.filter*2; j++ )
      {
      ecoef[j] *= opt.normal;
      ocoef[j] *= opt.normal;
      }

   /* if -1 is used shift the data one to the left */

   if( (opt.shift == 1) && (opt.action != 'i') )
      VectorShift( data, nn, 1, -1 );

   /*    Go through the passes of the wavelet filter and save the
	 high frequency component in the lower half of data and the
	 low frequency component in the upper half of data		*/

   width = nn;
   for( j=0; j<opt.level; j++ )
      {

      /* avoid decomposition if only wanting to do the inverse transform */

      if( opt.action != 'i' )
	 {

	 /* perform the wavelet transform */

	 VectorCopy( buf, data, width, 2*(opt.filter-1) );
	 Wavelet( buf, width, 2*opt.filter, ecoef, ocoef );
	 VectorPermute( data, buf, width, 0 );

	 /* rotate the low and high frequency numbers so they are
	    registered properly 					*/

	 if( opt.rotate )
	    {
	    VectorShift( data, width/2, wvshft[opt.filter][0], 1 );
	    VectorShift( data+width/2, width/2, wvshft[opt.filter][1], 1 );
	    }
	 }

      /* shift to the upper half so the decomposition can be done again */

      width /= 2;
      }

   /* if only transform results are requested save and quit */

   if( opt.action == 't' )
      {
      FileSave( data, opt.level, nn, opt );
      exit( 0 );
      }

   if( opt.action == 's' )
      {
      FileSave( data, 0, width, opt );
      offset = width;
      for( j=0; j<opt.level; j++ )
	 {
	 FileSave( data + offset, j+1, offset, opt );
	 offset *= 2;
	 }
      exit( 0 );
      }

   /* if filtering remove either the high or low frequency component  */

   if( (opt.action == 'h') || (opt.action == 'b') )
      {
      for( i=0; i<width; i++ )
	 data[i] = 0.;
      }

   if( opt.action == 'b' )
      {
      for( i=width*2; i<nn; i++ )
	 data[i] = 0.;
      }

   if( opt.action == 'k' )
      {
      for( i=width; i<width*2; i++ )
	 data[i] = 0.;
      }

   if( opt.action == 'l' )
      {
      for( i=width; i<nn; i++ )
	 data[i] = 0.;
      }

   /* set up the inverse coefficients */

   TranCoef( ecoef, ocoef, -opt.filter );

   /* apply normalization */

   for( j=0; j<opt.filter*2; j++ )
      {
      ecoef[j] /= opt.normal;
      ocoef[j] /= opt.normal;
      }

   /* reconstruct the data set from the wavelet results			*/

   for( j=0; j<opt.level; j++ )
      {

      /* expand to the next largest scale to decomposite again */

      width *= 2;

      /* shift the low and high frequency components for reintegration */

      if( opt.rotate )
	 {
	 VectorShift( data, width/2, wvshft[opt.filter][0], -1 );
	 VectorShift( data+width/2, width/2, wvshft[opt.filter][1], -1 );
	 }

      /* perform the inverse wavelet transform				*/

      VectorPermute( buf, data, -width, -2*(opt.filter-1) );
      Wavelet( buf, width, 2*opt.filter, ecoef, ocoef );
      VectorCopy( data, buf, width, 0 );
      }

   /* if -1 is used shift the data one to the left */

   if( opt.shift == 1 )
      VectorShift( data, nn, 1, 1 );

   /* send the results to the output */

      FileSave( data, opt.level, num, opt );
}

/* VectorCopy - This routine copies the data from one vector to another.
		If the padding is positive that number of values are
		copied from the left side of the vector and tagged onto
		the right side of the vector.  If negative then the
		right values are added to the left side of the vector.	*/

void VectorCopy( float *out, float *in, long length, int padding )
{
   long i;

   if( padding < 0 )
      {
      padding *= -1;

      for( i=0; i<length; i++ )
	 out[padding + i] = in[i];

      /* i%length is used to allow the data to be repeated multiple times
	 if it is shorter than padding */

      for( i=0; i<padding; i++ )
	 out[padding - 1 - i] = out[padding + length - 1 - i%length];
      }
   else
      {
      for( i=0; i<length; i++ )
	 out[i] = in[i];

      /* i%length is used to allow the data vector to be continually repeated
	 if it is shorter than padding */

      for( i=0; i<padding; i++ )
	 out[length+i] = out[i%length];
      }
}

/* Wavelet - This routine calculates the wavelet decomposition and
	     returns the values properly registered.  The high frequency
	     component is saved in the first half of the vector and the
	     low frequency is saved in the last half of the vector.	*/

void Wavelet( float *buf, long length, int width, double *ecoef,
							double *ocoef )
{
   long i;
   int j;
   int times;
   float even, odd;

   /* begin the loop through the data */

   for( i=0; i<length; i += 2 )
      {
      even = 0.;
      odd = 0.;

      for( j=0; j<width; j++ )
	 {
	 even += ecoef[j]*buf[i+j];
	 odd += ocoef[j]*buf[i+j];
	 }
      buf[i] = even;
      buf[i+1] = odd;
      }
}

/* VectorPermute - This routine takes one vector and permutes it and
		   places it in a second vector.  Alternate values are
		   placed together when the length is positive and
		   alternate values are split up if the length is negative.
		   If padding is negative then the right most values are
		   padded to the left of the vector else the left most
		   values are padded to the right of the vector.	*/

void VectorPermute( float *out, float *in, long length, int padding )
{
   long i;
   long halflength;
   int offset;

   if( length < 0 )
      halflength = -length/2;
   else
      halflength = length/2;

   if( padding < 0 )
      offset = -padding;
   else
      offset = 0;

   if( length < 0 )
      {
      /* place values alternately */

      length *= -1;
      for( i=0; i<halflength; i++ )
	 {
	 /* copy the even index - low pass result */

	 out[offset + i*2] = in[i];

	 /* copy the odd index - high pass result */

	 out[offset + i*2 + 1] = in[i + halflength];
	 }
      }
   else
      {
      /* split the alternate values up */

      for( i=0; i<halflength; i++ )
	 {
	 out[offset + i] = in[i*2];
	 out[offset + i + halflength] = in[i*2+1];
	 }
      }

   if( padding < 0 )
      {
      /* pad the left side */

      padding *= -1;

      /* i%length is used to allow the data to be repeated multiple times
	 if it is shorter than padding */

      for( i=0; i<padding; i++ )
	 out[padding - 1 - i] = out[padding + length - 1 - i%length];
      }
   else
      {
      /* pad the right side */

      /* i%length is used to allow the data vector to be continually repeated
	 if it is shorter than padding */

      for( i=0; i<padding; i++ )
	 out[length+i] = out[i%length];
      }
}

/* Vec_Shift - This routine shifts a vector either right or left depending
	       on the value of dir.  The left-over data values are then
	       wrapped around to the other end.				*/

void VectorShift( float *data, long length, int shift, int dir )
{
   float bfs[20];  /* is good up to D40 filter */
   long i;

   /* if shift distance exceeds the length then reduce the shift */

   shift %= length;

   if( dir == 1 )      /* rotate upward */
      {
      for( i=0; i<shift; i++ )
	 bfs[i] = data[length-shift+i];
      for( i=0; i<length-shift; i++ )
	 data[length-i-1] = data[length-shift-i-1];
      for( i=0; i<shift; i++ )
	 data[i] = bfs[i];
      }
   if( dir == -1 )      /* rotate downward */
      {
      for( i=0; i<shift; i++ )
	 bfs[i] = data[i];
      for( i=0; i<length-shift; i++ )

	 data[i] = data[i+shift];
      for( i=0; i<shift; i++ )
	 data[length-shift+i] = bfs[i];
      }
}

/* SpanData - This routine calculates the minimum length which is spans
	      the data and is also a power of 2.			*/

long SpanData( int *iterates, long number )
{
   int loops = 0;
   long nn = 1;

   while( 1 )
      {
      if( nn >= number )
	 break;
      nn <<= 1;
      loops++;
      }

   /* the analysis can not exceed the scale of the data */

   if( *iterates == 0 )
      {
      *iterates = loops;
      }
   else
      {
      if( *iterates > loops )
	 {
	 fprintf( stderr, "Data not long enough for the number of iterations specified\n" );
	 fprintf( stderr, "Max # of iterations possible is %d.\n", loops );
	 exit( 1 );
	 }
      }

   return( nn );
}

/* FileOpen - This routine opens up a file and checks for errors */

FILE *FileOpen( char *file, char *opt )
{
   FILE *ptr;

   if( (ptr = fopen( file, opt )) == NULL )
      {
      fprintf( stderr, "Can not open the file %s\n", file );
      exit( 1 );
      }
   return( ptr );
}

/* File_Load - This routine loads in a data file of floating point
	       numbers.  If the file does not start with a floating
	       point number, then the file is searched for the first
	       line that begins with a floating point number.  The
	       value returned is the number of points read.		*/
long FileLoad( float *data, struct opt opt )
{
   FILE *in;
   long i;
   long len;
   float tmp;

   /* get the input from stdin if the unix option is specified */

   if( opt.unix )
      {
      i = 0;
      while( i < MAXNUM )
	 {
	 if( fscanf( stdin, "%g", &tmp ) == EOF )
	    break;
	 data[i++] = tmp;
	 }
      return( i );
      }

   /* open the input file */

   if( opt.compress )
      {
      in = FileOpen( opt.infile, "rb" );
      fseek( in, 0L, 2 );
      len = ftell( in );
      fseek( in, 0L, 0 );
      if( len%sizeof( float ) )
	 {
	 fprintf( stderr, "Error in reading input file.\n" );
	 exit( 1 );
	 }
      else
	 {
	 i = len/sizeof( float );
	 if( i > MAXNUM )
	    {
	    fprintf( stderr, "Input file is too large.\n" );
	    exit( 1 );
	    }
	 else
	    {
	    if( fread( data, sizeof( float ), i, in ) != i )
	       {
	       fprintf( stderr, "Error reading input file.\n" );
	       exit( 1 );
	       }
	    }
	 }
      }
   else
      {
      in = FileOpen( opt.infile, "r" );
      i = 0;
      while( i < MAXNUM )
	 {
	 if( fscanf( in, "%g", &tmp ) == EOF )
	    break;
	 data[i++] = tmp;
	 }
      }

   fclose( in );
   return( i );
}

/* FileSave - This routine saves the results to the output using the
	      appropriate extension.					*/

void FileSave( float *data, int level, long length, struct opt opt )
{
   long i;
   FILE *out;

   /* send the output to stdout if the unix option is specified */

   if( opt.unix )
      {
      for( i=0; i<length; i++ )
	 {
	 if( fprintf( stdout, "%g\n", data[i] ) == EOF )
	    {
	    fprintf( stderr, "Error writing output file.\n" );
	    exit( 1 );
	    }
	 }
      return;
      }

   /* construct the output name */

   if( strlen( opt.root ) != NULL )
      {
      opt.extension[0] = '.';

      /* use the action type for the first byte */

      if( opt.action == 's' )
	 {
	 opt.extension[1] = HyperHex( opt.level );
	 }
      else
	 {
	 opt.extension[1] = opt.action;
	 }

      /* use the wavelet type for the second byte */

      opt.extension[2] = HyperHex( 2*opt.filter );

      /* use the level of the decomposition for the third byte */

      opt.extension[3] = HyperHex( level );
      opt.extension[4] = 0x00;
      strcpy( opt.outfile, opt.root );
      strcat( opt.outfile, opt.extension );
      }

   /* open the output file and save */

   if( opt.compress )
      {
      out = FileOpen( opt.outfile, "wb" );
      if( fwrite( data, sizeof( float ), length, out ) != length )
	 {
	 fprintf( stderr, "Error writing ouput file.\n" );
	 exit( 1 );
	 }
      }
   else
      {
      out = FileOpen( opt.outfile, "w" );
      for( i=0; i<length; i++ )
	 {
	 if( fprintf( out, "%g\n", data[i] ) == EOF )
	    {
	    fprintf( stderr, "Error writing output file.\n" );
	    exit( 1 );
	    }
	 }
      }
   fclose( out );
}

/* MemAlloc - This routine allocates memory, checks for errors and
	      sets it equal to zero.					*/

void *MemAlloc( int size, long length )
{
   char *ptr;
   long i;

   ptr = (char *)malloc( size*length );
   if( ptr == NULL )
      {
      fprintf( stderr, "Error in allocating memory.\n" );
      exit( 1 );
      }

   /* clear the memory */

   for( i=0; i<size*length; i++ )
      {
      ptr[i] = 0x00;
      }

   return( (void *)ptr );
}

/* TranCoef - This routine sets up the coefficients for the wavelet
	      transform.  If filt is negative then the coefficients are
	      set up to do the inverse transform.			*/

void TranCoef( double *even, double *odd, int filter )
{
   int i;

   if( filter > 0 )
      {
      for( i=0; i<2*filter; i++ )
	 {
	 /* low pass */

	 even[i] = coef[filter][i];

	 /* high pass */

	 odd[i] = coef[filter][2*filter - i -1];
	 if( i%2 )
	    odd[i] *= -1;
	 }
      }
   else
      {
      filter *= -1;
      for( i=0; i<filter; i++ )
	 {

	 /* inverse even index */

	 even[i*2] = coef[filter][filter*2 - 2*(i+1)];
	 even[i*2+1] = coef[filter][2*i+1];

	 /* inverse odd index */

	 odd[i*2] = coef[filter][filter*2 - (2*i+1)];
	 odd[i*2+1] = -coef[filter][i*2];
	 }
      }
}

/* InitOption - This routine takes the command line options and sets the
		appropriate values within the structure 'opt'		*/

struct opt InitOption( int argc, char *argv[] )
{
   int i;
   int lev;
   char *ptr;
   struct opt opt;

   /* set up the default values */

   strcpy( opt.infile, "noname.dat" );
   strcpy( opt.outfile, "" );
   strcpy( opt.root, "" );
   opt.filter = 2;
   opt.compress = 0;
   opt.normal = 1.;
   opt.action = 't';
   opt.level = 0;
   opt.shift = 0;
   opt.unix = 0;
   opt.rotate = 0;

   for( i=1; i<argc; i++ )
      {
      if( argv[i][0] == '-' )
	 {
	 switch( argv[i][1] )
	    {
	    case 'o':
	    case 'O':  strcpy( opt.outfile, argv[i+1] );
		       i++;
		       break;
	    case '?':  HelpFile();
		       exit( 0 );
		       break;
	    case '1':  opt.shift = 1;
		       break;
	    case 'n':
	    case 'N':  lev = atoi( &argv[i][2] );
                       if( (lev < 1) || (lev > 3) )
                          {
                          fprintf( stderr, "%s is not a valid option.\n", argv[i] );
                          exit( 1 );
                          }
                       if( atoi( &argv[i][2] ) != 2 )
			  opt.normal = 1.41421356237309504880;
		       if( atoi( &argv[i][2] ) > 2 )
			  opt.normal = 1/opt.normal;
		       break;
	    case 'u':
	    case 'U':  opt.unix = 1;
		       break;
	    case 'r':
	    case 'R':  opt.rotate = 1;
		       break;
	    case 'd':
	    case 'D':  opt.filter = atoi( &argv[i][2] );
		       if( (opt.filter%2 == 1) || (opt.filter < 2) ||
						  (opt.filter > 20) )
			  {
			  fprintf( stderr, "%s is not a valid option.\n", argv[i] );
			  exit( 1 );
			  }
		       else
			  {
			  opt.filter /= 2;
			  }
		       break;
	    case 'c':
	    case 'C':  opt.compress = 1;
		       break;
	    case 'l':
	    case 'L':  opt.action = 'l';
		       lev = atoi( &argv[i][2] );
		       if( lev > 0 )
			  opt.level = lev;
		       break;
	    case 'h':
	    case 'H':  opt.action = 'h';
		       lev = atoi( &argv[i][2] );
		       if( lev > 0 )
			  opt.level = lev;
		       break;
	    case 'b':
	    case 'B':  opt.action = 'b';
		       lev = atoi( &argv[i][2] );
		       if( lev > 0 )
			  opt.level = lev;
		       break;
	    case 'k':
	    case 'K':  opt.action = 'k';
		       lev = atoi( &argv[i][2] );
		       if( lev > 0 )
			  opt.level = lev;
		       break;
	    case 'i':
	    case 'I':  opt.action = 'i';
		       lev = atoi( &argv[i][2] );
		       if( lev > 0 )
			  opt.level = lev;
		       break;
	    case 't':
	    case 'T':  opt.action = 't';
		       lev = atoi( &argv[i][2] );
		       if( lev > 0 )
			  opt.level = lev;
		       break;
	    case 's':
	    case 'S':  opt.action = 's';
		       lev = atoi( &argv[i][2] );
		       if( lev > 0 )
			  opt.level = lev;
		       break;
	    default : fprintf( stderr, "%s not a valid option.\n", argv[i] );
		      exit( 1 );
	    }
	 }
      else
	 {
	 strcpy( opt.infile, argv[i] );
	 }
      }

   /* set up the root and extension of the output files */

   if( strlen( opt.outfile ) == 0 )
      {
      strcpy( opt.outfile, opt.infile );
      ptr = strchr( opt.outfile, '.' );
      if( ptr )
	 ptr[0] = 0x00;
      strcpy( opt.root, opt.outfile );
      }
   else
      {
      if( strchr( opt.outfile, '.' ) == NULL )
	 {
	 strcpy( opt.root, opt.outfile );
	 }
      else
	 {
	 if( opt.action == 's' )
	    {
	    fprintf( stderr, "Can not specify the output file extension with \
		     option -s\n" );
	    }
	 }
      }

   return( opt );
}

/* HyperHex - This routine takes an integer between 0 and 36 and converts
	      it into a character that goes from 0-9,A-Z.		*/

char HyperHex( int value )
{
   if( (value > 35) || (value < 0) )
      {
      fprintf( stderr, "Value too big to specify the file extension properly.\n" );
      exit( 1 );
      }
   if( value < 10 )
      {
      return( '0' + value );
      }
   else
      {
      return( 'A' + value - 10 );
      }
}

/* HelpFile - This routine prints out a help file to the screen 	*/

void HelpFile( void )
{
   printf( "DAUBWAVE - General purpose orthogonal wavelet program.\n" );
   printf( "Author: Steven Gollmer (nls@mace.cc.purdue.edu)   Date: October 23, 1992\n\n" );
   printf( "DAUBWAVE <input> (options)\t Default Options: -d4 -n2 -t\n" );
   printf( "Options:\n" );
   printf( "-o <output>\n\tSpecify the root or the whole name of the output file.\n" );
   printf( "-d#\tSpecify the wavelet transform d2 through d20.\n" );
   printf( "-n#\t1/2 Normalization. 1-On inverse, 2-Root on both, 3-On transform.\n" );
   printf( "-t#\tPerform the transform to # levels of the hierarchy.\n" );
   printf( "-i#\tPerform the inverse transform to # levels of the hierarchy.\n" );
   printf( "-s#\tSame as -t# but save each hierarchy as a separate file.\n" );
   printf( "-l#\tPerform a low pass filter on the data transformed to the # level.\n" );
   printf( "-h#\tPerform a high pass filter on the data transformed to the # level.\n" );
   printf( "-b#\tPerform a band pass filter on the data transformed to the # level.\n" );
   printf( "-k#\tPerform a notch filter on the data transformed to the # level.\n" );
   printf( "\t(Analyze to the highest level possible if # is not specified.)\n" );
   printf( "-r\tCorrelate coefficients at different levels by shifting results.\n" );
   printf( "-1\tShift the data by one before the transform is performed.\n" );
   printf( "-c\tUse IEEE binary floating point for input and output.\n" );
   printf( "-u\tImplement Unix type redirection for input and output.\n" );
   printf( "Reference:\tPress, William H., 'Numerical Recipes for Fortran, 2nd Ed.'\n" );
   printf( "\t\tNew York: Cambridge University Press, 1992.\n" );
}
