/* OXMOLES will compute the variation in the products of hydrocarbon
           combustion with the number of moles of Oxygen

   V1.0 (09/05/90): original version
   V1.1 (10/17/90): make interactive and run TPLOT
   V1.2 (08/07/91): update for CREST/V5.00
*/
#include <stdio.h>
#include <process.h>
#include <float.h>

#define elements 4
#define compounds 10
#define products (elements+compounds)
char *element[elements]={"C","H","S","O"};
char *compound[compounds]={"Odia","Cmonox","Cdiox","Hdia","Hydroxl",
              "Water","Smonox","Sdiox","Striox","Sulfuricacid"};
float rmoles[elements]={5.,4.,.1,0.};
float pfract[products];
#define O rmoles[3]

main()
  {
  FILE *file1,*file2;
  int i,l,n,N;
  char c,cbuf[81];
  #define filename static char
  filename data[]    ={"CREST.GAS"};
  filename reaction[]={"OXMOLES.REA"};
  filename spool[]   ={"OXMOLES.SPL"};
  filename output[]  ={"OXMOLES.OUT"};
  filename plot[]    ={"OXMOLES.PLT"};
  float P,T,Omin,Omax;

/* list heading */

  printf("OXMOLES/V1.2: hydrocarbon combustion products vs. moles oxygen\n");

/* get temperature from user */

  printf("enter temperature in R ");
  if(gets(cbuf)==NULL)quit("no user input");
  if(sscanf(cbuf,"%hf",&T)!=1)quit("scan error");
  breaktest();
  if(T<=0.)quit("temperature out of range");
  if(T>9000.)quit("temperature out of range");

/* get pressure from user */

  printf("enter pressure in psia ");
  if(gets(cbuf)==NULL)quit("no user input");
  if(sscanf(cbuf,"%hf",&P)!=1)quit("scan error");
  breaktest();
  if(P<=0.)quit("pressure out of range");

/* get minimum moles O from user */

  printf("enter minimum moles oxygen (try 5) ");
  if(gets(cbuf)==NULL)quit("no user input");
  if(sscanf(cbuf,"%hf",&Omin)!=1)quit("scan error");
  breaktest();
  if(Omin<=0.)quit("minimum moles oxygen out of range");

/* get maximum moles O from user */

  printf("enter maximum moles oxygen (try 20) ");
  if(gets(cbuf)==NULL)quit("no user input");
  if(sscanf(cbuf,"%hf",&Omax)!=1)quit("scan error");
  breaktest();
  if(Omax<=Omin)quit("maximum moles oxygen out of range");

/* number of steps */

  printf("The number of computations, N, is the number of times you want to run\n");
  printf("CREST, each time with a different number of moles of oxygen.  If, for\n");
  printf("instance, you selected Omin=5 and Omax=20 moles,  and  then  selected\n");
  printf("N=4, the computations would be done at O=5,10,15,20.  If you selected\n");
  printf("N=16, the computations would be done 5,6,7,...,18,19,20.\n");
  printf("enter number of computations, N (try 16) ");
  if(gets(cbuf)==NULL)quit("no user input");
  if(sscanf(cbuf,"%i",&N)!=1)quit("scan error");
  breaktest();
  if(N<2)quit("too few (minumum=2)");
  if(N>100)quit("too many (maxumum=100)");

/* open output file */

  if((file1=fopen(output,"wt"))==NULL)quit("unable to create output file");

/* step through moles of O range */

  for(n=0;n<N;n++)
    {
    O=Omin+(Omax-Omin)*(float)n/(float)(N-1);

/* test for user break */

    if(breaktest())quit("break detected at keyboard");

/* create reaction file */

    if((file2=fopen(reaction,"wt"))==NULL)quit("unable to create reaction file");
    l=fprintf(file2,"%hG%s",rmoles[0],element[0]);
    for(i=1;i<elements;i++)
      {
      l=l+fprintf(file2,"+%hG%s",rmoles[i],element[i]);
      if(l>60)
        {
        fprintf(file2,"\n");
        l=0;
        }
      }
    c='=';
    for(i=0;i<products;i++)
      {
      if(i<elements)
        l=l+fprintf(file2,"%c%s",c,element[i]);
      else
        l=l+fprintf(file2,"%c%s",c,compound[i-elements]);
      if(l>60)
        {
        fprintf(file2,"\n");
        l=0;
        }
      c='+';
      }
    fprintf(file2,"\n%c",26);
    fclose(file2);

/* remove existing spool file (just in case) */

    remove(spool);

/* run CREST and pass it the necessary files and data */

    sprintf(cbuf,"D=%s R=%s S=%s B=P:T:%hG:%hG:%hG",data,reaction,spool,T,P,T);
    if(prgexe("CREST.EXE",cbuf))quit("unable to run CREST");

/* read the results from the spool file and save in the output file */

    if((file2=fopen(spool,"rt"))==NULL)quit("unable to open spool file");
    look1:
    if(freads(file2,cbuf,81)==0)quit("read error in spool file");
    if(cbuf[0]!='=')goto look1;
    fprintf(file1,"%hG",O);
    for(i=0;i<products;i++)
      {
      if(freads(file2,cbuf,81)==0)quit("read error in spool file");
      if(sscanf(cbuf,"%*s %*s %*hf %hf",&pfract[i])!=1)
        quit("scan error in spool file");
      encode(pfract[i],cbuf);
      fprintf(file1," %s",cbuf);
      }
    fprintf(file1,"\n");
    fclose(file2);
    }

/* end and close the output file */

  fprintf(file1,"%c",26);
  fclose(file1);

/* create the plot command file */

  if((file1=fopen(plot,"wt"))==NULL)quit("unable to create plot command file");
  fprintf(file1,"TPLOT illustrate batch running CREST by OXMOLES.C\n");
  fprintf(file1,"???\n");
  fprintf(file1,"!STAND\n");
  fprintf(file1,"!MAP\n");
  fprintf(file1,"15 1 2 3 4 5 6 7 8 9 10 11 12 13 14\n");
  fprintf(file1,"!ALTER\n");
  fprintf(file1,"!PLOT\n");
  fprintf(file1,"SUPPRESS\n");
  fprintf(file1,"TICKS\n");
  fprintf(file1,"ASPECT\n");
  fprintf(file1,"2*1.25\n");
  fprintf(file1,"LOGY\n");
  fprintf(file1,"ABOVE\n");
  fprintf(file1,"OXMOLES: T=%hG^oR, P=%hG psia\n",T,P);
  fprintf(file1,"MOLES OF O\n");
  fprintf(file1,"%hG 0 %hG 0 0\n",Omin,Omax);
  fprintf(file1,"MOLE FRACTION OF PRODUCTS\n");
  fprintf(file1,"-6 1 2 2 0 0 2\n");
  fprintf(file1,"%s\n",output);
  fprintf(file1,"%i %i %i\n",products+1,products,(products+1)/2);
  for(i=0;i<products;i++)
    {
    fprintf(file1,"1 %i 1 3 %i %i\n",i+2,(i+1)%9+1,i+1);
      if(i<elements)
        fprintf(file1,"%s\n",element[i]);
      else
        fprintf(file1,"%s\n",compound[i-elements]);
    }
  fprintf(file1,"!END\n");
  fprintf(file1,"%c",26);
  fclose(file1);

/* run the plot program, TPLOT */

  prgexe("TPLOT.EXE",plot);
  printf("\nfor hardcopy or modifications, run PLOTS\n");
  }

encode(f,buf)                  /* encode and compress flt into string */
  float f;                     /* It's incredible that one must go to */
  char *buf;                   /* such lengths in order to do a simple*/
  {                            /* little thing like suppress all of   */
  char b;                      /* the unnecessary stuff in a floating */
  int i,j;                     /* number like--> 1E+007 or 1E-005 !   */
  sprintf(buf,"%hG",f);
  if(buf[0]=='0'&&buf[1]=='.')
    {
    i=1;
    while(1)
      {
      b=buf[i];
      buf[i-1]=b;
      if(b==0)break;
      i++;
      }
    }
  i=0;
  while(b=buf[i])
    {
    if(b=='E'&&(buf[i+1]=='+'||buf[i+1]=='-')&&buf[i+2]=='0')
      {
      j=i+3;
      while(1)
        {
        b=buf[j];
        buf[j-1]=b;
        if(b==0)break;
        j++;
        }
      }
    else
      i++;
    }
  i=0;
  while(b=buf[i])
    {
    if(b=='E'&&buf[i+1]=='+')
      {
      i+=2;
      while(1)
        {
        b=buf[i];
        buf[i-1]=b;
        if(b==0)break;
        i++;
        }
      break;
      }
    i++;
    }
  }

breaktest()                      /* test for an unsolicited keystroke */
  {
  if(!kbhit())return(0);
  if(!getch())getch();
  return(1);
  }

freads(stream,string,bytes)               /* It's hard to believe     */
  char *string;                           /* that one must go through */
  int bytes;                              /* all this trouble just to */
  FILE *stream;                           /* read a file!             */
  {
  char *eol,buf2[2];
  int b;
  eol=fgets(string,bytes,stream);
  if(eol==NULL)return(0);
  for(b=0;b<bytes;b++)
    {
    if(string[b]=='\n')
      {
      string[b]=0;
      return(b+1);
      }
    }
  string[bytes-1]=0;
  more:
  eol=fgets(buf2,2,stream);
  if((eol==NULL)||(!buf2[0])||(buf2[0]=='\n'))return(bytes);
  goto more;
  }

prgexe(prog,string)                                  /* run a program */
  char prog[],string[];
  {
  printf("%s %s\n",prog,string);
  return(spawnlp(0,prog,prog,string,NULL));
  }

quit(message)                            /* write a message and abort */
  char message[];
  {
  fcloseall();
  printf("\n%s\r",message);
  abort();
  }
