/* FindEnth will compute the enthalpy of the products of hydrocarbon
           combustion as a function of temperature

   V1.0 (10/17/90): original version
   V1.1 (08/07/91): update to CREST/V5.00
*/
#include <stdio.h>
#include <process.h>
#include <float.h>

#define elements 5
#define compounds 19
#define products (elements+compounds)
char *element[elements]={"C","H","S","O","N"};
char *compound[compounds]={"Odia","Cmonox","Cdiox","Hdia","Hydroxl",
              "Water","Smonox","Sdiox","Striox","Sulfuricacid","Ndia",
              "Nmonox","Ndiox","Ntriox","Nnmonox","Nndiox","Nntriox",
              "Nntetraox","Nnpentaox"};
float rmoles[elements];

main()
  {
  FILE *file1,*file2;
  int i,l,n,N;
  char c,cbuf[81];
  #define filename static char
  filename data[]    ={"CREST.GAS"};
  filename reaction[]={"FINDENTH.REA"};
  filename spool[]   ={"FINDENTH.SPL"};
  filename output[]  ={"FINDENTH.OUT"};
  filename plot[]    ={"FINDENTH.PLT"};
  float H,h,hmin,hmax,pmoles,P,T,Tmin,Tmax;

/* list heading */

  printf("FindEnth/V1.1: Find Enthalpy of hydrocarbon combustion products\n");

/* get moles C from user */

  printf("enter number moles of carbon (try 1) ");
  if(gets(cbuf)==NULL)quit("no user input");
  if(sscanf(cbuf,"%hf",&rmoles[0])!=1)quit("scan error");
  if(rmoles[0]<=0.)quit("moles out of range");

/* get moles H from user */

  printf("enter number moles of hydrogen (try %hG) ",rmoles[0]*2.2);
  if(gets(cbuf)==NULL)quit("no user input");
  if(sscanf(cbuf,"%hf",&rmoles[1])!=1)quit("scan error");
  if(rmoles[1]<=0.)quit("moles out of range");

/* get moles S from user */

  printf("enter number moles of sulfur (try %hG) ",rmoles[0]*.05);
  if(gets(cbuf)==NULL)quit("no user input");
  if(sscanf(cbuf,"%hf",&rmoles[2])!=1)quit("scan error");
  if(rmoles[2]<0.)quit("moles out of range");

/* get moles O from user */

  printf("enter number moles of oxygen (try %hG) ",
    rmoles[0]*2.+rmoles[1]*.5+rmoles[2]*2.);
  if(gets(cbuf)==NULL)quit("no user input");
  if(sscanf(cbuf,"%hf",&rmoles[3])!=1)quit("scan error");
  if(rmoles[3]<0.)quit("moles out of range");

/* set nitrogen moles from oxygen moles for 79/21 mixture */

  rmoles[4]=rmoles[3]*79./21.;

/* get minimum temperature from user */

  printf("enter minimum temperature in R ");
  if(gets(cbuf)==NULL)quit("no user input");
  if(sscanf(cbuf,"%hf",&Tmin)!=1)quit("scan error");
  if(Tmin<=0.)quit("temperature out of range");
  if(Tmin>9000.)quit("temperature out of range");

/* get maximum temperature from user */

  printf("enter maximum temperature in R ");
  if(gets(cbuf)==NULL)quit("no user input");
  if(sscanf(cbuf,"%hf",&Tmax)!=1)quit("scan error");
  if(Tmax<=Tmin)quit("temperature out of range");
  if(Tmax>9000.)quit("temperature out of range");

/* number of steps */

  printf("The number of temperatures, N, is the number of times you want to run\n");
  printf("CREST, each time with a different temperature.  If, for instance, you\n");
  printf("selected  Tmin=500R  and  Tmax=2000R,  and  then  selected N=4, the\n");
  printf("computations would be done at T=500,1000,1500,2000.  If you  selected\n");
  printf("N=16, the computations would be done 500,600,...,1900,2000.\n");
  printf("enter number of temperatures, N (try 10) ");
  if(gets(cbuf)==NULL)quit("no user input");
  if(sscanf(cbuf,"%i",&N)!=1)quit("scan error");
  if(N<2)quit("too few (minumum=2)");
  if(N>100)quit("too many (maxumum=100)");

/* get pressure from user */

  printf("enter pressure in psia ");
  if(gets(cbuf)==NULL)quit("no user input");
  if(sscanf(cbuf,"%hf",&P)!=1)quit("scan error");
  if(P<=0.)quit("pressure out of range");

/* create reaction file */

    if((file1=fopen(reaction,"wt"))==NULL)quit("unable to create reaction file");
    l=fprintf(file1,"%hG%s",rmoles[0],element[0]);
    for(i=1;i<elements;i++)
      {
      if(rmoles[i]>0.)
        {
        l=l+fprintf(file1,"+%hG%s",rmoles[i],element[i]);
        if(l>60)
          {
          fprintf(file1,"\n");
          l=0;
          }
        }
      }
    c='=';
    for(i=0;i<products;i++)
      {
      if(i<elements)
        {
        if(rmoles[i]>0.)
          l=l+fprintf(file1,"%c%s",c,element[i]);
        }
      else
        {
        if((compound[i-elements][0]!='S')||(rmoles[2]>0.))
          l=l+fprintf(file1,"%c%s",c,compound[i-elements]);
        }
      if(l>60)
        {
        fprintf(file1,"\n");
        l=0;
        }
      c='+';
      }
    fprintf(file1,"\n%c",26);
    fclose(file1);

/* open output file */

  if((file1=fopen(output,"wt"))==NULL)quit("unable to create output file");
  fprintf(file1,"elements=%hG%s",rmoles[0],element[0]);
  for(i=1;i<elements;i++)
    {
    if(rmoles[i]>0.)
      fprintf(file1,"+%hG%s",rmoles[i],element[i]);
    }
  fprintf(file1,"\nP=%hG psia\n",P);
  fprintf(file1," TR   moles    H/RTo    h/RTo\n");

/* step through temperature */

  for(n=0;n<N;n++)
    {
    T=Tmin+(Tmax-Tmin)*(float)n/(float)(N-1);

/* test for user break */

    if(breaktest())quit("break detected at keyboard");

/* remove existing spool file (just in case) */

    remove(spool);

/* run CREST and pass it the necessary files and data */

    sprintf(cbuf,"D=%s R=%s S=%s B=P:T:%hG:%hG:%hG",
      data,reaction,spool,T,P,T);
    if(prgexe("CREST.EXE",cbuf))quit("unable to run CREST");

/* read the results from the spool file and save in the output file */

    if((file2=fopen(spool,"rt"))==NULL)quit("unable to open spool file");
    look1:
    if(freads(file2,cbuf,81)==0)quit("read error in spool file");
    if(cbuf[0]!='=')goto look1;
    look2:
    if(freads(file2,cbuf,81)==0)quit("read error in spool file");
    if(cbuf[0]!='-')goto look2;
    if(freads(file2,cbuf,81)==0)quit("read error in spool file");
    if(sscanf(cbuf,"%*s %*s %hf %*hf %*hf %hf",&pmoles,&H)!=2)
      quit("scan error in spool file");
    h=H/pmoles;
    fprintf(file1,"%4.0hf %7.3hf %9.2hf %7.2hf\n",T,pmoles,H,h);
    if(n==0)
      {
      hmin=h;
      hmax=h;
      }
    else
      {
      if(h<hmin)hmin=h;
      if(h>hmax)hmax=h;
      }
    fclose(file2);
    }

/* end and close the output file */

  fprintf(file1,"\n%c",26);
  fclose(file1);

/* create the plot command file */

  if((file1=fopen(plot,"wt"))==NULL)quit("unable to create plot command file");
  fprintf(file1,"TPLOT illustrate batch running CREST by FindEnth.C\n");
  fprintf(file1,"????\n");
  fprintf(file1,"!STAND\n");
  fprintf(file1,"!MAP\n");
  fprintf(file1,"15 1 2 3 4 5 6 7 8 9 10 11 12 13 14\n");
  fprintf(file1,"!ALTER\n");
  fprintf(file1,"!PLOT\n");
  fprintf(file1,"SUPPRESS\n");
  fprintf(file1,"TICKS\n");
  fprintf(file1,"ASPECT\n");
  fprintf(file1,"2*1.25\n");
  fprintf(file1,"ABOVE\n");
  fprintf(file1,"FINDENTH: %hG%s",rmoles[0],element[0]);
  for(i=1;i<elements;i++)
    {
    if(rmoles[i]>0.)
      fprintf(file1,"+%hG%s",rmoles[i],element[i]);
    }
  fprintf(file1,", P=%hG psia\n",P);
  fprintf(file1,"TEMPERATURE ;[^oR;]\n");
  fprintf(file1,"%hG 0 %hG 0 0\n",Tmin,Tmax);
  fprintf(file1,"h/RTo ;[per mole;]\n");
  fprintf(file1,"%7.2hf 0 %7.2hf 0 0\n",hmin,hmax);
  fprintf(file1,"%s\n",output);
  fprintf(file1,"4 1 1\n");
  fprintf(file1,"1 4 1 3 2 2\n");
  fprintf(file1,"To TIMES THE SLOPE OF THIS LINE IS Cp/R\n");
  fprintf(file1,"!END\n");
  fprintf(file1,"%c",26);
  fclose(file1);

/* run the plot program, TPLOT */

  prgexe("TPLOT.EXE",plot);
  printf("\nfor hardcopy or modifications, run PLOTS\n\n");
  sprintf(cbuf,"COPY %s CON",output);
  printf(">%s\n\n",cbuf);
  system(cbuf);
  }

breaktest()                      /* test for an unsolicited keystroke */
  {
  if(!kbhit())return(0);
  if(!getch())getch();
  return(1);
  }

freads(stream,string,bytes)               /* It's hard to believe     */
  char *string;                           /* that one must go through */
  int bytes;                              /* all this trouble just to */
  FILE *stream;                           /* read a file!             */
  {
  char *eol,buf2[2];
  int b;
  eol=fgets(string,bytes,stream);
  if(eol==NULL)return(0);
  for(b=0;b<bytes;b++)
    {
    if(string[b]=='\n')
      {
      string[b]=0;
      return(b+1);
      }
    }
  string[bytes-1]=0;
  more:
  eol=fgets(buf2,2,stream);
  if((eol==NULL)||(!buf2[0])||(buf2[0]=='\n'))return(bytes);
  goto more;
  }

prgexe(prog,string)                                  /* run a program */
  char prog[],string[];
  {
  printf("%s %s\n",prog,string);
  return(spawnlp(0,prog,prog,string,NULL));
  }

quit(message)                            /* write a message and abort */
  char message[];
  {
  fcloseall();
  printf("\n%s\r",message);
  abort();
  }
