/*
 *  CCSIM...Control Chart SIMulator
 */

#include <stdio.h>
#include <math.h>
#include <signal.h>
#include <string.h>
#include <stdlib.h>
#include <time.h>
#include <esiprot1.h>  /* Essential Graphics ver2, South Mountain Software */
#include <culproto.h>  /*

/*------------------------------------------------------------------------*/
/*  CCSIM uses graphics/windowing functions from the South Mountain       */
/*  Software (Essential) Graphics ver.2 and Utilities ver.5 libraries     */
/*  (zero run-time royalty) to draw high-resolution plots on IBM-         */
/*  Compatible VGA (640x480 pixel) display screens and text-mode menus.   */
/*------------------------------------------------------------------------*/

#define K_ESC   0x01 /* get1key scan code */
#define PKEY    0x19 /* get1key scan code */
#define RKEY    0x13 /* get1key scan code */
#define SKEY    0x1f /* get1key scan code */
#define VKEY    0x2f /* get1key scan code */
#define XKEY    0x2d /* get1key scan code */
#define QKEY    0x10 /* get1key scan code */
#define LKEY    0x26 /* get1key scan code */
#define EKEY    0x12 /* get1key scan code */
#define GKEY    0x22 /* get1key scan code */
#define F08KEY  0x42 /* get1key scan code */
#define F09KEY  0x43 /* get1key scan code */
#define F10KEY  0x44 /* get1key scan code */
#define SPACEB  0x39 /* get1key scan code */

char    buf[256], ynqchar, outnam[13], zletter[100], updn[100];
double  score[100], memory[100], raws[100], truscore, truerror;
double  avgscore[2], varscore[2], minscore[2], maxscore[2];
double  errsmooth[2], ewma, ewmam1, ewmawgt, efactor;
FILE    *outfile;
int     kb, idx, cursrow, batch, total, theend, zones, output;
int     parmset, windowid, idum, savseed, zone[2][10];
int     rrcumtot, rrcum[2][8], rrops[2][8], rrule[8][100];
static  int rrspan[8] = { 1, 9, 6, 14, 3, 5, 15, 8 };
struct  tm *curtime;
time_t  *ltime, timex;
WIN_PTR w0, w1, w2, w3, w4;  /* some windows */

int     x, y, x2, y2;   /* Graphics cursor coordinates */
int     base[2];        /* Base Position...for Line Drawing */
int     xmin, ymin;     /* Mimimum x and y world or clip coordinates */
int     xmax, ymax;     /* Maximum x and y world or clip coordinates */
char    scrbuf[4000];

char    *winlgets( WIN_PTR wn, char *str, int va, char *vcl, int lmax );
double  box_mull( int *idum );
float   bucket( int *idum );
float   uniran16( int *idum );
float   uniran32( int *idum );
int     intget( int maxlen, int *val, int mode, int bell, int autr );
int     initseed( void );
time_t  exptime( time_t *ltime );
void    main( void );
void    VGAsetup( void );
void    VGArawval( void );
void    VGArrules( int rule, int obs );
void    VGAsmooth( void );
void    RunRules( int last );
void    handler( int sig );
void    setup( void );
void    summary( void );
void    zonedisp( void );

extern  int  isalpha( int );
extern  int  get1key( void );
extern  int  kbhit( void );
extern  int  grtext( int row, int col, int att, char *string );
extern  void exit( int code );

/*
 * Brute Force Box-Muller Transformation.
 */

double box_mull( idum )
int *idum; {

        static int iset=0;
        static double next;
        double rad,ang;

        if( iset == 0 ) {
            rad = sqrt( -2.0 * log( (double)bucket( idum ) ) );
            ang = 2.0 * 3.1415926536 * (double)bucket( idum );
            next = rad * cos( ang );
            iset = 1;
            return( rad * sin( ang ) );
            }
        else {
            iset=0;
            return next;
            }
        }

/*
 * Use a bucket to guard against time-series structure in the 32-bit
 * L'Ecuyer pseudo-random number generator; withdraw from the bucket
 * using the 16-bit L'Ecuyer pseudo-random number generator.
 */

float bucket( int *idum ) {

        static int iff=0;
        static float rand[100];
        int j;
        float temp;

        if( iff == 0 ) {
            iff = 1;
            for( j = 0; j < 100; j++ )
                rand[j] = uniran32( idum );
            }

        j = (int)( 100.0 * (double)uniran16( idum ) - 0.5 );
        if( j < 0 )
            j = 0;
        if( j > 99 )
            j = 99;

        temp = rand[j];
        rand[j] = uniran32( idum );
        return temp;
        }

/*
 * Portable 16 and 32-bit random number generators proposed by Pierre
 * L'Ecuyer(1988), Communications of the ACM 31, 742-749 and 774.
 */

/*
 * UNIRAN16: Static integer seed values s1, s2, and s3 must not only be
 * >= 1 but also s1 <= 32362, s2 <= 31726, and s3 <= 31656, respectively.
 */

float uniran16( int *idum ) {

        static int s1, s2, s3, ldum, iff=0;

        if( *idum < 0 || iff == 0 ) {
                ldum = abs(*idum);
                if( ldum < 1 )
                    ldum = 1;
                if( ldum > 31656 )
                    ldum = 31656;
                s1 = s2 = s3 = ldum;
                *idum = iff = 1;
                }

        ldum = s1 / 206;
        s1 =  157 * ( s1 - ldum * 206 ) - ldum * 21;
        if( s1 < 0 )
            s1 += 32363;

        ldum = s2 / 217;
        s2 =  146 * ( s2 - ldum * 217 ) - ldum * 45;
        if( s2 < 0 )
            s2 += 31727;

        ldum = s3 / 222;
        s3 =  142 * ( s3 - ldum * 222 ) - ldum * 133;
        if( s3 < 0 )
            s3 += 31657;

        ldum = s1 - s2;
        if( ldum > 706 )
            ldum -= 32362;
        ldum += s3;
        if( ldum < 1 )
            ldum += 32362;

        return (float)( (double)ldum * 3.0899e-5 );
        }

/*
 * UNIRAN32: Static long seed values s1 and s2 must not only be >= 1
 * but also s1 <= 2,147,483,562 and s2 <= 2,147,483,398, respectively.
 * Here, the initial seed is a 16-bit integer; s1 = s2 <= 32767.
 */

float uniran32( int *idum ) {

        static long s1, s2, ldum;
        static int iff=0;

        if( *idum < 0 || iff == 0 ) {
                ldum = (long)abs(*idum);
                if( ldum < (long)1 )
                    ldum = (long)1;
                s1 = s2 = ldum;
                *idum = iff = 1;
                }

        ldum = s1 / (long)53668;
        s1 =  (long)40014 * ( s1 - ldum * (long)53668 ) - ldum * (long)12211;
        if( s1 < 0 )
            s1 += (long)2147483563;

        ldum = s2 / (long)52774;
        s2 =  (long)40692 * ( s2 - ldum * (long)52774 ) - ldum * (long)3791;
        if( s2 < 0 )
            s2 += (long)2147483399;

        ldum = s1 - s2;
        if( ldum < (long)1 )
            ldum += (long)2147483562;

        return (float)( (double)ldum * 4.656613e-10 );
        }

/*
 *      exptime()...time expired running CCSIM (using the Microsoft
 *                   long time() function to read system clock.)
 */

time_t exptime( ltime )
time_t *ltime; {

    time_t etime;
    static time_t begin = (long)0;

    time( &etime );
    *ltime = etime ;
    if( begin == (long)0 ) {
        begin = etime ;
        return (long)0;
        }
    else {
        etime -= begin;
        begin = *ltime;
        return etime;
        }
    }

void handler( int sig ) {     /* Interrupt Handler */
      /* ^C Break Disabled */
      signal( SIGINT, handler );
      }

void RunRules( int last ) {     /* Detect "Run Rules" Violations */
      int i, k, code;

      k = 0;
      if( last >= batch/2 && ( truscore != 0.0 || truerror != 1.0 ) )
          k = 1;
      
      if( zletter[last] == 'O' )       /* Run Rule 1: 1 beyond A */
          rrule[0][last] = 1;
      else
          rrule[0][last] = 0;
      rrops[k][0]++;

      code = 0;                        /* Run Rule 2: 9 on same side */
      if( last > 7 ) {
          rrops[k][1]++;
          code = 1;
          for( i = 0; i < 8; i++ ) {
              if( ( raws[last-i] > efactor && raws[last-i-1] < -1.0 * efactor ) ||
                  ( raws[last-i] < -1.0 * efactor && raws[last-i-1] > efactor ) ) {
                  code = 0;
                  break;
                  }
              }
          }
      rrule[1][last] = code;

      code = 0;                        /* Run Rule 3: 6 increasing or decreasing */
      if( last > 4 ) {
          rrops[k][2]++;
          code = 1;
          for( i = 0; i < 5; i++ ) {
              if( updn[last-i] == 'E' || updn[last-i] != updn[last-i-1] ) {
                  code = 0;
                  break;
                  }
              }
          }
      rrule[2][last] = code;

      code = 0;                        /* Run Rule 4: 14 alternating up & down */
      if( last > 12 ) {
          rrops[k][3]++;
          code = 1;
          for( i = 0; i < 13; i++ ) {
              if( updn[last-i] == 'E' || updn[last-i] == updn[last-i-1] ) {
                  code = 0;
                  break;
                  }
              }
          }
      rrule[3][last] = code;

      code = 0;                        /* Run Rule 5: 2 of 3 in A+ */
      if( last > 1 ) {
          rrops[k][4]++;
          for( i = 0; i < 3; i++ )
              if( zletter[last-i] == 'A' || zletter[last-i] == 'O' )
                  code++;
          if( code >= 2 )
              code = 1;
          else
              code = 0;
          }
      rrule[4][last] = code;

      code = 0;                        /* Run Rule 6: 4 of 5 in B+ */
      if( last > 3 ) {
          rrops[k][5]++;
          for( i = 0; i < 5; i++ )
              if( zletter[last-i] != 'C' )
                  code++;
          if( code >= 4 )
              code = 1;
          else
              code = 0;
          }
      rrule[5][last] = code;

      code = 0;                        /* Run Rule 7: 15 within C */
      if( last > 13 ) {
          rrops[k][6]++;
          code = 1;
          for( i = 0; i < 15; i++ )
              if( zletter[last-i] != 'C' ) {
                  code = 0;
                  break;
                  }
          }
      rrule[6][last] = code;

      code = 0;                        /* Run Rule 8: 8 within A or B */
      if( last > 6 ) {
          rrops[k][7]++;
          code = 1;
          for( i = 0; i < 8; i++ )
              if( zletter[last-i] == 'C' || zletter[last-i] == 'O' ) {
                  code = 0;
                  break;
                  }
          }
      rrule[7][last] = code;
      }

/*
 *      initseed()...initialize pseudo-random sequence using the Microsoft
 *                   long time() function to read system clock.
 */

int initseed() {
      long ltime;
      int seed;

      time( &ltime );
      seed = (int)ltime ;
      if( seed > 0 )
          return -seed;
      else if( seed == 0 )
          return -13;
      else
          return seed;
      }

void VGAsetup() {
      char prstring[3];
      static char leftstr[] = "+3+2+1 0-1-2-3";

      if( windowid > 1 )
          grxlab( "    Shewhart Chart with Moving Average...", 7, 0, 0 );
      else if( ewmawgt > 0.0 )
          grxlab( "    Shewhart Chart with EWMA...", 7, 0, 0 );
      else
          grxlab( "    Shewhart Control Chart...", 7, 0, 0 );
      
      if( truscore != 0.0 || truerror != 1.0 ) {
          sprintf( buf, "[Second 50: Mean=%6.3lf", truscore );
          grxlab(  buf, 3, 37, 0 );
          if( windowid > 1 || ewmawgt > 0.0 )
              sprintf( buf, " & Std.Err.=%6.3lf]", errsmooth[1] );
          else
              sprintf( buf, " & Std.Err.=%6.3lf]", truerror );
          grxlab(  buf, 3, 60, 0 );
          }
      
      setview( 0, 16, 639, 479 );
      xmax = 10 * ( batch + 1 );
      xmin = -40;
      ymax =  510;
      ymin =  -10;
      setworld( xmin, ymax, xmax, ymin );
      worldon( 1 );
      
      prstring[2] = '\0'; /* initialize score labels */
      prstring[0] = leftstr[0];
      prstring[1] = leftstr[1];
      fcurloc( -35, 432 );                   /* +3 */
      fcursprt( 0, prstring, 0, 3, 0 );
      prstring[0] = leftstr[2];
      prstring[1] = leftstr[3];
      fcurloc( -35, 369 );                   /* +2 */
      fcursprt( 0, prstring, 0, 3, 0 );
      prstring[0] = leftstr[4];
      prstring[1] = leftstr[5];
      fcurloc( -35, 306 );                   /* +1 */
      fcursprt( 0, prstring, 0, 3, 0 );
      prstring[0] = leftstr[6];
      prstring[1] = leftstr[7];
      fcurloc( -35, 244 );                   /*  0 */
      fcursprt( 0, prstring, 0, 3, 0 );
      prstring[0] = leftstr[8];
      prstring[1] = leftstr[9];
      fcurloc( -35, 182 );                   /* -1 */
      fcursprt( 0, prstring, 0, 3, 0 );
      prstring[0] = leftstr[10];
      prstring[1] = leftstr[11];
      fcurloc( -35, 119 );                   /* -2 */
      fcursprt( 0, prstring, 0, 3, 0 );
      prstring[0] = leftstr[12];
      prstring[1] = leftstr[13];
      fcurloc( -35,  56 );                   /* -3 */
      fcursprt( 0, prstring, 0, 3, 0 );
      
      xmin = 0;
      worldpts( xmin, ymin, &x,  &y );
      worldpts( xmax, ymax, &x2, &y2 );
      
      worldon( 0 );
      setview( x, y2, x2, y );  /* new, smaller viewport */
      setworld( xmin, ymax, xmax, ymin );
      worldon( 1 );
      
      set_base( xmin, ymin );        /* base[0] = xmin; base[1] = ymin; */
      grmove( xmin, ymax, 3 );       /* 3 => cyan */
      grmove( xmax, ymax, 3 );
      grmove( xmax, ymin, 3 );
      grmove( xmin, ymin, 3 );
      
      y = 438;
      grline( xmin, y, xmax, y, 4 );     /* 4 => red */
      y = 375;
      grline( xmin, y, xmax, y, 3 );
      y = 313;
      grline( xmin, y, xmax, y, 3 );
      y = 250;
      grline( xmin, y, xmax, y, 2 );     /* 2 => green */
      y = 187;
      grline( xmin, y, xmax, y, 3 );
      y = 125;
      grline( xmin, y, xmax, y, 3 );
      y = 62;
      grline( xmin, y, xmax, y, 4 );
      
      /*
       * Draw vertical line separating 1st & 2nd 50 observations
       * when 2nd 50 have different mean or variance from 1st 50.
       */
      if( truscore != 0.0 || truerror != 1.0 ) {
          set_base( xmax/2, ymin );
          grmove( xmax/2, ymax, 4 );       /* 4 => red */
          }
      }

void VGArawval() {
      int j;

      for( j = 0; j < batch; j++ ) {

        if( j == 0 ) {
          x = 10;
          y = (int)( 62.5 * ( raws[j] + 4.0 ) );
          set_base( x, y );
          }
        else {
          x += 10;
          y = (int)( 62.5 * ( raws[j] + 4.0 ) );
          grmove( x, y, 7 );           /* 7 => white */
          }
        }
      }

void VGArrules( int rule, int obs ) {
      int j, lowlim;
      double rawtrun;

      /* connect and box offending observations */
      lowlim = obs - rrspan[rule];
      for( j = obs; j > lowlim; j-- ) {

        rawtrun = raws[j];
        if( rawtrun > +4.1 )
            rawtrun = +4.1;
        if( rawtrun < -4.1 )
            rawtrun = -4.1;

        if( j == obs ) {
          x = 10 * ( obs + 1 );
          y = (int)( 62.5 * ( rawtrun + 4.0 ) );
          }
        else {
          x -= 10;
          y = (int)( 62.5 * ( rawtrun + 4.0 ) );
          grmove( x, y, 14 );           /* 14 => bright yellow */
          }

        set_base( x+4, y+4 );
        grmove( x+4, y-4, 14 );
        grmove( x-4, y-4, 14 );
        grmove( x-4, y+4, 14 );
        grmove( x+4, y+4, 14 );

        set_base( x, y );
        }
      }

void VGAsmooth() {
      int j;

      for( j = 0; j < batch; j++ ) {
      
        if( j == 0 ) {
          x = 10;
          y = (int)( 62.5 * ( score[0] + 4.0 ) );
          set_base( x, y );
          }
        else {
          x += 10;
          y = (int)( 62.5 * ( score[j] + 4.0 ) );
          grmove( x, y, 3 );           /* 3 => cyan */
          }
        }
      }

void zonedisp() {

      int i, j, k;

      w4 = win_open(1,1,20,74,0,3,3,0,3," Display RANDOM Control Charts ",0,3,"",0,3);

      if( parmset == 0 ) {
          win_prtf( w4, 1, "\n\n\tYou must set Parameter values before starting simulation..." );
          win_prtf( w4, 1, "\n\n\tPress a Key to Continue..." );
          get1key();
          win_close( w4 );
          return;
          }

      zones = 1;
      if( windowid > 1 ) {
          win_prtf( w4, 1, "\n\n\tThe VGA graphics screens show a Moving Average process with" );
          win_prtf( w4, 1, "\n\tWindow Width = %d.  ", windowid );
          if( truscore != 0.0 || truerror != 1.0 ) {
              win_prtf( w4, 1, "The 1st 50 Observations have MEAN = 0.0" );
              win_prtf( w4, 1, "\n\t& STD.DEVIATION = %6.3lf.  ", errsmooth[0] );
              win_prtf( w4, 1, "Over the 2nd 50 Observations, the" );
              win_prtf( w4, 1, "\n\tMoving Moments approach MEAN=%6.3lf", truscore );
              win_prtf( w4, 1, " & STD.DEV.=%6.3lf.", errsmooth[1] );
              }
          else {
              win_prtf( w4, 1, "The underlying process is" );
              win_prtf( w4, 1, "\n\tCOMPLETELY STABLE with MEAN=0.00 and STD.DEV.=1.00." );
              win_prtf( w4, 1, "\n\tThus the Moving Average has STD.DEVIATION = %6.3lf ", errsmooth[0] );
              }
          }
      else if( ewmawgt > 0.0 ) {
          win_prtf( w4, 1, "\n\n\tThe VGA graphics screens show an EWMA process with Short-" );
          win_prtf( w4, 1, "\n\tTerm Memory Weight = %6.3lf.  ", ewmawgt );
          if( truscore != 0.0 || truerror != 1.0 ) {
              win_prtf( w4, 1, "The 1st 50 Observations have" );
              win_prtf( w4, 1, "\n\tMEAN=0.0 & STD.DEV.=%6.3lf.  ", errsmooth[0] );
              win_prtf( w4, 1, "Over the 2nd 50 Observations," );
              win_prtf( w4, 1, "\n\tthe EWMA Moments approach MEAN=%6.3lf", truscore );
              win_prtf( w4, 1, " & STD.DEV.=%6.3lf.", errsmooth[1] );
              }
          else {
              win_prtf( w4, 1, "The underlying process is" );
              win_prtf( w4, 1, "\n\tCOMPLETELY STABLE with MEAN=0.00 and STD.DEV.=1.00." );
              win_prtf( w4, 1, "\n\tThus the EWMA has STD.DEVIATION = %6.3lf ", errsmooth[0] );
              }
          }
      else {
          win_prtf( w4, 1, "\n\n\tThe VGA screens show a normally distributed process" );
          win_prtf( w4, 1,   "\n\twith statistically independent observations." );
          if( truscore != 0.0 || truerror != 1.0 ) {
              win_prtf( w4, 1, "\n\n\tThe 1st 50 Observations have MEAN= 0.00" );
              win_prtf( w4, 1, " and STD.DEVIATION= 1.00." );
              win_prtf( w4, 1, "\n\n\tThe 2nd 50 Observations have MEAN=%5.2lf", truscore );
              win_prtf( w4, 1, " and STD.DEV.=%6.3lf.", truerror );
              }
          else {
              win_prtf( w4, 1, "\n\n\tProcess is COMPLETELY STABLE" );
              win_prtf( w4, 1,   " with MEAN=0.0 and STD.DEV.=1.0." );
              }
          }

      win_prtf( w4, 1, "\n\n\tWARNING: You must press the Q, X or ESCape Key to stop." );
      win_prtf( w4, 1,   "\n\tWARNING: The Control-Break key combination has been disabled!" );
      win_prtf( w4, 1,   "\n\tWARNING: Pressing the Q, X or ESCape Key is the ONLY way out!" );

      win_prtf( w4, 1, "\n\n\tPress R Key or SPACEBAR repeatedly to see Run-Rule Violations..." );

      win_prtf( w4, 1, "\n\n\tSCREEN DUMPS to slave parallel PRINTERS on LPT1:" );
      win_prtf( w4, 1,   "\n\t  Press the L key or F10 for HP Laser/Desk Jet." );
      win_prtf( w4, 1,   "\n\t  Press the G key or F09 for IBM graphics dot matrix." );
      win_prtf( w4, 1,   "\n\t  Press the E key or F08 for Epson FX, JX or LQ." );

      win_prtf( w4, 1, "\n\n\tPress a Key to initiate Graphics..." );
      get1key();

      curtype(1,0,0);
      scrtomem( 2000, 0, scrbuf  );

      setvga();     /* assume IBM-compatible graphics adapter */
      initgraf( 18, 2, 0 );    /* Initialize vga graphics mode */
      fontinit();
      fontld( 0, "VGA8X16" ); /* Std. 8x16 Font from IBM ReadOnlyMemory */

      ewmam1 = 1.0 - ewmawgt;

      for( j = 0; j < 2; j++ )
          for( i = 0; i < 8; i++ )
              rrcum[j][i] = rrops[j][i] = 0;

      while( 1 ) {

          /* Clear out KeyBoard Buffer */
          while( kbhit() != 0 )
              get1key();

          VGAsetup();

          /* initial off-screen process history (before 1st point to be plotted.) */
          ewma = 0.0;
          for( j = 0; j < 100; j++ ) {
                  memory[j] = box_mull(&idum);
                  if( ewmawgt > 0.0 )
                      ewma = ewmawgt*memory[j] + ewmam1*ewma;
                  }

          /* on-screen values */
          total += batch;
          for( j = 0; j < batch; j++ ) {

              if( windowid > 1 ) {
                  score[j] = 0.0;
                  for( k = 0; k < windowid - 1; k++ )
                      score[j] += memory[k] = memory[k+1];
                  if( j < batch/2 )
                      score[j] += memory[windowid-1] = raws[j] = box_mull(&idum);
                  else
                      score[j] += memory[windowid-1] = raws[j] = truscore + truerror * box_mull(&idum);
                  score[j] /= (double)windowid;
                  }
              else {
                  if( j < batch/2 )
                      score[j] = raws[j] = box_mull(&idum);
                  else
                      score[j] = raws[j] = truscore + truerror * box_mull(&idum);
                  }

              if( ewmawgt > 0.0 )
                  score[j] = ewma = ewmawgt*score[j] + ewmam1*ewma;
            
              k = 0;
              if( j >= batch/2 && ( truscore != 0.0 || truerror != 1.0 ) )
                  k = 1;
              avgscore[k] += score[j];
              varscore[k] += score[j] * score[j];
              if( minscore[k] > score[j] )
                  minscore[k] = score[j];
              if( maxscore[k] < score[j] )
                  maxscore[k] = score[j];

              if( 4.0 < score[j] )
                  zone[k][9]++;
              else if( 3.0 < score[j] )
                  zone[k][8]++;
              else if( 2.0 < score[j] )
                  zone[k][7]++;
              else if( 1.0 < score[j] )
                  zone[k][6]++;
              else if( 0.0 < score[j] )
                  zone[k][5]++;
              else if( -1.0 < score[j] )
                  zone[k][4]++;
              else if( -2.0 < score[j] )
                  zone[k][3]++;
              else if( -3.0 < score[j] )
                  zone[k][2]++;
              else if( -4.0 < score[j] )
                  zone[k][1]++;
              else
                  zone[k][0]++;

              if( 3.0 < raws[j] )
                  zletter[j] = 'O';
              else if( 2.0 < raws[j] )
                  zletter[j] = 'A';
              else if( 1.0 < raws[j] )
                  zletter[j] = 'B';
              else if( -1.0 <= raws[j] )
                  zletter[j] = 'C';
              else if( -2.0 <= raws[j] )
                  zletter[j] = 'B';
              else if( -3.0 <= raws[j] )
                  zletter[j] = 'A';
              else
                  zletter[j] = 'O';

              if( j > 0 ) {
                  if( raws[j] > ( raws[j-1] + efactor ) )
                      updn[j] = 'U';
                  else if( raws[j] < ( raws[j-1] - efactor ) )
                      updn[j] = 'D';
                  else
                      updn[j] = 'E';
                  }

              RunRules( j );
              }

          rrcumtot = 0;
          for( j = 0; j < batch; j++ ) {
              k = 0;
              if( j >= batch/2 && ( truscore != 0.0 || truerror != 1.0 ) )
                  k = 1;
              for( i = 0; i < 8; i++ ) {
                  rrcum[k][i] += rrule[i][j];
                  rrcumtot += rrule[i][j];
                  }
              }

          VGArawval();

          if( ewmawgt > 0.0 || windowid > 1 )
              VGAsmooth();

          kb = get1key() >> 8;       /* Wait for user to press any key. */

          if( kb == RKEY || kb == SPACEB ) {
              if( rrcumtot == 0 ) {
                  grxlab( "    NO VIOLATIONS of the Run Rules were detected...", 2, 0, 0 );
                  kb = get1key() >> 8;
                  }
              else {
                  sprintf( buf, "    %d VIOLATIONS of the Run Rules were detected...", rrcumtot );
                  grxlab(  buf, 14, 0, 0 );
                  for( j = 0; j < batch; j++ ) {
                      for( i = 0; i < 8; i++ ) {
                          if( rrule[i][j] == 0 )
                              continue;
                          kb = get1key() >> 8;       /* Wait for user to press any key. */
                          if( kb != RKEY && kb != SPACEB )
                              rrcumtot = 0;
                          else {
                              initgraf( 18, 2, 0 );    /* Initialize vga graphics mode */
                              VGAsetup();
                              VGArawval();
                              sprintf( buf, "    VIOLATION of Run Rule No. %d at Observation No. %d...", i+1, j+1 );
                              grxlab(  buf, 14, 0, 0 );
                              VGArrules( i, j );
                              if( rrcumtot == 1 )
                                  kb = get1key() >> 8;
                              rrcumtot--;
                              }
                          if( rrcumtot <= 0 )
                              break;
                          }
                      if( rrcumtot <= 0 )
                          break;
                      }
                  }
              }

          if( kb == LKEY || kb == F10KEY ) {
                setlasrv();  /* laser vertical */
                devcdump( 3, 3, 80, 20, 1, 1, 0 );
                /* devcdump( 3 => LaserJet printer,
                             3 => size of graph...half-page, 1-shade, 100 dpi
                            80 => left size margin (graph is 640 of 800),
                            20 => top margin (graph is 480 of 1050),
                             1 => FormFeed YES,
                             1 => Copies,
                             0 => Color ); */
                /*  Dump Graph to HP LaserJet connected to LPT1.  */
                }

          if( kb == GKEY || kb == F09KEY ) {
                setprnth();  /* IBM or Epsom horizontal */
                devcdump( 1, 1, 0, 0, 1, 1, 0 );
                /*  Dump to IBM Graphics DotMatrix Printer on LPT1.  */
                }

          if( kb == EKEY || kb == F08KEY ) {
                setprnth();  /* IBM or Epsom horizontal */
                devcdump( 0, 1, 0, 0, 1, 1, 0 );
                /*  Dump to Epson FX, JX-80, LQ1500 on LPT1.  */
                }

          if( kb == K_ESC || total >= 30000 ||
              kb == QKEY  || kb == XKEY )
              break;

          initgraf( 18, 2, 0 );    /* Initialize vga graphics mode */
          }

      /* Clear out KeyBoard Buffer After (possible) Screen Dump */
      while( kbhit() != 0 )
              get1key();
      
      initgraf( 3, 0, 0 );  /* 80 col. color text */
      memtoscr( 2000, 0, scrbuf );
      curtype(1,0,0);
      win_close( w4 );
      }

void setup() {

      int k;
      parmset++;
      zones = 0;

      w3 = win_open(1,1,20,74,0,7,7,0,7," Set Parameter Values ",0,7,"",0,7);

      win_prtf( w3, 1, "\n\n\tAt colon Prompts : ...simply press ENTER to get the [default]." );

      win_prtf( w3, 1, "\n\n\tCCSIM generates normally distributed processes" );
      win_prtf( w3, 1,   "\n\twith statistically independent observations." );
      win_prtf( w3, 1, "\n\n\tThe 1st 50 Observations generated for EACH SCREEN always" );
      win_prtf( w3, 1,   "\n\thave MEAN = 0.00 and STD.DEVIATION = 1.00." );
      win_prtf( w3, 1, "\n\n\tThe MEAN value for the 2nd 50 observations my be set to any" );
      win_prtf( w3, 1,   "\n\tvalue in the Range from -3.00 to +3.00." );
      win_prtf( w3, 1,   "\n\tSpecify the 2nd 50 MEAN value. [%6.2lf] : ", truscore );
      winlgets( w3, buf, 6, "-.1234567890", 9 );  /* at most 8 digits & enter key */
      sscanf( buf, "%lf", &truscore );
      if( truscore < -3.0 )
          truscore = -3.0;
      if( truscore >  3.0 )
          truscore =  3.0;
      win_prtf( w3, 1, "\n\t2nd 50 Process MEAN = %6.2lf.", truscore );

      win_prtf( w3, 1, "\n\n\tThe STANDARD DEVIATION of the 2nd 50 observations may be set to" );
      win_prtf( w3, 1,   "\n\tany value in the Range from 0.01 to 6.00" );
      win_prtf( w3, 1,   "\n\tSpecify the 2nd 50 STD.DEVIATION. [%6.3lf] : ", truerror );
      winlgets( w3, buf, 6, ".1234567890", 9 );  /* at most 8 digits & enter key */
      sscanf( buf, "%lf", &truerror );
      if( truerror < 0.01 )
          truerror = 0.01;
      if( truerror >  6.0 )
          truerror =  6.0;
      win_prtf( w3, 1, "\n\t2nd 50 Process STD.DEVIATION = %6.3lf.", truerror );
      errsmooth[0] = 1.0;
      errsmooth[1] = truerror;

      windowid = 0;
      win_prtf( w3, 1, "\n\n\tCreate Screen Displays of AVERAGES within a Moving WINDOW?" );
      win_prtf( w3, 1,   "\n\tThe Window Width can be from 2 to 50." );
      win_prtf( w3, 1,   "\n\tSpecify WIDTH (Zero => NO WINDOW). [%d] : ", windowid );
      winlgets( w3, buf, 6, "1234567890", 3 );  /* at most 2 digits & enter key */
      sscanf( buf, "%d", &windowid );
      if( windowid < 2 || windowid > 50 )
          windowid = 0;
      if( windowid > 1 ) {
          win_prtf( w3, 1, "\n\tMoving Window Width = %d.", windowid );
          errsmooth[0] = 1.0 / sqrt( (double)windowid );
          errsmooth[1] = truerror / sqrt( (double)windowid );
          win_prtf( w3, 1, "\n\tMoving Average STD.DEVIATIONS = %6.3lf and %6.3lf.",
              errsmooth[0], errsmooth[1] );
          }
      else
          win_prtf( w3, 1, "\n\tNo Moving Average will be formed." );

      ewmawgt = 0.0;
      if( windowid == 0 ) {
          win_prtf( w3, 1, "\n\n\tCreate Screen Displays of Exponentially Weighted Moving AVERAGES?" );
          win_prtf( w3, 1,   "\n\tThe Short-Term Memory Weight can vary from 0.01 to 1.0" );
          win_prtf( w3, 1,   "\n\tSpecify WEIGHT (Zero => NO EWMA). [%6.3lf] : ", ewmawgt );
          winlgets( w3, buf, 6, ".1234567890", 9 );  /* at most 8 digits & enter key */
          sscanf( buf, "%lf", &ewmawgt );
          if( ewmawgt < 0.01 )
              ewmawgt = 0.0;
          if( ewmawgt > 1.0 )
              ewmawgt = 1.0;
          }
      if( ewmawgt > 0.0 ) {
          win_prtf( w3, 1, "\n\tEWMA Weight = %6.3lf.", ewmawgt );
          errsmooth[0] = 1.0 * sqrt( ewmawgt / ( 2.0 - ewmawgt ) );
          errsmooth[1] = truerror * sqrt( ewmawgt / ( 2.0 - ewmawgt ) );
          win_prtf( w3, 1, "\n\tEWMA STD.DEVIATIONS = %6.3lf and %6.3lf.",
              errsmooth[0], errsmooth[1] );
          }
      else
          win_prtf( w3, 1, "\n\tNo EWMA will be formed." );

      idum = 0;
      win_prtf( w3, 1, "\n\n\tInput Simulation Start-up Seed ( 0 means use your Clock ): " );
      winlgets( w3, buf, 6, "1234567890", 6 );  /* at most 5 digits & enter key */
      sscanf( buf, "%d", &idum );
      if( idum == 0 )
              idum = initseed();
      else if( idum > 0 )
              idum = -idum ;
      win_prtf( w3, 1, "\n\tEquivalent Start-Up Seed = %d.", idum );
      savseed = idum;

      /* seed bucket selection generator */
      uniran16(&savseed);
      savseed = idum;

      total = 0;
      for( k = 0; k < 2; k++ ) {
          avgscore[k] = varscore[k] = minscore[k] = maxscore[k] = 0.0;
          for( idx = 0; idx < 10; idx++ )
              zone[k][idx] = 0;
          }

      win_prtf( w3, 1, "\n\n\tPress Q to QUIT now...Other Key to Continue..." );
      if( ( get1key() >> 8 ) == QKEY ) {
          curlocat(24,0);
          printf( "\n" );
          coloreos( 2, 0 );
          exit( 0 );
          }

      win_close( w3 );
      }

void summary() {

      double stdev;

      w2 = win_open(1,1,20,74,0,2,2,0,2," Save Summary Statistics ",0,2,"",0,2);

      if( parmset == 0 || zones == 0 ) {
          win_prtf( w2, 1, "\n\n\tYou must set Parameter values and Simulate before Saving..." );
          win_prtf( w2, 1, "\n\n\tPress a Key to Continue..." );
          get1key();
          win_close( w2 );
          return;
          }

      win_prtf( w2, 1, "\n\n\tSaving Control Chart Simulation Results..." );

      if( output == 0 ) {

          win_prtf( w2, 1, "\n\n\tSpecify filename to SAVE Output [%s] : ", outnam );
          winlgets( w2, buf, 1, 0, 9 );
          if( buf[0] != '\0' && buf[0] != '\r' && buf[0] != '\n' )
                  sscanf( buf, "%s", outnam );
          outnam[ min( 8, (int)strcspn( outnam, ". " ) ) ] = '\0';
          if( outnam[0] == '\0' )
                  strcpy( outnam, "CCSIM" );
          strcat( outnam, ".ccs" );
          win_prtf( w2, 1, "\n\tThe CCSim Output file is to be: %s\n",
                  outnam );

          while( filexist( outnam ) ) {
              win_prtf( w2, 1, "\n\tOverwrite Existing %s file ? [Y|n] : ",
                  outnam );
              ynqchar = 'Y';
              winlgets( w2, buf, 6, "yYnN", 1 );
              if( buf[0] != '\0' && buf[0] != '\r' && buf[0] != '\n' )
                  sscanf( buf, "%c", &ynqchar );
              if( ynqchar == 'y' || ynqchar == 'Y' )
                  break;
              win_prtf( w2, 1, "\n\n\tRe-specify filename to SAVE Output [%s] : ",
                  outnam );
              winlgets( w2, buf, 1, 0, 9 );
              if( buf[0] != '\0' && buf[0] != '\r' && buf[0] != '\n' )
                  sscanf( buf, "%s", outnam );
              outnam[ min( 8, (int)strcspn( outnam, ". " ) ) ] = '\0';
              if( outnam[0] == '\0' )
                  strcpy( outnam, "CCSIM" );
              strcat( outnam, ".ccs" );
              }

          if( ( outfile = fopen( outnam, "w" ) ) == NULL ) {
                  win_prtf( w2, 1, "\tCannot write to Output filename : %s\n", outnam );
                  if( ( outfile = fopen( "ccsim.ccs", "w" ) ) == NULL )
                          win_prtf( w2, 1, "\tAlso cannot write to filename : ccsim.ccs\n" );
                  else {
                          win_prtf( w2, 1, "\t...using default Outfile name : ccsim.ccs\n" );
                          strcpy( outnam, "ccsim.ccs" );
                          output = 1;
                          }
                  }
          else
                  output = 1;
          }

      fprintf( outfile, "Control Chart Simulation: Parameter Set No. %d",
          parmset );
      timex = exptime( ltime );
      curtime = localtime( ltime );
      fprintf( outfile, "\nDate/Time Stamp : %s", asctime( curtime ) );
      fprintf( outfile, "\nRandom Number Seed  = %d", savseed );
      
      win_prtf( w2, 1, "\n\n\tNumber of Observations Simulated = %d.", total );
      fprintf( outfile, "\nNumber of Observations Simulated = %d.", total );

      if( truscore != 0.0 ) {
          win_prtf( w2, 1, "\n\n\t1st 50 Process MEAN =   0.00." );
          win_prtf( w2, 1,   "\n\t2nd 50 Process MEAN = %6.2lf.", truscore );
          fprintf( outfile, "\n1st 50 Process MEAN =   0.00." );
          fprintf( outfile, "\n2nd 50 Process MEAN = %6.2lf.", truscore );
          }
      else {
          win_prtf( w2, 1, "\n\n\tAll Observations have True MEAN = 0.00." );
          fprintf( outfile, "\nAll Observations have True MEAN = 0.00." );
          }

      if( windowid > 1 ) {
          win_prtf( w2, 1, "\n\tMoving Window Width = %d.", windowid );
          fprintf( outfile, "\nMoving Window Width = %d.", windowid );
          }
      if( ewmawgt > 0.0 ) {
          win_prtf( w2, 1, "\n\tEWMA Weight = %6.3lf.", ewmawgt );
          fprintf( outfile, "\nEWMA Weight = %6.3lf.", ewmawgt );
          }

      if( truscore != 0.0 || truerror != 1.0 ) {
          avgscore[0] /= (double)total / 2.0;
          win_prtf( w2, 1, "\n\tObserved 1st 50 MEAN = %6.2lf.", avgscore[0] );
          fprintf( outfile, "\nObserved 1st 50 MEAN = %6.2lf.", avgscore[0] );
          avgscore[1] /= (double)total / 2.0;
          win_prtf( w2, 1, "\n\tObserved 2nd 50 MEAN = %6.2lf.", avgscore[1] );
          fprintf( outfile, "\nObserved 2nd 50 MEAN = %6.2lf.", avgscore[1] );
          }
      else {
          avgscore[0] /= (double)total;
          win_prtf( w2, 1, "\n\tObserved Process MEAN = %6.2lf.", avgscore[0] );
          fprintf( outfile, "\nObserved Process MEAN = %6.2lf.", avgscore[0] );
          }

      if( truerror != 1.0 ) {
          win_prtf( w2, 1, "\n\n\t1st 50 Process STD.DEVIATION =   1.000." );
          win_prtf( w2, 1,   "\n\t2nd 50 Process STD.DEVIATION =  %6.3lf.", truerror );
          fprintf( outfile, "\n1st 50 Process STD.DEVIATION =   1.000." );
          fprintf( outfile, "\n2nd 50 Process STD.DEVIATION =  %6.3lf.", truerror );
          }
      else {
          win_prtf( w2, 1, "\n\n\tAll Observations have True STD.DEV. = 1.000." );
          fprintf( outfile, "\nAll Observations have True STD.DEV. = 1.000." );
          }

      if( truerror != 1.0 && windowid > 1 ) {
          win_prtf( w2, 1, "\n\t1st 50 Moving Avg. STD.DEVIATION = %6.3lf.", errsmooth[0] );
          win_prtf( w2, 1, "\n\t2nd 50 Moving Avg. Steady-State STD.DEVIATION = %6.3lf.", errsmooth[1] );
          fprintf( outfile, "\n1st 50 Moving Avg. STD.DEVIATION = %6.3lf.", errsmooth[0] );
          fprintf( outfile, "\n2nd 50 Moving Avg. Steady-State STD.DEVIATION = %6.3lf.", errsmooth[1] );
          }
      else if( windowid > 1 ) {
          win_prtf( w2, 1, "\n\tMoving Average Steady-State STD.DEVIATION = %6.3lf.", errsmooth[0] );
          fprintf( outfile, "\nMoving Average Steady-State STD.DEVIATION = %6.3lf.", errsmooth[0] );
          }

      if( truerror != 1.0 && ewmawgt > 0.0 ) {
          win_prtf( w2, 1, "\n\t1st 50 EWMA STD.DEVIATION = %6.3lf.", errsmooth[0] );
          win_prtf( w2, 1, "\n\t2nd 50 EWMA Steady-State STD.DEVIATION = %6.3lf.", errsmooth[1] );
          fprintf( outfile, "\n1st 50 EWMA STD.DEVIATION = %6.3lf.", errsmooth[0] );
          fprintf( outfile, "\n2nd 50 EWMA Steady-State STD.DEVIATION = %6.3lf.", errsmooth[1] );
          }
      else if( ewmawgt > 0.0 ) {
          win_prtf( w2, 1, "\n\tEWMA Steady-State STD.DEVIATION = %6.3lf.", errsmooth[0] );
          fprintf( outfile, "\nEWMA Steady-State STD.DEVIATION = %6.3lf.", errsmooth[0] );
          }

      if( truscore != 0.0 || truerror != 1.0 ) {
          varscore[0] -= avgscore[0] * avgscore[0] * (double)total / 2.0;
          stdev = sqrt( 2.0 * varscore[0] / (double)( total - 2 ) );
          win_prtf( w2, 1, "\n\tObserved 1st 50 STD.DEVIATION = %6.3lf.", stdev );
          fprintf( outfile, "\nObserved 1st 50 STD.DEVIATION = %6.3lf.", stdev );
          varscore[1] -= avgscore[1] * avgscore[1] * (double)total / 2.0;
          stdev = sqrt( 2.0 * varscore[1] / (double)( total - 2 ) );
          win_prtf( w2, 1, "\n\tObserved 2nd 50 STD.DEVIATION = %6.3lf.", stdev );
          fprintf( outfile, "\nObserved 2nd 50 STD.DEVIATION = %6.3lf.", stdev );
          }
      else {
          varscore[0] -= avgscore[0] * avgscore[0] * (double)total;
          stdev = sqrt( varscore[0] / (double)( total - 1 ) );
          win_prtf( w2, 1, "\n\tObserved Process STD.DEVIATION = %6.3lf.", stdev );
          fprintf( outfile, "\nObserved Process STD.DEVIATION = %6.3lf.", stdev );
          }

      win_prtf( w2, 1, "\n\n\tPress a Key to Continue..." );
      get1key();

      if( truscore != 0.0 || truerror != 1.0 ) {
          win_prtf( w2, 1, "\n\n\tStatistics for the 1st 50 Observations on each plot." );
          fprintf( outfile, "\n\nStatistics for the 1st 50 Observations on each plot." );
          total /= 2;
          }

      win_prtf( w2, 1, "\n\n\tObserved MAXIMUM = %6.3lf.", maxscore[0] );
      fprintf( outfile, "\nObserved MAXIMUM = %6.3lf.", maxscore[0] );
      
      win_prtf( w2, 1, "\n\tPercentage >=  4          was %6.1lf.",
          100.0 * (double)zone[0][9] / (double)total );
      fprintf( outfile, "\nPercentage >=  4          was %6.1lf.",
          100.0 * (double)zone[0][9] / (double)total );
      for( idx = 8; idx > 0; idx-- ) {
          win_prtf( w2, 1, "\n\tPercentage >= %2d and < %2d was %6.1lf.",
              idx-5, idx-4, 100.0 * (double)zone[0][idx] / (double)total );
          fprintf( outfile, "\nPercentage >= %2d and < %2d was %6.1lf.",
              idx-5, idx-4, 100.0 * (double)zone[0][idx] / (double)total );
          }
      win_prtf( w2, 1, "\n\tPercentage  < -4          was %6.1lf.",
          100.0 * (double)zone[0][0] / (double)total );
      fprintf( outfile, "\nPercentage  < -4          was %6.1lf.",
          100.0 * (double)zone[0][0] / (double)total );
      
      win_prtf( w2, 1, "\n\tObserved MINIMUM = %6.3lf.", minscore[0] );
      fprintf( outfile, "\nObserved MINIMUM = %6.3lf.\n\n", minscore[0] );

      if( truscore != 0.0 || truerror != 1.0 ) {

          win_prtf( w2, 1, "\n\n\tPress a Key to Continue..." );
          get1key();

          win_prtf( w2, 1, "\n\n\tStatistics for the 2nd 50 Observations on each plot." );
          fprintf( outfile, "Statistics for the 2nd 50 Observations on each plot." );

          win_prtf( w2, 1, "\n\n\tObserved MAXIMUM = %6.3lf.", maxscore[1] );
          fprintf( outfile, "\nObserved MAXIMUM = %6.3lf.", maxscore[1] );

          win_prtf( w2, 1, "\n\tPercentage >=  4          was %6.1lf.",
              100.0 * (double)zone[1][9] / (double)total );
          fprintf( outfile, "\nPercentage >=  4          was %6.1lf.",
              100.0 * (double)zone[1][9] / (double)total );
          for( idx = 8; idx > 0; idx-- ) {
              win_prtf( w2, 1, "\n\tPercentage >= %2d and < %2d was %6.1lf.",
                  idx-5, idx-4, 100.0 * (double)zone[1][idx] / (double)total );
              fprintf( outfile, "\nPercentage >= %2d and < %2d was %6.1lf.",
                  idx-5, idx-4, 100.0 * (double)zone[1][idx] / (double)total );
              }
          win_prtf( w2, 1, "\n\tPercentage  < -4          was %6.1lf.",
              100.0 * (double)zone[1][0] / (double)total );
          fprintf( outfile, "\nPercentage  < -4          was %6.1lf.",
              100.0 * (double)zone[1][0] / (double)total );

          win_prtf( w2, 1, "\n\tObserved MINIMUM = %6.3lf.", minscore[1] );
          fprintf( outfile, "\nObserved MINIMUM = %6.3lf.\n\n", minscore[1] );

          total *= 2;
          }

      win_prtf( w2, 1, "\n\n\tPress a Key to Continue..." );
      get1key();

      if( truscore != 0.0 || truerror != 1.0 ) {
          win_prtf( w2, 1, "\n\n\tRun-Rule Violations for the 1st 50 Observations on each plot." );
          fprintf( outfile, "\n\nRun-Rule Violations for the 1st 50 Observations on each plot." );
          }

      for( idx = 0; idx < 8; idx++ ) {
          win_prtf( w2, 1, "\n\n\tRun-Rule No. %d : %4d Violations => %7.3lf Percent.",
              idx+1, rrcum[0][idx], 100.0 * (double)rrcum[0][idx] / (double)rrops[0][idx] );
          fprintf( outfile, "\nRun-Rule No. %d : %4d Violations => %7.3lf Percent.",
              idx+1, rrcum[0][idx], 100.0 * (double)rrcum[0][idx] / (double)rrops[0][idx] );
          }

      if( truscore != 0.0 || truerror != 1.0 ) {

          win_prtf( w2, 1, "\n\n\tPress a Key to Continue..." );
          get1key();

          win_prtf( w2, 1, "\n\n\tRun-Rule Violations for the 2nd 50 Observations on each plot." );
          fprintf( outfile, "Run-Rule Violations for the 2nd 50 Observations on each plot." );

          for( idx = 0; idx < 8; idx++ ) {
              win_prtf( w2, 1, "\n\n\tRun-Rule No. %d : %4d Violations => %7.3lf Percent.",
                  idx+1, rrcum[1][idx], 100.0 * (double)rrcum[1][idx] / (double)rrops[1][idx] );
              fprintf( outfile, "\nRun-Rule No. %d : %4d Violations => %7.3lf Percent.",
                  idx+1, rrcum[1][idx], 100.0 * (double)rrcum[1][idx] / (double)rrops[1][idx] );
              }
          }

      win_prtf( w2, 1, "\n\n\tPress Q to QUIT now...Other Key to Continue..." );
      if( ( get1key() >> 8 ) == QKEY ) {
          curlocat(24,0);
          printf( "\n" );
          coloreos( 2, 0 );
          exit( 0 );
          }

      win_close( w2 );
      }

void main() {
      
      extern char wrap_flg; /* declare the text wrapping flag */
      wrap_flg = TRUE;
      if( signal( SIGINT, handler ) == SIG_ERR ) {
              perror( "Couldn't set SIGINT...Abort!\n\n" );
              exit( 1 );
              }

      win_init();
      curtype(1,0,0);
      
      w0 = win_open(0,0,23,78,7,1,1,7,1," Control Chart Simulator ",7,1,"",7,1);
      
      win_prtf( w0, 1, "\n\n\t                   CCSIM.EXE...Version 9402" );
      win_prtf( w0, 1, "\n\n\t *** Pseudo-Random Control Chart Displays in VGA Graphics ***\n");
      win_prtf( w0, 1, "\n\n\t              A Quality Assurance Training Tool:" );
      win_prtf( w0, 1,   "\n\t      Statistics Committee of the QA Section of the PMA" );
      win_prtf( w0, 1, "\n\n\t         Bob Obenchain, CompuServe User [72007,467]\n\n" );
      
      /* set up light source direction */
      curtype(1,0,0);
      set_shadow(1,8,0,0);
      
      w1 = win_open(15,3,3,72,1,7,7,1,7,"",1,7,"",1,7);
      win_prtf( w1, 1, "\n\tPress a key to see the Control Chart Simulator MENU..." );
      get1key();
      win_close( w1 );
      win_cls( w0 );

      strcpy( outnam, "ccsim.ccs" );
      output = theend = parmset = zones = 0;
      truscore = 0.0;
      truerror = 1.0;
      batch = 100;
      efactor = 0.005;
      w1 = win_open(4,22,12,36,1,7,7,1,7,"",1,7,"",1,7);
      while( 1 ) {
      
              while( kbhit() != 0 )
                  get1key();

              win_totop( w1 );
              win_cls( w1 );
              curtype(1,0,0);
              win_prtf( w1, 0, "            Main Menu" );
              win_prtf( w1, 1, "\n\n   P = set  Parameter values" );
              win_prtf( w1, 1, "\n\n   V = View simulated charts" );
              win_prtf( w1, 1, "\n\n   S = Save Summary Statistics" );
              win_prtf( w1, 1, "\n\n   X = eXit from the CCSIM system" );

              win_prtf( w1, 1, "\n\n   choice --> " );
              cursrow = w1->cur_row;
              while( 1 ) {
                  if( ( kb = get1key() >> 8 ) == PKEY
                     || kb == VKEY || kb == SKEY || kb == XKEY )
                        break;
                  win_curset( cursrow, 0 );
                  win_prtf( w1, 1, "   Press P V S or X --> " );
                  }
      
              if( kb == XKEY ) {
                      win_prtf( w1, 0, "X" );
                      theend = 1;
                      }
              else if( kb == PKEY ) {
                      win_prtf( w1, 0, "P" );
                      setup();
                      }
              else if( kb == VKEY ) {
                      win_prtf( w1, 0, "V" );
                      zonedisp();
                      }
              else {
                      win_prtf( w1, 0, "S" );
                      summary();
                      }
              if( theend == 1 )
                  break;
              }
      
      fcloseall();
      curlocat(24,0);
      printf( "\n" );
      coloreos( 2, 0 );
      printf( "\n\nREMINDER(S) :\n" );
      
      if( outfile != NULL ) {
              printf( "\nCCSIM created an output file named: %s\n", outnam );
              printf( "\n\tUse the DOS invocation... TYPE %s | MORE", outnam );
              printf( "\n\tto review detailed information about this" );
              printf( "\n\trun of the Control Chart Simulator.\n\n" );
              }
      
      exit( 0 );
      }

