'***************************************************************************
'*                                                                         *
'*                           Standard Menu Module                          *
'*                                 MENU.BAS                                *
'*                                                                Ver 1.00 *
'*                                                                         *
'* FUNCTION:  This module provides simple to complex menus on the          *
'*            screen that are easy to define and evoke.                    *
'*                                                                         *
'*            All menus to be used by a given program can be defined       *
'*            during initialization and then simply referenced             *
'*            throughout the program by a simple variable associated       *
'*            with the menu.                                               *
'*                                                                         *
'*            Defining the menu requires only a few calls that are self    *
'*            documenting in nature and very easy to code and modify.      *
'*            Many options are available to customize the menu to          *
'*            specific tastes.  All the positioning variables, counters,   *
'*            pointers, etc. are internally generated by this module.      *
'*                                                                         *
'*            This module should remain separate from the main module      *
'*            and others that call it.  The source code can remain as      *
'*            a Quick Library (.QLB) and stand-alone library (.LIB)        *
'*            during development.  However, this module contains many      *
'*            optional procedures that may not be used by a given          *
'*            program.  It might be desirable to delete the procedures     *
'*            that are not being used from the final version of a program. *
'*                                                                         *
'*                                                                         *
'*                                                                         *
'* *********************************************************************** *
'* PROCEDURES:                                                             *
'* *********************************************************************** *
'*                                                                         *
'*   EXTERNAL FUNCTIONS & SUBROUTINES CALLED BY THIS MODULE:               *
'*                                                                         *
'*   EXTERNAL FUNCTIONS:                                                   *
'*     CenteringPosition% : Returns horizontal position for text centering *
'*                                                                         *
'*   EXTERNAL SUBROUTINES:                                                 *
'*     Box           : Draw a box/boarder on the screen                    *
'*     CLA           : Clear an area of the screen                         *
'*     CursorOFF     : Turn the cursor OFF                                 *
'*     CursorOn      : Turn the cursor ON                                  *
'*     Flicker       : Flash/Flicker text on the screen                    *
'*     GetKeyAndCode : Get key press with ASCII code from user             *
'*                                                                         *
'*                                                                         *
'*   INTERNAL FUNCTIONS & SUBROUTINES CALLED BY THIS MODULE:               *
'*                                                                         *
'*   INTERNAL SUBROUTINES:                                                 *
'*     ColorsMenuDefault  : Sets up the default colors for the module      *
'*     MenuCommentDivdSet : Sets a divider line for the comment area       *
'*     MenuCommentTextSet : Sets a text line for the comment area          *
'*     MenuHighItem       : Highlights the current item                    *
'*     MenuItemSet        : Put an item into the menu                      *
'*     MenuModInit        : Initialize the menu module                     *
'*     MenuNormItem       : Removes the highlight                          *
'*     MenuOpen           : Returns next open menu for use                 *
'*     MenuTitleDivdSet   : Sets a divider line for the title area         *
'*     MenuTitleTextSet   : Sets a text line for the title area            *
'*     XAdjustAddCtr      : Sets the widest object pointer                 *
'*     XAdjustBdrSet      : Sets the pointers for the menu boarders        *
'*     XAdjustCalcSelSize : Calculate and set the selector size            *
'*     XAdjustPointers    : Adjust pointers on the X axis                  *
'*     YAdjust            : Adjust all the pointers on the Y axis          *
'*                                                                         *
'*                                                                         *
'* *********************************************************************** *
'* VARIABLES:                                                              *
'* *********************************************************************** *
'*                                                                         *
'*   CONSTANTS:                                                            *
'*                       LOGICAL CONSTANTS                                 *
'*     True            : Logical TRUE                                      *
'*     False           : Logical FALSE                                     *
'*     On.             : Logical TRUE                                      *
'*     Off.            : Logical FALSE                                     *
'*     Yes             : Logical TRUE                                      *
'*     No              : Logical FALSE                                     *
'*                       COLOR CONSTANTS                                   *
'*     Black           : Color constant                                    *
'*     Blue            : Color constant                                    *
'*     Green           : Color constant                                    *
'*     Cyan            : Color constant                                    *
'*     Red             : Color constant                                    *
'*     Magenta         : Color constant                                    *
'*     Brown           : Color constant                                    *
'*     White           : Color constant                                    *
'*     Grey            : Color constant                                    *
'*     LightBlue       : Color constant                                    *
'*     LightGreen      : Color constant                                    *
'*     LightCyan       : Color constant                                    *
'*     LightRed        : Color constant                                    *
'*     LightMagenta    : Color constant                                    *
'*     Yellow          : Color constant                                    *
'*     BrightWhite     : Color constant                                    *
'*                       VIDEO DISPLAY CONSTANTS                           *
'*     FullScreenX     : Screen Size X                                     *
'*     FullScreenY     : Screen Size Y                                     *
'*                       MODUAL DEFAULT CONSTANTS                          *
'*     MenuDefaultX    : Default location X                                *
'*     MenuDefaultY    : Default location Y                                *
'*     MenuDefaultWid  : Default Width                                     *
'*     MenuDefaultLen  : Default Length                                    *
'*                       DATA STRUCTURE CONSTANTS (Set As Needed)          *
'*     MaxMenus        : Max number of menus                               *
'*     MaxItems        : Max number of items in a menu                     *
'*     MaxTitleLines   : Max number of title lines in a menu               *
'*     MaxCommentLines : Max number of comment lines in a menu             *
'*                                                                         *
'*   GLOBAL TO SOME PROCEDURES:                                            *
'*     MenuInfo() AS MenuInformation:                                      *
'*       Basic menu information on each menu                               *
'*     MenuBoarder$()   : Menu boarder (code or definition)                *
'*     MenuLPtr$()      : Menu current item pointer on left side           *
'*     MenuLClr$()      : Menu pointer clear                               *
'*     MenuRPtr$()      : Menu current item pointer on right side          *
'*     MenuRClr$()      : Menu pointer clear                               *
'*     -Structures for the header/top of the list-                         *
'*     MenuTitle$()     : Title/Header Text                                *
'*     MenuTitleInfo() AS MenuTopBtmText:                                  *
'*       Title/Header Text - Pointers, Etc.                                *
'*     MenuTitleDivdInfo() AS MenuDividerLineInfo:                         *
'*       Title/Header Dividers - Pointers, Etc.                            *
'*     -Structures for the comment lines/end of list-                      *
'*     MenuComment$()   : Comment/Trailer Text                             *
'*     MenuCommentInfo() AS MenuTopBtmText:                                *
'*       Comment/Trailer Text - Pointers, Etc.                             *
'*     MenuCommentDivdInfo() AS MenuDividerLineInfo:                       *
'*       Comment/Trailer Dividers - Pointers, Etc.                         *
'*     -Selector and holders-                                              *
'*     MenuSelect$()    : Selector items for each menu item                *
'*     MenuSelectLH$()  : Menu selector left holder                        *
'*     MenuSelectRH$()  : Menu selector right holder                       *
'*     -Menu Items-                                                        *
'*     MenuItems$()     : The Menu Items Text                              *
'*     -Colors-                                                            *
'*     ClrFMenuBdr%     : Foreground menu boarder/divider lines color      *
'*     ClrBMenuBdr%     : Background menu boarder/divider lines color      *
'*     ClrFMenuInner%   : Foreground menu inner area color                 *
'*     ClrBMenuInner%   : Background menu inner area color                 *
'*     ClrFMenuTitle%   : Foreground menu title color                      *
'*     ClrBMenuTitle%   : Background menu title color                      *
'*     ClrFMenuCom%     : Foreground menu comment color                    *
'*     ClrBMenuCom%     : Background menu comment color                    *
'*     ClrFMenuItem%    : Foreground menu item text normal color           *
'*     ClrBMenuItem%    : Background menu item text normal color           *
'*     ClrFMenuItemCur% : Foreground menu item text current color          *
'*     ClrBMenuItemCur% : Background menu item text current color          *
'*     ClrFMenuPtrs%    : Foreground menu pointers color                   *
'*     ClrBMenuPtrs%    : Background menu pointers color                   *
'*     ClrFMenuSel%     : Foreground menu selector color                   *
'*     ClrBMenuSel%     : Background menu selector color                   *
'*     ClrFMenuExit%    : Foreground exit color                            *
'*     ClrBMenuExit%    : Background exit color                            *
'*                                                                         *
'*                                                                         *
'* *********************************************************************** *
'* NOTES & COMMENTS:                                                       *
'* *********************************************************************** *
'*                                                                         *
'*   NOTES:                                                                *
'*                                                                         *
'*     As an alternate method of controling colors for the module,         *
'*     a COMMON block can be used to pass colors throughout a              *
'*     program.  This will require adding the COMMON block to this         *
'*     module as well as the other modules of the program and then         *
'*     re-compiling.                                                       *
'*                                                                         *
'*     No error checking is performed in this module.  If more then        *
'*     the default number of menus is desired the CONT MaxMenus can        *
'*     be set to accommodate the need.                                     *
'*                                                                         *
'*                                                                         *
'*   EXAMPLES:                                                             *
'*                                                                         *
'*     'DEFINE A MENU *******                                              *
'*     InitMenu MenuNumber%                                                *
'*     'MENU TITLE ****                                                    *
'*     MenuTitleText MenuNumber%, 0, 0, "Menu Example"                     *
'*     MenuTitleDivd MenuNumber%, 0, "͹"                                 *
'*     'MENU ITEMS ****                                                    *
'*     MenuItem MenuNumber%, "1", "Option Number 1"                        *
'*     MenuItem MenuNumber%, "2", "Option Number 2"                        *
'*     MenuItem MenuNumber%, "3", "Option Number 3"                        *
'*     'MENU COMMENT ****                                                  *
'*     MenuCommentText MenuNumber%, 0, 0, ""                               *
'*     MenuCommentText MenuNumber%, 0, 0, "Select 1 - 3"                   *
'*     'MENU DISPLAY OPTIONS ****                                          *
'*     SpaceLMenuBdr MenuNumber%, 2                                        *
'*     SpaceLMenuItems MenuNumber%, 1                                      *
'*     SpaceRMenuBdr MenuNumber%, 2                                        *
'*     MenuHolders MenuNumber%, "[", "]"                                   *
'*     MenuItemPointerL MenuNumber%, ""                                   *
'*     MenuItemPointerR MenuNumber%, ""                                   *
'*     FlickerMenu MenuNumber%, 7                                          *
'*                                                                         *
'*                                                                         *
'*   LAST MODIFICATION:                                                    *
'*                                                                         *
'*     DATE        VERSION  PROGRAMMER      DESCRIPTION OF MODIFICATION    *
'*  12/26/1989       1.00    Bob Cox                    N/A                *
'*                                                                         *
'*                                                                         *
'* *********************************************************************** *
'* Bob Cox '89                                              QuickBASIC 4.5 *
'***************************************************************************


'***************************************************************************
'* TYPE DECLARATIONS                                                 *******
'***************************************************************************
TYPE MenuInformation
  Active       AS INTEGER  'Active menu flag
                           'Info about items in the menu
  Items        AS INTEGER  'Number of items in the menu
  ItemCur      AS INTEGER  'Currently/Last selected item
                           'Positioning info (supplied by caller)
  MULX         AS INTEGER  'Menu Upper Left X (0-Auto Center on X Axis)
  MULY         AS INTEGER  'Menu Upper Left Y (0-Auto Center on Y Axis)
                           'Extra spacing around pointers and menu items
  SpLBdr       AS INTEGER  'Space off left boarder (before Left pointer)
  SpLPtr       AS INTEGER  'Space off left pointer (next to selector or items text/after left pointer)
  SpLItm       AS INTEGER  'Space off left of items (next to left selector/left side of text)
  SpRBdr       AS INTEGER  'Space off right boarder (next to boarder/after right pointer)
  SpRItm       AS INTEGER  'Space off right of items (next to items text on right side/before right pointer)
                           'Positioning info calculated/passed
  MWid         AS INTEGER  'Menu total width including boarders
  MIWid        AS INTEGER  'Menu inner width (excluding boarders)
  MLen         AS INTEGER  'Menu total length including boarders
  WidOb        AS INTEGER  'Widest object
  WidItem      AS INTEGER  'Widest Item (Menu Item)
  SelSize      AS INTEGER  'Total size of the selector (holders included)
  BULX         AS INTEGER  'Boarder upper left X
  BULY         AS INTEGER  'Boarder upper left Y
  BLRX         AS INTEGER  'Boarder lower right X
  BLRY         AS INTEGER  'Boarder lower right Y
  IULX         AS INTEGER  'Inner upper left X
  IULY         AS INTEGER  'Inner upper left Y
  ILRX         AS INTEGER  'Inner lower right X
  ILRY         AS INTEGER  'Inner lower right Y
  SX           AS INTEGER  'Selector start X
  IStartX      AS INTEGER  'Items Start X (Actual pos on screen for items X)
  IStartY      AS INTEGER  'Items Start Y (Actual pos on screen for items Y)
                           'Options
  SetToTop     AS INTEGER  'Set menu current item to the top when called
  Header       AS INTEGER  'Header Flag
  Comment      AS INTEGER  'Comment Flag
  DrawMenu     AS INTEGER  'Draw the menu flag
  Boarder      AS INTEGER  'Boarder flag
  Selector     AS INTEGER  'Selector flag
  Spacing      AS INTEGER  'Spacing between items in menu (1-n)
  Flicker      AS INTEGER  'Flicker selected item counter (0-Disables Flicker)
                           'Exit options
  ExitESC      AS INTEGER  'Exit on escape
  ExitFunc     AS INTEGER  'Exit on Function keys
  ExitCtrl     AS INTEGER  'Exit on CTRL keys
  ExitAlt      AS INTEGER  'Exit on ALT keys
                           'Title/Header
  HeaderLines  AS INTEGER  'Total number of lines in header
  HeaderBdr    AS INTEGER  'Header starts in the boarder flag
  HeaderTextL  AS INTEGER  'Total number of text lines in header
  HeaderDivdL  AS INTEGER  'Total number of divider lines in header
                           'Comment/Ending Text
  CommentLines AS INTEGER  'Total number of lines in comment
  CommentBdr   AS INTEGER  'Comments start in the boarder
  CommentTextL AS INTEGER  'Total number of text lines in comment
  CommentDivdL AS INTEGER  'Total number of divider lines in comment
                           'Pointers to current item
  LPntrX       AS INTEGER  'Left pointer X position
  LPntrL       AS INTEGER  'Left pointer length
  RPntrX       AS INTEGER  'Right pointer X position
  RPntrL       AS INTEGER  'Right pointer length
END TYPE

TYPE MenuTopBtmText
  X        AS INTEGER    'X base position (0 - Center on X axis)
  Y        AS INTEGER    'Y base position (0 - Next line)
  AX       AS INTEGER    'Actual X location on screen
  AY       AS INTEGER    'Actual Y location on screen
  LenText  AS INTEGER    'Length of the text (for any later re-calc operations) (when=0 indicates NOT DEFINED)
END TYPE

TYPE MenuDividerLineInfo
  Y        AS INTEGER    'Relative Y position
  AY       AS INTEGER    'Actual Y location on screen
                         'Divider line characters
  DCL      AS STRING * 1 'Divider line left character
  DCC      AS STRING * 1 'Divider line center character (replicated to fill center)
  DCR      AS STRING * 1 'Divider line right character
END TYPE


'***************************************************************************
'* COMMON BLOCKS                                                     *******
'***************************************************************************


'***************************************************************************
'* INTERNAL PROCEDURE DECLARATIONS                                   *******
'***************************************************************************
DECLARE SUB ColorsMenuDefault ()
DECLARE SUB MenuCommentDivdSet (MenuID%, DivID%, Y%, Divd$)
DECLARE SUB MenuCommentTextSet (MenuID%, TextID%, X%, Y%, Text$)
DECLARE SUB MenuHighItem (MenuID%, Item%)
DECLARE SUB MenuItemSet (MenuID%, ItemID%, SelectKey$, MenuItemText$)
DECLARE SUB MenuModInit ()
DECLARE SUB MenuNormItem (MenuID%)
DECLARE SUB MenuOpen (MenuID%)
DECLARE SUB MenuTitleDivdSet (MenuID%, DivID%, Y%, Divd$)
DECLARE SUB MenuTitleTextSet (MenuID%, TextID%, X%, Y%, Text$)
DECLARE SUB XAdjustAddCtr (MenuID%, LenObject%)
DECLARE SUB XAdjustBdrSet (MenuID%)
DECLARE SUB XAdjustCalcSelSize (MenuID%)
DECLARE SUB XAdjustPointers (MenuID%)
DECLARE SUB YAdjust (MenuID%)


'***************************************************************************
'* EXTERNAL PROCEDURE DECLARATIONS                                   *******
'***************************************************************************
DECLARE FUNCTION CenteringPosition% (SizeOfArea%, SizeOfObjectToCenter%)

DECLARE SUB Box (Xtop%, Ytop%, Xbot%, Ybot%, TheBox$)
DECLARE SUB CLA (UprleftX%, UprleftY%, LowrigtX%, LowrigtY%)
DECLARE SUB CursorOFF ()
DECLARE SUB CursorON ()
DECLARE SUB Flicker (NumB%, LocX%, LocY%, HFG%, HBG%, NFG%, NBG%, FlickerStr$)
DECLARE SUB GetKeyAndCode (KeyPress$, ExtendedFlag%, KeyCode%)


'***************************************************************************
'* CONSTANTS                                                         *******
'***************************************************************************

'Logical Constants
CONST True = 1, False = 0
CONST On. = 1, Off. = 0
CONST Yes = 1, No = 0

'Color Constants ***
CONST Black = 0, Blue = 1, Green = 2, Cyan = 3
CONST Red = 4, Magenta = 5, Brown = 6, White = 7
CONST Grey = 8, LightBlue = 9, LightGreen = 10, LightCyan = 11
CONST LightRed = 12, LightMagenta = 13, Yellow = 14, BrightWhite = 15

'Video Display Constants
CONST FullScreenX = 80   'Size of screen on X axis (Number of Columns)
CONST FullScreenY = 25   'Size of screen on Y axis (Number of Rows)

'Set default positioning and size for an initialized menu
'Width and length will auto adjust as items are added to
'the menu definition
CONST MenuDefaultX = 0
CONST MenuDefaultY = 0
CONST MenuDefaultWid = 3
CONST MenuDefaultLen = 3

CONST MaxMenus = 3   'Max number of menus
CONST MaxItems = 23  'Max number of items in a menu

CONST MaxTitleLines = 5    'Max number of title lines in a menu
CONST MaxCommentLines = 4  'Max number of comment lines in a menu


'***************************************************************************
'* MODUAL LEVEL SIMPLE VARIABLES                                     *******
'***************************************************************************


'***************************************************************************
'* STORAGE/DATA STRUCTURES                                           *******
'***************************************************************************
DIM MenuInfo(1 TO MaxMenus) AS MenuInformation  'Basic menu information on each menu
DIM MenuBoarder$(1 TO MaxMenus)                 'Menu boarder (code or definition)

'Menu current item pointers
DIM MenuLPtr$(1 TO MaxMenus)                    'Menu current item pointer on left side
DIM MenuLClr$(1 TO MaxMenus)                    'Menu pointer clear
DIM MenuRPtr$(1 TO MaxMenus)                    'Menu current item pointer on right side
DIM MenuRClr$(1 TO MaxMenus)                    'Menu pointer clear

'Structures for the header/top of the list
DIM MenuTitle$(1 TO MaxMenus, 1 TO MaxTitleLines)                                'Title/Header for menu
DIM MenuTitleInfo(1 TO MaxMenus, 1 TO MaxTitleLines) AS MenuTopBtmText
DIM MenuTitleDivdInfo(1 TO MaxMenus, 1 TO MaxTitleLines)  AS MenuDividerLineInfo

'Structures for the comment lines/end of list
DIM MenuComment$(1 TO MaxMenus, 1 TO MaxCommentLines)
DIM MenuCommentInfo(1 TO MaxMenus, 1 TO MaxCommentLines) AS MenuTopBtmText
DIM MenuCommentDivdInfo(1 TO MaxMenus, 1 TO MaxCommentLines)  AS MenuDividerLineInfo

'Selector and holders
DIM MenuSelect$(1 TO MaxMenus, 1 TO MaxItems)   'Selector items for each menu item
DIM MenuSelectLH$(1 TO MaxMenus)                'Menu selector left holder
DIM MenuSelectRH$(1 TO MaxMenus)                'Menu selector right holder

'Menu items
DIM MenuItems$(1 TO MaxMenus, 1 TO MaxItems)    'The menu items text


'***************************************************************************
'* MODUAL ERROR ROUTINE                                              *******
'***************************************************************************


'***************************************************************************
'* MODUAL EVENT ROUTINES                                             *******
'***************************************************************************


'***************************************************************************
'* DATA STATEMENTS                                                   *******
'***************************************************************************

SUB ColorMenuBoarder (ColorFore%, ColorBack%)

'***************************************************************************
'*                                                                         *
'*                    Set the Color of the Menu Boarder                    *
'*                             ColorMenuBoarder                            *
'*                                                                Ver 1.00 *
'*                                                                         *
'* FUNCTION:  This routine sets the foreground and background              *
'*            color for the menu boarder.                                  *
'*                                                                         *
'*                                                                         *
'* *********************************************************************** *
'* ARGUMENTS:                                                              *
'* *********************************************************************** *
'*                                                                         *
'*   INPUT:                                                                *
'*     ColorFore% : Foreground color                                       *
'*     ColorBack% : Background color                                       *
'*                                                                         *
'*                                                                         *
'*                                                                         *
'* Bob Cox '89                                              QuickBASIC 4.5 *
'***************************************************************************

'***************************************************************************
'* SHARED VARIABLES                                                  *******
'***************************************************************************
SHARED ClrFMenuBdr%, ClrBMenuBdr% 'Menu Boader Color

'***************************************************************************
'* PROCEDURE LOGIC                                                   *******
'***************************************************************************

  ClrFMenuBdr% = ColorFore%
  ClrBMenuBdr% = ColorBack%

END SUB

SUB ColorMenuComment (ColorFore%, ColorBack%)

'***************************************************************************
'*                                                                         *
'*                    Set the Color of the Menu Comment                    *
'*                             ColorMenuComment                            *
'*                                                                Ver 1.00 *
'*                                                                         *
'* FUNCTION:  This routine sets the foreground and background              *
'*            color for the menu comment area.                             *
'*                                                                         *
'*                                                                         *
'* *********************************************************************** *
'* ARGUMENTS:                                                              *
'* *********************************************************************** *
'*                                                                         *
'*   INPUT:                                                                *
'*     ColorFore% : Foreground color                                       *
'*     ColorBack% : Background color                                       *
'*                                                                         *
'*                                                                         *
'*                                                                         *
'* Bob Cox '89                                              QuickBASIC 4.5 *
'***************************************************************************

'***************************************************************************
'* SHARED VARIABLES                                                  *******
'***************************************************************************
SHARED ClrFMenuCom%, ClrBMenuCom%            'Menu comment text colors

'***************************************************************************
'* PROCEDURE LOGIC                                                   *******
'***************************************************************************
 
  ClrFMenuCom% = ColorFore%
  ClrBMenuCom% = ColorBack%

END SUB

SUB ColorMenuExit (ColorFore%, ColorBack%)

'***************************************************************************
'*                                                                         *
'*                   Set the Exit Color for the Menu Module                *
'*                               ColorMenuExit                             *
'*                                                                Ver 1.00 *
'*                                                                         *
'* FUNCTION:  This routine sets the foreground and background              *
'*            color for exit.  This is the color the system is set         *
'*            to before returning to the caller.                           *
'*                                                                         *
'*                                                                         *
'* *********************************************************************** *
'* ARGUMENTS:                                                              *
'* *********************************************************************** *
'*                                                                         *
'*   INPUT:                                                                *
'*     ColorFore% : Foreground color                                       *
'*     ColorBack% : Background color                                       *
'*                                                                         *
'*                                                                         *
'*                                                                         *
'* Bob Cox '89                                              QuickBASIC 4.5 *
'***************************************************************************

'***************************************************************************
'* SHARED VARIABLES                                                  *******
'***************************************************************************
SHARED ClrFMenuExit%, ClrBMenuExit%          'Menu Exit Colors

'***************************************************************************
'* PROCEDURE LOGIC                                                   *******
'***************************************************************************

  ClrFMenuExit% = ColorFore%
  ClrBMenuExit% = ColorBack%

END SUB

SUB ColorMenuInner (ColorFore%, ColorBack%)

'***************************************************************************
'*                                                                         *
'*                   Set the Color of the Menu Inner Area                  *
'*                              ColorMenuInner                             *
'*                                                                Ver 1.00 *
'*                                                                         *
'* FUNCTION:  This routine sets the foreground and background              *
'*            color for the menu inner area.                               *
'*                                                                         *
'*                                                                         *
'* *********************************************************************** *
'* ARGUMENTS:                                                              *
'* *********************************************************************** *
'*                                                                         *
'*   INPUT:                                                                *
'*     ColorFore% : Foreground color                                       *
'*     ColorBack% : Background color                                       *
'*                                                                         *
'*                                                                         *
'*                                                                         *
'* Bob Cox '89                                              QuickBASIC 4.5 *
'***************************************************************************

'***************************************************************************
'* SHARED VARIABLES                                                  *******
'***************************************************************************
SHARED ClrFMenuInner%, ClrBMenuInner% 'Menu Inner Colors

'***************************************************************************
'* PROCEDURE LOGIC                                                   *******
'***************************************************************************

  ClrFMenuInner% = ColorFore%
  ClrBMenuInner% = ColorBack%

END SUB

SUB ColorMenuItemCur (ColorFore%, ColorBack%)

'***************************************************************************
'*                                                                         *
'*                  Set the Color of the Menu Current Item                 *
'*                             ColorMenuItemCur                            *
'*                                                                Ver 1.00 *
'*                                                                         *
'* FUNCTION:  This routine sets the foreground and background              *
'*            color for the current/active menu item.                      *
'*                                                                         *
'*                                                                         *
'* *********************************************************************** *
'* ARGUMENTS:                                                              *
'* *********************************************************************** *
'*                                                                         *
'*   INPUT:                                                                *
'*     ColorFore% : Foreground color                                       *
'*     ColorBack% : Background color                                       *
'*                                                                         *
'*                                                                         *
'*                                                                         *
'* Bob Cox '89                                              QuickBASIC 4.5 *
'***************************************************************************

'***************************************************************************
'* SHARED VARIABLES                                                  *******
'***************************************************************************
SHARED ClrFMenuItemCur%, ClrBMenuItemCur%    'Menu Current Item Colors (Highlighted Item)

'***************************************************************************
'* PROCEDURE LOGIC                                                   *******
'***************************************************************************

  ClrFMenuItemCur% = ColorFore%
  ClrBMenuItemCur% = ColorBack%

END SUB

SUB ColorMenuItems (ColorFore%, ColorBack%)

'***************************************************************************
'*                                                                         *
'*                  Set the Color of the Menu Normal Items                 *
'*                              ColorMenuItems                             *
'*                                                                Ver 1.00 *
'*                                                                         *
'* FUNCTION:  This routine sets the foreground and background              *
'*            color for the menu items (those not current).                *
'*                                                                         *
'*                                                                         *
'* *********************************************************************** *
'* ARGUMENTS:                                                              *
'* *********************************************************************** *
'*                                                                         *
'*   INPUT:                                                                *
'*     ColorFore% : Foreground color                                       *
'*     ColorBack% : Background color                                       *
'*                                                                         *
'*                                                                         *
'*                                                                         *
'* Bob Cox '89                                              QuickBASIC 4.5 *
'***************************************************************************

'***************************************************************************
'* SHARED VARIABLES                                                  *******
'***************************************************************************
SHARED ClrFMenuItem%, ClrBMenuItem% 'Menu Normal Items Colors

'***************************************************************************
'* PROCEDURE LOGIC                                                   *******
'***************************************************************************

  ClrFMenuItem% = ColorFore%
  ClrBMenuItem% = ColorBack%

END SUB

SUB ColorMenuPtrs (ColorFore%, ColorBack%)

'***************************************************************************
'*                                                                         *
'*                    Set the Color of the Menu Pointers                   *
'*                              ColorMenuPtrs                              *
'*                                                                Ver 1.00 *
'*                                                                         *
'* FUNCTION:  This routine sets the foreground and background              *
'*            color for the menu pointers.                                 *
'*                                                                         *
'*                                                                         *
'* *********************************************************************** *
'* ARGUMENTS:                                                              *
'* *********************************************************************** *
'*                                                                         *
'*   INPUT:                                                                *
'*     ColorFore% : Foreground color                                       *
'*     ColorBack% : Background color                                       *
'*                                                                         *
'*                                                                         *
'*                                                                         *
'* Bob Cox '89                                              QuickBASIC 4.5 *
'***************************************************************************

'***************************************************************************
'* SHARED VARIABLES                                                  *******
'***************************************************************************
SHARED ClrFMenuPtrs%, ClrBMenuPtrs% 'Menu Current Item Pointers Colors

'***************************************************************************
'* PROCEDURE LOGIC                                                   *******
'***************************************************************************

  ClrFMenuPtrs% = ColorFore%
  ClrBMenuPtrs% = ColorBack%

END SUB

SUB ColorMenuSelector (ColorFore%, ColorBack%)

'***************************************************************************
'*                                                                         *
'*                    Set the Color of the Menu Selector                   *
'*                            ColorMenuSelector                            *
'*                                                                Ver 1.00 *
'*                                                                         *
'* FUNCTION:  This routine sets the foreground and background              *
'*            color for the menu selectors.                                *
'*                                                                         *
'*                                                                         *
'* *********************************************************************** *
'* ARGUMENTS:                                                              *
'* *********************************************************************** *
'*                                                                         *
'*   INPUT:                                                                *
'*     ColorFore% : Foreground color                                       *
'*     ColorBack% : Background color                                       *
'*                                                                         *
'*                                                                         *
'*                                                                         *
'* Bob Cox '89                                              QuickBASIC 4.5 *
'***************************************************************************

'***************************************************************************
'* SHARED VARIABLES                                                  *******
'***************************************************************************
SHARED ClrFMenuSel%, ClrBMenuSel%  'Menu Selector Colors

'***************************************************************************
'* PROCEDURE LOGIC                                                   *******
'***************************************************************************

  ClrFMenuSel% = ColorFore%
  ClrBMenuSel% = ColorBack%

END SUB

SUB ColorMenuTitle (ColorFore%, ColorBack%)

'***************************************************************************
'*                                                                         *
'*                     Set the Color of the Menu Title                     *
'*                              ColorMenuTitle                             *
'*                                                                Ver 1.00 *
'*                                                                         *
'* FUNCTION:  This routine sets the foreground and background              *
'*            color for the menu title area.                               *
'*                                                                         *
'*                                                                         *
'* *********************************************************************** *
'* ARGUMENTS:                                                              *
'* *********************************************************************** *
'*                                                                         *
'*   INPUT:                                                                *
'*     ColorFore% : Foreground color                                       *
'*     ColorBack% : Background color                                       *
'*                                                                         *
'*                                                                         *
'*                                                                         *
'* Bob Cox '89                                              QuickBASIC 4.5 *
'***************************************************************************

'***************************************************************************
'* SHARED VARIABLES                                                  *******
'***************************************************************************
SHARED ClrFMenuTitle%, ClrBMenuTitle% 'Menu title text Colors

'***************************************************************************
'* PROCEDURE LOGIC                                                   *******
'***************************************************************************

  ClrFMenuTitle% = ColorFore%
  ClrBMenuTitle% = ColorBack%

END SUB

SUB ColorsMenuAllBack (ColorBack%)

'***************************************************************************
'*                                                                         *
'*             Set the Color of All the Menu Background Colors             *
'*                            ColorsMenuAllBack                            *
'*                                                                Ver 1.00 *
'*                                                                         *
'* FUNCTION:  This routine sets all the background colors to one color.    *
'*                                                                         *
'*                                                                         *
'* *********************************************************************** *
'* ARGUMENTS:                                                              *
'* *********************************************************************** *
'*                                                                         *
'*   INPUT:                                                                *
'*     ColorBack% : Background color for all                               *
'*                                                                         *
'*                                                                         *
'*                                                                         *
'* Bob Cox '89                                              QuickBASIC 4.5 *
'***************************************************************************

'***************************************************************************
'* SHARED VARIABLES                                                  *******
'***************************************************************************
SHARED ClrBMenuBdr%           'Menu Boader
SHARED ClrBMenuInner%         'Menu Inner Color
SHARED ClrBMenuTitle%         'Menu Title Text Colors
SHARED ClrBMenuCom%           'Menu Comment Text Colors
SHARED ClrBMenuItem%          'Menu Normal Items Colors
SHARED ClrBMenuItemCur%       'Menu Current Item Colors (Highlighted Item)
SHARED ClrBMenuPtrs%          'Menu Current Item Pointers Colors
SHARED ClrBMenuSel%           'Menu Selector Colors

'***************************************************************************
'* PROCEDURE LOGIC                                                   *******
'***************************************************************************

  ClrBMenuBdr% = ColorBack%
  ClrBMenuInner% = ColorBack%
  ClrBMenuTitle% = ColorBack%
  ClrBMenuCom% = ColorBack%
  ClrBMenuItem% = ColorBack%
  ClrBMenuItemCur% = ColorBack%
  ClrBMenuPtrs% = ColorBack%
  ClrBMenuSel% = ColorBack%

END SUB

SUB ColorsMenuAllFore (ColorFore%)

'***************************************************************************
'*                                                                         *
'*             Set the Color of All the Menu Foreground Colors             *
'*                            ColorsMenuAllFore                            *
'*                                                                Ver 1.00 *
'*                                                                         *
'* FUNCTION:  This routine sets all the foreground colors to one color.    *
'*                                                                         *
'*                                                                         *
'* *********************************************************************** *
'* ARGUMENTS:                                                              *
'* *********************************************************************** *
'*                                                                         *
'*   INPUT:                                                                *
'*     ColorFore% : Foreground color for all                               *
'*                                                                         *
'*                                                                         *
'*                                                                         *
'* Bob Cox '89                                              QuickBASIC 4.5 *
'***************************************************************************

'***************************************************************************
'* SHARED VARIABLES                                                  *******
'***************************************************************************
SHARED ClrFMenuBdr%           'Menu Boader
SHARED ClrFMenuInner%         'Menu Inner Color
SHARED ClrFMenuTitle%         'Menu Title Text Color
SHARED ClrFMenuCom%           'Menu Comment Text Color
SHARED ClrFMenuItem%          'Menu Normal Items Color
SHARED ClrFMenuItemCur%       'Menu Current Item Color (Highlighted Item)
SHARED ClrFMenuPtrs%          'Menu Current Item Pointers Color
SHARED ClrFMenuSel%           'Menu Selector Color

'***************************************************************************
'* PROCEDURE LOGIC                                                   *******
'***************************************************************************

  ClrFMenuBdr% = ColorFore%
  ClrFMenuInner% = ColorFore%
  ClrFMenuTitle% = ColorFore%
  ClrFMenuCom% = ColorFore%
  ClrFMenuItem% = ColorFore%
  ClrFMenuItemCur% = ColorFore%
  ClrFMenuPtrs% = ColorFore%
  ClrFMenuSel% = ColorFore%

END SUB

SUB ColorsMenuDefault

'***************************************************************************
'*                                                                         *
'*                     Set All the Default Menu Colors                     *
'*                            ColorsMenuDefault                            *
'*                                                                Ver 1.00 *
'*                                                                         *
'* FUNCTION:  This routine sets all the default foreground and background  *
'*            colors for the menu module.                                  *
'*                                                                         *
'*                                                                         *
'* *********************************************************************** *
'* VARIABLES:                                                              *
'* *********************************************************************** *
'*                                                                         *
'*   GLOBAL TO PROCEDURE:                                                  *
'*     Colors                                                              *
'*     ClrFMenuBdr%     : Foreground menu boarder/divider lines color      *
'*     ClrBMenuBdr%     : Background menu boarder/divider lines color      *
'*     ClrFMenuInner%   : Foreground menu inner area color                 *
'*     ClrBMenuInner%   : Background menu inner area color                 *
'*     ClrFMenuTitle%   : Foreground menu title color                      *
'*     ClrBMenuTitle%   : Background menu title color                      *
'*     ClrFMenuCom%     : Foreground menu comment color                    *
'*     ClrBMenuCom%     : Background menu comment color                    *
'*     ClrFMenuItem%    : Foreground menu item text normal color           *
'*     ClrBMenuItem%    : Background menu item text normal color           *
'*     ClrFMenuItemCur% : Foreground menu item text current color          *
'*     ClrBMenuItemCur% : Background menu item text current color          *
'*     ClrFMenuPtrs%    : Foreground menu pointers color                   *
'*     ClrBMenuPtrs%    : Background menu pointers color                   *
'*     ClrFMenuSel%     : Foreground menu selector color                   *
'*     ClrBMenuSel%     : Background menu selector color                   *
'*     ClrFMenuExit%    : Foreground exit color                            *
'*     ClrBMenuExit%    : Background exit color                            *
'*                                                                         *
'*                                                                         *
'* Bob Cox '89                                              QuickBASIC 4.5 *
'***************************************************************************

'***************************************************************************
'* SHARED VARIABLES                                                  *******
'***************************************************************************
SHARED ClrFMenuBdr%, ClrBMenuBdr%            'Menu Boader
SHARED ClrFMenuInner%, ClrBMenuInner%        'Menu Inner Color
SHARED ClrFMenuTitle%, ClrBMenuTitle%        'Menu Title Text Colors
SHARED ClrFMenuCom%, ClrBMenuCom%            'Menu Comment Text Colors
SHARED ClrFMenuItem%, ClrBMenuItem%          'Menu Normal Items Colors
SHARED ClrFMenuItemCur%, ClrBMenuItemCur%    'Menu Current Item Colors (Highlighted Item)
SHARED ClrFMenuPtrs%, ClrBMenuPtrs%          'Menu Current Item Pointers Colors
SHARED ClrFMenuSel%, ClrBMenuSel%            'Menu Selector Colors
SHARED ClrFMenuExit%, ClrBMenuExit%          'Menu Exit Colors

'***************************************************************************
'* PROCEDURE LOGIC                                                   *******
'***************************************************************************

  ClrFMenuBdr% = BrightWhite
  ClrBMenuBdr% = Black

  ClrFMenuInner% = White
  ClrBMenuInner% = Black

  ClrFMenuTitle% = BrightWhite
  ClrBMenuTitle% = Black

  ClrFMenuCom% = White
  ClrBMenuCom% = Black

  ClrFMenuItem% = White
  ClrBMenuItem% = Black

  ClrFMenuItemCur% = BrightWhite
  ClrBMenuItemCur% = Black

  ClrFMenuPtrs% = BrightWhite
  ClrBMenuPtrs% = Black

  ClrFMenuSel% = White
  ClrBMenuSel% = Black

  ClrFMenuExit% = White
  ClrBMenuExit% = Black

END SUB

SUB FlickerMenu (MenuID%, FlickerCount%)

'***************************************************************************
'*                                                                         *
'*                          Set the Flicker Value                          *
'*                               FlickerMenu                               *
'*                                                                Ver 1.00 *
'*                                                                         *
'* FUNCTION:  This routine sets the flicker counter to determine the       *
'*            number of times a selected item from the menu will           *
'*            flicker.  The range of FlickerCount% is 0 - 20 where         *
'*            a value of 0 will disable the flicker effect.                *
'*                                                                         *
'*                                                                         *
'* *********************************************************************** *
'* ARGUMENTS:                                                              *
'* *********************************************************************** *
'*                                                                         *
'*   INPUT:                                                                *
'*     MenuID%       : Menu being set                                      *
'*     FlickerCount% : Number of times to flicker (0 - disable flicker)    *
'*                                                                         *
'*                                                                         *
'*                                                                         *
'* Bob Cox '89                                              QuickBASIC 4.5 *
'***************************************************************************

'***************************************************************************
'* SHARED VARIABLES                                                  *******
'***************************************************************************
SHARED MenuInfo() AS MenuInformation

'***************************************************************************
'* PROCEDURE LOGIC                                                   *******
'***************************************************************************
 
  SELECT CASE FlickerCount%
    CASE IS < 0
      FlickerCount% = 0
    CASE IS > 20
      FlickerCount% = 20
  END SELECT

  MenuInfo(MenuID%).Flicker = FlickerCount%

END SUB

SUB InitMenu (MenuID%)

'***************************************************************************
'*                                                                         *
'*                            Initialize a Menu                            *
'*                                 InitMenu                                *
'*                                                                Ver 1.00 *
'*                                                                         *
'* FUNCTION:  This routine initializes a menu for use.  Before a menu can  *
'*            be defined (MenuItem, etc.) it must be initialized here.     *
'*            This routine will return the next available menu number/slot *
'*            that will continue to be used in all other support and       *
'*            definition routines for that menu.  This routine followed    *
'*            by other simple calls to build a menu can be used as an      *
'*            alternate to InitMenuFromDATA.                               *
'*                                                                         *
'*                                                                         *
'* *********************************************************************** *
'* ARGUMENTS:                                                              *
'* *********************************************************************** *
'*                                                                         *
'*   OUTPUT:                                                               *
'*     MenuID% : The menu's ID number                                      *
'*                 =0 - No menu area remains                               *
'*                 >0 - Menu's number/ID                                   *
'*                                                                         *
'* *********************************************************************** *
'* PROCEDURES:                                                             *
'* *********************************************************************** *
'*                                                                         *
'*   FUNCTIONS & SUBROUTINES CALLED:                                       *
'*                                                                         *
'*   SUBROUTINES:                                                          *
'*     MenuOpen    : Returns next open menu for use                        *
'*     MenuModInit : Initializes the menu module (default colors, etc.)    *
'*                                                                         *
'*                                                                         *
'*                                                                         *
'* Bob Cox '89                                              QuickBASIC 4.5 *
'***************************************************************************

'***************************************************************************
'* SHARED VARIABLES                                                  *******
'***************************************************************************
SHARED NumMenus%                       'Number of menus currently defined
SHARED MenuModuleInit%                 'Set externally by MenuModInit
SHARED MenuInfo()  AS MenuInformation  'Basic menu information on each menu
SHARED MenuBoarder$()                  'Menu boarder (code or definition)

'***************************************************************************
'* PROCEDURE LOGIC                                                   *******
'***************************************************************************

  MenuOpen MenuID%

  IF MenuID% <> 0 THEN
    'A menu area exists

    'Init the menu module if it has not been done yet
    'This will be done ONE time only
    IF MenuModuleInit% = False THEN MenuModInit

    'Init the menu
    MenuInfo(MenuID%).Active = True   'This is an active menu
    MenuInfo(MenuID%).Spacing = 1     'Default item spacing to 1
    MenuInfo(MenuID%).DrawMenu = On.  'Set the draw flag on
    MenuInfo(MenuID%).Boarder = On.   'Place a boarder around the menu
    MenuBoarder$(MenuID%) = "D"       'Set for a double box

    MenuInfo(MenuID%).ItemCur = 1     'Position to the top item for first call

    'Set the menu for default positioning
    MenuInfo(MenuID%).MULX = MenuDefaultX
    MenuInfo(MenuID%).MULY = MenuDefaultY
    'Set a default size for the menu window
    MenuInfo(MenuID%).MWid = MenuDefaultWid
    MenuInfo(MenuID%).MLen = MenuDefaultLen
    'Set default menu position
    MenuInfo(MenuID%).BULX = CenteringPosition%(FullScreenX, MenuInfo(MenuID%).MWid)
    MenuInfo(MenuID%).BULY = CenteringPosition%(FullScreenY, MenuInfo(MenuID%).MLen)
    MenuInfo(MenuID%).BLRX = MenuInfo(MenuID%).BULX + MenuInfo(MenuID%).MWid - 1
    MenuInfo(MenuID%).BLRY = MenuInfo(MenuID%).BULY + MenuInfo(MenuID%).MLen - 1

    NumMenus% = NumMenus% + 1         'Incr menu count (menus being managed by this module)
  END IF

END SUB

SUB InitMenuFromDATA (MenuID%)

'***************************************************************************
'*                                                                         *
'*                  Initialize a menu from DATA statements                 *
'*                             InitMenuFromDATA                            *
'*                                                                Ver 1.00 *
'*                                                                         *
'* FUNCTION:  This routine will load a menu that is defined in DATA        *
'*            statements.  The DATA statements need to follow the          *
'*            format described below and must be located in this module.   *
'*            This routine can be used as an alternate to InitMenu         *
'*            followed by all the other calls to build a menu.             *
'*                                                                         *
'*                                                                         *
'* *********************************************************************** *
'* ARGUMENTS:                                                              *
'* *********************************************************************** *
'*                                                                         *
'*   OUTPUT:                                                               *
'*     MenuID% : The menu number for the menu loaded from DATA             *
'*                                                                         *
'* *********************************************************************** *
'* PROCEDURES:                                                             *
'* *********************************************************************** *
'*                                                                         *
'*   FUNCTIONS & SUBROUTINES CALLED:                                       *
'*                                                                         *
'*   SUBROUTINES:                                                          *
'*     MenuOpen      : Returns next open menu for use                      *
'*     YAdjust       : Adjust pointers on Y axis                           *
'*     XAdjustBdrSet : Adjust pointers on X axis                           *
'*                                                                         *
'* *********************************************************************** *
'* NOTES & COMMENTS:                                                       *
'* *********************************************************************** *
'*                                                                         *
'* NOTES:                                                                  *
'*                                                                         *
'* Data Statements Format:                                                 *
'*                                                                         *
'*  Item:                   Example:      Comment:                         *
'*  _______________________ _____________ ________________________________ *
'*                                        Menu window or box               *
'*  Menu X Location         0             0=Auto Center on X axis          *
'*  Menu Y Location         0             0=Auto Center on Y Axis          *
'*  Boarder On/Off          1             1=On/0=Off                       *
'*  Boarder Type            "D"           Standard BOX types or define     *
'*                                        Pointers (current item)          *
'*  Space off left boarder  0             Spaces before pointer            *
'*  Left Side Pointer       "=>"          Any characters of any length     *
'*  Space off left pointer  0             Spaces after pointer             *
'*  Space off rght pointer  0             Spaces before pointer            *
'*  Right Side Pointer      "<="          Any characters of any length     *
'*  Space off rght boarder  0             Spaces before right boarder      *
'*                                        Header                           *
'*  Header On/Off           1             1=On/0=Off                       *
'*  Header in boarder       0             Start Header In Boarder 1/0      *
'*  Total Header Size       2             Size of the Header Area          *
'*  Number of Text Lines    1             Number of Text Lines in Header   *
'*  Number of Divider Lines 1             Number of Divd lines in Header   *
'*  [HEADER TEXT LINES]                   Text Lines In Header (0-n)       *
'*  Text header X position  0             X realitive position (0=Center)  *
'*  Text header Y position  1             Line position in header (1-n)    *
'*  Header Text             "Main Menu"   Any text any size                *
'*  [HEADER DIVIDER LINES]                Divider Lines In Header (0-n)    *
'*  Divider header Y pos    2             Line position in header (1-n)    *
'*  Divider characters      "|-|"         Divider segments (3 characters)  *
'*                                        Menu Items Selector              *
'*  Selector On/Off         1             1-On/0=Off                       *
'*  Left Selector Holder    "["           Any characters any length        *
'*  Right Selector Holder   "]"           Any characters any length        *
'*  Space off Selector      0             Spaces between selector & items  *
'*                                        Menu Items                       *
'*  Number of Menu Items    1             Menu Items (1-n | Norm:>1)       *
'*  Spacing Between Items   1             Line Spacing (1-n)               *
'*  Current Item            0             Which Item is Current Item       *
'*  [MENU ITEMS]                          All the Items in the Menu        *
'*  Selector Character      "E"           Single Key Select (0-1 chars)    *
'*  Item Text               "Exit Menu"   Any Text any Size                *
'*                                        Comment/Trailer                  *
'*  Trailer On/Off          1             1=On/0=Off                       *
'*  Trailer in boarder      0             End Trailer In Boarder 1/0       *
'*  Total Trailer Size      2             Size of the Trailer Area         *
'*  Number of Text Lines    1             Number of Text Lines in Trailer  *
'*  Number of Divider Lines 1             Number of Divd lines in Trailer  *
'*  [TRAILER TEXT LINES]                  Text Lines In Trailer (0-n)      *
'*  Text Trailer X position 0             X realitive position (0=Center)  *
'*  Text Trailer Y position 1             Line position in Trailer (1-n)   *
'*  Trailer Text            "Select..."   Any text any size                *
'*  [TRAILER DIVIDER LINES]               Divider Lines In Trailer (0-n)   *
'*  Divider Trailer Y pos    2            Line position in Trailer (1-n)   *
'*  Divider characters       "|-|"        Divider segments (3 characters)  *
'*                                        Colors (Foreground/Background)   *
'*  Boarder/Divider          15,0         Boarder and Divider Color        *
'*  Inner Menu                7,0         Inner Area of Menu Color         *
'*  Header                   15,0         Header Text Color                *
'*  Trailer                   7,0         Trailer Text Color               *
'*  Normal Item               7,0         Not Current Item Color           *
'*  Current Item             15,0         Current Item Color (highlighted) *
'*  Pointers                 15,0         Current Item Pointers Color      *
'*  Selector                  7,0         Item Selectors Color             *
'*                                        Exit Options                     *
'*  Exit on ESC               1           1=On/0=Off                       *
'*  Exit on Function Keys     0           1=On/0=Off                       *
'*  Exit on Control Keys      0           1=On/0=Off                       *
'*  Exit on Alter Keys        0           1=On/0=Off                       *
'*                                        Misc                             *
'*  Set To Top Of Items       1           Reset to Top of Menu (1/0)       *
'*  Flicker On Selection      0           Flicker Count (1-n | 0=None)     *
'*                                                                         *
'*                                                                         *
'* Bob Cox '89                                              QuickBASIC 4.5 *
'***************************************************************************

'***************************************************************************
'* SHARED VARIABLES                                                  *******
'***************************************************************************
SHARED NumMenus%                                    'Number of menus
SHARED MenuModuleInit%                              'Set externally by MenuModInit

SHARED MenuInfo() AS MenuInformation                'Basic menu information on each menu
SHARED MenuItems$()                                 'Menu items - text
                                                    'Various menu elements
SHARED MenuBoarder$()                               'Menu boarder (code or definition)
SHARED MenuSelect$()                                'Selector items for each menu item
SHARED MenuSelectLH$()                              'Menu selector left holder
SHARED MenuSelectRH$()                              'Menu selector right holder
                                                    'Title area
SHARED MenuTitle$()                                 'Title/Header for menu
SHARED MenuTitleInfo() AS MenuTopBtmText            'Info on the menu title text
SHARED MenuTitleDivdInfo() AS MenuDividerLineInfo   'Info on the menu dividers
                                                    'Comment area
SHARED MenuComment$()                               'Comment text for menu
SHARED MenuCommentInfo() AS MenuTopBtmText          'Info on the menu comment text
SHARED MenuCommentDivdInfo() AS MenuDividerLineInfo 'Info on the comment area dividers
                                                    'Pointer
SHARED MenuLPtr$()                                  'Menu current item pointer on left side
SHARED MenuRPtr$()                                  'Menu current item pointer on right side
SHARED MenuLClr$()                                  'Menu left pointer clear
SHARED MenuRClr$()                                  'Menu right pointer clear
                                                    'Colors
SHARED ClrFMenuBdr%, ClrBMenuBdr%                   'Menu Boader Colors
SHARED ClrFMenuTitle%, ClrBMenuTitle%               'Menu title text Colors
SHARED ClrFMenuCom%, ClrBMenuCom%                   'Menu comment text colors
SHARED ClrFMenuItem%, ClrBMenuItem%                 'Menu Normal Items Colors
SHARED ClrFMenuItemCur%, ClrBMenuItemCur%           'Menu Current Item Colors (Highlighted Item)
SHARED ClrFMenuInner%, ClrBMenuInner%               'Menu inner color
SHARED ClrFMenuSel%, ClrBMenuSel%                   'Menu Selector Colors

'***************************************************************************
'* PROCEDURE LOGIC                                                   *******
'***************************************************************************

  'Init the menu
  MenuOpen MenuID%
  MenuInfo(MenuID%).Active = True   'This is an active menu
  MenuInfo(MenuID%).DrawMenu = On.  'Draw the menu when called

  'Read in boarder definition first
  READ MenuInfo(MenuID%).MULX
  READ MenuInfo(MenuID%).MULY
  READ MenuInfo(MenuID%).Boarder
  READ MenuBoarder$(MenuID%)
  READ MenuInfo(MenuID%).SpLBdr
  READ MenuLPtr$(MenuID%)
  MenuInfo(MenuID%).LPntrL = LEN(MenuLPtr$(MenuID%))
  MenuLClr$(MenuID%) = SPACE$(MenuInfo(MenuID%).LPntrL)
  READ MenuInfo(MenuID%).SpLPtr
  READ MenuInfo(MenuID%).SpRItm
  READ MenuRPtr$(MenuID%)
  MenuInfo(MenuID%).RPntrL = LEN(MenuRPtr$(MenuID%))
  MenuRClr$(MenuID%) = SPACE$(MenuInfo(MenuID%).RPntrL)
  READ MenuInfo(MenuID%).SpRBdr

  'Header
  READ MenuInfo(MenuID%).Header
  READ MenuInfo(MenuID%).HeaderBdr
  READ MenuInfo(MenuID%).HeaderLines
  READ MenuInfo(MenuID%).HeaderTextL
  READ MenuInfo(MenuID%).HeaderDivdL
  'Header Text
  FOR I% = 1 TO MenuInfo(MenuID%).HeaderTextL
    READ MenuTitleInfo(MenuID%, I%).X
    READ MenuTitleInfo(MenuID%, I%).Y
    READ MenuTitle$(MenuID%, I%)
    MenuTitleInfo(MenuID%, I%).LenText = LEN(MenuTitle$(MenuID%, I%))
    IF MenuTitleInfo(MenuID%, I%).LenText > MenuInfo(MenuID%).WidOb THEN
      MenuInfo(MenuID%).WidOb = MenuTitleInfo(MenuID%, I%).LenText
    END IF
  NEXT
  'Header divider lines
  FOR I% = 1 TO MenuInfo(MenuID%).HeaderDivdL
    READ MenuTitleDivdInfo(MenuID%, I%).Y
    READ Divd$
    MenuTitleDivdInfo(MenuID%, I%).DCL = LEFT$(Divd$, 1)
    MenuTitleDivdInfo(MenuID%, I%).DCC = MID$(Divd$, 2, 1)
    MenuTitleDivdInfo(MenuID%, I%).DCR = RIGHT$(Divd$, 1)
  NEXT

  'Items Selector
  READ MenuInfo(MenuID%).Selector
  READ MenuSelectLH$(MenuID%)
  READ MenuSelectRH$(MenuID%)
  MenuInfo(MenuID%).SelSize% = LEN(MenuSelectLH$(MenuID%)) + LEN(MenuSelectRH$(MenuID%)) + 1
  READ MenuInfo(MenuID%).SpLItm

  'Menu Items
  READ MenuInfo(MenuID%).Items
  READ MenuInfo(MenuID%).Spacing
  READ MenuInfo(MenuID%).ItemCur
  FOR I% = 1 TO MenuInfo(MenuID%).Items
    READ MenuSelect$(MenuID%, I%)
    READ MenuItems$(MenuID%, I%)
    LenItem% = LEN(MenuItems$(MenuID%, I%))
    IF LenItem% > MenuInfo(MenuID%).WidItem THEN
      MenuInfo(MenuID%).WidItem = LenItem%
    END IF
    IF LenItem% > MenuInfo(MenuID%).WidOb THEN
      MenuInfo(MenuID%).WidOb = LenItem%
    END IF
  NEXT

  'Trailer/Commment
  READ MenuInfo(MenuID%).Comment
  READ MenuInfo(MenuID%).CommentBdr
  READ MenuInfo(MenuID%).CommentLines
  READ MenuInfo(MenuID%).CommentTextL
  READ MenuInfo(MenuID%).CommentDivdL
  'Trailer/Commment Text
  FOR I% = 1 TO MenuInfo(MenuID%).CommentTextL
    READ MenuCommentInfo(MenuID%, I%).X
    READ MenuCommentInfo(MenuID%, I%).Y
    READ MenuComment$(MenuID%, I%)
    MenuCommentInfo(MenuID%, I%).LenText = LEN(MenuComment$(MenuID%, I%))
    IF MenuCommentInfo(MenuID%, I%).LenText > MenuInfo(MenuID%).WidOb THEN
      MenuInfo(MenuID%).WidOb = MenuCommentInfo(MenuID%, I%).LenText
    END IF
  NEXT
  'Trailer/Commment divider lines
  FOR I% = 1 TO MenuInfo(MenuID%).CommentDivdL
    READ MenuCommentDivdInfo(MenuID%, I%).Y
    READ Divd$
    MenuCommentDivdInfo(MenuID%, I%).DCL = LEFT$(Divd$, 1)
    MenuCommentDivdInfo(MenuID%, I%).DCC = MID$(Divd$, 2, 1)
    MenuCommentDivdInfo(MenuID%, I%).DCR = RIGHT$(Divd$, 1)
  NEXT

  'Colors
  READ ClrFMenuBdr%       'Boarder/Divider color
  READ ClrBMenuBdr%
  READ ClrFMenuInner%     'Inner color
  READ ClrBMenuInner%
  READ ClrFMenuTitle%     'Header/Title color
  READ ClrBMenuTitle%
  READ ClrFMenuCom%       'Trailer/Comment color
  READ ClrBMenuCom%
  READ ClrFMenuItem%      'Items normal color
  READ ClrBMenuItem%
  READ ClrFMenuItemCur%   'Current item color
  READ ClrBMenuItemCur%
  READ ClrFMenuPtrs%      'Pointers color
  READ ClrBMenuPtrs%
  READ ClrFMenuSel%       'Selector color
  READ ClrBMenuSel%

  'Exit Flags
  READ MenuInfo(MenuID%).ExitESC
  READ MenuInfo(MenuID%).ExitFunc
  READ MenuInfo(MenuID%).ExitCtrl
  READ MenuInfo(MenuID%).ExitAlt

  'Misc Flags/Counters
  READ MenuInfo(MenuID%).SetToTop
  READ MenuInfo(MenuID%).Flicker

  'Adjust all the pointers on the Y and X axis
  YAdjust MenuID%
  XAdjustBdrSet MenuID%

END SUB

SUB KillMenu (MenuID%)

'***************************************************************************
'*                                                                         *
'*                            Kill/Delete a Menu                           *
'*                                 KillMenu                                *
'*                                                                Ver 1.00 *
'*                                                                         *
'* FUNCTION:  This routine removes a menu from memory and frees the        *
'*            space for another menu.                                      *
'*                                                                         *
'*                                                                         *
'* *********************************************************************** *
'* ARGUMENTS:                                                              *
'* *********************************************************************** *
'*                                                                         *
'*   INPUT:                                                                *
'*     MenuID% : Menu to kill/delete                                       *
'*                                                                         *
'* *********************************************************************** *
'* NOTES & COMMENTS:                                                       *
'* *********************************************************************** *
'*                                                                         *
'*   NOTES:                                                                *
'*          Deleted menu can be reused by another menu after deletion      *
'*                                                                         *
'*                                                                         *
'* Bob Cox '89                                              QuickBASIC 4.5 *
'***************************************************************************

'***************************************************************************
'* SHARED VARIABLES                                                  *******
'***************************************************************************
SHARED NumMenus%                       'Number of menus
SHARED MenuInfo()  AS MenuInformation  'Basic menu information on each menu
SHARED MenuSelectLH$()                 'Menu selector left holder
SHARED MenuSelectRH$()                 'Menu selector right holder
SHARED MenuLPtr$()                     'Menu current item pointer on left side
SHARED MenuRPtr$()                     'Menu current item pointer on right side
SHARED MenuLClr$()                     'Menu pointer clear
SHARED MenuRClr$()                     'Menu pointer clear

'***************************************************************************
'* PROCEDURE LOGIC                                                   *******
'***************************************************************************

  MenuInfo(MenuID%).Active = False

  'Reset Flags and counters
  MenuInfo(MenuID%).Items = None
  'Reset optional spacing
  MenuInfo(MenuID%).SpLBdr = None
  MenuInfo(MenuID%).SpLPtr = None
  MenuInfo(MenuID%).SpLItm = None
  MenuInfo(MenuID%).SpRBdr = None
  MenuInfo(MenuID%).SpRItm = None
  'Reset positioning parms
  MenuInfo(MenuID%).WidOb = None
  MenuInfo(MenuID%).WidItem = None
  MenuInfo(MenuID%).SelSize = None
  'Flags/Options
  MenuInfo(MenuID%).SetToTop = Off.
  MenuInfo(MenuID%).Header = Off.
  MenuInfo(MenuID%).Comment = Off.
  MenuInfo(MenuID%).Selector = Off.
  MenuInfo(MenuID%).Flicker = Off.
  'Exit flags
  MenuInfo(MenuID%).ExitESC = Off.
  MenuInfo(MenuID%).ExitFunc = Off.
  MenuInfo(MenuID%).ExitCtrl = Off.
  MenuInfo(MenuID%).ExitAlt = Off.
  'Header
  MenuInfo(MenuID%).HeaderBdr = Off.
  MenuInfo(MenuID%).HeaderLines = None
  MenuInfo(MenuID%).HeaderTextL = None
  MenuInfo(MenuID%).HeaderDivdL = None
  'Comment
  MenuInfo(MenuID%).CommentBdr = Off.
  MenuInfo(MenuID%).CommentLines = None
  MenuInfo(MenuID%).CommentTextL = None
  MenuInfo(MenuID%).CommentDivdL = None
  'Pointer length
  MenuInfo(MenuID%).LPntrL = None
  MenuInfo(MenuID%).RPntrL = None
  'Reset selector holders
  MenuSelectLH$(MenuID%) = ""      'Menu selector left holder
  MenuSelectRH$(MenuID%) = ""      'Menu selector right holder
  'Reset pointers
  MenuLPtr$(MenuID%) = ""          'Menu current item pointer on left side
  MenuRPtr$(MenuID%) = ""          'Menu current item pointer on right side
  MenuLClr$(MenuID%) = ""          'Menu pointer clear
  MenuRClr$(MenuID%) = ""          'Menu pointer clear

  NumMenus% = NumMenus% - 1

END SUB

SUB Menu (MenuID%, SI%, Rc%)

'***************************************************************************
'*                                                                         *
'*                                Do a Menu                                *
'*                                   Menu                                  *
'*                                                                Ver 1.00 *
'*                                                                         *
'* FUNCTION:  This routine displays a menu and allows the user to select   *
'*            an item from the menu either by using cursor keys to         *
'*            highlight a given item followed by pressing the [Enter] key  *
'*            or by pressing a single key associated with a menu item.     *
'*                                                                         *
'*                                                                         *
'* *********************************************************************** *
'* ARGUMENTS:                                                              *
'* *********************************************************************** *
'*                                                                         *
'*   INPUT:                                                                *
'*     MenuID% : Menu to display/select from                               *
'*                 Menu is not built in this routine                       *
'*                                                                         *
'*   OUTPUT:                                                               *
'*     SI%     : Selected Item from the menu                               *
'*                 1-n(number of items in menu)                            *
'*     Rc%     : Return Code - key pressed to exit routine                 *
'*                 0 - Normal Exit                                         *
'*                27 - [ESC] key pressed                                   *
'*              1-26 - [CTRL][A]-[CTRL]-[Z] (not [CTRL][M]/[Enter])        *
'*                     1000+ - Extended ASCII key pressed                  *
'*                               ALT KEYS                                  *
'*                     1016-1025 [ALT][Q,W,E,R,T,Y,U,I,O,P]                *
'*                     1030-1038 [ALT][A,S,D,F,G,H,J,K,L]                  *
'*                     1044-1050 [ALT][Z,X,C,V,B,N,M]                      *
'*                               FUNCTION KEYS                             *
'*                     1059-1068 [F1]-[F10]                                *
'*                     1084-1093 [Shift][F1]-[Shift][F10]                  *
'*                     1094-1103 [CTRL][F1]-[CTRL][F10]                    *
'*                     1104-1113 [ALT][F1]-[ALT][F10]                      *
'*                                                                         *
'* *********************************************************************** *
'* PROCEDURES:                                                             *
'* *********************************************************************** *
'*                                                                         *
'*   FUNCTIONS & SUBROUTINES CALLED:                                       *
'*                                                                         *
'*   SUBROUTINES:                                                          *
'*     Box           : Draws menu boarder                                  *
'*     CLA           : Clears inner area                                   *
'*     CursorOFF     : Turns off the cursor                                *
'*     CursorOn      : Turns on the cursor                                 *
'*     Flicker       : Flickers/flashes the selected item                  *
'*     GetKeyAndCode : Gets the user's response                            *
'*     MenuHighItem  : Highlights the current item                         *
'*     MenuNormItem  : Removes the highlight                               *
'*                                                                         *
'* *********************************************************************** *
'* NOTES & COMMENTS:                                                       *
'* *********************************************************************** *
'*                                                                         *
'*   NOTES:                                                                *
'*                                                                         *
'*                                                                         *
'* Bob Cox '89                                              QuickBASIC 4.5 *
'***************************************************************************

'***************************************************************************
'* SHARED VARIABLES                                                  *******
'***************************************************************************
SHARED MenuInfo() AS MenuInformation                'Basic menu information on each menu
SHARED MenuItems$()                                 'Menu items - text
                                                    'Various menu elements
SHARED MenuBoarder$()                               'Menu boarder (code or definition)
SHARED MenuSelect$()                                'Selector items for each menu item
SHARED MenuSelectLH$()                              'Menu selector left holder
SHARED MenuSelectRH$()                              'Menu selector right holder
                                                    'Title area
SHARED MenuTitle$()                                 'Title/Header for menu
SHARED MenuTitleInfo() AS MenuTopBtmText            'Info on the menu title text
SHARED MenuTitleDivdInfo() AS MenuDividerLineInfo   'Info on the menu dividers
                                                    'Comment area
SHARED MenuComment$()                               'Comment text for menu
SHARED MenuCommentInfo() AS MenuTopBtmText          'Info on the menu comment text
SHARED MenuCommentDivdInfo() AS MenuDividerLineInfo 'Info on the comment area dividers
                                                    'Colors
SHARED ClrFMenuBdr%, ClrBMenuBdr%                   'Menu Boader Colors
SHARED ClrFMenuTitle%, ClrBMenuTitle%               'Menu title text Colors
SHARED ClrFMenuCom%, ClrBMenuCom%                   'Menu comment text colors
SHARED ClrFMenuItem%, ClrBMenuItem%                 'Menu Normal Items Colors
SHARED ClrFMenuItemCur%, ClrBMenuItemCur%           'Menu Current Item Colors (Highlighted Item)
SHARED ClrFMenuInner%, ClrBMenuInner%               'Menu inner color
SHARED ClrFMenuSel%, ClrBMenuSel%                   'Menu Selector Colors
SHARED ClrFMenuExit%, ClrBMenuExit%                 'Menu Exit Colors
                                                    'Highlighting
SHARED CPos%                                        'Current pos (for flicker effect)

'***************************************************************************
'* PROCEDURE LOGIC                                                   *******
'***************************************************************************

  CursorOFF

  'Put the menu on the screen if draw flag is TRUE
  IF MenuInfo(MenuID%).DrawMenu = True THEN
    IF MenuInfo(MenuID%).Boarder = True THEN
      COLOR ClrFMenuBdr%, ClrBMenuBdr% 'Set the boarder color
      Box MenuInfo(MenuID%).BULX, MenuInfo(MenuID%).BULY, MenuInfo(MenuID%).BLRX, MenuInfo(MenuID%).BLRY, MenuBoarder$(MenuID%)
    END IF
    'Fill the inner area with inner color
    COLOR ClrFMenuInner%, ClrBMenuInner%  'Set the inner color
    CLA MenuInfo(MenuID%).IULX, MenuInfo(MenuID%).IULY, MenuInfo(MenuID%).ILRX, MenuInfo(MenuID%).ILRY
    IF MenuInfo(MenuID%).Header = True THEN
      'Display the header
      'Display boarder line(s)
      COLOR ClrFMenuBdr%, ClrBMenuBdr%
      FOR P% = 1 TO MenuInfo(MenuID%).HeaderDivdL
        LOCATE MenuTitleDivdInfo(MenuID%, P%).AY, MenuInfo(MenuID%).BULX
        PRINT MenuTitleDivdInfo(MenuID%, P%).DCL; STRING$(MenuInfo(MenuID%).MIWid, MenuTitleDivdInfo(MenuID%, P%).DCC); MenuTitleDivdInfo(MenuID%, P%).DCR
      NEXT
      'Display header text
      COLOR ClrFMenuTitle%, ClrBMenuTitle%
      FOR P% = 1 TO MenuInfo(MenuID%).HeaderTextL
        LOCATE MenuTitleInfo(MenuID%, P%).AY, MenuTitleInfo(MenuID%, P%).AX
        PRINT MenuTitle$(MenuID%, P%)
      NEXT
    END IF
    IF MenuInfo(MenuID%).Comment = True THEN
      'Display the trailer/comment area
      'Display boarder line(s)
      COLOR ClrFMenuBdr%, ClrBMenuBdr% 'Set the boarder color
      FOR P% = 1 TO MenuInfo(MenuID%).CommentDivdL
        LOCATE MenuCommentDivdInfo(MenuID%, P%).AY, MenuInfo(MenuID%).BULX
        PRINT MenuCommentDivdInfo(MenuID%, P%).DCL; STRING$(MenuInfo(MenuID%).MIWid, MenuCommentDivdInfo(MenuID%, P%).DCC); MenuCommentDivdInfo(MenuID%, P%).DCR
      NEXT
      'Display comment text
      COLOR ClrFMenuCom%, ClrBMenuCom%
      FOR P% = 1 TO MenuInfo(MenuID%).CommentTextL
        LOCATE MenuCommentInfo(MenuID%, P%).AY, MenuCommentInfo(MenuID%, P%).AX
        PRINT MenuComment$(MenuID%, P%)
      NEXT
    END IF
    'Display the menu items
    COLOR ClrFMenuItem%, ClrBMenuItem%
    PosY% = MenuInfo(MenuID%).IStartY
    FOR Item% = 1 TO MenuInfo(MenuID%).Items
      LOCATE PosY%, MenuInfo(MenuID%).IStartX
      PRINT MenuItems$(MenuID%, Item%)
      PosY% = PosY% + MenuInfo(MenuID%).Spacing
    NEXT
    IF MenuInfo(MenuID%).Selector = On. THEN
      'Display the selectors with their holders
      COLOR ClrFMenuSel%, ClrBMenuSel%
      PosY% = MenuInfo(MenuID%).IStartY
      FOR Item% = 1 TO MenuInfo(MenuID%).Items
        IF MenuSelect$(MenuID%, Item%) <> "" THEN
          'Only print the selector if a key has been set to it
          LOCATE PosY%, MenuInfo(MenuID%).SX
          PRINT MenuSelectLH$(MenuID%); MenuSelect$(MenuID%, Item%); MenuSelectRH$(MenuID%)
        END IF
        PosY% = PosY% + MenuInfo(MenuID%).Spacing
      NEXT
    END IF
  END IF

  'Set the current position
  IF MenuInfo(MenuID%).SetToTop = True THEN
    SI% = 1  'Set top item as current item
  ELSE
    SI% = MenuInfo(MenuID%).ItemCur 'Set current item as last item selected
  END IF
   
  MenuHighItem MenuID%, SI%   'Highlight new current item

  ItemSelected% = False
  WHILE ItemSelected% = False
    'Select Item from menu
    GetKeyAndCode KeyPress$, ExtendedFlag%, KeyCode%
    IF ExtendedFlag% = True THEN
      'Extended ASCII code
      SELECT CASE KeyCode%
        CASE 72 '[Up Arrow]
          IF SI% > 1 THEN
            SI% = SI% - 1 'Move up an item
          ELSE
            SI% = MenuInfo(MenuID%).Items 'Wrap back to bottom
          END IF
        CASE 80 '[Down Arrow]
          IF SI% < MenuInfo(MenuID%).Items THEN
            SI% = SI% + 1 'Move down an item
          ELSE
            SI% = 1 'Wrap back to top
          END IF
        CASE 71 '[Home]
          SI% = 1   'Top of the menu items
        CASE 79 '[End]
          SI% = MenuInfo(MenuID%).Items 'Bottom of menu items
        CASE 16 TO 25, 30 TO 38, 44 TO 50 '[ALT][A]-[ALT][Z]
          IF MenuInfo(MenuID%).ExitAlt = True THEN
            ItemSelected% = True
            Rc% = 1000 + KeyCode% 'Set return code to function key return code
          ELSE
            BEEP 'ALT key exit not valid
          END IF
        CASE 59 TO 68, 84 TO 113 '[F1]-[F10] (59-68)
                                 '[Shift][F1]-[Shift][F10] (84-93)
                                 '[CTRL][F1]-[CTRL][F10] (94-103)
                                 '[ALT][F1]-[ALT][F10] (104-113)
          IF MenuInfo(MenuID%).ExitFunc = True THEN
            ItemSelected% = True
            Rc% = 1000 + KeyCode% 'Set return code to function key return code
          ELSE
            BEEP 'Function key exit not valid
          END IF
        CASE ELSE
          BEEP  'Invalid key press
      END SELECT
    ELSE
      'Normal ASCII code
      SELECT CASE KeyCode%
        CASE 13 '[Enter]
          ItemSelected% = True
          Rc% = 0
        CASE 27 '[ESC]
          IF MenuInfo(MenuID%).ExitESC = True THEN
            ItemSelected% = True
            Rc% = KeyCode%
          ELSE
            BEEP
          END IF
        CASE 1 TO 26  '[CTRL][A]-[CTRL][Z] (not [CTRL][M]/[Enter])
          IF MenuInfo(MenuID%).ExitCtrl = True THEN
            ItemSelected% = True
            Rc% = KeyCode%
          ELSE
            BEEP
          END IF
        CASE ELSE
          IF MenuInfo(MenuID%).Selector = On. THEN
            'Find the option requested by using the selctor
            Sel% = 0  'Set selector pointer to SELECTOR NOT FOUND
            KeyPress$ = UCASE$(KeyPress$) 'Remove case sensitivity
            FOR P% = 1 TO MenuInfo(MenuID%).Items
              IF KeyPress$ = MenuSelect$(MenuID%, P%) THEN
                'Found the item user has selected
                Sel% = P% 'Indicate which item was selected
                P% = MenuInfo(MenuID%).Items
              END IF
            NEXT
            IF Sel% <> 0 THEN
              'Exit the routine with the selected item
              SI% = Sel%
              ItemSelected% = True
              Rc% = 0
            ELSE
              BEEP  'Invalid key press
            END IF
          ELSE
            BEEP  'Invalid key press - no selector keys active
          END IF
      END SELECT
    END IF
    MenuNormItem MenuID%        'Normalize previous item
    MenuHighItem MenuID%, SI%   'Highlight new current item
  WEND

  IF Rc% = 0 THEN
    'Normal selection - flicker item
    Flicker MenuInfo(MenuID%).Flicker, MenuInfo(MenuID%).IStartX, CPos%, ClrFMenuItemCur%, ClrBMenuItemCur%, ClrFMenuItem%, ClrBMenuItem%, MenuItems$(MenuID%, SI%)
  END IF

  MenuInfo(MenuID%).ItemCur = SI%     'Save position of last selected item
  COLOR ClrFMenuExit%, ClrBMenuExit%  'Set menu exit color
  CursorON

END SUB

SUB MenuBoarders (MenuID%, OnOff%)

'***************************************************************************
'*                                                                         *
'*                      Turn the Menu Boarders On/Off                      *
'*                               MenuBoarders                              *
'*                                                                Ver 1.00 *
'*                                                                         *
'* FUNCTION:  This routine controls the boarder option. The boarders can   *
'*            be turned ON or OFF.                                         *
'*                                                                         *
'*                                                                         *
'* *********************************************************************** *
'* ARGUMENTS:                                                              *
'* *********************************************************************** *
'*                                                                         *
'*   INPUT:                                                                *
'*     MenuID% : Menu ID                                                   *
'*     OnOff%  : Flag for Option                                           *
'*               1 - Boarders ON                                           *
'*               0 - Boarders OFF                                          *
'*                                                                         *
'* *********************************************************************** *
'* NOTES & COMMENTS:                                                       *
'* *********************************************************************** *
'*                                                                         *
'*   NOTES:                                                                *
'*          The space used by the boarder is always alocated even if the   *
'*          boarder is OFF it will remain as spaces.                       *
'*                                                                         *
'* Bob Cox '89                                              QuickBASIC 4.5 *
'***************************************************************************

'***************************************************************************
'* SHARED VARIABLES                                                  *******
'***************************************************************************
SHARED MenuInfo()  AS MenuInformation  'Basic menu information on each menu

'***************************************************************************
'* PROCEDURE LOGIC                                                   *******
'***************************************************************************

  MenuInfo(MenuID%).Boarder = OnOff%

END SUB

SUB MenuCommentArea (MenuID%, OnOff%)

'***************************************************************************
'*                                                                         *
'*                    Turn the Menu Comment Area On/Off                    *
'*                             MenuCommentArea                             *
'*                                                                Ver 1.00 *
'*                                                                         *
'* FUNCTION:  This routine diplays or removes the comment area at the      *
'*            bottom of the menu if a comment area has been set.           *
'*                                                                         *
'*                                                                         *
'* *********************************************************************** *
'* ARGUMENTS:                                                              *
'* *********************************************************************** *
'*                                                                         *
'*   INPUT:                                                                *
'*     MenuID% : Menu ID                                                   *
'*     OnOff%  : Flag for Option                                           *
'*               1 - Comment Area ON                                       *
'*               0 - Comment Area OFF                                      *
'*                                                                         *
'* *********************************************************************** *
'* PROCEDURES:                                                             *
'* *********************************************************************** *
'*                                                                         *
'*   FUNCTIONS & SUBROUTINES CALLED:                                       *
'*                                                                         *
'*   SUBROUTINES:                                                          *
'*     YAdjust : Adjusts pointers on the Y axis                            *
'*                                                                         *
'* Bob Cox '89                                              QuickBASIC 4.5 *
'***************************************************************************

'***************************************************************************
'* SHARED VARIABLES                                                  *******
'***************************************************************************
SHARED MenuInfo() AS MenuInformation           'Basic menu information on each menu

'***************************************************************************
'* PROCEDURE LOGIC                                                   *******
'***************************************************************************

  MenuInfo(MenuID%).Comment = OnOff%  'Set the flag
  YAdjust MenuID%                     'Adjust the Y axis pointers if needed

END SUB

SUB MenuCommentDivd (MenuID%, Y%, Divd$)

'***************************************************************************
'*                                                                         *
'*                   Place a Divider in the Comment Area                   *
'*                             MenuCommentDivd                             *
'*                                                                Ver 1.00 *
'*                                                                         *
'* FUNCTION:   This "routine" provides a simple calling interface to       *
'*             MenuCommentDivdSet.                                         *
'*                                                                         *
'*                                                                         *
'* *********************************************************************** *
'* ARGUMENTS:                                                              *
'* *********************************************************************** *
'*                                                                         *
'*   INPUT:                                                                *
'*     MenuID% : Menu ID                                                   *
'*     Y%      : Position in Comment Area                                  *
'*     Divd$   : Divider (3 character string)                              *
'*                                                                         *
'* *********************************************************************** *
'* PROCEDURES:                                                             *
'* *********************************************************************** *
'*                                                                         *
'*   FUNCTIONS & SUBROUTINES CALLED:                                       *
'*                                                                         *
'*   SUBROUTINES:                                                          *
'*     MenuCommentDivdSet : Sets a divider line for the comment area       *
'*                                                                         *
'* *********************************************************************** *
'* NOTES & COMMENTS:                                                       *
'* *********************************************************************** *
'*                                                                         *
'*   NOTES:                                                                *
'*                                                                         *
'*                                                                         *
'* Bob Cox '89                                              QuickBASIC 4.5 *
'***************************************************************************

'***************************************************************************
'* PROCEDURE LOGIC                                                   *******
'***************************************************************************

  MenuCommentDivdSet MenuID%, 0, Y%, Divd$

END SUB

SUB MenuCommentDivdSet (MenuID%, DivID%, Y%, Divd$)

'***************************************************************************
'*                                                                         *
'*                   Place a Divider in the Comment Area                   *
'*                            MenuCommentDivdSet                           *
'*                                                                Ver 1.00 *
'*                                                                         *
'* FUNCTION:  This routine places a divider line in the comment area.      *
'*            The divider can be used to separate the menu items from      *
'*            the comment text.                                            *
'*                                                                         *
'*                                                                         *
'* *********************************************************************** *
'* ARGUMENTS:                                                              *
'* *********************************************************************** *
'*                                                                         *
'*   INPUT:                                                                *
'*     MenuID% : Menu ID                                                   *
'*     DivID%  : Divider line array position (0 - Next avail)              *
'*     Y%      : Position in Comment Area                                  *
'*     Divd$   : Divider (3 character string)                              *
'*                                                                         *
'* *********************************************************************** *
'* PROCEDURES:                                                             *
'* *********************************************************************** *
'*                                                                         *
'*   FUNCTIONS & SUBROUTINES CALLED:                                       *
'*                                                                         *
'*   SUBROUTINES:                                                          *
'*     YAdjust : Adjust all the pointers on the Y axis                     *
'*                                                                         *
'* *********************************************************************** *
'* NOTES & COMMENTS:                                                       *
'* *********************************************************************** *
'*                                                                         *
'*   NOTES:                                                                *
'*          DivID% : if = 0 it defaults to next avail position.            *
'*                   if > 0 will redefine the line at that position.       *
'*                                                                         *
'*                                                                         *
'* Bob Cox '89                                              QuickBASIC 4.5 *
'***************************************************************************

'***************************************************************************
'* SHARED VARIABLES                                                  *******
'***************************************************************************
SHARED MenuInfo() AS MenuInformation                  'Basic menu information on each menu
SHARED MenuCommentDivdInfo() AS MenuDividerLineInfo   'Info on the menu comment dividers

'***************************************************************************
'* PROCEDURE LOGIC                                                   *******
'***************************************************************************

  MenuInfo(MenuID%).Comment = True  'Turn on the comment area

  IF DivID% = 0 THEN
    'This is a new divider line - assign to next position in array
    MenuInfo(MenuID%).CommentDivdL = MenuInfo(MenuID%).CommentDivdL + 1
    DivID% = MenuInfo(MenuID%).CommentDivdL
  END IF

  'Save the segments of the bar
  MenuCommentDivdInfo(MenuID%, DivID%).DCL = LEFT$(Divd$, 1)
  MenuCommentDivdInfo(MenuID%, DivID%).DCC = MID$(Divd$, 2, 1)
  MenuCommentDivdInfo(MenuID%, DivID%).DCR = RIGHT$(Divd$, 1)

  IF Y% = 0 THEN
    'Place the divider on the next avail line in the comment area
    MenuInfo(MenuID%).CommentLines = MenuInfo(MenuID%).CommentLines + 1
    MenuCommentDivdInfo(MenuID%, DivID%).Y = MenuInfo(MenuID%).CommentLines
  ELSE
    IF Y% > MenuInfo(MenuID%).CommentLines THEN
      MenuInfo(MenuID%).CommentLines = Y%
    END IF
    MenuCommentDivdInfo(MenuID%, DivID%).Y = Y%
  END IF

  'Calc actual Y of divider line from realitive positioning parm passed by caller
  'and adjust the menu on the Y axis if needed
  YAdjust MenuID%

END SUB

SUB MenuCommentInBdr (MenuID%, OnOff%)

'***************************************************************************
'*                                                                         *
'*                Start the Menu Comment Area in the Boarder               *
'*                             MenuCommentInBdr                            *
'*                                                                Ver 1.00 *
'*                                                                         *
'* FUNCTION:  This routine sets the "Comment In Boarder" flag ON/OFF.      *
'*            When ON the comment area extends into the boarder, when      *
'*            OFF the comment area is contained by the boarder.            *
'*                                                                         *
'*                                                                         *
'* *********************************************************************** *
'* ARGUMENTS:                                                              *
'* *********************************************************************** *
'*                                                                         *
'*   INPUT:                                                                *
'*     MenuID% : Menu ID                                                   *
'*     OnOff%  : Flag for Option                                           *
'*               1 - Comment Area Starts in Boarders ON                    *
'*               0 - Comment Area Starts in Boarders OFF                   *
'*                                                                         *
'* *********************************************************************** *
'* PROCEDURES:                                                             *
'* *********************************************************************** *
'*                                                                         *
'*   FUNCTIONS & SUBROUTINES CALLED:                                       *
'*                                                                         *
'*   SUBROUTINES:                                                          *
'*     YAdjust : Adjust all the pointers on the Y axis                     *
'*                                                                         *
'* *********************************************************************** *
'* NOTES & COMMENTS:                                                       *
'* *********************************************************************** *
'*                                                                         *
'*   NOTES:                                                                *
'*                                                                         *
'*                                                                         *
'* Bob Cox '89                                              QuickBASIC 4.5 *
'***************************************************************************

'***************************************************************************
'* SHARED VARIABLES                                                  *******
'***************************************************************************
SHARED MenuInfo()  AS MenuInformation  'Basic menu information on each menu

'***************************************************************************
'* PROCEDURE LOGIC                                                   *******
'***************************************************************************

  MenuInfo(MenuID%).CommentBdr = OnOff%
  YAdjust MenuID%

END SUB

SUB MenuCommentText (MenuID%, X%, Y%, Text$)

'***************************************************************************
'*                                                                         *
'*                 Place a Line of Text in the Comment Area                *
'*                             MenuCommentText                             *
'*                                                                Ver 1.00 *
'*                                                                         *
'* FUNCTION:   This "routine" provides a simple calling interface to       *
'*             MenuCommentTextSet.                                         *
'*                                                                         *
'*                                                                         *
'* *********************************************************************** *
'* ARGUMENTS:                                                              *
'* *********************************************************************** *
'*                                                                         *
'*   INPUT:                                                                *
'*     MenuID% : Menu ID                                                   *
'*     X%      : Relative X position from left side of boarder             *
'*                 0 - will auto center text between boarders              *
'*     Y%      : Relative Y position in the comment area                   *
'*                 0 - will become next line in comment area               *
'*     Text$   : Text to be placed in the comment area                     *
'*                                                                         *
'* *********************************************************************** *
'* PROCEDURES:                                                             *
'* *********************************************************************** *
'*                                                                         *
'*   FUNCTIONS & SUBROUTINES CALLED:                                       *
'*                                                                         *
'*   SUBROUTINES:                                                          *
'*     MenuCommentTextSet : Sets a text line for the comment area          *
'*                                                                         *
'* *********************************************************************** *
'* NOTES & COMMENTS:                                                       *
'* *********************************************************************** *
'*                                                                         *
'*   NOTES:                                                                *
'*                                                                         *
'*                                                                         *
'* Bob Cox '89                                              QuickBASIC 4.5 *
'***************************************************************************

'***************************************************************************
'* PROCEDURE LOGIC                                                   *******
'***************************************************************************

  MenuCommentTextSet MenuID%, 0, X%, Y%, Text$

END SUB

SUB MenuCommentTextSet (MenuID%, TextID%, X%, Y%, Text$)

'***************************************************************************
'*                                                                         *
'*                 Place a Line of Text in the Comment Area                *
'*                            MenuCommentTextSet                           *
'*                                                                Ver 1.00 *
'*                                                                         *
'* FUNCTION:  This routine places text in the comment area, multiple       *
'*            lines of text can be defined.                                *
'*                                                                         *
'*                                                                         *
'* *********************************************************************** *
'* ARGUMENTS:                                                              *
'* *********************************************************************** *
'*                                                                         *
'*   INPUT:                                                                *
'*     MenuID% : Menu ID                                                   *
'*     TextID% : Comment text array position (0 - Next avail)              *
'*     X%      : Relative X position from left side of boarder             *
'*                 0 - will auto center text between boarders              *
'*     Y%      : Relative Y position in the comment area                   *
'*                 0 - will become next line in comment area               *
'*     Text$   : Text to be placed in the comment area                     *
'*                                                                         *
'* *********************************************************************** *
'* PROCEDURES:                                                             *
'* *********************************************************************** *
'*                                                                         *
'*   FUNCTIONS & SUBROUTINES CALLED:                                       *
'*                                                                         *
'*   SUBROUTINES:                                                          *
'*     XAdjustAddCtr : Sets the widest object pointer                      *
'*     YAdjust       : Adjust all the pointers on the Y axis               *
'*                                                                         *
'* *********************************************************************** *
'* NOTES & COMMENTS:                                                       *
'* *********************************************************************** *
'*                                                                         *
'*   NOTES:                                                                *
'*                                                                         *
'*                                                                         *
'* Bob Cox '89                                              QuickBASIC 4.5 *
'***************************************************************************

'***************************************************************************
'* SHARED VARIABLES                                                  *******
'***************************************************************************
SHARED MenuInfo() AS MenuInformation       'Basic menu information on each menu
SHARED MenuComment$()                      'Comment text for menu
SHARED MenuCommentInfo() AS MenuTopBtmText 'Info on the menu comment text

'***************************************************************************
'* PROCEDURE LOGIC                                                   *******
'***************************************************************************

  MenuInfo(MenuID%).Comment = True  'Turn on the comment area

  IF TextID% = 0 THEN
    'This is new comment text - assign to next position in array
    MenuInfo(MenuID%).CommentTextL = MenuInfo(MenuID%).CommentTextL + 1
    TextID% = MenuInfo(MenuID%).CommentTextL
  END IF

  'Save the parms-may be needed for later recalc
  MenuCommentInfo(MenuID%, TextID%).X = X%
  MenuCommentInfo(MenuID%, TextID%).LenText = LEN(Text$)

  'Save the comment text
  MenuComment$(MenuID%, TextID%) = Text$

  'Calc actual X of comment text from realitive positioning parm passed by caller
  'and adjust the menu on the X axis if needed
  XAdjustAddCtr MenuID%, MenuCommentInfo(MenuID%, TextID%).LenText

  IF Y% = 0 THEN
    'Place the text on the next avail line in the comment area (bottom of menu)
    MenuInfo(MenuID%).CommentLines = MenuInfo(MenuID%).CommentLines + 1
    MenuCommentInfo(MenuID%, TextID%).Y = MenuInfo(MenuID%).CommentLines
  ELSE
    IF Y% > MenuInfo(MenuID%).CommentLines THEN
      MenuInfo(MenuID%).CommentLines = Y%
    END IF
    MenuCommentInfo(MenuID%, TextID%).Y = Y%
  END IF

  'Calc actual Y of comment text from realitive positioning parm passed by caller
  'and adjust the menu on the Y axis if needed
  YAdjust MenuID%

END SUB

SUB MenuDraw (MenuID%, OnOff%)

'***************************************************************************
'*                                                                         *
'*                       Set the Menu Draw Flag On/Off                     *
'*                                 MenuDraw                                *
'*                                                                Ver 1.00 *
'*                                                                         *
'* FUNCTION:  This routine sets the menu draw flag on/off.  This flag      *
'*            can be used to keep the menu from redrawing if it is         *
'*            still displayed on the screen after a menu selection.        *
'*                                                                         *
'*                                                                         *
'* *********************************************************************** *
'* ARGUMENTS:                                                              *
'* *********************************************************************** *
'*                                                                         *
'*   INPUT:                                                                *
'*     MenuID% : Menu ID                                                   *
'*     OnOff%  : Flag for Option                                           *
'*               1 - Draw the Menu when Menu is Called ON                  *
'*               0 - Draw the Menu when Menu is Called OFF                 *
'*                                                                         *
'* *********************************************************************** *
'* NOTES & COMMENTS:                                                       *
'* *********************************************************************** *
'*                                                                         *
'*   NOTES:                                                                *
'*                                                                         *
'*                                                                         *
'* Bob Cox '89                                              QuickBASIC 4.5 *
'***************************************************************************

'***************************************************************************
'* SHARED VARIABLES                                                  *******
'***************************************************************************
SHARED MenuInfo() AS MenuInformation       'Basic menu information on each menu

'***************************************************************************
'* PROCEDURE LOGIC                                                   *******
'***************************************************************************
 
  MenuInfo(MenuID%).DrawMenu = OnOff%

END SUB

SUB MenuExitALT (MenuID%, OnOff%)

'***************************************************************************
'*                                                                         *
'*                          Exit Menu on ALT Keys                          *
'*                               MenuExitALT                               *
'*                                                                Ver 1.00 *
'*                                                                         *
'* FUNCTION:  This routine sets the "Exit On ALT Keys" flag ON/OFF.        *
'*            When ON the menu can be exited on ALT keys.                  *
'*                                                                         *
'*                                                                         *
'* *********************************************************************** *
'* ARGUMENTS:                                                              *
'* *********************************************************************** *
'*                                                                         *
'*   INPUT:                                                                *
'*     MenuID% : Menu ID                                                   *
'*     OnOff%  : Flag for Option                                           *
'*               1 - Exit on ALT Keys ON                                   *
'*               0 - Exit on ALT Keys OFF                                  *
'*                                                                         *
'* *********************************************************************** *
'* NOTES & COMMENTS:                                                       *
'* *********************************************************************** *
'*                                                                         *
'*   NOTES:                                                                *
'*                                                                         *
'*                                                                         *
'* Bob Cox '89                                              QuickBASIC 4.5 *
'***************************************************************************

'***************************************************************************
'* SHARED VARIABLES                                                  *******
'***************************************************************************
SHARED MenuInfo() AS MenuInformation    'Basic menu information on each menu

'***************************************************************************
'* PROCEDURE LOGIC                                                   *******
'***************************************************************************

  MenuInfo(MenuID%).ExitAlt = OnOff%

END SUB

SUB MenuExitCTRL (MenuID%, OnOff%)

'***************************************************************************
'*                                                                         *
'*                          Exit Menu on CTRL Keys                         *
'*                               MenuExitCTRL                              *
'*                                                                Ver 1.00 *
'*                                                                         *
'* FUNCTION:  This routine sets the "Exit On CTRL Keys" flag ON/OFF.       *
'*            When ON the menu can be exited on CTRL keys.                 *
'*                                                                         *
'*                                                                         *
'* *********************************************************************** *
'* ARGUMENTS:                                                              *
'* *********************************************************************** *
'*                                                                         *
'*   INPUT:                                                                *
'*     MenuID% : Menu ID                                                   *
'*     OnOff%  : Flag for Option                                           *
'*               1 - Exit on CTRL Keys ON                                  *
'*               0 - Exit on CTRL Keys OFF                                 *
'*                                                                         *
'* *********************************************************************** *
'* NOTES & COMMENTS:                                                       *
'* *********************************************************************** *
'*                                                                         *
'*   NOTES:                                                                *
'*                                                                         *
'*                                                                         *
'* Bob Cox '89                                              QuickBASIC 4.5 *
'***************************************************************************

'***************************************************************************
'* SHARED VARIABLES                                                  *******
'***************************************************************************
SHARED MenuInfo() AS MenuInformation    'Basic menu information on each menu

'***************************************************************************
'* PROCEDURE LOGIC                                                   *******
'***************************************************************************

  MenuInfo(MenuID%).ExitCtrl = OnOff%

END SUB

SUB MenuExitESC (MenuID%, OnOff%)

'***************************************************************************
'*                                                                         *
'*                           Exit Menu on ESC Key                          *
'*                               MenuExitESC                               *
'*                                                                Ver 1.00 *
'*                                                                         *
'* FUNCTION:  This routine sets the "Exit On ESC Key" flag ON/OFF.         *
'*            When ON the menu can be exited on ESC key.                   *
'*                                                                         *
'* *********************************************************************** *
'* ARGUMENTS:                                                              *
'* *********************************************************************** *
'*                                                                         *
'*   INPUT:                                                                *
'*     MenuID% : Menu ID                                                   *
'*     OnOff%  : Flag for Option                                           *
'*               1 - Exit on ESC Key ON                                    *
'*               0 - Exit on ESC Key OFF                                   *
'*                                                                         *
'* *********************************************************************** *
'* NOTES & COMMENTS:                                                       *
'* *********************************************************************** *
'*                                                                         *
'*   NOTES:                                                                *
'*                                                                         *
'*                                                                         *
'* Bob Cox '89                                              QuickBASIC 4.5 *
'***************************************************************************

'***************************************************************************
'* SHARED VARIABLES                                                  *******
'***************************************************************************
SHARED MenuInfo() AS MenuInformation           'Basic menu information on each menu

'***************************************************************************
'* PROCEDURE LOGIC                                                   *******
'***************************************************************************

  MenuInfo(MenuID%).ExitESC = OnOff%

END SUB

SUB MenuExitFUNC (MenuID%, OnOff%)

'***************************************************************************
'*                                                                         *
'*                        Exit Menu on Function Keys                       *
'*                               MenuExitFUNC                              *
'*                                                                Ver 1.00 *
'*                                                                         *
'* FUNCTION:  This routine sets the "Exit On Function Keys" flag ON/OFF.   *
'*            When ON the menu can be exited on Function keys.             *
'*                                                                         *
'*                                                                         *
'* *********************************************************************** *
'* ARGUMENTS:                                                              *
'* *********************************************************************** *
'*                                                                         *
'*   INPUT:                                                                *
'*     MenuID% : Menu ID                                                   *
'*     OnOff%  : Flag for Option                                           *
'*               1 - Exit on FUNCTION Keys ON                              *
'*               0 - Exit on FUNCTION Keys OFF                             *
'*                                                                         *
'* *********************************************************************** *
'* NOTES & COMMENTS:                                                       *
'* *********************************************************************** *
'*                                                                         *
'*   NOTES:                                                                *
'*                                                                         *
'*                                                                         *
'* Bob Cox '89                                              QuickBASIC 4.5 *
'***************************************************************************

'***************************************************************************
'* SHARED VARIABLES                                                  *******
'***************************************************************************
SHARED MenuInfo() AS MenuInformation    'Basic menu information on each menu

'***************************************************************************
'* PROCEDURE LOGIC                                                   *******
'***************************************************************************

  MenuInfo(MenuID%).ExitFunc = OnOff%

END SUB

SUB MenuHighItem (MenuID%, Item%)

'***************************************************************************
'*                                                                         *
'*                      Highlight Current Item in Menu                     *
'*                               MenuHighItem                              *
'*                                                                Ver 1.00 *
'*                                                                         *
'* FUNCTION:  This routine highlights the current item in the menu         *
'*            using color changes and/or pointer(s).                       *
'*                                                                         *
'*                                                                         *
'* *********************************************************************** *
'* ARGUMENTS:                                                              *
'* *********************************************************************** *
'*                                                                         *
'*   INPUT:                                                                *
'*     MenuID% : Menu ID                                                   *
'*     Item%   : Item to Highlight/Set to Current Item                     *
'*                                                                         *
'* *********************************************************************** *
'* NOTES & COMMENTS:                                                       *
'* *********************************************************************** *
'*                                                                         *
'*   NOTES: This routine is the opposite of MenuNormItem and shares some   *
'*          global variables with it.                                      *
'*                                                                         *
'*                                                                         *
'* Bob Cox '89                                              QuickBASIC 4.5 *
'***************************************************************************

'***************************************************************************
'* SHARED VARIABLES                                                  *******
'***************************************************************************
SHARED MenuInfo() AS MenuInformation   'Basic menu information on each menu
SHARED MenuItems$()                    'Menu items - text
SHARED MenuLPtr$()                     'Menu current item pointer on left side
SHARED MenuRPtr$()                     'Menu current item pointer on right side

SHARED PPos%                           'Previous Position
SHARED PItem%                          'Previous Item
SHARED CPos%                           'Current pos (for flicker effect)

'Colors
SHARED ClrFMenuItemCur%, ClrBMenuItemCur%    'Menu Current Item Colors (Highlighted Item)
SHARED ClrFMenuPtrs%, ClrBMenuPtrs%          'Menu Current Item Pointers Colors

'***************************************************************************
'* PROCEDURE LOGIC                                                   *******
'***************************************************************************

    CPos% = MenuInfo(MenuID%).IStartY + (Item% - 1) * MenuInfo(MenuID%).Spacing
    PItem% = Item%

    'Highlight current item
    COLOR ClrFMenuItemCur%, ClrBMenuItemCur%
    LOCATE CPos%, MenuInfo(MenuID%).IStartX
    PRINT MenuItems$(MenuID%, Item%)

    'Display the pointers
    COLOR ClrFMenuPtrs%, ClrBMenuPtrs%
    LOCATE CPos%, MenuInfo(MenuID%).LPntrX
    PRINT MenuLPtr$(MenuID%) 'Display left pointer
    LOCATE CPos%, MenuInfo(MenuID%).RPntrX
    PRINT MenuRPtr$(MenuID%) 'Display right pointer

    PPos% = CPos% 'Set previous item

END SUB

SUB MenuHolders (MenuID%, LHolder$, RHolder$)

'***************************************************************************
'*                                                                         *
'*                       Set the Menu Selector Holders                     *
'*                               MenuHolders                               *
'*                                                                Ver 1.00 *
'*                                                                         *
'* FUNCTION:  This routine sets the selector key holders or removes then.  *
'*            One or both holders can be set.                              *
'*                                                                         *
'*                                                                         *
'* *********************************************************************** *
'* ARGUMENTS:                                                              *
'* *********************************************************************** *
'*                                                                         *
'*   INPUT:                                                                *
'*     MenuID%  : Menu ID                                                  *
'*     LHolder$ : Left holder character(s) to set or null to remove        *
'*     RHolder$ : Right holder character(s) to set or null to remove       *
'*                                                                         *
'* *********************************************************************** *
'* PROCEDURES:                                                             *
'* *********************************************************************** *
'*                                                                         *
'*   FUNCTIONS & SUBROUTINES CALLED:                                       *
'*                                                                         *
'*   SUBROUTINES:                                                          *
'*     XAdjustCalcSelSize : Reset/adjust the pointers on the X axis        *
'*                                                                         *
'* *********************************************************************** *
'* NOTES & COMMENTS:                                                       *
'* *********************************************************************** *
'*                                                                         *
'*   NOTES:                                                                *
'*          When called the menu pointers are updated on the X axis        *
'*          to reflect the holders                                         *
'*                                                                         *
'*          The Selector flag is set to ON                                 *
'*                                                                         *
'*                                                                         *
'* Bob Cox '89                                              QuickBASIC 4.5 *
'***************************************************************************

'***************************************************************************
'* SHARED VARIABLES                                                  *******
'***************************************************************************
SHARED MenuInfo() AS MenuInformation  'Basic menu information on each menu
SHARED MenuSelectLH$()                'Menu selector left holder
SHARED MenuSelectRH$()                'Menu selector right holder

'***************************************************************************
'* PROCEDURE LOGIC                                                   *******
'***************************************************************************

  MenuInfo(MenuID%).Selector = On.

  MenuSelectLH$(MenuID%) = LHolder$
  MenuSelectRH$(MenuID%) = RHolder$
  XAdjustCalcSelSize MenuID%

END SUB

SUB MenuItem (MenuID%, SelectKey$, MenuItemText$)

'***************************************************************************
'*                                                                         *
'*                        Put an Item into the Menu                        *
'*                                 MenuItem                                *
'*                                                                Ver 1.00 *
'*                                                                         *
'* FUNCTION:   This "routine" provides a simple calling interface to       *
'*             MenuItemSet.                                                *
'*                                                                         *
'*                                                                         *
'* *********************************************************************** *
'* ARGUMENTS:                                                              *
'* *********************************************************************** *
'*                                                                         *
'*   INPUT:                                                                *
'*     MenuID%       : Menu ID                                             *
'*     SelectKey$    : Selector key for the menu item                      *
'*     MenuItemText$ : Text of the menu item                               *
'*                                                                         *
'* *********************************************************************** *
'* PROCEDURES:                                                             *
'* *********************************************************************** *
'*                                                                         *
'*   FUNCTIONS & SUBROUTINES CALLED:                                       *
'*                                                                         *
'*   SUBROUTINES:                                                          *
'*     MenuItemSet : Put an item into the menu                             *
'*                                                                         *
'* *********************************************************************** *
'* NOTES & COMMENTS:                                                       *
'* *********************************************************************** *
'*                                                                         *
'*   NOTES:                                                                *
'*                                                                         *
'*                                                                         *
'* Bob Cox '89                                              QuickBASIC 4.5 *
'***************************************************************************

'***************************************************************************
'* PROCEDURE LOGIC                                                   *******
'***************************************************************************

  MenuItemSet MenuID%, 0, SelectKey$, MenuItemText$

END SUB

SUB MenuItemPointerL (MenuID%, Pointer$)

'***************************************************************************
'*                                                                         *
'*                  Set the Left Side Pointer to the Items                 *
'*                             MenuItemPointerL                            *
'*                                                                Ver 1.00 *
'*                                                                         *
'* FUNCTION:  This routine sets the left pointer that will point at the    *
'*            current or highlighted item.  Any character(s) can act as    *
'*            the pointer, null removes it.                                *
'*                                                                         *
'*                                                                         *
'* *********************************************************************** *
'* ARGUMENTS:                                                              *
'* *********************************************************************** *
'*                                                                         *
'*   INPUT:                                                                *
'*     MenuID%  : Menu ID                                                  *
'*     Pointer$ : Character(s) to use for the left pointer                 *
'*                                                                         *
'* *********************************************************************** *
'* PROCEDURES:                                                             *
'* *********************************************************************** *
'*                                                                         *
'*   FUNCTIONS & SUBROUTINES CALLED:                                       *
'*                                                                         *
'*   SUBROUTINES:                                                          *
'*     XAdjustBdrSet : Sets the pointers for the menu boarders             *
'*                                                                         *
'* *********************************************************************** *
'* NOTES & COMMENTS:                                                       *
'* *********************************************************************** *
'*                                                                         *
'*   NOTES:                                                                *
'*                                                                         *
'*                                                                         *
'* Bob Cox '89                                              QuickBASIC 4.5 *
'***************************************************************************

'***************************************************************************
'* SHARED VARIABLES                                                  *******
'***************************************************************************
SHARED MenuInfo()  AS MenuInformation  'Basic menu information on each menu
SHARED MenuLPtr$()                     'Menu current item pointer on left side
SHARED MenuLClr$()                     'Menu left pointer clear

'***************************************************************************
'* PROCEDURE LOGIC                                                   *******
'***************************************************************************

  MenuInfo(MenuID%).LPntrL = LEN(Pointer$)
  MenuLPtr$(MenuID%) = Pointer$
  MenuLClr$(MenuID%) = SPACE$(MenuInfo(MenuID%).LPntrL)
  XAdjustBdrSet MenuID%

END SUB

SUB MenuItemPointerR (MenuID%, Pointer$)

'***************************************************************************
'*                                                                         *
'*                 Set the Right Side Pointer to the Items                 *
'*                             MenuItemPointerR                            *
'*                                                                Ver 1.00 *
'*                                                                         *
'* FUNCTION:  This routine sets the right pointer that will point at the   *
'*            current or highlighted item.  Any character(s) can act as    *
'*            the pointer, null removes it.                                *
'*                                                                         *
'*                                                                         *
'* *********************************************************************** *
'* ARGUMENTS:                                                              *
'* *********************************************************************** *
'*                                                                         *
'*   INPUT:                                                                *
'*     MenuID%  : Menu ID                                                  *
'*     Pointer$ : Character(s) to use for the right pointer                *
'*                                                                         *
'* *********************************************************************** *
'* PROCEDURES:                                                             *
'* *********************************************************************** *
'*                                                                         *
'*   FUNCTIONS & SUBROUTINES CALLED:                                       *
'*                                                                         *
'*   SUBROUTINES:                                                          *
'*     XAdjustBdrSet : Sets the pointers for the menu boarders             *
'*                                                                         *
'* *********************************************************************** *
'* NOTES & COMMENTS:                                                       *
'* *********************************************************************** *
'*                                                                         *
'*   NOTES:                                                                *
'*                                                                         *
'*                                                                         *
'* Bob Cox '89                                              QuickBASIC 4.5 *
'***************************************************************************

'***************************************************************************
'* SHARED VARIABLES                                                  *******
'***************************************************************************
SHARED MenuInfo()  AS MenuInformation  'Basic menu information on each menu
SHARED MenuRPtr$()                     'Menu current item pointer on right side
SHARED MenuRClr$()                     'Menu right pointer clear

'***************************************************************************
'* PROCEDURE LOGIC                                                   *******
'***************************************************************************

  MenuInfo(MenuID%).RPntrL = LEN(Pointer$)
  MenuRPtr$(MenuID%) = Pointer$
  MenuRClr$(MenuID%) = SPACE$(MenuInfo(MenuID%).RPntrL)
  XAdjustBdrSet MenuID%

END SUB

SUB MenuItemSet (MenuID%, ItemID%, SelectKey$, MenuItemText$)

'***************************************************************************
'*                                                                         *
'*                        Put an Item into the Menu                        *
'*                               MenuItemSet                               *
'*                                                                Ver 1.00 *
'*                                                                         *
'* FUNCTION:  This routine places the menu items text and their selector   *
'*            keys into the menu.  This routine is the main menu building  *
'*            routine if InitMenuFromDATA is not used to build the menu.   *
'*                                                                         *
'*                                                                         *
'* *********************************************************************** *
'* ARGUMENTS:                                                              *
'* *********************************************************************** *
'*                                                                         *
'*   INPUT:                                                                *
'*     MenuID%       : Menu ID                                             *
'*     ItemID%       : Item Position in the Menu                           *
'*     SelectKey$    : Select Key/Character for Menu Item                  *
'*     MenuItemText$ : Menu Item Text                                      *
'*                                                                         *
'* *********************************************************************** *
'* PROCEDURES:                                                             *
'* *********************************************************************** *
'*                                                                         *
'*   FUNCTIONS & SUBROUTINES CALLED:                                       *
'*                                                                         *
'*   SUBROUTINES:                                                          *
'*     XAdjustAddCtr      : Sets the widest object pointer                 *
'*     XAdjustCalcSelSize : Calculate and set the selector size            *
'*     YAdjust            : Adjust all the pointers on the Y axis          *
'*                                                                         *
'* *********************************************************************** *
'* NOTES & COMMENTS:                                                       *
'* *********************************************************************** *
'*                                                                         *
'*   NOTES:                                                                *
'*                                                                         *
'*                                                                         *
'* Bob Cox '89                                              QuickBASIC 4.5 *
'***************************************************************************

'***************************************************************************
'* SHARED VARIABLES                                                  *******
'***************************************************************************
SHARED MenuInfo()  AS MenuInformation  'Basic menu information on each menu
SHARED MenuItems$()                    'The menu items text
SHARED MenuSelect$()                   'Selector items for each menu item

'***************************************************************************
'* PROCEDURE LOGIC                                                   *******
'***************************************************************************

  IF ItemID% = 0 THEN
    'Add item and it's selection key as the next menu item
    MenuInfo(MenuID%).Items = MenuInfo(MenuID%).Items + 1
    ItemID% = MenuInfo(MenuID%).Items
  END IF

  'Put item and it's selection key into the menu
  MenuSelect$(MenuID%, ItemID%) = UCASE$(SelectKey$) 'Display selector as upper case (remove case sensitivity from selector)
  'Save the menu item - expand items to fill center area if needed
  LenItem% = LEN(MenuItemText$)
  IF LenItem% > MenuInfo(MenuID%).WidItem THEN
    'Resize all the other items in the menu - pad them with spaces
    MenuInfo(MenuID%).WidItem = LenItem%
    MenuItems$(MenuID%, ItemID%) = MenuItemText$
    FOR I% = ItemID% TO 1 STEP -1
      MenuItems$(MenuID%, I%) = MenuItems$(MenuID%, I%) + SPACE$(LenItem% - LEN(MenuItems$(MenuID%, I%)))
    NEXT
  ELSE
    'Pad new item to width of the other items
    MenuItems$(MenuID%, ItemID%) = MenuItemText$ + SPACE$(MenuInfo(MenuID%).WidItem - LenItem%)
  END IF

  'Recalc/position the menu
  XAdjustAddCtr MenuID%, LenItem%

  'If a selector key has been passed then set the selctor flag to ON
  IF SelectKey$ <> "" AND MenuInfo(MenuID%).Selector = Off. THEN
    MenuInfo(MenuID%).Selector = On.
    XAdjustCalcSelSize MenuID%
  END IF

  'Adjust the menu on the Y axis
  YAdjust MenuID%

END SUB

SUB MenuItemSetCurItem (MenuID%, Item%)

'***************************************************************************
'*                                                                         *
'*             Set Which Item in a Menu Will be the Current One            *
'*                            MenuItemSetCurItem                           *
'*                                                                Ver 1.00 *
'*                                                                         *
'* FUNCTION:  This routine sets which item in the menu will be the         *
'*            current item/highlighted item in the menu when Menu          *
'*            is called.                                                   *
'*                                                                         *
'*                                                                         *
'* *********************************************************************** *
'* ARGUMENTS:                                                              *
'* *********************************************************************** *
'*                                                                         *
'*   INPUT:                                                                *
'*     MenuID% : Menu ID                                                   *
'*     Item%   : Item in the menu that will be current (1-num menu items)  *
'*                                                                         *
'* *********************************************************************** *
'* NOTES & COMMENTS:                                                       *
'* *********************************************************************** *
'*                                                                         *
'*   NOTES:                                                                *
'*                                                                         *
'*                                                                         *
'* Bob Cox '89                                              QuickBASIC 4.5 *
'***************************************************************************

'***************************************************************************
'* SHARED VARIABLES                                                  *******
'***************************************************************************
SHARED MenuInfo() AS MenuInformation   'Basic menu information on each menu

'***************************************************************************
'* PROCEDURE LOGIC                                                   *******
'***************************************************************************

  MenuInfo(MenuID%).ItemCur = Item%
  MenuInfo(MenuID%).SetToTop = Off.
 
END SUB

SUB MenuItemSpacing (MenuID%, Spacing%)

'***************************************************************************
'*                                                                         *
'*                 Set Spacing Interval Between Menu Items                 *
'*                             MenuItemSpacing                             *
'*                                                                Ver 1.00 *
'*                                                                         *
'* FUNCTION:  This routine sets the spacing between menu items in the      *
'*            menu.  The default spacing is one, no blank lines between    *
'*            menu items.  If spacing is set to two then one blank line    *
'*            will appear between the items in the menu.                   *
'*                                                                         *
'*                                                                         *
'* *********************************************************************** *
'* ARGUMENTS:                                                              *
'* *********************************************************************** *
'*                                                                         *
'*   INPUT:                                                                *
'*     MenuID%  : Menu ID                                                  *
'*     Spacing% : Spacing interval                                         *
'*                  (1-no blank lines/default, 2-one blank line)           *
'*                                                                         *
'* *********************************************************************** *
'* PROCEDURES:                                                             *
'* *********************************************************************** *
'*                                                                         *
'*   FUNCTIONS & SUBROUTINES CALLED:                                       *
'*                                                                         *
'*   SUBROUTINES:                                                          *
'*     YAdjust : Adjust all the pointers on the Y axis                     *
'*                                                                         *
'* *********************************************************************** *
'* NOTES & COMMENTS:                                                       *
'* *********************************************************************** *
'*                                                                         *
'*   NOTES:                                                                *
'*                                                                         *
'*                                                                         *
'* Bob Cox '89                                              QuickBASIC 4.5 *
'***************************************************************************

'***************************************************************************
'* SHARED VARIABLES                                                  *******
'***************************************************************************
SHARED MenuInfo() AS MenuInformation      'Basic menu information on each menu

'***************************************************************************
'* PROCEDURE LOGIC                                                   *******
'***************************************************************************

  MenuInfo(MenuID%).Spacing = Spacing%
  YAdjust MenuID%

END SUB

SUB MenuModInit

'***************************************************************************
'*                                                                         *
'*                        Initialize the Menu Module                       *
'*                               MenuModInit                               *
'*                                                                Ver 1.00 *
'*                                                                         *
'* FUNCTION:  This is the initialization routine for the module.           *
'*            This routine needs to be called one time.                    *
'*                                                                         *
'*                                                                         *
'* *********************************************************************** *
'* PROCEDURES:                                                             *
'* *********************************************************************** *
'*                                                                         *
'*   FUNCTIONS & SUBROUTINES CALLED:                                       *
'*                                                                         *
'*   SUBROUTINES:                                                          *
'*     ColorsMenuDefault : Sets up the default colors for the module       *
'*                                                                         *
'* *********************************************************************** *
'* NOTES & COMMENTS:                                                       *
'* *********************************************************************** *
'*                                                                         *
'*   NOTES:                                                                *
'*                                                                         *
'*                                                                         *
'* Bob Cox '89                                              QuickBASIC 4.5 *
'***************************************************************************

'***************************************************************************
'* SHARED VARIABLES                                                  *******
'***************************************************************************
SHARED MenuModuleInit% 'Flag indicates if this routine has been called

'***************************************************************************
'* PROCEDURE LOGIC                                                   *******
'***************************************************************************

  'Initialization steps
  ColorsMenuDefault      'Setup default colors so menu will display on screen
  MenuModuleInit% = True 'Menu module has been initialized

END SUB

SUB MenuNormItem (MenuID%)

'***************************************************************************
'*                                                                         *
'*                 Normalize Previous Current Item in Menu                 *
'*                               MenuNormItem                              *
'*                                                                Ver 1.00 *
'*                                                                         *
'* FUNCTION:  This routine normalizes the previous current item in the     *
'*            menu that was highlighted.                                   *
'*                                                                         *
'*                                                                         *
'* *********************************************************************** *
'* ARGUMENTS:                                                              *
'* *********************************************************************** *
'*                                                                         *
'*   INPUT:                                                                *
'*     MenuID% : Menu ID                                                   *
'*                                                                         *
'* *********************************************************************** *
'* NOTES & COMMENTS:                                                       *
'* *********************************************************************** *
'*                                                                         *
'*   NOTES: This routine is the opposite of MenuHighItem and shares some   *
'*          global variables with it.                                      *
'*                                                                         *
'*                                                                         *
'* Bob Cox '89                                              QuickBASIC 4.5 *
'***************************************************************************

'***************************************************************************
'* SHARED VARIABLES                                                  *******
'***************************************************************************
SHARED MenuItems$()                    'Menu items - text
SHARED MenuInfo() AS MenuInformation   'Basic menu information on each menu
SHARED MenuLClr$()                     'Menu left pointer clear
SHARED MenuRClr$()                     'Menu right pointer clear
SHARED ClrFMenuInner%, ClrBMenuInner%  'Menu inner color
SHARED ClrFMenuItem%, ClrBMenuItem%    'Menu Normal Items Colors
SHARED PPos%                           'Previous Position
SHARED PItem%                          'Previous Item

'***************************************************************************
'* PROCEDURE LOGIC                                                   *******
'***************************************************************************

    'Normalize previous item
    COLOR ClrFMenuItem%, ClrBMenuItem%
    LOCATE PPos%, MenuInfo(MenuID%).IStartX
    PRINT MenuItems$(MenuID%, PItem%)

    'Normalize pointers
    COLOR ClrFMenuInner%, ClrBMenuInner%
    LOCATE PPos%, MenuInfo(MenuID%).LPntrX
    PRINT MenuLClr$(MenuID%)  'Erase left pointer
    LOCATE PPos%, MenuInfo(MenuID%).RPntrX
    PRINT MenuRClr$(MenuID%)  'Erase right pointer

END SUB

SUB MenuOpen (MenuID%)

'***************************************************************************
'*                                                                         *
'*                 Returns the Next Open or Available Menu                 *
'*                                 MenuOpen                                *
'*                                                                Ver 1.00 *
'*                                                                         *
'* FUNCTION:  This routine finds an empty/available menu area that will    *
'*            be managed by this module.  The menu position is returned    *
'*            as an integer.  The position or Menu ID number will be       *
'*            used to identify the menu in all subsequent calls to this    *
'*            module.                                                      *
'*                                                                         *
'*                                                                         *
'* *********************************************************************** *
'* ARGUMENTS:                                                              *
'* *********************************************************************** *
'*                                                                         *
'*   OUTPUT:                                                               *
'*     MenuID% : Menu ID                                                   *
'*                                                                         *
'* *********************************************************************** *
'* NOTES & COMMENTS:                                                       *
'* *********************************************************************** *
'*                                                                         *
'*   NOTES:                                                                *
'*           If more menus are needed the constant MaxMenus can be         *
'*           increased in size to accomidate the additions.                *
'*                                                                         *
'*                                                                         *
'* Bob Cox '89                                              QuickBASIC 4.5 *
'***************************************************************************

'***************************************************************************
'* SHARED VARIABLES                                                  *******
'***************************************************************************
SHARED MenuInfo() AS MenuInformation                'Basic menu information on each menu

'***************************************************************************
'* PROCEDURE LOGIC                                                   *******
'***************************************************************************

  'Find/Return next avail or open menu
  MenuID% = 0
  IF NumMenus% < MaxMenus THEN
    'A menu manager slot is available - find it
    FOR M% = 1 TO MaxMenus
      IF MenuInfo(M%).Active = No THEN
        MenuID% = M%
        M% = MaxMenus
      END IF
    NEXT
  END IF

END SUB

SUB MenuSelector (MenuID%, OnOff%)

'***************************************************************************
'*                                                                         *
'*                   Turn the Menu Item Selectors On/Off                   *
'*                               MenuSelector                              *
'*                                                                Ver 1.00 *
'*                                                                         *
'* FUNCTION:  Each item in the menu can have a single key or selector      *
'*            set for it.  This routine displays/uses the selectors ON     *
'*            or removes/ignores the selectors OFF.                        *
'*                                                                         *
'*                                                                         *
'* *********************************************************************** *
'* ARGUMENTS:                                                              *
'* *********************************************************************** *
'*                                                                         *
'*   INPUT:                                                                *
'*     MenuID% : Menu ID                                                   *
'*     OnOff%  : Flag for Option                                           *
'*               1 - Selector(s) ON                                        *
'*               0 - Selector(s) OFF                                       *
'*                                                                         *
'* *********************************************************************** *
'* PROCEDURES:                                                             *
'* *********************************************************************** *
'*                                                                         *
'*   FUNCTIONS & SUBROUTINES CALLED:                                       *
'*                                                                         *
'*   SUBROUTINES:                                                          *
'*     XAdjustBdrSet : Sets the pointers for the menu boarders             *
'*                                                                         *
'* *********************************************************************** *
'* NOTES & COMMENTS:                                                       *
'* *********************************************************************** *
'*                                                                         *
'*   NOTES:                                                                *
'*                                                                         *
'*                                                                         *
'* Bob Cox '89                                              QuickBASIC 4.5 *
'***************************************************************************

'***************************************************************************
'* SHARED VARIABLES                                                  *******
'***************************************************************************
SHARED MenuInfo()  AS MenuInformation  'Basic menu information on each menu

'***************************************************************************
'* PROCEDURE LOGIC                                                   *******
'***************************************************************************

  MenuInfo(MenuID%).Selector = OnOff%
  XAdjustBdrSet MenuID%
 
END SUB

SUB MenuSetLocation (MenuID%, X%, Y%)

'***************************************************************************
'*                                                                         *
'*                   Set the Menu Location On the Screen                   *
'*                             MenuSetLocation                             *
'*                                                                Ver 1.00 *
'*                                                                         *
'* FUNCTION:  This routine positions the menu on the screen.  The default  *
'*            settings (X=0, Y=0) center the menu on each axis, placing    *
'*            the menu in the center of the screen.                        *
'*                                                                         *
'*                                                                         *
'* *********************************************************************** *
'* ARGUMENTS:                                                              *
'* *********************************************************************** *
'*                                                                         *
'*   INPUT:                                                                *
'*     MenuID% : Menu ID                                                   *
'*     X%      : Upper left X corner of menu (0-Center menu on X axis)     *
'*     Y%      : Upper left Y corner of menu (0-Center menu on Y axis)     *
'*                                                                         *
'*                                                                         *
'* *********************************************************************** *
'* PROCEDURES:                                                             *
'* *********************************************************************** *
'*                                                                         *
'*   FUNCTIONS & SUBROUTINES CALLED:                                       *
'*                                                                         *
'*   SUBROUTINES:                                                          *
'*     XAdjustBdrSet : Sets the pointers for the menu boarders             *
'*     YAdjust       : Adjust all the pointers on the Y axis               *
'*                                                                         *
'* *********************************************************************** *
'* NOTES & COMMENTS:                                                       *
'* *********************************************************************** *
'*                                                                         *
'*   NOTES:                                                                *
'*                                                                         *
'*                                                                         *
'* Bob Cox '89                                              QuickBASIC 4.5 *
'***************************************************************************

'***************************************************************************
'* SHARED VARIABLES                                                  *******
'***************************************************************************
SHARED MenuInfo() AS MenuInformation 'Basic menu information on each menu

'***************************************************************************
'* PROCEDURE LOGIC                                                   *******
'***************************************************************************

  MenuInfo(MenuID%).MULX = X%
  MenuInfo(MenuID%).MULY = Y%

  'Reset the pointers on the X and Y axis
  XAdjustBdrSet MenuID%
  YAdjust MenuID%

END SUB

SUB MenuSetToTop (MenuID%, OnOff%)

'***************************************************************************
'*                                                                         *
'*                Turn Auto Reset to Top of Menu Flag On/Off               *
'*                               MenuSetToTop                              *
'*                                                                Ver 1.00 *
'*                                                                         *
'* FUNCTION:  This routine sets the "Reset to Top of Menu" flag ON/OFF.    *
'*            When the flag is ON, the current/highlighted item in the     *
'*            menu will always be the top or first item in the menu.       *
'*            When the flag is OFF, the last item that was current will    *
'*            be current when Menu is called.                              *
'*                                                                         *
'*                                                                         *
'* *********************************************************************** *
'* ARGUMENTS:                                                              *
'* *********************************************************************** *
'*                                                                         *
'*   INPUT:                                                                *
'*     MenuID% : Menu ID                                                   *
'*     OnOff%  : Flag for Option                                           *
'*               1 - Set to top ON                                         *
'*               0 - Set to top OFF                                        *
'*                                                                         *
'* *********************************************************************** *
'* NOTES & COMMENTS:                                                       *
'* *********************************************************************** *
'*                                                                         *
'*   NOTES:                                                                *
'*                                                                         *
'*                                                                         *
'* Bob Cox '89                                              QuickBASIC 4.5 *
'***************************************************************************

'***************************************************************************
'* SHARED VARIABLES                                                  *******
'***************************************************************************
SHARED MenuInfo() AS MenuInformation   'Basic menu information on each menu

'***************************************************************************
'* PROCEDURE LOGIC                                                   *******
'***************************************************************************

  MenuInfo(MenuID%).SetToTop = OnOff%

END SUB

SUB MenuTitleArea (MenuID%, OnOff%)

'***************************************************************************
'*                                                                         *
'*                  Turn the Menu Header/Title Area On/Off                 *
'*                              MenuTitleArea                              *
'*                                                                Ver 1.00 *
'*                                                                         *
'* FUNCTION:  This routine sets the "Title Area" flag ON/OFF.  When the    *
'*            flag is ON, the title area will be displayed.  When the      *
'*            flag is OFF, the title area is not displayed.                *
'*                                                                         *
'*                                                                         *
'* *********************************************************************** *
'* ARGUMENTS:                                                              *
'* *********************************************************************** *
'*                                                                         *
'*   INPUT:                                                                *
'*     MenuID% : Menu ID                                                   *
'*     OnOff%  : Flag for Option                                           *
'*               1 - Title in Boarder ON                                   *
'*               0 - Title in Boarder OFF                                  *
'*                                                                         *
'* *********************************************************************** *
'* PROCEDURES:                                                             *
'* *********************************************************************** *
'*                                                                         *
'*   FUNCTIONS & SUBROUTINES CALLED:                                       *
'*                                                                         *
'*   SUBROUTINES:                                                          *
'*     YAdjust : Adjust all the pointers on the Y axis                     *
'*                                                                         *
'* *********************************************************************** *
'* NOTES & COMMENTS:                                                       *
'* *********************************************************************** *
'*                                                                         *
'*   NOTES:                                                                *
'*                                                                         *
'*                                                                         *
'* Bob Cox '89                                              QuickBASIC 4.5 *
'***************************************************************************

'***************************************************************************
'* SHARED VARIABLES                                                  *******
'***************************************************************************
SHARED MenuInfo() AS MenuInformation           'Basic menu information on each menu

'***************************************************************************
'* PROCEDURE LOGIC                                                   *******
'***************************************************************************

  MenuInfo(MenuID%).Header = OnOff%  'Set the flag
  YAdjust MenuID%                    'Reset the pointers on the Y axis

END SUB

SUB MenuTitleDivd (MenuID%, Y%, Divd$)

'***************************************************************************
'*                                                                         *
'*                 Place a Divider in the Title/Header Area                *
'*                              MenuTitleDivd                              *
'*                                                                Ver 1.00 *
'*                                                                         *
'* FUNCTION:   This "routine" provides a simple calling interface to       *
'*             MenuTitleDivdSet.                                           *
'*                                                                         *
'*                                                                         *
'* *********************************************************************** *
'* ARGUMENTS:                                                              *
'* *********************************************************************** *
'*                                                                         *
'*   INPUT:                                                                *
'*     MenuID% : Menu ID                                                   *
'*     Y%      : Position in Title Area                                    *
'*     Divd$   : Divider (3 character string)                              *
'*                                                                         *
'* *********************************************************************** *
'* PROCEDURES:                                                             *
'* *********************************************************************** *
'*                                                                         *
'*   FUNCTIONS & SUBROUTINES CALLED:                                       *
'*                                                                         *
'*   SUBROUTINES:                                                          *
'*     MenuTitleDivdSet : Sets a divider line for the title area           *
'*                                                                         *
'* *********************************************************************** *
'* NOTES & COMMENTS:                                                       *
'* *********************************************************************** *
'*                                                                         *
'*   NOTES:                                                                *
'*                                                                         *
'*                                                                         *
'* Bob Cox '89                                              QuickBASIC 4.5 *
'***************************************************************************

'***************************************************************************
'* PROCEDURE LOGIC                                                   *******
'***************************************************************************

  MenuTitleDivdSet MenuID%, 0, Y%, Divd$

END SUB

SUB MenuTitleDivdSet (MenuID%, DivID%, Y%, Divd$)

'***************************************************************************
'*                                                                         *
'*                 Place a Divider in the Title/Header Area                *
'*                             MenuTitleDivdSet                            *
'*                                                                Ver 1.00 *
'*                                                                         *
'* FUNCTION:  This routine places a divider line in the title area.        *
'*            The divider can be used to separate the menu items from      *
'*            the title text.                                              *
'*                                                                         *
'*                                                                         *
'* *********************************************************************** *
'* ARGUMENTS:                                                              *
'* *********************************************************************** *
'*                                                                         *
'*   INPUT:                                                                *
'*     MenuID% : Menu ID                                                   *
'*     DivID%  : Divider line array position (0 - Next avail)              *
'*     Y%      : Position in Title Area                                    *
'*     Divd$   : Divider (3 character string)                              *
'*                                                                         *
'* *********************************************************************** *
'* PROCEDURES:                                                             *
'* *********************************************************************** *
'*                                                                         *
'*   FUNCTIONS & SUBROUTINES CALLED:                                       *
'*                                                                         *
'*   SUBROUTINES:                                                          *
'*     YAdjust : Adjust all the pointers on the Y axis                     *
'*                                                                         *
'* *********************************************************************** *
'* NOTES & COMMENTS:                                                       *
'* *********************************************************************** *
'*                                                                         *
'*   NOTES:                                                                *
'*                                                                         *
'*                                                                         *
'* Bob Cox '89                                              QuickBASIC 4.5 *
'***************************************************************************

'***************************************************************************
'* SHARED VARIABLES                                                  *******
'***************************************************************************
SHARED MenuInfo() AS MenuInformation              'Basic info on each menu
SHARED MenuTitleDivdInfo() AS MenuDividerLineInfo 'Info on the menu dividers

'***************************************************************************
'* PROCEDURE LOGIC                                                   *******
'***************************************************************************

  MenuInfo(MenuID%).Header = True 'Turn on the header

  IF DivID% = 0 THEN
    'This is a new divider line - assign to next position in array
    MenuInfo(MenuID%).HeaderDivdL = MenuInfo(MenuID%).HeaderDivdL + 1
    DivID% = MenuInfo(MenuID%).HeaderDivdL
  END IF

  'Save the segments of the bar
  MenuTitleDivdInfo(MenuID%, DivID%).DCL = LEFT$(Divd$, 1)
  MenuTitleDivdInfo(MenuID%, DivID%).DCC = MID$(Divd$, 2, 1)
  MenuTitleDivdInfo(MenuID%, DivID%).DCR = RIGHT$(Divd$, 1)

  IF Y% = 0 THEN
    'Place the divider on the next avail line in the title
    MenuInfo(MenuID%).HeaderLines = MenuInfo(MenuID%).HeaderLines + 1
    MenuTitleDivdInfo(MenuID%, DivID%).Y = MenuInfo(MenuID%).HeaderLines
  ELSE
    IF Y% > MenuInfo(MenuID%).HeaderLines THEN
      MenuInfo(MenuID%).HeaderLines = Y%
    END IF
    MenuTitleDivdInfo(MenuID%, DivID%).Y = Y%
  END IF
  YAdjust MenuID%

END SUB

SUB MenuTitleInBdr (MenuID%, OnOff%)

'***************************************************************************
'*                                                                         *
'*                 Start the Menu Title Area in the Boarder                *
'*                               MenuTitleInBdr                            *
'*                                                                Ver 1.00 *
'*                                                                         *
'* FUNCTION:  This routine sets the "Title In Boarder" flag ON/OFF.        *
'*            When ON the title area extends into the boarder, when        *
'*            OFF the title area is contained by the boarder.              *
'*                                                                         *
'*                                                                         *
'* *********************************************************************** *
'* ARGUMENTS:                                                              *
'* *********************************************************************** *
'*                                                                         *
'*   INPUT:                                                                *
'*     MenuID% : Menu ID                                                   *
'*     OnOff%  : Flag for Option                                           *
'*               1 - Title Area Starts in Boarders ON                      *
'*               0 - Title Area Starts in Boarders OFF                     *
'*                                                                         *
'* *********************************************************************** *
'* PROCEDURES:                                                             *
'* *********************************************************************** *
'*                                                                         *
'*   FUNCTIONS & SUBROUTINES CALLED:                                       *
'*                                                                         *
'*   SUBROUTINES:                                                          *
'*     YAdjust : Adjust all the pointers on the Y axis                     *
'*                                                                         *
'* *********************************************************************** *
'* NOTES & COMMENTS:                                                       *
'* *********************************************************************** *
'*                                                                         *
'*   NOTES:                                                                *
'*                                                                         *
'*                                                                         *
'* Bob Cox '89                                              QuickBASIC 4.5 *
'***************************************************************************

'***************************************************************************
'* SHARED VARIABLES                                                  *******
'***************************************************************************
SHARED MenuInfo()  AS MenuInformation  'Basic menu information on each menu

'***************************************************************************
'* PROCEDURE LOGIC                                                   *******
'***************************************************************************

  MenuInfo(MenuID%).HeaderBdr = OnOff%
  YAdjust MenuID%

END SUB

SUB MenuTitleText (MenuID%, X%, Y%, Text$)

'***************************************************************************
'*                                                                         *
'*                  Place a Line of Text in the Title Area                 *
'*                               MenuTitleText                             *
'*                                                                Ver 1.00 *
'*                                                                         *
'* FUNCTION:   This "routine" provides a simple calling interface to       *
'*             MenuTitleTextSet.                                           *
'*                                                                         *
'*                                                                         *
'* *********************************************************************** *
'* ARGUMENTS:                                                              *
'* *********************************************************************** *
'*                                                                         *
'*   INPUT:                                                                *
'*     MenuID% : Menu ID                                                   *
'*     X%      : Relative X position from left side of boarder             *
'*                 0 - will auto center text between boarders              *
'*     Y%      : Relative Y position from the top of the title area        *
'*                 0 - will become next line in title area                 *
'*     Text$   : Text to be placed in the title area                       *
'*                                                                         *
'* *********************************************************************** *
'* PROCEDURES:                                                             *
'* *********************************************************************** *
'*                                                                         *
'*   FUNCTIONS & SUBROUTINES CALLED:                                       *
'*                                                                         *
'*   SUBROUTINES:                                                          *
'*     MenuTitleTextSet : Sets a text line for the title area              *
'*                                                                         *
'* *********************************************************************** *
'* NOTES & COMMENTS:                                                       *
'* *********************************************************************** *
'*                                                                         *
'*   NOTES:                                                                *
'*                                                                         *
'*                                                                         *
'* Bob Cox '89                                              QuickBASIC 4.5 *
'***************************************************************************

'***************************************************************************
'* PROCEDURE LOGIC                                                   *******
'***************************************************************************

  MenuTitleTextSet MenuID%, 0, X%, Y%, Text$

END SUB

SUB MenuTitleTextSet (MenuID%, TextID%, X%, Y%, Text$)

'***************************************************************************
'*                                                                         *
'*                  Place a Line of Text in the Title Area                 *
'*                             MenuTitleTextSet                            *
'*                                                                Ver 1.00 *
'*                                                                         *
'* FUNCTION:  This routine places text in the title area, multiple         *
'*            lines of text can be defined.                                *
'*                                                                         *
'*                                                                         *
'* *********************************************************************** *
'* ARGUMENTS:                                                              *
'* *********************************************************************** *
'*                                                                         *
'*   INPUT:                                                                *
'*     MenuID% : Menu ID                                                   *
'*     TextID% : Title text array position (0 - Next avail)                *
'*     X%      : Relative X position from left side of boarder             *
'*                 0 - will auto center text between boarders              *
'*     Y%      : Relative Y position from the top of the title area        *
'*                 0 - will become next line in title area                 *
'*     Text$   : Text to be placed in the title area                       *
'*                                                                         *
'* *********************************************************************** *
'* PROCEDURES:                                                             *
'* *********************************************************************** *
'*                                                                         *
'*   FUNCTIONS & SUBROUTINES CALLED:                                       *
'*                                                                         *
'*   SUBROUTINES:                                                          *
'*     XAdjustAddCtr : Sets the widest object pointer                      *
'*     YAdjust       : Adjust all the pointers on the Y axis               *
'*                                                                         *
'* *********************************************************************** *
'* NOTES & COMMENTS:                                                       *
'* *********************************************************************** *
'*                                                                         *
'*   NOTES:                                                                *
'*                                                                         *
'*                                                                         *
'* Bob Cox '89                                              QuickBASIC 4.5 *
'***************************************************************************

'***************************************************************************
'* SHARED VARIABLES                                                  *******
'***************************************************************************
SHARED MenuInfo() AS MenuInformation      'Basic menu information on each menu
SHARED MenuTitle$()                       'Title/Header for menu
SHARED MenuTitleInfo() AS MenuTopBtmText  'Info on the menu title text

'***************************************************************************
'* PROCEDURE LOGIC                                                   *******
'***************************************************************************

  MenuInfo(MenuID%).Header = True  'Turn on the header

  IF TextID% = 0 THEN
    'This is new title text - assign to next position in array
    MenuInfo(MenuID%).HeaderTextL = MenuInfo(MenuID%).HeaderTextL + 1
    TextID% = MenuInfo(MenuID%).HeaderTextL
  END IF

  'Save the parms-may be needed for later recalc
  MenuTitleInfo(MenuID%, TextID%).X = X%
  MenuTitleInfo(MenuID%, TextID%).LenText = LEN(Text$)

  'Save the header text
  MenuTitle$(MenuID%, TextID%) = Text$

  'Calc actual X of header text from realitive positioning parm passed by caller
  'and adjust the menu on the X axis if needed
  XAdjustAddCtr MenuID%, MenuTitleInfo(MenuID%, TextID%).LenText

  IF Y% = 0 THEN
    'Place the text on the next avail line in the title
    MenuInfo(MenuID%).HeaderLines = MenuInfo(MenuID%).HeaderLines + 1
    MenuTitleInfo(MenuID%, TextID%).Y = MenuInfo(MenuID%).HeaderLines
  ELSE
    MenuTitleInfo(MenuID%, TextID%).Y = Y%
    IF Y% > MenuInfo(MenuID%).HeaderLines THEN
      MenuInfo(MenuID%).HeaderLines = Y%
    END IF
  END IF

  'Calc actual Y of comment text from realitive positioning parm passed by caller
  'and adjust the menu on the Y axis if needed
  YAdjust MenuID%

END SUB

SUB SpaceLMenuBdr (MenuID%, NumSpaces%)

'***************************************************************************
'*                                                                         *
'*                Add Spaces on the Left of the Menu Boarder               *
'*                              SpaceLMenuBdr                              *
'*                                                                Ver 1.00 *
'*                                                                         *
'* FUNCTION:  This routine adds/removes extra spaces on the left side      *
'*            of the menu boarder.                                         *
'*                                                                         *
'*                                                                         *
'* *********************************************************************** *
'* ARGUMENTS:                                                              *
'* *********************************************************************** *
'*                                                                         *
'*   INPUT:                                                                *
'*     MenuID%    : Menu ID                                                *
'*     NumSpaces% : Number of Spaces                                       *
'*                                                                         *
'* *********************************************************************** *
'* PROCEDURES:                                                             *
'* *********************************************************************** *
'*                                                                         *
'*   FUNCTIONS & SUBROUTINES CALLED:                                       *
'*                                                                         *
'*   SUBROUTINES:                                                          *
'*     XAdjustBdrSet : Sets the pointers for the menu boarders             *
'*                                                                         *
'* *********************************************************************** *
'* NOTES & COMMENTS:                                                       *
'* *********************************************************************** *
'*                                                                         *
'*   NOTES:                                                                *
'*                                                                         *
'*                                                                         *
'* Bob Cox '89                                              QuickBASIC 4.5 *
'***************************************************************************

'***************************************************************************
'* SHARED VARIABLES                                                  *******
'***************************************************************************
SHARED MenuInfo()  AS MenuInformation  'Basic menu information on each menu

'***************************************************************************
'* PROCEDURE LOGIC                                                   *******
'***************************************************************************

  MenuInfo(MenuID%).SpLBdr = NumSpaces%
  XAdjustBdrSet MenuID%

END SUB

SUB SpaceLMenuItems (MenuID%, NumSpaces%)

'***************************************************************************
'*                                                                         *
'*              Add Spaces on the Left Side of the Menu Items              *
'*                             SpaceLMenuItems                             *
'*                                                                Ver 1.00 *
'*                                                                         *
'* FUNCTION:  This routine adds/removes extra spaces on the left side      *
'*            of the menu items.                                           *
'*                                                                         *
'*                                                                         *
'* *********************************************************************** *
'* ARGUMENTS:                                                              *
'* *********************************************************************** *
'*                                                                         *
'*   INPUT:                                                                *
'*     MenuID%    : Menu ID                                                *
'*     NumSpaces% : Number of Spaces                                       *
'*                                                                         *
'* *********************************************************************** *
'* PROCEDURES:                                                             *
'* *********************************************************************** *
'*                                                                         *
'*   FUNCTIONS & SUBROUTINES CALLED:                                       *
'*                                                                         *
'*   SUBROUTINES:                                                          *
'*     XAdjustBdrSet : Sets the pointers for the menu boarders             *
'*                                                                         *
'* *********************************************************************** *
'* NOTES & COMMENTS:                                                       *
'* *********************************************************************** *
'*                                                                         *
'*   NOTES:                                                                *
'*                                                                         *
'*                                                                         *
'* Bob Cox '89                                              QuickBASIC 4.5 *
'***************************************************************************

'***************************************************************************
'* SHARED VARIABLES                                                  *******
'***************************************************************************
SHARED MenuInfo()  AS MenuInformation  'Basic menu information on each menu

'***************************************************************************
'* PROCEDURE LOGIC                                                   *******
'***************************************************************************

  MenuInfo(MenuID%).SpLItm = NumSpaces%
  XAdjustBdrSet MenuID%

END SUB

SUB SpaceLMenuPtr (MenuID%, NumSpaces%)

'***************************************************************************
'*                                                                         *
'*             Add Spaces on the Left Side of the Menu Pointer             *
'*                               SpaceLMenuPtr                             *
'*                                                                Ver 1.00 *
'*                                                                         *
'* FUNCTION:  This routine adds/removes extra spaces on the left side      *
'*            of the menu pointer.                                         *
'*                                                                         *
'*                                                                         *
'* *********************************************************************** *
'* ARGUMENTS:                                                              *
'* *********************************************************************** *
'*                                                                         *
'*   INPUT:                                                                *
'*     MenuID%    : Menu ID                                                *
'*     NumSpaces% : Number of Spaces                                       *
'*                                                                         *
'* *********************************************************************** *
'* PROCEDURES:                                                             *
'* *********************************************************************** *
'*                                                                         *
'*   FUNCTIONS & SUBROUTINES CALLED:                                       *
'*                                                                         *
'*   SUBROUTINES:                                                          *
'*     XAdjustBdrSet : Sets the pointers for the menu boarders             *
'*                                                                         *
'* *********************************************************************** *
'* NOTES & COMMENTS:                                                       *
'* *********************************************************************** *
'*                                                                         *
'*   NOTES:                                                                *
'*                                                                         *
'*                                                                         *
'* Bob Cox '89                                              QuickBASIC 4.5 *
'***************************************************************************

'***************************************************************************
'* SHARED VARIABLES                                                  *******
'***************************************************************************
SHARED MenuInfo()  AS MenuInformation  'Basic menu information on each menu

'***************************************************************************
'* PROCEDURE LOGIC                                                   *******
'***************************************************************************

  MenuInfo(MenuID%).SpLPtr = NumSpaces%
  XAdjustBdrSet MenuID%

END SUB

SUB SpaceRMenuBdr (MenuID%, NumSpaces%)

'***************************************************************************
'*                                                                         *
'*               Add Spaces on the Right of the Menu Boarder               *
'*                              SpaceRMenuBdr                              *
'*                                                                Ver 1.00 *
'*                                                                         *
'* FUNCTION:  This routine adds/removes extra spaces on the right side     *
'*            of the menu boarder.                                         *
'*                                                                         *
'*                                                                         *
'* *********************************************************************** *
'* ARGUMENTS:                                                              *
'* *********************************************************************** *
'*                                                                         *
'*   INPUT:                                                                *
'*     MenuID%    : Menu ID                                                *
'*     NumSpaces% : Number of Spaces                                       *
'*                                                                         *
'* *********************************************************************** *
'* PROCEDURES:                                                             *
'* *********************************************************************** *
'*                                                                         *
'*   FUNCTIONS & SUBROUTINES CALLED:                                       *
'*                                                                         *
'*   SUBROUTINES:                                                          *
'*     XAdjustBdrSet : Sets the pointers for the menu boarders             *
'*                                                                         *
'* *********************************************************************** *
'* NOTES & COMMENTS:                                                       *
'* *********************************************************************** *
'*                                                                         *
'*   NOTES:                                                                *
'*                                                                         *
'*                                                                         *
'* Bob Cox '89                                              QuickBASIC 4.5 *
'***************************************************************************

'***************************************************************************
'* SHARED VARIABLES                                                  *******
'***************************************************************************
SHARED MenuInfo()  AS MenuInformation  'Basic menu information on each menu

'***************************************************************************
'* PROCEDURE LOGIC                                                   *******
'***************************************************************************

  MenuInfo(MenuID%).SpRBdr = NumSpaces%
  XAdjustBdrSet MenuID%

END SUB

SUB SpaceRMenuItems (MenuID%, NumSpaces%)

'***************************************************************************
'*                                                                         *
'*              Add Spaces on the Right Side of the Menu Items             *
'*                             SpaceRMenuItems                             *
'*                                                                Ver 1.00 *
'*                                                                         *
'* FUNCTION:  This routine adds/removes extra spaces on the right side     *
'*            of the menu items.                                           *
'*                                                                         *
'*                                                                         *
'* *********************************************************************** *
'* ARGUMENTS:                                                              *
'* *********************************************************************** *
'*                                                                         *
'*   INPUT:                                                                *
'*     MenuID%    : Menu ID                                                *
'*     NumSpaces% : Number of Spaces                                       *
'*                                                                         *
'* *********************************************************************** *
'* PROCEDURES:                                                             *
'* *********************************************************************** *
'*                                                                         *
'*   FUNCTIONS & SUBROUTINES CALLED:                                       *
'*                                                                         *
'*   SUBROUTINES:                                                          *
'*     XAdjustBdrSet : Sets the pointers for the menu boarders             *
'*                                                                         *
'* *********************************************************************** *
'* NOTES & COMMENTS:                                                       *
'* *********************************************************************** *
'*                                                                         *
'*   NOTES:                                                                *
'*                                                                         *
'*                                                                         *
'* Bob Cox '89                                              QuickBASIC 4.5 *
'***************************************************************************

'***************************************************************************
'* SHARED VARIABLES                                                  *******
'***************************************************************************
SHARED MenuInfo()  AS MenuInformation  'Basic menu information on each menu

'***************************************************************************
'* PROCEDURE LOGIC                                                   *******
'***************************************************************************

  MenuInfo(MenuID%).SpRItm = NumSpaces%
  XAdjustBdrSet MenuID%

END SUB

SUB XAdjustAddCtr (MenuID%, LenObject%)

'***************************************************************************
'*                                                                         *
'*              Added Item to Center of Menu Adjust on X Axis              *
'*                              XAdjustAddCtr                              *
'*                                                                Ver 1.00 *
'*                                                                         *
'* FUNCTION:  This routine sets the widest object pointer that is used     *
'*            in calculations for other pointers on the X axis.            *
'*                                                                         *
'*                                                                         *
'* *********************************************************************** *
'* ARGUMENTS:                                                              *
'* *********************************************************************** *
'*                                                                         *
'*   INPUT:                                                                *
'*     MenuID%    : Menu ID                                                *
'*     LenObject% : Length/size of object (string) being placed in menu    *
'*                                                                         *
'* *********************************************************************** *
'* PROCEDURES:                                                             *
'* *********************************************************************** *
'*                                                                         *
'*   FUNCTIONS & SUBROUTINES CALLED:                                       *
'*                                                                         *
'*   SUBROUTINES:                                                          *
'*     XAdjustBdrSet : Sets the pointers for the menu boarders             *
'*                                                                         *
'* *********************************************************************** *
'* NOTES & COMMENTS:                                                       *
'* *********************************************************************** *
'*                                                                         *
'*   NOTES:                                                                *
'*                                                                         *
'*                                                                         *
'* Bob Cox '89                                              QuickBASIC 4.5 *
'***************************************************************************

'***************************************************************************
'* SHARED VARIABLES                                                  *******
'***************************************************************************
SHARED MenuInfo()  AS MenuInformation  'Basic menu information on each menu

'***************************************************************************
'* PROCEDURE LOGIC                                                   *******
'***************************************************************************

  'Adjust the menu on the X axis if needed
  IF LenObject% > MenuInfo(MenuID%).WidOb THEN
    'This is the widest object to appear in the menu - menu must be adjusted
    MenuInfo(MenuID%).WidOb = LenObject%
  END IF

  XAdjustBdrSet MenuID% 'Adjust the boarders and pointers

END SUB

SUB XAdjustBdrSet (MenuID%)

'***************************************************************************
'*                                                                         *
'*                           Adjust the Boarders                           *
'*                              XAdjustBdrSet                              *
'*                                                                Ver 1.00 *
'*                                                                         *
'* FUNCTION:  This routine sets the pointers for the menu boarders.        *
'*                                                                         *
'*                                                                         *
'* *********************************************************************** *
'* ARGUMENTS:                                                              *
'* *********************************************************************** *
'*                                                                         *
'*   INPUT:                                                                *
'*     MenuID% : Menu to set the boarders on                               *
'*                                                                         *
'* *********************************************************************** *
'* PROCEDURES:                                                             *
'* *********************************************************************** *
'*                                                                         *
'*   FUNCTIONS & SUBROUTINES CALLED:                                       *
'*                                                                         *
'*   SUBROUTINES:                                                          *
'*     XAdjustPointers : Adjust pointers on the X axis                     *
'*                                                                         *
'* *********************************************************************** *
'* NOTES & COMMENTS:                                                       *
'* *********************************************************************** *
'*                                                                         *
'*   NOTES:                                                                *
'*                                                                         *
'*                                                                         *
'* Bob Cox '89                                              QuickBASIC 4.5 *
'***************************************************************************

'***************************************************************************
'* SHARED VARIABLES                                                  *******
'***************************************************************************
SHARED MenuInfo()  AS MenuInformation  'Basic menu information on each menu

'***************************************************************************
'* PROCEDURE LOGIC                                                   *******
'***************************************************************************

  MenuInfo(MenuID%).MIWid = MenuInfo(MenuID%).WidOb + MenuInfo(MenuID%).SpLBdr + MenuInfo(MenuID%).SpLPtr + MenuInfo(MenuID%).SpLItm + MenuInfo(MenuID%).SpRBdr + MenuInfo(MenuID%).SpRItm + MenuInfo(MenuID%).LPntrL + MenuInfo(MenuID%).RPntrL +  _
MenuInfo(MenuID%).SelSize * MenuInfo(MenuID%).Selector
  MenuInfo(MenuID%).MWid = MenuInfo(MenuID%).MIWid + 2

  IF MenuInfo(MenuID%).MULX = 0 THEN
    'Auto centering is active on the X axis
    'Calc boarder start X position
    MenuInfo(MenuID%).BULX = CenteringPosition%(FullScreenX, MenuInfo(MenuID%).MWid) 'Calc upper left boarder position
  ELSE
    MenuInfo(MenuID%).BULX = MenuInfo(MenuID%).MULX
  END IF
  'Calc boarder right side position
  MenuInfo(MenuID%).BLRX = MenuInfo(MenuID%).BULX + MenuInfo(MenuID%).MWid - 1

  'Set the inner area pointers
  MenuInfo(MenuID%).IULX = MenuInfo(MenuID%).BULX + 1
  MenuInfo(MenuID%).ILRX = MenuInfo(MenuID%).BLRX - 1

  XAdjustPointers MenuID% 'Adjust all the individual pointers (all based on the boarders)

END SUB

SUB XAdjustCalcSelSize (MenuID%)

'***************************************************************************
'*                                                                         *
'*                   Calculate and Set the Selector Size                   *
'*                            XAdjustCalcSelSize                           *
'*                                                                Ver 1.00 *
'*                                                                         *
'* FUNCTION:  This routine sets the selector size parameter for            *
'*            calculations of the X pointers.                              *
'*                                                                         *
'*                                                                         *
'* *********************************************************************** *
'* ARGUMENTS:                                                              *
'* *********************************************************************** *
'*                                                                         *
'*   INPUT:                                                                *
'*     MenuID% : Menu ID                                                   *
'*                                                                         *
'* *********************************************************************** *
'* PROCEDURES:                                                             *
'* *********************************************************************** *
'*                                                                         *
'*   FUNCTIONS & SUBROUTINES CALLED:                                       *
'*                                                                         *
'*   SUBROUTINES:                                                          *
'*     XAdjustBdrSet : Sets the pointers for the menu boarders             *
'*                                                                         *
'* *********************************************************************** *
'* NOTES & COMMENTS:                                                       *
'* *********************************************************************** *
'*                                                                         *
'*   NOTES:                                                                *
'*                                                                         *
'*                                                                         *
'* Bob Cox '89                                              QuickBASIC 4.5 *
'***************************************************************************

'***************************************************************************
'* SHARED VARIABLES                                                  *******
'***************************************************************************
SHARED MenuInfo()  AS MenuInformation  'Basic menu information on each menu
SHARED MenuSelectLH$()                 'Menu selector left holder
SHARED MenuSelectRH$()                 'Menu selector right holder

'***************************************************************************
'* PROCEDURE LOGIC                                                   *******
'***************************************************************************

  MenuInfo(MenuID%).SelSize% = LEN(MenuSelectLH$(MenuID%)) + LEN(MenuSelectRH$(MenuID%)) + 1
  XAdjustBdrSet MenuID%  'Readjust all the X pointers

END SUB

SUB XAdjustPointers (MenuID%)

'***************************************************************************
'*                                                                         *
'*                      Adjust Pointers On the X Axis                      *
'*                             XAdjustPointers                             *
'*                                                                Ver 1.00 *
'*                                                                         *
'* FUNCTION:  This routine sets all the major pointers for the X axis.     *
'*                                                                         *
'*                                                                         *
'* *********************************************************************** *
'* ARGUMENTS:                                                              *
'* *********************************************************************** *
'*                                                                         *
'*   INPUT:                                                                *
'*     MenuID% : Menu ID                                                   *
'*                                                                         *
'* *********************************************************************** *
'* PROCEDURES:                                                             *
'* *********************************************************************** *
'*                                                                         *
'*   FUNCTIONS & SUBROUTINES CALLED:                                       *
'*                                                                         *
'*   FUNCTIONS:                                                            *
'*     CenteringPosition% : Returns horizontal position for text centering *
'*                                                                         *
'* *********************************************************************** *
'* NOTES & COMMENTS:                                                       *
'* *********************************************************************** *
'*                                                                         *
'*   NOTES:                                                                *
'*                                                                         *
'*                                                                         *
'* Bob Cox '89                                              QuickBASIC 4.5 *
'***************************************************************************

'***************************************************************************
'* SHARED VARIABLES                                                  *******
'***************************************************************************
SHARED MenuInfo()  AS MenuInformation      'Basic menu information on each menu
SHARED MenuTitleInfo() AS MenuTopBtmText   'Info on the menu title text
SHARED MenuCommentInfo() AS MenuTopBtmText 'Info on the menu comment text

'***************************************************************************
'* PROCEDURE LOGIC                                                   *******
'***************************************************************************

  'Calc left pointer position
  MenuInfo(MenuID%).LPntrX = MenuInfo(MenuID%).BULX + MenuInfo(MenuID%).SpLBdr + 1

  'Calc selector position
  MenuInfo(MenuID%).SX = MenuInfo(MenuID%).LPntrX + MenuInfo(MenuID%).LPntrL + MenuInfo(MenuID%).SpLPtr

  'Calc Start of items text
  MenuInfo(MenuID%).IStartX = MenuInfo(MenuID%).SX + MenuInfo(MenuID%).SelSize * MenuInfo(MenuID%).Selector + MenuInfo(MenuID%).SpLItm

  'Calc right pointer position
  MenuInfo(MenuID%).RPntrX = MenuInfo(MenuID%).BLRX - MenuInfo(MenuID%).SpRBdr - MenuInfo(MenuID%).RPntrL

  'Set actual positions of text in title and comment areas
  'Text for title area
  IF MenuInfo(MenuID%).HeaderTextL > 0 THEN
    FOR P% = 1 TO MenuInfo(MenuID%).HeaderTextL
      IF MenuTitleInfo(MenuID%, P%).X = 0 THEN
        'Auto center on the X axis
        MenuTitleInfo(MenuID%, P%).AX = CenteringPosition%(MenuInfo(MenuID%).MWid, MenuTitleInfo(MenuID%, P%).LenText) + MenuInfo(MenuID%).BULX - 1
      ELSE
        MenuTitleInfo(MenuID%, P%).AX = MenuInfo(MenuID%).BULX + MenuTitleInfo(MenuID%, P%).X
      END IF
    NEXT
  END IF
  'Text for comment area
  IF MenuInfo(MenuID%).CommentTextL > 0 THEN
    FOR P% = 1 TO MenuInfo(MenuID%).CommentTextL
      IF MenuCommentInfo(MenuID%, P%).X = 0 THEN
        'Auto center on the X axis
        MenuCommentInfo(MenuID%, P%).AX = CenteringPosition%(MenuInfo(MenuID%).MWid, MenuCommentInfo(MenuID%, P%).LenText) + MenuInfo(MenuID%).BULX - 1
      ELSE
        MenuCommentInfo(MenuID%, P%).AX = MenuInfo(MenuID%).BULX + MenuCommentInfo(MenuID%, P%).X
      END IF
    NEXT
  END IF

END SUB

SUB YAdjust (MenuID%)

'***************************************************************************
'*                                                                         *
'*                  Adjust All the Pointers on the Y Axis                  *
'*                                 YAdjust                                 *
'*                                                                Ver 1.00 *
'*                                                                         *
'* FUNCTION:  This routine adjusts the pointers on the Y axis.             *
'*                                                                         *
'*                                                                         *
'* *********************************************************************** *
'* ARGUMENTS:                                                              *
'* *********************************************************************** *
'*                                                                         *
'*   INPUT:                                                                *
'*     MenuID% : Menu ID                                                   *
'*                                                                         *
'* *********************************************************************** *
'* NOTES & COMMENTS:                                                       *
'* *********************************************************************** *
'*                                                                         *
'*   NOTES:                                                                *
'*                                                                         *
'*                                                                         *
'* Bob Cox '89                                              QuickBASIC 4.5 *
'***************************************************************************

'***************************************************************************
'* SHARED VARIABLES                                                  *******
'***************************************************************************
SHARED MenuInfo()  AS MenuInformation  'Basic menu information on each menu
SHARED MenuTitleInfo() AS MenuTopBtmText  'Info on the menu title text
SHARED MenuTitleDivdInfo() AS MenuDividerLineInfo  'Info on the menu dividers

SHARED MenuCommentInfo() AS MenuTopBtmText  'Info on the menu comment text
SHARED MenuCommentDivdInfo() AS MenuDividerLineInfo  'Info on the comment area dividers

'***************************************************************************
'* PROCEDURE LOGIC                                                   *******
'***************************************************************************

  'Add to the length by the spacing interval
  MenuInfo(MenuID%).MLen = (MenuInfo(MenuID%).Items - 1) * MenuInfo(MenuID%).Spacing + (MenuInfo(MenuID%).HeaderLines - MenuInfo(MenuID%).HeaderBdr) * MenuInfo(MenuID%).Header + (MenuInfo(MenuID%).CommentLines - MenuInfo(MenuID%).CommentBdr) *  _
MenuInfo(MenuID%).Comment + 3
  IF MenuInfo(MenuID%).MULY = 0 THEN
    'Auto centering is active on the Y axis
    'Calc upper boarder position
    MenuInfo(MenuID%).BULY = CenteringPosition%(FullScreenY, MenuInfo(MenuID%).MLen) 'Calc upper left boarder position
  ELSE
    MenuInfo(MenuID%).BULY = MenuInfo(MenuID%).MULY
  END IF
  'Calc Start Y for items
  MenuInfo(MenuID%).IStartY = MenuInfo(MenuID%).BULY + (MenuInfo(MenuID%).HeaderLines - MenuInfo(MenuID%).HeaderBdr) * MenuInfo(MenuID%).Header + 1
  'Calc lower boarder position
  MenuInfo(MenuID%).BLRY = MenuInfo(MenuID%).BULY + MenuInfo(MenuID%).MLen - 1

  'Set inner menu pointers
  MenuInfo(MenuID%).IULY = MenuInfo(MenuID%).BULY + 1
  MenuInfo(MenuID%).ILRY = MenuInfo(MenuID%).BLRY - 1

  'Set actual positions of dividers and text in title and comment areas
  'Header Area
  IF MenuInfo(MenuID%).HeaderTextL > 0 THEN
    FOR P% = 1 TO MenuInfo(MenuID%).HeaderTextL
      MenuTitleInfo(MenuID%, P%).AY = MenuInfo(MenuID%).BULY + MenuTitleInfo(MenuID%, P%).Y - MenuInfo(MenuID%).HeaderBdr
    NEXT
  END IF
  IF MenuInfo(MenuID%).HeaderDivdL > 0 THEN
    FOR P% = 1 TO MenuInfo(MenuID%).HeaderDivdL
      MenuTitleDivdInfo(MenuID%, P%).AY = MenuInfo(MenuID%).BULY + MenuTitleDivdInfo(MenuID%, P%).Y - MenuInfo(MenuID%).HeaderBdr
    NEXT
  END IF
  'Comment Area
  IF MenuInfo(MenuID%).CommentTextL > 0 THEN
    StartOfComments% = MenuInfo(MenuID%).BLRY - MenuInfo(MenuID%).CommentLines + MenuInfo(MenuID%).CommentBdr - 1
    FOR P% = 1 TO MenuInfo(MenuID%).CommentTextL
      MenuCommentInfo(MenuID%, P%).AY = StartOfComments% + MenuCommentInfo(MenuID%, P%).Y
    NEXT
  END IF
  IF MenuInfo(MenuID%).CommentDivdL > 0 THEN
    StartOfComments% = MenuInfo(MenuID%).BLRY - MenuInfo(MenuID%).CommentLines + MenuInfo(MenuID%).CommentBdr - 1
    FOR P% = 1 TO MenuInfo(MenuID%).CommentDivdL
      MenuCommentDivdInfo(MenuID%, P%).AY = StartOfComments% + MenuCommentDivdInfo(MenuID%, P%).Y
    NEXT
  END IF

END SUB

