    ' *******************************************************************
    '
    '          GALAXY.BAS - QuickBASIC or QBX/PDS source code
    '
    '          Generates a random galaxy map.  Star systems may
    '          include planets and population.  The mouse is used
    '          to select a system for display.
    '
    '          QB or QBX must be started with the /L option to
    '          use the interrupt driven mouse routines.
    '
    '          Written by Rich Dersheimer in May of 1993.
    '          72123,1521
    '
    ' *******************************************************************

    DEFINT A-Z

    TYPE RegType
        ax      AS INTEGER          ' This stuff sets up the
        bx      AS INTEGER          ' data passed to the Interrupt
        cx      AS INTEGER          ' routine that calls the mouse driver.
        dx      AS INTEGER
        bp      AS INTEGER          ' You must have started QB or QBX
        si      AS INTEGER          ' with the /L option to load the
        di      AS INTEGER          ' QB or QBX quick library.
        flags   AS INTEGER
    END TYPE
    
    TYPE StarSystem
        StarX AS INTEGER
        StarY AS INTEGER
        Starname AS STRING * 20     ' This sets up the different data
        Planet1 AS STRING * 20      ' fields for each system location,
        Planet1Pop AS INTEGER       ' name, number of planets, and each
        Planet2 AS STRING * 20      ' planets population.  You could add
        Planet2Pop AS INTEGER       ' more info, like planet defenses,
        Planet3 AS STRING * 20      ' production value, etc.
        Planet3Pop AS INTEGER
    END TYPE

    ' This is the Interrupt routine in your QBX or QB Quick Library.
    ' In GALAXY, it is used to control the mouse driver.
    DECLARE SUB Interrupt (IntNum%, iReg AS RegType, oReg AS RegType)

    ' The DrawFont Sub draws whatever text you need to display on
    ' the screen.  You pass it the text coordinates (1 to 43 vertical,
    ' 1 to 80 horizontal), the string to display, and the color for the
    ' characters.  It calculates the graphics position for the string,
    ' draws it in the proper color, with a black shadow.
    DECLARE SUB DrawFont (DrawFontY, DrawFontX, Display$, FontColor)

    ' These Subs handle the mouse functions.
    DECLARE SUB MouseInit ()
    DECLARE SUB ShowMouse ()
    DECLARE SUB HideMouse ()
    DECLARE SUB PollMouse ()


    ' A bunch of variables I'm using in various Subs.
    DIM SHARED CurX, CurY
    DIM SHARED Star(1 TO 50) AS StarSystem
    DIM SHARED iReg AS RegType
    DIM SHARED oReg AS RegType
    DIM SHARED Font$(0 TO 40)

    Null$ = STRING$(20, 0)  ' Initial value for StarSystem strings.

    Font$(0) = ""                       ' space
    Font$(1) = "u5er2fd2l4r4d3bl4"      ' A
    Font$(2) = "u6r3fdgl3r3fdgl3"       ' B
    Font$(3) = "buu4er2fbd4gl2hbd"      ' C
    Font$(4) = "u6r2f2d2g2l2"           ' D
    Font$(5) = "u6r4lbd3l3d3r4l4"       ' E
    Font$(6) = "u6r4l4d3r3l3d3"         ' F     These are the upper-case
    Font$(7) = "buu4er2fbg2r2d3l3hbd"   ' G     characters and some symbols.
    Font$(8) = "u6d3r4u3d6bl4"          ' H     You can add to or change
    Font$(9) = "r4l2u6r2l4bd6"          ' I     these to suit your needs.
    Font$(10) = "bu3d2fr2eu5l2bl2bd6"   ' J
    Font$(11) = "u6d3re3g3f3bl4"        ' K
    Font$(12) = "u6d6r4l4"              ' L
    Font$(13) = "u6f2e2d6bl4"           ' M
    Font$(14) = "u6f4u4d6bl4"           ' N
    Font$(15) = "buu4er2fd4gl2hbd"      ' O
    Font$(16) = "u6r3fdgl3d3"           ' P
    Font$(17) = "buu4er2fd4guhf2l3hbd"  ' Q
    Font$(18) = "u6r3fdgl2f3bl4"        ' R
    Font$(19) = "bufr2euhl2huer2fbd5bl4"' S
    Font$(20) = "br2u6r2l4bd6"          ' T
    Font$(21) = "bu6d5fr2eu5bl4bd6"     ' U
    Font$(22) = "bu6d4f2e2u4bl4bd6"     ' V
    Font$(23) = "bu6d5feu2d2feu5bl4bd6" ' W
    Font$(24) = "bu6df2e2ubd6uh2g2d"    ' X
    Font$(25) = "bu6d2f2d2u2e2u2bl4bd6" ' Y
    Font$(26) = "bu6r4dg4dr4l4"         ' Z
    Font$(27) = "brr2lu6gbd5bl"         ' 1
    Font$(28) = "r4l4u3r4u3l4bd6"       ' 2
    Font$(29) = "r4u3l3r3u3l4bd6"       ' 3
    Font$(30) = "br4u6d3l4u3bd6"        ' 4
    Font$(31) = "r4u3l4u3r4l4bd6"       ' 5
    Font$(32) = "u6d3r4d3l4"            ' 6
    Font$(33) = "bu5er3d6bl4"           ' 7
    Font$(34) = "u6r4d3l4r4d3l4"        ' 8
    Font$(35) = "bu3r4l4u3r4d6bl4"      ' 9
    Font$(36) = "u6r4d6l4"              ' 0
    Font$(37) = "br2buubu2ubl2bd5"      ' :
    Font$(38) = "bu3r4l4bd3"            ' -
    Font$(39) = "br2bu6d2bd4bl3"        ' '
    Font$(40) = "brruldbl"              ' .

    SCREEN 9
    WIDTH 80, 43
    RANDOMIZE -TIMER
    COLOR 7, 0
    CLS

    ' Draw the main screen - light gray with a black inset box.
    LINE (0, 0)-(640, 350), 7, BF
    LINE (16, 8)-(624, 240), 0, BF
    LINE (14, 6)-(626, 6), 8
    LINE (15, 7)-(625, 7), 8
    LINE (14, 6)-(14, 241), 8
    LINE (15, 7)-(15, 240), 8
    LINE (626, 6)-(626, 241), 15
    LINE (625, 7)-(625, 240), 15
    LINE (14, 242)-(626, 242), 15
    LINE (15, 241)-(625, 241), 15

    ' Add some sprinkles in the background.
    FOR Sprinkle = 1 TO 250
        PSET (RND(1) * 608 + 16, RND(1) * 232 + 8), 7
    NEXT Sprinkle

    ' Set the number of star systems in the galaxy.
    ' The number should match the number used in the
    ' DIMENSION statement for the Star() StarSystem-type array.
    NumberOfStars = 50

    FOR C = 1 TO NumberOfStars

        DO
            RndX = INT(RND(1) * 76 + 3)     ' Give each system a unique
            RndY = INT(RND(1) * 29 + 2)     ' coordinate location.
            StarDuplicated = -1

            FOR S = 1 TO C
                IF RndX = Star(S).StarX AND RndY = Star(S).StarY THEN
                    StarDuplicated = 0
                END IF
            NEXT S

        LOOP WHILE NOT StarDuplicated

        Star(C).StarX = RndX                ' Give each system a name.
        Star(C).StarY = RndY
        READ Star(C).Starname
        LName$ = RTRIM$(Star(C).Starname)   ' Trim the extra spaces off.

        ' Decide how many habitable planets &
        ' how much population each system has.
        ' The probablity for having planets
        ' is 1 in 10 for a Venus orbit type
        ' planet, 1 in 3 for an Earth orbit
        ' planet, and 1 in 8 for a Mars orbit
        ' habitable planet.  These can be changed
        ' to suit your personal universe preference.

        IF INT(RND(1) * 10) = 1 THEN
            Star(C).Planet1 = LName$ + "-2"
            Star(C).Planet1Pop = INT(RND(1) * 100 + 1)
        END IF

        IF INT(RND(1) * 3) = 1 THEN
            Star(C).Planet2 = LName$ + "-3"
            Star(C).Planet2Pop = INT(RND(1) * 100 + 1)
        END IF

        IF INT(RND(1) * 8) = 1 THEN
            Star(C).Planet3 = LName$ + "-4"
            Star(C).Planet3Pop = INT(RND(1) * 100 + 1)
        END IF

        ' Draw a little sun at each system location.
        ' You could use different PAINT colors to show star
        ' type (Type-G, white dwarf, red giant, whatever), or
        ' ownership, or if the system has been scouted, etc.
        CIRCLE ((RndX - 1) * 8 + 3, (RndY - 1) * 8 + 3), 3, 12
        PAINT ((RndX - 1) * 8 + 3, (RndY - 1) * 8 + 3), 14, 12
    NEXT C

    DrawFont 33, 3, "GALAXY SECTOR MAP 1", 14
    
    MouseInit   ' Set up the mouse cursor.
    ShowMouse

    ' Here's the main loop for program execution.

    DO

    ShowMouse   ' Wait for a left-button click.
    PollMouse
    HideMouse
    
    DRAW "S8"   ' Change to large typestyle.

    LINE (175, 245)-(630, 350), 7, BF    ' Clear the text area.
    
    FOR C = 1 TO NumberOfStars

        ' If the player clicked on a star system,
        ' display the system name, coordinates,
        ' number of planets, and total population.
        IF Star(C).StarX = CurX AND Star(C).StarY = CurY THEN
            DrawFont 34, 32, "SYSTEM:", 14
            DrawFont 34, 46, (RTRIM$(Star(C).Starname)), 15
            DrawFont 37, 32, "SECTOR:", 14
            COORD$ = LTRIM$(STR$(CurX - 2)) + "-" + LTRIM$(STR$(CurY - 1))
            DrawFont 37, 46, COORD$, 15
            DrawFont 40, 30, "PLANETS:", 14
            Planets = 0
            IF Star(C).Planet1 <> Null$ THEN Planets = Planets + 1
            IF Star(C).Planet2 <> Null$ THEN Planets = Planets + 1
            IF Star(C).Planet3 <> Null$ THEN Planets = Planets + 1
            TotalPop = Star(C).Planet1Pop + Star(C).Planet2Pop
            TotalPop = TotalPop + Star(C).Planet3Pop
            
            IF Planets THEN
                DrawFont 40, 46, LTRIM$(STR$(Planets)), 15
                DrawFont 43, 24, "Population:", 14
                DrawFont 43, 44, STR$(TotalPop), 15
            ELSE
                DrawFont 40, 46, "NONE", 15
            END IF

            EXIT FOR    ' If you've displayed a system, don't look for more.
        END IF

    NEXT C
    
    LOOP

        ' Here are the system names.  You can change them,
        ' or even have the player name the systems.
        DATA Lee's Gate, Procyon, Antares, Agena, Adara, Sigh-Mee
        DATA Nihal, Vega, Saiph, Algol, Schedar, Rastaban
        DATA Izar, Sirius, Nunki, Etamin, Alioth, Wesen
        DATA Mirzam, Ascella, Spica, Gemma, Wolf, Arneb
        DATA Mintaka, Aron Minor, Canopus, Altair, Mizar, Polaris
        DATA Rich-D-37, Capella, Zosma, Almak, Alhena, Chan
        DATA Thuban, Nath, Deneb, Ross, Regulus, Hamal, Castor
        DATA Centauri, Rigel, Acrux, Pollux, Phaeda, Sabik, Megrez

'
'
'
' SUBROUTINE: DrawFont **********************************************
SUB DrawFont (DrawFontY, DrawFontX, Display$, FontColor) STATIC

    ' Test$ is searched to find a valid, displayable character.
    Test$ = " ABCDEFGHIJKLMNOPQRSTUVWXYZ1234567890:-'."

    ' Find the graphic location for the first character.
    DFX = (DrawFontX - 1) * 8 + 1
    DFY = (DrawFontY - 1) * 8 + 1

    ' Move the graphic cursor to the first character location.
    DRAW "BM" + STR$(DFX) + "," + STR$(DFY)

    ' Make sure there are no lower case letters.
    Display$ = UCASE$(Display$)

    ' Draw the shadow, then the main color for each character.
    FOR x = 1 TO LEN(Display$)
        a = INSTR(Test$, MID$(Display$, x, 1)) - 1
        IF a < 0 THEN a = 0
        DRAW "C0"                       ' Use 0 for the shadow color.
        DRAW Font$(a)                   ' Draw the shadow.
        DRAW "bhc" + STR$(FontColor)    ' Shift up & left and use the
        DRAW Font$(a)                   ' main color to draw the character.
        DRAW "br8bf"                    ' Move to next character place.
    NEXT x

END SUB

'
'
'
' SUBROUTINE: HideMouse *********************************************
SUB HideMouse

    ' Hide the mouse cursor.
    iReg.ax = 2
    Interrupt &H33, iReg, oReg

END SUB

'
'
'
' SUBROUTINE: MouseInit *********************************************
SUB MouseInit

    ' Reset the mouse driver.
    iReg.ax = 0
    Interrupt &H33, iReg, oReg

    ' Tell the mouse driver what video mode you are using.
    iReg.ax = 40
    iReg.cx = 9
    iReg.dx = 0
    Interrupt &H33, iReg, oReg

END SUB

'
'
'
' SUBROUTINE: PollMouse *********************************************
SUB PollMouse

    ' Poll the mouse for a left-button press
    oReg.bx = 0
    
    DO

        iReg.ax = 5
        iReg.bx = 0
        Interrupt &H33, iReg, oReg

        ' This returns the text screen y and x for the mouse cursor.
        CurX = INT(oReg.cx / 8) + 1
        CurY = INT(oReg.dx / 8) + 1

        ' Press the ESC key to end the program.
        IF INKEY$ = CHR$(27) THEN
            HideMouse                   ' Make sure you hide the mouse
            SCREEN 0                    ' cursor & reset the screen width!
            WIDTH 80, 25
            COLOR 7, 0
            CLS
            END
        END IF

    LOOP WHILE oReg.bx = 0

END SUB

'
'
'
' SUBROUTINE: ShowMouse *********************************************
SUB ShowMouse

    ' Show the mouse cursor.
    iReg.ax = 1
    Interrupt &H33, iReg, oReg

END SUB

