PROC ZView(InfoDyn)

; ZView - Generic table view control procedure.

; Copyright 1992 by WHITE HEAT, Inc.  All Rights Reserved.

; GENERAL NOTES: This proc allows simple access to single or multitable
;                (two tables only) Paradox objects, either in table or
;                form views.  A secondary form (based on the main table)
;                can also be accessed.  Pass the required parameters to
;                the proc when called, and it will deal with them accord-
;                ingly.  It traps for most errors, but if you call it with
;                a "Link" table, the secondary form mentioned, you must 
;                supply the right parameters for success, for example.
;                Since "InfoDyn" is a dynamic array, you need only pass
;                the minimum assigned variables in the bag.  See comments
;                as follows for more information regarding the bag elements
;                and their functions.  This subsystem includes full network
;                locking and control access via Paradox locking facilities.
;                Access to key fields can also be restricted.  All table
;                locking can be bypassed by setting InfoDyn["DoLock"] = 
;                FALSE.  All record locking will by bypassed if 
;                InfoDyn["RecordLock"] = FALSE.
;
; NOTE #1: InfoDyn["Access"] should be:  0 = No Add, Edit, or Delete access.
;                                        1 = Edit only.
;                                        2 = Add and Edit only.
;                                        3 = Add, Edit, and Delete access.
;
; NOTE #2: If a form is utilized, "Add" & "Modify" procs fill, by default: 
;
;                                        1 = [Number] field (N type assumed).
;                                        2 = [Date] field (D type assumed).
;                                        3 = [Time] field (A8 type assumed).
;
; NOTE #3: The master table will be locked by virtue of its being included
;          in the InfoDyn["TableName"] variable.  Any "linked" detail table,
;          i.e., any second form view's detail table to be accessed, will 
;          also be locked due to its name being assigned to the variable
;          InfoDyn["LinkTable"].  Any other tables need to be specifically
;          included in the InfoDyn["LockString"] string, including the lock-
;          ing construct "PFL" in the string; create the string as if there
;          were no other tables with which to be concerned, such as: 
;          "TableA PFL, TableB PFL" for example.
;
; NOTE #4: "ZModify" and "ZAdd," called from this procedure, assign ERRORPROC
;          and release the variable when they exit; you must reassign to
;          restore any previously-existing setting.

PRIVATE
   Result,
   TableVar,
   CurrAccess,
   LinkVar,
   Linkage,
   LinkEdit,
   Linked,
   Lockstring,
   NoForm,
   TableHandle,
   ZoomFlag

; Check three global variables first.

IF (NOT ISASSIGNED(ErrorFlag)) THEN
   ErrorFlag = FALSE ; In case not assigned elsewhere.
ENDIF

IF (NOT ISASSIGNED(G.LockTries)) THEN
   G.LockTries = 20 ; Until "Unable to lock" message given.
ENDIF

IF (NOT ISASSIGNED(G.LockSleepTime)) THEN
   G.LockSleepTime = 100 ; Equals 1/10th second.
ENDIF

; Process dynarray variables.

IF (NOT ISASSIGNED(InfoDyn["TableName"])) THEN ; Main table name required.
   BEEP
   RETURN FALSE
ENDIF

IF (NOT ISASSIGNED(InfoDyn["MultiLock"])) THEN ; Multiform table setting - 
   InfoDyn["MultiLock"] = FALSE                ;   use for a linked detail
ENDIF                                          ;   table on the master.

Linkage = TRUE

IF (NOT ISASSIGNED(InfoDyn["LinkForm"])) THEN ; Will there be a "linked" 
   InfoDyn["LinkForm"] = ""                   ;    or detail table involved
   Linkage = FALSE                            ;    on an additional form
ENDIF                                         ;    that they can swith to?

IF (NOT ISASSIGNED(InfoDyn["LinkTable"])) THEN ; Name of linked detail table
   InfoDyn["LinkTable"] = ""                   ;    that they can choose 
ENDIF                                          ;    with the "LINK" choice.

IF (NOT ISASSIGNED(InfoDyn["DoLock"])) THEN ; Bother with table locks?
   InfoDyn["DoLock"] = TRUE
ENDIF

IF (NOT ISASSIGNED(InfoDyn["RecordLock"])) THEN ; Bother with record locks?
   InfoDyn["RecordLock"] = TRUE
ENDIF
   
IF (NOT ISASSIGNED(InfoDyn["FieldName"])) THEN ; Name of indexed view field.
   InfoDyn["FieldName"] = ""
ENDIF

IF (NOT ISASSIGNED(InfoDyn["IndexName"])) THEN ; Name of index if different
   InfoDyn["IndexName"] = ""                   ;    from field name.
ENDIF

IF (NOT ISASSIGNED(InfoDyn["Last"])) THEN ; Initially move to end of table?
   InfoDyn["Last"] = FALSE
ENDIF

IF (NOT ISASSIGNED(InfoDyn["FormName"])) THEN ; Name of initial form view.
   InfoDyn["FormName"] = ""
ENDIF

IF (NOT ISASSIGNED(InfoDyn["HideTable"])) THEN ; Hide table view window?
   InfoDyn["HideTable"] = FALSE
ENDIF

IF InfoDyn["FormName"] = "" THEN
   InfoDyn["HideTable"] = FALSE ; If no form, can't hide table view window.
ENDIF

IF (NOT ISASSIGNED(InfoDyn["EditForm"])) THEN ; Name of editing form.
   InfoDyn["EditForm"] = ""
ENDIF

IF (NOT ISASSIGNED(InfoDyn["Access"])) THEN ; Access rights - default is
   InfoDyn["Access"] = 3                    ;    full access.
ENDIF

CurrAccess = InfoDyn["Access"] ; Link table rights may <> main table rights.

IF Linkage THEN
   IF (NOT ISASSIGNED(InfoDyn["LinkAccess"])) THEN
      InfoDyn["LinkAccess"] = InfoDyn["Access"] ; If no setting, default to
   ENDIF                                        ;    same as main table.

   IF (NOT ISASSIGNED(InfoDyn["LinkEditForm"])) THEN ; Assign edit form for
      InfoDyn["LinkEditForm"] = InfoDyn["LinkForm"]  ;    "Link" table.
   ENDIF
ENDIF

IF (NOT ISASSIGNED(InfoDyn["KeyAccess"])) THEN ; Use IMAGERIGHTS UPDATE?
   InfoDyn["KeyAccess"] = TRUE ; If not specifeid, default to full access.
ENDIF

IF (NOT ISASSIGNED(InfoDyn["SQLProc"])) THEN ; Name of any SQL pre-proc.
   InfoDyn["SQLProc"] = ""
ENDIF

IF (NOT ISASSIGNED(InfoDyn["ArrProc"])) THEN ; Name of any arrival proc.
   InfoDyn["ArrProc"] = ""
ENDIF

IF (NOT ISASSIGNED(InfoDyn["DepProc"])) THEN ; Name of any departure proc.
   InfoDyn["DepProc"] = ""
ENDIF

IF (NOT ISASSIGNED(InfoDyn["ZModArr"])) THEN ; Name of any ZModify arrival
   InfoDyn["ZModArr"] = ""                   ;    proc.
ENDIF

IF (NOT ISASSIGNED(InfoDyn["ZModDep"])) THEN ; Name of any ZModify departure
   InfoDyn["ZModDep"] = ""                   ;    proc.
ENDIF

IF (NOT ISASSIGNED(InfoDyn["ZDelArr"])) THEN ; Name of any ZDelete arrival
   InfoDyn["ZDelArr"] = ""                   ;    proc.
ENDIF

IF (NOT ISASSIGNED(InfoDyn["ZDelDep"])) THEN ; Name of any ZDelete departure
   InfoDyn["ZDelDep"] = ""                   ;    proc.
ENDIF

IF (NOT ISASSIGNED(InfoDyn["ZAddArr"])) THEN ; Name of any ZAdd arrival proc.
   InfoDyn["ZAddArr"] = ""
ENDIF

IF (NOT ISASSIGNED(InfoDyn["ZAddDep"])) THEN ; Name of any ZAdd departure
   InfoDyn["ZAddDep"] = ""                   ;    proc.
ENDIF

IF (NOT ISASSIGNED(InfoDyn["PassMsg"])) THEN ; Initial screenmessage, if any.
   InfoDyn["PassMsg"] = ""
ENDIF   

; Handle necessary locking.

IF InfoDyn["DoLock"] THEN
   IF (NOT ISASSIGNED(InfoDyn["LockString"])) THEN ; InfoDyn["LockString"] is
      InfoDyn["LockString"] = ""                   ;    name of detail table,
   ENDIF                                           ;    if any - not master.

   TableVar = InfoDyn["TableName"] ; Set to main table.
   Lockstring = "TableVar PFL"     ; Initial lock string uses main table.

   IF (InfoDyn["LinkTable"] <> "") THEN
      IF (UPPER(InfoDyn["LinkTable"]) <> UPPER(InfoDyn["TableName"])) THEN
         LinkVar = InfoDyn["LinkTable"]            ; If "link" involved, add
         LockString = LockString + ", LinkVar PFL" ;    to lock string if not
      ENDIF                                        ;    same as master.
   ENDIF

   IF (InfoDyn["LockString"] <> "") THEN ; Add extra tables string, if any.
      Lockstring = InfoDyn["LockString"] + ", " + LockString
   ENDIF   

   Result = ZTLock(LockString,0,0) ; Attempt locking.

   IF Result = "No" THEN
      RETURN FALSE
   ENDIF
ENDIF   

Linked = FALSE
ZoomFlag = FALSE
ScreenMessage = ""

IF (InfoDyn["PassMsg"] <> "") THEN ; If initial message passed, prepare it.
   ScreenMessage = InfoDyn["PassMsg"]
ENDIF

CLEAR

IF (InfoDyn["SQLProc"] <> "") THEN ; If SQL Pre-proc called for, execute it.

   EXECPROC InfoDyn["SQLProc"]
   
ENDIF

; Set up the table for working.

IF (InfoDyn["SQLProc"] = "") THEN ; SQL pre-proc should prepare/view table.
   VIEW InfoDyn["TableName"]      ; Otherwise, view it now.
ENDIF
   
TableVar = TABLE() ; "TableVar" variable now utilized differently.

; Deal with indexing as appropriate.

IF (InfoDyn["FieldName"] <> "") THEN
   IF InfoDyn["IndexName"] = "" THEN ; If "", set to same as field name.
      InfoDyn["IndexName"] = InfoDyn["FieldName"]
   ENDIf

   MOVETO FIELD InfoDyn["FieldName"] 
   ORDERTABLE
   SELECT InfoDyn["IndexName"]
   
   IF UPPER(MENUCHOICE()) = "CANCEL" THEN ; If no such index, avoid break.
      MENU
      ESC
   ENDIF
ENDIF

IF InfoDyn["Last"] THEN ; Move to head or tail of table as appropriate.
   END
ELSE
   HOME
ENDIF

IF InfoDyn["HideTable"] THEN ; Hide the table view window if called for.
   WINDOW HANDLE CURRENT TO TableHandle
   WINDOW MOVE TableHandle TO -100, -100
ENDIF

; Deal with form or table view and associated forms settings.

NoForm = TRUE

IF InfoDyn["FormName"] <> "" THEN
   NoForm = FALSE

   IF (NOT ISEMPTY(TABLE())) THEN
      PICKFORM InfoDyn["FormName"]
      
      IF (NOT ISMULTIFORM(TABLE(),FORM())) THEN
         InfoDyn["MultiLock"] = FALSE ; Reassign if necessary.
      ENDIF
   ENDIF
ENDIF

IF InfoDyn["EditForm"] = "" THEN ; If "", defaults to view form (if any).
   InfoDyn["EditForm"] = InfoDyn["FormName"] 
ENDIF

IF (InfoDyn["ArrProc"] <> "") THEN ; Execute arrival procedure, if any.

   EXECPROC InfoDyn["ArrProc"]
   
ENDIF

; Establish an overall loop for control of user action.

WHILE (TRUE)

IF ScreenMessage <> "" THEN
   BEEP SLEEP 25 BEEP
ENDIF

MESSAGE ScreenMessage

; Set up appropriate overall pulldown menu, using appropriate menu options.

IF (NOT InfoDyn["MultiLock"]) AND (NOT Linkage) THEN
   SHOWPULLDOWN
      "Search" : "Search for a record based on current field."  : "Search",
      "Next"   : "Search for next occurence of previous value." : "Next",
      "Toggle" : "Toggle from Form to Table view."              : "Toggle",
      "Edit"   : "Edit this record."                            : "Edit",
      "Delete" : "Delete this record."                          : "Delete",   
      "Add"    : "Add a new record."                            : "Add",
      "Quit"   : "Quit and return to previous menu."            : "Quit"
   ENDMENU
ENDIF

IF (NOT InfoDyn["MultiLock"]) AND Linkage AND (NOT Linked) THEN
   SHOWPULLDOWN
      "Search" : "Search for a record based on current field."  : "Search",
      "Next"   : "Search for next occurence of previous value." : "Next",
      "Toggle" : "Toggle from Form to Table view."              : "Toggle",
      "Link"   : "Access linked/detail view."                   : "Link",
      "Edit"   : "Edit this record."                            : "Edit",
      "Delete" : "Delete this record."                          : "Delete",   
      "Add"    : "Add a new record."                            : "Add",
      "Quit"   : "Quit and return to previous menu."            : "Quit"
   ENDMENU
ENDIF

IF (NOT InfoDyn["MultiLock"]) AND Linkage AND Linked THEN
   SHOWPULLDOWN
      "Search" : "Search for a record based on current field."  : "Search",
      "Next"   : "Search for next occurence of previous value." : "Next",
      "Toggle" : "Toggle from Form to Table view."              : "Toggle",
      "Master" : "Return to Master view."                       : "Master",
      "Edit"   : "Edit this record."                            : "Edit",
      "Delete" : "Delete this record."                          : "Delete",   
      "Add"    : "Add a new record."                            : "Add",
      "Quit"   : "Quit and return to previous menu."            : "Quit"
   ENDMENU
ENDIF

IF InfoDyn["MultiLock"] AND (NOT Linkage) THEN
   SHOWPULLDOWN
      "Search" : "Search for a record based on current field."  : "Search",
      "Next"   : "Search for next occurence of previous value." : "Next",
      "Toggle" : "Toggle from Form to Table view."              : "Toggle",
      "Upper"  : "Move to upper master record."                 : "Upper",
      "Lower"  : "Move to lower linked records."                : "Lower",
      "Edit"   : "Edit this record."                            : "Edit",
      "Delete" : "Delete this record."                          : "Delete",   
      "Add"    : "Add a new record."                            : "Add",
      "Quit"   : "Quit and return to previous menu."            : "Quit"
   ENDMENU
ENDIF   
   
IF InfoDyn["MultiLock"] AND Linkage AND (NOT Linked) THEN
   SHOWPULLDOWN
      "Search" : "Search for a record based on current field."  : "Search",
      "Next"   : "Search for next occurence of previous value." : "Next",
      "Toggle" : "Toggle from Form to Table view."              : "Toggle",
      "Link"   : "Access linked/detail view."                   : "Link",
      "Upper"  : "Move to upper master record."                 : "Upper",
      "Lower"  : "Move to lower linked records."                : "Lower",
      "Edit"   : "Edit this record."                            : "Edit",
      "Delete" : "Delete this record."                          : "Delete",   
      "Add"    : "Add a new record."                            : "Add",
      "Quit"   : "Quit and return to previous menu."            : "Quit"
   ENDMENU
ENDIF

IF InfoDyn["MultiLock"] AND Linkage AND Linked THEN
   SHOWPULLDOWN
      "Search" : "Search for a record based on current field."  : "Search",
      "Next"   : "Search for next occurence of previous value." : "Next",
      "Toggle" : "Toggle from Form to Table view."              : "Toggle",
      "Master" : "Return to Master view."                       : "Master",
      "Upper"  : "Move to upper master record."                 : "Upper",
      "Lower"  : "Move to lower linked records."                : "Lower",
      "Edit"   : "Edit this record."                            : "Edit",
      "Delete" : "Delete this record."                          : "Delete",   
      "Add"    : "Add a new record."                            : "Add",
      "Quit"   : "Quit and return to previous menu."            : "Quit"
   ENDMENU
ENDIF

; Apply menu restrictions as appropriate.

IF NoForm OR InfoDyn["MultiLock"] OR InfoDyn["HideTable"] THEN
   MENUDISABLE "Toggle" ; "Toggle" disabled for linked detail views.
ENDIF

IF CurrAccess < 3 THEN
   MENUDISABLE "Delete"
ENDIF

IF CurrAccess < 2 THEN
   MENUDISABLE "Edit"
ENDIF

IF CurrAccess < 1 THEN
   MENUDISABLE "Add"
ENDIF

; Give user information on how to access the menu. 

PROMPT " Press [F10] or use Mouse to access Menu."
ECHO NORMAL

; Start the WAIT loop.

WAIT TABLE PROC "ZWait"
   KEY "F2","F3","F4","F7","F9","F10","ESC","DEL","INS",15,-18,-24,26,-31,
       -37,-38,-44,-45,-98,-100
   MESSAGE "CLOSE","MENUSELECT"
ENDWAIT

IF retval = 2005 THEN ; Default when other than 0, 1, or 2 returned.
   QUITLOOP
ENDIF

ENDWHILE ; Overall loop.

; Clean house before returning.

CLEAR
ECHO OFF
PROMPT
CLEARPULLDOWN

IF (InfoDyn["DepProc"] <> "") THEN ; Execute departure procedure, if any.

   EXECPROC InfoDyn["DepProc"]

ENDIF

CLEARIMAGE

IF InfoDyn["DoLock"] THEN
   EXECUTE "UNLOCK " + LockString
ENDIF
   
RETURN TRUE

ENDPROC ; ZView(InfoDyn)

WRITELIB G.AppLib ZView
RELEASE PROCS ZView