; ===========================================================================
;      AUTHOR: Copyright (c) 1992, 1993 - Daniel J. Paolini II
; DESCRIPTION: Routines to establish and control a SpeedButton bar in a
;              Window on the desktop.  The SpeedButtons provide shortcut
;              icons for maneuvering in a workspace image.  Routines include:
;                 utSpeedButtonsEnable.u   - called to place Buttons on screen
;                 utSpeedButtonsSetup.u    - called to create Button Window
;                 utSpeedButtonsPressed.u  - called when Button is clicked
;                 utSpeedButtonsDispatch.n - handles selected Button action
;                 utSpeedButtonsHelp.u     - description of Button icons
;                 utSpeedButtonsHelpDB.u   - called to set scroll rate
; ===========================================================================
;       TITLE: utSpeedButtonsEnable.u
;     RETURNS: No value
; DESCRIPTION: Places SpeedButtons at desired location; creates Window if it
;              does not exist.
; ---------------------------------------------------------------------------
PROC utSpeedButtonsEnable.u(     ; Restores or establishes SpeedButtons
         row.n,                  ; Row to establish SpeedButtons window
         column.n,               ; Column to establish SpeedButtons window
         colors.v)               ; DynArray of custom colors, or ""
;Global  g.handles.y             ; Stores application window handles
   IF NOT IsAssigned(g.handles.y) THEN
      DynArray g.handles.y[]
   ENDIF

   IF NOT IsAssigned(g.handles.y["SpeedButtons"]) OR
      NOT IsWindow(g.handles.y["SpeedButtons"]) THEN
      utSpeedButtonsSetup.u(colors.v)        ; Establish a new window
   ENDIF

   Window MOVE g.handles.y["SpeedButtons"]
          To row.n, column.n                 ; Bring it to desired location
   Return
ENDPROC
; ===========================================================================
;       TITLE: utSpeedButtonsSetup.u
;     RETURNS: No Value
; DESCRIPTION: Sets up mouse SpeedButtons
; ---------------------------------------------------------------------------
PROC utSpeedButtonsSetup.u(      ; Generic Mouse SpeedButtons Setup
         colors.v)               ; DynArray of Colors, or ""
Private  current.w,              ; Current Window Handle
         canvas.w,               ; Current Canvas Window Handle
         speedbuttons.y,         ; SpeedButtons window dynarray
         iconcolor.n,            ; Color of SpeedButton icons
         barcolor.n,             ; Color of SpeedButton divider bars
         n,                      ; Loop incrementer
         y                       ; Transient window attributes dynarray
;Global  g.handles.y             ; Global window handle dynarray

   IF NOT IsAssigned(g.handles.y) THEN
      DynArray g.handles.y[]     ; Create window-tracking dynarray
   ENDIF

   iconcolor.n = IIF(IsBlank(colors.v),SysColor(1003),colors.v["1003"])
   barcolor.n  = IIF(IsBlank(colors.v),SysColor(1001),colors.v["1001"])
   Window HANDLE CURRENT To current.w        ; Save current window handle
   canvas.w = GetCanvas()                    ; Save current window handle

   DynArray speedbuttons.y[]                 ; Create a dynamic array for specs
      speedbuttons.y["CanClose"] = False
      speedbuttons.y["CanMaximize"] = False
      speedbuttons.y["CanMove"] = False
      speedbuttons.y["CanResize"] = False
      speedbuttons.y["Echo"] = False
      speedbuttons.y["HasShadow"] = False
      speedbuttons.y["HasFrame"] = False     ; IF Framed, window is *5* rows!!!
      speedbuttons.y["Style"] = iconcolor.n
   Window CREATE  FLOATING @ -200,-200
                  HEIGHT 1 WIDTH 37
                  ATTRIBUTES speedbuttons.y To g.handles.y["SpeedButtons"]

   SetCanvas g.handles.y["SpeedButtons"]     ; Set Canvas to SpeedButtons Window
   @ 0,0 ?? " \30  \174  \27 Pg\24 ? Pg\25 \26  \175  \31 "
   FOR n From 0 To 9                         ; Color divider bars
      PaintCanvas ATTRIBUTE barcolor.n  0,0+(n*4),0,0+(n*4)
   ENDFOR

   IF IsWindow(canvas.w) THEN                ; Restore focus
      SetCanvas canvas.w
   ELSE
      SetCanvas Default
   ENDIF

   IF IsWindow(current.w) THEN
      Window SELECT current.w                ; Restore original Window
   ENDIF
   Return
ENDPROC
; ===========================================================================
;       TITLE: utSpeedButtonsPressed.u
;     RETURNS: No value
; DESCRIPTION: Determines which button was selected, colors it to appear
;              depressed, and calls the SpeedBar dispatch procedure
; ---------------------------------------------------------------------------
PROC utSpeedButtonsPressed.u(    ; Handles Mouse Events on Buttons window
         event.y,                ; Wait Proc Event DynArray
         pushcolor.n)            ; Color for "depressed" button (11 is good)
Private  canvas.w,               ; Current canvas
         current.w,              ; Current window
         button.n,               ; Which button was "pressed"
         y                       ; DynArray of Window attributes
   IF NImages() = 0 OR IsEmpty(Table()) THEN
      msWorking.u("Table is Empty",79,3,2)
   ELSE
      ; You may need code here to block activity if editing/adding a record,
      ;  if you do not control how this proc is called from within your wait
      ;  handler.
      canvas.w = GetCanvas()                 ; Current canvas focus
      LocalizeEvent event.y                  ; Set Row/Column position
      SetCanvas g.handles.y["SpeedButtons"]  ;  relative to current window
                                             ; Determines current Style attrib
      Window GetAttributes g.handles.y["SpeedButtons"] To y

      IF Mod(event.y["Col"],4) <> 0 THEN     ; 0 = Clicked on a divider bar
         button.n = Int(event.y["Col"]/4)+1  ; Buttons are evenly spaced
         PaintCanvas Attribute pushcolor.n 0,(button.n*4)-3,0,(button.n*4)-1
         utSpeedButtonsDispatch.u(button.n)
         Sleep 300                           ; Pause for "depressed" effect
         PaintCanvas Attribute y["Style"]  0,(button.n*4)-3,0,(button.n*4)-1
      ELSE
         Beep
      ENDIF

      IF IsWindow(canvas.w) THEN             ; Restore focus
         SetCanvas canvas.w
      ELSE
         SetCanvas Default
      ENDIF
   ENDIF
   Return
ENDPROC
; ===========================================================================
;       TITLE: utSpeedButtonsDispatch.u
;     RETURNS: No value
; DESCRIPTION: Dispatches actions based upon which button was pressed
; ---------------------------------------------------------------------------
PROC utSpeedButtonsDispatch.u(   ; Calls action appropriate to button
         button.n)               ; Button number
Private  y                       ; Transient GetEvent DynArray
;Global  g.scrollrate.n          ; Scroll rate in milliseconds
   IF NOT IsAssigned(g.scrollrate.n) THEN    ; Initialize scroll variable
      g.scrollrate.n = 300
   ENDIF

   SWITCH
      CASE button.n = 1 :                    ; Home
         Home
         Message "Beginning of Table..."
      CASE button.n = 2 :                    ; Reverse Scroll
         IF NOT AtFirst() THEN
            WHILE NOT AtFirst()
               Skip -1
               Echo NORMAL Echo OFF
               Message "Reverse Scroll, Record ",RecNo()," - MouseClick or Press Any Key to Stop..."
               Sleep g.scrollrate.n
               GetEvent ALL To y
               IF (y["Type"] = "MOUSE" AND y["Action"] = "UP") OR
                  y["Type"] = "KEY" THEN
                  QUITLOOP
               ENDIF
            ENDWHILE
            Message "You are on Record " + StrVal(RecNo()) + "..."
         ELSE
            Beep
            Message "You are at the First Record in this Image..."
         ENDIF
      CASE button.n = 3 :                    ; Skip -1
         IF NOT AtFirst() THEN
            Skip -1
            Message "Record " +Strval([#])+ "..."
         ELSE
            Beep
            Message "You are at the First Record in this Image..."
         ENDIF
      CASE button.n = 4 :                    ; PgUp
         IF IsFormView() THEN
            IF AtFirst() AND PageNo() = 1 THEN
               Beep
               IF NPages() = 1 THEN
                  Message "You are at the First Record in this Image..."
               ELSE
                  Message "You are at the First Record's First Page in this Image..."
               ENDIF
            ELSE
               PgUp
               IF NPages() = 1 THEN
                  Message "Record " +StrVal([#])+ "..."
               ELSE
                  Message "Page " +StrVal(PageNo())+ " of Record "+Strval([#])+ "..."
               ENDIF
            ENDIF
         ELSE
            PgUp
            Message "Record " +StrVal([#])+ "..."
         ENDIF
      CASE button.n = 5 :                    ; Help
         utSpeedButtonsHelp.u()
      CASE button.n = 6 :                    ; PgDn
         IF IsFormView() THEN
            IF AtLast() AND PageNo() = NPages() THEN
               Beep
               IF NPages() = 1 THEN
                  Message "You are at the Last Record in this Image..."
               ELSE
                  Message "You are at the Last Record's Last Page in this Image..."
               ENDIF
            ELSE
               PgDn
               IF NPages() = 1 THEN
                  Message "Record " +StrVal([#])+ "..."
               ELSE
                  Message "Page " +StrVal(PageNo())+ " of Record "+Strval([#])+ "..."
               ENDIF
            ENDIF
         ELSE
            PgDn
            Message "Record " +StrVal([#])+ "..."
         ENDIF
      CASE button.n = 7 :                    ; Skip 1
         IF NOT AtLast() THEN
            Skip 1
            Message "Record " +Strval([#])+ "..."
         ELSE
            Beep
         ENDIF
      CASE button.n = 8 :                    ; Forward Scroll
         IF NOT AtLast() THEN
            WHILE NOT AtLast()
               Skip 1
               Echo NORMAL Echo OFF
               Message "Forward Scroll, Record ",RecNo()," - MouseClick or Press Any Key to Stop..."
               Sleep g.scrollrate.n
               GetEvent ALL To y
               IF (y["Type"] = "MOUSE" AND y["Action"] = "UP") OR
                  y["Type"] = "KEY" THEN
                  QUITLOOP
               ENDIF
            ENDWHILE
            Message "You are on Record " + StrVal(RecNo()) + "..."
         ELSE
            Beep
            Message "You are at the Last Record in this Image..."
         ENDIF
      CASE button.n = 9 :                    ; End
         End
         Message "End of Table..."
      OTHERWISE   : Beep                     ; Clicked a divider bar
   ENDSWITCH
   Return
ENDPROC
; ===========================================================================
;       TITLE: utSpeedButtonsHelp.u
;     RETURNS: No value
; DESCRIPTION: Popup Dialog with descriptions of SpeedButton icons, and
;              embedded Dialog Box to set scroll rate in milliseconds
; ---------------------------------------------------------------------------
PROC utSpeedButtonsHelp.u()      ; Description of SpeedButton icons
Private  button.l                ; Pushbutton variable
   SHOWDIALOG "Help on Using Speed Buttons"
      Proc "utSpeedButtonsHelpDB.l" Trigger "UPDATE"
      @ 1,0
      Height 18 Width 37

      @ 1,1 ?? "Ŀ"
      @ 2,1 ?? "                               "
      @ 3,1 ?? "  \030  Home: 1st record in table "
      @ 4,1 ?? "  \174  Reverse continuous scroll "
      @ 5,1 ?? "  \027  Back/Up one record        "
      @ 6,1 ?? " Pg\024 Page up                   "
      @ 7,1 ?? " Pg\025 Page down                 "
      @ 8,1 ?? "  \026  Next/Down one record      "
      @ 9,1 ?? "  \175  Forward continuous scroll "
      @10,1 ?? "  \031  End: Last record in table "
      @11,1 ?? "                               "
      @12,1 ?? ""
      PaintCanvas Attribute 48 1,1,12,33
      PaintCanvas Attribute 59 1,33,12,33
      PaintCanvas Attribute 59 12,2,12,33
      PaintCanvas Attribute 59 3,3,10,5

      PushButton @ 14,3
         Width 14 "~C~ontinue"
         OK Default Value true Tag "OK"
      To button.l

      PushButton @ 14,19
         Width 14 "~S~crollRate"
         Value false Tag "RATE"
      To button.l
   ENDDIALOG
   Return
ENDPROC
; ===========================================================================
;       TITLE: utSpeedButtonsHelpDB.l
;     RETURNS: No value
; DESCRIPTION: Embedded Dialog Box to set scroll rate for Scrolling icons
; ---------------------------------------------------------------------------
PROC utSpeedButtonsHelpDB.l(     ; Set scroll rate for SpeedButtons
         type.a,                 ; EVENT or TRIGGER
         tag.a,                  ; Control element tag or null
         event.v,                ; DynArray of GetEvent, or control value
         element.a)              ; Checkbox label or null
Private  button.l
   IF type.a = "UPDATE" AND tag.a = "RATE" THEN
      SHOWDIALOG "In Tenth Seconds"
         @ 17,15
         Height 6 Width 26

         PaintCanvas Fill "1   5   9  13  17 20"
                     Attribute SysColor(1003) 1,2,1,21

         Slider @ 0,1
            Horizontal Length 22 Min 100 Max 2000
            ArrowStep 100 PageStep 500 Tag "SLIDER"
         To g.scrollrate.n

         PushButton @ 2,7
            Width 10 "~S~elect"
            OK Default Value true Tag "OK"
         To button.l
      ENDDIALOG
   ENDIF
   Return true
ENDPROC

