;---------------------------------------------------------------------
; RENSETAP.SC - Renaissance (Setup - Action Points)
;---------------------------------------------------------------------


IF NOT ISASSIGNED(LibName) THEN
  Ver = STRVAL(VERSION())
  DotLoc = SEARCH(".", Ver)
  IF DotLoc > 0 THEN
    Ver = SUBSTR(Ver, 1, DotLoc - 1) + SUBSTR(Ver, DotLoc + 1, 2)
  ENDIF
  LibName = SDIR() + "RenDev" + Ver
  RELEASE VARS Ver, DotLoc
ENDIF


PROC RenSetup.ActionPoint(RenObject, RenTbl, RenFld)
  PRIVATE RenNum, RenActPt, RenAct, RenActNum, RenTag, RenTitle,
          RenCodeTitle, RenActPtList, RenNone, RenMainActPt

  RenSetup.MainActPt(RenObject, RenTbl, RenFld)

  RenSetup.BuildActPtList(VARVAL(RenObject), RenTbl, RenFld, "")

  RenNum = 1

  WHILE TRUE
    RenSetup.ActionPointDlg(RenObject, VARVAL(RenObject), RenTbl, RenFld, RenNum)

    IF RETVAL = 0 THEN
      RETURN TRUE
    ELSE
      RenNum = RETVAL
    ENDIF

    RenActPt = RenActPtList[RenNum]
    IF SUBSTR(RenActPt, 29, 1) = "" THEN
      RenActPt = SUBSTR(RenActPt, 1, 27)
      WHILE MATCH(RenActPt, ".. ", RenActPt)
      ENDWHILE
    ENDIF

    RenTag = RenActPt
    IF RenFld <> "" THEN
      RenTag = RenTag + "," + RenTbl + "," + RenFld
    ELSE
      IF RenTbl <> "" THEN
        RenTag = RenTag + "," + RenTbl
      ENDIF
    ENDIF

    RenAct    = RenDyn.Get(RenObject, RenTag)
    RenActNum = RenSetup.ActionNum(RenAct)

    IF RenActNum = 0 THEN
      RenCodeTitle = RenSetup.GetName(RenObject)
      IF RenFld <> "" THEN
        RenCodeTitle = RenCodeTitle + ":" + RenTbl + "->" + RenFld
      ELSE
        IF RenTbl <> "" THEN
          RenCodeTitle = RenCodeTitle + ":" + RenTbl
        ENDIF
      ENDIF
      RenCodeTitle = RenCodeTitle + "::" + RenActPt

      RenSetup.ProcessAction("ACTIONPOINT", RenActPt, RenCodeTitle, RenObject,
                             VARVAL(RenObject), RenTag, RenActNum, RenAct)

      IF RETVAL = "External" THEN
        LOOP
      ELSE
        RETURN TRUE
      ENDIF
    ENDIF

    IF RenActNum = 2 THEN
      RenActNum = 1
    ENDIF

    IF NOT ISASSIGNED(RenActList) THEN
      RenSetup.RenActList()
    ENDIF

    RenSetup.ActionDlg(RenActPt, RenActNum)

    IF RETVAL > 0 THEN
      IF RenActNum > 2 AND RETVAL < 2 THEN
        ; Switch from action command to custom code or none
        RenAct = ""
      ENDIF

      RenActNum = RETVAL

      RenCodeTitle = RenSetup.GetName(RenObject)
      IF RenFld <> "" THEN
        RenCodeTitle = RenCodeTitle + ":" + RenTbl + "->" + RenFld
      ELSE
        IF RenTbl <> "" THEN
          RenCodeTitle = RenCodeTitle + ":" + RenTbl
        ENDIF
      ENDIF
      RenCodeTitle = RenCodeTitle + "::" + RenActPt

      RenSetup.ProcessAction("ACTIONPOINT", RenActPt, RenCodeTitle, RenObject,
                             VARVAL(RenObject), RenTag, RenActNum, RenAct)

      IF RETVAL = 1 THEN ; <Custom Code>
        RETURN TRUE
      ENDIF

      RenSetup.BuildActPtList(VARVAL(RenObject), RenTbl, RenFld, RenActPt)
      RenNum = RETVAL

      IF RenNone = TRUE THEN
        RenNum = 1
      ENDIF
    ENDIF
  ENDWHILE
ENDPROC
WRITELIB LibName RenSetup.ActionPoint
RELEASE PROCS RenSetup.ActionPoint


PROC RenSetup.BuildActPtList(RenObjectPtr, RenTbl, RenFld, RenActPt)
  PRIVATE RenTag, RenSize, i, RenCtr, RenNum, RenTag2

  IF NOT ISASSIGNED(RenMainActPt) THEN
    RenSetup.MainActPt(RenObject, RenTbl, RenFld)
  ENDIF

  IF RenFld <> "" THEN
    RenTag = "," + RenTbl + "," + RenFld
  ELSE
    IF RenTbl <> "" THEN
      RenTag = "," + RenTbl
    ELSE
      RenTag = ""
    ENDIF
  ENDIF

  RenSize = ARRAYSIZE(RenMainActPt)

  ARRAY RenActPtList[RenSize]

  RenCtr = 0
  RenNum = 1
  RenActPtr = UPPER(RenActPt)

  FOR i FROM 1 TO RenSize
    RenTag2 = RenMainActPt[i] + RenTag

    IF ISASSIGNED(RenObjectPtr[RenTag2]) AND RenObjectPtr[RenTag2] <> "" THEN
      RenCtr = RenCtr + 1

      RenActPtList[RenCtr] = FORMAT("W27", RenMainActPt[i]) + " "

      IF RenActPt = UPPER(RenMainActPt[i]) THEN
        RenNum = RenCtr
      ENDIF
    ENDIF
  ENDFOR

  FOR i FROM 1 TO RenSize
    RenTag2 = RenMainActPt[i] + RenTag

    IF ISASSIGNED(RenObjectPtr[RenTag2]) AND RenObjectPtr[RenTag2] <> "" THEN
      ; Already processed this element.
    ELSE
      RenCtr = RenCtr + 1

      RenActPtList[RenCtr] = RenMainActPt[i]

      IF RenActPt = UPPER(RenMainActPt[i]) THEN
        RenNum = RenCtr
      ENDIF
    ENDIF
  ENDFOR

  RETURN RenNum
ENDPROC
WRITELIB LibName RenSetup.BuildActPtList
RELEASE PROCS RenSetup.BuildActPtList


PROC RenSetup.ActionPointDlg(RenObject, RenObjectPtr, RenTbl, RenFld, RenNum)
  PRIVATE RenTitle, RenSys, RenRow, RenCol, RenButton, RenLastEvent,
    PaintPAL.OriginalPalette, PaintPAL.ColorDeviations

  SWITCH
    CASE RenFld <> "" : RenTitle = RenFld + " Field"
    CASE RenTbl <> "" : RenTitle = RenTbl + " Table"
    OTHERWISE         : RenTitle = RenSetup.GetName(RenObject)
  ENDSWITCH
  RenTitle = RenTitle + " Action Points"

  SYSINFO TO RenSys
  RenRow = INT((RenSys["SCREENHEIGHT"] - 19) / 2)
  RenCol = INT((RenSys["SCREENWIDTH"]  - 52) / 2)

  RenButton = ""

  ; PaintPAL_Generated_Code_Begin(ACTIONPOINTLIST)

  ; PaintPAL_Color_Setup_Begin
  GETCOLORS TO PaintPAL.OriginalPalette
  DYNARRAY PaintPAL.ColorDeviations[]
  PaintPAL.ColorDeviations[1037] = 127  ;Label Inactive
  PaintPAL.ColorDeviations[1038] = 112  ;Label Selected
  PaintPAL.ColorDeviations[1039] = 112  ;Label Shortcut
  PaintPAL.ColorDeviations[1040] = 112  ;Pushbutton Normal
  PaintPAL.ColorDeviations[1041] = 112  ;Pushbutton Default
  PaintPAL.ColorDeviations[1042] = 127  ;Pushbutton Selected
  PaintPAL.ColorDeviations[1044] = 127  ;Pushbutton Shortcut
  SETCOLORS FROM PaintPAL.ColorDeviations
  RELEASE VARS PaintPAL.ColorDeviations
  ; PaintPAL_Color_Setup_End

  SHOWDIALOG RenTitle
     PROC "RenSetup.ActionPointDlgProc"
        MOUSE "DOWN"
        KEY "ESC", "TAB", "REVERSETAB"
        TRIGGER "UPDATE", "SELECT", "ARRIVE"

     @RenRow,RenCol HEIGHT 19 WIDTH 52

     ; PaintPAL_Frame_Begin
     FRAME SINGLE FROM 0,1 TO 16,34
     PAINTCANVAS ATTRIBUTE 112 0,1,0,34
     PAINTCANVAS ATTRIBUTE 127 16,1,16,34
     PAINTCANVAS ATTRIBUTE 112 0,1,16,1
     PAINTCANVAS ATTRIBUTE 127 0,34,16,34
     ; PaintPAL_Frame_End

     PICKARRAY @1,2 HEIGHT 15 WIDTH 31
        RenActPtList
        TAG "ActPts"
        TO RenNum

     PUSHBUTTON @5,36 WIDTH 12
        "~S~elect"
        DEFAULT
        VALUE "Select"
        TAG "Select"
        TO RenButton

     LABEL @4,35
        "~~"
        FOR "Select"

     LABEL @6,35
        "~~"
        FOR "Select"

     LABEL @5,35
        ""
        FOR "Select"

     LABEL @5,47
        "~~"
        FOR "Select"

     PUSHBUTTON @8,35 WIDTH 14
        "~C~lear"
        VALUE "Clear"
        TAG "Clear"
        TO RenButton

     LABEL @8,47
        "~~"
        FOR "Clear"

     LABEL @8,35
        ""
        FOR "Clear"

     LABEL @9,35
        "~~"
        FOR "Clear"

     LABEL @7,35
        "~~"
        FOR "Clear"

     PUSHBUTTON @11,35 WIDTH 14
        "D~o~ne"
        VALUE "Done"
        TAG "Done"
        TO RenButton

     LABEL @10,35
        "~~"
        FOR "Done"

     LABEL @12,35
        "~~"
        FOR "Done"

     LABEL @11,35
        ""
        FOR "Done"

     LABEL @11,47
        "~~"
        FOR "Done"
  ENDDIALOG

  ; PaintPAL_Color_Cleanup_Begin
  SETCOLORS FROM PaintPAL.OriginalPalette
  RELEASE VARS PaintPAL.OriginalPalette
  ; PaintPAL_Color_Cleanup_End

  ; PaintPAL_Generated_Code_End(ACTIONPOINTLIST)

  IF RenButton = "Select" THEN
    GETEVENT ALL TO RETVAL
    RETURN RenNum
  ELSE
    GETEVENT ALL TO RETVAL
    RETURN 0
  ENDIF
ENDPROC
WRITELIB LibName RenSetup.ActionPointDlg
RELEASE PROCS RenSetup.ActionPointDlg


PROC RenSetup.ActionPointDlgProc(RenTrigger, RenTag, RenEvent, RenElement)
  PRIVATE RenWin, RenActPt, RenTag

  IF NOT ISASSIGNED(RenLastEvent) THEN
    RenLastEvent = ""
  ENDIF

  IF RenTrigger = "EVENT" AND RenEvent["TYPE"] <> "IDLE" AND RenEvent["TYPE"] <> "MESSAGE" THEN
    RenLastEvent = RenEvent["TYPE"]
  ENDIF

  SWITCH
    CASE RenTrigger = "EVENT" AND RenEvent["TYPE"] = "MOUSE" : ; MOUSE DOWN
      WINDOW HANDLE DIALOG TO RenWin
      IF WINDOWAT(RenEvent["ROW"], RenEvent["COL"]) <> RenWin THEN
        CANCELDIALOG
        RETURN FALSE
      ELSE
        RETURN TRUE
      ENDIF

    CASE (RenTrigger = "ARRIVE" AND RenLastEvent = "MOUSE") OR
      RenTrigger = "UPDATE" :

      SWITCH
        CASE RenTag = "Clear" :
          RenNum = CONTROLVALUE("ActPts")

          RenActPt = RenActPtList[RenNum]
          IF SUBSTR(RenActPt, 29, 1) = "" THEN
            RenActPt = SUBSTR(RenActPt, 1, 27)
            WHILE MATCH(RenActPt, ".. ", RenActPt)
            ENDWHILE
            RenYesNo("Warning", "Clear the code from the \"" + RenActPt + "\" action point?")
          ELSE
            RETVAL = FALSE
          ENDIF

          IF NOT RETVAL THEN
            SELECTCONTROL "ActPts"
            RETURN FALSE
          ENDIF

          IF RenFld <> "" THEN
            RenTag = RenActPt + "," + RenTbl + "," + RenFld
          ELSE
            IF RenTbl <> "" THEN
              RenTag = RenActPt + "," + RenTbl
            ELSE
              RenTag = RenActPt
            ENDIF
          ENDIF

          RELEASE VARS RenObjectPtr[RenTag]

          RenSetup.BuildActPtList(RenObjectPtr, RenTbl, RenFld, RenActPt)
          RenNum = 1

          SELECTCONTROL "ActPts"
          REFRESHDIALOG

          RETURN FALSE

        CASE RenTag = "Select" :
          ACCEPTDIALOG
          RenButton = "Select"
          RETURN FALSE

        CASE RenTag = "Done" :
          CANCELDIALOG
          RenButton = "Done"
          RETURN FALSE
      ENDSWITCH

    CASE RenTrigger = "SELECT" :
      IF RenTag = "ActPts" THEN
        RenButton = "Select"
        ACCEPTDIALOG
        RETURN FALSE
      ENDIF

    CASE RenTrigger = "EVENT" AND RenEvent["TYPE"] = "KEY" :
      IF RenEvent["KEYCODE"] = 27 THEN
        CANCELDIALOG
        RETURN FALSE
      ENDIF
  ENDSWITCH

  RETURN TRUE
ENDPROC
WRITELIB libname RenSetup.ActionPointDlgProc
RELEASE PROCS RenSetup.ActionPointDlgProc


PROC RenSetup.ActionDlg(RenActPt, RenActNum)
  PRIVATE RenTitle, RenSys, RenRow, RenCol, RenButton, RenActTitle, RenLastEvent,
    PaintPAL.OriginalPalette, PaintPAL.ColorDeviations

  RenTitle = RenSetup.GetName(RenObject)
  IF RenFld <> "" THEN
    RenTitle = RenTitle + ":" + RenTbl + "->" + RenFld
  ELSE
    IF RenTbl <> "" THEN
      RenTitle = RenTitle + ":" + RenTbl
    ENDIF
  ENDIF

  SYSINFO TO RenSys
  RenRow = INT((RenSys["SCREENHEIGHT"] - 19) / 2)
  RenCol = INT((RenSys["SCREENWIDTH"]  - 52) / 2)

  RenButton = ""

  RenActTitle = "Action for " + RenActPt
  IF LEN(RenActTitle) > 30 THEN
    RenActTitle = SUBSTR(RenActTitle, 1, 32)
  ENDIF
  RenActTitle = " " + RenActTitle + " "
  RenActCol = 18 - INT(LEN(RenActTitle) / 2)

  ; PaintPAL_Generated_Code_Begin(ACTIONLIST)

  ; PaintPAL_Color_Setup_Begin
  GETCOLORS TO PaintPAL.OriginalPalette
  DYNARRAY PaintPAL.ColorDeviations[]
  PaintPAL.ColorDeviations[1037] = 127  ;Label Inactive
  PaintPAL.ColorDeviations[1038] = 112  ;Label Selected
  PaintPAL.ColorDeviations[1039] = 112  ;Label Shortcut
  PaintPAL.ColorDeviations[1040] = 112  ;Pushbutton Normal
  PaintPAL.ColorDeviations[1041] = 112  ;Pushbutton Default
  PaintPAL.ColorDeviations[1042] = 127  ;Pushbutton Selected
  PaintPAL.ColorDeviations[1044] = 127  ;Pushbutton Shortcut
  SETCOLORS FROM PaintPAL.ColorDeviations
  RELEASE VARS PaintPAL.ColorDeviations
  ; PaintPAL_Color_Setup_End

  SHOWDIALOG RenTitle
     PROC "RenSetup.ActionDlgProc"
        MOUSE "DOWN"
        KEY "ESC", "TAB", "REVERSETAB"
        TRIGGER "UPDATE", "SELECT", "ARRIVE"

     @RenRow,RenCol HEIGHT 19 WIDTH 52

     ; PaintPAL_Frame_Begin
     FRAME SINGLE FROM 0,1 TO 16,34
     PAINTCANVAS ATTRIBUTE 112 0,1,0,34
     PAINTCANVAS ATTRIBUTE 127 16,1,16,34
     PAINTCANVAS ATTRIBUTE 112 0,1,16,1
     PAINTCANVAS ATTRIBUTE 127 0,34,16,34
     ; PaintPAL_Frame_End

     ; PaintPAL_Static_Text_Begin
     PAINTCANVAS FILL " " ATTRIBUTE 112 0,13,0,21
     @0,RenActCol
     ?? RenActTitle
     PAINTCANVAS ATTRIBUTE 112 0,13,0,21
     ; PaintPAL_Static_Text_End

     PICKARRAY @1,2 HEIGHT 15 WIDTH 31
        RenActList
        TAG "Actions"
        TO RenActNum

     PUSHBUTTON @6,35 WIDTH 14
        "~O~K"
        DEFAULT
        VALUE "OK"
        TAG "OK"
        TO RenButton

     LABEL @5,35
        "~~"
        FOR "OK"

     LABEL @7,35
        "~~"
        FOR "OK"

     LABEL @6,35
        ""
        FOR "OK"

     LABEL @6,47
        "~~"
        FOR "OK"

     PUSHBUTTON @9,35 WIDTH 14
        "~C~ancel"
        VALUE "Cancel"
        TAG "Cancel"
        TO RenButton

     LABEL @9,47
        "~~"
        FOR "Cancel"

     LABEL @9,35
        ""
        FOR "Cancel"

     LABEL @10,35
        "~~"
        FOR "Cancel"

     LABEL @8,35
        "~~"
        FOR "Cancel"
  ENDDIALOG

  ; PaintPAL_Color_Cleanup_Begin
  SETCOLORS FROM PaintPAL.OriginalPalette
  RELEASE VARS PaintPAL.OriginalPalette
  ; PaintPAL_Color_Cleanup_End

  ; PaintPAL_Generated_Code_End(ACTIONLIST)

  IF RenButton = "OK" THEN
    GETEVENT ALL TO RETVAL
    RETURN RenActNum
  ELSE
    GETEVENT ALL TO RETVAL
    RETURN 0
  ENDIF
ENDPROC
WRITELIB LibName RenSetup.ActionDlg
RELEASE PROCS RenSetup.ActionDlg


PROC RenSetup.ActionDlgProc(RenTrigger, RenTag, RenEvent, RenElement)
  PRIVATE RenWin, RenActPt, RenTag

  IF NOT ISASSIGNED(RenLastEvent) THEN
    RenLastEvent = ""
  ENDIF

  IF RenTrigger = "EVENT" AND RenEvent["TYPE"] <> "IDLE" AND RenEvent["TYPE"] <> "MESSAGE" THEN
    RenLastEvent = RenEvent["TYPE"]
  ENDIF

  SWITCH
    CASE RenTrigger = "EVENT" AND RenEvent["TYPE"] = "MOUSE" : ; MOUSE DOWN
      WINDOW HANDLE DIALOG TO RenWin
      IF WINDOWAT(RenEvent["ROW"], RenEvent["COL"]) <> RenWin THEN
        CANCELDIALOG
        RETURN FALSE
      ELSE
        RETURN TRUE
      ENDIF

    CASE (RenTrigger = "ARRIVE" AND RenLastEvent = "MOUSE") OR
      RenTrigger = "UPDATE" :

      SWITCH
        CASE RenTag = "OK" :
          ACCEPTDIALOG
          RenButton = "OK"
          RETURN FALSE

        CASE RenTag = "Cancel" :
          CANCELDIALOG
          RenButton = "Cancel"
          RETURN FALSE
      ENDSWITCH

    CASE RenTrigger = "SELECT" :
      IF RenTag = "Actions" THEN
        RenButton = "OK"
        ACCEPTDIALOG
      ENDIF

    CASE RenTrigger = "EVENT" AND RenEvent["TYPE"] = "KEY" :
      IF RenEvent["KEYCODE"] = 27 THEN
        CANCELDIALOG
        RETURN FALSE
      ENDIF
  ENDSWITCH

  RETURN TRUE
ENDPROC
WRITELIB libname RenSetup.ActionDlgProc
RELEASE PROCS RenSetup.ActionDlgProc


PROC RenSetup.ActionNum(RenVal)
  PRIVATE RenAct, NumAct, i, RenLen

  IF NOT ISASSIGNED(RenActList) THEN
    RenSetup.RenActList()
  ENDIF

  IF RenVal = "" THEN
    RETURN 2
  ENDIF

  RenLen = LEN(RenVal)

  SWITCH
    CASE UPPER(SUBSTR(RenVal, 1, 11)) = "RENACTION(\"" AND
         SUBSTR(RenVal, RenLen - 1, 2) = "\")" :

      NumAct = ARRAYSIZE(RenActList)
      RenAct = UPPER(SUBSTR(RenVal, 12, RenLen - 13))

      IF MATCH(RenAct, "..,..", RenAct) THEN
        ; Removed action command parameters
      ENDIF

      FOR i FROM 3 TO NumAct
        IF RenAct = UPPER(RenActList[i]) THEN
          RETURN i
        ENDIF
      ENDFOR

    CASE UPPER(RenVal) = ";NONE" :
      RETURN 2
  ENDSWITCH

  RETURN 0
ENDPROC
WRITELIB LibName RenSetup.ActionNum
RELEASE PROCS RenSetup.ActionNum


PROC RenSetup.ProcessAction(RenType, RenActPt, RenTitle, RenObject,
                            RenObjectPtr, RenTag, RenActNum, RenAct)

  PRIVATE RenVal, RenCompVal, RenProps, RenParm, RenProc, RenProcFnd,
          RenCode, RenConfig

  RenObject = UPPER(RenObject)
  RenType   = UPPER(RenType)
  RenTag    = UPPER(RenTag)
  RenNone   = FALSE

  SWITCH
    CASE RenActNum = 0 OR RenActNum = 1 : ; <Custom Code>
      RenActNum = 1

    CASE RenActNum = 2 : ; <None>
      RenAct = ";None"
      RenNone = TRUE

    CASE RenActNum > 2 :
      IF ISASSIGNED(RenActParm[RenActNum]) THEN
        RenParm = ""

        IF RenActNum = RenSetup.ActionNum(RenAct) THEN
          IF MATCH(RenAct, "RenAction(..,..)", RenAct, RenParm) THEN
            RenParm = SUBSTR(RenParm, 1, LEN(RenParm) - 1)
          ENDIF
        ENDIF

        RenAct = RenActList[RenActNum]

        RenSetup.GetActParms(RenAct, RenParm, RenActNum, TRUE)
        IF RETVAL = FALSE THEN
          RETURN FALSE
        ENDIF

        IF RETVAL = "" THEN
          RenAct = "RenAction(\"" + RenAct + "\")"
        ELSE
          RenAct = "RenAction(\"" + RenAct + "," + RETVAL + "\")"
        ENDIF
      ELSE
        RenAct = RenActList[RenActNum]
        RenAct = "RenAction(\"" + RenAct + "\")"
      ENDIF
  ENDSWITCH

  SWITCH
    CASE RenType = "ACTIONPOINT" :
      IF RenActNum = 2 THEN
        RELEASE VARS RenObjectPtr[RenTag]
      ELSE
        RenObjectPtr[RenTag] = RenAct
      ENDIF

    CASE RenType = "BUTTON" :
      RenObjectPtr[RenTag] = RenAct
      RenSetup.RedisplayButtons(RenObject)

    OTHERWISE :
      RenObjectPtr[RenTag] = RenAct
  ENDSWITCH

  IF RenActNum = 1 THEN
    RenCompVal = RenObject + "," + RenTag

    IF ISASSIGNED(RenCodeObjects[RenCompVal]) THEN
      RenObject.Select(RenCodeObjects[RenCompVal])
      RETURN 2
    ENDIF

    RenProc = RenSetup.ProcName(RenObject, RenTag, RenActPt, RenTitle)
    RenCode = RenSetup.GetProc(RenProc)

    DYNARRAY RenConfig[]
    RenSetup.GetConfig(RenConfig)

    IF RenConfig["EDITOR"] <> "" THEN
      RenSetup.CallEditor(RenConfig["EDITOR"], RenConfig["SCLOC"], RenObject, RenTag, RenCode, RenProc,  RenTitle)
      RETURN "External"
    ELSE
      DYNARRAY RenProps[]
      RenProps["FROMTYPE"] = RenType
      RenProps["FROMTAG"]  = RenTag
      RenProps["PROC"]     = RenProc
      RenProps["CODE"]     = RenCode
      RenProps["TITLE"]    = RenTitle

      RenCode.Create(RenObject, RenProps)
      IF RETVAL <> "" THEN
        RenObject.Select(RETVAL)
        RETURN 1
      ELSE
        RenContinue("Could not open custom code window")
        RETURN FALSE
      ENDIF
    ENDIF
  ELSE
    RETURN TRUE
  ENDIF
ENDPROC
WRITELIB LibName RenSetup.ProcessAction
RELEASE PROCS RenSetup.ProcessAction


PROC RenSetup.ProcName(RenObject, RenTag, RenActPt, RenCom)
  PRIVATE RenProc, i, RenProcFnd, RenCode

  RenProc = RenDyn.Get(RenObject, RenTag)
  IF RenProc <> "" THEN
    RenSetup.GetProc(RenProc)
    IF RenProcFnd = TRUE THEN
      RETURN RenProc
    ENDIF
    RELEASE VARS RETVAL
    RenCode = RenProc
  ELSE
    RenCode = "\n"
  ENDIF

  ECHO OFF

  EDITOR OPEN PRIVDIR() + "_Renwork.sc"

  i = 0
  WHILE TRUE
    i = i + 1
    RenProc = Ren["SETUPPROC"] + RenActPt + STRVAL(i)
    EDITOR FIND "PROC " + RenProc + "("
    IF NOT RETVAL THEN
      CANCELEDIT
      RenSetup.SaveProc(RenProc + "()", RenCode, TRUE, FALSE, RenCom)
      RenDyn.Set(RenObject, RenTag, RenProc + "()")
      RETURN RenProc + "()"
    ENDIF
    CTRLPGUP
  ENDWHILE
ENDPROC
WRITELIB LibName RenSetup.ProcName
RELEASE PROCS RenSetup.ProcName


PROC RenSetup.GetProc(RenProc)
  PRIVATE RenStr, RenStart, RenEnd

  RenProcFnd = FALSE

  FILEREAD PRIVDIR() + "_Renwork.sc" TO RenStr

  RenStart = SEARCH("PROC " + RenProc, RenStr)

  IF RenStart = 0 THEN
    RETURN ""
  ENDIF

  RenStart = SEARCHFROM("\n", RenStr, RenStart)

  IF RenStart = 0 THEN
    RETURN ""
  ENDIF

  RenStart = RenStart + 1

  RenEnd = SEARCHFROM("ENDPROC", RenStr,  RenStart)

  IF RenEnd = 0 THEN
    RETURN ""
  ENDIF

  RenProcFnd = TRUE

  RETURN SUBSTR(RenStr, RenStart, RenEnd - RenStart)
ENDPROC
WRITELIB LibName RenSetup.GetProc
RELEASE PROCS RenSetup.GetProc


PROC RenSetup.SaveProc(RenProc, RenVal, RenWrite, RenClosed, RenCom)
  PRIVATE RenPrefix, RenStr, RenStart, RenEnd, RenProc2

  IF SUBSTR(RenVal, LEN(RenVal), 1) <> "\n" THEN
    RenVal = RenVal + "\n"
  ENDIF

  FILEREAD PRIVDIR() + "_Renwork.sc" TO RenStr

  IF RenClosed THEN
    RenPrefix = "PROC CLOSED "
  ELSE
    RenPrefix = "PROC "
  ENDIF

  RenStart = SEARCH(RenPreFix + RenProc, RenStr)

  IF RenStart = 0 THEN
    IF RenClosed THEN
      RenPrefix = "PROC "
    ELSE
      RenPrefix = "PROC CLOSED "
    ENDIF

    RenStart = SEARCH(RenPreFix + RenProc, RenStr)

    IF RenStart = 0 THEN
      IF RenClosed THEN
        RenPrefix = "PROC CLOSED "
      ELSE
        RenPrefix = "PROC "
      ENDIF
    ENDIF
  ENDIF

  IF RenStart > 0 THEN
    RenEnd = SEARCHFROM("ENDPROC", RenStr, RenStart)
    IF RenEnd > 0 THEN
      IF RenCom = "Sets up autolib and calls the main workspace procedure" THEN
        RETVAL = SEARCHFROM("ENDPROC", RenStr, RenEnd + 1)
        IF RETVAL > 0 THEN
          RenEnd = RETVAL
        ENDIF
      ENDIF
      IF RenEnd > 0 THEN
        FILEWRITE PRIVDIR() + "_Renwork.sc" FROM
          SUBSTR(RenStr, 1, RenStart - 1) +
          RenPrefix + RenProc + "\n" +
          RenVal +
          SUBSTR(RenStr, RenEnd, LEN(RenStr) - RenEnd + 1)
        RETURN
      ENDIF
    ENDIF
  ENDIF

  RELEASE VARS RenStr

  IF NOT MATCH(RenProc, "..()", RenProc2) THEN
    RenProc2 = RenProc
  ENDIF

  EDITOR OPEN PRIVDIR() + "_Renwork.sc"

  EDITOR FIND ";*** RENAISSANCE: END OF PROCEDURES"
  IF RETVAL THEN
    HOME
    ENTER
    ENTER
    UP
    UP
  ELSE
    CTRLPGDN
    ENTER
    ENTER
  ENDIF

  EDITOR INSERT
    ";==============================================================================\n" +
    "; " + RenProc2 + "\n" +
    ";\n" +
    "; " + RenCom + "\n" +
    ";==============================================================================\n" +
    RenPrefix + RenProc + "\n" +
    RenVal +
    "ENDPROC\n"

  IF RenWrite = TRUE THEN
    EDITOR INSERT
      "WRITELIB Lib " + RenProc2 + "\n" +
      "RELEASE PROCS " + RenProc2 + "\n"
  ENDIF

  EDITOR INSERT "\n\n"

  DO_IT!
ENDPROC
WRITELIB LibName RenSetup.SaveProc
RELEASE PROCS RenSetup.SaveProc


PROC RenSetup.CallEditor(RenEditor, RenScLoc, RenObject, RenTag, RenCode, RenProc, RenTitle)
  PRIVATE RenTemp, RenSaveErrorProc, RenError, RenFromType, RenCmd,
    RenFromObj, RenFromTag, RenWin, RenAttr, RenInfo, RenTxt, RenEdLen

  RenTemp = PRIVDIR() + "_Rencode.sc"

  RenSetSkipError()
  FILEWRITE RenTemp FROM RenCode
  RenUnSetSkipError()

  IF RenError THEN
    RenContinue("Could not open custom code window")
    RETURN FALSE
  ENDIF

  RenEdLen = LEN(RenEditor)

  WHILE TRUE
    RenMsg("Calling editor...")

    IF RenScLoc > 0 THEN
      RenCmd = SUBSTR(RenEditor, 1, RenScLoc - 1) + RenTemp +
               SUBSTR(RenEditor, RenScLoc + 1, RenEdLen - RenScLoc + 1)
    ELSE
      RenCmd = RenEditor + " " + RenTemp
    ENDIF

    RUN BIG NOREFRESH RenCmd

    RenMsgClose()

    WHILE CHARWAITING()
      RETVAL = GETCHAR()
    ENDWHILE

    IF NOT ISFILE(RenTemp) THEN
      RETURN FALSE
    ENDIF

    FILEREAD RenTemp TO RenCode

    ECHO OFF

    RenMsg("Checking syntax...")

    RenCheckSyntaxStr(RenCode)
    IF NOT RETVAL THEN
      RenMsgClose()
      RenButtonDlg("Problem", "Paradox found syntax errors in your code...\n\n" +
                   "Do you want to edit the code or save it with syntax errors?",
                   "Edit,Save")
      IF RETVAL = "Edit" THEN
        LOOP
      ENDIF
    ENDIF

    RenMsgClose()

    RenSetup.SaveProc(RenProc, RenCode, TRUE, FALSE, RenTitle)

    RETURN TRUE
  ENDWHILE
ENDPROC
WRITELIB LibName RenSetup.CallEditor
RELEASE PROCS RenSetup.CallEditor


PROC RenSetup.GetActParms(RenAct, RenParm, RenActNum, RenCheck)
  PRIVATE RenType, RenFormObj, RenTbl, RenFrm, RenList, RenDef,
          RenClass, RenSize, i, RenSaveTbl, RenSaveFld, RenFld,
          RenLast, RenCnt, RenStr, RenIdx

  RenType = RenActParm[RenActNum]

  SWITCH
    CASE RenType = "Message" :
      RenAccept(RenAct, "Message:", "A255", RenParm)
      IF RETVAL = FALSE THEN
        RETURN FALSE
      ELSE
        RETURN RETVAL
      ENDIF

    CASE RenType = "Object" :
      IF ISASSIGNED(RenTags[RenParm]) THEN
        RenDef = RenTags[RenParm]
      ELSE
        RenDef = 0 ; No default
      ENDIF

      RenSetup.SelectObject(RenAct + " Object Selection", RenDef, FALSE, FALSE, FALSE, TRUE)
      IF RETVAL = "" THEN
        RETURN FALSE
      ELSE
        RETURN RenDyn.Get(RETVAL, "TAG")
      ENDIF

    CASE RenType = "FormTable" :
      IF RenCheck = FALSE THEN
        IF RenParm = "" THEN
          RETURN "Table"
        ELSE
          RETURN RenParm
        ENDIF
      ENDIF

      IF RenObjectPtr["CLASS"] = "RENBUTTONBAR" THEN
        RenFormObj = Ren["CURRENTOBJECT"]
      ELSE
        RenFormObj = RenObject
      ENDIF

      IF RenDyn.Get(RenFormObj, "CLASS") <> "RENFORM" THEN
        RenContinue("The \"" + RenAct + "\" action command\n" +
                    "can only be used with forms")
        RETURN FALSE
      ENDIF

      RenTbl = RenDyn.Get(RenFormObj, "TABLE")
      RenFrm = RenDyn.Get(RenFormObj, "FORM")

      FORMTABLES RenTbl RenFrm RenList
      IF NOT RETVAL THEN
        RenContinue("The \"" + RenAct + "\" action command can only be\n"+
                    "used with forms that have embedded tables")
        RETURN FALSE
      ENDIF

      RenSize = ARRAYSIZE(RenList)
      RenDef = 1
      FOR i FROM 1 TO RenSize
        IF UPPER(RenList[i]) = UPPER(RenParm) THEN
          RenDef = i
          QUITLOOP
        ENDIF
      ENDFOR

      RenPickArray(RenAct + " Table Selection", RenList, RenDef)
      IF RETVAL = 0 THEN
        RETURN FALSE
      ELSE
        RETURN RenList[RETVAL]
      ENDIF

    CASE RenType = "Field" :
      IF RenCheck = FALSE THEN
        IF RenParm = "" THEN
          RETURN "Field"
        ELSE
          RETURN RenParm
        ENDIF
        RETURN FALSE
      ENDIF

      IF RenObjectPtr["CLASS"] = "RENBUTTONBAR" THEN
        RenFormObj = Ren["CURRENTOBJECT"]
      ELSE
        RenFormObj = RenObject
      ENDIF

      RenClass = RenDyn.Get(RenFormObj, "CLASS")
      IF RenClass <> "RENFORM" AND RenClass <> "RENTABLE" THEN
        RenContinue("The \"" + RenAct + "\" action command can\n" +
                    "only be used with tables and forms")
        RETURN FALSE
      ENDIF

      IF RenClass = "RENFORM" THEN
        RenSaveTbl = TABLE()
        RenSaveFld = FIELD()

        RenTbl = RenDyn.Get(RenFormObj, "TABLE")

        IF UPPER(RenSaveTbl) <> UPPER(RenTbl) THEN
          MOVETO RenTbl
        ENDIF

        DYNARRAY RenList[]
        RenGetFields(RenList, TRUE, TRUE, FALSE, "Value")

        MOVETO RenSaveTbl
        MOVETO FIELD RenSaveFld
      ELSE
        RenGetFields(RenList, FALSE, FALSE, FALSE, "Value")
      ENDIF

      RenDef = ""
      FOREACH i IN RenList
        IF UPPER(RenList[i]) = UPPER(RenParm) THEN
          RenDef = i
          QUITLOOP
        ENDIF
      ENDFOREACH

      RenPickDynarray(RenAct + " Field Selection", RenList, RenDef)
      IF RETVAL = "" THEN
        RETURN FALSE
      ELSE
        RETURN RenList[RETVAL]
      ENDIF

    CASE RenType = "Records" :
      IF NUMVAL(RenParm) = "Error" THEN
        RenDef = BLANKNUM()
      ELSE
        RenDef = NUMVAL(RenParm)
      ENDIF

      RenAccept(RenAct, "Number of Records:", "S", RenDef)
      IF RETVAL = FALSE THEN
        RETURN FALSE
      ELSE
        RETURN STRVAL(RETVAL)
      ENDIF

    CASE RenType = "Button" :
      IF NUMVAL(RenParm) = "Error" THEN
        RenDef = BLANKNUM()
      ELSE
        RenDef = NUMVAL(RenParm)
      ENDIF

      ARRAY RenList[15]
      FOR i FROM 1 TO 15
        RenList[i] = STRVAL(i)
      ENDFOR

      RenPickArray(RenAct + " Selection", RenList, RenDef)
      IF RETVAL = 0 THEN
        RETURN FALSE
      ELSE
        RETURN RenList[RETVAL]
      ENDIF

    CASE RenType = "Field,Index" :
      IF RenCheck = FALSE THEN
        IF RenParm = "" THEN
          RETURN "Field,Index"
        ELSE
          RETURN RenParm
        ENDIF
        RETURN FALSE
      ENDIF

      IF NOT MATCH(RenParm, "..,..", RenFld, RenIdx) THEN
        RenFld = ""
        RenIdx = ""
      ENDIF

      IF RenObjectPtr["CLASS"] = "RENBUTTONBAR" THEN
        RenFormObj = Ren["CURRENTOBJECT"]
      ELSE
        RenFormObj = RenObject
      ENDIF

      RenClass = RenDyn.Get(RenFormObj, "CLASS")
      IF RenClass <> "RENFORM" AND RenClass <> "RENTABLE" THEN
        RenContinue("The \"" + RenAct + "\" action command can\n" +
                    "only be used with tables and forms")
        RETURN FALSE
      ENDIF

      RenSaveTbl = TABLE()
      RenSaveFld = FIELD()

      IF RenClass = "RENFORM" THEN
        RenTbl = RenDyn.Get(RenFormObj, "TABLE")

        IF UPPER(RenSaveTbl) <> UPPER(RenTbl) THEN
          MOVETO RenTbl
        ENDIF

        DYNARRAY RenList[]
        RenGetFields(RenList, FALSE, FALSE, FALSE, "Value")

        MOVETO RenSaveTbl
        MOVETO FIELD RenSaveFld
      ELSE
        RenGetFields(RenList, FALSE, FALSE, FALSE, "Value")
      ENDIF

      RenDef = ""
      FOREACH i IN RenList
        IF UPPER(RenList[i]) = UPPER(RenFld) THEN
          RenDef = i
          QUITLOOP
        ENDIF
      ENDFOREACH

      RenList[" "] = "<Current Field>"

      RenPickDynarray(RenAct + " Field Selection", RenList, RenDef)
      IF RETVAL = "" THEN
        MOVETO RenSaveTbl
        MOVETO FIELD RenSaveFld
        RETURN FALSE
      ENDIF

      IF RETVAL = " " THEN
        MOVETO RenSaveTbl
        MOVETO FIELD RenSaveFld
        RETURN ""
      ENDIF

      RenFld = RenList[RETVAL]

      MOVETO FIELD RenFld
      ORDERTABLE
      CTRLPGDN
      RenLast = MENUCHOICE()

      RenDef = ""

      CTRLPGUP
      DYNARRAY RenList[]
      RenCnt = 0
      WHILE TRUE
        RenCnt = RenCnt + 1
        RenStr = FILL("0", 4 - LEN(RenCnt)) + STRVAL(RenCnt)
        RenList[RenStr] = MENUCHOICE()
        IF UPPER(MENUCHOICE()) = UPPER(RenIdx) THEN
          RenDef = RenStr
        ENDIF
        IF RenList[RenStr] = RenLast THEN
          QUITLOOP
        ELSE
          DOWN
        ENDIF
      ENDWHILE
      ESC

      MOVETO RenSaveTbl
      MOVETO FIELD RenSaveFld

      IF RenCnt = 1 THEN
        RETURN RenFld + "," + RenList["0001"]
      ENDIF

      RenPickDynarray(RenAct + " Index Selection", RenList, RenDef)
      IF RETVAL = "" THEN
        RETURN ""
      ENDIF

      RETURN RenFld + "," + RenList[RETVAL]
  ENDSWITCH

  RETURN ""
ENDPROC
WRITELIB LibName RenSetup.GetActParms
RELEASE PROCS RenSetup.GetActParms


PROC RenSetup.RenActList()
  ARRAY RenActList[53]                  ARRAY RenActParm[53]
  RenActList[01] = "<Custom Code>"
  RenActList[02] = "<None>"
  RenActList[03] = "AddToLookup"
  RenActList[04] = "Browse"
  RenActList[05] = "Button"             RenActParm[05] = "Button"
  RenActList[06] = "Calendar"
  RenActList[07] = "Cancel"
  RenActList[08] = "Clock"
  RenActList[09] = "Close"
  RenActList[10] = "Copy"
  RenActList[11] = "Delete"
  RenActList[12] = "DownImage"
  RenActList[13] = "Erase"
  RenActList[14] = "FieldView"
  RenActList[15] = "Filter"
  RenActList[16] = "FirstRecord"
  RenActList[17] = "FormToggle"
  RenActList[18] = "Globe"
  RenActList[19] = "Help"
  RenActList[20] = "Insert"
  RenActList[21] = "LastRecord"
  RenActList[22] = "LockRecord"
  RenActList[23] = "Lookup"
  RenActList[24] = "Message"            RenActParm[24] = "Message"
  RenActList[25] = "MoveToField"        RenActParm[25] = "Field"
  RenActList[26] = "MoveToTable"        RenActParm[26] = "FormTable"
  RenActList[27] = "MoveToWindow"       RenActParm[27] = "Object"
  RenActList[28] = "NextMasterRecord"
  RenActList[29] = "NextRecord"
  RenActList[30] = "NextWindow"
  RenActList[31] = "OkDialog"           RenActParm[31] = "Message"
  RenActList[32] = "Options"
  RenActList[33] = "OrderTable"         RenActParm[33] = "Field,Index"
  RenActList[34] = "Paste"
  RenActList[35] = "PostRecord"
  RenActList[36] = "PrevMasterRecord"
  RenActList[37] = "PrevRecord"
  RenActList[38] = "PrevWindow"
  RenActList[39] = "Prompt"             RenActParm[39] = "Message"
  RenActList[40] = "Quit"
  RenActList[41] = "Renaissance"
  RenActList[42] = "Save"
  RenActList[43] = "ShowPulldown"
  RenActList[44] = "Skip"               RenActParm[44] = "Records"
  RenActList[45] = "Undo"
  RenActList[46] = "UpImage"
  RenActList[47] = "Video25"
  RenActList[48] = "Video50"
  RenActList[49] = "WalkingMan"
  RenActList[50] = "WindowList"
  RenActList[51] = "XCut"
  RenActList[52] = "Zoom"
  RenActList[53] = "ZoomNext"
ENDPROC
WRITELIB LibName RenSetup.RenActList
RELEASE PROCS RenSetup.RenActList


PROC RenSetup.MainActPt(RenObject, RenTbl, RenFld)
  PRIVATE RenClass

  RenClass = RenDyn.Get(RenObject, "CLASS")

  SWITCH
    CASE RenClass = "RENSETUP" :
      RenSetup.MainActPtWorkspace()

    CASE RenClass = "RENTABLE" :
      IF RenFld <> "" THEN
        RenSetup.MainActPtField()
      ELSE
        RenSetup.MainActPtTable()
      ENDIF

    CASE RenClass = "RENFORM" :
      IF RenFld <> "" THEN
        RenSetup.MainActPtField()
      ELSE
        IF RenTbl <> "" THEN
          RenSetup.MainActPtFormTable()
        ELSE
          RenSetup.MainActPtForm()
        ENDIF
      ENDIF

    CASE RenClass = "RENMEMO" :
      RenSetup.MainActPtMemo()

    CASE RenClass = "RENEDITOR" :
      RenSetup.MainActPtEditor()

    OTHERWISE :
      RenSetup.MainActPtWindow()
  ENDSWITCH
ENDPROC
WRITELIB LibName RenSetup.MainActPt
RELEASE PROCS RenSetup.MainActPt


PROC RenSetup.MainActPtWorkspace()
  ARRAY RenMainActPt[79]

  RenMainActPt[01] = "AfterAddToLookup"
  RenMainActPt[02] = "AfterCancelDelete"
  RenMainActPt[03] = "AfterCancelInsert"
  RenMainActPt[04] = "AfterCancelModify"
  RenMainActPt[05] = "AfterDelete"
  RenMainActPt[06] = "AfterLookup"
  RenMainActPt[07] = "AfterPost"
  RenMainActPt[08] = "AfterPostInsert"
  RenMainActPt[09] = "AfterPostModify"
  RenMainActPt[10] = "AfterQuit"
  RenMainActPt[11] = "ArriveField"
  RenMainActPt[12] = "ArrivePage"
  RenMainActPt[13] = "ArriveRow"
  RenMainActPt[14] = "ArriveTable"
  RenMainActPt[15] = "ArriveWindow"
  RenMainActPt[16] = "BeforeCancelInsert"
  RenMainActPt[17] = "BeforeCancelModify"
  RenMainActPt[18] = "BeforeDelete"
  RenMainActPt[19] = "BeforeEdit"
  RenMainActPt[20] = "BeforeEditInsert"
  RenMainActPt[21] = "BeforeEditModify"
  RenMainActPt[22] = "BeforeFieldView"
  RenMainActPt[23] = "BeforeInsert"
  RenMainActPt[24] = "BeforeLookup"
  RenMainActPt[25] = "BeforeModify"
  RenMainActPt[26] = "BeforePostInsert"
  RenMainActPt[27] = "BeforePostModify"
  RenMainActPt[28] = "BeforeQuit"
  RenMainActPt[29] = "BeforeSetup"
  RenMainActPt[30] = "BeforeVerifyDelete"
  RenMainActPt[31] = "BeforeWait"
  RenMainActPt[32] = "BeforeWaitLookup"
  RenMainActPt[33] = "Browse"
  RenMainActPt[34] = "Cancel"
  RenMainActPt[35] = "CheckRecord"
  RenMainActPt[36] = "Copy"
  RenMainActPt[37] = "DepartField"
  RenMainActPt[38] = "DepartPage"
  RenMainActPt[39] = "DepartRow"
  RenMainActPt[40] = "DepartTable"
  RenMainActPt[41] = "DepartWindow"
  RenMainActPt[42] = "DisplayOnly"
  RenMainActPt[43] = "DownImage"
  RenMainActPt[44] = "Erase"
  RenMainActPt[45] = "Filter"
  RenMainActPt[46] = "FormToggle"
  RenMainActPt[47] = "HelpAction"
  RenMainActPt[48] = "Idle"
  RenMainActPt[49] = "ImageRights"
  RenMainActPt[50] = "KeyAll"
  RenMainActPt[51] = "MessageAll"
  RenMainActPt[52] = "MessageClose"
  RenMainActPt[53] = "MessageMaximize"
  RenMainActPt[54] = "MessageMenuKey"
  RenMainActPt[55] = "MessageMenuSelect"
  RenMainActPt[56] = "MouseAll"
  RenMainActPt[57] = "MouseAuto"
  RenMainActPt[58] = "MouseDoubleLeft"
  RenMainActPt[59] = "MouseDoubleRight"
  RenMainActPt[60] = "MouseDown"
  RenMainActPt[61] = "MouseLeft"
  RenMainActPt[62] = "MouseMove"
  RenMainActPt[63] = "MouseRight"
  RenMainActPt[64] = "MouseUp"
  RenMainActPt[65] = "Options"
  RenMainActPt[66] = "OrderTable"
  RenMainActPt[67] = "PassRights"
  RenMainActPt[68] = "Paste"
  RenMainActPt[69] = "PostRecord"
  RenMainActPt[70] = "ReadOnly"
  RenMainActPt[71] = "Save"
  RenMainActPt[72] = "ShowPullDown"
  RenMainActPt[73] = "TouchRecord"
  RenMainActPt[74] = "UpImage"
  RenMainActPt[75] = "ValCheck"
  RenMainActPt[76] = "WindowList"
  RenMainActPt[77] = "XCut"
  RenMainActPt[78] = "Zoom"
  RenMainActPt[79] = "ZoomNext"
ENDPROC
WRITELIB LibName RenSetup.MainActPtWorkspace
RELEASE PROCS RenSetup.MainActPtWorkspace


PROC RenSetup.MainActPtForm()
  ARRAY RenMainActPt[68]

  RenMainActPt[01] = "AfterAddToLookup"
  RenMainActPt[02] = "AfterCancelDelete"
  RenMainActPt[03] = "AfterCancelInsert"
  RenMainActPt[04] = "AfterCancelModify"
  RenMainActPt[05] = "AfterDelete"
  RenMainActPt[06] = "AfterLookup"
  RenMainActPt[07] = "AfterPost"
  RenMainActPt[08] = "AfterPostInsert"
  RenMainActPt[09] = "AfterPostModify"
  RenMainActPt[10] = "ArriveField"
  RenMainActPt[11] = "ArrivePage"
  RenMainActPt[12] = "ArriveRow"
  RenMainActPt[13] = "ArriveTable"
  RenMainActPt[14] = "ArriveWindow"
  RenMainActPt[15] = "BeforeCancelInsert"
  RenMainActPt[16] = "BeforeCancelModify"
  RenMainActPt[17] = "BeforeDelete"
  RenMainActPt[18] = "BeforeEdit"
  RenMainActPt[19] = "BeforeEditInsert"
  RenMainActPt[20] = "BeforeEditModify"
  RenMainActPt[21] = "BeforeFieldView"
  RenMainActPt[22] = "BeforeInsert"
  RenMainActPt[23] = "BeforeLookup"
  RenMainActPt[24] = "BeforeModify"
  RenMainActPt[25] = "BeforePostInsert"
  RenMainActPt[26] = "BeforePostModify"
  RenMainActPt[27] = "BeforeVerifyDelete"
  RenMainActPt[28] = "BeforeWaitLookup"
  RenMainActPt[29] = "Browse"
  RenMainActPt[30] = "CheckRecord"
  RenMainActPt[31] = "DepartField"
  RenMainActPt[32] = "DepartPage"
  RenMainActPt[33] = "DepartRow"
  RenMainActPt[34] = "DepartTable"
  RenMainActPt[35] = "DepartWindow"
  RenMainActPt[36] = "DisplayOnly"
  RenMainActPt[37] = "DownImage"
  RenMainActPt[38] = "Filter"
  RenMainActPt[39] = "FormToggle"
  RenMainActPt[40] = "HelpAction"
  RenMainActPt[41] = "Idle"
  RenMainActPt[42] = "ImageRights"
  RenMainActPt[43] = "KeyAll"
  RenMainActPt[44] = "MessageAll"
  RenMainActPt[45] = "MessageClose"
  RenMainActPt[46] = "MessageMaximize"
  RenMainActPt[47] = "MessageMenuKey"
  RenMainActPt[48] = "MessageMenuSelect"
  RenMainActPt[49] = "MouseAll"
  RenMainActPt[50] = "MouseAuto"
  RenMainActPt[51] = "MouseDoubleLeft"
  RenMainActPt[52] = "MouseDoubleRight"
  RenMainActPt[53] = "MouseDown"
  RenMainActPt[54] = "MouseLeft"
  RenMainActPt[55] = "MouseMove"
  RenMainActPt[56] = "MouseRight"
  RenMainActPt[57] = "MouseUp"
  RenMainActPt[58] = "Options"
  RenMainActPt[59] = "OrderTable"
  RenMainActPt[60] = "PassRights"
  RenMainActPt[61] = "PostRecord"
  RenMainActPt[62] = "ReadOnly"
  RenMainActPt[63] = "ShowPullDown"
  RenMainActPt[64] = "TouchRecord"
  RenMainActPt[65] = "UpImage"
  RenMainActPt[66] = "ValCheck"
  RenMainActPt[67] = "Zoom"
  RenMainActPt[68] = "ZoomNext"
ENDPROC
WRITELIB LibName RenSetup.MainActPtForm
RELEASE PROCS RenSetup.MainActPtForm


PROC RenSetup.MainActPtTable()
  ARRAY RenMainActPt[66]

  RenMainActPt[01] = "AfterAddToLookup"
  RenMainActPt[02] = "AfterCancelDelete"
  RenMainActPt[03] = "AfterCancelInsert"
  RenMainActPt[04] = "AfterCancelModify"
  RenMainActPt[05] = "AfterDelete"
  RenMainActPt[06] = "AfterLookup"
  RenMainActPt[07] = "AfterPost"
  RenMainActPt[08] = "AfterPostInsert"
  RenMainActPt[09] = "AfterPostModify"
  RenMainActPt[10] = "ArriveField"
  RenMainActPt[11] = "ArriveRow"
  RenMainActPt[12] = "ArriveTable"
  RenMainActPt[13] = "ArriveWindow"
  RenMainActPt[14] = "BeforeCancelInsert"
  RenMainActPt[15] = "BeforeCancelModify"
  RenMainActPt[16] = "BeforeDelete"
  RenMainActPt[17] = "BeforeEdit"
  RenMainActPt[18] = "BeforeEditInsert"
  RenMainActPt[19] = "BeforeEditModify"
  RenMainActPt[20] = "BeforeFieldView"
  RenMainActPt[21] = "BeforeInsert"
  RenMainActPt[22] = "BeforeLookup"
  RenMainActPt[23] = "BeforeModify"
  RenMainActPt[24] = "BeforePostInsert"
  RenMainActPt[25] = "BeforePostModify"
  RenMainActPt[26] = "BeforeVerifyDelete"
  RenMainActPt[27] = "BeforeWaitLookup"
  RenMainActPt[28] = "Browse"
  RenMainActPt[29] = "CheckRecord"
  RenMainActPt[30] = "DepartField"
  RenMainActPt[31] = "DepartRow"
  RenMainActPt[32] = "DepartTable"
  RenMainActPt[33] = "DepartWindow"
  RenMainActPt[34] = "DisplayOnly"
  RenMainActPt[35] = "DownImage"
  RenMainActPt[36] = "Filter"
  RenMainActPt[37] = "FormToggle"
  RenMainActPt[38] = "HelpAction"
  RenMainActPt[39] = "Idle"
  RenMainActPt[40] = "ImageRights"
  RenMainActPt[41] = "KeyAll"
  RenMainActPt[42] = "MessageAll"
  RenMainActPt[43] = "MessageClose"
  RenMainActPt[44] = "MessageMaximize"
  RenMainActPt[45] = "MessageMenuKey"
  RenMainActPt[46] = "MessageMenuSelect"
  RenMainActPt[47] = "MouseAll"
  RenMainActPt[48] = "MouseAuto"
  RenMainActPt[49] = "MouseDoubleLeft"
  RenMainActPt[50] = "MouseDoubleRight"
  RenMainActPt[51] = "MouseDown"
  RenMainActPt[52] = "MouseLeft"
  RenMainActPt[53] = "MouseMove"
  RenMainActPt[54] = "MouseRight"
  RenMainActPt[55] = "MouseUp"
  RenMainActPt[56] = "Options"
  RenMainActPt[57] = "OrderTable"
  RenMainActPt[58] = "PassRights"
  RenMainActPt[59] = "PostRecord"
  RenMainActPt[60] = "ReadOnly"
  RenMainActPt[61] = "ShowPullDown"
  RenMainActPt[62] = "TouchRecord"
  RenMainActPt[63] = "UpImage"
  RenMainActPt[64] = "ValCheck"
  RenMainActPt[65] = "Zoom"
  RenMainActPt[66] = "ZoomNext"
ENDPROC
WRITELIB LibName RenSetup.MainActPtTable
RELEASE PROCS RenSetup.MainActPtTable


PROC RenSetup.MainActPtFormTable()
  ARRAY RenMainActPt[61]

  RenMainActPt[01] = "AfterAddToLookup"
  RenMainActPt[02] = "AfterCancelDelete"
  RenMainActPt[03] = "AfterCancelInsert"
  RenMainActPt[04] = "AfterCancelModify"
  RenMainActPt[05] = "AfterDelete"
  RenMainActPt[06] = "AfterLookup"
  RenMainActPt[07] = "AfterPost"
  RenMainActPt[08] = "AfterPostInsert"
  RenMainActPt[09] = "AfterPostModify"
  RenMainActPt[10] = "ArriveField"
  RenMainActPt[11] = "ArriveRow"
  RenMainActPt[12] = "ArriveTable"
  RenMainActPt[13] = "BeforeCancelInsert"
  RenMainActPt[14] = "BeforeCancelModify"
  RenMainActPt[15] = "BeforeDelete"
  RenMainActPt[16] = "BeforeEdit"
  RenMainActPt[17] = "BeforeEditInsert"
  RenMainActPt[18] = "BeforeEditModify"
  RenMainActPt[19] = "BeforeFieldView"
  RenMainActPt[20] = "BeforeInsert"
  RenMainActPt[21] = "BeforeLookup"
  RenMainActPt[22] = "BeforeModify"
  RenMainActPt[23] = "BeforePostInsert"
  RenMainActPt[24] = "BeforePostModify"
  RenMainActPt[25] = "BeforeVerifyDelete"
  RenMainActPt[26] = "BeforeWaitLookup"
  RenMainActPt[27] = "Browse"
  RenMainActPt[28] = "CheckRecord"
  RenMainActPt[29] = "DepartField"
  RenMainActPt[30] = "DepartRow"
  RenMainActPt[31] = "DepartTable"
  RenMainActPt[32] = "DisplayOnly"
  RenMainActPt[33] = "DownImage"
  RenMainActPt[34] = "Filter"
  RenMainActPt[35] = "FormToggle"
  RenMainActPt[36] = "HelpAction"
  RenMainActPt[37] = "ImageRights"
  RenMainActPt[38] = "KeyAll"
  RenMainActPt[39] = "MessageAll"
  RenMainActPt[40] = "MessageClose"
  RenMainActPt[41] = "MessageMaximize"
  RenMainActPt[42] = "MessageMenuKey"
  RenMainActPt[43] = "MessageMenuSelect"
  RenMainActPt[44] = "MouseAll"
  RenMainActPt[45] = "MouseAuto"
  RenMainActPt[46] = "MouseDoubleLeft"
  RenMainActPt[47] = "MouseDoubleRight"
  RenMainActPt[48] = "MouseDown"
  RenMainActPt[49] = "MouseLeft"
  RenMainActPt[50] = "MouseMove"
  RenMainActPt[51] = "MouseRight"
  RenMainActPt[52] = "MouseUp"
  RenMainActPt[53] = "OrderTable"
  RenMainActPt[54] = "PassRights"
  RenMainActPt[55] = "PostRecord"
  RenMainActPt[56] = "ReadOnly"
  RenMainActPt[57] = "TouchRecord"
  RenMainActPt[58] = "UpImage"
  RenMainActPt[59] = "ValCheck"
  RenMainActPt[60] = "Zoom"
  RenMainActPt[61] = "ZoomNext"
ENDPROC
WRITELIB LibName RenSetup.MainActPtFormTable
RELEASE PROCS RenSetup.MainActPtFormTable


PROC RenSetup.MainActPtField()
  ARRAY RenMainActPt[25]

  RenMainActPt[01] = "AfterAddToLookup"
  RenMainActPt[02] = "AfterLookup"
  RenMainActPt[03] = "ArriveField"
  RenMainActPt[04] = "BeforeFieldView"
  RenMainActPt[05] = "BeforeLookup"
  RenMainActPt[06] = "BeforeWaitLookup"
  RenMainActPt[07] = "DepartField"
  RenMainActPt[08] = "Filter"
  RenMainActPt[09] = "HelpAction"
  RenMainActPt[10] = "KeyAll"
  RenMainActPt[11] = "MessageAll"
  RenMainActPt[12] = "MessageClose"
  RenMainActPt[13] = "MessageMaximize"
  RenMainActPt[14] = "MessageMenuKey"
  RenMainActPt[15] = "MessageMenuSelect"
  RenMainActPt[16] = "MouseAll"
  RenMainActPt[17] = "MouseAuto"
  RenMainActPt[18] = "MouseDoubleLeft"
  RenMainActPt[19] = "MouseDoubleRight"
  RenMainActPt[20] = "MouseDown"
  RenMainActPt[21] = "MouseLeft"
  RenMainActPt[22] = "MouseMove"
  RenMainActPt[23] = "MouseRight"
  RenMainActPt[24] = "MouseUp"
  RenMainActPt[25] = "ValCheck"
ENDPROC
WRITELIB LibName RenSetup.MainActPtField
RELEASE PROCS RenSetup.MainActPtField


PROC RenSetup.MainActPtMemo()
  ARRAY RenMainActPt[29]

  RenMainActPt[01] = "ArriveWindow"
  RenMainActPt[02] = "Cancel"
  RenMainActPt[03] = "Copy"
  RenMainActPt[04] = "DepartWindow"
  RenMainActPt[05] = "Erase"
  RenMainActPt[06] = "HelpAction"
  RenMainActPt[07] = "Idle"
  RenMainActPt[08] = "ImageRights"
  RenMainActPt[09] = "KeyAll"
  RenMainActPt[10] = "MessageAll"
  RenMainActPt[11] = "MessageClose"
  RenMainActPt[12] = "MessageMaximize"
  RenMainActPt[13] = "MessageMenuKey"
  RenMainActPt[14] = "MessageMenuSelect"
  RenMainActPt[15] = "MouseAll"
  RenMainActPt[16] = "MouseAuto"
  RenMainActPt[17] = "MouseDoubleLeft"
  RenMainActPt[18] = "MouseDoubleRight"
  RenMainActPt[19] = "MouseDown"
  RenMainActPt[20] = "MouseLeft"
  RenMainActPt[21] = "MouseMove"
  RenMainActPt[22] = "MouseRight"
  RenMainActPt[23] = "MouseUp"
  RenMainActPt[24] = "Options"
  RenMainActPt[25] = "Paste"
  RenMainActPt[26] = "Save"
  RenMainActPt[27] = "ShowPullDown"
  RenMainActPt[28] = "TouchRecord"
  RenMainActPt[29] = "XCut"
ENDPROC
WRITELIB LibName RenSetup.MainActPtMemo
RELEASE PROCS RenSetup.MainActPtMemo


PROC RenSetup.MainActPtEditor()
  ARRAY RenMainActPt[27]

  RenMainActPt[01] = "ArriveWindow"
  RenMainActPt[02] = "Cancel"
  RenMainActPt[03] = "Copy"
  RenMainActPt[04] = "DepartWindow"
  RenMainActPt[05] = "Erase"
  RenMainActPt[06] = "HelpAction"
  RenMainActPt[07] = "Idle"
  RenMainActPt[08] = "KeyAll"
  RenMainActPt[09] = "MessageAll"
  RenMainActPt[10] = "MessageClose"
  RenMainActPt[11] = "MessageMaximize"
  RenMainActPt[12] = "MessageMenuKey"
  RenMainActPt[13] = "MessageMenuSelect"
  RenMainActPt[14] = "MouseAll"
  RenMainActPt[15] = "MouseAuto"
  RenMainActPt[16] = "MouseDoubleLeft"
  RenMainActPt[17] = "MouseDoubleRight"
  RenMainActPt[18] = "MouseDown"
  RenMainActPt[19] = "MouseLeft"
  RenMainActPt[20] = "MouseMove"
  RenMainActPt[21] = "MouseRight"
  RenMainActPt[22] = "MouseUp"
  RenMainActPt[23] = "Options"
  RenMainActPt[24] = "Paste"
  RenMainActPt[25] = "Save"
  RenMainActPt[26] = "ShowPullDown"
  RenMainActPt[27] = "XCut"
ENDPROC
WRITELIB LibName RenSetup.MainActPtEditor
RELEASE PROCS RenSetup.MainActPtEditor


PROC RenSetup.MainActPtWindow()
  ARRAY RenMainActPt[21]

  RenMainActPt[01] = "ArriveWindow"
  RenMainActPt[02] = "DepartWindow"
  RenMainActPt[03] = "HelpAction"
  RenMainActPt[04] = "Idle"
  RenMainActPt[05] = "KeyAll"
  RenMainActPt[06] = "MessageAll"
  RenMainActPt[07] = "MessageClose"
  RenMainActPt[08] = "MessageMaximize"
  RenMainActPt[09] = "MessageMenuKey"
  RenMainActPt[10] = "MessageMenuSelect"
  RenMainActPt[11] = "MouseAll"
  RenMainActPt[12] = "MouseAuto"
  RenMainActPt[13] = "MouseDoubleLeft"
  RenMainActPt[14] = "MouseDoubleRight"
  RenMainActPt[15] = "MouseDown"
  RenMainActPt[16] = "MouseLeft"
  RenMainActPt[17] = "MouseMove"
  RenMainActPt[18] = "MouseRight"
  RenMainActPt[19] = "MouseUp"
  RenMainActPt[20] = "Options"
  RenMainActPt[21] = "ShowPullDown"
ENDPROC
WRITELIB LibName RenSetup.MainActPtWindow
RELEASE PROCS RenSetup.MainActPtWindow


;PROC RenSetup.MainActPtWorkspace()
;  ARRAY RenMainActPt[78]
;
;  RenMainActPt[01] = "BeforeSetup"             ;;; Workspace
;  RenMainActPt[02] = "BeforeWait"
;  RenMainActPt[03] = "BeforeQuit"
;  RenMainActPt[04] = "AfterQuit"
;  RenMainActPt[05] = "WindowList"
;  RenMainActPt[06] = "ArriveWindow"            ;;; Window
;  RenMainActPt[07] = "DepartWindow"
;  RenMainActPt[08] = "Options"
;  RenMainActPt[09] = "ShowPullDown"
;  RenMainActPt[10] = "ArriveRow"               ;;; Table
;  RenMainActPt[11] = "DepartRow"
;  RenMainActPt[12] = "ArriveTable"
;  RenMainActPt[13] = "DepartTable"
;  RenMainActPt[14] = "BeforeEdit"
;  RenMainActPt[15] = "PostRecord"
;  RenMainActPt[16] = "CheckRecord"
;  RenMainActPt[17] = "AfterPost"
;  RenMainActPt[18] = "BeforeInsert"
;  RenMainActPt[19] = "BeforeEditInsert"
;  RenMainActPt[20] = "BeforePostInsert"
;  RenMainActPt[21] = "AfterPostInsert"
;  RenMainActPt[22] = "BeforeCancelInsert"
;  RenMainActPt[23] = "AfterCancelInsert"
;  RenMainActPt[24] = "BeforeModify"
;  RenMainActPt[25] = "BeforeEditModify"
;  RenMainActPt[26] = "BeforePostModify"
;  RenMainActPt[27] = "AfterPostModify"
;  RenMainActPt[28] = "BeforeCancelModify"
;  RenMainActPt[29] = "AfterCancelModify"
;  RenMainActPt[30] = "BeforeVerifyDelete"
;  RenMainActPt[31] = "BeforeDelete"
;  RenMainActPt[32] = "AfterDelete"
;  RenMainActPt[33] = "AfterCancelDelete"
;  RenMainActPt[34] = "ImageRights"             ; Also Memo
;  RenMainActPt[35] = "Browse"
;  RenMainActPt[36] = "OrderTable"
;  RenMainActPt[37] = "FormToggle"
;  RenMainActPt[38] = "DownImage"
;  RenMainActPt[39] = "UpImage"
;  RenMainActPt[40] = "Zoom"
;  RenMainActPt[41] = "ZoomNext"
;  RenMainActPt[42] = "TouchRecord"             ; Also Memo
;  RenMainActPt[43] = "PassRights"
;  RenMainActPt[44] = "DisplayOnly"
;  RenMainActPt[45] = "ReadOnly"
;  RenMainActPt[46] = "ArriveField"             ;;; Field
;  RenMainActPt[47] = "DepartField"
;  RenMainActPt[48] = "BeforeFieldView"
;  RenMainActPt[49] = "BeforeLookup"
;  RenMainActPt[50] = "BeforeWaitLookup"
;  RenMainActPt[51] = "AfterLookup"
;  RenMainActPt[52] = "ValCheck"
;  RenMainActPt[53] = "Filter"
;  RenMainActPt[54] = "ArrivePage"              ;;; Form
;  RenMainActPt[55] = "DepartPage"
;  RenMainActPt[56] = "Save"                    ;;; Editor
;  RenMainActPt[57] = "Cancel"
;  RenMainActPt[58] = "XCut"
;  RenMainActPt[59] = "Copy"
;  RenMainActPt[60] = "Paste"
;  RenMainActPt[61] = "Erase"
;  RenMainActPt[62] = "HelpAction"                    ;;; All
;  RenMainActPt[63] = "MouseLeft"
;  RenMainActPt[64] = "MouseRight"
;  RenMainActPt[65] = "MouseDoubleLeft"
;  RenMainActPt[66] = "MouseDoubleRight"
;  RenMainActPt[67] = "KeyAll"
;  RenMainActPt[68] = "MouseAll"
;  RenMainActPt[69] = "MouseDown"
;  RenMainActPt[70] = "MouseUp"
;  RenMainActPt[71] = "MouseMove"
;  RenMainActPt[72] = "MouseAuto"
;  RenMainActPt[73] = "MessageAll"
;  RenMainActPt[74] = "MessageClose"
;  RenMainActPt[75] = "MessageMaximize"
;  RenMainActPt[76] = "MessageMenuSelect"
;  RenMainActPt[77] = "MessageMenuKey"
;  RenMainActPt[78] = "Idle"
;ENDPROC
;WRITELIB LibName RenSetup.MainActPtWorkspace
;RELEASE PROCS RenSetup.MainActPtWorkspace
;
;
;PROC RenSetup.MainActPtForm()
;  ARRAY RenMainActPt[67]
;
;  RenMainActPt[01] = "ArriveWindow"            ;;; Window
;  RenMainActPt[02] = "DepartWindow"
;  RenMainActPt[03] = "Options"
;  RenMainActPt[04] = "ShowPullDown"
;  RenMainActPt[05] = "ArriveRow"               ;;; Table
;  RenMainActPt[06] = "DepartRow"
;  RenMainActPt[07] = "ArriveTable"
;  RenMainActPt[08] = "DepartTable"
;  RenMainActPt[09] = "BeforeEdit"
;  RenMainActPt[10] = "PostRecord"
;  RenMainActPt[11] = "CheckRecord"
;  RenMainActPt[12] = "AfterPost"
;  RenMainActPt[13] = "BeforeInsert"
;  RenMainActPt[14] = "BeforeEditInsert"
;  RenMainActPt[15] = "BeforePostInsert"
;  RenMainActPt[16] = "AfterPostInsert"
;  RenMainActPt[17] = "BeforeCancelInsert"
;  RenMainActPt[18] = "AfterCancelInsert"
;  RenMainActPt[19] = "BeforeModify"
;  RenMainActPt[20] = "BeforeEditModify"
;  RenMainActPt[21] = "BeforePostModify"
;  RenMainActPt[22] = "AfterPostModify"
;  RenMainActPt[23] = "BeforeCancelModify"
;  RenMainActPt[24] = "AfterCancelModify"
;  RenMainActPt[25] = "BeforeVerifyDelete"
;  RenMainActPt[26] = "BeforeDelete"
;  RenMainActPt[27] = "AfterDelete"
;  RenMainActPt[28] = "AfterCancelDelete"
;  RenMainActPt[29] = "ImageRights"             ; Also Memo
;  RenMainActPt[30] = "Browse"
;  RenMainActPt[31] = "OrderTable"
;  RenMainActPt[32] = "FormToggle"
;  RenMainActPt[33] = "DownImage"
;  RenMainActPt[34] = "UpImage"
;  RenMainActPt[35] = "Zoom"
;  RenMainActPt[36] = "ZoomNext"
;  RenMainActPt[37] = "TouchRecord"             ; Also Memo
;  RenMainActPt[38] = "PassRights"
;  RenMainActPt[39] = "DisplayOnly"
;  RenMainActPt[40] = "ReadOnly"
;  RenMainActPt[41] = "ArriveField"             ;;; Field
;  RenMainActPt[42] = "DepartField"
;  RenMainActPt[43] = "BeforeFieldView"
;  RenMainActPt[44] = "BeforeLookup"
;  RenMainActPt[45] = "BeforeWaitLookup"
;  RenMainActPt[46] = "AfterLookup"
;  RenMainActPt[47] = "ValCheck"
;  RenMainActPt[48] = "Filter"
;  RenMainActPt[49] = "ArrivePage"              ;;; Form
;  RenMainActPt[50] = "DepartPage"
;  RenMainActPt[51] = "HelpAction"                    ;;; All
;  RenMainActPt[52] = "MouseLeft"
;  RenMainActPt[53] = "MouseRight"
;  RenMainActPt[54] = "MouseDoubleLeft"
;  RenMainActPt[55] = "MouseDoubleRight"
;  RenMainActPt[56] = "KeyAll"
;  RenMainActPt[57] = "MouseAll"
;  RenMainActPt[58] = "MouseDown"
;  RenMainActPt[59] = "MouseUp"
;  RenMainActPt[60] = "MouseMove"
;  RenMainActPt[61] = "MouseAuto"
;  RenMainActPt[62] = "MessageAll"
;  RenMainActPt[63] = "MessageClose"
;  RenMainActPt[64] = "MessageMaximize"
;  RenMainActPt[65] = "MessageMenuSelect"
;  RenMainActPt[66] = "MessageMenuKey"
;  RenMainActPt[67] = "Idle"
;ENDPROC
;WRITELIB LibName RenSetup.MainActPtForm
;RELEASE PROCS RenSetup.MainActPtForm
;
;
;PROC RenSetup.MainActPtTable()
;  ARRAY RenMainActPt[65]
;
;  RenMainActPt[01] = "ArriveWindow"            ;;; Window
;  RenMainActPt[02] = "DepartWindow"
;  RenMainActPt[03] = "Options"
;  RenMainActPt[04] = "ShowPullDown"
;  RenMainActPt[05] = "ArriveRow"               ;;; Table
;  RenMainActPt[06] = "DepartRow"
;  RenMainActPt[07] = "ArriveTable"
;  RenMainActPt[08] = "DepartTable"
;  RenMainActPt[09] = "BeforeEdit"
;  RenMainActPt[10] = "PostRecord"
;  RenMainActPt[11] = "CheckRecord"
;  RenMainActPt[12] = "AfterPost"
;  RenMainActPt[13] = "BeforeInsert"
;  RenMainActPt[14] = "BeforeEditInsert"
;  RenMainActPt[15] = "BeforePostInsert"
;  RenMainActPt[16] = "AfterPostInsert"
;  RenMainActPt[17] = "BeforeCancelInsert"
;  RenMainActPt[18] = "AfterCancelInsert"
;  RenMainActPt[19] = "BeforeModify"
;  RenMainActPt[20] = "BeforeEditModify"
;  RenMainActPt[21] = "BeforePostModify"
;  RenMainActPt[22] = "AfterPostModify"
;  RenMainActPt[23] = "BeforeCancelModify"
;  RenMainActPt[24] = "AfterCancelModify"
;  RenMainActPt[25] = "BeforeVerifyDelete"
;  RenMainActPt[26] = "BeforeDelete"
;  RenMainActPt[27] = "AfterDelete"
;  RenMainActPt[28] = "AfterCancelDelete"
;  RenMainActPt[29] = "ImageRights"             ; Also Memo
;  RenMainActPt[30] = "Browse"
;  RenMainActPt[31] = "OrderTable"
;  RenMainActPt[32] = "FormToggle"
;  RenMainActPt[33] = "DownImage"
;  RenMainActPt[34] = "UpImage"
;  RenMainActPt[35] = "Zoom"
;  RenMainActPt[36] = "ZoomNext"
;  RenMainActPt[37] = "TouchRecord"             ; Also Memo
;  RenMainActPt[38] = "PassRights"
;  RenMainActPt[39] = "DisplayOnly"
;  RenMainActPt[40] = "ReadOnly"
;  RenMainActPt[41] = "ArriveField"             ;;; Field
;  RenMainActPt[42] = "DepartField"
;  RenMainActPt[43] = "BeforeFieldView"
;  RenMainActPt[44] = "BeforeLookup"
;  RenMainActPt[45] = "BeforeWaitLookup"
;  RenMainActPt[46] = "AfterLookup"
;  RenMainActPt[47] = "ValCheck"
;  RenMainActPt[48] = "Filter"
;  RenMainActPt[49] = "HelpAction"                    ;;; All
;  RenMainActPt[50] = "MouseLeft"
;  RenMainActPt[51] = "MouseRight"
;  RenMainActPt[52] = "MouseDoubleLeft"
;  RenMainActPt[53] = "MouseDoubleRight"
;  RenMainActPt[54] = "KeyAll"
;  RenMainActPt[55] = "MouseAll"
;  RenMainActPt[56] = "MouseDown"
;  RenMainActPt[57] = "MouseUp"
;  RenMainActPt[58] = "MouseMove"
;  RenMainActPt[59] = "MouseAuto"
;  RenMainActPt[60] = "MessageAll"
;  RenMainActPt[61] = "MessageClose"
;  RenMainActPt[62] = "MessageMaximize"
;  RenMainActPt[63] = "MessageMenuSelect"
;  RenMainActPt[64] = "MessageMenuKey"
;  RenMainActPt[65] = "Idle"
;ENDPROC
;WRITELIB LibName RenSetup.MainActPtTable
;RELEASE PROCS RenSetup.MainActPtTable
;
;
;PROC RenSetup.MainActPtFormTable()
;  ARRAY RenMainActPt[60]
;
;  RenMainActPt[01] = "ArriveRow"               ;;; Table
;  RenMainActPt[02] = "DepartRow"
;  RenMainActPt[03] = "ArriveTable"
;  RenMainActPt[04] = "DepartTable"
;  RenMainActPt[05] = "BeforeEdit"
;  RenMainActPt[06] = "PostRecord"
;  RenMainActPt[07] = "CheckRecord"
;  RenMainActPt[08] = "AfterPost"
;  RenMainActPt[09] = "BeforeInsert"
;  RenMainActPt[10] = "BeforeEditInsert"
;  RenMainActPt[11] = "BeforePostInsert"
;  RenMainActPt[12] = "AfterPostInsert"
;  RenMainActPt[13] = "BeforeCancelInsert"
;  RenMainActPt[14] = "AfterCancelInsert"
;  RenMainActPt[15] = "BeforeModify"
;  RenMainActPt[16] = "BeforeEditModify"
;  RenMainActPt[17] = "BeforePostModify"
;  RenMainActPt[18] = "AfterPostModify"
;  RenMainActPt[19] = "BeforeCancelModify"
;  RenMainActPt[20] = "AfterCancelModify"
;  RenMainActPt[21] = "BeforeVerifyDelete"
;  RenMainActPt[22] = "BeforeDelete"
;  RenMainActPt[23] = "AfterDelete"
;  RenMainActPt[24] = "AfterCancelDelete"
;  RenMainActPt[25] = "ImageRights"             ; Also Memo
;  RenMainActPt[26] = "Browse"
;  RenMainActPt[27] = "OrderTable"
;  RenMainActPt[28] = "FormToggle"
;  RenMainActPt[29] = "DownImage"
;  RenMainActPt[30] = "UpImage"
;  RenMainActPt[31] = "Zoom"
;  RenMainActPt[32] = "ZoomNext"
;  RenMainActPt[33] = "TouchRecord"             ; Also Memo
;  RenMainActPt[34] = "PassRights"
;  RenMainActPt[35] = "DisplayOnly"
;  RenMainActPt[36] = "ReadOnly"
;  RenMainActPt[37] = "ArriveField"             ;;; Field
;  RenMainActPt[38] = "DepartField"
;  RenMainActPt[39] = "BeforeFieldView"
;  RenMainActPt[40] = "BeforeLookup"
;  RenMainActPt[41] = "BeforeWaitLookup"
;  RenMainActPt[42] = "AfterLookup"
;  RenMainActPt[43] = "ValCheck"
;  RenMainActPt[44] = "Filter"
;  RenMainActPt[45] = "HelpAction"                    ;;; All
;  RenMainActPt[46] = "MouseLeft"
;  RenMainActPt[47] = "MouseRight"
;  RenMainActPt[48] = "MouseDoubleLeft"
;  RenMainActPt[49] = "MouseDoubleRight"
;  RenMainActPt[50] = "KeyAll"
;  RenMainActPt[51] = "MouseAll"
;  RenMainActPt[52] = "MouseDown"
;  RenMainActPt[53] = "MouseUp"
;  RenMainActPt[54] = "MouseMove"
;  RenMainActPt[55] = "MouseAuto"
;  RenMainActPt[56] = "MessageAll"
;  RenMainActPt[57] = "MessageClose"
;  RenMainActPt[58] = "MessageMaximize"
;  RenMainActPt[59] = "MessageMenuSelect"
;  RenMainActPt[60] = "MessageMenuKey"
;ENDPROC
;WRITELIB LibName RenSetup.MainActPtFormTable
;RELEASE PROCS RenSetup.MainActPtFormTable
;
;
;PROC RenSetup.MainActPtField()
;  ARRAY RenMainActPt[24]
;
;  RenMainActPt[01] = "ArriveField"             ;;; Field
;  RenMainActPt[02] = "DepartField"
;  RenMainActPt[03] = "BeforeFieldView"
;  RenMainActPt[04] = "BeforeLookup"
;  RenMainActPt[05] = "BeforeWaitLookup"
;  RenMainActPt[06] = "AfterLookup"
;  RenMainActPt[07] = "ValCheck"
;  RenMainActPt[08] = "Filter"
;  RenMainActPt[09] = "HelpAction"                    ;;; All
;  RenMainActPt[10] = "MouseLeft"
;  RenMainActPt[11] = "MouseRight"
;  RenMainActPt[12] = "MouseDoubleLeft"
;  RenMainActPt[13] = "MouseDoubleRight"
;  RenMainActPt[14] = "KeyAll"
;  RenMainActPt[15] = "MouseAll"
;  RenMainActPt[16] = "MouseDown"
;  RenMainActPt[17] = "MouseUp"
;  RenMainActPt[18] = "MouseMove"
;  RenMainActPt[19] = "MouseAuto"
;  RenMainActPt[20] = "MessageAll"
;  RenMainActPt[21] = "MessageClose"
;  RenMainActPt[22] = "MessageMaximize"
;  RenMainActPt[23] = "MessageMenuSelect"
;  RenMainActPt[24] = "MessageMenuKey"
;ENDPROC
;WRITELIB LibName RenSetup.MainActPtField
;RELEASE PROCS RenSetup.MainActPtField
;
;
;PROC RenSetup.MainActPtMemo()
;  ARRAY RenMainActPt[29]
;
;  RenMainActPt[01] = "ArriveWindow"            ;;; Window
;  RenMainActPt[02] = "DepartWindow"
;  RenMainActPt[03] = "Options"
;  RenMainActPt[04] = "ShowPullDown"
;  RenMainActPt[05] = "ImageRights"             ; Also Memo
;  RenMainActPt[06] = "TouchRecord"             ; Also Memo
;  RenMainActPt[07] = "Save"                    ;;; Editor
;  RenMainActPt[08] = "Cancel"
;  RenMainActPt[09] = "XCut"
;  RenMainActPt[10] = "Copy"
;  RenMainActPt[11] = "Paste"
;  RenMainActPt[12] = "Erase"
;  RenMainActPt[13] = "HelpAction"                    ;;; All
;  RenMainActPt[14] = "MouseLeft"
;  RenMainActPt[15] = "MouseRight"
;  RenMainActPt[16] = "MouseDoubleLeft"
;  RenMainActPt[17] = "MouseDoubleRight"
;  RenMainActPt[18] = "KeyAll"
;  RenMainActPt[19] = "MouseAll"
;  RenMainActPt[20] = "MouseDown"
;  RenMainActPt[21] = "MouseUp"
;  RenMainActPt[22] = "MouseMove"
;  RenMainActPt[23] = "MouseAuto"
;  RenMainActPt[24] = "MessageAll"
;  RenMainActPt[25] = "MessageClose"
;  RenMainActPt[26] = "MessageMaximize"
;  RenMainActPt[27] = "MessageMenuSelect"
;  RenMainActPt[28] = "MessageMenuKey"
;  RenMainActPt[29] = "Idle"
;ENDPROC
;WRITELIB LibName RenSetup.MainActPtMemo
;RELEASE PROCS RenSetup.MainActPtMemo
;
;
;PROC RenSetup.MainActPtEditor()
;  ARRAY RenMainActPt[27]
;
;  RenMainActPt[01] = "ArriveWindow"            ;;; Window
;  RenMainActPt[02] = "DepartWindow"
;  RenMainActPt[03] = "Options"
;  RenMainActPt[04] = "ShowPullDown"
;  RenMainActPt[05] = "Save"                    ;;; Editor
;  RenMainActPt[06] = "Cancel"
;  RenMainActPt[07] = "XCut"
;  RenMainActPt[08] = "Copy"
;  RenMainActPt[09] = "Paste"
;  RenMainActPt[10] = "Erase"
;  RenMainActPt[11] = "HelpAction"                    ;;; All
;  RenMainActPt[12] = "MouseLeft"
;  RenMainActPt[13] = "MouseRight"
;  RenMainActPt[14] = "MouseDoubleLeft"
;  RenMainActPt[15] = "MouseDoubleRight"
;  RenMainActPt[16] = "KeyAll"
;  RenMainActPt[17] = "MouseAll"
;  RenMainActPt[18] = "MouseDown"
;  RenMainActPt[19] = "MouseUp"
;  RenMainActPt[20] = "MouseMove"
;  RenMainActPt[21] = "MouseAuto"
;  RenMainActPt[22] = "MessageAll"
;  RenMainActPt[23] = "MessageClose"
;  RenMainActPt[24] = "MessageMaximize"
;  RenMainActPt[25] = "MessageMenuSelect"
;  RenMainActPt[26] = "MessageMenuKey"
;  RenMainActPt[27] = "Idle"
;ENDPROC
;WRITELIB LibName RenSetup.MainActPtEditor
;RELEASE PROCS RenSetup.MainActPtEditor
;
;
;PROC RenSetup.MainActPtWindow()
;  ARRAY RenMainActPt[21]
;
;  RenMainActPt[01] = "ArriveWindow"            ;;; Window
;  RenMainActPt[02] = "DepartWindow"
;  RenMainActPt[03] = "Options"
;  RenMainActPt[04] = "ShowPullDown"
;  RenMainActPt[05] = "HelpAction"                    ;;; All
;  RenMainActPt[06] = "MouseLeft"
;  RenMainActPt[07] = "MouseRight"
;  RenMainActPt[08] = "MouseDoubleLeft"
;  RenMainActPt[09] = "MouseDoubleRight"
;  RenMainActPt[10] = "KeyAll"
;  RenMainActPt[11] = "MouseAll"
;  RenMainActPt[12] = "MouseDown"
;  RenMainActPt[13] = "MouseUp"
;  RenMainActPt[14] = "MouseMove"
;  RenMainActPt[15] = "MouseAuto"
;  RenMainActPt[16] = "MessageAll"
;  RenMainActPt[17] = "MessageClose"
;  RenMainActPt[18] = "MessageMaximize"
;  RenMainActPt[19] = "MessageMenuSelect"
;  RenMainActPt[20] = "MessageMenuKey"
;  RenMainActPt[21] = "Idle"
;ENDPROC
;WRITELIB LibName RenSetup.MainActPtWindow
;RELEASE PROCS RenSetup.MainActPtWindow


PROC RenSetup.GetConfig(RenDyn)
  PRIVATE RenFile, RenStr, RenTemp1, RenTemp2, RenEdLen, RenLoc

  RenDyn["EDITOR"] = ""
  RenDyn["SCLOC"]  = 0

  RenFile = PRIVDIR() + "Renconf.txt"

  IF ISFILE(RenFile) THEN
    FILEREAD RenFile TO RenStr
    IF MATCH(RenStr, "..[EDITOR=..]..", RenTemp1, RenEditor, RenTemp2) THEN
      RenDyn["EDITOR"] = RenEditor

      RenEdLen = LEN(RenEditor)

      RenLoc = SEARCH(" *", RenEditor)
      IF RenLoc > 0 THEN
        IF RenLoc + 1 = LEN(RenEditor) THEN
          RenDyn["SCLOC"] = RenLoc + 1
        ELSE
          IF SUBSTR(RenEditor, RenLoc + 2, 1) = " " THEN
            RenDyn["SCLOC"] = RenLoc + 1
          ENDIF
        ENDIF
      ENDIF

    ENDIF
  ENDIF
ENDPROC
WRITELIB LibName RenSetup.GetConfig
RELEASE PROCS RenSetup.GetConfig


RELEASE VARS LibName

