; pxsys.sc
;
; Datadictionary and system maintenance for Paradox developers
; pre-release
;
; (c)      Ajan Tietotekniikka ATT Oy 1992
; Author:  Tapani Isoranta (Compuserve ID: 100021,3453)
;
; Address: Itlahdenkatu 21 A
;          00210 Helsinki
;          Finland
;
; Tel:     358-0-671699     in Finland (90)-671699
; Fax:     358-0-671761                (90)-671761
;
; ATT Oy is specialized in Paradox ja FoxPro developing
;
; Notes:   Use those table and field procedures with extremely care
;          You must understand the filter-table
;          !!! FIELD procedure is activated by every field in filter-table
;


;------------------------------------------------ create a library
;

if isfile("Pxsys.lib") then
  run "del Pxsys.lib"
endif

libname = "Pxsys"

createlib libname


;------------------------------------------------ px.variables ()
; PX-system variables
;
; Parameters:
;
; Globals:
;
; Return:
;


proc px.variables ()

  g_basetbl      = "Pxtbl"
  g_fldtbl       = "Pxfld"
  g_objtbl       = "Pxobj"
  g_setuptbl     = "Pxsetup"
  g_updtbl       = "Pxfld"
  g_errtbl       = "Pxerr"
  g_port         = "Lpt1"
  g_setup        = ""
  g_pagelen      = 64
  g_dir          = sdir()

  g_flashtbl     = "Pxlogo"
  g_mnutbl       = "Pxmnu"
  g_hlptbl       = "Pxhlp"
  g_hlptxttbl    = "Pxhlptxt"
  g_hlpreftbl    = "Pxhlpref"
  g_topicfld     = "Topic"
  g_stopicfld    = "Stopic"
  g_topictmpfld  = "Topictmp"
  g_stopictmpfld = "Stopictmp"
  g_rowfld       = "Row"

  g_screenproc   = "prg.background"

endproc

writelib libname px.variables


;----------------------------------------------- prg.background ()
; Background screen
; Created by snipper
;
; Parameters:
;
; Globals:    g_updtbl,
;             g_dir
;             g_port
;
; Return:
;


proc prg.background ()

  private titlecolor,
          backcolor,
          blackcolor,
          column,
          str

  titlecolor = 95
  backcolor  = 23
  blackcolor = 15

  canvas off
  @2,0
  clear eos

  setmargin off
  @2,0
text
                     PXsystem for Paradox Developers             (c) ATT Oy 1992
 TableDateĿ ObjDescriptionĿ
           ۳۳
 DescriptionĴ ۳۳
 ۳ ۳۳
 ۳ ۳۳
 ۳ ۳۳
 ۳ ۳۳
 ۳ ۳۳
  
FieldTypeDescriptionSPicRLookupJPFNLwHwDefĿ
۳۳۳۳۳۳۳۳۳۳۳۳۳۳
۳۳۳۳۳۳۳۳۳۳۳۳۳۳
۳۳۳۳۳۳۳۳۳۳۳۳۳۳
۳۳۳۳۳۳۳۳۳۳۳۳۳۳
۳۳۳۳۳۳۳۳۳۳۳۳۳۳
۳۳۳۳۳۳۳۳۳۳۳۳۳۳
۳۳۳۳۳۳۳۳۳۳۳۳۳۳
۳۳۳۳۳۳۳۳۳۳۳۳۳۳
۳۳۳۳۳۳۳۳۳۳۳۳۳۳
۳۳۳۳۳۳۳۳۳۳۳۳۳۳


endtext


  paintcanvas attribute titlecolor   2,0,2,79
  paintcanvas attribute backcolor    3,0,24,79
  paintcanvas attribute blackcolor   24,0,24,79
  style attribute blackcolor
  str = g_dir+"   "+g_updtbl+"   "+g_port
  column = int((79 - len(str))/2)
  @24,column ?? str
  style

  canvas on

endproc

writelib libname prg.background


;------------------------------------------------ px.setfiltertbl ()
; Temporary filtertbl
;
; Parameters:
;
; Globals:    g_updtbl
;             g_moment
;
; Return      g_updtbl
;


proc px.setfiltertbl ()

  private newtbl

  msg.dialog ("Filtertable: ","A8","","",g_updtbl)

  if istable(retval) then
    newtbl = retval
    pro.password (newtbl)
    if retval = False then
      return False
    else
      g_updtbl = newtbl
    endif
  else
    if not isblank(retval) then
      beep message "Unknown table!" sleep g_moment
    endif
  endif

endproc

writelib libname px.setfiltertbl


;------------------------------------------------ px.errfile ()
; Look errorfile
;
; Parameters: choice, selected function
;
; Globals:    g_errtbl
;             g_errfile
;
; Return :
;


proc px.errfile (choice)

  if not istable (g_errtbl) then
    create g_errtbl "Text" : "A80"
  endif

  switch
    case choice = "Look" :
      if not isfile (g_errfile) then
        view g_errtbl
      else
        {Tools} {ExportImport} {Import} {Ascii} {AppendDelimited}
        typein g_errfile enter select g_errtbl
        execute "run norefresh \"del \"+g_errfile+\"\""
      endif
      prompt g_escape,""
      wait table until "Esc"
      do_it!
      clearimage
    case choice = "Print" :
      out.selectdevice ()
      if isblank(retval) then
        return
      endif
      out.print (g_errtbl,"R")
    case choice = "Delete" :
      msg.areyousure ()
      if retval then
        execute "run norefresh \"del \"+g_errfile+\"\""
        empty g_errtbl
      endif
  endswitch

endproc

writelib libname px.errfile


;------------------------------------------------ px.sysreport ()
; Data dictionary and documentation reports
;
; Parameters: choice, selected report
;
; Globals:    g_fldtbl
;
; Return:
;


proc px.report (choice)

  private choice

  switch
    case choice = "Short datadict" :
      out.selectdevice ()
      if not isblank(retval) then
        out.print (g_fldtbl,2)
      endif
    case choice = "Long datadict" :
      out.selectdevice ()
      if not isblank(retval) then
        out.print (g_fldtbl,1)
      endif
    case choice = "Table docum" :
      px.printall ()
  endswitch

endproc

writelib libname px.report


;----------------------------------------------- px.printall ()
; Print table documentation
;
; Parameters:
;
; Globals:    g_waiting
;             g_updtbl
;             g_waitblank
;
; Return:
;


proc px.printall ()

  private i,
          r,
          tbl,
          firsttime

  out.selectdevice ()
  if isblank(retval) then
    return
  endif

  msg.waiting (g_waiting)

  {Ask} select g_updtbl
  moveto [Table] Check Do_It!
  rename "Answer" "List"
  clearall

  Query

   List | Table    |
        | Check _t |

   Pxfld | Table | Row   | Field | Type  | Description |
         | _t    | Check | Check | Check | Check       |

  Endquery

  do_it!
  clearall
  rename "Answer" "Entry001"


  Query

   List | Table    |
        | Check _t |

   Pxdoc | Table | Row   | Text  |
         | _t    | Check | Check |

  Endquery

  do_it!
  clearall
  rename "Answer" "Entry002"


  Query

   List | Table    |
        | Check _t |

   Pxobj | Table | Object | Description |
         | _t    | Check  | Check       |

  Endquery

  do_it!
  clearall
  rename "Answer" "Entry003"

  create "Entry004" like "Pxdummy"
  copyreport "Pxdummy" 1 "Entry004" 1

  view   "Entry004"
  view   "Entry003"
  view   "Entry002"
  coedit "Entry001"

  i = 0

  while True

    tbl = [Table]

    moveto "Entry004"
    i = i + 1
    down
    [Table] = tbl
    [Row]   = i

    moveto "Entry002"
    firsttime = True
    while tbl = [Table]
      copytoarray r
      if not isblank(r["Text"]) then
        moveto "Entry004"
        if firsttime then
          firsttime = False
          i = i + 1
          down
          [Table] = tbl
          [Row]   = i
        endif
        i = i + 1
        down
        [Table] = tbl
        [Row]   = i
        [Text]  = r["Text"]

        moveto "Entry002"
      endif
      if atlast () then quitloop endif
      skip
    endwhile

    moveto "Entry003"
    firsttime = True
    while tbl = [Table]
      copytoarray r
      if not isblank(r["Object"]) then
        moveto "Entry004"
        if firsttime then
          firsttime = False
          i = i + 1
          down
          [Table] = tbl
          [Row]   = i
        endif
        i = i + 1
        down
        [Table]   = tbl
        [Row]     = i
        [Text]    = r["Object"] + spaces(26-len(r["Object"])) +
                    r["Description"]
        moveto "Entry003"
      endif
      if atlast () then quitloop endif
      skip
    endwhile

    moveto "Entry001"
    firsttime = True
    while tbl = [Table]
      copytoarray r
      if not isblank(r["Field"]) then
        moveto "Entry004"
        if firsttime then
          firsttime = False
          i = i + 1
          down
          [Table] = tbl
          [Row]   = i
        endif
        i = i + 1
        down
        [Table] = tbl
        [Row]   = i
        [Text]  = r["Field"] + spaces(26-len(r["Field"])) +
                  r["Type"]  + spaces(6-len(r["Type"])) +
                  r["Description"]
        moveto "Entry001"
      endif
      if atlast () then quitloop endif
      skip
    endwhile

    if [Table] = tbl and atlast () then
      quitloop
    endif

  endwhile

  do_it!
  clearall

  msg.waiting (g_waitblank)

  out.print ("Entry004",1)

endproc

writelib libname px.printall


;----------------------------------------------- px.setupcode ()
; Set printer setup code
;
; Parameters:
;
; Globals:
;
; Return:
;


proc px.setupcode ()

  private g_popitem,
          g_popval,
          g_popwidth,
          g_popind,
          setupcode

  pop.setpopup ("Pxsetup","Printer","Setup")

  pop.popup (4,15,1)
  if not isblank(retval) then
    setupcode = g_popval[retval]
  else
    setupcode = ""
  endif

  out.printsetup (setupcode)

endproc

writelib libname px.setupcode


;------------------------------------------------ px.exeproc ()
; Database scanning based on table or field or any procedure
;
; Parameters: proctype, procedure type  Field or Table
;             procedur, procedure name
;
; Globals:    g_updtbl
;
; Return:
;
; Notice:     table procedure activates according to "Table"-field
;             field procedure activates according to "Table"-field and "Field"-field
;             !!! needs better algorithm for future versions
;             "Table"-field excpected for table-procedures
;             "Table"- and Field"-field excpected for fielf procedures
;

proc px.exeproc (proctype,procedur,warning)

  private tbl,
          fld,
          g_firsttime,
          g_popitem,
          g_popind,
          g_popwidth

  if not isblank(warning) then
    msg.dialog ("Are you extremely sure now Y/N ?","A2","{Y,N}","","")
    if retval <> "Y" then
      return
    endif
  endif
  if isblank(procedur) then
    msg.dialog ("Procedure: ","A60","","","")
    if isblank(retval) then
      return
    else
      procedur = retval
      retval = True
    endif
  endif

  view g_updtbl

  g_firsttime = True

  while not isempty(g_updtbl)

    tbl  = [Table]     ; crash, if table-field not found()... needs better algorithm

    pro.password (tbl)
    if retval = True then
      if proctype = "Field" then
        fld  = [Field] ; crash, if field-field not found()... needs better algorithm
      endif
      msg.message ("Processing: "+tbl,"","")
      execute procedur
      moveto g_updtbl
      if retval = False then
        quitloop
      endif
      g_firsttime = False
    endif

    if atlast () then
      quitloop
    endif
    skip
    if proctype = "Table" then
      while tbl = [Table] and not atlast ()
        skip
      endwhile
      if tbl = [Table] then quitloop endif
    endif

  endwhile

  if sysmode() <> "Main" then do_it! endif
  clearall

endproc

writelib libname px.exeproc


;------------------------------------------------ px.editsetuptbl ()
; Update printer setups
;
; Parameters:
;
; Globals:    g_setuptbl
;             g_escape
;
; Return:
;


proc px.editsetuptbl ()

  coedit g_setuptbl

  prompt g_escape,""
  wait table
    until "Esc"

  do_it!
  clearimage

endproc

writelib libname px.editsetuptbl


;------------------------------------------------ px.editbasetbl ()
; Update base table
;
; Parameters:
;
; Globals:    g_error
;             g_normalmode
;             g_basetbl
;             g_help
;             g_query
;             g_search
;             g_zoom
;             g_nxtwin
;             g_swifrm
;             g_fldedit
;             g_delete
;             g_insert
;             g_escape
;             g_ctrlpgup
;             g_ctrlpgdn
;
; Return:
;


proc px.editbasetbl ()

  private g_query,
          g_look,
          g_mode,
          g_mastertbl,
          g_masterfrm,
          g_mkeyfld,
          g_bkeyfld,
          g_memotbl,
          g_update

  g_query     = "F3=Query   "
  g_look      = "F4=LookObj "
  g_mode      = g_normalmode
  g_mastertbl = g_basetbl
  g_masterfrm = "1"
  g_mkeyfld   = "Table"
  g_bkeyfld   = "Table"
  g_memotbl   = "Pxdoc"    ; or "Pxfld"
  g_update    = True

  retval = ""
  edit g_mastertbl
  pickform g_masterfrm
  if retval = g_error then
    return False
  endif

  while true
    prompt g_help   + g_query  + g_look   + g_search + g_zoom + g_swifrm,
           g_nxtwin + g_delete + g_insert + g_escape

    key.wait ()

    switch
      case retval = "Esc" :
        do_it!
        clearall
        quitloop
      case retval = "FieldView"  :
        key.fldedit ()
      case retval = "F1" :
        hlp.hlpsystem ()
      case retval = "Del" :
        key.delrec ()
      case retval = "Ins" :
        key.insrec ()
      case retval = "Home" :
        key.homerec ()
      case retval = "End" :
        key.endrec ()
      case retval = "PgDn" :
        key.nextrec ()
      case retval = "PgUp" :
        key.prevrec ()
      case retval = g_ctrlpgdn :
        key.nextrec ()
      case retval = g_ctrlpgup :
        key.prevrec ()
      case retval = "F1" :
        hlp.hlpsystem ()
      case retval = "F2" :
      case retval = "F3" :
        if isruntime () then
          msg.message
          ("Not a possible menuchoice with runtime-Paradox!",1,g_moment)
          loop
        endif
        do_it!
        clearall
        px.queryfldtbl ()
        coedit g_mastertbl
        pickform g_masterfrm
        canvas on
      case retval = "F4" :
        px.lookobject ()
      case retval = "F5"  :
        key.search ()
      case retval = "F6"  :
        key.zoomdetails ()
      case retval = "F7"  :
        key.switchfrm ()
      case retval = "F8"  :
      case retval = "F9"  :
      case retval = "F10" :
      case retval = "ReverseTab" :
        switch
          case table() = "Pxobj" :
            g_memotbl = "Pxdoc"
          case table() = "Pxtbl" :
            g_memotbl = "Pxfld"
        endswitch
        key.prevtbl ()
      case retval = "Tab" :
        switch
          case table() = "Pxtbl" :
            g_memotbl = "Pxdoc"
          case table() = "Pxobj" :
            g_memotbl = "Pxfld"
        endswitch
        key.nexttbl ()
      case retval = "Zoom" :
        key.locaterec ()
      otherwise:
        key.anykey(retval)
    endswitch
  endwhile

endproc

writelib libname  px.editbasetbl


;------------------------------------------------ px.lookobject ()
; Look table'c contents or objects
;
; Parameters :
;
; Globals:
;
; Return:
;


proc px.lookobject ()

  private objtbl,
          tbl,
          frm

  frm    = isformview()
  objtbl = table()
  tbl    = [Table]

  pro.password (tbl)
  if retval = False then
    return False
  endif

  switch
    case objtbl = "Pxobj" :
      switch
        case search("R",[Object])> 0 :
          do_it!
          tbl.editrpt ()
        case search("F",[Object])> 0 :
          do_it!
          tbl.editfrm ()
      endswitch
    otherwise:
      do_it!
      coedit tbl
      firstshow
      prompt g_escape,""
      wait table until "Esc"
      do_it!
      clearimage
  endswitch

  moveto g_mastertbl
  if frm <> isformview() then
    formkey
  endif
  coeditkey
  moveto objtbl

endproc

writelib libname px.lookobject


;------------------------------------------------ px.queryfldtbl ()
; Query field definition table
;
; Parameters:
;
; Globals:    g_fldtbl
;
; Return:
;


proc px.queryfldtbl ()

  private keyshr,
          g_doit,
          g_edit,
          g_escape,
          alt_f8,
          f8

  g_doit   = "F2=Do_it! "
  g_edit   = "F9=Edit "
  g_menu   = "F10=Menu "
  g_escape = "Alt_F8=Escape "
  alt_f8   = -111
  f8       = -66

  {Ask} select g_fldtbl
  echo normal

  while true

    prompt g_escape + g_doit + g_edit + g_menu,
            window ()

    keychr = getchar()

    switch
      case sysmode () = "Main" and table () = g_fldtbl and keychr = f8 :
        loop
      case sysmode () = "Main" and keychr = alt_f8 :
        canvas off
        clearall
        quitloop
      otherwise :
        keypress keychr
    endswitch

  endwhile

  echo off

endproc

writelib libname px.queryfldtbl


;----------------------------------------------- px.readtblstruct ()
; Read directory's table structures and field definitions
;
; Parameters:
;
; Globals:    g_basetbl
;             g_fldtbl
;             g_objtbl
;
; Return:
;


proc px.readtblstruct ()

  private tbl,
          keyflds,
          fldcount,
          cind,
          ind

  msg.dialog ("Overwrite Basetables Y/N ","A2","!","","")
  if retval <> "Y" then
    return
  endif

  empty g_basetbl
  empty g_fldtbl
  empty g_objtbl

  px.readtbllist ()

  view "List"

  scan

    tbl = [Name]
    pro.password (tbl)
    if retval = False then
      loop
    endif

    msg.message ("Processing: "+tbl,"","")

    coedit g_basetbl
    ins
    [Table]   = tbl
    [Date]    = today()
    do_it!
    clearimage

    retval    = ""
    keyflds   = nkeyfields(tbl)
    if retval = g_error then loop endif

    px.writefamilyobj (tbl)
    if retval = False then loop endif

    px.writefieldattrib (tbl)

    moveto "List"

  endscan

  do_it!
  clearall

endproc

writelib libname px.readtblstruct


;----------------------------------------------- px.readtbllist ()
; Read directory's tables
;
; Paremeters:
;
; Globals:
;
; Return:     List-table, inventory of directory tables
;


proc px.readtbllist ()

  {Tools} {Info} {Inventory} {Tables} Enter
  moveto [List->Name]
  coeditkey

  locate "Struct"
  if retval  then del endif

  locate "Family"
  if retval  then del endif

  locate "Changed"
  if retval  then del endif

  locate "Problems"
  if retval  then del endif

  locate "List"
  if retval  then del endif

  locate "Answer"
  if retval  then del endif

  locate "Deleted"
  if retval  then del endif

  locate "Keyviol"
  if retval  then del endif

  locate pattern "Entry"+".."
  while retval
    del
    locate pattern "Entry" + ".."
  endwhile

  locate pattern "Px" + ".."
  while retval
    del
    locate pattern "Px" + ".."
  endwhile

  do_it!
  clearimage

endproc

writelib libname px.readtbllist


;------------------------------------------------ px.writefamilyobj ()
; First read and then write family objects
;
; Parameters: tbl, prosessed table
;
; Globals:    g_objtbl
;             g_error
;
; Return:     retval, True or False
;


proc px.writefamilyobj (tbl)

  private i,       ; count index
          obj,     ; family objects, reports and forms
          objdes   ; object descriptions

  array obj[15]
  array objdes[15]

  retval = ""
  Menu {Tools} {Info} {Family} select tbl
  moveto [Name]
  if retval = g_error then
    return False
  endif

  i = 0

  scan
    switch
      case substr([Name],1,6) = "Report" :
        i = i + 1
        obj[i] = substr([Name],8,len([Name])-7)
        if len(obj[i]) > 1 then
          obj[i] = substr(obj[i],2,len(obj[i]))
        endif
        Menu {Report} {Change} select tbl
        select obj[i]
        objdes[i] = menuchoice()
        Esc Esc Esc Esc Esc
        obj[i] = "R"+obj[i]
      case substr([Name],1,4) = "Form" :
        i = i + 1
        obj[i] = substr([Name],6,len([Name])-5)
        if len(obj[i]) > 1 then
          obj[i] = substr(obj[i],2,len(obj[i]))
        endif
        Menu {Forms} {Change} select tbl
        select obj[i]
        objdes[i] = menuchoice()
        Esc Esc Esc Esc Esc
        obj[i] = "F"+obj[i]
    endswitch
  endscan

  clearimage

  retval = ""
  coedit g_objtbl
  if retval = g_error then
    return False
  endif

  for ind from 1 to i
    ins
    [Table]       = tbl
    [Object]      = obj[ind]
    [Description] = objdes[ind]
  endfor

  do_it!
  clearimage

  return True

endproc

writelib libname px.writefamilyobj


;----------------------------------------------- px.writefieldattrib ()
; Fist read and then write field attributes
;
; Parameters : tbl, processed table
;
; Globals:     g_error
;              g_fldtbl
;
; Return:      retval, True or False
;


proc px.writefieldattrib (tbl)

  private fldnum,
          fldcount,
          fld,
          fldtype,
          speedup,
          required,
          picture,
          lookup,
          justall,
          privhelp,
          nformat,
          decimals,
          lowval,
          highval,
          default

  retval = ""
  fldcount = nfields(tbl)
  if retval = g_error then
    return False
  endif

  retval = ""
  Menu {Tools} {Info} {Family}
  select tbl moveto [Name]
  if retval = g_error then
    menu
    return False
  endif

  retval = ""
  edit tbl
  if retval = g_error then
    return False
  endif

  px.initfieldarrays (fldcount)

  moveto [#]
  for fldnum from 1 to fldcount
    right
    fld[fldnum] = field ()
    if keyflds < fldnum then
      fldtype[fldnum] = fieldtype ()
    else
      fldtype[fldnum] = fieldtype () + "*"
    endif

    px.readfieldattrib (fldnum)

    moveto tbl
  endfor

  do_it!
  clearimage
  moveto "Family" clearimage

  retval = ""
  coedit g_fldtbl
  if retval = g_error then
    return False
  endif

  for fldnum from 1 to fldcount
    ins
    [Table]    = tbl
    [Row]      = fldnum
    [Field]    = fld[fldnum]
    [Type]     = fldtype[fldnum]
    [Speedup]  = speedup[fldnum]
    [Required] = required[fldnum]
    [Picture]  = picture[fldnum]
    [Lookup]   = lookup[fldnum]
    [Justall]  = justall[fldnum]
    [Privhelp] = privhelp[fldnum]
    [NFormat]  = nformat[fldnum]
    [Decimals] = decimals[fldnum]
    [Lowval]   = lowval[fldnum]
    [Highval]  = highval[fldnum]
    [Default]  = default[fldnum]
  endfor

  do_it!
  clearimage

  return True

endproc

writelib libname px.writefieldattrib


;----------------------------------------------- px.initfieldarrays ()
; Init field arrays
;
; Parameters : fldcount, field count of table
;
; Globals:
;
; Return:      array definitions
;


proc px.initfieldarrays (fldcount)

  array fld[fldcount]
  array fldtype[fldcount]
  array speedup[fldcount]
  array required[fldcount]
  array picture[fldcount]
  array lookup[fldcount]
  array justall[fldcount]
  array privhelp[fldcount]
  array nformat[fldcount]
  array decimals[fldcount]
  array lowval[fldcount]
  array highval[fldcount]
  array default[fldcount]

endproc

writelib libname px.initfieldarrays


;------------------------------------------------ px.readfieldattrib ()
; Read field definitions on edit-mode
;
; Parameters: fldnum, field number
;
; Globals:    array fld
;             array nformat
;             array decimals
;             array lowval
;             array highval
;             array default
;             array lookup
;             array picture
;             array required
;             array lookup
;             array justall
;             array privhelp
;             g_errfile
;             g_moment
;
; Return:
;


proc px.readfieldattrib (fldnum)

  private errmess

  if fieldtype() = "N" or fieldtype () = "$" then
    Menu {Image} {Format} Enter
    nformat[fldnum]  = menuchoice() enter
    decimals[fldnum] = menuchoice() enter
  else
    nformat[fldnum]  = ""
    decimals[fldnum] = ""
  endif

  Menu {ValCheck} {Define} Enter {LowValue}
  lowval[fldnum] = menuchoice () Esc

  {HighValue}   highval[fldnum]  = menuchoice () Esc
  {Default}     default[fldnum]  = menuchoice () Esc
  {TableLookup} lookup[fldnum]   = menuchoice () Esc
  {Picture}     picture[fldnum]  = menuchoice () Esc
  {Required}    required[fldnum] = menuchoice () Esc

  if lookup[fldnum] <> "" then
    {TableLookup}  enter
    if menuchoice () <> "JustCurrentField" and
       menuchoice () <> "AllCorrespondingFields" then
      errmess = "Lookup error: " + lookup[fldnum] + " " +
                table () + "  " + field ()
      msg.message (errmess,1,g_moment)
      print file g_errfile format("w50",errmess) + " " +
                 " " + format("d3", today()) + "\n"
      justall[fldnum]    = ""
      privhelp[fldnum]   = ""
      lookup[fldnum]     = ""
      esc esc esc esc
    else
      justall[fldnum]  = menuchoice () enter
      privhelp[fldnum] = menuchoice () enter
    endif
  else
    justall[fldnum]  = ""
    privhelp[fldnum] = ""
    esc esc esc esc
  endif

  esc esc esc esc

  moveto "Family"
  locate "Speedup for "+fld[fldnum]+" [Maintained]"
  if retval then
    speedup[fldnum] = "*"
  else
    speedup[fldnum] = ""
  endif

endproc

writelib libname px.readfieldattrib


;------------------------------------------------ tbl.restructure
; Create a new table or restructure old table
;
; Parameters: restrtype, insert field or modify field name
;
; Globals:    g_updtbl
;             g_error
;             tbl
;
; Return:     retval, True or False
;


proc tbl.restructure (restrtype)

  private i,
          is,
          fldcount

  while [Table] = tbl
    fldcount = [Row]
    execute "copytoarray dbdef"+strval(fldcount)+""
    if atlast () then quitloop endif
    skip
  endwhile

  if not istable (tbl) then
    Menu {Create} typein tbl enter
    for i from 1 to fldcount
      is = strval(i)
      execute "typein dbdef"+is+"[\"Field\"]" enter
      execute "typein dbdef"+is+"[\"Type\"]" enter
    endfor
    do_it!
  else
    retval = ""
    Menu {Modify} {Restructure}
    select tbl
    if retval = g_error then
      menu
      return False
    else
      if menuchoice () = "Cancel" then
        {OK}
      endif
    endif
    moveto [Field Name]

    for i from 1 to fldcount
      if restrtype = 1 then
        ins
      else
        ctrlbackspace
      endif
      execute "typein dbdef"+strval(i)+"[\"Field\"]"
      enter ctrlbackspace
      execute "typein dbdef"+strval(i)+"[\"Type\"]" enter
      moveto [Field Name]
    endfor

    r = recno ()
    while recno () = r
      del
    endwhile

    retval = ""
    do_it!
    if retval = g_error then
      return False
    endif

    while sysmode () = "Restructure"
      enter
    endwhile

    if table () = "Keyviol" then
      clearimage
    endif

    if table () = "Problems" then
      clearimage
    endif

    clearimage

  endif

  moveto g_updtbl
  if [Table] <> tbl then
    skip -1
  endif

  return True

endproc

writelib libname tbl.restructure


;------------------------------------------------ tbl.tutilityverify ()
; Verify tables, log table status to g_errfile
;
; Parameters:
;
; Globals     g_errtbl
;             tbl
;
; Return      retval, True
;


proc tbl.tutilityverify ()

  if not istable(g_errtbl) then
    create g_errtbl "Text" : "A80"
  endif

  execute "run norefresh \"tutility \"+tbl+\" > NULL\""

  {Tools} {ExportImport} {Import} {Ascii}
  {AppendDelimited} {null.} select g_errtbl
  end
  coeditkey
  del del del del
  up up up up del del
  do_it!
  clearimage

  return True

endproc

writelib libname tbl.tutilityverify


;------------------------------------------------ tbl.tutilityrebuild ()
; Rebuild tables and log status to errfile
;
; Parameters:
;
; Globals:    g_errtbl
;             tbl
;
; Return:     retval, True
;


proc tbl.tutilityrebuild ()

  if not istable(g_errtbl) then
    create g_errtbl "Text" : "A80"
  endif

  execute "run norefresh \"tutility -rebuild \"+tbl+\" > NULL\""
  {Tools} {ExportImport} {Import} {Ascii}
  {AppendDelimited} {null.} select g_errtbl
  end
  coeditkey
  del del del del
  up up up up del del
  do_it!
  clearimage

  return True

endproc

writelib libname tbl.tutilityrebuild


;------------------------------------------------ tbl.checktbl ()
; Check tables, log corrupted tables
;
; Parameters:
;
; Globals:    g_error
;             tbl
;
; Return:     retval, True
;


proc tbl.checktbl ()

  retval = ""
  {Modify} {Restructure}
  select tbl
  if retval = g_error then
  else
    if sysmode () = "Restructure" then
      menu {Cancel}
    endif
  endif

  menu
  return True

endproc

writelib libname tbl.checktbl


;------------------------------------------------ tbl.editfrm ()
; Edit all forms
;
; Parameters:
;
; Globals: g_error
;          g_escape
;          g_save
;          g_menu
;          tbl
;
; Return   retval, True or False
;


proc tbl.editfrm ()

  private i,
          r,
          first,
          cind,
          frm

  if isruntime () then
    msg.message
    ("Not a possible menuchoice with runtime-Paradox!",1,g_moment)
    return False
  endif

  array frm[15]

  retval = ""
  {Forms} {Change}
  select tbl
  if retval = g_error then
    menu
    moveto g_updtbl
    return False
  endif

  first = True
  i     = 0
  r     = menuchoice()

  while True
    if menuchoice () = r and not first then
      quitloop
    else
      first = False
    endif
    i = i + 1
    frm[i] = menuchoice ()
    retval = ""
    select frm[i]
    if retval = g_error then
      i = 1 - 1
    else
      if menuchoice() = "Standard form" then
        i = i - 1
      endif
      esc
    endif
    right
  endwhile

  menu

  for cind from 1 to i
    {Forms} {Change}
    select tbl
    select frm[cind]
    enter
    echo normal
    while true
      prompt  g_escape + g_save + g_menu,
              tbl + " F" + strval(frm[cind]) + "  " + fieldinfo ()
      keychr = getchar()
      keypress keychr
      switch
        case sysmode () = "Main" :
          quitloop
        case keychr = asc("Esc") :
          menu {Cancel} {Yes}
          echo off
          return False
      endswitch
    endwhile
    echo off
  endfor

  return True

endproc

writelib libname tbl.editfrm


;------------------------------------------------ tbl.checkfrm ()
; Check forms, log corrupted
;
; Paremeters:
;
; Globals:    g_error
;             tbl
;
; Return:     retval, True or False
;


proc tbl.checkfrm ()

  private i,
          cind,
          frm,
          first,
          r

  array frm[15]

  retval = ""
  {Forms} {Change}
  select tbl
  if retval = g_error then
    menu
    return False
  endif

  first = True
  i     = 0
  r     = menuchoice()

  while True
    if menuchoice () = r and not first then
      quitloop
    else
      first = False
    endif
    i = i + 1
    frm[i] = menuchoice ()
    retval = ""
    select frm[i]
    if retval = g_error then
      i = i - 1
    else
      esc
    endif
    right
  endwhile

  menu
  return True

endproc

writelib libname tbl.checkfrm


;------------------------------------------------ tbl.editrpt ()
; Edit reports
;
; Parameters:
;
; Globals:    g_error
;             g_escape
;             g_save
;             g_menu
;             g_moment
;             tbl
;
; Return:     retval, True or False
;


proc tbl.editrpt ()

  private i,
          r,
          first,
          cind,
          rep,
          keychr

  if isruntime () then
    msg.message
    ("Not a possible menuchoice with runtime-Paradox!",1,g_moment)
    return False
  endif

  array rep[15]

  retval = ""
  {Report} {Change}
  select tbl
  if retval = g_error then
    menu
    return False
  endif

  i = 0
  r = menuchoice()
  first = true
  while True
    if menuchoice () = r and not first then
      quitloop
    else
      first = False
    endif
    i = i + 1
    rep[i] = menuchoice ()
    retval = ""
    select rep[i]
    if retval = g_error then
      i = i - 1
    else
      if menuchoice() = "Standard report" then
        i = i - 1
      endif
      esc
    endif
    right
  endwhile

  menu

  for cind from 1 to i
    {Report} {Change}
    select tbl
    select rep[cind]
    enter
    if sysmode () <> "Report" then
      menu
      loop
    endif
    echo normal
    while true
      prompt g_escape + g_save + g_menu,
             tbl + " R" + strval(rep[cind]) + "  " +
             bandinfo () + " " + fieldinfo ()
      keychr = getchar()
      keypress keychr
      switch
        case sysmode () = "Main" :
          quitloop
        case keychr = asc("Esc") :
          menu {Cancel} {Yes}
          echo off
          return False
      endswitch
    endwhile
    echo off
  endfor

  return True

endproc

writelib libname tbl.editrpt


;------------------------------------------------ tbl.checkrpt ()
; Check reports
;
; Parameters:
;
; Globals:    g_error
;
; Return:     retval, True or False
;


proc tbl.checkrpt ()

  private i,
          cind,
          rep,
          first,
          r

  array rep[15]

  retval = ""
  {Report} {Change}
  select tbl
  if retval = g_error then
    menu
    return False
  endif

  first = True
  i     = 0
  r     = menuchoice()

  while True
    if menuchoice () = r and not first then
      quitloop
    else
      first = False
    endif
    i = i + 1
    rep[i] = menuchoice ()
    retval = ""
    select rep[i]
    if retval = g_error then
      i = i - 1
    else
      esc
    endif
  endwhile

  menu

  return True

endproc

writelib libname tbl.checkrpt


;------------------------------------------------ tbl.pagelength ()
; Set pagelegth to reports
;
; Parameters:
;
; Globals: g_error
;          g_pagelen
;          tbl
;
; Return:  retval, True or False
;


proc tbl.pagelength ()

  private i,
          ind,
          rep,
          repname

  array rep[14]

  retval = ""
  {Report} {Change}
  select tbl
  if retval = g_error then
    menu
    return False
  endif

  i = 0

  while True
    right
    if menuchoice () = "R" then quitloop endif
    i = i + 1
    rep[i] = menuchoice ()
    retval = ""
    select rep[i]
    if retval = g_error then
      i = i - 1
    else
      esc
    endif
  endwhile

  menu

  for ind from 1 to i
    {Report} {Change}
    select tbl
    select rep[ind]
    repname = menuchoice ()
    enter

    if sysmode () <> "Report" then
      menu
      loop
    endif
    Menu {Setting} {PageLayout}
    {Length}
    g_pagelen = menuchoice ()
    msg.message (tbl+"  "+"  "+rep[ind]+"    "+repname,"","")
    msg.dialog ("Pagelength:","A3","{C,##[#]}","",g_pagelen)
    if not isblank(retval) then
      g_pagelen = retval
    endif
    ctrlbackspace typein g_pagelen enter
    do_it!
  endfor

  return True

endproc

writelib libname tbl.pagelength


;------------------------------------------------ tbl.copyfamily ()
; Copy tablefamilies to other directory
;
; Parameters:
;
; Globals:    g_error
;             g_firsttime
;             tbl
;
; Return:     retval, True or False
;


proc tbl.copyfamily ()

  private targettbl

  if g_firsttime then
    env.checkdir ("")
    if isblank(retval) then
      return False
    else
      targetdir = retval
    endif
    if substr(targetdir,len(targetdir),1) <> "\\" then
      targetdir = targetdir+"\\"
    endif
  endif

  targettbl = targetdir + tbl

  pro.password (targettbl)
  if retval = False then
     return False
  endif

  retval = ""
  {Tools} {Copy} {JustFamily} select tbl
  select targettbl
  if retval = g_error then
    menu
  else
    {Replace}
  endif

  return True

endproc

writelib libname tbl.copyfamily


;------------------------------------------------ tbl.adddata ()
; Add data from tables to another directory tables
;
; Parameters:
;
; Globals:    g_error
;             g_firsttime
;             tbl
;
; Return:     retval, True or False
;


proc tbl.adddata ()

  if g_firsttime then
    env.checkdir ("")
    if isblank(retval) then
      return False
    else
      targetdir = retval
    endif
    if substr(targetdir,len(targetdir),1) <> "\\" then
      targetdir = targetdir+"\\"
    endif
endif

  targettbl = targetdir + tbl

  pro.password (targettbl)
  if retval = False then
     return False
  endif

  retval = ""
  {Tools} {More} {Add} select tbl
  select targettbl
  if retval = g_error then
    menu
  else
    if menuchoice () = "NewEntries" then
      {Update}
    endif
    clearimage
  endif

  return True

endproc

writelib libname tbl.adddata


;------------------------------------------------ tbl.copytbl ()
; Copy tables to other directory
;
; Parameters:
;
; Globals:    g_error
;             g_firsttime
;             tbl
;
; Return:     retval, True or False
;


proc tbl.copytbl ()

  if g_firsttime then
    env.checkdir ("")
    if isblank(retval) then
      return False
    else
      targetdir = retval
    endif
    if substr(targetdir,len(targetdir),1) <> "\\" then
      targetdir = targetdir+"\\"
    endif
  endif

  targettbl = targetdir + tbl

  pro.password (targettbl)
  if retval = False then
     return False
  endif

  copy tbl targettbl
  menu

  return True

endproc

writelib libname tbl.copytbl


;------------------------------------------------ tbl.emptytbl ()
; Empty tables
;
; Parameters:
;
; Globals:    g_error
;             g_firsttime
;             tbl
;
; Return:     retval, True or False
;


proc tbl.emptytbl ()

  if g_firsttime then
    env.checkdir ("")
    if isblank(retval) then
      return False
    else
      targetdir = retval
    endif
    if substr(targetdir,len(targetdir),1) <> "\\" then
      targetdir = targetdir+"\\"
    endif
  endif

  targettbl = targetdir + tbl

  pro.password (targettbl)
  if retval = False then
     return False
  endif

  msg.dialog ("Empty "+upper(targettbl)+"  Y/N :","A2","!","","")
  switch
    case retval = "Y" :
      empty targettbl
    case isblank(retval) :
      return False
  endswitch

  return True

endproc

writelib libname tbl.emptytbl


;----------------------------------------------- fld.createfielddef ()
; Define field attributes
;
; Parameters:
;
; Globals:    g_error
;             g_errfile
;             g_moment
;             g_updtbl
;             tbl
;             fld
;
; Return:     retval, True
;


proc fld.createfielddef ()

  private errmess,
          flddef

  if isfile(tbl+".set") then
    retval = ""
    Menu {Tools} {Delete} {KeepSet} select tbl
    if retval = g_error then
      menu
      moveto g_updtbl
      return
    endif
  endif

  if isfile(tbl+".val") then
    retval = ""
    Menu {Tools} {Delete} {ValCheck} select tbl
    if retval = g_error then
      menu
      moveto g_updtbl
      return
    endif
  endif

  edit tbl
  moveto g_updtbl

  while [Table] = tbl
    copytoarray flddef
    fld = flddef["Field"]
    moveto tbl
    moveto field fld
    if (fieldtype() = "N" or fieldtype() = "$") then
      if flddef["Nformat"] <> "" and flddef["Decimals"] <> "" then
        Menu {Image} {Format} Enter
        typein flddef["Nformat"]  ctrlbackspace
        typein flddef["Decimals"]
        enter
        Menu {Image} {Keepset}
      endif
    endif
    Menu {ValCheck} {Define} Enter {LowValue}
    typein flddef["Lowval"] enter
    Menu {ValCheck} {Define} Enter {HighValue}
    typein flddef["Highval"] enter
    Menu {ValCheck} {Define} Enter {Default}
    typein flddef["Default"]  enter
    Menu {ValCheck} {Define} Enter {Picture}
    typein flddef["Picture"] enter
    Menu {ValCheck} {Define} Enter {Required}
    typein flddef["Required"]
    if not isblank(flddef["Lookup"]) then
      Menu {ValCheck} {Define} enter {TableLookup}
      typein flddef["Lookup"] Enter
      if menuchoice () <> "JustCurrentField" and
        menuchoice () <> "AllCorrespondingFields" then
        errmess = "Wrong lookup-table: "+menuchoice () + " -> " + table() + "  " + field ()
        msg.message (errmess,1,g_moment)
        print file g_errfile format("w50", errmess) + " " + " " + format("d3", today()) + "\n"
        esc esc esc esc esc
      else
        typein flddef["Justall"]
        typein flddef["Privhelp"]
      endif
    endif
    moveto g_updtbl
    if atlast() then quitloop endif
    skip
  endwhile

  moveto tbl
  do_it!
  clearimage

  moveto g_updtbl
  if [Table] <> tbl then
    skip -1
  endif
  return True

endproc

writelib libname fld.createfielddef


;----------------------------------------------- fld.createsecindex ()
; Create secondary index
;
; Parameters:
;
; Globals:    g_updtbl
;             tbl
;             fld
;
; Return:     retval, True
;


proc fld.createsecindex ()

  if not isblank([Speedup]) then
    Menu {Tools} {Delete} {QuerySpeed} select tbl
    index maintained tbl on fld
  endif

  moveto g_updtbl
  return True

endproc

writelib libname fld.createsecindex


;!***************************************************************
;! Here are some examples of field procedures
;! put the procedures to pxmnu-table
;!***************************************************************

;------------------------------------------------ fld.firstnol ()
; Write first nils to A-type fields
;
; Parameters:
;
; Globals:    g_error
;             tbl
;             fld
;
; Return:     retval, True
;


proc fld.firstnol ()

  private fldlen

  retval = ""
  coedit tbl
  if retval = g_error then
    return
  endif
  retval = ""
  moveto field fld
  if retval = g_error then
    do_it!
    clearimage
    return
  endif

  if numval ([]) = "Error" then
    return True
  endif

  fldlen = numval(substr(fieldtype(),2,len(fieldtype())-1))

  scan
    if not isblank([]) then
      [] = fill ("0",fldlen-len([]))+[]
    endif
  endscan

  do_it!
  clearimage

  return True

endproc

writelib libname fld.firstnol


;------------------------------------------------ fld.upper ()
; Upper to field values
;
; Parameters:
;
; Globals:    g_error,
;             tbl
;             fld
;
; Return:     retval, True
;


proc fld.upper ()

  retval = ""
  coedit tbl
  if retval = g_error then
    return
  endif
  retval = ""
  moveto field fld
  if retval = g_error then
    do_it!
    clearimage
    return
  endif

  scan
    [] = upper([])
  endscan

  do_it!
  clearimage

  return True

endproc

writelib libname fld.upper


;------------------------------------------------ fld.lower ()
; Lower field values
;
; Parameters:
;
; Globals:    g_error
;             tbl
;             fld
;
; Return:     retval, True
;


proc fld.lower ()

  retval = ""
  coedit tbl
  if retval = g_error then
    return
  endif

  retval = ""
  moveto field fld
  if retval = g_error then
    do_it!
    clearimage
    return
  endif

  scan
    [] = lower([])
  endscan

  do_it!
  clearimage

  return True

endproc

writelib libname fld.lower


;------------------------------------------------ fld.capletter ()
; Capitalize field words
;
; Parameters:
;
; Globals:    g_error
;             tbl
;             fld
;
; Return:     retval, True
;


proc fld.capletter ()

  private dta

  retval = ""
  coedit tbl
  if retval = g_error then
    return
  endif

  retval = ""
  moveto field fld
  if retval = g_error then
    do_it!
    clearimage
    return
  endif

  scan
    dta = []
    ctrlbackspace
    typein format("cc",lower(dta))
  endscan

  do_it!
  clearimage

  return True

endproc

writelib libname fld.capletter
