;**********************************************************************
;(c) Copyright 1992, Blue Pearl Software, Mountain View, CA

;You may use this code for any purpose as long as it is not sold or
;included in any software toolkit with other tools and as long as
;the copyright notice is not removed.

;Enjoy... David Gassner, CIS:72077,2207; Tel: (415) 968-0545
;**********************************************************************

Message "Creating PickRec library . . ."
libname = "Pickrec"
Createlib libname

;**********************************************************************
;Name:		PickRecord
;Purpose:	Use a table in Table view as a picklist

;Calls:		PickRecord.eh (event handler from wait record)
;		PickRecord.hl (Redraws hilite bar on current record)

;Arguments:	win.t - Title of window.  If blank, defaults to table name.
;		prompt.txt - Text of prompt on bottom line; if blank, has default
;		return.field - name of field to return
;		win.height - Window height
;              	win.width - Window width
;              	win.row - Window Top row
;              	win.col - Window Top column

;Returns:	Value of field indicated by return.field if selection is made
;		Blank ("") if cancelled by Esc or Mouse click off window

;Notes:		Cursor is left in selected record and returned field at
;		end of proc
;----------------------------------------------------------------------

Proc PickRecord(win.t,		;Window title
              prompt.txt,	;Prompt text
              return.field,	;Field to return
              win.height,	;Window height
              win.width,	;Window width
              win.row,		;Window Top row
              win.col)		;Window Top column

  private win.a,		;Attributes array
          win.h,		;Window handle
          keys.a,		;Cursor movement keys array
          searchtxt,		;Search value
          startcol,		;Cursor start column
          choice,		;Choice you make
          back.h,		;Background handle
          back.a,		;Background Attributes
          upbutton,		;Scroll Up Button Location
          downbutton,		;Scroll Down Button Location
          hilite,		;Hilite attribute,
          title.length		;Length of title text

Echo off
hilite = 112			;should be Syscolor(1057), but it's ugly

Window Handle Current to win.h		;Get window handle, assign attributes
Window Getattributes win.h to win.a	;Set table window attributes
win.a["HEIGHT"]       = win.height-2
win.a["WIDTH"]        = win.width-2
win.a["ORIGINROW"]    = win.row+1
win.a["ORIGINCOL"]    = win.col+1
win.a["CANMOVE"]      = false
win.a["CANRESIZE"]    = false
win.a["CANMAXIMIZE"]  = false
win.a["CANCLOSE"]     = false
win.a["HASFRAME"]     = false
win.a["HASSHADOW"]    = false
win.a["ECHO"]         = false

win.t = " "+Iif(Isblank(win.t),win.a["TITLE"],win.t)+" "

Dynarray back.a[]		;Set background window attributes
back.a["HEIGHT"]      = win.height
back.a["WIDTH"]       = win.width
back.a["ORIGINROW"]   = win.row
back.a["ORIGINCOL"]   = win.col
back.a["CANMOVE"]     = false
back.a["CANRESIZE"]   = false
back.a["CANMAXIMIZE"] = false
back.a["CANCLOSE"]    = false
back.a["HASFRAME"]    = false

Dynarray keys.a[]			;Set cursor keys array
keys.a[Asc("Down")]  = true  		;These are keys that cause movement
keys.a[Asc("Up")]    = true  		;in the table.
keys.a[Asc("Home")]  = true
keys.a[Asc("End")]   = true
keys.a[Asc("PgUp")]  = true
keys.a[Asc("PgDn")]  = true

searchtxt = ""				;Initialize search variable

If Isblank(prompt.txt) then		;Set default prompt
  Prompt " [Enter] Select   [Esc] Cancel  "+
         "\179  Lookup Help"
Else
  Prompt prompt.txt			;Set custom prompt
Endif

Window Create 				;Create background window
  Attributes back.a
  to back.h

scroll.col = back.a["WIDTH"]-1		;Put in scroll buttons

Frame Double				;Put in frame
  from 0,0
  to back.a["HEIGHT"]-1,scroll.col

title.length = Min(back.a["WIDTH"]-2,len(win.t))	;Write title
@ 0,Int( (back.a["WIDTH"] / 2) - (title.length / 2) )
?? Format("A"+Strval(title.length),win.t)

Paintcanvas Border Attribute Syscolor(1008)		;Color the frame
  0,0,back.a["HEIGHT"]-1,back.a["WIDTH"]-1

@ 1,scroll.col				;Up button position
Setmargin scroll.col

Style Attribute Syscolor(1011)		;Set button color
?? "\30"				;Write Up Button
? "\254"                        	;Write Thumb
thumb.row = Row()

Style Attribute Syscolor(1010)		;Scroll Bar color
For n from 1 to back.a["HEIGHT"]-5
  ? " "					;Write Scroll background
Endfor

Style Attribute Syscolor(1011)		;Set button color
? "\31"					;Write Down Button

Setmargin Off

upbutton = 1				;Get scroll button row numbers
downbutton = Row()

Window Select win.h			;Go to table window
Window Setattributes win.h from win.a
Setcanvas win.h

CtrlHome Right				;Position table in window
key.field = Field()			;to hide the Record Number column
CtrlRight
While Field() <> key.field
  Left
Endwhile
Synccursor				;Get starting column number
startcol = Col()-len([])

PickRecord.hl(0)			;Do first hilite

Echo Normal
Wait Field				;Do the picklist until
  Proc "PickRecord.eh"			;user presses Enter or Esc or
  Key "ALL"				;double-clicks mouse or
  Mouse "DOWN"				;clicks away from table window
Endwait

Echo Off
Window Select back.h			;Clear background
Window Close

Prompt					;Clear prompt

Return choice				;Return variable

Endproc
Writelib libname PickRecord
Release procs PickRecord

;**********************************************************************
;Proc:		PickRecord.eh
;Purpose:	Handle mouse and keystroke events from PickRecord

;Calls		PickRecord.hl

;Called from	PickRecord

Proc PickRecord.eh(tt,ev,cn)
  private mousedirection

If ev["TYPE"] = "KEY" then
  key = ev["KEYCODE"]			;Keystroke processing

  Switch
    Case isassigned(keys.a[key]) :	;Cursor keys
      Keypress key			;Execute movement
      PickRecord.cs()
      PickRecord.hl(0)			;Redraw hilite, no cursor
      Return 1

    Case key >= 32 and			;Searchable characters - Do search
         key <= 126:
      Locate IndexOrder Pattern searchtxt+Chr(key)+".."
      If retval then
        PickRecord.cs()
        PickRecord.hl(len(searchtxt)+1)
        searchtxt = searchtxt + Chr(key)
      Else
        Beep
      Endif
      Return 1
  
    Case key = Asc("Enter") :		;Enter - select record
      Moveto Field return.field
      choice = []
      Return 2
  
    Case key = Asc("Esc") :		;Esc - Cancel
      choice = ""
      Return 2
  
    Otherwise :				;Illegal key!
      Return 1
   
  Endswitch

Else

  Switch				;Handle mouse events

    Case Windowat(ev["ROW"],ev["COL"]) = win.h :	;Mouse on table
      LocalizeEvent ev
      SyncCursor
      If ev["ROW"] = 0 then		;Click on table heading
        @ Row(),startcol		;clears search, restores cursor
        searchtxt = ""
        Return 1
      Endif
      torec = (ev["Row"] - Row())	;Get click location,
      If torec <> 0 then		;Move cursor, redo hilite
        Skip torec
        PickRecord.cs()
        PickRecord.hl(0)
      Else
        @Row(),startcol			;Re-set cursor
        searchtxt = ""			;Clear search string
      Endif

      If ev["DOUBLECLICK"] then		;If double-click, it's a selection!
        Moveto Field return.field
        choice = []
        Return 2
      Else				;Otherwise, clear cycle, return
        Return 1
      Endif

    Case Windowat(ev["ROW"],ev["COL"]) = back.h :	;Mouse on background
      LocalizeEvent ev
      Switch
        Case ev["COL"] <> scroll.col :	;Ignore if not on scroll column
          Return 1
        Case ev["ROW"] = upbutton :	;Execute Up button
          Up
          mousedirection = "UP"
          PickRecord.cs()		;Check Slider
          PickRecord.hl(0)
        Case ev["ROW"] = downbutton :	;Execute Down button
          Down
          mousedirection = "DOWN"
          PickRecord.cs()		;Check Slider
          PickRecord.hl(0)
        Otherwise :
          PickRecord.th(ev["ROW"])	;Process Thumb work
          Return 1
      Endswitch

      While true			;Repeat button action
        GetEvent
          Mouse "Up","Move","Auto"
          to mouseevent
        If mouseevent["ACTION"] = "UP" or
           mouseevent["ACTION"] = "MOVE" then
          Return 1
        Else
          Keypress mousedirection
          PickRecord.cs()		;Check Slider
          PickRecord.hl(0)
        Endif
      Endwhile

    Otherwise :			;Ignore Mouse click away from both windows
      Return 1

  Endswitch

Endif

Endproc
Writelib libname PickRecord.eh
Release procs PickRecord.eh

;**********************************************************************
;Proc:		PickRecord.hl
;Purpose:	Redraw the hilite bar from within PickRecord display

;Called from	PickRecord, PickRecord.eh

Proc PickRecord.hl(search.n)
  private currow,
          recstext

SetCanvas back.h		;Show record number
@back.a["HEIGHT"]-1,2
Style Attribute Syscolor(1008)
recstext = " "+Strval(Recno())+" of "+Strval(Nimagerecords())+" "
?? recstext+Fill("\205",back.a["WIDTH"]-len(recstext)-3)

SetCanvas win.h
Synccursor
currow = Row()

Window Echo win.h true		;Echo new window image to canvas
Window Echo win.h false

Paintcanvas Attribute hilite	;Repaint the hilite bar
  currow,0,currow,(win.a["WIDTH"]-1)

@ currow,(startcol+search.n)	;Set cursor position
If search.n = 0 then		;Re-initialize search text if necessary
  searchtxt = ""
Endif
Cursor Normal

Endproc
Writelib libname PickRecord.hl
Release procs PickRecord.hl

;**********************************************************************
;Name:		PickRecord.th
;Purpose:	Handle thumb movement and related table movement
;Calls:		PickRecord.nr (new record selection)
;Called from:	PickRecord.eh (event handler)

Proc PickRecord.th(mouse.row)
  private mouseevent

Switch

  Case mouse.row < upbutton or			;Ignore clicks on top and
       mouse.row > downbutton :			;bottom of box
    Return

  Case mouse.row = thumb.row :			;Process click on thumb
    While true
      GetEvent Mouse "Up","Move"
        to mouseevent
      
      Switch
        Case mouseevent["ACTION"] = "UP" :	;Mouse up means done!
          Return
        Case Windowat(mouseevent["ROW"],
                      mouseevent["COL"]) <> back.h :
          Loop
      Endswitch

      SetCanvas back.h				;Now process mouse moves
      LocalizeEvent mouseevent
      mouse.row = mouseevent["ROW"]

      If mouseevent["COL"] <> scroll.col or	;Outside acceptable boundaries
         mouse.row <= upbutton or
         mouse.row >= downbutton then
        Loop
      Endif

      @ thumb.row,scroll.col
      Style Attribute Syscolor(1010)		;Scroll Bar color
      ?? " "					;Write Scroll background
      @mouse.row,scroll.col
      Style Attribute Syscolor(1011)		;Set thumbcolor
      ?? "\254"                        		;Write Thumb
      thumb.row = mouse.row			;Reset thumb position
      PickRecord.nr()				;Move to new record
    Endwhile

  Otherwise :					;Process page up and down
    While true
      SetCanvas back.h
      @thumb.row,scroll.col
      Style Attribute Syscolor(1010)		;Scroll Bar color
      ?? " "					;Write Scroll background
      Switch
        Case mouse.row < thumb.row :
          thumb.row = thumb.row - 1
        Case mouse.row > thumb.row :
          thumb.row = thumb.row + 1
        Otherwise :				;This only happens on repeats
          @thumb.row,scroll.col
          Style Attribute Syscolor(1011)
          ?? "\254"
          Return
      Endswitch
      @thumb.row,scroll.col
      Style Attribute Syscolor(1011)		;Set thumbcolor
      ?? "\254"                        		;Write Thumb
      PickRecord.nr()				;Move to new record
      If thumb.row = (upbutton + 1) or		;If at top or bottom, all done
         thumb.row = (downbutton - 1) then
        Return
      Endif
      GetEvent Mouse "Up","Auto" to mouseevent	;Cycle for Idle events
      If mouseevent["ACTION"] = "UP" then
        Return
      Endif
      If Windowat(mouseevent["ROW"],
          mouseevent["COL"]) <> back.h then
        Return
      Endif
      LocalizeEvent mouseevent
      mouserow = mouseevent["ROW"]
    Endwhile

Endswitch

Endproc
Writelib libname PickRecord.th
Release procs PickRecord.th

;**********************************************************************

Proc PickRecord.nr()
  private newrecord

Window Select win.h
Switch					;Get new record position
  Case thumb.row = 2 :
    newrecord = 1
  Case thumb.row = (downbutton - 1) :
    newrecord = Nimagerecords()
  Otherwise :
    newrecord = Int((Nimagerecords() /
                (back.a["HEIGHT"]-1)) *
                (thumb.row - 1))
Endswitch
Moveto Record newrecord			;Moveto correct record
PickRecord.hl(0)			;Rewrite highlight

Endproc
Writelib libname PickRecord.nr
Release procs PickRecord.nr

;**********************************************************************
;  PICKRECORD.CS - CHECK SLIDER POSITION AND ADJUST IF NECESSARY
;**********************************************************************

Proc PickRecord.cs()
  private newthumb.row

Switch
  Case Recno() = 1 :
    newthumb.row = 2
  Case Recno() = NimageRecords() :
    newthumb.row = downbutton - 1
  Otherwise :
    newthumb.row = Int( ( Recno() / Nimagerecords() ) *
                        (back.a["HEIGHT"] - 4) ) + 2
                   
Endswitch

If newthumb.row <> thumb.row then		;If different, adjust
  SetCanvas back.h
  @thumb.row,scroll.col
  Style Attribute Syscolor(1010)		;Scroll Bar color
  ?? " "					;Write Scroll background
  thumb.row = newthumb.row
  @thumb.row,scroll.col
  Style Attribute Syscolor(1011)		;Set thumbcolor
  ?? "\254"                        		;Write Thumb
Endif

Endproc
Writelib libname PickRecord.cs
Release procs PickRecord.cs

;*************************DEMO STARTS HERE*************************

autolib = "Pickrec"

View "Customer"					;Open demo table
Message ""					;Clear the message

pickchoice = PickRecord("Select a Customer",	;Call the procedure
                        "","Cust ID",
                        15,55,4,11)

ClearImage					;Clear the table

If Isblank(pickchoice) then
  Return "PickRecord cancelled!"
Else
  Return "You selected Customer #"+Strval(pickchoice)+"!"
Endif
