/* pdoxrpt.c      Jul 1991
 *
 * This program provides access to Paradox tables for
 * the purpose of generating quick reports.  The report
 * writing capabilities include the specification of
 * report title and column headings, the selection of
 * fields to be included, and the positioning of fields
 * within the report.
 */

#include <stdlib.h>
#include <stdio.h>
#include <conio.h>
#include <dos.h>
#include <bios.h>
#include <ctype.h>
#include <string.h>
#include "pxengine.h"

#define TRUE     1
#define FALSE    0
#define BS       8
#define CR      13
#define ESC     27
#define F6     320
#define F7     321
#define UP     328
#define LT     331
#define RT     333
#define DN     336


/* global variables */
TABLEHANDLE   tblHandle;
RECORDHANDLE  recHandle;
RECORDNUMBER  nRecords;

char datascr[]=
"  Table                       Report Specification                              "
"             Title:                                                             "
"                                                                                "
"            FIELD NAME           INCL LINE COL#          COLUMN TITLE           "
"                                                                                "
"                                                                                "
"                                                                                "
"                                                                                "
"                                                                                "
"                                                                                "
"                                                                                "
"                                                                                "
"                                                                                "
"                                                                                "
"                                                                                "
"                                                                                "
"                                                                                "
"                                                                                "
"                                                                                "
"                                                                                "
"                                                                                "
"                                                                                "
"                                                                                "
"                                                                                "
"[F6] Toggle INCL    [F7] Print    [ESC] Quit                                    ";

char old_screen[4000],
     rpt_title[59],
     col_title[19][32],
     temp_input[59],
     *names[256],
     *types[256];

int  day,month,year,
     nfields,
     tot_cols,
     max_lines,
     setup[19][3],
     col_width[19],
     rec_matrix[10][10][19];

struct text_info screen_info;
struct date today;
struct {
  char incl_flag[2];
  char line[3];
  char col[3];
  char title[31];
}fields[19];


/* function prototypes */
void data_entry(char *);
int  get_input(int,int,int,int,int);
void print_rpt(void);
void setup_rpt(void);
void get_field_data(FIELDHANDLE,char *);
int  getkey(void);
void get_struc_data(void);
void free_struc_data(void);
void initialize(void);
void draw_screen(void);
void display_field_names(char *);
void save_screen(void);
void restore_screen(void);


/* main program function - uses Engine library functions
 * (PX...) to open the commandline passed table name and
 * establish a record buffer; get_struc_data() is called
 * to get table structure information and store said field
 * data in names[] and types[]; it calls data_entry() to
 * get the report specification input and print the
 * resultant report; upon completion, it closes up and
 * restores the preexisting screen
 */
void main(int argc,char *argv[])
{
  if(argc != 2)
  {
    puts("Program call requires a table name or path:");
    puts("       pdoxrpt c:\\pdox35\\mytable ");
    exit(0);
  }
  else
  {
    PXInit();
  }

  if((PXTblOpen(argv[1],&tblHandle,0,0)) != PXSUCCESS)
  {
    cprintf("Cannot find or open file \"%s\"\n",argv[1]);
  }
  else
  {
    PXTblNRecs(tblHandle,&nRecords);
    PXRecBufOpen(tblHandle,&recHandle);
    get_struc_data();
    save_screen();

    data_entry(argv[1]);

    free_struc_data();
    PXRecBufClose(recHandle);
    PXTblClose(tblHandle);
    restore_screen();
  }

  PXExit();
}


/* data_entry() displays the report specification input
 * screen with a call to draw_screen(), makes a call to
 * display_field_names() to write the name of the accessed
 * table and field names to the display, initializes sets
 * of variables, and waits for user input
 */
void data_entry(char *pathName)
{
  int  c,col,row,len,pos,field_num=0,data_col=0;

  draw_screen();
  display_field_names(pathName);
  initialize();

  while(TRUE)
  {
    /* setup cursor location and initialize temporary data
     * string (temp_imput) for current data entry location
     * (data_col) where 0=title, 1=INCL column, 2=LINE
     * column, 3=COL# and 4=COLUMN TITLE
     */
    switch (data_col)
    {
      case (0):
        strcpy(temp_input,rpt_title);
        col=21;
        row=2;
        len=58;
        pos=strlen(temp_input);
        break;
      case (1):
        strcpy(temp_input,fields[field_num].incl_flag);
        col=35;
        row=5+field_num;
        len=1;
        pos=strlen(fields[field_num].incl_flag);
        break;
      case (2)
        strcpy(temp_input,fields[field_num].line);
        col=40;
        row=5+field_num;
        len=1;
        pos=strlen(fields[field_num].line);
        break;
      case (3):
        strcpy(temp_input,fields[field_num].col);
        col=45;
        row=5+field_num;
        len=2;
        pos=strlen(fields[field_num].col);
        break;
      case (4):
        strcpy(temp_input,fields[field_num].title);
        col=49;
        row=5+field_num;
        len=30;
        pos=strlen(fields[field_num].title);
        break;
    }

    /* get user input */
    c=get_input(col,row,len,pos,data_col);

    /* upon return (user has completed data entry for a given
     * field), copy field input from the temporary input string
     * to the field specific data structure element
     */
    switch (data_col)
    {
      case (0):
        strcpy(rpt_title,temp_input);
        break;
      case (1):
        strcpy(fields[field_num].incl_flag,temp_input);
        break;
      case (2):
        strcpy(fields[field_num].line,temp_input);
        break;
      case (3):
        strcpy(fields[field_num].col,temp_input);
        break;
      case (4):
        strcpy(fields[field_num].title,temp_input);
        break;
    }

    /* based upon the keystroke entered at completion of data
     * entry (returned by get_input()), move to indicated next
     * field, print report or quit
     */
    switch (c)
    {
      case (UP):
        if(field_num==0)
        {
          putch('\a');
          break;
        }
        else if(field_num==1)
          data_col=0;
        --field_num;
        break;
      case (DN):
        if(field_num==nfields)
        {
          if(data_col==4)
            putch('\a');
          else
          {
            ++data_col;
            field_num=1;
          }
          break;
        }
        if(field_num==0)
        {
          ++data_col;
        }
        ++field_num;
        break;
      case (CR):
      case (RT):
        if(field_num==nfields && data_col==4)
        {
          putch('\a');
          break;
        }
        if(data_col==0 || data_col==4)
          ++field_num;
        if(data_col==4)
          data_col=1;
        else
          ++data_col;
        break;
      case (LT):
        if(field_num==0)
        {
          putch('\a');
          break;
        }
        if(data_col==1)
          --field_num;
        if(data_col==1 && field_num !=0)
          data_col=4;
        else
          --data_col;
        break;
      case (F7):
        print_rpt();
        break;
      case (ESC):
        return;
    }
  }
}


/* get_input() is the common function for all data entry where
 * the starting coordinate location, maximum length of input,
 * and current position within the data entry string are passed
 */
int get_input(int x,int y,int len,int pos,int data_col)
{
  int c;

  gotoxy(x+pos,y);
  while(TRUE)
  {
    c=getkey();
    if(data_col!=1 && c>31 && c<127 && pos<len)
    {
      putch(c);
      temp_input[pos++]=c;
      continue;
    }
    switch (c)
    {
      case (BS):
      /* a backspace, space, backspace sequence is used to
       * erase the last character entered
       */
        if(pos>0)
        {
          temp_input[--pos]='\0';
          cputs("\x08\x20\x08");
        }
        else
          putch('\a');
        break;
      case (CR):
      case (UP):
      case (LT):
      case (RT):
      case (DN):
      case (F7):
      /* check for validity of input and append null character
       * to terminate data string prior to return - data check
       * is only for LINE and COL#, which require numerics
       */
        if(data_col==2 || data_col==3)
        {
          if(temp_input[0] && !isdigit(temp_input[0]))
          {
            putch('\a');
            break;
          }
        }
        temp_input[pos]='\0';
        return c;
      case (F6):
      /* toggle check mark on-off for INCL */  
        if(data_col==1)
        {
          if(temp_input[0])
          {
            cputs("\x08\x20\x08");
            temp_input[--pos]='\0';
          }
          else
          {
            putch('\xfb');
            temp_input[pos++]='1';
          }
        }
        else
          putch('\a');
        break;
      case (ESC):
        return c;
      default:
        putch('\a');
        break;
    }
  }
}


/* print_rpt() controls all report printing activity; it uses
 * Engine library functions (PX....) to access succeeding
 * records and calls get_field_data() to obtain the data in
 * the indicated field (fld_num); the three-dimensional array
 * "rec_matrix" is initialized by setup_rpt() and is used to
 * determine which fields get printed in what order and on
 * which line
 */
void print_rpt(void)
{
  register int i,j,k;
  int  rec_count=0,
       page_num=0,
       fld_num,
       line_num,
       col_sp;
  char buf1 [BUFSIZ],
       buf2 [BUFSIZ],
       underline[]="--------------------------------------------------";

  setup_rpt();

  /* move to the first record in the table */
  PXRecFirst(tblHandle);

  /* continue to print while the count of records printed
   * is less than the table total
   */
  while(rec_count < nRecords)
  {
    ++page_num;
    line_num=6;

    /* print date, page number and title */
    fprintf(stdprn,"%02d/%02d/%04d%62sPage %2d\r\n\n",month,day,year,
                   "",page_num);
    fprintf(stdprn,"%*s%s\r\n\n\n",40-(strlen(rpt_title)/2),"",
                   rpt_title);

    /* print underlined column headings */
    for(i=1;i<=tot_cols;i++)
    {
      if(i==tot_cols)
        col_sp=0;
      else
        col_sp=2;
      fprintf(stdprn,"%-*.*s",col_width[i],col_width[i]-col_sp,col_title[i]);
    }
    fprintf(stdprn,"\r\n");
    for(i=1;i<=tot_cols;i++)
    {
      if(i==tot_cols)
        col_sp=0;
      else
        col_sp=2;
      fprintf(stdprn,"%-*.*s",col_width[i],col_width[i]-col_sp,underline);
    }
    fprintf(stdprn,"\r\n");

    /* print specified record data while the line count is less
     * than 60 and the end of file (nRecords = total records)
     * has not been reached; when the line_num reaches 60, a
     * new page (with headings) is started
     */
    while(line_num<60 && rec_count++ < nRecords)
    {
      PXRecGet(tblHandle,recHandle);
      for(i=1;i<=max_lines;i++)
      {
        for(j=1;j<=tot_cols;j++)
        {
          k=1;
          buf2[0]=NULL;
          while((fld_num=rec_matrix[i][j][k++])>0)
          {
            get_field_data(fld_num,buf1);
            strcat(buf2,buf1);
            strcat(buf2," "); 
          }
          if(j==tot_cols)
            col_sp=0;
          else
            col_sp=2;
          fprintf(stdprn,"%-*.*s",col_width[j],col_width[j]-col_sp,buf2);
        }
        fprintf(stdprn,"\r\n");
      }
      fprintf(stdprn,"\r\n");
      PXRecNext(tblHandle);
      line_num+=(max_lines+1);
    }
    /* formfeed the page */
    fprintf(stdprn,"\f");
  }
}


/* setup_rpt() collates the information from the inputed report
 * specifications and reduces that data to a format usable in
 * the report printing process
 */
void setup_rpt(void)
{
  register int i;
  int default_col=1,
      fld_count,
      line_num,
      col_num=0,
      line_count=0,
      max_len,
      len;

  tot_cols=0;
  max_lines=0;
  memset(rec_matrix,0,sizeof(rec_matrix));

  for(i=1;i<=nfields;i++)
  {
    /* initialize data structures */
    setup[i][0]=0;
    setup[i][1]=0;
    setup[i][2]=0;
    col_title[i][0]=NULL;

    if(fields[i].incl_flag[0] != NULL)
    {
      /* get line and column data or use default values if
       * none given
       */
      if(fields[i].line[0] != NULL)
        setup[i][0]=atoi(fields[i].line);
      else
        setup[i][0]=1;
      if(fields[i].col[0] != NULL)
        setup[i][1]=atoi(fields[i].col);
      else
        setup[i][1]=default_col++;

      /* get width of column from field type data */
      switch (types[i-1][0])
      {
        case 'A':
          setup[i][2]=atoi(types[i-1]+1);
          break;
        case 'D':
          setup[i][2]=10;
          break;
        case 'N':
        case '$':
          setup[i][2]=15;
          break;
        case 'S':
          setup[i][2]=6;
          break;
      }

      /* get column name or default to field name */
      if(col_num != setup[i][1])
      {
        col_num=setup[i][1];
        if(fields[i].title[0]==NULL)
          strcpy(col_title[col_num],names[i-1]);
        else
          strcpy(col_title[col_num],fields[i].title);
      }
    }
  }

  /* setup field location information for printed report; this
   * includes determining the maximum column width for those
   * in which more than one field is printed on a given line
   */
  for(i=1;i<=nfields;)
  {
    col_width[i]=0;

    if(fields[i].incl_flag[0]==NULL)
    {
      ++i;
      continue;
    }
    line_count=0;
    max_len=0;
    col_num=setup[i][1];

    do{
      len=0;
      fld_count=0;
      line_num=setup[i][0];
      do{
        if(fields[i].incl_flag[0]!=NULL)
        {
          rec_matrix[line_num][col_num][++fld_count]=i;
          len+=(setup[i][2]+1);
        }
      }while((line_num==setup[++i][0] || setup[i][0]==0) &&
             (col_num==setup[i][1]) || setup[i][1]==0));
      ++line_count;
      if(len>max_len)
        max_len=len;
    }while(col_num==setup[i][1]);

    if(line_count>max_lines)
      max_lines=line_count;
    col_width[col_num]=max_len+1;
    ++tot_cols;
  }

  /* determine the total number of columns to be printed based
   * upon a page width of 80 characters; reduce width of last
   * column by the two-char width used to separate columns
   */
  i=0;
  len=0;
  while(len<82 && i<tot_cols)
    len+=col_width[++i];

  if(len>82)
    tot_cols=i-1;
  else if(len==82)
    tot_cols=i;
  col_width[tot_cols]-=2;
}


/* get_field_data() is called by print_rpt() to obtain data
 * contained in the specified field (fh) for the current
 * record; it is returned formatted and stored in the string
 * array s[]; the Engine function (PX....) used to read the
 * data is based upon the field type specified in types[]
 */
void get_field_data(FIELDHANDLE fh, char s[])
{
  short  shortnum;
  int    month,day,year,blankfld;
  long   date;
  double numeric;

  /* check for blank field */
  PXFldBlank(recHandle,fh,&blankfld);
  if(blankfld)
  {
    s[0]='\0';
    return;
  }

  switch(types[fh-1][0])
  {
    case 'A':
      PXGetAlpha(recHandle,fh,BUFSIZ,s);
      break;
    case 'D':
      PXGetDate(recHandle,fh,&date);
      PXDateDecode(date,&month,&day,&year);
      sprintf(s,"%02d/%02d/%04d",month,day,year);
      break;
    case 'N':
      PXGetDoub(recHandle,fh,&numeric);
      sprintf(s,"%15.4lf",numeric);
      break;
    case '$':
      PXGetDoub(recHandle,fh,&numeric);
      sprintf(s,"%15.2lf",numeric);
      break;
    case 'S':
      PXGetShort(recHandle,fh,&shortnum);
      sprintf(s,"%d",shortnum);
      break;
  }
}


/* get_key() is used to process all keystrokes and interpret
 * extended keycodes
 */
int getkey(void)
{
  static int key,hi,lo;

  key=bioskey(0);
  lo =key & 0x00ff;
  hi =(key & 0xff00) >> 8;

  return ((lo==0) ? hi+256 : lo);
}


/* get_struc_data() is called by main() to initialize names[]
 * and types[] arrays (actually arrays of pointers to memory
 * locations) to the field names and field types of the named
 * named table
 */
void get_struc_data(void)
{
  char name[BUFSIZ],
       type[BUFSIZ];

  FIELDHANDLE i;

  nfields=0;

  PXRecNFlds(tblHandle,&nfields);
  for(i=1;i<=nfields;i++)
  {
    PXFldName(tblHandle,i,BUFSIZ,name);
    PXFldType(tblHandle,i,BUFSIZ,type);
    names[i-1]=strdup(name);
    types[i-1]=strdup(type);
  }
}


/* free_struc_data() called by main() at end of program to free
 * memory associated with storage of names[] and types[] data
 */
void free_struc_data(void)
{
  int i;

  for(i=0;i<nfields;i++)
  {
    free(names[i]);
    free(types[i]);
  }
}


/* initialize() gets machine date for use in dating report and
 * sets the report title variable and the array of "field"
 * structure elements to a NULL
 */
void initialize(void)
{
  register int i;

  getdate(&today);
  day=today.da_day;
  month=today.da_mon;
  year=today.da_year;

  rpt_title[0]='\0';

  for(i=0;i<18;i++)
  {
    fields[i].incl_flag[0]='\0';
    fields[i].line[0]='\0';
    fields[i].col[0]='\0';
    fields[i].title[0]='\0';
  }
}


/* draw_screen() writes the dataentry screen defined by datascr[]
 * directly to video memory; because "datascr[]" is an array of
 * type signed character (the default type) it must be cast to
 * type unsigned character ("(unsigned char)") prior to a bit-wise
 * or'ing with the screen attribute for a correct display
 */
void draw_screen(void)
{
  int far *farptr;
  register int addr;

  farptr=(int far *)0xb8000000;
  clrscr();
  for(addr=0;addr<sizeof(datascr);addr++)
    *(farptr + addr)=(unsigned char)datascr[addr] | 0x0700;
}


/* display_field_names() displays the field names obtained by
 * get_struc_data() on the data entry screen; only the first
 * 18 fields are used
 */
void display_field_names(char *pathName)
{
  register int  i;
  char *tabName;

  tabName=strrchr(pathName,92);
  if(!tabName)
    tabName=pathName;
  else
    tabName+=1;

  gotoxy(3,2);
  cputs(tabName);

  if(nfields>18)
    nfields=18;

  for(i=0;i<nfields;i++)
  {
    gotoxy(2,6+i);
    cputs(names[i]);
  }
}


/* save_screen() and restore_screen() are used to restore the
 * screen that existed prior to calling the program; text
 * attribute is set to same as that used in draw_screen()
 */
void save_screen(void)
{
  gettextinfo(&screen_info);
  gettext(1,1,80,25,old_screen);
  textattr(7);
}


void restore_screen(void)
{
  puttext(1,1,80,25,old_screen);
  textattr(screen_info.attribute);
  gotoxy(screen_info.curx,screen_info.cury);
}
