; ****************************************************************************
; SCRIPT NAME: ds4_IO.sc                 (c) 1992, 1993 DataStar International
; APPDEV TEAM: Dan Paolini, David Kelton, Micah Bleecher, Patty Paolini
; APPLICATION: DataStar Utilities for Paradox 4
; DESCRIPTION: Contains the following Input/Output procedures:
;                 ioAcceptDialog.v
;                 ioAcceptDialogHidden.v
;                 ioAcceptDialogValue.v
;                 ioAcceptDirectory.v
;                 ioAcceptDirectoryUpdate.u
;                 ioAcceptDirectoryPath.v
;                 ioPickArrayDialog.v
;                 ioPickDynArrayDialog.v
;                 ioPickDynArrayDialogOption.v
;                 ioPickDynArrayIndexDialog.v
;                 ioPickDynArrayMultiDialog.v
;                 ioPickDynArrayMultiUpdate.v
;                 ioPickMemoDialog.v
;                 ioPickFromTable.l
;                 ioPrinterStatus.l
;                 ioPrintLine.u
;                 ioReportToFile.u
;                 ioSelectOutput.u
;                 ioSelectOutputPrinter.u
;                 ioSelectOutputProcess.l
;                 ioUserLogin.l
;                 ioUserLoginAccept.l
;                 ioUserLoginKey.l
; ============================================================================
? Format("w40"," ds4_IO.sc   - Input/Output Routines")
IF NOT IsAssigned(libname.a) THEN
   Beep Beep Beep QUIT "Variable \"libname.a\" is not assigned!.."
ENDIF
IF NOT IsFile(libname.a+".LIB") THEN
   CreateLib libname.a Size 640
ENDIF
@ Row(),40
; ============================================================================
;       TITLE: ioAcceptDialog.v          (c) 1992, 1993 DataStar International
;     RETURNS: Value Entered, or false if Cancelled
; DESCRIPTION: Generic routine for accepting data from user, with or without
;              a Picture or Default value, Hidden or unhidden.
; ----------------------------------------------------------------------------
PROC ioAcceptDialog.v(           ; One value DialogBox Accept
         top.n,                  ; Top Row for Box (999 = Centered)
         left.n,                 ; Left Column (999 = Centered)
         title.a,                ; Title for dBox
         prompt.a,               ; Data Input Prompt
         type.a,                 ; Type of Data Input
         picture.a,              ; Additional validity string
         default.v,              ; Any Default for the Accept Value?
         hidden.l,               ; Hidden, or not?
         dBoxPalette.a)          ; Color Palette or ""
Private  width.n,                ; Width of Dialog Box
         length.n,               ; Length of Input
         right.n,                ; Right edge of Box
         input.v,                ; Value entered by user
         oldColors.y,            ; Previous Color Set
         accept.v,               ; Variable to capture Accept
         spot.n,                 ; Where to begin Prompt
         pushButton.a,              ; Pushbutton variable
         dBoxColors.y
;Global  g.sysInfo.y
   IF NOT IsAssigned(g.sysInfo.y) THEN
      SysInfo To g.sysInfo.y             ; Determine Screen Size
   ENDIF
   IF Len(prompt.a) > 50 THEN          ; Must keep to a reasonable length
      accept.v = false
      Message "ERROR - Prompt is too Long!!!"
      Beep Beep Beep
      Sleep 5000
   ELSE
      SWITCH                           ; Determine length of Accept Datatype
         CASE type.a = "D" :           ; Set Default value to passed value
            length.n = 11              ;  or a blank value if none passed
            accept.v = IIF(IsBlank(default.v),BlankDate(),default.v)
         CASE type.a = "N" OR type.a = "$"   :
            length.n = 20
            accept.v = IIF(IsBlank(default.v),BlankNum(),default.v)
         CASE type.a = "S" :
            length.n = 8
            accept.v = IIF(IsBlank(default.v),BlankNum(),default.v)
         OTHERWISE         :
            length.n = NumVal(SubStr(type.a,2,3)) + 3
            accept.v = default.v
      ENDSWITCH                        ; Are we beyond 80 column screen width?
      IF length.n + Len(prompt.a) > 69 THEN
         length.n = 69 - Len(prompt.a)
         spot.n = 1
      ENDIF
      width.n = Min(74,Max(32,Max(Len(title.a)+10,length.n+Len(prompt.a)+5)))
      IF NOT IsAssigned(spot.n) THEN   ; Calculate starting spot if needed
         spot.n = Int((width.n - 3 - length.n - Len(prompt.a))/2)
      ENDIF
      IF IsBlank(picture.a) THEN       ; Set "global" Picture if none passed
         IF type.a = "D" THEN          ; Dates are tricky!
            picture.a = "{"+StrVal(Month(Today()))+",#[#]}"+"/"+
                        "{"+StrVal(Day(Today()))+",#[#]}"+"/"+
                        "{"+SubStr(StrVal(Year(Today())),3,2)+",#[#[#[#]]]}"
         ELSE
            picture.a = "*@"
         ENDIF
      ENDIF
      top.n = IIF(top.n = 999, Int((g.sysInfo.y["ScreenHeight"]-8)/2), top.n)
      top.n = IIF(top.n < 0 OR top.n > g.sysInfo.y["ScreenHeight"]-8, 8, top.n)
      left.n = IIF(left.n = 999 OR left.n < 0 OR
                  left.n > g.sysInfo.y["ScreenWidth"]-width.n-3,
                  Int((g.sysInfo.y["ScreenWidth"]-width.n)/2), left.n)
      IF hidden.l THEN
         accept.v = ioAcceptDialogHidden.v(top.n, left.n, title.a,
                                          prompt.a, type.a, picture.a,
                                          width.n, spot.n)
      ELSE
         accept.v = ioAcceptDialogValue.v(top.n, left.n, title.a,
                                          prompt.a, type.a, picture.a,
                                          width.n, spot.n)
      ENDIF
   ENDIF
   Return accept.v                     ; Return entered value or FALSE
ENDPROC
WriteLib libname.a ioAcceptDialog.v
?? "\004"
; ============================================================================
;       TITLE: ioAcceptDialogHidden.v    (c) 1992, 1993 DataStar International
;     RETURNS: Value Entered, or false if Cancelled
; DESCRIPTION: Dialog Box definition to accept a Hidden Value
; ----------------------------------------------------------------------------
PROC ioAcceptDialogHidden.v(     ; Accepts value, using HIDDEN parameter
         topRow.n,               ; Top Row for Box
         leftCol.n,              ; Left Column
         title.a,                ; Title for dBox
         prompt.a,               ; Data Input Prompt
         type.a,                 ; Type of Data Input
         picture.a,              ; Additional validity string
         width.a,                ; Width of dialog box
         spot.n)                 ; Where to begin prompt
Private  pushButton.l
;Global  accept.v                ; Variable to capture Accept
   SHOWDIALOG title.a                  ; Begin DialogBox definition
      Proc "dbEventHandler.l"
         Trigger "OPEN"
      @ -200,-200 Height 7 Width width.n
      @ 1,spot.n ?? prompt.a+":"
      Accept @ 1,spot.n+Len(prompt.a)+1 Width length.n type.a
         Picture picture.a
         Hidden
         Tag "ACCEPT"
      To accept.v
      PushButton @3,3 Width 10 "~O~K"
         OK Default Value dbButtonPress.v(true) Tag "OK"
      To pushButton.l
      PushButton @3,width.n-15 Width 10 "~C~ancel"
         Cancel Value dbButtonPress.v(false) Tag "CANCEL"
      To pushButton.l
   ENDDIALOG                           ; Now evaluate results
   IF NOT retval THEN
      accept.v = false
   ENDIF
   Return accept.v                     ; Return entered value or FALSE
ENDPROC
WriteLib libname.a ioAcceptDialogHidden.v
?? "\004"
; ============================================================================
;       TITLE: ioAcceptDialogValue.v     (c) 1992, 1993 DataStar International
;     RETURNS: Value Entered, or false if Cancelled
; DESCRIPTION: Generic routine for accepting data from user, with or without
;              a Picture or Default value.  Use ioAcceptHidden.v for hidden.
; ----------------------------------------------------------------------------
PROC ioAcceptDialogValue.v(      ; Accepts value from user
         topRow.n,               ; Top Row for Box
         leftCol.n,              ; Left Column
         title.a,                ; Title for dBox
         prompt.a,               ; Data Input Prompt
         type.a,                 ; Type of Data Input
         picture.a,              ; Additional validity string
         width.a,                ; Width of dialog box
         spot.n)                 ; Where to begin prompt
Private  pushButton.l
;Global  accept.v                ; Variable to capture Accept
   SHOWDIALOG title.a                  ; Begin DialogBox definition
      Proc "dbEventHandler.l"
         Trigger "OPEN"
      @ -200,-200 Height 7 Width width.n

      @ 1,spot.n ?? prompt.a+":"
      Accept @ 1,spot.n+Len(prompt.a)+1 Width length.n type.a
         Picture picture.a
         Tag "ACCEPT"
      To accept.v
      PushButton @3,3 Width 10 "~O~K"
         OK Default Value dbButtonPress.v(true) Tag "OK"
      To pushButton.l
      PushButton @3,width.n-15 Width 10 "~C~ancel"
         Cancel
         Value "CANCEL"
         Tag "CANCEL"
      To pushButton.a
   EndDialog                           ; Now evaluate results
   IF NOT retval THEN
      accept.v = false
   ENDIF
   Return accept.v                     ; Return entered value or FALSE
ENDPROC
WriteLib libname.a ioAcceptDialogValue.v
?? "\004"
; ============================================================================
;       TITLE: ioAcceptDirectory.l       (c) 1992, 1993 DataStar International
;     RETURNS: True if not Canceled
; DESCRIPTION: Presents User with dialog box and returns floppy
;              drive letter of selected Drive, False if cancel was selected
; ----------------------------------------------------------------------------
PROC ioAcceptDirectory.v(        ; Accepts Drive or Directory
         title.a,                ; Title of dialog box passed
         directory.a)            ; Default Path displayed in accept statement
Private  path.a,
         frameHigh.n,
         frameLow.n,
         frameTag.a,
         dBoxPalette.a

   frameHigh.n   = 127
   frameLow.n    = 112
   dBoxPalette.a = "GRAY"
   frameTag.a    = "PATH"
   path.a        = "*CANCEL*"
   DynArray dBoxProcs.y[]
      dBoxProcs.y["UPDATE"] = "ioAcceptDirectoryUpdate.l"
   topRow.n = 5
   leftCol.n = 10

   SHOWDIALOG title.a
      Proc "dbEventHandler.l"
         Trigger "OPEN", "UPDATE", "ARRIVE", "DEPART"
      @ -200,-200 Height 11 Width 59

      Frame Single From 0,0 To 2,56
         PaintCanvas Attribute IIF(frameTag.a = "PATH",frameHigh.n,frameLow.n)
                     0,0,2,56
         PaintCanvas Attribute IIF(frameTag.a = "PATH",frameLow.n,frameHigh.n)
                     0,56,2,56
         PaintCanvas Attribute IIF(frameTag.a = "PATH",frameLow.n,frameHigh.n)
                     2,1,2,56

      Frame Single From 3,0 To 8,56
         PaintCanvas Attribute IIF(frameTag.a = "PATH",frameLow.n,frameHigh.n)
                     0,0,2,56
         PaintCanvas Attribute IIF(frameTag.a = "PATH",frameHigh.n,frameLow.n)
                     0,56,2,56
         PaintCanvas Attribute IIF(frameTag.a = "PATH",frameHigh.n,frameLow.n)
                     2,1,2,56

      PaintCanvas Fill "Select Floppy Drive Letter or"
                  Attribute 112 6,13,6,41
      PaintCanvas Fill "Enter Data Directory (IE: C:\\PATH\\)"
                  Attribute 112 7,10,7,44

      Accept @ 1,14 Width 41
         "A40" Picture "*!" Tag "PATH"
      To directory.a

      Label @ 1,2
         "Directory:"
      For "PATH"

      PushButton @4,5 Width 12 "~A~-Drive"
         Value "A:\\" Tag "A"
      To path.a

      PushButton @4,17 Width 12 "~B~-Drive"
         Value "B:\\" Tag "B"
      To path.a

      PushButton @4,29 Width 12 "~C~ancel"
         CANCEL Value "*CANCEL*" Tag "CANCEL"
      To path.a

      PushButton @4,41 Width 12  "~O~k"
         Default Value ioAcceptDirectoryPath.v(directory.a) Tag "OK"
      To path.a
   ENDDIALOG

   IF Search(" " + path.a + " ", " A B *CANCEL* ") = 0 THEN
      IF SubStr(path.a, Len(path.a), 1) <> "\\" THEN
         path.a = path.a + "\\"
      ENDIF
   ENDIF
   Return path.a
ENDPROC
WriteLib libname.a ioAcceptDirectory.v
?? "\004"
; ============================================================================
;       TITLE: ioAcceptDirectoryPath.a   (c) 1992, 1993 DataStar International
;     RETURNS: Logical true/false if directory exists
; DESCRIPTION: Called by the OK button in ioAcceptDirectory.v. Evaluates
;              the accept statements variable.
; ----------------------------------------------------------------------------
PROC ioAcceptDirectoryPath.a(    ; Returns valid directory, or false
         directory.a)            ; Directory name entered by user
   IF DirExists(directory.a) = 1 THEN
      AcceptDialog
   ELSE
      SWITCH
         CASE IsBlank(directory.a)        :
            msContinue.u("","Cannot Enter a Blank Directory, Try Again or Cancel",
                          31,"BLUE",1)
         CASE DirExists(directory.a) = 0  :
            msContinue.u("",directory.a + " Does Not Exist, Try Again or Cancel",
                          31,"BLUE",1)
         CASE DirExists(directory.a) = -1 :
            msContinue.u("", directory.a + " is an Invalid Directory Name, Try Again or Cancel",
                          31,"BLUE",1)
      ENDSWITCH
      SelectControl "PATH"
      directory.a = ""
   ENDIF
   Return directory.a
ENDPROC
WriteLib libname.a ioAcceptDirectoryPath.a
?? "\004"
; ============================================================================
;       TITLE: ioAcceptDirectoryUpdate.l (c) 1992, 1993 DataStar International
;     RETURNS: None
; DESCRIPTION: Event Proc for ioAcceptDirectory.l
; ----------------------------------------------------------------------------
PROC ioAcceptDirectoryUpdate.u() ; Update Trigger from AcceptDirectory
;Global  tag.a
   IF tag.a = "A" OR tag.a = "B" THEN
      IF DriveStatus(tag.a) THEN
         AcceptDialog
      ELSE
         msContinue.u("","Drive " + tag.a + " Not Ready, Try Again or Cancel",
                        79,"RED",1)
      ENDIF
   ENDIF
   Return true
ENDPROC
WriteLib libname.a ioAcceptDirectoryUpdate.u
?? "\004"
; ============================================================================
;       TITLE: ioPickArrayDialog.v       (c) 1992, 1993 DataStar International
;     RETURNS: Value selected, or ""
; DESCRIPTION:
; ----------------------------------------------------------------------------
PROC ioPickArrayDialog.v(        ; Generic PickArray dBox
         listArray.r,            ; Array to use for Picklist
         pickWidth.n,            ; Width for Picklist
         title.a,                ; Title for dBox
         label.a,                ; Label to place above Picklist
         topRow.n,               ; Top row (999 = centered vertically)
         leftCol.n,              ; Left Column (999 = centered horizontally)
         dBoxPalette.a)          ; Dynamic Array of colors, or ""
Private  dBoxHeight.n,
         dBoxWidth.n,
         pushButton.l,
         retval.n

   IF NOT IsAssigned(g.sysInfo.y) THEN
      SysInfo To g.sysInfo.y
   ENDIF
   dBoxWidth.n  = Max(pickWidth.n + 4,30)
   dBoxHeight.n = Max(10,Min(DynArraySize(listArray.y) + 7,
                             g.sysInfo.y["ScreenHeight"] - 3))
   IF topRow.n = 999 THEN
      topRow.n = Int((g.sysInfo.y["ScreenHeight"]-dBoxHeight.n)/2)
   ENDIF
   IF leftCol.n = 999 THEN
      leftCol.n = Int((g.sysInfo.y["ScreenWidth"]-dBoxWidth.n)/2)
   ENDIF
   retval.n     = 0
   pushButton.l = false
   SHOWDIALOG title.a
      PROC "dbEventHandler.l"  Trigger "ARRIVE", "OPEN"
      @ -200, -200 Height dBoxHeight.n Width dBoxWidth.n

;      @ 1,Int((dBoxWidth.n-Len(label.a)-2)/2) label.a "PICKLIST"

      PickArray
         @ 2,Int((dBoxWidth.n-pickWidth.n-2)/2)
         Height dBoxHeight.n - 7 Width pickWidth.n
         listArray.r Tag "PICKLIST"
      To retval.n

      PushButton @ dBoxHeight.n - 4, 2 Width 10 "~S~elect"
         OK Default Value True Tag "BUTTON"
      To pushButton.l

      PushButton @ dBoxHeight.n - 4, dBoxWidth.n - 14 Width 10 "~C~ancel"
         Cancel Value False Tag "BUTTON"
      To pushButton.l
   ENDDIALOG
   Return IIF(retval.n=0,false,listArray.r[retval.n])
ENDPROC
WriteLib libname.a ioPickArrayDialog.v
?? "\004"
;=============================================================================
;       TITLE: ioPickDynArrayDialog.v          (c) 1993 DataStar International
;     RETURNS: Value selected, or ""
; DESCRIPTION:
;-----------------------------------------------------------------------------
PROC ioPickDynArrayDialog.v(     ; Generic PickDynArray dBox
         listArray.y,            ; Array to use for Picklist
         pickWidth.n,            ; Width for Picklist
         title.a,                ; Title for dBox
         label.a,                ; Label to place above Picklist
         topRow.n,               ; Top row (999 = centered vertically)
         leftCol.n,              ; Left Column (999 = centered horizontally)
         dBoxPalette.a)          ; Dynamic Array of colors, or ""
Private  dBoxHeight.n,
         dBoxWidth.n,
         pushButton.l

   IF NOT IsAssigned(g.sysInfo.y) THEN
      SysInfo To g.sysInfo.y
   ENDIF
   dBoxWidth.n  = Max(pickWidth.n + 4,30)
   dBoxHeight.n = Max(10,Min(DynArraySize(listArray.y) + 7,
                             g.sysInfo.y["ScreenHeight"] - 3))
   IF topRow.n = 999 THEN
      topRow.n = Int((g.sysInfo.y["ScreenHeight"]-dBoxHeight.n)/2)
   ENDIF
   IF leftCol.n = 999 THEN
      leftCol.n = Int((g.sysInfo.y["ScreenWidth"]-dBoxWidth.n)/2)
   ENDIF
   retval.v     = ""
   pushButton.l = false
   SHOWDIALOG title.a
      PROC "dbEventHandler.l"  Trigger "ARRIVE"
      @ -200,-200 Height dBoxHeight.n Width dBoxWidth.n

      PickDynArray
         @ 2,Int((dBoxWidth.n-pickWidth.n-2)/2)
         Height dBoxHeight.n - 7 Width pickWidth.n
         listArray.y Tag "PICKLIST"
      To retval.v

      PushButton @ dBoxHeight.n - 4, 2 Width 10 "~S~elect"
         OK Default Value True Tag "BUTTON"
      To pushButton.l

      PushButton @ dBoxHeight.n - 4, dBoxWidth.n - 14 Width 10 "~C~ancel"
         Cancel Value False Tag "BUTTON"
      To pushButton.l
   ENDDIALOG
   Return IIF(IsAssigned(listArray.y[retval.v]),listArray.y[retval.v],"")
ENDPROC
WriteLib libname.a ioPickDynArrayDialog.v
?? "\004"
; ============================================================================
;       TITLE: ioPickDynArrayDialogOption.v    (c) 1993 DataStar International
;     RETURNS: Value selected, or ""
; DESCRIPTION:
; ----------------------------------------------------------------------------
PROC ioPickDynArrayDialogOption.v(  ; Generic PickDynArray dBox
         listArray.y,            ; Array to use for Picklist
         pickWidth.n,            ; Width for Picklist
         title.a,                ; Title for dBox
         label.a,                ; Label to place above Picklist
         topRow.n,               ; Top row (999 = centered vertically)
         leftCol.n,              ; Left Column (999 = centered horizontally)
         dBoxPalette.a,          ; Dynamic Array of colors, or ""
         index.l)                ; return index if true, value otherwase

Private  dBoxHeight.n,
         dBoxWidth.n,
         pushButton.l

   IF NOT IsAssigned(g.sysInfo.y) THEN
      SysInfo To g.sysInfo.y
   ENDIF
   dBoxWidth.n  = Max(pickWidth.n + 4,30)
   dBoxHeight.n = Max(10,Min(DynArraySize(listArray.y) + 7,
                             g.sysInfo.y["ScreenHeight"] - 3))
   IF topRow.n = 999 THEN
      topRow.n = Int((g.sysInfo.y["ScreenHeight"]-dBoxHeight.n)/2)
   ENDIF
   IF leftCol.n = 999 THEN
      leftCol.n = Int((g.sysInfo.y["ScreenWidth"]-dBoxWidth.n)/2)
   ENDIF
   retval.v     = ""
   pushButton.l = false
   SHOWDIALOG title.a
      PROC "dbEventHandler.l"  Trigger "OPEN", "ARRIVE"
      @ -200,-200 Height dBoxHeight.n Width dBoxWidth.n

;      @ 1,Int((dBoxWidth.n-Len(label.a)-2)/2) label.a "PICKLIST"

      PickDynArray
         @ 2,Int((dBoxWidth.n-pickWidth.n-2)/2)
         Height dBoxHeight.n - 7 Width pickWidth.n
         listArray.y Tag "PICKLIST"
      To retval.v

      PushButton @ dBoxHeight.n - 4, 2 Width 10 "~S~elect"

         OK Default Value True Tag "BUTTON"
      To pushButton.l

      PushButton @ dBoxHeight.n - 4, dBoxWidth.n - 14 Width 10 "~C~ancel"
         Cancel Value False Tag "BUTTON"
      To pushButton.l
   ENDDIALOG
   SWITCH
     CASE NOT retval :
       retval.v = ""
     CASE index.l = TRUE :
       ;do nothing
     OTHERWISE :
      retval.v =  IIF(IsAssigned(listArray.y[retval.v]),listArray.y[retval.v],"")
   ENDSWITCH
   RETURN retval.v
ENDPROC
WriteLib libname.a ioPickDynArrayDialogOption.v
?? "\004"
;=============================================================================
;       TITLE: ioPickDynArrayIndexDialog.v     (c) 1993 DataStar International
;     RETURNS: Value selected, or ""
; DESCRIPTION:
;-----------------------------------------------------------------------------
PROC ioPickDynArrayIndexDialog.v(; Generic PickDynArray dBox
         listArray.y,            ; Array to use for Picklist
         pickWidth.n,            ; Width for Picklist
         title.a,                ; Title for dBox
         label.a,                ; Label to place above Picklist
         topRow.n,               ; Top row (999 = centered vertically)
         leftCol.n,           ; Left Column (999 = centered horizontally)
         dBoxPalette.a)          ; Dynamic Array of colors, or ""
Private  dBoxHeight.n,
         dBoxWidth.n,
         pushButton.l,
         retval.v

   IF NOT IsAssigned(g.sysInfo.y) THEN
      SysInfo To g.sysInfo.y
   ENDIF
   dBoxWidth.n  = Max(pickWidth.n + 4,30)
   dBoxHeight.n = Max(10,Min(DynArraySize(listArray.y) + 7,
                             g.sysInfo.y["ScreenHeight"] - 3))
   IF topRow.n = 999 THEN
      topRow.n = Int((g.sysInfo.y["ScreenHeight"]-dBoxHeight.n)/2)
   ENDIF
   IF leftCol.n = 999 THEN
      leftCol.n = Int((g.sysInfo.y["ScreenWidth"]-dBoxWidth.n)/2)
   ENDIF
   retval.v     = ""
   pushButton.l = false
   SHOWDIALOG title.a
      PROC "dbEventHandler.l"  Trigger "ARRIVE", "SELECT", "OPEN"
      @ -200,-200 Height dBoxHeight.n Width dBoxWidth.n

;      @ 1,Int((dBoxWidth.n-Len(label.a)-2)/2) label.a "PICKLIST"

      PickDynArrayIndex
         @ 2,Int((dBoxWidth.n-pickWidth.n-2)/2)
         Height dBoxHeight.n - 7 Width pickWidth.n
         listArray.y Tag "PICKLIST"
      To retval.v

      PushButton @ dBoxHeight.n - 4, 2 Width 10 "~S~elect"
         OK Default Value True Tag "BUTTON"
      To pushButton.l

      PushButton @ dBoxHeight.n - 4, dBoxWidth.n - 14 Width 10 "~C~ancel"
         Cancel Value False Tag "BUTTON"
      To pushButton.l
   ENDDIALOG
   Return retval.v
ENDPROC
WriteLib libname.a ioPickDynArrayIndexDialog.v
?? "\004"
; ============================================================================
;       TITLE: ioPickDynArrayMultiDialog.v     (c) 1993 DataStar International
;     RETURNS: retval.v
; DESCRIPTION: Uses an array to select values from multiple fields
; ----------------------------------------------------------------------------
PROC ioPickDynArrayMultiDialog.v(; MultiField Picklist
         pickList.y,             ; Array to use for Picklist
         pickedList.y,           ; Array of selected items
         title.a,                ; Title for dBox
         listLabel.a,            ; Label to place above Picklist
         pickLabel.a,            ; Label to place above Selected list
         dBoxPalette.a)          ; Dynamic Array of colors, or ""
Private  dBoxHeight.n,
         dBoxWidth.n,
         pushButton.l,
         pickedList.a,
         pickList.a,
         frameHigh.n,
         frameLow.n,
         frameTag.a

   IF NOT IsAssigned(g.sysInfo.y) THEN
      SysInfo To g.sysInfo.y
   ENDIF
   retval.v     = ""
   pushButton.l = false
   topRow.n = 4
   leftCol.n = 3
   DynArray dBoxProcs.y[]
      dBoxProcs.y["UPDATE"] = "ioPickDynArrayMultiUpdate.l"
      dBoxProcs.y["KEY"]    = "ioPickDynArrayMultiUpdate.l"
   frameHigh.n = 127
   frameLow.n = 112
   frameTag.a = "PICKLIST"

   SHOWDIALOG title.a
      PROC "dbEventHandler.l"
         Key 13
         Trigger "ARRIVE", "OPEN", "UPDATE"
      @ -200, -200  Height 17 Width 74

      Frame Single From 0,0 To 14,25
         PaintCanvas Border Attribute IIF(frameTag.a="PICKLIST",
                                          frameHigh.n,frameLow.n) 0,0,14,25
         PaintCanvas Attribute IIF(frameTag.a="PICKLIST",
                                   frameLow.n,frameHigh.n) 0,0,0,24
         PaintCanvas Attribute IIF(frameTag.a="PICKLIST",
                                   frameLow.n,frameHigh.n) 0,0,14,0
      Frame Single From 0,26 To 14,45
         PaintCanvas Border Attribute IIF(Search("PB_",frameTag.a) = 1,
                                          frameHigh.n,frameLow.n) 0,26,14,45
         PaintCanvas Attribute IIF(Search("PB_",frameTag.a) = 1,
                                   frameLow.n,frameHigh.n) 0,26,0,44
         PaintCanvas Attribute IIF(Search("PB_",frameTag.a) = 1,
                                   frameLow.n,frameHigh.n) 0,26,14,26
      Frame Single From 0,46 To 14,71
         PaintCanvas Border Attribute IIF(frameTag.a="PICKEDLIST",
                                          frameHigh.n,frameLow.n) 0,46,14,71
         PaintCanvas Attribute IIF(frameTag.a="PICKEDLIST",
                                   frameLow.n,frameHigh.n) 0,46,0,70
         PaintCanvas Attribute IIF(frameTag.a="PICKEDLIST",
                                   frameLow.n,frameHigh.n) 0,46,14,46

      Label @ 0,5 listLabel.a For "PICKLIST"
      PickDynArray
         @ 2,2
         Height 12 Width 22
         pickList.y Tag "PICKLIST"
      To pickList.a

      Label @ 0,51 pickLabel.a For "PICKEDLIST"
      PickDynArray
         @ 2,48
         Height 12 Width 22
         pickedList.y Tag "PICKEDLIST"
      To pickedList.a

      PushButton @ 2,27  Width 18 "Pick Item  "
         Value True Tag "PB_SELECTITEM"
      To pushButton.l

      PushButton @ 4,27  Width 18 "~P~ick All  "
         Value True Tag "PB_SELECTALL"
      To pushButton.l

      PushButton @ 6,27  Width 18 "  Remove Item"
         Value True Tag "PB_REMOVEITEM"
      To pushButton.l

      PushButton @ 8,27  Width 18 " ~R~emove All"
         Value True Tag "PB_REMOVEALL"
      To pushButton.l

      PushButton @ 10,27 Width 18 "~O~K"
         OK Value True Tag "PB_ACCEPT"
      To pushButton.l

      PushButton @ 12,27 Width 18 "~C~ancel"
         Cancel Value False Tag "PB_CANCEL"
      To pushButton.l
   ENDDIALOG
   Return retval.v
ENDPROC
WriteLib libname.a ioPickDynArrayMultiDialog.v
?? "\004"
; ============================================================================
;       TITLE: ioPickDynArrayMultiUpdate.l     (c) 1993 DataStar International
;     RETURNS:
; DESCRIPTION:
; ----------------------------------------------------------------------------
PROC ioPickDynArrayMultiUpdate.l()
   IF type.a = "EVENT" THEN
      SWITCH
         CASE tag.a = "PICKLIST" :
            IF NOT IsBlank(pickList.a) THEN
               pickedList.y[pickList.a] = pickList.y[pickList.a]
               Release VARS pickList.y[pickList.a]
               RefreshControl "PICKLIST"
               RefreshControl "PICKEDLIST"
            ELSE
               Beep
            ENDIF
         CASE tag.a = "PICKEDLIST" :
            IF NOT IsBlank(pickedList.a) THEN
               pickList.y[pickedList.a] = pickedList.y[pickedList.a]
               Release VARS pickedList.y[pickedList.a]
               RefreshControl "PICKLIST"
               RefreshControl "PICKEDLIST"
            ELSE
               Beep
            ENDIF
      ENDSWITCH
   ELSE
      SWITCH
         CASE tag.a = "PB_SELECTITEM" :
            IF NOT IsBlank(pickList.a) THEN
               pickedList.y[pickList.a] = pickList.y[pickList.a]
               Release VARS pickList.y[pickList.a]
               RefreshControl "PICKLIST"
               RefreshControl "PICKEDLIST"
            ELSE
               Beep
            ENDIF
         CASE tag.a = "PB_REMOVEITEM" :
            IF NOT IsBlank(pickedList.a) THEN
               pickList.y[pickedList.a] = pickedList.y[pickedList.a]
               Release VARS pickedList.y[pickedList.a]
               RefreshControl "PICKLIST"
               RefreshControl "PICKEDLIST"
            ELSE
               Beep
            ENDIF
         CASE tag.a = "PB_SELECTALL" :
            FOREACH a In pickList.y
               pickedList.y[a] = pickList.y[a]
               Release VARS pickList.y[a]
            ENDFOREACH
            RefreshControl "PICKLIST"
            RefreshControl "PICKEDLIST"
         CASE tag.a = "PB_REMOVEALL" :
            FOREACH a In pickedList.y
               pickList.y[a] = pickedList.y[a]
               Release VARS pickedList.y[a]
            ENDFOREACH
            RefreshControl "PICKLIST"
            RefreshControl "PICKEDLIST"
      ENDSWITCH
   ENDIF
   Return true
ENDPROC
WriteLib libname.a ioPickDynArrayMultiUpdate.l
?? "\004"
; ============================================================================
;       TITLE: ioPickMemoDialog.v        (c) 1992, 1993 DataStar International
;     RETURNS: Original value, Original value plus New value, or New value
; DESCRIPTION: Uses an array to select values to append to or overwrite an
;              existing memo value (to build a memo from boilerplate items).
; ----------------------------------------------------------------------------
PROC ioPickMemoDialog.v(         ; Generic PickDynArray dBox
         current.v,              ; Current value of string to modify
         listArray.y,            ; Array to use for Picklist
         pickWidth.n,            ; Width for Picklist
         title.a,                ; Title for dBox
         label.a,                ; Label to place above Picklist
         labelColor.n,           ; Color for Label
         topRow.n,               ; Top row (999 = centered vertically)
         leftCol.n,              ; Left Column (999 = centered horizontally)
         colors.v)               ; Dynamic Array of colors, or ""
Private  dBoxHeight.n,
         dBoxWidth.n,
         pushButton.a,
         selection.v

   IF NOT IsAssigned(g.sysInfo.y) THEN
      SysInfo To g.sysInfo.y
   ENDIF
   dBoxWidth.n  = Max(pickWidth.n + 4,51)
   dBoxHeight.n = Max(10,Min(DynArraySize(listArray.y) + 7,
                             g.sysInfo.y["ScreenHeight"] - 3))
   topRow.n = IIF(topRow.n = 999,
                  Int((g.sysInfo.y["ScreenHeight"]-dBoxHeight.n)/2),
                  topRow.n)
   leftCol.n = IIF(leftCol.n = 999,
                  Int((g.sysInfo.y["ScreenWidth"]-dBoxWidth.n)/2),
                  leftCol.n)
   selection.v  = ""
   pushButton.a = "CANCEL"
   SHOWDIALOG title.a
      PROC "dbEventHandler.l"
         Trigger "OPEN"
      @ -200,-200 Height dBoxHeight.n Width dBoxWidth.n

      PaintCanvas Fill Format("w"+StrVal(dBoxWidth.n-4)+",ac",label.a)
                  Attribute labelColor.n 1,1,1,dBoxWidth.n-4

      PickDynArray
         @ 2,Int((dBoxWidth.n-pickWidth.n-2)/2)
         Height dBoxHeight.n - 7 Width pickWidth.n
         listArray.y Tag "PICKLIST"
      To selection.v

      PushButton @ dBoxHeight.n - 4, 2 Width 10 "~A~ppend"
         OK Value "APPEND" Tag "BUTTON"
      To pushButton.a

      PushButton @ dBoxHeight.n - 4, dBoxWidth.n - 31 Width 13 "~O~verwrite"
         OK Value "OVERWRITE" Tag "BUTTON"
      To pushButton.a

      PushButton @ dBoxHeight.n - 4, dBoxWidth.n - 14 Width 10 "~C~ancel"
         Cancel Value "CANCEL" Tag "BUTTON"
      To pushButton.a
   ENDDIALOG
   Return IIF(pushButton.a = "OVERWRITE",
              selection.v,
              current.v + StrVal(selection.v))
ENDPROC
WriteLib libname.a ioPickMemoDialog.v
?? "\004"
; ============================================================================
;       TITLE: ioPickFromTable.l         (c) 1992, 1993 DataStar International
;     RETURNS: Logical true/false
; DESCRIPTION: Generic item selection.  This is not a true WAIT handler.
;              Use it to have the user select a Master record,
; ----------------------------------------------------------------------------
PROC ioPickFromTable.l(          ; Generic PickDynArray dBox
         table.a,                ; table to select from
         form.a,                 ; form to use
         prompt.a)               ; prompt Line 1
Private  retval.l
;Global  pickRecord.r            ; array to hold selected record
   retval.l = false
   IF NOT IsEmpty(table.a) THEN
      Lock table.a PFL
      IF NOT retval THEN
          msContinue.u("","Unable to Lock the " + table.a + " Table.  " +
                          "Please try again later.",31,"BLUE",2)
      ELSE
         View table.a
         IF NOT IsBlank(form.a) THEN
            wsPickForm.l(form.a)
         ENDIF
         ShowPullDown
            "Press ~<F2>~ to Select  " : "" : "",
            "~<Esc>~ to Cancel  "      : "" : "",
            "~<Ctrl><Z>~ to Search"    : "" : ""
         EndMenu
         WHILE true
            WAIT Table Prompt
               prompt.a
            Until "Dos", "DosBig", "F35", "FieldView", "Zoom", "Ins", "Rotate",
                  "ZoomNext", "Del", "F2", "Esc", "F1", -35
            SWITCH
               CASE retval = "Zoom"       : wsZoom.l("FIRST")
               CASE retval = "ZoomNext"   : wsZoom.l("NEXT")
               CASE retval = "Esc"        :
                  ClearAll
                  QUITLOOP
               CASE retval = "F1"         : lkHelpTable.u("Select")
               CASE retval = -35          : hsEngine.u("Select",g.help.y)
               CASE retval = "F2"  :
                  CopyToArray pickRecord.r  ; assign selected record to array
                  retval.l = true
                  QUITLOOP
               OTHERWISE           : Beep
            ENDSWITCH
         ENDWHILE
      ENDIF
   ENDIF
   Return retval.l
ENDPROC
WriteLib libname.a ioPickFromTable.l
?? "\004"
; ============================================================================
;       TITLE: ioPrinterStatus.l         (c) 1992, 1993 DataStar International
;     RETURNS: logical true or false if printer available
; DESCRIPTION: Generic printer status, called from ErrorProc
; ----------------------------------------------------------------------------
PROC ioPrinterStatus.l()         ; Generic printer status
Private  retval.l                ; Value to return
   retval.l = true
   msWorking.u("Checking Printer Status",111,0,0)
   WHILE NOT PrinterStatus()
      msWorkingClear.u()
      retval.l = msConfirm.l("","N",79,"RED",3,"~R~eady","~C~ancel",true)
      IF NOT retval.l THEN
         QUITLOOP
      ENDIF
      msWorking.u("Checking Printer Status",111,0,0)
   ENDWHILE
   IF NOT retval.l THEN
      msContinue.u("","The Report has been Canceled - Attempting to " +
                       "Continue with Application",31,"BLUE",1)
   ENDIF
   msWorkingClear.u()
   Return retval.l
ENDPROC
WriteLib libname.a ioPrinterStatus.l
?? "\004"
; ============================================================================
;       TITLE: ioPrintLine.u             (c) 1992, 1993 DataStar International
;     RETURNS: No value
; DESCRIPTION: Generic PAL Print Line routine
; ----------------------------------------------------------------------------
PROC ioPrintLine.u(              ; Generic Print Line in PAL
         line.a,                 ; Formatted Line to Print
         margin.n,               ; Standard Left Margin
         file.a)                 ; Name of File, or "" for Printer
   IF IsBlank(file.a) THEN
      Print Spaces(margin.n) + line.a + "\n"
   ELSE
      Print File file.a Spaces(margin.n) + line.a + "\n"
   ENDIF
   Return
ENDPROC
WriteLib libname.a ioPrintLine.u
?? "\004"
; ============================================================================
;       TITLE: ioReportToFile.u          (c) 1992, 1993 DataStar International
;     RETURNS: No Value
; DESCRIPTION: Prints a Report to a designated File Name
; ----------------------------------------------------------------------------
PROC ioReportToFile.u(           ; Generic report to file
         table.a,                ; Table to Report on
         report.a,               ; Report to Output
         file.a)                 ; Name of File to Output to
Private  g.config.y              ; This set embedded printer control
                                 ;  variables to blank

   msWorking.u("Preparing Report - Please Wait",111,0,0)
   ; Sets blank Setup string
   {Report} {Change} Select table.a Select report.a Enter
   {Setting} {Setup} {Custom} Enter       ; Enter chooses default Port
      Select "" Select ""                 ; Second SELECT removes Reset
   {Output} {File} Select file.a          ; Outputs to File
   IF MenuChoice() = "Cancel" THEN
      {Replace}
   ENDIF
   {Cancel} {Yes}
   msWorkingClear.u()
   Return
ENDPROC
WriteLib libname.a ioReportToFile.u
?? "\004"
; ============================================================================
;       TITLE: ioSelectOutput.u          (c) 1992, 1993 DataStar International
;     RETURNS: No value
; DESCRIPTION: Generic Output Loop
; ----------------------------------------------------------------------------
PROC ioSelectOutput.u(           ; Generic Report Output Loop
         title.a,                ; Report Title
         table.a,                ; Table to report on
         report.a,               ; Report number
         default.a,              ; default printer port for Local
         printerName.a,          ; printer name
         setup.a,                ; Report Setup String
         setupField.a,           ; Printer Setup String Field in printTable.a
         printTable.a,            ; Full path to Printers table, or ""
         custom.a,               ; Custom Printer Select routine or ""
         pause.l)                ; Pause before Printing?
Private  menu.a,                 ; destination for report
         file.a,                 ; name for saved report file
         n,                      ; Menu selection number
         file.l,                 ; Is Report already in file?
         netPort.a,              ; selected network port
         destination.a,          ; Report destination
         pushButton.l,
         screenFile.a,
         printers.r,
         frameHigh.n,
         frameLow.n,
         titleLength.n,
         titleLeft.n,
         frameTag.a,
         destination.n,
         dBoxPalette.a
;Global  g.config.y,
;;;        pica.a,
;;;        elite.a,
;;;        condensed.a,
;;;        compressed.a,
;;;        picalandscape.a,
;;;        elitelandscape.a,
;;;        condensedlandscape.a,
;;;        boldon.a,
;;;        boldoff.a,
;;;        reset.a,

   msWorking.u("W",111,0,0)
   IF NOT IsBlank(printTable.a) THEN
      ioSelectOutputPrinter.u(printTable.a)
   ENDIF

   file.l         = false
   screenFile.a   = PrivDir() + StrVal(Ticks())+".sc"
   file.a         = PrivDir() + "Filesave.rpt"
   destination.n  = 1
   pushButton.l   = false
   frameHigh.n    = inAttributeConvert.n(g.appcolors.y["1036"],true)
   frameLow.n     = inAttributeConvert.n(g.appcolors.y["1036"],false)
   frameTag.a     = "OUTPUT"
   titleLength.n  = Min(54,Len(title.a)+2)
   titleLeft.n    = 28 - Int(titleLength.n/2)
   dBoxPalette.a  = "GRAY"
   dbPaletteSet.u("GRAY")

   msWorkingClear.u()
   topRow.n = 4
   leftCol.n = 10

   SHOWDIALOG "Select Report Destination for"
      PROC "dbEventHandler.l"
         Trigger "UPDATE", "ARRIVE", "DEPART", "OPEN"
      @ -200,-200 Height 13 Width 60

      Frame Single From 3,7 To 6,50
         PaintCanvas Border
                     Attribute IIF(frameTag.a="DESTINATION",frameHigh.n,frameLow.n)
                     3,7,6,50
         PaintCanvas Border
                     Attribute IIF(frameTag.a="DESTINATION",frameLow.n,frameHigh.n)
                     3,7,3,49
         PaintCanvas Border
                     Attribute IIF(frameTag.a="DESTINATION",frameLow.n,frameHigh.n)
                     3,7,6, 7
      Frame Single From 7,7 To 10,50
         PaintCanvas Border
                     Attribute IIF(frameTag.a="OK" OR frameTag.a = "CANCEL",
                     frameHigh.n,frameLow.n)
                     7,7,10,50
         PaintCanvas Border
                     Attribute IIF(frameTag.a="OK" OR frameTag.a = "CANCEL",
                     frameLow.n,frameHigh.n)
                     7,7,7,49
         PaintCanvas Border
                     Attribute IIF(frameTag.a="OK" OR frameTag.a = "CANCEL",
                     frameLow.n,frameHigh.n)
                     7,7,10,7

      @ 1,titleLeft.n ?? Format("w"+StrVal(titleLength.n)+",ac",title.a)
         PaintCanvas Attribute 95 1,titleLeft.n,1,titleLeft.n+titleLength.n-1

      @ 2,titleLeft.n+1 ?? Fill("",titleLength.n)
         PaintCanvas Attribute frameLow.n 2,titleLeft.n+1,
                                          2,titleLeft.n+titleLength.n

      @ 1,titleLeft.n+titleLength.n ?? ""
         PaintCanvas Attribute frameLow.n 1,titleLeft.n+titleLength.n,
                                          1,titleLeft.n+titleLength.n

      RadioButtons @ 4,8 Height 2 Width 42
         "Screen",
         "Printer",
         "Spreadsheet",
         "DiskFile",
         "Table"
         Tag "DESTINATION"
      To destination.n

      PushButton @ 8,11 Width 12
         "~O~utput"
         Default Value ioSelectOutputProcess.l() Tag "OK"
      To pushButton.l

      PushButton @ 8,35 Width 12
         "~C~ontinue"
         Cancel Value dbButtonPress.v(false) Tag "CANCEL"
      To pushButton.l
   ENDDIALOG

   msWorking.u("W",111,0,0)
   {Report} {SetPrinter} {Regular}
   {Report} {SetPrinter} {Override} {EndOfPage} {FormFeed}

   IF file.l THEN
      Run NOREFRESH "Del " + screenFile.a + " > NUL"
   ENDIF

   IF NOT IsBlank(printTable.a) THEN
      tbView.u(printTable.a,true)
      ClearImage
      UnLock printTable.a PFL
   ENDIF

   msWorkingClear.u()
   Return
ENDPROC
WriteLib libname.a ioSelectOutput.u
?? "\004"
; ============================================================================
;       TITLE: ioSelectOutputPrinter.u   (c) 1992, 1993 DataStar International
;     RETURNS: No value
; DESCRIPTION: Reads Printers from printer table
; ----------------------------------------------------------------------------
PROC ioSelectOutputPrinter.u(    ; Reads Printers from printer table
         printTable.a)
Private  count.n,
         w
   count.n = 0
   Lock printTable.a PFL
   WHILE NOT retval AND count.n < 5
      count.n = count.n + 1
      Sleep 1000
      Lock printTable.a PFL
   ENDWHILE
   IF count.n = 5 THEN
      printTable.a = ""
   ELSE
      View printTable.a
      Window Handle Image ImageNo() To w
      MoveTo [Printer Name]
      Array printers.r[NImageRecords()]
      SCAN
         printers.r[RecNo()] = []
         IF [] = printerName.a Then
            Moveto Field setupField.a
            setup.a=[]
            Moveto [Printer Name]
         ENDIF
      ENDSCAN
   ENDIF
   wsWindowPark.u(w)
   Return
ENDPROC
WriteLib libname.a ioSelectOutputPrinter.u
?? "\004"
; ============================================================================
;       TITLE: ioSelectOutputProcess.l   (c) 1992, 1993 DataStar International
;     RETURNS: Logical true/false
; DESCRIPTION: Generic Output Dialog Box Proc
; ----------------------------------------------------------------------------
PROC ioSelectOutputProcess.l()   ; Generic Report Output Loop
Private  print.l,
         v,
         altPrinter.n,
         pushButton.l,
         printer.r
;Global  printTable.a
;        g.config.y,
;        setup.a,
;        custom.a,
;        g.printers.r
;        printerName.a

   print.l = false
   SWITCH
      CASE destination.n = 1  :
         IF NOT file.l THEN
            ioReportToFile.u(table.a,report.a,screenFile.a)
         ENDIF
         msWorking.u("Sending Report to Screen - Press <Esc> when Finished Viewing",
                        31, 1, 0)
         Run NoRefresh "Readme " + screenFile.a
         file.l = true
      CASE destination.n = 2  :
         IF NOT IsBlank(printTable.a) THEN
            tbView.u(printTable.a,true)
            Right
            Locate printerName.a
            IF retval THEN
               CopyToArray printer.r
               g.config.y["printer.a"]             = user.r[2]
               g.config.y["port.a"]                = user.r[3]
               g.config.y["pica.a"]                = user.r[4]
               g.config.y["elite.a"]               = user.r[5]
               g.config.y["condensed.a"]           = user.r[6]
               g.config.y["compressed.a"]          = user.r[7]
               g.config.y["picaLandscape.a"]       = user.r[8]
               g.config.y["eliteLandscape.a"]      = user.r[9]
               g.config.y["condensedLandscape.a"]  = user.r[10]
               g.config.y["compressedLandscape.a"] = user.r[11]
               g.config.y["boldOn.a"]              = user.r[12]
               g.config.y["boldOff.a"]             = user.r[13]
               g.config.y["reset.a"]               = user.r[14]
               setup.a = []
            ENDIF
         ENDIF
         print.l = true
      CASE destination.n = 3 :
      CASE destination.n = 3        :
         WHILE true
            msWorking.u("Enter 1-8 Character Table Name to be placed in YOUR Directory",31,0,0)
            file.a = ioAcceptDialog.v(999,999,"Enter Table Name",
                                    "Filename:","A8","*{&,#,.,$}","",false,"")
            IF NOT file.a = false THEN
               IF IsTable(PrivDir()+file.a) THEN        ; Whoops!
                  msConfirm.l("", "That Filename Exists.  Do you want to " +
                                  "Overwrite it, or Try Again?", 31, "BLUE",
                              1, "~T~ry Again", "~O~verwrite", false)
                  IF NOT retval THEN
                     LOOP
                  ENDIF
               ENDIF                         ; Save to specified file name
               Copy table.a PrivDir() + file.a
            ENDIF
            QUITLOOP
         ENDWHILE
      CASE destination.n = 4        :
         WHILE true
            msWorking.u("Enter 1-8 Character Spreadsheet Name to be placed in YOUR Directory",31,0,0)
            file.a = ioAcceptDialog.v(999,999,"Enter Spreadsheet Name",
                                     "Filename:","A8","*{&,#,.,$}","",false,"")
            IF NOT file.a = false THEN
               IF IsFile(PrivDir()+file.a+".WK1") THEN        ; Whoops!
                  msConfirm.l("", "That Filename Exists.  Do you want to " +
                                 "Overwrite it, or Try Again?", 31, "BLUE",
                              1, "~T~ry Again", "~O~verwrite", false)
                  IF NOT retval THEN
                     LOOP
                  ENDIF
               ENDIF                         ; Save to specified file name
               {Tools} {ExportImport} {Export} {1-2-3} {2) 1-2-3 Release 2}
                  Select table.a
                  Select PrivDir() + file.a
                  IF MenuChoice() = "Cancel" THEN
                     {Replace}
                  ENDIF
            ENDIF
            QUITLOOP
         ENDWHILE
      CASE destination.n = 5  : ; SaveTable()
         WHILE true
            msWorking.u("Enter 1-8 Character ASCII Report Name to be placed in YOUR Directory",31,2,0)
            file.a = ioAcceptDialog.v(999,999,"Enter ASCII File Name",
                                     "Filename:","A8","*{&,#,.,$}","",false,"")
            IF NOT file.a = false THEN
               IF IsFile(PrivDir()+file.a) THEN        ; Whoops!
                  msConfirm.l("", "That Filename Exists.  Do you want to " +
                                  "Overwrite it, or Try Again?", 31, "BLUE",
                              1, "~T~ry Again", "~O~verwrite", false)
                  IF NOT retval THEN
                     LOOP
                  ENDIF
               ENDIF
               IF NOT file.l THEN
                  msWorking.u("One Moment - Preparing Report",49,0,0)
                  ioReportToFile.u(table.a,report.a,screenFile.a)
                  file.l = true
               ENDIF
               msWorking.u("Saving Report as "+PrivDir()+file.a,49,0,0)
               Run NOREFRESH "Copy " + screenFile.a + " " + PrivDir()+file.a
            ENDIF
            QUITLOOP
         ENDWHILE
   ENDSWITCH
   IF print.l THEN
      IF pause.l THEN
         msContinue.u("Make sure Printer is Ready with the proper paper",79,999,999,"",2)
      ENDIF
      SetPrinter default.a
      ioPrinterStatus.l()                ; Check printer status
      IF retval THEN
         {Report} {SetPrinter} {Override} {Setup} Select setup.a
         Report table.a report.a
      ENDIF
   ENDIF
   msWorkingClear.u()
   Return true
ENDPROC
WriteLib libname.a ioSelectOutputProcess.l
?? "\004"
; ============================================================================
;       TITLE: ioUserLogin.l             (c) 1992, 1993 DataStar International
;     RETURNS: GUEST, CANCEL, ERROR, LOCKED or UserRecord Array
; DESCRIPTION: Generic routine for accepting password from user.
; ----------------------------------------------------------------------------
PROC ioUserLogin.l(              ; Generic User/Password Login
         topRow.n,               ; Top Row for Box
         dBoxPalette.a,          ; Custom dBox Color Palette
         table.a,                ; Users Password Table
         a,                      ; Password for Password Table
         password.l,             ; Get Password?
         misses.n,               ; How many misses allowed
         case.l,                 ; Case Sensitive?
         guest.l)                ; Guest Rights if Failed?
Private  miss.n,
         startTicks.n,
         user.a,
         pushButton.l,
         oldColors.y,
         leftCol.n,
         count.n,
         dBoxProcs.y,
         userName.a
   count.n = 0
   Lock table.a PFL
   WHILE NOT retval AND count.n < 5
      Sleep 1000
      count.n = count.n + 1
      Lock table.a PFL
   ENDWHILE
   IF NOT retval THEN
      user.a = "LOCKED"
   ELSE
      SetCanvas DEFAULT
      Password a
      View table.a
      UnPassword a
      a = ""
      Right
      miss.n = 1
      user.a = "USER"
      userName.a = ""
      pushButton.l = false
      startTicks.n = Ticks()
      SWITCH
         CASE guest.l AND password.l :
            msWorking.u("Enter User Name and Password - If GUEST, ignore Password",111,0,0)
         CASE password.l :
            msWorking.u("Enter your User Name and Password",111,0,0)
         CASE NOT password.l :
            msWorking.u("Enter your User Name",111,0,0)
      ENDSWITCH
      DynArray dBoxProcs.y[]
         dBoxProcs.y["KEY"]    = "ioUserLoginKey.l"
         dBoxProcs.y["ACCEPT"] = "ioUserLoginAccept.l"
      leftCol.n = 19
      SWITCH
         CASE password.l AND case.l       :
            SHOWDIALOG "Enter Your Username and Password"
               PROC "dbEventHandler.l"
                  Key 13
                  Trigger "ACCEPT", "OPEN"
                  Idle
               @ -200,-200  Height 9 Width 42

               LABEL @ 1,7 "~U~sername:" For "USERNAME"
               Accept @ 1,17 Width 15 "A12"
                  Tag "USERNAME"
               To userName.a
               LABEL @ 3,7 "~P~assword:" For "A"
               Accept @ 3,17 Width 15 "A12"
                  Hidden Tag "A"
               To a

               PushButton @ 5,3 Width 16 "~OK~ Proceed"
                  OK DEFAULT Value true Tag "BUTTON"
               To pushButton.l
               PushButton @ 5,21 Width 16 "~<Esc>~ Cancel"
                  Cancel Value false Tag "BUTTON"
               To pushButton.l
            ENDDIALOG
         CASE password.l AND NOT case.l   :
            topRow.n = 4
            leftCol.n = 18
            frameTag.a = "USERNAME"
            SHOWDIALOG "Enter Username and Password"
               PROC "dbEventHandler.l"
                  Trigger "OPEN", "ARRIVE", "ACCEPT"
                  Key 13
                  Idle
               @-200,-200 Height 17 Width 44

               Frame Single From 2,6 To 6,35
                  PaintCanvas Attribute IIF(frameTag.a="BUTTON",112,127)
                              2,6,6,35
                  PaintCanvas Attribute IIF(frameTag.a="BUTTON",127,112)
                              2,6,2,34
                  PaintCanvas Attribute IIF(frameTag.a="BUTTON",127,112)
                              2,6,6,6

               Frame Single From 8,6 To 12,35
                  PaintCanvas Attribute IIF(frameTag.a="BUTTON",127,112)
                              8,6,12,35
                  PaintCanvas Attribute IIF(frameTag.a="BUTTON",112,127)
                              8,6,8,34
                  PaintCanvas Attribute IIF(frameTag.a="BUTTON",112,127)
                              8,6,12,6

               Label @ 3,8 "~U~sername:" For "USERNAME"
               Accept @ 3,18 Width 15 "A12"
                  Picture "*!" Tag "USERNAME"
               To userName.a

               Label @ 5,8 "~P~assword:" For "A"
               Accept @ 5,18 Width 15 "A12"
                  Picture "*!" Hidden Tag "A"
               To a

               PushButton @ 10,8 Width 12 "~OK~-Login"
                  OK DEFAULT Value true Tag "BUTTON"
               To button.l

               PushButton @ 10,22 Width 12 "~Esc~-Quit"
                  Cancel Value false Tag "BUTTON"
               To button.l
            ENDDIALOG
         CASE NOT password.l AND case.l   :
            SHOWDIALOG "Enter Your Username"
               PROC "dbEventHandler.l"
                  Key 13
                  Trigger "ACCEPT", "OPEN"
                  Idle
               @ -200,-200  Height 9 Width 42

               LABEL @ 1,7 "~U~sername:" For "USERNAME"
               Accept @ 1,17 Width 15 "A12"
                  Tag "USERNAME"
               To userName.a

               PushButton @ 5,3 Width 16 "~OK~ Proceed"
                  OK DEFAULT Value true Tag "BUTTON"
               To pushButton.l
               PushButton @ 5,21 Width 16 "~<Esc>~ Cancel"
                  Cancel Value false Tag "BUTTON"
               To pushButton.l
            ENDDIALOG
         OTHERWISE                        :
            SHOWDIALOG "Enter Your Username"
               PROC "dbEventHandler.l"
                  Key 13
                  Trigger "ACCEPT", "OPEN"
                  Idle
               @ -200,-200  Height 9 Width 42

               LABEL @ 1,7 "~U~sername:" For "USERNAME"
               Accept @ 1,17 Width 15 "A12"
                  Picture "*!" Tag "USERNAME"
               To userName.a

               PushButton @ 5,3 Width 16 "~OK~ Proceed"
                  OK DEFAULT Value true Tag "BUTTON"
               To pushButton.l
               PushButton @ 5,21 Width 16 "~<Esc>~ Cancel"
                  Cancel Value false Tag "BUTTON"
               To pushButton.l
            ENDDIALOG
      ENDSWITCH
   ENDIF
   IF NOT retval THEN
      msWorking.u("The Login has been Canceled",79,1,2)
      user.a = "CANCEL"
   ENDIF
   g.config.y["logName.a"] = IIF(user.a = "USER", g.config.y["logName.a"], user.a)
   UnLock table.a PFL
   ClearImage
   msWorkingClear.u()
   Return user.a = "USER" OR (guest.l AND user.a = "GUEST")
ENDPROC
WriteLib libname.a ioUserLogin.l
?? "\004"
; ============================================================================
;       TITLE: ioUserLoginAccept.l       (c) 1992, 1993 DataStar International
;     RETURNS: GUEST, USER, CANCEL, ERROR
; DESCRIPTION: Generic routine for accepting password from user.
; ----------------------------------------------------------------------------
PROC ioUserLoginAccept.l()       ; User Login Validation DialogBox Proc
Private  l,
         retval.l,
         pwMsg.a,
         user.r
;Global  startTicks.n
;        user.a
;        userName.a
;        miss.n
;        misses.n
;        guest.l
;        topRow.n
;        a
   retval.l = true
   IF password.l THEN
      pwMsg.a = "Invalid Username/Password Combination"
   ELSE
      pwMsg.a = "Invalid Username"
   ENDIF
   SWITCH
      CASE IsBlank(userName.a) :
         SelectControl "USERNAME"
         Message "Please Enter a Username, or select <Cancel>..."
         retval.l = false
      CASE password.l AND IsBlank(a) AND NOT guest.l :
         SelectControl "A"
         Message "Please Enter a Password..."
         retval.l = false
      OTHERWISE :
         Locate userName.a
         l = retval
         Right
         IF NOT l OR (password.l AND [] <> a) THEN
            SelectControl "USERNAME"
            SWITCH
               CASE miss.n < misses.n AND guest.l :
                  msConfirm.l("",pwMsg.a+" - Are you a Guest? ",
                              79,"RED",1,"~T~ry Again","~G~uest",true)
                  IF NOT retval THEN
                     user.a = "GUEST"
                     g.config.y["logName.a"]             = userName.a
                     g.config.y["fullName.a"]            = "GUEST: " + userName.a
                     g.config.y["accessLevel.a"]         = "GUEST"
                     g.config.y["printer.a"]             = ""
                     g.config.y["port.a"]                = ""
                     g.config.y["pica.a"]                = ""
                     g.config.y["elite.a"]               = ""
                     g.config.y["condensed.a"]           = ""
                     g.config.y["compressed.a"]          = ""
                     g.config.y["picaLandscape.a"]       = ""
                     g.config.y["eliteLandscape.a"]      = ""
                     g.config.y["condensedLandscape.a"]  = ""
                     g.config.y["compressedLandscape.a"] = ""
                     g.config.y["boldOn.a"]              = ""
                     g.config.y["boldOff.a"]             = ""
                     g.config.y["reset.a"]               = ""
                     View "SYS\\Printers"   ; First Field Must be Printer Name/ID
                     Array g.printers.r[NImageRecords()]
                     Right
                     SCAN
                        g.printers.r[RecNo()] = []
                     ENDSCAN
                     ClearImage
                  ELSE
                     miss.n = miss.n + 1
                     retval.l = false
                     Left
                  ENDIF
               CASE miss.n < misses.n :
                  miss.n = miss.n + 1
                  Beep Beep Beep
                  Message pwMsg.a + " - Please try again..."
                  retval.l = false
                  Left
               CASE guest.l   :
                  msContinue.u("","You are not a Registered User " +
                                    "- You are granted only Guest Rights",
                                 79, "RED",3)
                  user.a = "GUEST"
               OTHERWISE      :
                  msContinue.u("","Sorry, too many Errors for User Name - Returning",
                              79, "RED", 5)
                  user.a = "ERROR"
            ENDSWITCH
         ELSE
         ;  The User Login Table must have the following virtue structure,
         ;  regardless of field name
         ;        Login Name
         ;        Login Password
         ;        Full User Name
         ;        Security Level/Group/Rights
         ;        Default Printer Name
         ;        Default Printer Port
         ;        Portrait Pica        (10 cpi)
         ;        Portrait Elite       (12 cpi)
         ;        Portrait Condensed   (16-17 cpi)
         ;        Portrait Compressed  (16-17 cpi, 8 lpi)
         ;        Bold On
         ;        Bold Off
         ;        Landscape Pica       (10 cpi)
         ;        Landscape Elite      (12 cpi)
         ;        Landscape Condensed  (16-17 cpi)
         ;        Landscape Compressed (16-17 cpi, 8 lpi)
         ;        Reset
         ;        Any Other Fields

            CopyToArray user.r
            user.r[3]                           = "***pw***"
            g.config.y["logName.a"]             = user.r[2]
            g.config.y["fullName.a"]            = user.r[4]
            g.config.y["accessLevel.a"]         = user.r[5]
            g.config.y["printer.a"]             = user.r[6]
            g.config.y["port.a"]                = user.r[7]
            g.config.y["pica.a"]                = user.r[8]
            g.config.y["elite.a"]               = user.r[9]
            g.config.y["condensed.a"]           = user.r[10]
            g.config.y["compressed.a"]          = user.r[11]
            g.config.y["picaLandscape.a"]       = user.r[12]
            g.config.y["eliteLandscape.a"]      = user.r[13]
            g.config.y["condensedLandscape.a"]  = user.r[14]
            g.config.y["compressedLandscape.a"] = user.r[15]
            g.config.y["boldOn.a"]              = user.r[16]
            g.config.y["boldOff.a"]             = user.r[17]
            g.config.y["reset.a"]               = user.r[18]
            View "SYS\\Printers"   ; First Field Must be Printer Name/ID
            Array g.printers.r[NImageRecords()]
            Right
            SCAN
               g.printers.r[RecNo()] = []
            ENDSCAN
            ClearImage
         ENDIF
   ENDSWITCH
   Return retval.l
ENDPROC
WriteLib libname.a ioUserLoginAccept.l
?? "\004"
; ============================================================================
;       TITLE: ioUserLoginKey.l          (c) 1992, 1993 DataStar International
;     RETURNS: GUEST, USER, CANCEL, ERROR
; DESCRIPTION: Generic routine for accepting password from user.
; ----------------------------------------------------------------------------
PROC ioUserLoginKey.l()          ; User Login Validation DialogBox Proc
;Global  tag.a,
;        event.v
   IF tag.a = "USERNAME" THEN
      event.v["KeyCode"] = 9
      Release Vars event.v["ScanCode"]
   ENDIF
   Return true
ENDPROC
WriteLib libname.a ioUserLoginKey.l
?? "\004"
