; ****************************************************************************
; SCRIPT NAME: ds4_DATE.sc               (c) 1992, 1993 DataStar International
; APPDEV TEAM: Dan Paolini, David Kelton, Micah Bleecher, Patty Paolini
; APPLICATION: DataStar Utilities for Paradox 4
; DESCRIPTION: Contains the following Date and/or Time procedures:
;                 dtInsertSlashes.a
;                 dtInsertSlashesYearMonthDay.a
;                 dtSerialCreate.a       
;                 dtSerialDays.a         
;                 dtSerialMinutes.a      
;                 dtSerialNow.a          
;                 dtSerialRange.l        
;                 dtSerialRestore.a      
;                 dtSystemTime.u
;                 dtSystemDateSet.l
;                 dtSystemTimeSet.l
;                 dtSystemTimePosition.u
;                 dtSystemTimeUpdate.l
;                 dtTicksToMinutes.n
;                 dtTimeToMinutes.v
;                 dtTwelveHourTime.a
; ============================================================================
? Format("w40"," ds4_DATE.sc - Date/TIme Routines")
@ Row(),40
IF NOT IsAssigned(libname.a) THEN
   Beep Beep Beep QUIT "Variable \"libname.a\" is not assigned!.."
ENDIF
IF NOT IsFile(libname.a+".LIB") THEN
   CreateLib libname.a Size 640
ENDIF
; ============================================================================
;       TITLE: dtInsertSlashes.a         (c) 1992, 1993 DataStar International
;     RETURNS: An 8 character string with slashes in the 3rd and 6th position
; DESCRIPTION: Adds slashes to six-character date string
; ----------------------------------------------------------------------------
PROC dtInsertSlashes.a(          ; Inserts slashes into six-character Date
         date.a)                 ; Date to convert
Private  retval.a
   retval.a = SubStr(date.a,1,2) + "/" + SubStr(date.a,3,2) + "/" +
              SubStr(date.a,5,2)
   Return IIF(DateVal(retval.a) = "Error","",retval.a)
ENDPROC
?? "\004"
WRITELIB libname.a dtInsertSlashes.a
; ===========================================================================
;       TITLE: dtInsertSlashesYearMonthDay.a       (c) 1992, 1993 DataStar International
;     RETURNS: An 8 character string with slashes in the 3rd and 6th position
;              Or "Error" if not a Valid Date
; DESCRIPTION: Adds slashes to six-character date string
; ---------------------------------------------------------------------------
PROC dtInsertSlashesYearMonthDay.a( ; Inserts slashes into six-character Date
         date.a)                 ; Date to convert
Private  retval.a
   retval.a = SubStr(date.a,3,2) + "/" + SubStr(date.a,5,2) + "/" +
              SubStr(date.a,1,2)
   Return IIF(DateVal(retval.a) = "Error","",retval.a)
ENDPROC
?? "\004"
WRITELIB libname.a dtInsertSlashesYearMonthDay.a
; ===========================================================================
;       TITLE: dtSerialCreate.a       (c) 1992, 1993 DataStar International
;     RETURNS: Days and minutes since 1/1/1900 as an A10 string.
; DESCRIPTION: Converts date and time passed to a serialized date/time string
;              from the date 1/1/1900.  Time may be passed in numeric minutes
;              from midnight or an A4 military time string.  Example: A date
;              of 10/19/92, and a time of 0100 or 60 returns "33894.0060" .
; ---------------------------------------------------------------------------
PROC dtSerialCreate.a(           ; Creates serial date/time string
         date.d,                 ; Date to convert
         time.v)                 ; Minutes from midnight or military time
   RETURN StrVal(date.d-1/1/1900) + "." +
          SubStr(Format("w5,ez",IIF(Type(time.v) = "A4",
                                    dtTimeToMinutes.v(time.v), time.v)),2,4)
ENDPROC
?? "\004"
WRITELIB libname.a dtSerialCreate.a
; ===========================================================================
;       TITLE: dtSerialDays.a          (c) 1992, 1993 DataStar International
;     RETURNS: Alphanumeric hypertime or "Error" -  "A10"
; DESCRIPTION: Based on dtSerialCreate.a, adds or subtracts days to serial date
;              on the passed hypertime.
; ---------------------------------------------------------------------------
PROC dtSerialDays.a(             ; Add/Subtract days to Serial date
         serialDate.a,           ; Hypertime string
         days.n,                 ; Days to add
         add.l)                  ; True = Add, False = Subtract
Private  retval.a,               ; Return variable
         a2, a1                  ; Transient Match variables
   IF Match(serialDate.a,"..\".\"..",a1,a2) THEN
      retval.a = StrVal(NumVal(a1) + IIF(add.l,days.n,-days.n)) + "." + a2
   ELSE
      retval.a = "Error"
   ENDIF
   Return retval.a
ENDPROC
?? "\004"
WRITELIB libname.a dtSerialDays.a
; ============================================================================
;       TITLE: dtSerialMinutes.a         (c) 1992, 1993 DataStar International
;     RETURNS: Serial Date/Time String (A10) or "Error"
; DESCRIPTION: Based on dtSerialCreate.a, adds or subtracts minutes to time
; ----------------------------------------------------------------------------
PROC dtSerialMinutes.a(          ; Adds or Subtracts minutes from serial date
         serialDate.a,           ; Serial Date/Time string
         minutes.n,              ; Minutes to adjust
         add.l)                  ; True = Add, False = Subtract
Private  retval.a,               ; Return variable
         time.n,                 ; Time portion of serialdate
         date.n,                 ; Date portion of serialdate
         a2, a1                  ; Transient Match variables
   IF Match(serialDate.a,"..\".\"..",a1,a2) THEN
      date.n = NumVal(a1)
      time.n = NumVal(a2)
      IF add.l THEN
         IF minutes.n + time.n > 1439 THEN
            date.n = date.n + Int((minutes.n + time.n)/1440)
            time.n = Mod(minutes.n + time.n,1440)
         ELSE
            time.n = time.n + minutes.n
         ENDIF
      ELSE
         IF time.n - minutes.n < 0 THEN
            date.n = date.n + Int((minutes.n + time.n)/1440) - 1
            time.n = 1440 + Mod(time.n - minutes.n,1440)
         ELSE
            time.n = time.n - minutes.n
         ENDIF
      ENDIF
      retval.a = StrVal(date.n) + "." +
                 SubStr(Format("w5,ez",time.n),2,4)
   ELSE
      retval.a = "Error"
   ENDIF
   Return retval.a
ENDPROC
?? "\004"
WRITELIB libname.a dtSerialMinutes.a
; ============================================================================
;       TITLE: dtSerialNow.a             (c) 1992, 1993 DataStar International
;     RETURNS: Alphanumeric "A10"
; DESCRIPTION: Based on dtSerialCreate.a, returns current date/time as serial
; ----------------------------------------------------------------------------
PROC dtSerialNow.a()
   Return dtSerialCreate.a(Today(),(SubStr(Time(),1,2)+SubStr(Time(),3,2)))
ENDPROC
?? "\004"
WRITELIB libname.a dtSerialNow.a
; ============================================================================
;       TITLE: dtSerialRange.l           (c) 1992, 1993 DataStar International
;     RETURNS: Logical true/false if date false between INCLUSIVE range
; DESCRIPTION: Based on dtSerialCreate.a, this procedure determines if
;              a serialdate falls on or between two serialdates.  Begindate.a
;              is starting serialdate and enddate.a is ending serialdate.
;              checkdate.a is the serialdate that is being checked.
; ----------------------------------------------------------------------------
PROC dtSerialRange.l(            ; Checks if serial date falls within range
         beginDate.a,            ; Beginning serial date
         endDate.a,              ; Ending serial date
         checkDate.a)            ; Serial date to check
Private  retval.l                ; Variable to return
   Return (checkDate.a >= beginDate.a AND checkDate.a <= endDate.a)
ENDPROC
?? "\004"
WRITELIB libname.a dtSerialRange.l
; ============================================================================
;       TITLE: dtSerialRestore.a         (c) 1992, 1993 DataStar International
;     RETURNS: Alphanumeric string "A13"
; DESCRIPTION: Based on dtSerialCreate.a, restores serial date/time back to
;              alphanumeric representation of date and minutes.  Example:
;              "33894.0060" would return "10/19/92.0060". The resulting
;              string could be parsed out with the match function.  Example:
;                 IF Match(datetime.a,"..\".\"..",a1,a2) THEN
;                    date.d = DateVal(a1)    ; Date
;                    minutes.n = NumVal(a2)  ; Minutes from midnight
;                 ENDIF
; ----------------------------------------------------------------------------
PROC dtSerialRestore.a(          ; Changes serial time to alphanumeric
         serialDate.a)           ; Serial date/time string
Private  retval.a,               ; Rreturn variable
         a1,a2                   ; Match variables
   IF Match(serialDate.a,"..\".\"..",a1,a2) THEN
      retval.a = StrVal(1/1/1900 + NumVal(a1)) + "." + a2
   ELSE
      retval.a = "Error"
   ENDIF
   Return retval.a
ENDPROC
?? "\004"
WRITELIB libname.a dtSerialRestore.a
; ============================================================================
;       TITLE: dtSystemTime.u            (c) 1992, 1993 DataStar International
;     RETURNS: No Value
; DESCRIPTION: System Time and Date Dialog Box
; ----------------------------------------------------------------------------
PROC dtSystemTime.u()            ; Generic Time and Date DialogBox
Private  time.a,
         ticks.n,
         dBoxPalette.a,
         dBoxProcs.y,
         icon.a,
         button.l,               ; Value of selected Pushbutton
         frameHigh.n,
         frameLow.n

   SetCanvas DEFAULT
   IF NOT IsAssigned(g.sysinfo.y) THEN
      SysInfo to g.sysinfo.y
   ENDIF

   time.a = dtTwelveHourTime.a(true)
   ticks.n = Ticks()
   dBoxPalette.a = "CYAN"
   DynArray dBoxProcs.y[]
      dBoxProcs.y["IDLE"] = "dtSystemTimeUpdate.l"
   icon.a = "   12    " +
            "         " +
            " 9    3 " +
            "         " +
            "    6    "
   frameHigh.n = 59
   frameLow.n  = 48
   button.l    = true
   topRow.n    = 7
   leftCol.n   = Int((g.sysinfo.y["ScreenWidth"]-60)/2)

   SHOWDIALOG "System Date and Time"
      Proc "dbEventHandler.l"
         Idle
         Trigger "OPEN"    ; Wait for Key Alert
      @ -200,-200
      Height 11 Width 60

      Frame From 1,1 To 7,11
         PaintCanvas Border Attribute 16 1,1,7,11
         PaintCanvas Border Attribute 25 1,1,1,10
         PaintCanvas Border Attribute 25 1,1,7,1
         PaintCanvas Fill icon.a
                     Attribute 30 2,2,6,10

      Frame From 1,14 To 7,44
         PaintCanvas Border Attribute frameHigh.n 1,14,7,44
         PaintCanvas Border Attribute frameLow.n  1,14,1,43
         PaintCanvas Border Attribute frameLow.n  1,14,7,14
         PaintCanvas Fill Spaces(27) +
                          Format("w27,ac","Today is " +
                          DOW(Today()) + ", " + Format("D5",Today())) +
                          Spaces(27) +
                          Format("w27,ac","The Time is " + time.a) +
                          Spaces(27)
                     Attribute 63 2,16,6,42

      PushButton @ 2,45
         Width 12 "~C~ontinue"
         OK Default Value dbButtonPress.v(true) Tag "OK"
      To button.l
      PushButton @ 4,45
         Width 12 "Set ~D~ate"
         Value dtSystemDateSet.l() Tag "SET.DATE"
      To button.l
      PushButton @ 6,45
         Width 12 "Set ~T~ime"
         Value dtSystemTimeSet.l() Tag "SET.TIME"
      To button.l
   ENDDIALOG
   msWorkingClear.u()
   Return
ENDPROC
?? "\004"
WriteLib libname.a dtSystemTime.u
; ============================================================================
;       TITLE: dtSystemDateSet.l         (c) 1992, 1993 DataStar International
;     RETURNS: No Value
; DESCRIPTION: System Time and Date Dialog Box
; ----------------------------------------------------------------------------
PROC dtSystemDateSet.l()         ; Generic Time Set Utility
Private  button.l,
         accept.d
   accept.d = Today()
   SHOWDIALOG "New Date"
      @ 13,54
      Height 7 Width 18

      Accept @ 1,3 Width 10 "D"
         Tag "ACCEPT"
      To accept.d

      PushButton @ 3,5
         Width 6 "~O~K"
         OK Default Value true Tag "OK"
      To button.l
   ENDDIALOG

   Run NOREFRESH "DATE " + StrVal(accept.d)
   Return true
ENDPROC
?? "\004"
WriteLib libname.a dtSystemDateSet.l
; ============================================================================
;       TITLE: dtSystemTimeSet.l         (c) 1992, 1993 DataStar International
;     RETURNS: No Value
; DESCRIPTION: System Time and Date Dialog Box
; ----------------------------------------------------------------------------
PROC dtSystemTimeSet.l()         ; Generic Time Set Utility
Private  button.l,
         accept.a
   accept.a = SubStr(time.a,1,5)
   IF Search("AM",time.a) <> 0 THEN
      accept.a = SubStr(Format("w3,ez",NumVal(SubStr(accept.a,1,2))-12),2,2) +
                 SubStr(accept.a,3,3)
   ENDIF
   SHOWDIALOG "New Time"
      @ 15,54
      Height 7 Width 18

      Accept @ 1,4 Width 8 "A5"
         Picture "##:##"
         Tag "ACCEPT"
      To accept.a

      PushButton @ 3,5
         Width 6 "~O~K"
         OK Default Value true Tag "OK"
      To button.l
   ENDDIALOG

   Run NOREFRESH "TIME " + accept.a
   time.a = dtTwelveHourTime.a(true)
   Return true
ENDPROC
?? "\004"
WriteLib libname.a dtSystemTimeSet.l
; ============================================================================
;       TITLE: dtSystemTimePosition.u    (c) 1992, 1993 DataStar International
;     RETURNS: No Value
; DESCRIPTION: System Time and Date Dialog Box
; ----------------------------------------------------------------------------
PROC dtSystemTimePosition.u(     ; Generic Time and Date DialogBox
         topRow.n,               ; Top Row (999 = Centered Vertically)
         leftCol.n)              ; Left Column (999 = Centered Horizontally)
Private  time.a,
         ticks.n,
         button.a,               ; Value of selected Pushbutton
         dBoxPalette.a,
         dBoxProcs.y

   SetCanvas DEFAULT
   IF NOT IsAssigned(g.sysinfo.y) THEN
      SysInfo to g.sysinfo.y
   ENDIF
   topRow.n = 7
   leftCol.n = Int((g.sysinfo.y["ScreenWidth"]-32)/2)
   time.a = dtTwelveHourTime.a(true)
   ticks.n = Ticks()
   dBoxPalette.a = "GRAY"
   DynArray dBoxProcs.y[]
      dBoxProcs.y["IDLE"] = "dtSystemTimeUpdate.l"

   SHOWDIALOG "System Time"
      Proc "dbEventHandler.l" IDLE
      @ topRow.n, leftCol.n
      Height 7 Width 32

      @ 1,2 ?? time.a + "  " + DOW(Today()) + ", " + Format("D5",Today())

      PushButton @ 3,9 Width 12 "~C~ontinue"
         OK Default Value "OK" Tag "OK"
      To button.a
   ENDDIALOG
   Return
ENDPROC
?? "\004"
WriteLib libname.a dtSystemTimePosition.u
; ============================================================================
;       TITLE: dtSystemTimeUpdate.l      (c) 1992, 1993 DataStar International
;     RETURNS: No Value
; DESCRIPTION: System Time and Date Dialog Box
; ----------------------------------------------------------------------------
PROC dtSystemTimeUpdate.l()      ; Generic Time and Date DialogBox
;Global  ticks.n
   IF ticks.n < Ticks() - 990 THEN
      time.a = dtTwelveHourTime.a(true)
      ticks.n = Ticks()
   ENDIF
   Return true
ENDPROC
?? "\004"
WriteLib libname.a dtSystemTimeUpdate.l
; ============================================================================
;       TITLE: dtTicksToMinutes.n()      (c) 1992, 1993 DataStar International
;     RETURNS: Numeric
; DESCRIPTION: Converts Ticks() to  minutes from midnight
; ----------------------------------------------------------------------------
PROC dtTicksToMinutes.n()
   Return Int((Ticks() / 1000) / 60)
ENDPROC
?? "\004"
WriteLib libname.a dtTicksToMinutes.n
; ============================================================================
;       TITLE: dtTimeToMinutes.n         (c) 1992, 1993 DataStar International
;     RETURNS: Returns Numeric
; DESCRIPTION: Accepts Numeric parameter in military time and calculates
;              the minutes from midnight.
; ----------------------------------------------------------------------------
PROC dtTimeToMinutes.v(
         time.v)
   IF Search(Type(time.v),"SN$") = 0 THEN
      time.v = NumVal(time.v)
   ENDIF
   Return IIF(time.v = "Error","Error",(Int(time.v/100) * 60) +
                                       Mod(time.v,100))
ENDPROC
?? "\004"
WriteLib libname.a dtTimeToMinutes.v
; ============================================================================
;       TITLE: dtTwelveHourTime.a        (c) 1992, 1993 DataStar International
;     RETURNS: String of current 12 hour time
; DESCRIPTION: seconds.l=true - include seconds
; ----------------------------------------------------------------------------
PROC dtTwelveHourTime.a(         ; Returns 12-Hour Time from System Time
         seconds.l)              ; Include Seconds?
Private  hours.a,                ; Holds current hours as string
         n,                      ; Variable index for SubStr() function
         time.a
   n = IIF(seconds.l,8,5)
   time.a = Time()
   hours.a = Substr(time.a,1,2)
   Return IIF(hours.a > "11",
            IIF(hours.a <> "12",
               StrVal(NumVal(hours.a)-12) + Substr(time.a,3,n-2) + " pm",
               hours.a + Substr(time.a,3,n-2) + " pm"),
            IIF(hours.a = "00","12"+Substr(time.a,3,n-2) + " am",
               IIF(Search("0",time.a) = 1,
                  SubStr(time.a,2,n-1) + " am",
                  SubStr(time.a,1,n) + " am")))
ENDPROC
?? "\004"
WriteLib libname.a dtTwelveHourTime.a
