; ****************************************************************************
; SCRIPT NAME: ds4_INFO.sc               (c) 1992, 1993 DataStar International
; APPDEV TEAM: Dan Paolini, David Kelton, Micah Bleecher, Patty Paolini
; APPLICATION: DataStar Utilities for Paradox 4
; DESCRIPTION: Contains the following Information procedures:
;                 inAllFieldsChecked.l
;                 inAttributeConvert.n
;                 inAttributeForeground.n
;                 inBackSlashDouble.a
;                 inCanvasColorPalette.n
;                 inErrorHandler.n
;                    epErrorReset.n
;                 inErrorLog.u
;                 inLeapYear.l
;                 inNextNumber.v
;                 inNextNumberbyYear.v
;                 inNotAvailable.u
;                 inBackSlashDouble.a
;                 inPathStrip.a
;                 inQuotesEmbedded.a
;                 inRound.n
;                 inSmartKey.l
;                 inStartUp.l
;                 inStartupColors.u
;                 inTrimSpaces.a
;                 inUserLog.u
;                 inWordWrap.a
; ============================================================================
? Format("w40"," ds4_INFO.sc - Information Routines")
IF NOT IsAssigned(libName.a) THEN
   Beep Beep Beep QUIT "Variable \"libName.a\" is not assigned!.."
ENDIF
IF NOT IsFile(libName.a+".LIB") THEN
   CreateLib libName.a Size 640
ENDIF
@ Row(),40
; ============================================================================
;       TITLE: inAllFieldsChecked.l()    (c) 1992, 1993 DataStar International
;     RETURNS: Logical true/false if all fields are checked
; DESCRIPTION: Checks all query fields to determine if all have a positive
;              check mark (check, checkplus or groupby)
; ----------------------------------------------------------------------------
PROC inAllFieldsChecked.l(
         fieldOrder.r,           ; Sequential order of fields
         checkStatus.y,          ; Check mark status of the field
         columns.n)              ; Number of columns in the image
Private  firstCheck.a,           ; Status of the first field
         retval.l,               ; Return variable
         n                       ; Transient loop counter
   firstCheck.a = checkStatus.y[fieldOrder.r[2]]
   retval.l = false
   IF NOT IsBlank(firstCheck.a) THEN
      retval.l = true
      FOR n From 2 To columns.n
         IF firstCheck.a <> checkStatus.y[fieldOrder.r[n]] THEN
            retval.l = false
            QUITLOOP
         ENDIF
      ENDFOR
   ENDIF
   Return retval.l
ENDPROC
?? "\004"
WriteLib libName.a inAllFieldsChecked.l
; ============================================================================
;       TITLE: inAttributeConvert.n      (c) 1992, 1993 DataStar International
;     RETURNS: Color attribute
; DESCRIPTION: Returns either the intense foreground of a background color if
;              highlight.l = true, else black on background color.
; ----------------------------------------------------------------------------
PROC inAttributeConvert.n(       ; Converts color into highlight or lowlight
         color.n,                ; Background color
         highlight.l)            ; True=highlight, false=lowlight
   Return (Int(color.n/16)*16) + IIF(highlight.l,Int(color.n/16)+8,0)
ENDPROC
?? "\004"
WriteLib libName.a inAttributeConvert.n
; ============================================================================
;       TITLE: inAttributeForeground.n   (c) 1992, 1993 DataStar International
;     RETURNS: Color attribute
; DESCRIPTION: Returns the designated foreground on the background of color,
;              which is usually passed as a SysColor() or GetColors element.
; ----------------------------------------------------------------------------
PROC inAttributeForeground.n(    ; Returns specified foreground on background
         color.n,                ; Background color
         foreground.n)           ; Foreground color to return
   Return (Int(color.n/16)*16) + foreground.n
ENDPROC
?? "\004"
WriteLib libName.a inAttributeForeground.n
; ============================================================================
;       TITLE: inBackSlashDouble.a       (c) 1992, 1993 DataStar International
;     RETURNS: No Value
; DESCRIPTION: Doubles backslashes in a String
; ----------------------------------------------------------------------------
PROC inBackSlashDouble.a(        ; Doubles backslashes in a string
         path.a)                 ; Path to double
Private  a1, a2, a3              ; Transient string variables
   a1 = path.a
   a2 = ""
   WHILE Match(a1,"..\\..",a3,a1)
      a2 = a2 + a3 +"\\\\"
   ENDWHILE
   Return IIF(a1 = path.a, path.a, a2 + a1)
ENDPROC
WriteLib libName.a inBackSlashDouble.a
?? "\004"
; ============================================================================
;       TITLE: inCanvasColorPalette.n    (c) 1992, 1993 DataStar International
;     RETURNS: Attribute selected, or 256 if cancelled
; DESCRIPTION: Color Attribute Selection Box
; -----------------------------------------------------------------------------
PROC inCanvasColorPalette.n()    ; Color Attribute selection box
Private  column.n,               ; Column location for placement on screen
         n1, n2,                 ; Loop counters
         cursorRow.n,            ; Tracks cursor row position
         cursorColumn.n,         ; Tracks cursor column position
         key.n,                  ; Key pressed
         attribute.n
   Canvas OFF
   SyncCursor
   IF Col() > 41 THEN
      column.n = 6
   ELSE
      column.n = 56
   ENDIF
   cursorRow.n = 0
   cursorColumn.n = 0
   PaintCanvas Attribute 8 6, column.n+2, 19, column.n + 19
   PaintCanvas Fill "Ŀ" +
                    "Select Attribute" +
                    "Ĵ" +
                    "" +
                    "" +
                    "" +
                    "" +
                    "" +
                    "" +
                    "" +
                    "" +
                    "Ĵ" +
                    " Attribute #    " +
                    ""
               Attribute 63 5, column.n, 18, column.n+17
   FOR n1 From 1 To 8
      FOR n2 From 1 To 16
         PaintCanvas Attribute ((n1-1)*16)+n2-1  7+n1,column.n+n2,7+n1,column.n+n2
      ENDFOR
   ENDFOR
   WHILE true
      Style Attribute 62
      @ 17, column.n+13 ?? SubStr(Format("w4,ar",(cursorRow.n*16)+cursorColumn.n),2,3)
      Style                                     ; Set Blinking Attribute
      PaintCanvas Attribute (cursorRow.n*16)+cursorColumn.n+128
                            cursorRow.n+8, cursorColumn.n+column.n+1,
                            cursorRow.n+8, cursorColumn.n+column.n+1
      Canvas ON
      Canvas OFF
      key.n = GetChar()                         ; Remove Blinking Attribute
      PaintCanvas Attribute (cursorRow.n*16)+cursorColumn.n
                            cursorRow.n+8, cursorColumn.n+column.n+1,
                            cursorRow.n+8, cursorColumn.n+column.n+1
      SWITCH                                    ; Evaluate Keypress
         CASE key.n = -75  : cursorColumn.n = cursorColumn.n - 1
         CASE key.n = -77  : cursorColumn.n = cursorColumn.n + 1
         CASE key.n = -72  : cursorRow.n = cursorRow.n - 1
         CASE key.n = -80  : cursorRow.n = cursorRow.n + 1
         CASE key.n = -71  : cursorColumn.n = 0
         CASE key.n = -79  : cursorColumn.n = 15
         CASE key.n = -119 : cursorRow.n = 0
         CASE key.n = -117 : cursorRow.n = 7
         CASE key.n =  13  : QUITLOOP
         CASE key.n =  27  : Canvas ON Return 256
      ENDSWITCH
      SWITCH                                    ; Out of Range trap
         CASE cursorRow.n = -1      : cursorRow.n = 7
         CASE cursorRow.n =  8      : cursorRow.n = 0
         CASE cursorColumn.n = -1   : cursorColumn.n = 15
         CASE cursorColumn.n = 16   : cursorColumn.n = 0
      ENDSWITCH
   ENDWHILE
   attribute.n = (cursorRow.n*16) + cursorColumn.n
   msConfirm.l("","Make the Attribute Blink?",attribute.n,"GRAY",1,
                  "~N~o Blink","~B~linking",false)
   IF retval THEN
      attribute.n = attribute.n + 128
   ENDIF
   Canvas ON
   Return attribute.n
ENDPROC
WriteLib libName.a InCanvasColorPalette.n
?? "\004"
; ============================================================================
;       TITLE: inErrorHandler.n          (c) 1992, 1993 DataStar International
;     RETURNS: Error Continuation Code
; DESCRIPTION: Main Error Handling Procedure - calls inErrorLog.u
;              The initial switch deals with specific errors, and attempts
;              to continue the application.  You should do this only when
;              you are sure it won't end up breaking something else (e.g.
;              If you continue from a query error, and later code expects
;              that the query will have performed successfully, you are
;              just postponing the inevitable.  That is one reason to use a
;              Query Execute procedure, so that you can interrupt the
;              process in the event of an error.
; ----------------------------------------------------------------------------
PROC inErrorHandler.n()          ; Main Error Handler
Private  errorproc,              ; Keeps errorproc from being recursive
         error.y,                ; DynArray from ErrorInfo
         message.a,              ; Formatted message to user
         script.a,               ; Concatonated re-named Savevars.sc
         errorWindow.a,          ; Paradox Window()
         a,                      ; Counter for FOREACH command
         windows.r,              ; Array of Windows from WINDOW LIST
         n1, n2,                 ; Transient Loop Counters
             logAllErrors.l
;Global  g.sysInfo.y             ; System info dynarray
;        g.debug.l               ; Development DEBUG flag
;        g.y                     ; Dynarray of Passwords
;        error.l                 ; Error flag passed back to routine
;        g.config.y              ; Check g.config.y for "logAllErrors"
   errorWindow.a = Window()                     ; Capture the Paradox Window
   IF IsAssigned(g.config.y) AND IsAssigned(g.config.y["logAllErrors"]) THEN
      logAllErrors.l = g.config.y["logAllErrors"]
   ELSE
      logAllErrors.l = false
   ENDIF
   msWorkingClear.u()
   IF NImages() > 0 AND ImageType() <> "Query" THEN
      SetBatch Off                           ; Just in case
   ENDIF
   ErrorInfo to error.y                      ; Capture the error info bag
   retval.n = 2                              ; Initialize returned value
   SWITCH
      CASE error.y["Proc"] = "WSDITTO.U"        :
         msContinue.u("","You cannot ditto " + StrVal(record.r[Field()]) +
                          " - " + errorWindow.a,79,"RED",1)
         retval.n = 1                        ; Ignore Ditto
      CASE error.y["Proc"] = "WSFIELDVIEW.U" AND error.y["Code"] = 23 :
         msContinue.u("","The Field Value does not satisfy current validity " +
                          "checks.  Current field value is:  " +
                           StrVal([]),30,"BLUE",1)
         error.l = True                      ; Set error flag
         retval.n = 1                        ; Step over the []=[] assignment
      CASE error.y["Proc"] = "WSPICKFORM.L"     :
         error.l = True                      ; Set error flag
         msContinue.u("",error.y["Message"],79,"RED",1)
         retval.n = 1
      CASE error.y["Proc"] = "WSCOPYFROMARRAY.U"   :
         SWITCH
            CASE (error.y["Code"] = 60 AND
                 Match (error.y["Message"],"..linked fields in ..") OR
                 Match (error.y["Message"],"..master record is blank..")) OR
                 (error.y["Code"] = 23 AND
                 Match(error.y["Message"],"..value must be provided..")):
               retval.n = 1
            CASE error.y["Code"] = 23 AND
                 MATCH(error.y["Message"],"..not one of the possible value.."):
               wsCopyFromArrayRecover.u(arrayname.a)
         ENDSWITCH
      CASE error.y["Code"] = 23
       AND ImageType() = "Query"
       AND error.y["Proc"] = "QUEXECUTE.L" :
         a = []
         CtrlBackSpace                       ; Eliminate offending expression
         msContinue.u("","","The invalid query criterion: " + a +
                             " was deleted from the " + Field() + " field," +
                             " so that the Query could continue.",31,"BLUE",1)
         retval.n = 1                        ; Skip over error command
      CASE error.y["Code"] = 34
       AND Search("procedure",error.y["Message"]) <> 0  :
         SWITCH
            CASE Search("!",error.y["Message"]) <> 0    :
               error.l = true
               retval.n = 1
            CASE Search("EditSetup",error.y["Message"]) <> 0    :
               msContinue.u("","The Procedure to be called to setup the Edit " +
                               "Session for Menu Choice: " + menutag.a +
                               " has not been defined.", 79, "RED", 4)
               retval.n = 1
            CASE Search("help",error.y["Message"]) <> 0 :
               helpchoice.a = "HELP"
               helpmenu.a = "DEFAULT"
               retval.n = 0
         ENDSWITCH
      CASE error.y["Code"] = 27              ; Using quExecute.l proc
       AND ImageType() = "Query"
       AND error.y["Proc"] = "QUEXECUTE.L" :
         error.l = true                      ; Set Query Error flag
         retval.n = 1                        ; Skip over error command
      CASE error.y["Code"] = 27              ; Not using quExecute.l proc
       AND ImageType() = "Query" :
         msContinue.u("","Query Error - " +Window(),79,"RED",3)
         retval.n = 1                        ; Skip over error command
      CASE error.y["Code"] = 27 AND Search("Returned a Value",error.y["Message"]) = 0 :
         msContinue.u("","Sorry, the Process could NOT be Completed.  " +
                          error.y["Message"],79,"RED",3)
         retval.n = 1                        ; Skip over error command
      CASE error.y["Code"] = 43
        OR error.y["Message"] = "Printer not ready" :
         ioPrinterStatus.l()
         IF retval THEN
            retval.n = 0
         ELSE
            retval.n = 1
         ENDIF
      CASE error.y["Proc"] = "INSTARTUP.L"
       AND error.y["Code"] = 11     :        ; PrivDir conflict
         retval.n = 1
      CASE error.y["Proc"] = "INERRORRESET.U"
       AND error.y["Code"] = 30     :        ; ErrorReset
         retval.n = 1
   ENDSWITCH

   IF retval.n = 2 THEN                      ; Error still not resolved
      msWorking.u("Error in Procedure: " + error.y["Proc"],79,0,0)
      IF NOT IsAssigned(g.debug.l) OR NOT g.debug.l THEN
         Echo OFF
         password.a = ""                        ; Deassign any password variables
         IF NOT IsAssigned(g.sysInfo.y) THEN
            SysInfo to g.sysInfo.y              ; Capture System Info
         ENDIF

         IF g.sysInfo.y["UIMode"] = "COMPATIBLE" THEN
            Canvas ON                           ; Just in case
         ENDIF

         IF IsAssigned(g.y) THEN                ; Deassign any password variables
            FOREACH a In g.y
               UnPassword g.y[a]
               g.y[a] = "********"
            ENDFOREACH
         ENDIF

         IF IsAssigned(g.a) THEN
            UnPassword g.a
            g.a = "********"
         ENDIF

         IF IsAssigned(t.a) THEN
            UnPassword t.a
            t.a = "********"
         ENDIF

         IF IsAssigned(chars.a) THEN
            chars.a = "********"
         ENDIF

         IF IsAssigned(g.config.y["pw.a"]) THEN
            g.config.y["pw.a"] = "********"
         ENDIF

         msContinue.u("",error.y["MESSAGE"],79,"RED",4)
         IF DirExists("ERR") = 0 THEN        ; Create an ERR directory if none
            Run NOREFRESH "MD ERR"           ; Store error logs in separate Dir
         ENDIF                               ; Log the error info
         script.a = "ERR\\"+StrVal(Ticks())  ; Easy Unique Name

         inErrorLog.u(error.y,g.sysInfo.y)   ; Log the error to disk and printer

         msWorking.u("Saving Current Variable Assignments to Disk",110,0,0)
         SaveVars ALL                        ; Rename Savevars.sc for posterity
         IF Sysmode() <> "Main" Then
            RUN NOREFRESH "REN "+PrivDir()+"savevars.sc "+Directory()+"\\"+script.a
         ELSE
            {Tools} {Rename} {Script} Select "Savevars" Select script.a
            IF MenuChoice() = "Cancel" THEN     ; VERY unlikely
               {Replace}
            ENDIF
         ENDIF
      ELSE
         msContinue.u("",error.y["MESSAGE"],79,"RED",1)
      ENDIF
      msWorkingClear.u()                     ; Removes message window

      IF NOT IsAssigned(g.debug.l) OR NOT g.debug.l THEN
         msContinue.u("","Log Complete - Please Contact Technical Support",
                       31,"BLUE",1)
         Reset
         {Tools} {More} {Protect} {Clearpasswords}
         SetColors DEFAULT
         EXIT
      ELSE                                   ; Allow access to DEBUG prompt
         msConfirm.l("","IF <Debug>, Use <Ctrl><P> to Pop Back to Error",79,
                      "RED",3,"~D~ebug","~C~ancel",true)
         IF retval THEN
            Debug ; Use <Ctrl><P> to Pop back to error
            retval.n = 0
         ELSE
            Reset
            {Tools} {More} {Protect} {Clearpasswords}
            SetColors DEFAULT
            QUIT "You have Canceled the Application from the Error Prompt..."
         ENDIF
      ENDIF
   ELSE
      IF logAllErrors.l THEN
         inErrorLog.u(error.y,g.sysInfo.y)   ; Log the error to disk and printer
      ENDIF
      PROC epErrorReset.n()                  ; Reset the ErrorCode
      Private errorproc
         Return 1
      ENDPROC
      errorproc = "epErrorReset.n"           ; Specialized errorproc
      retval = 1 + "A"                       ; Create errorcode 30
      errorproc = ""                         ; Deassign errorproc
      Release Procs epErrorReset.n           ; Release procedure
   ENDIF
   Return retval.n                           ; 0, 1 or 2
ENDPROC
WriteLib libName.a inErrorHandler.n
?? "\004"
; ============================================================================
;       TITLE: inErrorLog.u              (c) 1992, 1993 DataStar International
;     RETURNS: No value
; DESCRIPTION: Error Logging Procedure - called by inErrorHandler.n
;              Creates a Memo Variable and writes it to disk from the
;              contents of error.y (ErrorInfo, SysInfo & selected info).
; ----------------------------------------------------------------------------
PROC inErrorLog.u(               ; Logs Error to file and printer
         error.y,                ; ErrorInfo DynArray
         g.sysInfo.y)            ; SysInfo DynArray
Private  a,                      ; Tag of error.y in FOREACH loop
         error.m                 ; Memo variable holding errorlog
;Global  g.debug.l               ; Development DEBUG flag
   msWorking.u("An Error has occurred, please wait while it is logged",79,3,0)

   error.y["Date of Error"] = Today()
   error.y["Working Directory"] = Directory()
   error.y["Working Drivespace"] = DriveSpace(SubStr(Directory(),1,1))
   error.y["Current MemLeft"] = MemLeft()
   error.y["Private Directory"] = PrivDir()
   error.y["Private Drivespace"] = DriveSpace(SubStr(PrivDir(),1,1))
   error.y["Printer Status"] = Format("LO",PrinterStatus())
   error.y["RunTime"] = Format("LY",IsRunTime())
   error.y["Current SysMode"] = SysMode()
   error.y["Time of Error"] = Time()
   error.y["Paradox version"] = Version()

   error.y["Paradox Build"] = g.sysInfo.y["Build"]
   error.y["Current Extended Memory"] = g.sysInfo.y["Extended"]
   error.y["Current Expanded Memory"] = g.sysInfo.y["Expanded"]
   error.y["Mouse Available"] = g.sysInfo.y["Mouse"]
   error.y["Screen Height"] = StrVal(g.sysInfo.y["ScreenHeight"]) + " Rows"
   error.y["Screen Width"] = StrVal(g.sysInfo.y["ScreenWidth"]) + " Columns"
   error.y["UI Mode"] = g.sysInfo.y["UIMode"]

   IF NImages() <> 0 THEN                    ; occurred on image on workspace
      error.y["Number of Images"] = NImages()
      error.y["Current Table"] = Table()
      error.y["Current Image Type"] = ImageType()
      error.y["Current Field"] = Field()
      IF ImageType() = "Display" THEN
         error.y["Current Field Value"] = IIF(NImageRecords() <> 0,[],"No Records Present")
      ELSE
         error.y["Current Field Value"] = []
      ENDIF

      error.y["Shared Table"] = IsShared(Table())
      IF error.y["Current Image Type"] = "Query" THEN
         IF CheckMarkStatus() <> "" THEN ; store checkmark if appropriate
            error.y["Current Field Value"] = CheckMarkStatus()+" "+[]
         ENDIF
         error.y["Formview"] = "N/A"
         error.y["Record Number"] = "N/A"
      ELSE
         error.y["Formview"] = Format("LN",IsFormView())
         error.y["Record Number"] = RecNo()
      ENDIF
      error.y["Number of Records"] = NRecords(TABLE())
   ELSE                                      ; not in an image
      error.y["Number of Images"] = "N/A"
      error.y["Current Table"] = "N/A"
      error.y["Current Image Type"] = "N/A"
      error.y["Current Field"] = "N/A"
      error.y["Current Field Value"] = "N/A"
      error.y["Shared Table"] = "N/A"
      error.y["Number of Records"] = "N/A"
      error.y["Formview"] = "N/A"
      error.y["Record Number"] = "N/A"
   ENDIF

   IF IsAssigned(g.sysInfo.y["Starting MemLeft"]) THEN
      error.y["Starting MemLeft"] = g.sysInfo.y["Starting MemLeft"]
   ELSE
      error.y["Starting MemLeft"] = "UA"
   ENDIF

   IF error.y["User"] = "" THEN
      error.y["User"] = "N/A"
   ENDIF

   IF IsAssigned(g.config.y) THEN
      FOREACH a In g.config.y
         error.y[a] = g.config.y[a]
      ENDFOREACH
   ENDIF

   error.m = Fill("-",80) + "\n" +
             Format("w80,ac","*** Error while in Procedure " +
                              error.y["Proc"] + " ***") + "\n" +
              Spaces(8) + "Error: #" + StrVal(error.y["Code"]) + " - " +
              error.y["Message"] + "\n" + Spaces(8) + Fill("-",64) + "\n"
   FOREACH a In error.y
      error.m = error.m + Format("w31,ar",a) + ":  " + StrVal(error.y[a]) + "\n"
   ENDFOREACH
                                             ; Write memo variable to diskfile
   msWorking.u("Writing Error Log to Disk",31,0,0)
   FileWrite APPEND "ERR\\Errorlog.sc" From error.m
   IF NOT IsAssigned(g.debug.l) OR NOT g.debug.l THEN
      IF PrinterStatus() THEN             ; prints log if printer is available
         msWorking.u("Writing Error Log to Printer",111,0,0)
         Open PRINTER
         FileWrite PrivDir()+"Errorlog" FROM error.m
         RUN NoRefresh "Copy "+PrivDir()+"Errorlog LPT1 > NUL"
         Editor New PrivDir()+"Errorlog"
         {Cancel} {Yes}
         Close PRINTER
      ENDIF
   ENDIF
   Return
ENDPROC
WriteLib libName.a inErrorLog.u
?? "\004"
; ============================================================================
;       TITLE: inLeapYear.l              (c) 1992, 1993 DataStar International
;     RETURNS: Logical true/false if year is leap year
; DESCRIPTION: Determines if Year of date passed is a Leap Year
; ----------------------------------------------------------------------------
PROC inLeapYear.l(               ; Determines if date is in leap year
         date.d)                 ; Date to check
   Return (DateVal("2/29/"+Strval(Year(date.d))) <> "Error")
ENDPROC
?? "\004"
WriteLib libName.a inLeapYear.l
; ============================================================================
;       TITLE: inNextNumber.v            (c) 1992, 1993 DataStar International
;     RETURNS: Next value, or false
; DESCRIPTION: Increments next ID number stored in table - based upon
;              various similar routines, especially Alan Zenreich's
; ----------------------------------------------------------------------------
PROC inNextNumber.v(             ; Next Number routine
         table.a,                ; Name of next number table.
         field.a,                ; Name of field containing next number.
         embedded.l,             ; True if table is embedded in the current form.
         increment.n,            ; Amount to increment number by.
         working.a)              ; Message to display while getting next number.
Private  n,                      ; Loop counter.
         v,                      ; Incrementor and error flag.
         oldTable.a,             ; Original table name.
         oldImage.n,             ; Master image if any.
         formView.l,             ; Original form status.
         record.n                ; Current record number.
   msWorking.u(working.a,111,0,0)
   Echo Off
   oldImage.n  = ImageNo()
   record.n    = RecNo()
   oldTable.a  = Table()   ; Find the original table name
   formView.l  = IsformView()   ; Lets us put it back.
   SWITCH
      CASE NOT embedded.l AND formView.l :
         FORMKEY
         IF MATCH(WINDOW(), "Can't leave the record..") THEN
         ; Record locked or on new detail record.
            WHILE NOT ISMULTIFORM(TABLE(), FORM())
               UPIMAGE   ; Move to master table.
            ENDWHILE
            UNLOCKRECORD
            FORMKEY
         ENDIF
         masterimage.n = IMAGENO()
      CASE embedded.l AND NOT formView.l :
         FORMKEY
         masterimage.n = IMAGENO()
   ENDSWITCH

   MoveTo table.a
   MoveTo FIELD field.a
   SetBatch ON
   FOR n FROM 1 TO 10               ; Attempt to lock record up to 10 times.
      LockRecord
      IF retval OR ErrorCode() = 55 THEN
         QUITLOOP
      ENDIF
      SLEEP 500                     ; wait 1/2 second.
   ENDFOR
   IF NOT RETVAL AND ErrorCode() <> 55 THEN               ; 10 attempts.
      SetBatch Off
      msContinue.u("", ErrorUser()+ " has locked " + table.a,79,"RED",2)
      v = false
   ELSE
      v = []                        ; Grab the current field.
      [] = [] + increment.n         ; Increment or decrement the field.
   ENDIF

   UnLockRecord                     ; Free it up for others.
   SetBatch Off
   SWITCH
      CASE NOT embedded.l AND formView.l :
         MoveTo oldImage.n          ; The master table in table view.
         FormKey
         MoveTo oldTable.a
         IF detail.l THEN           ; Cursor is now at the top of the table.
            MoveTo RECORD record.n  ; Find the old position in the table even
         ENDIF
      OTHERWISE :
      MoveTo oldTable.a             ; Back to the original table.
   ENDSWITCH
   Return v
ENDPROC
WriteLib libName.a inNextNumber.v
?? "\004"
; ============================================================================
;       TITLE: inNextNumberByYear.v      (c) 1992, 1993 DataStar International
;     RETURNS: Next value, or false
; DESCRIPTION: Increments next ID number stored in table, based upon the
;              year value passed, for systems that re-initialize the next
;              number seed value each year.
; ----------------------------------------------------------------------------
PROC inNextNumberByYear.v(       ; Next Number routine, by Year
         table.a,                ; Name of next number table.
         field.a,                ; Name of field containing next number.
         year.s,                 ; Year to Locate
         embedded.l,             ; True if table is embedded in the current form.
         increment.n,            ; Amount to increment number by.
         working.a)              ; Message to display while getting next number.
Private  n,                      ; Loop counter.
         v,                      ; Incrementor and error flag.
         oldTable.a,             ; Original table name.
         oldImage.n ,            ; Master image if any.
         formView.l ,            ; Original form status.
         record.n ,              ; Current record number.
         detail.l                ; Is this an embedded detail table?
   msWorking.u(working.a,111,0,0)
   detail.l    = LinkType() <> "None"
   record.n     = RecNo()
   oldTable.a  = Table()   ; Find the original table name
   formView.l  = IsformView()   ; Lets us put it back.
   IF NOT IsValid() THEN
      CtrlBackSpace             ; Clear the field.
   ENDIF
   IF NOT embedded.l AND formView.l THEN
      FormKey
   ENDIF

   oldImage.n = ImageNo()
   MoveTo table.a
   CtrlHome                         ; Assumes that YEAR is first field
   IF ColNo() = 1 THEN              ; RecNo() field (TableView)
      Right
   ENDIF
   Locate year.s
   IF NOT retval THEN
      ImageRights
      Ins
      CtrlHome                         ; Assumes that YEAR is first field
      IF ColNo() = 1 THEN              ; RecNo() field (TableView)
         Right
      ENDIF
      [] = year.s
      Right
      WHILE ColNo() > 1
         [] = 1
         Right
      ENDWHILE
      v = 1
   ELSE
      MoveTo FIELD field.a
      SetBatch ON
      FOR n FROM 1 TO 10               ; Attempt to lock record up to 10 times.
         LockRecord
         IF retval THEN
            QUITLOOP
         ENDIF
         SLEEP 500                     ; wait 1/2 second.
      ENDFOR
      IF NOT RETVAL THEN               ; 10 attempts.
         SetBatch Off
         msContinue.u("", ErrorUser()+ " has locked " + table.a,79,"RED",2)
         v = false
      ELSE
         v = []                        ; Grab the current field.
         [] = [] + increment.n         ; Increment or decrement the field.
      ENDIF

      UnLockRecord                     ; Free it up for others.
      SetBatch Off
   ENDIF

   SWITCH
      CASE NOT embedded.l AND formView.l :
         MoveTo oldImage.n          ; The master table in table view.
         FormKey
         MoveTo oldTable.a
         IF detail.l THEN           ; Cursor is now at the top of the table.
            MoveTo RECORD record.n   ; Find the old position in the table even
         ENDIF
      OTHERWISE :
      MoveTo oldTable.a             ; Back to the original table.
   ENDSWITCH
   msWorkingClear.u()
   Return v
ENDPROC
WriteLib libName.a inNextNumberByYear.v
?? "\004"
; ============================================================================
;       TITLE: inNotAvailable.u          (c) 1992, 1993 DataStar International
;     RETURNS: No value
; DESCRIPTION: Generic Feature Unavailable
; ----------------------------------------------------------------------------
PROC inNotAvailable.u()          ; Use if a routine is not finished
   msContinue.u("","This feature is undergoing testing - It is unavailable",
                 95,"MAGENTA",1)
   Return
ENDPROC
WriteLib libName.a inNotAvailable.u
?? "\004"
; ============================================================================
;       TITLE: inPathStrip.a             (c) 1992, 1993 DataStar International
;     RETURNS: String stripped of drive and path
; DESCRIPTION: This strips the drive specifier and DOS path from a string.
; ----------------------------------------------------------------------------
PROC inPathStrip.a(              ; Strips PATH from specified string
         string.a)               ; Usually a full table name.
Private  a1                      ; Match variable
   WHILE MATCH(string.a, "..\\..", a1, string.a)   ; Kill anything prior to
   ENDWHILE                                        ;  last backslash, if any
   WHILE MATCH(string.a, "@:..", a1, string.a)     ; Kill drive letter and
   ENDWHILE                                        ;  colon, if any
   Return string.a
ENDPROC
WriteLib libName.a inPathStrip.a
?? "\004"
; ============================================================================
;       TITLE: inQuotesEmbedded.a        (c) 1992, 1993 DataStar International
;     RETURNS: String with backslash preceding all embedded quotes
; DESCRIPTION: Adds escape characters (backslashes) to quoted strings
; ----------------------------------------------------------------------------
PROC inQuotesEmbedded.a(         ; Adds backslashes preceding quotes
         string.a)               ; String to process
Private  len.n,                  ; Origional length of string
         n                       ; Loop counter
   IF Search("\"",string.a) > 0 THEN
      len.n = Len(string.a)
      FOR n From len.n To 1 Step -1          ; Step backwords to account for
         IF SubStr(string.a,n,1) = "\"" THEN ; increasing length of string
            string.a = SubStr(string.a,1,n-1) + "\\" +
                       SubStr(string.a,n,Len(string.a))
         ENDIF
      ENDFOR
   ENDIF
   Return string.a
ENDPROC
?? "\004"
WriteLib libName.a inQuotesEmbedded.a
; ============================================================================
;       TITLE: inRound.n                 (c) 1992, 1993 DataStar International
;     RETURNS: Value rounded to desired precision
; DESCRIPTION: This strips the drive specifier and DOS path from a string.
; ----------------------------------------------------------------------------
PROC inRound.n(                  ; Generic rounding procedure
         value.n,                ; Value to round
         prec.n)                 ; Level of precision
Private  retval.n                ; Value to return
   IF prec.n > 0 AND value.n <> 0 THEN
      IF value.n >= 0 THEN
         retval.n = Round(((value.n * Pow(10,prec.n)) + .01) /
                            Pow(10,prec.n),prec.n)
      ELSE
         retval.n = Round(((value.n * Pow(10,prec.n)) - .01) /
                            Pow(10,prec.n),prec.n)
      ENDIF
   ELSE
      retval.n = Round(value.n,prec.n)
   ENDIF
   Return retval.n
ENDPROC
WriteLib libName.a inRound.n
?? "\004"
; ============================================================================
;       TITLE: inSmartKey.l              (c) 1992, 1993 DataStar International
;     RETURNS: Logical true/false if successful
; DESCRIPTION: Generates a 4, 6 or 8 char smart ID Key based upon other fields
; ----------------------------------------------------------------------------
PROC inSmartKey.l(               ; SmartKey generator (4, 6 or 8 chars)
         size.n,                 ; Length of ID string (4,6,8)
         field1.n,               ; First Field to use
         field2.n,               ; Second Field to use (if any)
         field3.n)               ; Third Field to use (if any)
Private  r,
         a,
         key.a,
         oktopost.l,
         retval.l
   ImageRights
   CopyToArray r
   Del
   retval.l = true
   SWITCH
      CASE field2.n = 0 AND size.n = 4   :
         key.a = Format("w2,cu",r[field1.n])
      CASE field2.n = 0 AND size.n = 6   :
         key.a = Format("w4,cu",r[field1.n])
      CASE field3.n = 0 AND size.n = 6   :
         key.a = Format("w2,cu",r[field1.n]) + Format("w2,cu",r[field2.n])
      CASE field3.n = 0 AND size.n = 8   :
         key.a = Format("w4,cu",r[field1.n]) + Format("w2,cu",r[field2.n])
      CASE field3.n > 0 AND size.n = 8   :
         key.a = Format("w3,cu",r[field1.n]) + Format("w2,cu",r[field2.n]) +
                 Format("w1,cu",r[field3.n])
      OTHERWISE         :
         msContinue.u("","Sorry, improper combination of size and fields!",
                       79,"RED",1)
         retval.l = false
   ENDSWITCH

   n = 0
   msWorking.u("Attempting to Post Record with this Key",111,0,0)
   WHILE retval.l
      IF n > 99 THEN
         msContinue.u("","There are already 100 Records with this Key!!!  Sorry",
                       79,"RED",2)
         retval.l = false
         QUITLOOP
      ENDIF
      r[2] = key.a + SubStr(Format("w3,ez",n),2,2)
      AppendArray r
      IF retval = 1 THEN
         retval.l = true
         QUITLOOP
      ENDIF
      n = n + 1
   ENDWHILE
   msWorkingClear.u()
   Return retval.l
ENDPROC
WriteLib libName.a inSmartKey.l
?? "\004"
; ============================================================================
;       TITLE: inStartApplication.u            (c) 1993 DataStar International
;     RETURNS: No value
; DESCRIPTION: Standard Application Startup
; ----------------------------------------------------------------------------
PROC inStartApplication.u(       ; Application Startup Procedure
         g.appCode.a,            ; 4-Character application code
         autolib,                ; Pass standard autolib from DRIVER
         g.debug.l)              ; True skips error logging
Private  erroproc,               ; Error handler (inErrorHandler.n)
         g.menuProcs.y,          ; List of ExecProc'd Menu Procedures
         g.appColors.y,          ; Default application colors
         g.config.y,             ; Standard configuration information
         g.handles.y,            ; Tracks window handles throughout
         g.printer.y,            ; Printer configuration
         g.sysInfo.y,            ; System Information
         g.global.y,             ; Global ordinal variables
         g.arrivalValue.v,       ; Value when arriving on field in WP
         a1,                     ; Transient menu tag match variable
         y                       ; Splashscreen Window Dynarray

   errorproc = "inErrorHandler.n"         ; Main Error Handler
   DynArray g.handles.y[]
   msWorking.u("Initializing " + g.appCode.a + " - One Moment",111,0,0)
   SysInfo To g.sysInfo.y                 ; Always capture SysInfo
   ExecProc g.appCode.a + "Initialize.u"  ; Sets Initialization Variables
   DynArray y[]
      y["HasFrame"] = false
      y["HasShadow"] = false
      y["CanMove"] = false
      y["CanResize"] = false
      y["Maximized"] = true
      y["CanMaximize"] = false
      y["CanClose"] = false
      y["CanvasHeight"] = 25
      y["CanvasWidth"] = 80
      y["Echo"] = true
      y["Floating"] = false
      y["OriginCol"] = 0
      y["OriginRow"] = 0
      y["Height"] = 25
      y["Width"] = 80
   Window Create Attributes y To g.handles.y["splashBackground"]
   Window Select g.handles.y["splashBackground"]
   SetCanvas g.handles.y["splashBackground"]
   PaintCanvas Fill g.config.y["deskTop.a"] Attribute 55 ALL
   Echo Normal Echo Off
   IF NOT IsAssigned(g.appColors.y) THEN
      g.appColors.y = ""
   ENDIF

   IF inStartupVerify.l(g.config.y["RequiredSpace.n"],
                        g.config.y["RequiredUI.a"],
                        g.appColors.y) THEN
      g.config.y["Logtime"] = Time()
      g.config.y["Logdate"] = Today()
      g.config.y["RetryPeriod"] = RetryPeriod()
      GetColors to g.appColors.y
      SetCanvas DEFAULT
      PaintCanvas FILL Spaces(g.sysInfo.y["Screenwidth"]-1)
                  Attribute SysColor(1001) 0,0,0,(g.sysInfo.y["SCREENWIDTH"]-1)
      PaintCanvas FILL Spaces(g.sysInfo.y["Screenwidth"]-1)
                  Attribute SysColor(1001) (g.sysInfo.y["ScreenHeight"]-1),0,
                  (g.sysInfo.y["SCREENHEIGHT"]-1),(g.sysInfo.y["SCREENWIDTH"]-1)

      IF ioUserLogin.l(g.config.y["pwRow.n"],
                       g.config.y["pwPalette.a"],
                       g.config.y["pwTable.a"],
                       g.config.y["pw.a"],
                       g.config.y["pwRequired.l"],
                       g.config.y["pwMisses.n"],
                       g.config.y["pwCase.l"],
                       g.config.y["pwGuest.l"]) THEN
         inUserLog.u(g.config.y["logTable.a"],
                     g.config.y["logName.a"],
                     g.config.y["logTime.a"],
                     g.config.y["logDate.d"],"IN")
         Canvas OFF
         PaintCanvas Fill Spaces(g.sysInfo.y["Screenwidth"])
                     Attribute SysColor(1001) g.sysInfo.y["ScreenHeight"]-1,
                                             0,
                                             g.sysInfo.y["SCREENHEIGHT"]-1,
                                             g.sysInfo.y["SCREENWIDTH"]-1

         msSplashWindow.u(g.config.y["SplashStyle.a"], 11, 60,
                          g.appCode.a + "SplashText.u","SPLASH")
         Canvas Off
         WHILE true
            IF NOT IsWindow(GetCanvas()) THEN
               SetCanvas Default
            ENDIF
            Canvas Off
            ExecProc g.appCode.a + "MenuDefine" + g.config.y["accessLevel.a"] + ".u"
            menukey.n = 255
            menutag.a = ""
            Message ""
            msSplashWindow.u(g.config.y["SplashStyle.a"], 11, 60,
                             g.appCode.a + "SplashText.u","SPLASH")
            msWorkingClear.u()
            Canvas On

            GetMenuSelection KeyTo menukey.n To menutag.a
            ExecProc g.appCode.a + "MenuDispatch.l"
            IF NOT retval THEN
               QUITLOOP
            ENDIF
            msWorkingClear.u()
         ENDWHILE
         inUserLog.u(g.config.y["logTable.a"],
                     g.config.y["logName.a"],
                     g.config.y["logTime"],
                     g.config.y["logDate"],"OUT")
      ENDIF
   ENDIF
   Return
ENDPROC
?? "\004"
Writelib libName.a inStartApplication.u
; ============================================================================
;       TITLE: inStartupVerify.l         (c) 1992, 1993 DataStar International
;     RETURNS: No value
; DESCRIPTION: Start up Verification procedure to initialize application
;              Place Startup Splash Screen first
;              g.appColors.y     Custom DynArray must use this name
; ----------------------------------------------------------------------------
PROC inStartupVerify.l(          ; Generic Startup routine
         spaceNeeded.n,          ; Minimum Available Diskspace
         uiMode.a,               ; UI Mode Required, or BOTH
         g.appColors.y)          ; Desired Custom Colors, or ""
Private  n,                      ; Available drivespace
         retval.l
;Global  g.sysInfo.y
;        g.appColors.y
   IF Version() < 4 THEN
      Reset
      QUIT "Sorry, you need at least Paradox 4 for this Program..."
   ENDIF
   SysInfo To g.sysInfo.y
   g.sysInfo.y["Starting MemLeft"] = MemLeft()
   IF Upper(uiMode.a) = "COMPATIBLE" THEN
      SetUIMode COMPATIBLE
   ELSE
      SetUIMode STANDARD
   ENDIF
   retval.l = true

   ; These next sections check for necessary environmental conditions
   IF PrivDir() = "" OR PrivDir() = Directory() THEN
      WHILE true
         n = 1
         IF DirExists(Substr(Directory(),1,1)+":\\PRI"+StrVal(n)) = 0 THEN
            Run NOREFRESH "MD "+Substr(Directory(),1,1)+":\\PRI"+Strval(n)
         ENDIF
         SetPrivDir Substr(Directory(),1,1)+":\\PRI"+StrVal(n)
         IF (Upper(PrivDir()) = Upper(SubStr(Directory(),1,1)+":\\PRI"+StrVal(n))+"\\") THEN
            QUITLOOP
         ENDIF
         n = n + 1
      ENDWHILE
   ENDIF
   g.sysInfo.y["PrivSpace"] = DriveSpace(SubStr(PrivDir(),1,1))

   IF g.sysInfo.y["PrivSpace"] < spaceNeeded.n THEN
      msContinue.u("", "Directory = " + StrVal(g.sysInfo.y["PrivSpace"]) +
                        " bytes - " + StrVal(spaceNeeded.n) + " bytes" +
                        " are Required to Start Program.  Try Deleting" +
                        " any Unnecessary Files to free Diskspace.",
                     79,"RED",2)
      retval.l = false
   ELSE
      IF Type(g.appColors.y) <> "DY" THEN
         inStartupColors.u()
      ENDIF
      SetColors From g.appColors.y

      SetRetryPeriod 3
      SetRestartCount 15
      SetAutoSave 3
      {Report} {SetPrinter} {Override} {EndOfPage} {FormFeed}
      IF IsFile(PrivDir()+"Username.sc") THEN
         Play "UserName"                     ; User-specific routine
      ENDIF
   ENDIF

   IF NOT IsBlankZero() THEN
      msWorking.u("Reconfigure Paradox for \"Blanks=Zero\"",79,2,2)
      retval.l = false
   ENDIF

   msWorkingClear.u()
   Return retval.l
ENDPROC
WriteLib libName.a inStartupVerify.l
?? "\004"
; ============================================================================
;       TITLE: inStartupColors.u         (c) 1992, 1993 DataStar International
;     RETURNS: No value
; DESCRIPTION:
; ----------------------------------------------------------------------------
PROC inStartupColors.u()         ; Sets up standard app color palette
;Global  g.appColors.y
   DynArray g.appColors.y[]
      g.appColors.y["0"]    = 120
      g.appColors.y["1"]    = 0
      g.appColors.y["2"]    = 31
      g.appColors.y["3"]    = 79
      g.appColors.y["4"]    = 116
      g.appColors.y["5"]    = 27
      g.appColors.y["6"]    = 59
      g.appColors.y["7"]    = 31
      g.appColors.y["8"]    = 48
      g.appColors.y["9"]    = 19
      g.appColors.y["10"]   = 31
      g.appColors.y["11"]   = 121
      g.appColors.y["12"]   = 31
      g.appColors.y["13"]   = 31
      g.appColors.y["14"]   = 30
      g.appColors.y["15"]   = 63
      g.appColors.y["16"]   = 30
      g.appColors.y["17"]   = 31
      g.appColors.y["18"]   = 116
      g.appColors.y["19"]   = 113
      g.appColors.y["20"]   = 31
      g.appColors.y["21"]   = 31
      g.appColors.y["22"]   = 31
      g.appColors.y["23"]   = 30
      g.appColors.y["24"]   = 27
      g.appColors.y["25"]   = 49
      g.appColors.y["26"]   = 63
      g.appColors.y["27"]   = 30
      g.appColors.y["28"]   = 31
      g.appColors.y["29"]   = 62
      g.appColors.y["30"]   = 0
      g.appColors.y["31"]   = 0
      g.appColors.y["1000"] = 113
      g.appColors.y["1001"] = 112
      g.appColors.y["1002"] = 120
      g.appColors.y["1003"] = 116
      g.appColors.y["1004"] = 32
      g.appColors.y["1005"] = 8
      g.appColors.y["1006"] = 36
      g.appColors.y["1007"] = 56
      g.appColors.y["1008"] = 63
      g.appColors.y["1009"] = 58
      g.appColors.y["1010"] = 49
      g.appColors.y["1011"] = 19
      g.appColors.y["1012"] = 48
      g.appColors.y["1013"] = 31
      g.appColors.y["1014"] = 0
      g.appColors.y["1015"] = 55
      g.appColors.y["1016"] = 63
      g.appColors.y["1017"] = 58
      g.appColors.y["1018"] = 19
      g.appColors.y["1019"] = 19
      g.appColors.y["1020"] = 62
      g.appColors.y["1021"] = 33
      g.appColors.y["1022"] = 0
      g.appColors.y["1023"] = 112
      g.appColors.y["1024"] = 127
      g.appColors.y["1025"] = 122
      g.appColors.y["1026"] = 19
      g.appColors.y["1027"] = 19
      g.appColors.y["1028"] = 112
      g.appColors.y["1029"] = 127
      g.appColors.y["1030"] = 0
      g.appColors.y["1031"] = 112
      g.appColors.y["1032"] = 127
      g.appColors.y["1033"] = 122
      g.appColors.y["1034"] = 19
      g.appColors.y["1035"] = 19
      g.appColors.y["1036"] = 112
      g.appColors.y["1037"] = 112
      g.appColors.y["1038"] = 127
      g.appColors.y["1039"] = 126
      g.appColors.y["1040"] = 32
      g.appColors.y["1041"] = 43
      g.appColors.y["1042"] = 47
      g.appColors.y["1043"] = 120
      g.appColors.y["1044"] = 46
      g.appColors.y["1045"] = 112
      g.appColors.y["1046"] = 112
      g.appColors.y["1047"] = 127
      g.appColors.y["1048"] = 126
      g.appColors.y["1049"] = 31
      g.appColors.y["1050"] = 47
      g.appColors.y["1051"] = 26
      g.appColors.y["1052"] = 32
      g.appColors.y["1053"] = 114
      g.appColors.y["1054"] = 49
      g.appColors.y["1055"] = 49
      g.appColors.y["1056"] = 48
      g.appColors.y["1057"] = 47
      g.appColors.y["1058"] = 62
      g.appColors.y["1059"] = 49
      g.appColors.y["1060"] = 19
      g.appColors.y["1061"] = 0
      g.appColors.y["1062"] = 0
      g.appColors.y["1063"] = 112
      g.appColors.y["1064"] = 120
      g.appColors.y["1065"] = 116
      g.appColors.y["1066"] = 31
      g.appColors.y["1067"] = 8
      g.appColors.y["1068"] = 31
      g.appColors.y["1069"] = 120
      g.appColors.y["1070"] = 48
      g.appColors.y["1071"] = 112
      g.appColors.y["1072"] = 112
      g.appColors.y["1073"] = 120
      g.appColors.y["1074"] = 116
      g.appColors.y["1075"] = 32
      g.appColors.y["1076"] = 8
      g.appColors.y["1077"] = 36
ENDPROC
?? "\004"
WriteLib libName.a inStartupColors.u
; ============================================================================
;       TITLE: inTrimSpaces.a            (c) 1992, 1993 DataStar International
;     RETURNS: String with no spaces
; DESCRIPTION: Trim trailing and leading spaces from string
; ----------------------------------------------------------------------------
PROC inTrimSpaces.a(             ; Trims trailing/leading blanks
         string.a)               ; String to trim
   WHILE Match(string.a," ..",string.a)
   ENDWHILE
   WHILE Match(string.a,".. ",string.a)
   ENDWHILE
   Return string.a
ENDPROC
WriteLib libName.a inTrimSpaces.a
?? "\004"
; ============================================================================
;       TITLE: inUserLog.u               (c) 1992, 1993 DataStar International
;     RETURNS: No Value
; DESCRIPTION: Initial Login Routine
; ----------------------------------------------------------------------------
PROC inUserLog.u(                ; User Login/Logout routine
         logTable.a,             ; Name of Log Table
         userName.a,             ; Name of user
         logTime.a,              ; Original login time
         logDate.d,              ; Current date
         inOut.a)                ; IN or OUT
Private  r
msWorking.u("One Moment - Logging " +inOut.a+ " " + userName.a,111,0,0)
Lock logTable.a PFL
IF NOT retval THEN
   Sleep 1000
   Lock logTable.a PFL
   IF retval THEN
      WHILE true
         CoEdit "Sys\\Userlog"
         SetBatch On
         IF Upper(inOut.a) = "IN" THEN
            Ins
            CopyToArray r
            r[2] = userName.a
            r[3] = logDate.d
            r[4] = logTime.a
            r[5] = userName()
         ELSE
            Locate userName.a, logDate.d, logTime.a
            IF NOT retval THEN
               Ins
               CopyToArray r
               r[2] = userName.a
               r[3] = logDate.d
               r[4] = logTime.a
               r[5] = UserName()
               r[6] = SubStr(Time(),1,5)
            ELSE
               CopyToArray r
               r[6] = SubStr(Time(),1,5)
            ENDIF
         ENDIF
         CopyFromArray r
         LockRecord
         IF NOT retval THEN
            LockKey
         ENDIF
         SetBatch Off
         Do_It!
         ClearImage QUITLOOP
      ENDWHILE
   ENDIF
ENDIF
UnLock logTable.a PFL
Sleep 200                                 ; Pause so screen doesn't flash
Return
ENDPROC
WriteLib libName.a inUserLog.u
?? "\004"
; ============================================================================
;       TITLE: inWordWrap.n              (c) 1992, 1993 DataStar International
;     RETURNS: Size of Dynarray
; DESCRIPTION: Formats message
; ----------------------------------------------------------------------------
PROC inWordWrap.n(               ; Formats memo into wordwrapped DynArray
         message.a,              ; Text to wrap
         length1.n,              ; Length to wrap first line
         length2.n)              ; Length to wrap second line
Private  n1,
         n2,
         n3,
         formatLength.a,
         lines.n,
         firstLine.l,
         length.n
;Global  wrapped.y               ; DynArray of wrapped text

   DynArray wrapped.y[]
   lines.n = 1
   length.n = length1.n
   formatLength.a = "W"+StrVal(length.n)
   firstLine.l = true
   WHILE true
      messagelength.n = Len(message.a)
      IF messagelength.n = 0 THEN
         QUITLOOP
      ENDIF
      IF messagelength.n <= length.n AND Search("\n",message.a) = 0 THEN
         wrapped.y[StrVal(lines.n)] = Format(formatLength.a,message.a)
         QUITLOOP
      ELSE
         ; Strip out leading carriage returns
         WHILE SubStr(message.a,1,1) = "\n"
            message.a = SubStr(message.a,2,messagelength.n)
         ENDWHILE
         ; Locate embedded carriage returns to break line
         n1 = Search("\n",message.a)
         IF n1 > 0 AND n1 <= length.n THEN
            wrapped.y[StrVal(lines.n)] = Format(formatLength.a,
                                                SubStr(message.a,1,n1 - 1))
         ELSE
            n1 = length.n
            WHILE SubStr(message.a, n1, 1) <> " " AND n1 > 0
               n1 = n1 - 1
            ENDWHILE
            IF n1 > 0 THEN
               wrapped.y[StrVal(lines.n)] = Format(formatLength.a,SubStr(message.a,1,n1))
            ENDIF
         ENDIF
         n2 = n1 + 1
         WHILE SubStr(message.a, n2, 1) = " "
            n2 = n2 + 1
         ENDWHILE
         message.a = SubStr(message.a,n2,messagelength.n)
         lines.n = lines.n + 1
      ENDIF
      IF firstLine.l THEN
         firstLine.l = false
         length.n = length2.n
         formatLength.a = "W"+StrVal(length.n)
      ENDIF
   ENDWHILE
   ; Global wrapped.y
   lines.n = DynArraySize(wrapped.y)
   Return lines.n
ENDPROC
?? "\004"
WriteLib libName.a inWordWrap.n
