;Source File     : OkKey4.sc
;Submitted By    : Alan Zenreich and James Kocis, PAL By Example.
;Version         : 1.01
;                ; This version supercedes all prior versions.  It fixes a
;                ; potential problem with locating memo fields.
;Last revision   : 2/5/93
;Description     : These procedures post records and trap for key violations
;                ; in "real-time" during the CoEdit of a table in Paradox 4.0.
;                ; See text below for full commentary.
;
;
; This code is excerpted from the upcoming book:
;    The Paradox Programmer's Guide:  PAL By Example
;    By Alan Zenreich and James M. Kocis.
;    Published by Bantam/Borland.
;
; These procedures were developed for, and are used extensively by, WaitPlus
; and WaitPlus Professional data entry systems available from:
;    PAL By Example
;    One Newark Street
;    Hoboken, NJ  07030
;    201-659-3021    800-262-8069   Fax: 201-795-9237
;
; You might ask "Why are Alan and Jim sharing these procedures?"
; We're publishing them because key violation checking is very tricky in the
; current Paradox versions, but very important to implement.
; Unique keys is an integral part of any well designed system, and these
; procedures can help maintain the integrity of your data.
;
; The code below uses many techniques to get the desired effect, and is
; optimized to take advantage of version specific features of Paradox.
; The issues are complicated, but the source code is pretty well commented.
;
; This is only a small sampling of the source code provided in WaitPlus Pro.
; PAL programmers are granted rights to use the code listed in this
; script in their applications, provided that the following copyright
; statement is declared in their source code (along with their own copyright
; notice):
;
;     "Portions of this program are COPYRIGHT 1993 PAL By Example"
;
; You must also agree not to use this code in a product that generally
; competes with WaitPlus Pro.  This agreement is simliar in intent to
; the Paradox Runtime agreement.  Special licensing agreements for commercial
; products must be arranged in writing through PAL By Example.
;
;
;---------------------------------------------------------------------------
; Posting records in CoEdit mode has always been a problem in Paradox.
; Code written for Paradox versions prior to 4.0 (for example our OkKey35.l
; procedure) attempted to cope with the fact that there was no Paradox command
; that proactively posted a record under all CoEdit scenarios (master tables,
; detail tables, TableView, etc.)
;
; Our requirements for a routine that posts a record are, on the surface,
; simple:
;  * Attempt to post the record.
;  * If it posted, keep the cursor with the record and keep the record
;    locked.
;  * If it failed to post, don't change the record state, and tell us
;    why it failed.  Among the many things that can interfere with posting
;    a record are:
;     * Key violation  - another record with this key exists.
;     * Table is Write Locked with an explicit lock, or an ACCEPT with a
;       lookup option.
;     * A Query has temporarily place a special kind of implicit write lock.
;     * While adding a new record in table view, a Group Lock placed
;       by another user modifying the key of a master can either prohibit
;       the record from posting, or allow the post but not allow the record
;       to lock (or permit any further changes).  This is particulary
;       problematic.

; We had hoped that when Paradox 4.0 was introduced, that the
;    POSTRECORD NOPOST LEAVELOCKED
; command would accomplish the task.  This unfortunately is not the case.
; The NOPOST option doesn't set ERRORCODE() or ERRORMESSAGE() so if a post
; failed, there was no way of telling the user if it was a key violation
; or another user locking the table.

; Using the POSTRECORD KEYVIOL LEAVELOCKED gets the ERRORCODE() and
; ERRORMESSAGE(), but there is no generic way of getting out of the "KeyViol"
; recordstatus, especially if there is the potential for data in exisiting
; records that no longer passes the current validity checks.

; When a query starts it might place a special kind of write lock on the table.
; This Query Write Lock could be considered a "pending write lock", in
; that it allows an edited record to post then sets a write lock so you
; cannot lock or edit other records.  So, an edited existing record might post,
; but not remain locked.  New records are not allowed to post.

; Viewing a table in a secondary index order could cause an existing record
; to fly away if the indexed field has changed, and there is no way for
; POSTRECORD NOPOST LEAVELOCKED to stay with the record and leave it locked.
; Furthermore, there is a problem with the initial release of Paradox 4.0
; that creates rogue record locks in that scenario, so we want to avoid
; the problem entirely.

;--------------------------------------------------------------------------
; This script contains two main procedures that control posting records in
; CoEdit mode, and displaying any failure to post to the user:
;  postRecord4.l()  and  postRecord4.y()
; In addition, there is one more procedure that can be used in
; Paradox 4.0 Compatibility mode: okKey4.l().  The only difference
; between okKey4.l() and postRecord4.l() is in the way the error
; messages are displayed.   okKey4.l() shows errors at the top of the screen
; and postRecord4.l() uses a dialog box.

; The postRecord4.l() procedure is the one that most programmers will use.
; It is used wherever you might need to commit a record to the table.
; For those programmers already familiar with (and use) our OkKey35.l and
; okKey4.l() procedures, the postRecord4.l() procedure supercedes them
; and should be substituted for use with Paradox 4.0.
; We changed the name of the procedure because although it is generally used
; to verify the key, it now also allows posting in unkeyed tables.

; When invoked, postRecord4.l() calls a subordinate routine named
; postRecord4.y() which actually attempts to post the record and reports
; back any problems.

; If postRecord4.l() sees that the record failed to post for any reason
; it displays a dialog box to the user, describing why the post failed.

; The postRecord4.y() procedure is the routine that actually tries to commit
; the record to the table.  It updates a dynamic array that contains
; information about the final post status.

; The reason why the postRecord4.y() code is not included within the
; postRecord4.l() procedure is that you might want to use the actual posting
; mechanism with your own routines that display error messages.
; So, splitting out the user interface from the posting mechanism gives us
; more flexibility than a single procedure.

; So the postRecord4.l() and postRecord4.y() procedures provide the
; functionality that we need, and avoid the potential problems wherever
; possible.

; The procedures attempt to post the current record and maintain
; control of the record, leaving it locked.  The code deals with the
; various problems associated with records flying away, invalid data,
; query write locks, secondary index order, and all other known posting
; problems.  As of this writing, there is only one condition that will allow
; a record to post and not let you maintain control over the record.  This is
; is a very obscure case, and returns an error code of 98, which describes
; a new record that posted in spite of a group lock and then cannot be relocked.

; The listings explain the syntax for the procedure calls,
; allowing you to fine tune the locking, echo, and requiredcheck states.
;
; A typical call to post the record might look like:
;   l = postRecord4.l(TRUE, TRUE, FALSE)
;     ; Which specifies that the record must lock, ECHO is reset to
;     ; normal, and RequiredCheck is off.
;   IF l THEN
;     ; Do whatever is appropriate when post succeeds (perhaps exit CoEdit)
;   ELSE
;     ; Do whatever is appropriate after the user sees the dialog
;     ; box explaining what the problem was (perhaps return them to editing
;     ; the record or bring up additional help).
;   ENDIF
;
;----------------------------------------------------------------------------

  ; The postRecord4.l() procedure invokes the postRecord4.y() procedure
  ; and then displays a dialog box if the post failed.
PROC postRecord4.l(mustLock.l, echoOn.l, requiredCheck.l)
  ; mustLock.l -- is TRUE if routine should not attempt to post if it cannot
  ;  guarantee that the record will remain locked after posting.  If set
  ;  to FALSE, it means that you don't care if the record is locked or not.
  ;  The only reason a record might post but remain unlocked is if a
  ;  query has the table write locked and this is an existing, edited record.
  ; echoOn.l -- determines if the ECHO should be set to Normal when the proc
  ;  is finished.  The subordinate postRecord.y() procedure needs to turn echo
  ;  off for the duration of the post, so this variable makes sure that it
  ;  knows if it has to turn the  echo back to normal when it is done.
  ;  If echoOn.l = True then the proc issues an ECHO NORMAL.
  ;  Any other value leaves echo off.
  ; requiredcheck.l -- determines if required fields should be reinstated when
  ;  leaving the proc.  If requiredcheck.l is any value other than FALSE,
  ;  REQUIREDCHECK will be set back to the normal ON status.
  PRIVATE postStatus.y, title.a, line1.a, line2.a, sys.y, button.a, erroruser.a

    ; The next few lines allow you to specify an alternate post record
    ; procedure, perhaps necessary under special circumstances, for example
    ; when you know that there can't possibly be a key conflict, and you
    ; don't need to remain on the record, you might have a procedure that
    ; simply unlocks the record and returns TRUE.
    ;
    ; To invoke an alternate post record procedure, assign a variable called
    ; postRecordProc.a to the name of the alternate procedure prior to invoking
    ; the postRecord4.l procedure. The alternate procedure must return a TRUE
    ; if the key is okay, or FALSE if there is a key violation or any other
    ; failure to post.
    ;
    ; This can be handy for performance gains by making the key checking
    ; more specific to the particular needs of the table. Remember
    ; to release or reassign postRecordProc.a = "" when no longer needed.

  IF ISASSIGNED (postRecordProc.a) AND postRecordProc.a <> "" THEN
    EXECPROC postRecordProc.a ; Do the special procedure and return its value.
    RETURN RETVAL
  ENDIF

   ; Continue with the normal postRecord.l procedure:

   ; Declare a dynamic array to hold results of record posting procedure.
  DYNARRAY postStatus.y[]
   ; Attempt to post the record.
  postRecord4.y(postStatus.y, mustLock.l, echoOn.l, requiredCheck.l)
   ; The postRecord4.u() procedure fills the dynarray with elements for
   ;  postStatus.y["ERRORCODE"] = 0   ; or  3, 9, 53, 65, 98, 99
   ;  postStatus.y["ERRORUSER"] = ""  ; or ERRORUSER()
   ;  postStatus.y["ERRORMESSAGE"] = ""  ; or an errror message.
   ;  Note that the postRecord.l() procedure doesn't do anything with
   ;  the ["ERRORMESSAGE"] value, but the subordinate procedure supplies
   ;  this information in case it is called by another procedure that wants
   ;  to display messages based on the full information.

  IF postStatus.y["ERRORCODE"] <> 0 THEN
    line1.a = ""    ; Initialize variables for dialog box.
    line2.a = ""
    title.a = "Cannot post record"
    erroruser.a = IIF(postStatus.y["ERRORUSER"] = "",
                      "another user", postStatus.y["ERRORUSER"])

    SWITCH
      CASE postStatus.y["ERRORCODE"] = 3:   ; Another user has control of table.
        line1.a = "Table is currently locked by"
        line2.a = IIF(postStatus.y["ERRORUSER"] = "", "another user",
                      postStatus.y["ERRORUSER"]) + "."
      CASE postStatus.y["ERRORCODE"] = 9:   ; Another user has control of record.
        line1.a = "There is a record lock conflict"
        line2.a = "with " + IIF(postStatus.y["ERRORUSER"] = "", "another user",
                                postStatus.y["ERRORUSER"])
      CASE postStatus.y["ERRORCODE"] = 65 :  ; Group lock.
        line1.a = "Record is part of group locked"
        line2.a = "by " + IIF(postStatus.y["ERRORUSER"] = "",
                          "another user", postStatus.y["ERRORUSER"]) + "."
      CASE postStatus.y["ERRORCODE"] = 23 :   ; Invalid data.
        line1.a= "Current field has invalid data."
      CASE postStatus.y["ERRORCODE"] = 98 :
         ; Posted but not locked because of group lock.
         ; This error only occurs when mustLock.l = TRUE and a group lock
         ; lets a new record post, but prevents it from remaining locked.
        title.a = "Record posted, cannot lock"
        line1.a = "Record is now in a group locked"
        line2.a = "by " + IIF(postStatus.y["ERRORUSER"] = "",
                          "another user", postStatus.y["ERRORUSER"]) + "."
      CASE postStatus.y["ERRORCODE"] = 99 :   ; Blank unmodified record.
        line1.a =  "Record is blank and unmodified."
        line2.a =  "Cannot post and remain locked."
      OTHERWISE :                             ; A key conflict.
        line1.a = "A record with this key"
        line2.a = "already exists."
    ENDSWITCH

     ; Display a dialog that show the reason the post failed.
    SYSINFO TO sys.y    ; Get system attributes including screen size.
    BEEP BEEP
    MESSAGE ""
    SHOWDIALOG title.a
      @ INT((sys.y["SCREENHEIGHT"] - 8) /2),  ; Center dialog on screen
        INT((sys.y["SCREENWIDTH"] - 40) /2)

      HEIGHT 8 WIDTH 40
      @1,6  ?? FORMAT("W31,AC", line1.a)    ; Center text.
      @2,6  ?? FORMAT("W31,AC", line2.a)

      @1,3 ?? ""     ; An exclamation point.
      @2,3 ?? ""
      @3,3 ?? ""

      PUSHBUTTON
        @4,14 WIDTH 12
        "Ok"
        CANCEL   ; So [Enter] or [Esc] will remove dialog.
        VALUE ""
        TAG "okTag"
      TO button.a
    ENDDIALOG
  ENDIF
  RETURN postStatus.y["ERRORCODE"] = 0  ; TRUE if post was ok, FALSE otherwise.
ENDPROC
WRITELIB libname.a postRecord4.l
RELEASE PROCS      postRecord4.l

;----------------------------------------------------------------------------
; The postRecord4.y() procedure actually tries to commit the record to
; the table.  A dynamic array is passed in as the first parameter to
; hold the results of the post.  The procedure is typicaly called by
; the postRecord4.l() procedure and passed the parameters specified in
; that procedure.
;
; The code is heavily commented to explain the reasons for the coding
; techniques.

PROC postRecord4.y(postStatus.y, mustLock.l, echoOn.l, requiredCheck.l)
  ; postStatus.y -- is an pre-created dynamic array that will be loaded
  ;  with the results of the attempt to post.
  ;    ["ERRORCODE"] is set to 0 if post succeeds, 53, 3, 9, 65, 98 or 99
  ;    if it fails.  See notes in the code to determine what circumstances
  ;    cause each of the error codes.
  ; mustLock.l -- is TRUE if routine should not attempt to post if it cannot
  ;  guarantee that the record will remain locked after posting.  If set
  ;  to FALSE, it means that you don't care if the record is locked or not.
  ;  The only reason a record might post but remain unlocked is if a
  ;  query has the table write locked and this is an existing, edited record.
  ; echoOn.l -- determines if the ECHO should be set to Normal when the proc
  ;  is finished.  The procedure needs to turn echo off for the duration of the
  ;  post, so this variable makes sure that it knows what to do with the echo
  ;  when it is done.  If echoOn.l = True then  the proc issues an ECHO NORMAL.
  ;  Any other value leaves ECHO OFF.
  ; requiredcheck.l -- determines if required fields should be reinstated when
  ;  leaving the proc.  If requiredcheck.l is any value other than FALSE,
  ;  REQUIREDCHECK will be set back to the normal ON status.
  ;

  PRIVATE priorPWL.l, keys.n, maxkeys.n, r, isformview.l,
          new.l, modified.l, values.a, n, field.a, continue.l, ERRORPROC

   ; Initialize values in the dynamic array.
  postStatus.y["ERRORCODE"] = 0
  postStatus.y["ERRORUSER"] = ""
  postStatus.y["ERRORMESSAGE"] = ""
  new.l = RECORDSTATUS("NEW")
  modified.l = RECORDSTATUS("MODIFIED")
  REQUIREDCHECK OFF  ; Turn required check off.
  ECHO OFF           ; Turn the ECHO OFF.

   ; Let's see if we already have a Prevent Write Lock on this table.
  priorPWL.l = LOCKSTATUS(TABLE(), "PWL") > 0
   ; priorPWL.l = TRUE if we have at least one PWL currently in place.

   ; The reason we want to see if a PWL exists before continuing is that
   ; we need to make sure that a PWL would succeed in a few areas of the
   ; code that follows.

   ; Placing a PWL requires a critical section lock (requiring a lock file
   ; access, so we don't want to (or need to) add an additional PWL if one
   ; is already in place.  Fortunately determining if we already have a PWL
   ; is fast.  LOCKSTATUS() doesn't require a disk read, and not subject to
   ; a critical section lock placed by another user.
   ; So the priorPWL.l variable lets us optimize the code by not placing
   ; any redundant locks.

   ; Now determine the posting approach based on the current record status.
  SWITCH
     CASE NOT new.l AND NOT modified.l :
       ; The record is untouched, there is no reason to post it.
       ; This CASE statemtent is commented out just to point out that the
       ; initialization of the variables above takes care of the return
       ; status.
       IF NOT RECORDSTATUS("LOCKED") AND mustLock.l THEN
         LOCKRECORD
         IF NOT RETVAL THEN
            ; The record cannot be locked.  Either it is locked by another user
            ; or subject to a write lock or query write lock.
           postStatus.y["ERRORCODE"] = ERRORCODE()
            ; Errorcode() could be 3 (table in use by another user) or
            ; 9 (another user has the record locked).  It could also be a 65
            ; if there is a Group lock in place while this record is in
            ; tableview and another user is modifying the key of a master
            ; record that uses this table as a detail.
            ;
           postStatus.y["ERRORMESSAGE"] = ERRORMESSAGE()
           postStatus.y["ERRORUSER"] = ERRORUSER()
         ENDIF
       ENDIF

    CASE NOT ISVALID():
       ; The current field has invalid data.
      postStatus.y["ERRORCODE"] = 23
      postStatus.y["ERRORMESSAGE"] = "Field has invalid data"

    CASE new.l AND NOT modified.l:
       ; With a blank, new record there are several things that could happen;
       ; let's look at what Paradox does:
       ; * If it is a keyed table, POSTRECORD will close up the untouched row.
       ; * If it is keyed POSTRECORD KEYVIOL LEAVELOCKED will close up the
       ;     untouched row, and obviously will not lock the record.
       ; * If it is unkeyed table, POSTRECORD will post the blank record.

       ; So, Paradox behavior in keyed table is different from unkeyed tables.
       ; What should we do?  In a keyed table, we can simply delete the
       ; record, if the user is not insisting that the mustLock.l = TRUE.

       ; If mustLock.l = TRUE then we should return an errorcode stating that
       ; it is impossible to post an unmodified new record.

       ; The last possibility is that we can force the blank record into the
       ; table and leave it locked.  However, if the user really wants to
       ; do this, it's a simple matter of touching the record before calling
       ; this proc.  This would be useful if the user is trying to post a blank
       ; record in an unkeyed table, so they need to "touch" it first by
       ; doing anything that causes RECORDSTATUS("Modified") = TRUE.
       ; Simply assigning a field to itslef, or entering and ending FieldView
       ; can do this.

      IF mustLock.l THEN
         ; Refuse to post unmodified new record.
        postStatus.y["ERRORCODE"] = 99
        postStatus.y["ERRORMESSAGE"] = "Record is blank"
      ELSE
        IF NKEYFIELDS(TABLE()) = 0 THEN
          DEL  ; Remove blank record, closing untouched row.
        ELSE
          ; There is no way for us to know if there is an IMAGERIGHTS UPDATE
          ; currently on the table.  If there is currently an IMAGERIGHTS
          ; restriction, you can't delete the record with a DEL command,
          ; but POSTRECORD is able to remove the record.
          POSTRECORD
        ENDIF
      ENDIF
       ; We are not setting any errorcode or message when a row is closed
       ; up because if mustLock.l = FALSE, we don't much care what happens
       ; to the record because we are specifying that we don't need control
       ; of it.

    CASE new.l:
       ; A new, modified record.
      SETBATCH ON   ; Make sure no other users can change table.
      RETVAL = TRUE ; Initialize variable.
      IF NOT priorPWL.l THEN
         ; If there is not already a PWL in place then we will attempt to put
         ; a Prevent Write Lock on the table.
         ; It is possible that a Write Lock or a Query Write lock is in place,
         ; and could prevent us from posting the record.
        LOCK TABLE() PWL
      ENDIF
      IF RETVAL THEN
         ; We were able to get a prevent write lock so no query is holding us
         ; up.
        LOCKRECORD
         ; If successful, the record will post, the cursor will stay
         ; with the record and the record will remain locked.
        IF NOT RETVAL THEN
           ; There must be a key conflict.  We could either be in KeyViol state
           ; or if in a master form, a key violation exists, but we are not
           ; in the KeyViol state.
          postStatus.y["ERRORCODE"] = ERRORCODE()
           ; Errorcode() could be 53 (key exists) or 9 (key exists but another
           ; user has the conflicting record locked).  It could also be a 65
           ; if there is a Group lock in place while this record is in tableview
           ; and another user is modifying the key of a master record that
           ; uses this table as a detail.
           ;
          postStatus.y["ERRORMESSAGE"] = ERRORMESSAGE()
          postStatus.y["ERRORUSER"] = ERRORUSER()
          SWITCH
            CASE RECORDSTATUS("KeyViol"):
               ; We need to get out of the key violation state because we don't
               ; want subsequent attempts to lock this record overwriting
               ; a conflicting record.
               ; Because this is a new, unposted record, we don't have to concern
               ; ourselves with invalid data that could be in existing records.
              COPYTOARRAY r    ; Copy record to an array.
                ; "Touch" the key fields to remove keyviol status.
              COPYFROMARRAY r
            CASE mustlock.l AND NOT RECORDSTATUS("NEW") AND
                 NOT RECORDSTATUS("LOCKED"):
               ; Even though the LOCKRECORD set RETVAL = FALSE there
               ; is the possibility that the record posted, but cannot be locked.
               ; This can happen in TableView when a group lock takes over.
               ; This only happens when the record is inserted at a point in the
               ; table between records that are not part of the target group.
              postStatus.y["ERRORCODE"] = 98
          ENDSWITCH
          ; The post succeeded, let's check for the remote possibility
          ; that
        ENDIF
        IF NOT priorPWL.l THEN
          UNLOCK TABLE() PWL  ; Remove the temporary Prevent Write Lock.
        ENDIF
      ELSE
         ; Could not get a PWL , if a WL was placed (of any kind) the record
         ; can not possibly post.
        postStatus.y["ERRORCODE"] = 3  ; Table in use.
        postStatus.y["ERRORUSER"] = ERRORUSER()
        postStatus.y["ERRORMESSAGE"] = ERRORMESSAGE()
      ENDIF
      SETBATCH OFF

    CASE NOT new.l AND modified.l:
       ; This is an existing modified record, already locked.
      continue.l = TRUE
       ; Get current field because we might have to move from it.
      field.a = FIELD()
      IF mustLock.l AND NOT priorPWL.l THEN
         ; User has specified that this task must fail if we can't keep the
         ; record locked.  At this time the priorPWL.l variable tells us
         ; that we don't currently have a PWL for the table.
         ; Place a Prevent Write Lock on the table to make sure nothing is
         ; (or will be) preventing us from assuring that the record can be
         ; locked after posting.
        LOCK TABLE() PWL  ; Place a Prevent Write Lock.
        IF NOT RETVAL THEN
          ; The PWL failed.
          postStatus.y["ERRORCODE"] = 3
          postStatus.y["ERRORUSER"] = ERRORUSER()
          postStatus.y["ERRORMESSAGE"] = ERRORMESSAGE()
          continue.l = FALSE
        ENDIF
      ENDIF
      IF continue.l THEN ; We got a PWL, or we don't care if record locks.
        keys.n = NKEYFIELDS(TABLE())
         ; We are going to try to POSTRECORD NOPOST (which posts) the record,
         ; then attempt to locate the record and  re-lock it.
         ; When the record is unlocked and the table is keyed, we need to be
         ; able to find it again.
         ; If the table is keyed we'll copy the values to an array to save
         ; the edited stated of the record for the subsequent LOCATE.
        IF keys.n > 0 THEN
          COPYTOARRAY r
           ; Now we'll construct a string to execute that will find the
           ; record in the event that it flies away.
           ; For example, let's assume there are 3 keys.  To find the records
           ; we'd have to issue LOCATE r[2],r[3],r[4] which will find the
           ; unique record.  Remember that the first element of the array is
           ; the table name so it is not used.
           ; The second element is the first key field.

          IF keys.n > 1 THEN
             ; Construct the search string to be EXECUTE'd
            values.a = ""  ; initialize string

             ; Let's make sure that we don't have more than 20 keys
             ; because the EXECUTE string below would be too long.
            maxkeys.n = MIN(keys.n, 21)
            FOR n FROM 1 TO maxkeys.n
              IF n > 1 THEN
                values.a = values.a +","  ; Add a comma before the next value.
              ENDIF
              values.a = values.a + "r["+STRVAL(n+1)+"]"
            ENDFOR
             ; We now have a string that might look like
             ; "r[2],r[3],r[4]"
          ENDIF
        ENDIF

         ; SETBATCH ON makes sure that no one can stop us from finding the
         ; record once posted, even if the record flies away.
        SETBATCH ON   ; Get short term exclusive use of the table.
        POSTRECORD NOPOST  ; Attempt to post the record.
         ; We use NOPOST so that we don't have the possibility of getting
         ; into the KeyViol state (unlike using UNLOCKRECORD).
         ; We can't use POSTRECORD NOPOST LEAVELOCKED for two reasons:

         ; If we did use LEAVELOCKED, there is a situation where the
         ; record may post but not remain locked.  If a query is in progress
         ; it might place a special kind of write lock.  If this lock is
         ; in place when the post is attempted, the post will succeed, but
         ; the LEAVELOCKED option will fail.  In this case RETVAL = TRUE and
         ; ERRORCODE() = 3.

         ; Another potential cause for problems is that in Paradox 4.0
         ; using the command to post a record while viewing the table in a
         ; secondary index order can add rogue locks to the lock file.
         ; If the indexed field has changed, the record will fly away and
         ; unlock anyway, so we might just as well attempt to post it and
         ; chase it down later.  Even if the rogue lock situation is fixed,
         ; the record is still going to fly away, so using LEAVELOCKED
         ; doesn't gain us anything.

         ; The previous SETBATCH ON statement assures us that we will find
         ; the record, and that no one else can delete or lock it before we
         ; get to it.

        IF NOT RETVAL THEN
           ; The POSTRECORD NOPOST failed.
           ; This can't be a normal Write Lock because record was already
           ; locked.  Nor can it be a Query Write Lock because that kind
           ; of lock allows modified records to post.
           ; So it must be a key violation.
          postStatus.y["ERRORCODE"] = 53
          postStatus.y["ERRORMESSAGE"] = "Key Violation"
        ELSE
           ; The record posted.
           ; If it is keyed, we have to locate and lock it again.
          SWITCH
            CASE keys.n = 0 :    ; No key fields, don't do anything.

            CASE keys.n = 1 :
              isformview.l = ISFORMVIEW()
              IF ARRAYSIZE(r) = 2 THEN
                ; We must have only one field so it must be placed as a regular
                ; field on any form.
                ; Make sure we move to the field for LOCATE, whether in table
                ; or form view.
                CTRLEND
              ELSE
                ; Try to move to the key field.  If we are in form view and
                ; the key field has not been placed as a regular field we must
                ; switch to table view in order to move to the field and do
                ; a single-value LOCATE.
                ; We would have preferred to simply append another array element
                ; to the LOCATE command if there are more fields in the table.
                ; However the second field might be a memo or blob, which
                ; can't be used in a multi-field locate; so we have to move to
                ; the field.
                MENU {IMAGE} {ZOOM} {FIELD} ENTER
                IF WINDOW() = "Field has not been placed on the form" THEN
                  MENU
                  ESC
                  FORMKEY
                  MENU {IMAGE} {ZOOM} {FIELD} ENTER
                ENDIF
              ENDIF
              LOCATE r[2]
              IF isformview.l AND NOT ISFORMVIEW() THEN
                ; If we were originally in form view, switch back.
                FORMKEY
              ENDIF
              MOVETO FIELD field.a  ; In case we moved from the field during LOCATE.

            OTHERWISE :
              ; Two or more key fields.
              ; LOCATE using the multi-field string created earlier.
              EXECUTE "LOCATE " + values.a
               ; Because setbatch is on, the locate can't fail, nor can any
               ; other user get control of the record before we do.

          ENDSWITCH
        ENDIF
          ; Attempt to lock the record.  If we previously specified
          ; mustLock.l = TRUE, then there is no way this can fail.  If we
          ; don't care if the record locks or not, this might fail if a write
          ; lock is in place, in which case the record remains unlocked.
        LOCKRECORD
        SETBATCH OFF  ; Release the exclusive table lock.
      ENDIF
      IF mustLock.l AND NOT priorPWL.l THEN
         ; Remove the temporary PWL placed on the table.
        UNLOCK TABLE() PWL
      ENDIF

  ENDSWITCH
  IF requiredcheck.l <> FALSE THEN
    REQUIREDCHECK ON
  ENDIF
  IF echoOn.l = TRUE THEN
    ECHO NORMAL
  ENDIF
ENDPROC
WRITELIB libname.a postRecord4.y
RELEASE PROCS      postRecord4.y
;---------------------------------------------------------------------
;---------------------------------------------------------------------
; The next procedure is the functional equivilant of postRecord4.l()
; but this time for Paradox's compatibility mode.
; The only real difference is that it doesn't use a dialog box to
; display messages; it places the text on the top of the screen and
; waits for a keypress.  If your code doesn't run in compatibility
; mode, the procedure is not needed.  The procedure checks to see if the
; standard user interface is in use, and if so it invokes the
; postRecord4.l() procedure automatically.

PROC okKey4.l(mustLock.l, echoOn.l, requiredCheck.l)

  PRIVATE postStatus.y, title.a, line1.a, line2.a, sys.y, button.a,
          erroruser.a, n
    ; The next few lines allow you to specify an alternate post record
    ; procedure, perhaps necessary under special circumstances, for example
    ; when you know that there can't possibly be a key conflict, and you
    ; don't need to remain on the record, you might have a procedure that
    ; simply unlocks the record and returns TRUE.
    ;
    ; To invoke an alternate post record procedure, assign a variable called
    ; postRecordProc.a to the name of the alternate procedure prior to invoking
    ; the postRecord4.l procedure. The alternate procedure must return a TRUE
    ; if the key is okay, or FALSE if there is a key violation or any other
    ; failure to post.
    ;
    ; This can be handy for performance gains by making the key checking
    ; more specific to the particular needs of the table. Remember
    ; to release or reassign postRecordProc.a = "" when no longer needed.

  IF ISASSIGNED (postRecordProc.a) AND postRecordProc.a <> "" THEN
    EXECPROC postRecordProc.a ; Do the special procedure and return its value.
    RETURN RETVAL
  ENDIF

  IF sys.y["UIMODE"] = "STANDARD" THEN
    ; Switch back to the normal postrecord4.l() procedure with dialog box.
    postRecord4.l(mustLock.l, echoOn.l, requiredCheck.l)
    RETURN RETVAL
  ENDIF

   ; Continue with the normal postRecord procedure:

   ; Declare a dynamic array to hold results of record posting procedure.
  DYNARRAY postStatus.y[]
   ; Attempt to post the record.
  postRecord4.y(postStatus.y, mustLock.l, echoOn.l, requiredCheck.l)
   ; The postRecord4.u() procedure fills the dynarray with elements for
   ;  postStatus.y["ERRORCODE"] = 0   ; or  3, 9, 53, 65, 98, 99
   ;  postStatus.y["ERRORUSER"] = ""  ; or ERRORUSER()
   ;  postStatus.y["ERRORMESSAGE"] = ""  ; or an errror message.
   ;  Note that the postRecord.l() procedure doesn't do anything with
   ;  the ["ERRORMESSAGE"] value, but the subordinate procedure supplies
   ;  this information in case it is called by another procedure that wants
   ;  to display messages based on the full information.

  IF postStatus.y["ERRORCODE"] <> 0 THEN
    line1.a = ""    ; Initialize variables for dialog box.
    line2.a = ""
    erroruser.a = IIF(postStatus.y["ERRORUSER"] = "",
                      "another user", postStatus.y["ERRORUSER"])

    SWITCH
      CASE postStatus.y["ERRORCODE"] = 3:   ; Another user has control of table.
        line1.a = "Table is currently locked by"
        line2.a = IIF(postStatus.y["ERRORUSER"] = "", "another user",
                      postStatus.y["ERRORUSER"]) + "."
      CASE postStatus.y["ERRORCODE"] = 9:   ; Another user has control of record.
        line1.a = "There is a record lock conflict"
        line2.a = "with " + IIF(postStatus.y["ERRORUSER"] = "", "another user",
                                postStatus.y["ERRORUSER"])
      CASE postStatus.y["ERRORCODE"] = 65 :  ; Group lock.
        line1.a = "Record is part of group locked"
        line2.a = "by " + IIF(postStatus.y["ERRORUSER"] = "",
                          "another user", postStatus.y["ERRORUSER"]) + "."
      CASE postStatus.y["ERRORCODE"] = 23 :   ; Invalid data.
        line1.a= "Current field has invalid data."
      CASE postStatus.y["ERRORCODE"] = 98 :
         ; Posted but not locked because of group lock.
         ; This error only occurs when mustLock.l = TRUE and a group lock
         ; lets a new record post, but prevents it from remaining locked.
        line1.a = "Record is now in a group locked"
        line2.a = "by " + IIF(postStatus.y["ERRORUSER"] = "",
                          "another user", postStatus.y["ERRORUSER"]) + "."
      CASE postStatus.y["ERRORCODE"] = 99 :   ; Blank unmodified record.
        line1.a =  "Record is blank and unmodified."
        line2.a =  "Cannot post and remain locked."
      OTHERWISE :                             ; A key conflict.
        line1.a = "A record with this key"
        line2.a = "already exists."
    ENDSWITCH
  ENDIF
  IF NOT ok.l THEN
     STYLE ATTRIBUTE SYSCOLOR(3)        ; Use same color as Paradox messages.
     @ 0,0 ?? FORMAT("W80,AC",line1.a)  ; Center the first line.
     ?? FORMAT("W80,AC", line2.a+"  Press any key.") ; Center the second line.
     STYLE
     WHILE CHARWAITING()           ; User might be typing ahead of the program.
       n = GETCHAR()               ; Grab keystroke from buffer assign to n.
     ENDWHILE
     WHILE NOT CHARWAITING()       ; Waits for keystroke from operator.
       BEEP BEEP                   ; Two quick beeps.
       SLEEP 1000                  ; Pauses for a second.
     ENDWHILE
     n = GETCHAR()                 ; Retrieves the operator keystroke.
     @1,0 CLEAR EOL                ; Clean up after messages, except for
     @0,0 CLEAR EOL
  ENDIF
  RETURN postStatus.y["ERRORCODE"] = 0  ; TRUE if post was ok, FALSE otherwise.
ENDPROC
WRITELIB libname.a okKey4.l
RELEASE PROCS      okKey4.l
