;==============================================================================
; (c) Copyright Elect Software International Inc., 1992, Toronto. Anyone can
; use this code for anything as long as it is not resold as a software
; development resource, as long as the copyright notice isn't removed, and as
; long as changes are clearly marked as to authorship.
; Comments welcome. Henrik Bechmann, CIS:72701,3717; Tel:416-534-8176.
;==============================================================================

;==============================================================================
;                             CALENDAR INTERFACE
;==============================================================================
; Calendar.Constructor() ; called at start of session in which calendar will be
;                        ; used.
; Calendar.Destructor()  ; called after session in which calendar is used, to
;                        ; clean up.
; Calendar.Dialog(Date)  ; called with a default date (usually today()).
;                        ; Returns selected date, or BlankDate() if no date
;                        ; was selected.
; Calendar.GetDate(DayOfMonth,MonthOfYear,CalendarYear)
;                        ; can be used as a general utility; returns date based
;                        ; on integer input.
; Calendar.GetHighDay(MonthOfYear,CalendarYear)
;                        ; can be used as a general utility; returns last day
;                        ; of the month (28-31) based on month and year
;                        ; integers
;==============================================================================
;                          CALENDAR IMPLEMENTATION
;==============================================================================
Proc Calendar.Constructor()
   Private
      i,j,k
   Dynarray Calendar.DOWNum[]
   Calendar.DOWNum["Sun"] = 1
   Calendar.DOWNum["Mon"] = 2
   Calendar.DOWNum["Tue"] = 3
   Calendar.DOWNum["Wed"] = 4
   Calendar.DOWNum["Thu"] = 5
   Calendar.DOWNum["Fri"] = 6
   Calendar.DOWNum["Sat"] = 7

   Array Calendar.MonthName[12]
   Calendar.MonthName[1] = "Jan"
   Calendar.MonthName[2] = "Feb"
   Calendar.MonthName[3] = "Mar"
   Calendar.MonthName[4] = "Apr"
   Calendar.MonthName[5] = "May"
   Calendar.MonthName[6] = "Jun"
   Calendar.MonthName[7] = "Jul"
   Calendar.MonthName[8] = "Aug"
   Calendar.MonthName[9] = "Sep"
   Calendar.MonthName[10] = "Oct"
   Calendar.MonthName[11] = "Nov"
   Calendar.MonthName[12] = "Dec"

   ;---------------------------------------------------------------------------
   ; A map is set up as follows: bracketed numbers ([]) represent
   ; Calendar.Picklist cells, unbracketed numbers represent
   ; Calendar.MapToPickList cells. Calendar.MapToPickList cells contain
   ; pointers to the Calendar.PickList cells. The current calendar month is
   ; mapped onto the Calendar.MapToPickList array with an offset, being the
   ; number of cells of the month that are blank. The first cell
   ; of the Calendar.MapToPickList array that points to a day value in picklist
   ; is the first Calendar.DOWNum of the month.
   ;
   ;  1[ 1]   2[ 7]   3[13]   4[19]   5[25]   6[31]   7[37]
   ;  8[ 2]   9[ 8]  10[14]  11[20]  12[26]  13[32]  14[38]
   ; 15[ 3]  16[ 9]  17[15]  18[21]  19[27]  20[33]  21[39]
   ; 22[ 4]  23[10]  24[16]  25[22]  26[28]  27[34]  28[40]
   ; 29[ 5]  30[11]  31[17]  32[23]  33[29]  34[35]  35[41]
   ; 36[ 6]  37[12]  38[18]  39[24]  40[30]  41[36]  42[42]
   ;
   ; Therefore:
   ; - To get the picklist number for the current day of the month, choose the
   ; pointer in Calendar.MapToPickList with an index of the day of the month
   ; plus the month offset.
   ; - To get the date value from the Calendar.Picklist index, simply choose
   ; the value found in the Calendar.PickList of the index.
   ;---------------------------------------------------------------------------
   Array Calendar.MapToPickList[42]
   k = 1
   For i From 0 to 6
      For j From  1 to 42 step 7
         Calendar.MapToPickList[i + j] = k
         k = k + 1
      EndFor
   EndFor
   ;---------------------------------------------------------------------------
   ; Generate an array that allows easy lookup of the implied index of
   ; Calendar.MapToPickList given an index of Calendar.PickList. This
   ; allows resynchronization from a pointer into Picklist that points to a
   ; blank, to a pointer that points to a date within range. (See
   ; Calendar!ResyncDatePick())
   ;---------------------------------------------------------------------------
   Array Calendar.MapToMap[42]
   For i From 1 to 42
      Calendar.MapToMap[Calendar.MapToPickList[i]] = i
   EndFor
EndProc ; Calendar.Constructor

writelib Libname Calendar.Constructor
release procs Calendar.Constructor
? "Calendar.Constructor()"
ColumnSet()





Proc Calendar.Destructor()
   Release Vars
      Calendar.DOWNum,
      Calendar.MonthName,
      Calendar.MapToPickList,
      Calendar.MapToMap
EndProc

writelib Libname Calendar.Destructor
release procs Calendar.Destructor
? "Calendar.Destructor()"
ColumnSet()




Proc Calendar.Dialog(Date)
   Private
      Calendar.Day,
      Calendar.Month,
      Calendar.Year,
      Calendar.PickList,
      Calendar.Offset,
      Calendar.HighDay,
      Calendar.Prompt,
      DatePick,
      PopupVar,
      ControlVar,
      i
   Calendar.Day = Day(Date)
   Calendar.Month = Month(Date)
   Calendar.Year = Year(Date)
   Calendar!GeneratePickList()

   ShowDialog "Calendar"
      Proc "Calendar!EventResponse"
         Key "PgUp","PgDn","Home","End"
         Trigger "Accept"
      @5,2 Height 15 Width 38
      @0,0 ?? Format("W36,AC",Calendar.Prompt)
      @1,1 ?? "Sun  Mon  Tue  Wed  Thu  Fri  Sat  "
      PickArray
         @2,1 Height 6 Width 34
         Columns 7
         Calendar.PickList
         Tag "Pick"
         To DatePick
      PushButton
         @9,1 Width 10
         "~M~onth"
         Value Calendar!ChangeMonth()
         Tag "Month"
         To PopupVar
      PushButton
         @9,13 Width 10
         "~Y~ear"
         Value Calendar!ChangeYear()
         Tag "Year"
         To PopupVar
      PushButton
         @9,25 Width 10
         "~T~oday"
         Value Calendar!MovetoToday()
         Tag "Today"
         To PopupVar
      PushButton
         @11,5 Width 10
         "~O~K"
         OK
         Default
         Value "OK"
         Tag "OK"
         To ControlVar
      PushButton
         @11,20 Width 10
         "~C~ancel"
         Cancel
         Value "Cancel"
         Tag "Cancel"
         To ControlVar
   EndDialog

   If retval Then
      Date = Calendar.GetDate(Calendar.PickList[DatePick],
               Calendar.Month,Calendar.Year)
   Else
      Date = BlankDate()
   Endif
   Return Date
EndProc ; Calendar.Dialog

writelib Libname Calendar.Dialog
release procs Calendar.Dialog
? "Calendar.Dialog()"
ColumnSet()






;----------------------------------------------------------------------------
; Calendar!GeneratePickList() expects three global variables to be set:
; Calendar.Day, Calendar.Month, and Calendar.Year. If Calendar.Day is greater
; than the last day of the current month, it will be set to the highest day of
; the month.
; In any case DatePick is set to the index value of Calendar.Picklist that
; points to Calendar.Day
;----------------------------------------------------------------------------
Proc Calendar!GeneratePickList()
   Array Calendar.PickList[42]
   Calendar.HighDay = Calendar.GetHighDay(Calendar.Month,Calendar.Year)
   If Calendar.Day > Calendar.HighDay Then
      Calendar.Day = Calendar.HighDay
   Endif
   Calendar.Offset =
      Calendar.DOWNum[
         DOW(Calendar.GetDate(
            1,Calendar.Month,Calendar.Year))] - 1
   For i From 1 To Calendar.HighDay
      Calendar.PickList[Calendar.MapToPickList[i + Calendar.Offset]] = i
   EndFor
   For i From 1 To Calendar.Offset
      Calendar.PickList[Calendar.MapToPickList[i]] = BlankNum()
   EndFor
   For i From Calendar.HighDay + Calendar.Offset + 1 To 42
      Calendar.PickList[Calendar.MapToPickList[i]] = BlankNum()
   EndFor
   DatePick = Calendar.MapToPickList[Calendar.Day + Calendar.Offset]
   Calendar.Prompt = Calendar.MonthName[Calendar.Month] + " " +
      StrVal(Calendar.Year)
EndProc

writelib Libname Calendar!GeneratePickList
release procs Calendar!GeneratePickList
? "Calendar!GeneratePickList()"
ColumnSet()





Proc Calendar!EventResponse(EventType,TagValue,EventInfo,ElementValue)
   Private
      Key,
      OK
   Switch
      Case EventType = "ACCEPT":
         If Not IsBlank(Calendar.PickList[DatePick]) Then
            OK = True
         Else
            Beep
            OK = false
            SelectControl "Pick"
         Endif
      Case EventType = "EVENT":
         Key = EventInfo["KeyCode"]
         Switch
            Case Key = Asc("PgUp"):
               If (Calendar.Month = 1) Then
                  If (Calendar.Year > 100) Then
                     Calendar.Month = 12
                     Calendar.Year = Calendar.Year - 1
                  Endif
               Else
                  Calendar.Month = Calendar.Month - 1
               Endif
               Calendar!ResyncDatePick()
               Calendar!GeneratePickList()
            Case Key = Asc("PgDn"):
               If (Calendar.Month = 12) Then
                  If (Calendar.Year < 9999) Then
                     Calendar.Month = 1
                     Calendar.Year = Calendar.Year + 1
                  Endif
               Else
                  Calendar.Month = Calendar.Month + 1
               Endif
               Calendar!ResyncDatePick()
               Calendar!GeneratePickList()
            Case Key = Asc("Home"):
               DatePick = Calendar.MapToPickList[Calendar.Offset + 1]
            Case Key = Asc("End"):
               DatePick = Calendar.MapToPickList[Calendar.Offset + Calendar.HighDay]
         EndSwitch
         RefreshDialog
         OK = False
   EndSwitch
   Return OK
EndProc ; Calendar!EventResponse

writelib Libname Calendar!EventResponse
release procs Calendar!EventResponse
? "Calendar!EventResponse()"
ColumnSet()





Proc Calendar!MovetoToday()
   Date = Today()
   Calendar.Day = Day(Date)
   Calendar.Month = Month(Date)
   Calendar.Year = Year(Date)
   Calendar!GeneratePickList()
   SelectControl "Pick"
   RefreshDialog
   Return True
EndProc

writelib Libname Calendar!MovetoToday
release procs Calendar!MovetoToday
? "Calendar!MovetoToday()"
ColumnSet()





Proc Calendar!ChangeMonth()
   Private
      MonthPick,
      ControlVar
   MonthPick = Calendar.Month
   ShowDialog "Select month"
      @8,30 Height 8 Width 27
      PickArray
         @0,1 Height 3 Width 23
         Columns 4
         Calendar.MonthName
         Tag "Pick"
         To MonthPick
      PushButton
         @4,2 Width 10
         "~O~K"
         OK
         Default
         Value "OK"
         Tag "OK"
         To ControlVar
      PushButton
         @4,14 Width 10
         "~C~ancel"
         Cancel
         Value "Cancel"
         Tag "Cancel"
         To ControlVar
   EndDialog
   If retval Then
      Calendar.Month = MonthPick
      Calendar!ResyncDatePick()
      Calendar!GeneratePickList()
      RefreshDialog
   Endif
   SelectControl "Pick"
   Return True
EndProc

writelib Libname Calendar!ChangeMonth
release procs Calendar!ChangeMonth
? "Calendar!ChangeMonth()"
ColumnSet()





Proc Calendar!ChangeYear()
   Private
      ControlVar,
      InputYear
   RefreshControl "Pick"
   InputYear = Calendar.Year
   ShowDialog "Select year"
      @8,30 Height 7 Width 27
      @1,1 ?? "Enter year: "
      Accept
         @1,13 Width 7
         "S" Min 100 Max 9999 Required Tag "Accept"
         To InputYear
      PushButton
         @3,2 Width 10
         "~O~K"
         OK
         Default
         Value "OK"
         Tag "OK"
         To ControlVar
      PushButton
         @3,14 Width 10
         "~C~ancel"
         Cancel
         Value "Cancel"
         Tag "Cancel"
         To ControlVar
   EndDialog
   If RetVal Then
      Calendar.Year = InputYear
      Calendar!ResyncDatePick()
      Calendar!GeneratePickList()
      RefreshDialog
   Endif
   SelectControl "Pick"
   Return True
EndProc

writelib Libname Calendar!ChangeYear
release procs Calendar!ChangeYear
? "Calendar!ChangeYear()"
ColumnSet()





Proc Calendar!ResyncDatePick()
   Private
      MapPointer
   If IsBlank(Calendar.PickList[DatePick]) Then
      MapPointer = Calendar.MapToMap[DatePick]
      If MapPointer > 15 Then
         DatePick = Calendar.MapToPickList[Calendar.Offset + Calendar.HighDay]
         Calendar.Day = Calendar.HighDay
      Else
         DatePick = Calendar.MapToPickList[Calendar.Offset + 1]
         Calendar.Day = 1
      Endif
   Else
      Calendar.Day = Calendar.PickList[DatePick]
   Endif
EndProc

writelib Libname Calendar!ResyncDatePick
release procs Calendar!ResyncDatePick
? "Calendar!ResyncDatePick()"
ColumnSet()





Proc Calendar.GetDate(DayOfMonth,MonthOfYear,CalendarYear)
   Return DateVal(StrVal(MonthOfYear) + "/" +
                  StrVal(DayOfMonth) + "/" +
                  StrVal(CalendarYear))
EndProc

writelib Libname Calendar.GetDate
release procs Calendar.GetDate
? "Calendar.GetDate()"
ColumnSet()





Proc Calendar.GetHighDay(MonthOfYear,CalendarYear)
   Private
      MonthOfYear
   If MonthOfYear = 12 Then
      Return 31 ; December
   Else
      MonthOfYear = MonthOfYear + 1
      Return Day(Calendar.GetDate(1,MonthOfYear,CalendarYear) - 1)
   Endif
EndProc

writelib Libname Calendar.GetHighDay
release procs Calendar.GetHighDay
? "Calendar.GetHighDay()"
ColumnSet()





;Calendar.Constructor()
;Calendar.Dialog(Today())
;Calendar.Destructor()
