; Licensed software Copyright (C) 1993 Financial Modeling Specialists, Inc.
;
; 04/10/92-Created by Ned Levy
; 04/07/93-Modified by Luke Chung

; 1993 Paradox Conference in Washington, DC
;
; Presentation: Developing Ad Hoc Query Modules in Paradox DOS Applications
;
;           by: Luke Chung
;               President
;               Financial Modeling Specialists, Inc.
;               8027 Leesburg Pike, Suite 410
;               Vienna, Virginia 22182
;               USA
;               703-356-4700, fax 703-448-3861
;               Compuserve: 73710,463

; OVERVIEW
; This script contains a sample ad hoc query system.  Most of the procedures
; are discussed in this presentation's paper.
; Refer to the paper for more details.
;
; This script demonstrates a simple ad hoc query system for a customer
; invoicing system.  There are two tables: customer and invoice.
; The tables are linked by the [Customer No] field.  A Query by Form (QBF)
; method is used to show an easy-to-use yet powerful method for retrieving
; data.  The basic steps of a flexible ad hoc query system are illustrated:
;            - Data grouping
;            - Data filtering
;            - Queries (individual tables & combining results)
;            - Data retrieval (output generation)
;            - Data viewing and printing

; PARADOX SORT ORDER
; Because of different Paradox sort orders, this program automatically
; indexes the tables if key fields are not found.  Should you have a problem
; due to sort order differences, just delete all the *.PX files.

; NOTE
; This presentation and script was first given at the 1992 Conference in
; Palm Springs.  Modifications have been made to show Paradox 4 techniques.

; DISCLAIMER
; This script is for reference only and is provided on an "as is" basis.
; You are welcome to use these scripts in your applications, but neither
; Financial Modeling Specialists, Inc. nor Luke Chung offer any guarantees
; or warranties to its usability.  Steps have not been taken to fully
; "bullet-proof" this application.  All liabilities are assumed by the user.
; You are welcome to use the procedures in this script provided you include
; our copyright notice at the top of your script.

; PROCEDURE        DESCRIPTION               CREATED     MODIFIED    CALLED BY
; QueryDemo.u      Main menu                 04/13/92-LC 04/23/92-LC
; CheckKeys.u      Key tables                04/22/92-LC             QueryDemo.u
; HelpText.u       Help screen               04/13/92-LC 04/13/92-LC QueryDemo.u
; ViewTable.u      View table with forms     04/13/92-LC 04/23/92-LC QueryDemo.u
; LateInvoices.u   Late invoice query        04/13/92-LC             QueryDemo.u
; QueryEngine.u    Main query procedure      04/10/92-NL 04/07/93-LC QueryDemo.u, LateInvoices.u
; SetVars.u        Set program vars          04/10/92-NL 04/12/92-LC QueryEngine.u
; QueryData.l      Coordinate query          04/10/92-NL             QueryEngine.u
; SelectGroup.l    Select grouping           04/10/92-NL 09/21/93-LC QueryEngine.u
; EnterFilter.l    Enter filter criteria     04/10/92-NL 09/21/93-LC QueryEngine.u
; QueryVars.u      Retrieve query filters    04/10/92-NL 04/07/93-LC QueryData.l
; QueryInvoice.l   Perform invoice query     04/10/92-NL 04/07/93-LC QueryData.l
; QueryCustomer.l  Perform customer query    04/10/92-NL 04/07/93-LC QueryData.l
; QueryOrders.l    Perform orders query      04/07/93-LC             QueryData.l
; CombineTables.l  Combine results           04/10/92-NL 04/07/93-LC QueryData.l
; CreateOutput.u   Main output procedure     04/10/92-NL 04/12/92-LC QueryEngine.u
; CreateInvoice.u  Create invoice output     04/10/92-NL 04/12/92-LC CreateOutput.u
; CreateCustomer.u Create customer output    04/10/92-NL 04/12/92-LC CreateOutput.u
; ViewOutput.u     View output results/print 04/10/92-NL 09/21/93-LC QueryEngine.u
; DoIt.l           Utility do_it! proc       04/10/92-NL 04/07/93-LC Several
; ShowMsg.u        Show top message          04/10/92-NL 04/07/93-LC Several
; PrintReport.u    Handle report printing    10/25/88-LC 05/01/91-LC ViewOutput.u
; PrintRep.l       Send report to printer    03/01/91-LC 05/01/91-LC *PrintReport.u
; MaxWindow.u      Maximize & remove frame   09/21/93-LC              EnterFilter.l, SelectGroup.l, ViewOutput.u

? "04/07/93-LC Compiling Query"
if not isassigned(Libname.a) then Libname.a = "adhoc" endif
if not isfile(Libname.a+".lib") then
  createlib Libname.a
endif

; Main menu of script.
; Called by
proc QueryDemo.u()
  private choice.a
  CheckKeys.u(Customer.t, 1)           ; Make sure tables are keyed.
  CheckKeys.u(Invoice.t, 2)
  choice.a = "Invoices"
  while TRUE
    HelpText.u("QueryDemo")
    showmenu
      "Invoices"  : "View invoice table.",
      "Customers" : "View customer table.",
      "Orders"    : "View orders table.",
      "Query"     : "Run ad hoc query.",
      "Late"      : "Get list of invoices 15 days late.",
      "Exit"      : "Leave application."
      default choice.a
    to choice.a
    switch
      case choice.a = "Esc"       : quitloop
      case choice.a = "Exit"      : quitloop
      case choice.a = "Invoices"  : ViewTable.u(Invoice.t,  "F", "1")
      case choice.a = "Customers" : ViewTable.u(Customer.t, "F", "")
      case choice.a = "Orders"    : ViewTable.u(Orders.t, "2", "")
      case choice.a = "Query"     : QueryEngine.u(Filter.t, TRUE)
      case choice.a = "Late"      : LateInvoices.u()
    endswitch
  endwhile
endproc
writelib libname.a QueryDemo.u
release procs      QueryDemo.u

; Check key fields.
; This application does not index the tables until you run it the first time.
; This allows it to use your sort order.
; INPUT: Table.t        Table to place key fields.
;        KeyFld.n       Number of key fields.
; Called by CreateReport.u.
proc CheckKeys.u(Table.t, KeyFld.n)
  private proc.a, x
  proc.a = "CheckKeys.u"
  if nkeyfields(Table.t) = 0 then
    ? "Keying "+Table.t+" table to your sort order"
    Menu {Modify} {Restructure}
      select Table.t
      right right
      for x from 1 to KeyFld.n
        "*"  down                           ; Add "*" to end of field type.
      endfor
    do_it!  clearall
  endif
endproc
writelib libname.a CheckKeys.u
release procs      CheckKeys.u

; Help screen for menus.
; INPUT: Menu.a         Determines which screen to display.
; Called by QueryDemo.u.
proc HelpText.u(Menu.a)
  canvas off
  clear
  @5,0
  switch
    case Menu.a = "QueryDemo" :
      text
                 AD HOC QUERY TECHNIQUES IN PARADOX APPLICATIONS


               INVOICES        View invoice table.

               CUSTOMERS       View customer table.

               ORDERS          View orders table.

               QUERY           Run ad hoc query.

               LATE            Get list of invoices 15 days late.
                                Sample of a "canned" query.

               EXIT            Exit application.
      endtext
  endswitch
  if monitor() = "Color" then
    paintcanvas attribute 31 0,0,24,79      ; Color the entire screen.
  endif
  canvas on
endproc
writelib libname.a HelpText.u
release procs      HelpText.u

; View a table and let user toggle between multi-record and multi-table forms.
; INPUT: Table.t        Table to view.
;        RecForm.a      Multi-record form.
;        TableForm.a    Multi-table form.
; Called by QueryDemo.u.
proc ViewTable.u(Table.t, RecForm.a, TableForm.a)
  private currForm.a, prompt.a
  currForm.a = RecForm.a          ; Start with multi-record form.
  view Table.t
    pickform currForm.a
    if TableForm.a = ""
      then prompt.a = "Press [Esc] to return."
      else prompt.a = "Press [Esc] to return, [F7] to switch forms."
    endif
    while TRUE
      wait table
        prompt prompt.a
      until "Esc", "F7"
      switch
        case retval = "Esc" : quitloop
        case retval = "F7" and TableForm.a <> "" :
          if currForm.a = RecForm.a
            then currForm.a = TableForm.a
            else currForm.a = RecForm.a
          endif
          pickform currForm.a
      endswitch
    endwhile
  clearall
endproc
writelib libname.a ViewTable.u
release procs      ViewTable.u

; Example of creating a non-ad hoc query using the ad hoc query engine.
; Just fill the filter table with the desired criteria, and run the engine.
; All the queries, data retrieval, viewing, etc. is handled!
; Some extra steps are taken to keep the current filter criteria.  That way,
; the user's last selections are not lost.  This makes this query seem totally
; disconnected from the ad hoc query section.
; Called by QueryDemo.u.
proc LateInvoices.u()
  private Original.r
  ShowMsg.u("Creating list of late invoices")
  edit Filter.t
    copytoarray Original.r             ; Keep current record to restore later.
    del                                ; Eliminate current values.
    [Invoice]  = ""                   ; Select Invoice as group.
    [ShipDateHigh] = today()-15        ; Enter criteria: 15 days late.
    [BalanceLow]   = 0                 ; Balance > 0.
  do_it!  clearall
  QueryEngine.u(Filter.t, False)
  edit Filter.t
    copyfromarray Original.r           ; Restore original values.
  do_it!  clearall
endproc
writelib libname.a LateInvoices.u
release procs      LateInvoices.u

; Main procedure to enter selections and perform query.
; INPUT: Filter.t       Table with filter criteria (may already contain data).
;        Enter.l        TRUE if user enters filters, FALSE if passing pre-selected data.
; Called by QueryDemo.u, LateInvoices.u.
proc QueryEngine.u(Filter.t, Enter.l)
  private Group.a, TempInv.t, TempCust.t, InvTemp.t, CustTemp.t, Combine.t,
          Output.t, ok.l
  ShowMsg.u("Loading query engine")
  clearall
  SetVars.u()
  if Enter.l then
    ok.l = SelectGroup.l(Filter.t, "F")         ;Returns TRUE if group selected, FALSE for [Esc].
    if ok.l then
      ShowMsg.u("Prepare to enter filter options")
      ok.l = EnterFilter.l(Filter.t, "1")       ;Returns TRUE if accepted, FALSE if cancelled.
    endif
  else
    ok.l = TRUE                                 ;Always OK if no entry needed.
  endif
  if ok.l then                                  ;Begin retrieving data.
    QueryData.l(Filter.t)                       ;Run queries to extract data.
    if retval then                              ;...returns FALSE if no data found.
      CreateOutput.u(Group.a, Combine.t)        ;Create output table.
      switch                                    ;View results.
        case Group.a = "Invoices"  : ViewOutput.u(Group.a, Output.t, "1", "F", "1")
        case Group.a = "Customers" : ViewOutput.u(Group.a, Output.t, "1", "", "1")
      endswitch
    else
      ShowMsg.u("No data found")
      sleep 1000
    endif
  endif
  ShowMsg.u("Leaving query engine")
  setdir directory()                            ;Delete temporary tables.
endproc
writelib libname.a QueryEngine.u
release procs      QueryEngine.u

; Set variables for query engine.
; Called by QueryEngine.u.
proc SetVars.u()
  TempInv.t    = "Entry01"    ; Temporary table--filtered list of invoices.
  TempCust.t   = "Entry02"    ; Temporary table--filtered list of customers.
  TempOrd.t    = "Entry03"    ; Temporary table--filtered list of customers.
  Combine.t    = "Entry04"    ; Temporary table--combined filtered lists.
  Output.t     = "Output"     ; Output table showing results to user.
  InvTemp.t    = "InvTemp"    ; Template table for Output.t list of invoices.
  CustTemp.t   = "CustTemp"   ; Template table for Output.t list of customers.
  OrderTemp.t  = "OrdTemp"    ; Template table for Output.t list of orders.
endproc
writelib libname.a SetVars.u
release procs      SetVars.u

; Create list of records meeting selection criteria.  Returns customer
; numbers if Group.a is "Customer" or invoice numbers if Group.a is "Invoice"
; INPUT: Filter.t       Filter table
; Return TRUE if combine table created, FALSE if not.
; Called by QueryEngine.u
proc QueryData.l(Filter.t)
  private shipL.d, shipH.d, salesL.n, salesH.n, balH.d, balL.d,
          custNo.s, state.a, product.a, name.a
  QueryVars.u(Filter.t)                         ;Sets variables from filter.t
  QueryInvoice.l(Invoice.t, TempInv.t)          ;Query invoice table.
  if retval then                                ;Continue if any invoices passed.
    QueryCustomer.l(Customer.t, TempCust.t)     ;Query customer table.
    if retval then                              ;Continue if any customers passed.
      QueryOrders.l(Orders.t, TempOrd.t)        ;Query order table.
      if retval then                            ;Continue if any orders passed.
        CombineTables.l(Group.a, Combine.t, TempInv.t, TempCust.t, TempOrd.t) ;Combine tables.
        return retval                             ;TRUE if data found.
      endif
    endif
  endif
 return FALSE
endproc
writelib libname.a QueryData.l
release procs      QueryData.l

; Display a form for user to select form with the [F6] key.
; INPUT: Group.t        Group selection table.
;        Form.a         Form to use for selection.
; Returns Group selected if [F2] pressed, blank if [Esc] pressed.
; Called by QueryEngine.u.
proc SelectGroup.l(Group.t, Form.a)
  private key.v                             ; Key pressed.
  edit Group.t                              ; Edit table with group options.
    pickform Form.a                         ; Use single record form.
    MaxWindow.u(TRUE)                       ; Maximize & remove frame.
    while TRUE
      wait record
        prompt "Move to group desired and press [F6] to select.",
               "Press [F2] to finish, [Esc] to cancel."
      until "F2", "Esc", "F6", "", "Backspace", "CtrlBackspace"
      key.v = retval
      switch
        case key.v = "F6" or key.v = "" :        ;Alt 251 is the check key.
          [Customer] = ""                         ;Remove all checks.
          [Invoice]  = ""
          []         = ""                        ;Place check in current field.
        case key.v = "F2" :
          switch                                  ;Determine if group selected.
            case [Customer] = "" : quitloop
            case [Invoice]  = "" : quitloop
            otherwise : message "Select group or cancel..."
                        sleep 1000
          endswitch
        case key.v = "Esc" : Canceledit quitloop
        otherwise : beep                          ;Prevent deleting checks
      endswitch
    endwhile
  do_it! clearimage
  return (key.v = "F2")
endproc
writelib libname.a SelectGroup.l
release procs      SelectGroup.l

; Retrieves filter criteria from user using QBF
; INPUT: Filter.t       Table with filter criteria to fill.
;        Form.a         Form on Filter.t to use for data entry.
; Returns TRUE if [F2] pressed, FALSE if [Esc] pressed.
; Called by QueryEngine.u
proc EnterFilter.l(Filter.t, Form.a)
  private key.v,                                 ; Key pressed by user.
          customer.a, invoice.a                  ; Grouping criteria.
  edit Filter.t                                  ; Retrieve table with filter criteria.
    pickform Form.a                              ; Display form showing fields to fill.
    MaxWindow.u(TRUE)                            ; Maximize and eliminate frame.
    while TRUE
      wait record                                ; Let user type anything in this record.
        prompt "Enter filter criteria desired.",
               "Press [F2] to finish, [Esc] to cancel, [Del] to clear."
      until "F2", "Esc", "Del", "Home", "End"
      key.v = retval
      switch
        case key.v = "F2"   : do_it! quitloop
        case key.v = "Esc"  : canceledit quitloop
        case key.v = "Del"  : customer.a = [Customer]       ; Save group fields.
                              invoice.a  = [Invoice]
                              del
                              [Customer] = customer.a
                              [Invoice]  = invoice.a
        case key.v = "Home" : CtrlHome
        case key.v = "End"  : CtrlEnd
      endswitch
    endwhile
  clearimage
  return (key.v = "F2")
endproc
writelib libname.a EnterFilter.l
release procs      EnterFilter.l

; Assign values selected during group and filter selection.
; INPUT: Filter.t       Table with filter criteria.
; Called by QueryData.l.
proc QueryVars.u(Filter.t)
  view Filter.t
    switch
      case [Customer] = "" : group.a = "Customers"
      case [Invoice]  = "" : group.a = "Invoices"
    endswitch
    name.a    = [Salesman]
    shipL.d   = [ShipDateLow]
    shipH.d   = [ShipDateHigh]
    salesL.n  = [SalesLow]
    salesH.n  = [SalesHigh]
    balL.n    = [BalanceLow]
    balH.n    = [BalanceHigh]
    custNo.s  = [Customer No]
    state.a   = [State]
    product.a = [Product No]
  clearimage
endproc
writelib libname.a QueryVars.u
release procs      QueryVars.u

; Query invoice table using values assigned during filter selection.
; INPUT: Invoice.t      Invoice table.
;        TempInv.t      List of invoices passing filter.
; Returns TRUE if records found, FALSE if not.
; If no filter on fields, TempInv.t does not exist.
; Called by QueryData.l
proc QueryInvoice.l(Invoice.t, TempInv.t)
  ShowMsg.u("Querying Invoice Table")           ;Show working message.
  Menu {Ask} select Invoice.t                   ;Query Invoice table.
    moveto[Invoice No]    Check                 ;Get list of invoices.
    moveto[Customer No]   Check                 ;Get link field to customer table.
    if not isblank(name.a) then
      [Salesman] = "~name.a"                    ;Salesman filter if selected.
    endif
    switch                                      ;Range criteria filter for ship dates.
      case isblank(shipL.d) and isblank(shipH.d) :
      case isblank(shipL.d) : [Ship Date] = "<= ~shipH.d"
      case isblank(shipH.d) : [Ship Date] = ">= ~shipL.d"
      otherwise             : [Ship Date] = ">= ~shipL.d, <= ~shipH.d"
    endswitch
    switch                                      ;Range criteria filter for sales.
      case isblank(salesL.n) and isblank(salesH.n) :
      case isblank(salesL.n) : [Total] = "<= ~salesH.n"
      case isblank(salesH.n) : [Total] = "> ~salesL.n"
      otherwise              : [Total] = "> ~salesL.n, <= ~salesH.n"
    endswitch
    switch                                      ;Range criteria filter for balance.
      case isblank(balL.n) and isblank(balH.n) :
      case isblank(balL.n) : [Balance] = "<= ~balH.n"
      case isblank(balH.n) : [Balance] = "> ~balL.n"
      otherwise            : [Balance] = "> ~balL.n, <= ~balH.n"
    endswitch
  DoIt.l(TRUE, TempInv.t)                       ;Run the query.
  if retval then
    return not isempty(TempInv.t)               ;If no data return FALSE
  else
    return FALSE
  endif
endproc
writelib libname.a QueryInvoice.l
release procs      QueryInvoice.l

; Query customer table using values assigned during filter selection.
; INPUT: Customer.t     Customer table.
;        TempCust.t     List of customers passing filter.
; Returns TRUE if answer records, FALSE if not.
; Called by QueryData.l
proc QueryCustomer.l(Customer.t, TempCust.t)
  ShowMsg.u("Querying Customer Table")           ; Show message.
  Menu {Ask} select Customer.t                   ; Query the customer table.
    moveto[Customer No]   Check                  ; Get list of customers.
    if not isblank(custNo.s) then                ; Customer filter.
      [Customer No] = "~custNo.s"
    endif
    if not isblank(state.a) then                 ; State filter.
      [State] = "~state.a"
    endif
  DoIt.l(TRUE, TempCust.t)                       ; Run the query.
  if retval then
    return not isempty(TempCust.t)               ; If no data return FALSE.
  else
    return FALSE                                 ; Query failed.
  endif
endproc
writelib libname.a QueryCustomer.l
release procs      QueryCustomer.l

; Query customer table using values assigned during filter selection.
; INPUT: Orders.t       Orders table.
;        TempOrd.t      List of invoices passing filter.
; Returns TRUE if answer records, FALSE if not.
; If no filter on orders, delete the TempOrd table and return TRUE.
; By not querying on orders if there is no filter (by definition, all invoices
; pass), the system runs faster.
; Called by QueryData.l
proc QueryOrders.l(Orders.t, TempOrd.t)
  private ok.l
  ShowMsg.u("Querying Orders Table")             ; Show message.
  if Product.a = "" then
    ok.l = TRUE
    if istable(TempOrd.t) then delete TempOrd.t endif
  else
    Menu {Ask} select Orders.t                   ; Query the orders table.
      moveto[Invoice No]     Check               ; Get list of invoices.
      if not isblank(Product.a) then             ; Product filter.
        [Product No] = "~Product.a"
      endif
    DoIt.l(TRUE, TempOrd.t)                      ; Run the query.
    ok.l = retval
    if ok.l then
      ok.l = not isempty(TempOrd.t)              ; If no data return FALSE.
    endif
  endif
  return ok.l
endproc
writelib libname.a QueryOrders.l
release procs      QueryOrders.l

; Combine filtered list of invoices and customers into list of Group.a.
; INPUT: Group.a        Group selected by user.
;        Combine.t      Table to be created with list passing filter.
;        TempInv.t      Invoices passing invoice filter.
;        TempCust.t     Customers passing customer filter.
; Return TRUE if anything passed, FALSE if no data passed.
; Called by QueryData.l
proc CombineTables.l(Group.a, Combine.t, TempInv.t, TempCust.t, TempOrd.t)
  ShowMsg.u("Combining Results")                ; Show working message.
  switch
    case Group.a = "Invoices"  :
      Menu {Ask} select TempInv.t
        moveto[Invoice No]  Check "_i"          ; Get invoice numbers.
        [Customer No]       = "_c"              ; Place example link.
      Menu {Ask} select TempCust.t
        [Customer No]       = "_c"              ; Place example link.
      if istable(TempOrd.t) then
        Menu {Ask} select TempOrd.t             ; TempOrd.t does not exist if no filters are on orders.
          [Invoice No] = "_i"                   ; Link on invoice number.
      endif
    case Group.a = "Customers" :
      Menu {Ask} select TempCust.t
        moveto[Customer No] Check "_c"          ;Place example link and get customer numbers.
      Menu {Ask} select TempInv.t
        [Invoice No]   = "_i"
        [Customer No]  = "_c"
      if istable(TempOrd.t) then
        Menu {Ask} select TempOrd.t             ; TempOrd.t does not exist if no filters are on orders.
          [Invoice No] = "_i"                   ; Link on invoice number.
      endif
  endswitch
  DoIt.l(TRUE, Combine.t)                       ;Run query.
  return not isempty(Combine.t)
endproc
writelib libname.a CombineTables.l
release procs      CombineTables.l

; Main procedure controlling output table creation.
; INPUT: Group.a        Group selected to view.
;        Combine.t      Table with records passing filter.
proc CreateOutput.u(Group.a, Combine.t)
  switch
    case Group.a = "Invoices"  : CreateInvoice.u(Output.t,  InvTemp.t,  Combine.t)
    case Group.a = "Customers" : CreateCustomer.u(Output.t, CustTemp.t, Combine.t)
  endswitch
endproc
writelib libname.a CreateOutput.u
release procs      CreateOutput.u

; Procedure for generating Invoice Group output table.
; INPUT: Output.t       Table to be created.
;        Template.t     Empty table with reports, forms, field settings, etc.
;        Combine.t      Table of records passing filter.
; Called by CreateOutput.u.
proc CreateInvoice.u(Output.t, Template.t, Combine.t)
  ShowMsg.u("Creating Invoice Output")          ;Show working message.
  copy Template.t Output.t                      ;Copy table which will contain the new data.
  Menu {Ask} select Combine.t                   ;Query list of filtered records.
    [Invoice No]  = "_i"                        ;Place example link.
  Menu {Ask} select Invoice.t                   ;Query invoice table.
    [Invoice No]  = "_i"
    [Customer No] = "_c"
    [Salesman]    = "_sm"
    [Order Date]  = "_od"
    [Ship Date]   = "_sd"
    [Ship Type]   = "_st"
    [Sub-total]   = "_sub"
    [Taxes]       = "_tax"
    [Total]       = "_tot"
    [Paid]        = "_paid"
  Menu {Ask} select Customer.t                  ;Add customer information to output table.
    [Customer No] = "_c"
    [Name]        = "_name"
    [State]       = "_state"
  Menu {Ask} select Output.t                    ;Put data into output table.
   "fast insert"                                ;Use an INSERT query.
    [Invoice No]  = "_i"
    [Customer No] = "_c"
    [Name]        = "_name"
    [State]       = "_state"
    [Salesman]    = "_sm"
    [Order Date]  = "_od"
    [Ship Date]   = "_sd"
    [Days Late]   = "today - _sd"               ;Calc days late.
    [Ship Type]   = "_st"
    [Sub-total]   = "_sub"
    [Taxes]       = "_tax"
    [Total]       = "_tot"
    [Paid]        = "_paid"
    [Balance]     = "_tot - _paid"              ;Calculations can be made "on-the-fly"
  DoIt.l(TRUE, "")                              ;Run the query.
  sort Output.t on "Invoice No"
  clearall
endproc
writelib libname.a CreateInvoice.u
release procs      CreateInvoice.u

; Procedure for generating Customer Group output table.
; INPUT: Output.t       Table to be created.
;        Template.t     Empty table with reports, forms, field settings, etc.
;        Combine.t      Table of records passing filter.
; Called by CreateOutput.u.
proc CreateCustomer.u(Output.t, Template.t, Combine.t)
  ShowMsg.u("Creating Customer Output")
  copy Template.t Output.t                      ;Copy table which will contain the new data.
  Menu {Ask} select Combine.t                   ;Query list of filtered records.
    [Customer No]  = "_c"
  Menu {Ask} select Customer.t                  ;Query customer table.
    [Customer No] = "_c"
    [Name]        = "_name"
    [Address]     = "_add"
    [City]        = "_city"
    [State]       = "_st"
    [Zip Code]    = "_zip"
  Menu {Ask} select Output.t                    ;Put data into output table.
   "fast insert"                                ;Use an INSERT query.
    [Customer No] = "_c"
    [Name]        = "_name"
    [Address]     = "_add"
    [City]        = "_city"
    [State]       = "_st"
    [Zip Code]    = "_zip"
  DoIt.l(TRUE, "")                              ;Run the query.
  sort Output.t on "Customer No"
  clearall
endproc
writelib libname.a CreateCustomer.u
release procs      CreateCustomer.u

; View results from query.
; INPUT: Group.a        Group selected.
;        Output.t       Table with ad hoc query results.
;        Form1.a        First form to use.
;        Form2.a        Second form to use.
;        Report.a       Report to use.
; Called by QueryEngine.u
proc ViewOutput.u(Group.a, Output.t, Form1.a, Form2.a, Report.a)
  private form.a, key.v, multiForm.l
  form.a = Form1.a
  multiForm.l = FALSE
  view Output.t
    if form.a <> "" then
      pickform form.a
      multiForm.l = ismultiform(Output.t, form.a)
    endif
    MaxWindow.u(TRUE)                       ; Maximize & remove frame.
    while TRUE
      wait table
        prompt "Filter results.",
               "Press [Esc] to return, [F7] to toggle form, [Ctrl-P] to print report."
      until "Esc", "F7", "F3", "F4", 16
      key.v = retval
      switch
        case key.v = "Esc"   : quitloop
        case key.v = "F7"    :
          if form.a = Form1.a
            then form.a = Form2.a
            else form.a = Form1.a
          endif
          if form.a = "" then
            formkey                        ; Toggles table and form views.
          else
            pickform form.a                ; Switches forms.
            multiForm.l = ismultiform(Output.t, form.a)
          endif
          MaxWindow.u(FALSE)                      ; Maximize & remove frame.
        case key.v = "F3" or key.v = "F4" :
          if multiForm.l then
            keypress key.v
          endif
        case key.v = 16      :
          if Report.a <> "" then
            PrintReport.u(table(), Report.a)
          endif
      endswitch
    endwhile
  clearall
endproc
writelib libname.a ViewOutput.u
release procs      ViewOutput.u

; Executes the query on the workspace and checks if it has failed.
; INPUT: Clear.l        TRUE clears workspace, FALSE does nothing.
;        Rename.t       Table to rename ANSWER if given.
; Returns TRUE if okay, FALSE if not.
; Called by several.
proc DoIt.l(Clear.l, Rename.t)
  private error.l
  do_it!                                    ; Run query.
  error.l = (window() <> "")                ; Check if an error message is given.
  ; For Paradox 4 set to continuous queries (no restart), in a multi-user
  ; situation, the window will display the number of times the tables changed
  ; during the query: "Other users changed query tables # times during query"
  ; The query is successful, but WINDOW() is not blank.
  if error.l and version() >= 4 then
    error.l = not match(window(), "Other users changed query ..")
  endif
  if error.l then                           ; If window is not blank an error occurred.
    message window()," press any key..."    ; Show message to user.
    retval = getchar()                      ; Wait for keypress.
    return FALSE                            ; Tell calling proc that query failed.
  else
    if Rename.t <> "" then
      rename "Answer" Rename.t
    endif
    if Clear.l then clearall endif          ; Clear images.
    return TRUE                             ; Return TRUE if query ran.
  endif
endproc
writelib libname.a DoIt.l
release procs      DoIt.l

; Show a message with blinking dots.
; INPUT: Msg.a          Message to display.
; Called by several.
proc ShowMsg.u(Msg.a)
  cursor off
  if monitor() = "Color"
    then style attribute 14
    else style reverse
  endif
    @0,0 ?? format("w80,ac",Msg.a)   ; Display the message.
  if monitor() = "Color"
    then style attribute 14+128
    else style reverse, blink
  endif
    @0, int(40+len(Msg.a)/2)
    ?? "...."
  style
endproc
writelib libname.a ShowMsg.u
release procs      ShowMsg.u

; Show reports attached to table and send to screen or printer.
; INPUT: Table.t        Table to print.
;        Rep.a          Report to print.
; Called by ViewOutput.u.
proc PrintReport.u(Table.t, Rep.a)
  private proc.a, x, M, temp.t
  proc.a = "PrintReport.u"
  showmenu
    "Printer" : "Send report to printer.",
    "Screen"  : "Send report to screen."
  to M
  @0,0 clear eol
  @1,0 clear eol
  switch
    case M = "Printer" : PrintRep.l(Table.t, Rep.a)
    case M = "Screen" :
      ShowMsg.u("Sending report to screen")
      if isfile(sdir()+"README.COM") then
        temp.t = privdir() + "xxxxtemp"                      ; Temp file
        Menu {Report} {SetPrinter} {Override} {Setup} { }
          Menu {Report} {Output} select Table.t select Rep.a {File} ; Send to file.
            select Temp.t+".sc"
          if menuchoice() = "Cancel" then {Replace} endif
          run norefresh sdir()+"README "+Temp.t+".sc"
          Menu {Tools} {Delete} {Script} select Temp.t {Ok}
        Menu {Report} {SetPrinter} {Regular}
      else
        Menu {Report} {Output} select Table.t select Rep.a {Screen}
      endif
    case M = "Esc" :     ; do nothing.
  endswitch
endproc
writelib libname.a PrintReport.u
release procs      PrintReport.u

; Handles printing reports to printer.
; If printer is not ready, the user is asked to retry or cancel.
; INPUT: Report.t   Table with report to print.
;        Rep.a      Report to print.
; Returns TRUE if okay, FALSE if cancelled by user.
; Called by PrintReport.u
proc PrintRep.l(Report.t, Rep.a)
  private proc.a, x
  proc.a = "PrintRep.l"
  if not istable(Report.t) or isempty(Report.t) then
    return TRUE
  else
    while TRUE
      ShowMsg.u("Checking printer")
      if printerstatus() then
        ShowMsg.u("Sending report to printer")
        report Report.t Rep.a
        return TRUE
      else
        clear
        style reverse
          @22,0  beep
          ?? format("w80,ac", "Printer not ready.   Press any key to retry, [Esc] to cancel.")
        style
        while charwaiting() x = getchar() endwhile      ; Clear buffer.
        x = getchar()                                   ; Get character
        if x = 27 then return FALSE endif
        clear
      endif
    endwhile
  endif
endproc
writelib libname.a PrintRep.l
release procs      PrintRep.l

; Maximize the current window and remove its frame if it is a form.
; INPUT: MenuRemove.l   TRUE removes menu, FALSE leaves menu alone.
; Called by EnterFilter.l, SelectGroup.l, ViewOutput.u
proc MaxWindow.u(MenuRemove.l)
  private win.w, winMax.r                        ; Window settings.
  dynarray winMax.r[]
  window handle current to win.w
  winMax.r["Maximized"] = TRUE
  winMax.r["HasFrame"]  = not isformview()
  window setattributes win.w from winMax.r     ; Maximize and eliminate frame.
  if MenuRemove.l then
    showpulldown endmenu                         ; Remove menu.
  endif
endproc
writelib libname.a MaxWindow.u
release procs      MaxWindow.u



; Alternatives to using QUERY..ENDQUERY statements:
;
; Query                                 ; query saved with {Querysave}.
;
;   Invoice  | Invoice No |        Ship Date        |  Balance |
;            | Check      | Check <=today - ~days.n | Check >0 |
;
; Endquery
; Do_it!  clearall
;
;
; This can be translated to:
;
; clearall                           ; Make sure other queries are not on workspace.
; Menu {Ask} select Invoice.t        ; Where Invoice.t = "Invoice"
;   moveto[Invoice No]  Check
;   moveto[Ship Date]   Check  "<= today - ~days.n"
;   moveto[Balance]     Check  "> 0"
; Do_it!  clearall
;
;
; If you are not placing a check in a field, you can assign its criteria
; just like placing a value in a field.:
;
;                              [Balance] = "> 0"
