; Licensed software Copyright (C) 1993 Financial Modeling Specialists, Inc.
; Sample dialog box query routine.
; Dialog.sc
; 02/03/93-LC

if not isassigned(Libname.a) then Libname.a = "adhoc" endif

; OVERVIEW
; Example of using a dialog box for filter input.

; PROCESS
; Display dialog box.  When completed, process results.

; PROCEDURE      DESCRIPTION               CREATED     MODIFIED    CALLED BY
; QueryEngine4.u Query engine in 4         02/03/93-LC 04/07/93-LC
; DialogFilter.l Dialog filter routine     02/03/93-LC 04/07/93-LC *QueryEngine4.u
; DialogCtrl.l   Dialog box control proc   02/03/93-LC 04/07/93-LC *DialogFilter.l
; DlgLookup.a    Lookup list for dlg box   04/07/93-LC             *DialogCtrl.l
; KeyCode.v      Keycode translation       07/20/92-LC 09/28/92-LC *DialogCtrl.l
; DlgMsg.u       Dialog message box        07/16/92-LC 09/08/92-LC  several
; KeyMsg4.n      Trap for key or mouse     09/06/92-LC              KeyMsg.n, DlgMsg.u, DlgMsg.n
; ViewOutput4.u  View output results/print 04/10/92-NL 04/08/93-LC *QueryEngine4.u

; MODIFICATIONS

; Main procedure to enter selections and perform query.
; INPUT: Filter.t       Table with filter criteria (may already contain data).
;        Enter.l        TRUE if user enters filters, FALSE if passing pre-selected data.
; Called by QueryDemo.u, LateInvoices.u.
proc QueryEngine4.u(Filter.t, Enter.l)
  private TempInv.t, TempCust.t, InvTemp.t, CustTemp.t, Combine.t,
          Output.t, ok.l
  DlgMsg.u_FMS("Loading query engine", "", FALSE)
  clearall
  SetVars.u()
  if Enter.l then
    ok.l = DialogFilter.l(Filter.t)             ; Get user input.
  else
    ok.l = TRUE                                 ; Always OK if no entry needed.
  endif
  if ok.l then                                  ; Begin retrieving data.
    QueryData.l(Filter.t)                       ; Run queries to extract data.
    if retval then                              ; ...returns FALSE if no data found.
      view Filter.t
        if [Invoice] = ""
          then Group.a = "Invoices"
          else Group.a = "Customers"
        endif
      clearimage
      CreateOutput.u(Group.a, Combine.t)        ; Create output table.
      switch                                    ;View results.
        case Group.a = "Invoices"  : ViewOutput4.u(Group.a, Output.t, "1", "F", "1")
        case Group.a = "Customers" : ViewOutput4.u(Group.a, Output.t, "1", "", "1")
      endswitch
    else
      ShowMsg.u("No data found")
      sleep 1000
    endif
  endif
  DlgMsg.u_FMS("Leaving query engine", "", FALSE)
  setdir directory()                            ;Delete temporary tables.
  window select Msg.w
  window close
endproc
writelib libname.a QueryEngine4.u
release procs      QueryEngine4.u

; Filter input in a dialog box.
; INPUT: Filter.t       Input table with previous filter selections.
;                       User selections are placed here.
; Called by QueryEngine4.u
proc DialogFilter.l(Filter.t)
  private Salesman.t, Group.n, Salesman.a, ShipLow.d, ShipHigh.d,
          SalesLow.n, SalesHigh.n, BalanceLow.n, BalanceHigh.n,
          Customer.s, State.a, Product.a, Product.t, ok.l
  Salesman.t = "Salesman"                              ; Lookup table.
  Product.t  = "Product"

  ; Set initial values of dialog box.
  view Filter.t
    if [Invoice] = ""
      then Group.n = 1
      else Group.n = 2
    endif
    Salesman.a    = [Salesman]
    SalesLow.n    = [SalesLow]
    SalesHigh.n   = [SalesHigh]
    ShipLow.d     = [ShipDateLow]
    ShipHigh.d    = [ShipDateHigh]
    BalanceLow.n  = [BalanceLow]
    BalanceHigh.n = [BalanceHigh]
    Customer.s    = [Customer No]
    State.a       = [State]
    Product.a     = [Product No]
  clearimage

  showpulldown endmenu                 ; Remove menus
  DlgMsg.u_FMS("", "", FALSE)          ; Remove message box

  SHOWDIALOG "Enter Filter Options"
    PROC "DialogCtrl.l"
    @2,1 HEIGHT 21 WIDTH 78

    FRAME SINGLE FROM 7,1 TO 15,46
    PAINTCANVAS ATTRIBUTE 127 7,1,7,46
    PAINTCANVAS ATTRIBUTE 112 15,1,15,46
    PAINTCANVAS ATTRIBUTE 127 7,1,15,1
    PAINTCANVAS ATTRIBUTE 112 7,46,15,46

    FRAME SINGLE FROM 4,48 TO 9,74
    PAINTCANVAS ATTRIBUTE 127 4,48,4,74
    PAINTCANVAS ATTRIBUTE 112 9,48,9,74
    PAINTCANVAS ATTRIBUTE 127 4,48,9,48
    PAINTCANVAS ATTRIBUTE 112 4,74,9,74

    FRAME DOUBLE FROM 1,14 TO 5,33
    PAINTCANVAS ATTRIBUTE 127 1,14,1,33
    PAINTCANVAS ATTRIBUTE 112 5,14,5,33
    PAINTCANVAS ATTRIBUTE 127 1,14,5,14
    PAINTCANVAS ATTRIBUTE 112 1,33,5,33

    FRAME SINGLE FROM 10,48 TO 15,74
    PAINTCANVAS ATTRIBUTE 127 10,48,10,74
    PAINTCANVAS ATTRIBUTE 112 15,48,15,74
    PAINTCANVAS ATTRIBUTE 127 10,48,15,48
    PAINTCANVAS ATTRIBUTE 112 10,74,15,74

    @2,19   ?? "Data Group"
    @5,54   ?? "Customer Fields"
    @8,52   ?? "State"
    @11,4   ?? "Sales"
    @11,14  ?? ">="
    @11,30  ?? "<="
    @8,16   ?? "Invoice Fields"
    @10,4   ?? "Salesman"
    @12,4   ?? "Ship Date"
    @12,14  ?? ">="
    @12,30  ?? "<="
    @13,4   ?? "Balance"
    @13,14  ?? ">"
    @13,30  ?? "<="

    @11,56  ?? "Order Fields"
    @13,52  ?? "Product"
    @7,52   ?? "Customer"

    RADIOBUTTONS @3,17 HEIGHT 2 WIDTH 14
      "Invoice",
      "Customer"
      TAG "Group"
    TO Group.n

    ACCEPT @10,17 WIDTH 6 "A3" PICTURE "*!" LOOKUP Salesman.t
      TAG "Salesman"
    TO Salesman.a

    ACCEPT @11,17 WIDTH 11 "N"
      TAG "SalesLow"
    TO SalesLow.n

    ACCEPT @11,33 WIDTH 11 "N"
      TAG "SalesHigh"
    TO SalesHigh.n

    ACCEPT @12,17 WIDTH 11 "D"
      TAG "ShipDateLow"
    TO ShipLow.d

    ACCEPT @12,33 WIDTH 11 "D"
      TAG "ShipDateHigh"
    TO ShipHigh.d

    ACCEPT @13,17 WIDTH 9 "N"
      TAG "BalanceLow"
    TO BalanceLow.n

    ACCEPT @13,33 WIDTH 9 "N"
      TAG "BalanceHigh"
    TO BalanceHigh.n

    ACCEPT @7,61 WIDTH 9 "S" MIN 0
      TAG "Customer"
    TO Customer.s

    ACCEPT @8,61 WIDTH 5 "A2" PICTURE "!!"
      TAG "State"
    TO State.a

    ACCEPT @13,61 WIDTH 11 "A8" PICTURE "*!" LOOKUP Product.t
      TAG "Product"
    TO Product.a

    PUSHBUTTON @17,24 WIDTH 10
      "~O~k"
      OK DEFAULT
      VALUE "Ok"
      TAG "Ok"
    TO Tag.a

    PUSHBUTTON @17,41 WIDTH 10
      "~C~ancel"
      CANCEL
      VALUE "Cancel"
      TAG "Cancel"
    TO Tag.a
  ENDDIALOG

  ok.l = retval
  if ok.l then             ; Store results in Filter table.
    DlgMsg.u_FMS("Retrieving data", "", FALSE)
    coedit Filter.t
      if Group.n = 1 then
        [Invoice]  = ""
        [Customer] = ""
      else
        [Invoice]  = ""
        [Customer] = ""
      endif
      [Salesman]     = Salesman.a
      [SalesLow]     = SalesLow.n
      [SalesHigh]    = SalesHigh.n
      [ShipDateLow]  = ShipLow.d
      [ShipDateHigh] = ShipHigh.d
      [BalanceLow]   = BalanceLow.n
      [BalanceHigh]  = BalanceHigh.n
      [Customer No]  = Customer.s
      [State]        = State.a
      [Product No]   = Product.a
    do_it! clearimage
  endif
  return ok.l
endproc
writelib Libname.a DialogFilter.l
release procs  DialogFilter.l

; Dialog box control procedure.
; INPUT: standard dialog proc parameters.
; Returns TRUE if finished.
; Called by DialogFilter.l
proc DialogCtrl.l(Trigger.a, Tag.a, Event.r, Element.a)
  private key.v
  switch
    case Trigger.a = "EVENT" and Event.r["Type"] = "KEY" :
      key.v = KeyCode.v(Event.r["Keycode"])
      if key.v = "F1" then
        switch
          case Tag.a = "Salesman" :
            DlgLookup.a(Salesman.t, "Salesman ID", "Name")
            if retval <> "" then
              Salesman.a = retval
            endif
          case Tag.a = "Customer" :
            DlgLookup.a(Customer.t, "Customer No", "Name")
            if retval <> "" then
              Customer.s = numval(retval)        ; Since field is numeric (short).
            endif
          case Tag.a = "Product" :
            DlgLookup.a(Product.t, "Product No", "Description")
            if retval <> "" then
              Product.a = retval
            endif
          otherwise : beep
                      retval = ""
        endswitch
        if retval <> "" then
          REFRESHCONTROL Tag.a
        endif
     endif
  endswitch
endproc
writelib Libname.a DialogCtrl.l
release procs      DialogCtrl.l

; Create popup menu of elements from a table for lookup selection in
; dialog boxes.  The values to select are in the menu and their descriptions
; are displayed in the prompt line.
; INPUT: Lookup.t       Table with lookup values to present.
;        ItemField.a    Field with items to display.
;        DescField.a    Description field to put in prompt.
; Returns value of item selected, blank if [Esc] pressed.
; Called by DialogCtrl.l
proc DlgLookup.a(Lookup.t, ItemField.a, DescField.a)
  private recs.n, item.r, desc.r, x, M
  recs.n = nrecords(Lookup.t)
  array item.r[recs.n]
  array desc.r[recs.n]
  view Lookup.t
    x = 1
    moveto field ItemField.a
    scan
      item.r[x] = strval([])
      x = x + 1
    endscan
    if DescField.a = "" then
      for x from 1 to recs.n
        desc.r[x] = item.r[x]
      endfor
    else
      moveto field DescField.a
      x = 1
      scan
        desc.r[x] = strval([])
        x = x + 1
      endscan
    endif
  clearimage
  showarray
    item.r desc.r
  to M
  if M <> "Esc"
    then return M
    else return ""
  endif
endproc
writelib Libname.a DlgLookup.a
release procs      DlgLookup.a

; Translates a keycode to a PAL code.
; INPUT: KeyCode.n      Keycode provided by Paradox.
; Returns ASCII representation of key pressed (e.g. "F2" instead of -60).
; Called by MainDlgSet.l_FMS
proc KeyCode.v(KeyCode.n)
  private key.v
  switch
    case (KeyCode.n <= -59)  and (KeyCode.n >= -68)  : key.v = "F"+strval(-KeyCode.n-58)    ; F1..F10
    case (KeyCode.n <= -84)  and (KeyCode.n >= -92)  : key.v = "F1"+strval(-KeyCode.n-83)   ; F11..F19  Shift
    case (KeyCode.n <= -93)  and (KeyCode.n >= -102) : key.v = "F2"+strval(-KeyCode.n-93)   ; F20..F29  Ctrl
    case (KeyCode.n <= -103) and (KeyCode.n >= -112) : key.v = "F3"+strval(-KeyCode.n-103)  ; F30..F39  Alt
    case KeyCode.n = 4    : key.v = "Ditto"                ; Ctrl-D
    case KeyCode.n = 6    : key.v = "FieldView"            ; Ctrl-F
    case KeyCode.n = 8    : key.v = "BackSpace"
    case KeyCode.n = 9    : key.v = "Tab"
    case KeyCode.n = 13   : key.v = "Enter"
    case KeyCode.n = 26   : key.v = "Zoom"
    case KeyCode.n = 27   : key.v = "Esc"
    case KeyCode.n = -44  : key.v = "ZoomNext"
    case KeyCode.n = -71  : key.v = "Home"
    case KeyCode.n = -72  : key.v = "Up"
    case KeyCode.n = -73  : key.v = "PgUp"
    case KeyCode.n = -75  : key.v = "Left"
    case KeyCode.n = -77  : key.v = "Right"
    case KeyCode.n = -79  : key.v = "End"
    case KeyCode.n = -80  : key.v = "Down"
    case KeyCode.n = -81  : key.v = "PgDn"
    case KeyCode.n = -82  : key.v = "Ins"
    case KeyCode.n = -83  : key.v = "Del"
    case KeyCode.n = -115 : key.v = "CtrlLeft"
    case KeyCode.n = -116 : key.v = "CtrlRight"
    case KeyCode.n = -117 : key.v = "CtrlEnd"
    case KeyCode.n = -118 : key.v = "CtrlPgDn"
    case KeyCode.n = -119 : key.v = "CtrlHome"
    case KeyCode.n = -132 : key.v = "CtrlPgUp"
    case KeyCode.n = -15  : key.v = "ReverseTab"
    case KeyCode.n >= 48 and KeyCode.n <= 122 : key.v = upper(chr(KeyCode.n))
    otherwise : key.v = KeyCode.n          ; Other keys.
  endswitch
  return key.v
endproc
writelib Libname.a KeyCode.v
release procs  KeyCode.v

; Dialog message with option to press a key.
; INPUT: Msg1.a         Line one message.
;        Msg2.a         Line two message.
;        PressKey.l     TRUE if key is pressed, FALSE if not.
; Creates Msg.w   Message handle.  Message box remains throughout program.
;                 Message FLOATING status is moved.
; Called by several.
proc DlgMsg.u_FMS(Msg1.a, Msg2.a, PressKey.l)
  private proc.a, win.r, current.w, height.n, width.n, len1.n, len2.n,
          style.n, close.r
  proc.a = "DlgMsg.u_FMS"
  if Msg1.a + Msg2.a = "" then         ; Remove message.
    if isassigned(Msg.w) and iswindow(Msg.w) then
      dynarray Close.r[]
      close.r["originrow"] = -10
      close.r["origincol"] = -80
      WINDOW SETATTRIBUTES Msg.w FROM close.r
    endif
  else
    height.n = 3
    if Msg2.a <> "" then height.n = height.n + 1 endif
    if PressKey.l   then height.n = height.n + 1 endif
    width.n  = 50
    WINDOW HANDLE CURRENT TO current.w
    DYNARRAY win.r[]
    if PressKey.l
      then style.n = 79  ; WarnColor.n
      else style.n = 47  ; BoxColor.n
    endif
    win.r["style"]    = style.n
    win.r["width"]    = width.n
    win.r["height"]   = height.n
    if not isassigned(Msg.w) or not iswindow(Msg.w) then
      win.r["canmove"]   = FALSE
      win.r["hasframe"]  = FALSE
      win.r["originrow"] = -10
      win.r["origincol"] = -80
      WINDOW CREATE FLOATING HEIGHT height.n WIDTH width.n
        ATTRIBUTES win.r
      TO Msg.w
    endif
    if Msg1.a <> "" then
      SETCANVAS Msg.w
      canvas off
        WINDOW SETATTRIBUTES Msg.w FROM win.r
        clear
        frame double from 0,1 to height.n-1,width.n-2
        width.n = width.n - 2                  ; Adjust for shadow & frame.
        len1.n = int((width.n-len(Msg1.a))/2)
        if len1.n < 1 then len1.n = 1 endif
        len2.n = int((width.n-len(Msg2.a))/2)
        if len2.n < 0 then len2.n = 0 endif
        if (Msg2.a = "") then
          if PressKey.l then len1.n = len1.n + 1 endif     ; Center is no dots.
          @ 1,len1.n ?? Msg1.a
        else
          @ 1,len1.n+1 ?? Msg1.a
          if PressKey.l then len2.n = len2.n + 1 endif     ; Center is no dots.
          @ 2,len2.n ?? Msg2.a
        endif
        if PressKey.l then
          len1.n = int((width.n-len("Press any key."))/2)+1
          if Msg2.a = ""
            then @2,len1.n
            else @3,len1.n
          endif
          ?? "Press any key."
        endif
        paintcanvas attribute style.n all
        if not PressKey.l then
          style attribute style.n + 128
            ?? "..."
          style
        endif
        cursor off
        win.r["floating"]  = TRUE
        win.r["originrow"] = 14
        win.r["origincol"] = int((80-width.n)/2)
        WINDOW SETATTRIBUTES Msg.w FROM win.r
      canvas on
      window select Msg.w
      if PressKey.l then
        dynarray Close.r[]
        Close.r["floating"] = FALSE
        beep
        KeyMsg4.n_FMS()                          ; Wait for key or mouse UP.
        WINDOW SETATTRIBUTES Msg.w FROM close.r
      endif
    endif
    if (current.w <> 0) then
      WINDOW SELECT current.w
    endif
  endif
endproc
writelib libname.a DlgMsg.u_FMS
release procs      DlgMsg.u_FMS

; Handle mouse and key events.
; Returns key pressed or 0 if mouse down.  If mouse DOWN, waits for mouse up.
; Called by KeyMsg.n, DlgMsg.u, DlgMsg.n
proc KeyMsg4.n_FMS()
  private proc.a, Event.r, char.n
  proc.a = "KeyMsg4.n_FMS"
  while charwaiting() retval = getchar() endwhile
  GETEVENT MOUSE "Down" KEY "All" TO Event.r
  if Event.r["TYPE"] = "KEY" then
    char.n = Event.r["Keycode"]
  else
    char.n = 0
    while TRUE
      GETEVENT MOUSE "Up" KEY "All" TO Event.r
      if Event.r["TYPE"] = "MOUSE"
        then quitloop
        else beep
      endif
    endwhile
  endif
  return char.n
endproc
writelib libname.a KeyMsg4.n_FMS
release procs      KeyMsg4.n_FMS

; View results from query.
; INPUT: Group.a        Group selected.
;        Output.t       Table with ad hoc query results.
;        Form1.a        First form to use.
;        Form2.a        Second form to use.
;        Report.a       Report to use.
; Called by QueryEngine4.u
proc ViewOutput4.u(Group.a, Output.t, Form1.a, Form2.a, Report.a)
  private win.r, form.w, form.a, key.v, multiform.l
  form.a = Form1.a
  view Output.t
    if form.a <> "" then
      pickform form.a
      multiForm.l = ismultiform(Output.t, form.a)
    endif
    DlgMsg.u_FMS("", "", FALSE)          ; Remove message box
    cursor normal
    DYNARRAY win.r[]
    win.r["hasFrame"]  = not isformview()
    win.r["title"]     = ""
    win.r["floating"]  = TRUE
    win.r["maximized"] = TRUE
    win.r["canmove"]   = FALSE
    win.r["canresize"] = FALSE
    win.r["canmaximize"] = FALSE
    if isformview()
      then window handle form to Form.w
      else window handle image 1 to Form.w
    endif
    window setattributes Form.w from win.r
    window select Form.w
    while TRUE
      wait table
        prompt "Press [Esc] to return, [F7] to toggle form, [Ctrl-P] to print report."
      until "Esc", "F3", "F4", "F7", 16
      key.v = retval
      switch
        case key.v = "Esc"   : quitloop
        case key.v = "F7"    :
          if form.a = Form1.a
            then form.a = Form2.a
            else form.a = Form1.a
          endif
          if form.a = "" then
            formkey                        ; Toggles table and form views.
          else
            pickform form.a                ; Switches forms.
            multiForm.l = ismultiform(Output.t, form.a)
          endif
          win.r["hasFrame"]  = not isformview()
          if isformview()
            then window handle form to Form.w
            else window handle image 1 to Form.w
          endif
          window setattributes Form.w from win.r
        case key.v = "F3" or key.v = "F4" :
          if multiForm.l
            then keypress key.v
            else beep
          endif
        case key.v = 16      :
          if Report.a <> "" then
            PrintReport.u(table(), Report.a)
          endif
      endswitch
    endwhile
  clearall
  echo normal
  echo off
endproc
writelib libname.a ViewOutput4.u
release procs      ViewOutput4.u
