; -----------------------------------------------------
;  JD_ENTRY.SC - defines procedure JulianDateEntry()
;
;  Converts Julian date input to Paradox dates.
;
;  When your data entry source material contains Julian
;  dates, this procedure allows you to input them
;  directly to a Paradox date field.  Typical use is to
;  call the procedure when a selected key is pressed
;  during a 'wait .. until' command.  For example, add
;  the numeric value 10 to the 'until' list of keys,
;  and add 'case retval = 10: JulianDateEntry()' to the
;  switch case structure - then when you press CtrlJ
;  (ASCII 10) the procedure checks if you are in a date
;  field, in an edit mode, gets a 5-digit Julian date
;  from you and makes sure the number of days is right,
;  and finally converts the value to the equivalent
;  Paradox date.  May be used interactively by writing
;  the procedure to a library on the autolib path, and
;  establishing a setkey - on CtrlJ, for example - to
;  call the procedure:  setkey 10 JulianDateEntry()
;
;
;  Also included is a separate short procedure to 
;  convert a Paradox date value to a 5 digit Julian
;  date (example 91001).
;
;
;  Written by Steve Caple,
;  Sacramento Paradox User's Group
;  Revised 12-14-91 wsc
; -----------------------------------------------------
proc JulianDateEntry()
  private R,        ; row location of canvas cursor
          C,        ; column location of canvas cursor
          L,        ; current field length
          Ans,      ; showmenu response variable
          JDate,    ; 5-digit Julian date from user
          JYear,    ; first 2 chars in JDate
          JDays,    ; numval of last 3 chars of JDate
          LeapYear, ; logical, true if leap year
          GoodDays, ; logical, true if good Jdays
          DateMsg,  ; string for date error message
          MsgAttr   ; attribute for error messages

  ; ---------------------------------------------------
  ; get lower right corner message attribute from
  ; current configuration
  ; ---------------------------------------------------
  MsgAttr = syscolor(3)

  ; ---------------------------------------------------
  ; make sure we're in an edit mode AND a date field
  ; - if not, beep & return
  ; ---------------------------------------------------
  if search(sysmode(),"CoEditDataEntry") < 1 then
    beep
    ; -----------------------------------------------
    ; Msg2() and Msg() are my utility messaging procs
    ; substitute one of your own, or hard code the
    ; messages as in '@ 2,0 ?? "You must be in ...'
    ; Msg procs were uploaded to SC_MSG.ZIP, Lib 6
    ; -----------------------------------------------
    Msg2("You must be in CoEdit mode to use this key.",
         2,true,MsgAttr,1200,1800,false)
    return false
  endif

  if (fieldtype() <> "D") then
    beep
    Msg2("You must be in Date field to use this key.",
         2,true,MsgAttr,1200,1800,false)
    return false
  endif

  ; ---------------------------------------------------
  ; if you are editing the table using a linked multi-
  ; table form and toggle to tableview, the image will
  ; be "link locked," forcing you to return to the form
  ; to make any changes - checking islinklocked() will
  ; determine at once that (1) you are in a multitable
  ; form and (2) you are in table view - in which case
  ; you need to return to the form view
  ; ---------------------------------------------------
  if islinklocked() then
    formkey
  endif

  ; ---------------------------------------------------
  ; synchronize the canvas cursor to the workplace
  ; cursor, and get the position of the FIRST character
  ; of the current field
  ; ---------------------------------------------------
  synccursor
  R = row()
  if len([]) = 7 then ; compensate for right justified
    L = 8             ; dates with single digit months
  else
    L = len([])
  endif
  C = col()-L


  ; ---------------------------------------------------
  ; MAIN LOOP - stay in this loop accepting dates until
  ; we get a good date or user chooses to Quit
  ; ---------------------------------------------------

  while true
    ; -------------------------------------------------
    ; put canvas cursor at the location of the current
    ; date field, and accept a Julian date string at
    ; that spot - we use A8 as the length to coincide
    ; with the length of the mm/dd/yy date field, but
    ; our picture clause keeps to 5 number characters
    ; - picture clause could be simpler, just "#####"
    ; -------------------------------------------------
    prompt "",""
    prompt
    style attribute 112
    @ 0,0 ?? spaces(80)
    @ 1,0 ?? spaces(80)
    Msg("Enter 5-digit Julian Date (YYDDD format) for " +
        field(),2,true,MsgAttr,0,false)
    style reverse
    @ R,C accept "A8" picture "{9,#}{0,#}{0,1,2,3}{##}"
            to JDate
    style
    if not retval then         ; user pressed Esc
      return false             ; so quit
    endif
    if isblank(JDate) then     ; user entered blank
      beep
      style reverse
      @ R,C ?? "BLANK   "      ; put warning over field
      style
      showmenu
        "Retry  ": "Try to enter a valid Julian Date",
        "Quit"   : "Abandon Julian Date entry"
      to Ans
      ; -----------------------------------------------
      ; treat Esc as a Quit key
      ; -----------------------------------------------
      if Ans = "Quit"  or  Ans = "Esc"  then
        return false
      else             ; user chose "Try again", so
        loop           ; go back and start over
      endif
    endif

    ; -------------------------------------------------
    ; if we get this far, we've got a 5 digit value 
    ; (yyddd) - check it out:
    ; > get yy portion into JYear
    ; > get numeric value for the 'ddd' portion
    ; -------------------------------------------------
    JYear = substr(JDate,1,2)
    JDays = numval(substr(JDate,3,3))

    ; -------------------------------------------------
    ; check for valid number of days, warn user
    ; (note: easy and accurate way to check leap year)
    ; -------------------------------------------------
    if day(dateval("3/1/"+JYear)-1) = 29 then
      LeapYear = true
    else
      LeapYear = false
    endif

    switch
      case JDays > 0 and JDays <= 365:
        GoodDays = true
      case JDays < 1:
        GoodDays = false
        DateMsg = "A Julian date cannot have zero days"
      case JDays > 365 and not LeapYear:
        GoodDays = false
        DateMsg = "Year cannot have more than 365 days"
      case JDays > 366:
        GoodDays = false
        DateMsg = "Year cannot have more than 366 days"
    endswitch

    ; -----------------------------------------------
    ; if number of days invalid, force retry or quit
    ; -----------------------------------------------
    if not GoodDays then
      beep
      style reverse, blink
      @ R,C ?? format("w8",JDate)
      style
      Msg(DateMsg,2,true,MsgAttr,0,false)

      showmenu
        "Retry  ": "Try to enter a valid Julian Date",
        "Quit"   : "Abandon Julian Date entry"
      to Ans

      ; -----------------------------------------------
      ; treat Esc as a Quit key
      ; -----------------------------------------------
      if Ans = "Quit"  or  Ans = "Esc"  then
        return false   ; leave procedure
      else             ; user chose "Try again", so
        loop           ; go back and start over
      endif
    endif

    ; -------------------------------------------------
    ; we can't get to this point without entering an
    ; acceptable date string - otherwise we would have
    ; failed the tests above and either have return-ed
    ; or loop-ed back to the accept statement - so we
    ; are OK and can leave the loop
    ; -------------------------------------------------

    quitloop       ; leave MAIN LOOP

  endwhile         ; end of MAIN LOOP

  ; ---------------------------------------------------
  ; set date to the Jan 1st of JYear, then add the
  ; Julian days number-1, and assign to current field
  ; ---------------------------------------------------
  [] = dateval("1/1/"+JYear) + JDays-1
  prompt

endproc  ; JulianDateEntry()





; -----------------------------------------------------
; simple proc to convert Paradox date to 5 digit Julian
; -----------------------------------------------------
proc JDate(DateIn)
  private t,yr,mo,dy,jdays
  t = match(DateIn,"../../..",mo,dy,yr)
  jdays = 1 + (DateIn - dateval("1/1/"+yr))
  jdays = strval(jdays)
  jdays = fill("0",3-len(jdays)) + jdays
  return yr + jdays
endproc










