;-------------------------------------------------------------------------------
;    PROCNAME: INSPECT.U
; DESCRIPTION: Paradox For DOS Object inspector. This utility uses commands
;	       available only in Paradox 4.5 for DOS, and will not execute
;	       under earlier versions of Paradox.
;	       INSPECT.U is the main driver for this utility, and uses
;	       a GETEVENT in an infinite loop to trap events. All events
;	       except for the right mouse click will be passed through the
;	       object inspector to Paradox.
;
;	       SPECIAL NOTE:
;	       This program is for use in interactive Paradox only. This
;	       program is not intended for use under Paradox runtime.
;
;     EXAMPLE: Inspect.u()
;  PARAMETERS: None
;     RETURNS: None
;SETS GLOBALS:
;USES GLOBALS: None
;  PROCEDURES: Various
;      AUTHOR: Harry Goldman
;	       DataBase Designs, Inc.
;	       Copyright 1993, All Rights Reserved
;     CREATED: 6/93 - 10/93
; ATTRIBUTION: none
; MODIFIED BY:
;    COMMENTS: Authorization is hearby granted for the reuse and or
;	  distribution of this application in part or in whole with
;	       proper attribution to the author.
;-------------------------------------------------------------------------------

Proc Inspect.u()
   private Rec.r,	; Dynamic array of the current event
      Pop.a,		; Menu choice made by user
      Error.n,		; Error trapping variable
      Line1.r,		; Dynamic array of first line prompts
      Line2.r,		; Dynamic array of second line prompts
      Formats.r,	; Dynamic array of inspector window sizes and locations
      CRanges.r,	; Dynamic array of color groupings
      Handles.r,  ; Dynamic array of window handles
      Win.r,      ; Dynamic array of window attributes
      Low.n,		; Low color attribute in a range
      High.n,		; High color attribute in a range
      Tbl.a,		; Current table
      Fld.a,		; Current field
      Row.n,		; Current workspace cursor row
      Col.n,		; Current workspace cursor column
      L 	  ; Logical variable

Loader.u()    ; Call a loading routine to populate the Dynamic arrays


; Begin an infinite loop to trap all events. If the inspector is not active
; pass the event through to Paradox for execution.
; This will allow almost everything that Paradox can do to still be done,
; sort of like a Paradox TSR.
; While this script executes the MESSAGE box will be used to tell the user
; that INSPECT is running.
; To exit this utility press [Ctrl] [Q]
; NOTE: The alternate menu [Alt][F10] is not available durring the use of
; this utility.

While True
   Message "INSPECT! " + Strval(Today()) + " " + Strval(Time()) +
      " [Ctrl][Q] To Quit "
   Echo Normal
   Error.n = 0
   GetEvent                                     ; trap all events
      Key "ALL"
      Mouse "ALL"
      Message "ALL"
      Idle
      To Rec.r
   Switch                                       ; If [Ctrl][Q] is pressed
      Case Rec.r["Type"] = "KEY" :              ; quit
         If Rec.r["KeyCode"] = 17 Then
            Quitloop
         Endif

; If the right mouse is clicked call a routine to determine the location
; and type of inspector menu to show.

      Case rec.r["Type"] = "MOUSE" :
         If Rec.r["Buttons"] = "RIGHT" And Rec.r["Action"] = "DOWN" Then
            GetClick.u()
            If ClickTYpe.u < 1 Then
               Loop
            Endif
            Row.n = Min(Rec.r["Row"],18)             ; Get the workspace
            Col.n = Min(Rec.r["Col"],60)             ; coordiantes for the
                                                     ; inspector menu
            Switch
               Case ClickType.u = 1 : SystemMenu.u() ; Display the inspector
               Case Clicktype.u = 2 : WindowMenu.u() ; menu.
               Case Clicktype.u = 3 : TableMenu.u()  ;
               Case Clicktype.u = 4 : FieldMenu.u()  ;
               Case Clicktype.u = 5 : FieldMenu.u()  ;
               Otherwise            : Quitloop
            Endswitch

; Selections from the inpector menu will go to the variable POP.A. If the
; user pressed [Esc] the menu is canceled and no other action is taken.
;
; If a menu selection is made, branch to that part of the utility.
;
; NOTE: In a popup menu, clicking with the mouse off of the menu is
; equivalent in Paradox to the pressing of the [Esc] key.

            Switch
               Case Pop.a = "Esc"  : Loop
               Case Pop.a = "Quit" : Quit
               Case Substr(Pop.a,1,5) = "Color" :

; If the user selected a color inspector, use a dynamic array to
; determine the range of color attributes that are needed on screen.

                  L = Match(Cranges.r[Pop.a],"..,..",Low.n,High.n)
                  ShowColors.u(Low.n,High.n,Pop.a)

; If the user wants to set the highlight color for either field or
; record, call the highlight portion of the utility.

               Case Substr(Pop.a,1,2) = "Hl" : DoHighlight.u()

; In all other cases, go into the body of the inspector

               Otherwise     : Insp.u(Pop.a)
            Endswitch
            Loop
         Endif
   EndSwitch
   ExecEvent Rec.r                       ; Execute the event that was trapped
   Release Vars Rec.r                    ; Release the event dynamic array
Endwhile                                 ; End of the infinite loop

Endproc             ; Inspect.u()
;-------------------------------------------------------------------------------
;    PROCNAME: Loader
; DESCRIPTION: Load dynamic arrays with values
;     EXAMPLE: Loader.u()
;  PARAMETERS: None
;     RETURNS: None
;SETS GLOBALS: All dynamic arrays
;USES GLOBALS: None
;  PROCEDURES: none
;      AUTHOR: Harry Goldman
;	       DataBase Designs, Inc.
;	       Copyright 1993, All Rights Reserved
;     CREATED:
; ATTRIBUTION: none
; MODIFIED BY:
;    COMMENTS: Authorization is hearby granted for the reuse and or
;	  distribution of this application in part or in whole with
;	       proper attribution to the author.
;-------------------------------------------------------------------------------

Proc Loader.u()

If IsRuntime() Then
   Quit "This Application Is Not To Be Used Under Runtime"
Endif

Dynarray Line1.r[]
Line1.r["Struct"] = "Table Structure Information - "
Line1.r["flds"]   = "Field List - "
Line1.r["forms"]  = "Form List - "
Line1.r["Idx"]    = "Index List - "
Line1.r["Rpt"]    = "Report List - "
Line1.r["Set"]    = "Setting Information - "
Line1.r["Setf"]   = "Field Settings - "
Line1.r["Vlchk"]  = "Valcheck List - "
Line1.r["Sysd"]   = "Drives Available - Double Click For Drive Space"
Line1.r["Sysc"]   = "System Configuration"
Line1.r["Syse"]   = "System Environment"
Line1.r["Sysp"]   = "Paradox Environment"
Line1.r["Win"]    = "Window Attributes: "

Dynarray Line2.r[]
Line2.r["Struct"] = "Item,Setting"
Line2.r["flds"]   = "Field Name,Field Type"
Line2.r["forms"]  = "Form,Form Name"
Line2.r["Idx"]    = "Index,Index Info"
Line2.r["Rpt"]    = "Report,Report Name"
Line2.r["Set"]    = "Setting Type,Setting"
Line2.r["Setf"]   = "Setting Type,Setting"
Line2.r["Vlchk"]  = "Valcheck Type,Valcheck Setting"
Line2.r["Sysd"]   = "Drive,Type"
Line2.r["Sysc"]   = "Item,Value"
Line2.r["Syse"]   = "Item,Value"
Line2.r["Sysp"]   = "Item,Value"
Line2.r["Win"]    = "Item,Value"

Dynarray Formats.r[]
Formats.r["Struct"] = "5,25,20"
Formats.r["flds"]   = "5,35,10"
Formats.r["forms"]  = "5,5,40"
Formats.r["Idx"]    = "5,22,23"
Formats.r["Rpt"]    = "5,5,40"
Formats.r["Set"]    = "5,25,20"
Formats.r["Setf"]   = "5,25,20"
Formats.r["Vlchk"]  = "5,15,30"
Formats.r["Sysd"]   = "5,20,25"
Formats.r["Sysc"]   = "5,25,20"
Formats.r["Syse"]   = "2,8,40"
Formats.r["Sysp"]   = "2,18,30"
Formats.r["Win"]    = "2,18,30"

Dynarray CRanges.r[]
CRanges.r["ColorsM"] = "1000,1006"
CRanges.r["ColorsW"] = "1007,1014"
CRanges.r["ColorsD"] = "1031,1062"
CRanges.r["ColorsS"] = "1063,1071"
CRanges.r["ColorsP"] = "1072,1077"
CRanges.r["ColorsR"] = "1000,1077"

Dynarray Changes.r[]
Changes.r["Privdir"]    = "Private Directory,A30,35,*!,SetPrivDir,T"
Changes.r["Autosave"]   = "Autosave Setting,S,6,#[#][#],Array,F"
Changes.r["PageLength"] = "Report Page Length,S,6,#[#][#],Array,F"
Changes.r["PageWidth"]  = "Report Page Width,S,6,#[#][#],Array,F"
Changes.r["Margin"]     = "Report Margin,S,6,#[#][#],Array,F"
Changes.r["Wait"]       = "Pause After Each Page: True / False,A6,8,TF,Array,F"
Changes.r["GroupRepeats"] = "Surpresses Repeating Values In Reports: True / False,A6,8,TF,Array,F"
Changes.r["Quote Char"] = "Enter A Character To Delimit Strings,A1,4,!,Array,F"
Changes.r["Separator"]  = "Enter A Field Seperator,A1,4,!,Array,F"
Changes.r["ASCII Decimal Point"]  = "Enter A Period Or Comma,A1,4,P,Array,F"
Changes.r["Report Decimal Point"] = "Enter A Period Or Comma,A1,4,P,Array,F"
Changes.r["BlankZero"]  = "Blank = Zero: True / False,A6,8,TF,Array,F"
Changes.r["Canclose"]   = "Allow Window Close Button: True / False,A6,8,TF,Array,F"
Changes.r["Canmaximize"]= "Allow Window Maximize: True / False,A6,8,TF,Array,F"
Changes.r["Canmove"]    = "Allow Window Move: True / False,A6,8,TF,Array,F"
Changes.r["Canresize"]  = "Allow Window Resize: True / False,A6,8,TF,Array,F"
Changes.r["Canvas"]     = "Allow Canvas Commands to Show: True / False,A6,8,TF,Array,F"
Changes.r["HasFrame"]   = "Window Frame: True / False,A6,8,TF,Array,F"
Changes.r["HasShadow"]  = "Window Shadow: True / False,A6,8,TF,Array,F"
Changes.r["Height"]     = "Window Height,S,8,*#,Array,F"
Changes.r["Maximized"]  = "Maximize Window: True / False,A6,8,TF,Array,F"
Changes.r["OriginRow"]  = "Enter New Origin Row,S,8,*#,Array,F"
Changes.r["OriginCol"]  = "Enter New Origin Column,S,8,*#,Array,F"
Changes.r["ScrollRow"]  = "Enter Scroll Row,S,8,*#,Array,F"
Changes.r["ScrollCol"]  = "Enter Scroll Column,S,8,*#,Array,F"
Changes.r["Style"]      = "Enter Window Color,S,8,*#,Array,F"
Changes.r["Title"]      = "Window Title,A40,44,*!,Array,F"
Changes.r["Width"]      = "Window Width,S,8,*#,Array,F"

Dynarray ColorSet.r[]

Colorset.r[0] ="Menu Background"
Colorset.r[1] ="Reserved"
Colorset.r[2] ="Current Menu Selection"
Colorset.r[3] ="Message and Debugger Line"
Colorset.r[4] ="Menu Selection Description"
Colorset.r[5] ="Mode Indicator"
Colorset.r[6] ="Non Current Display"
Colorset.r[7] ="Selected Area"
Colorset.r[8] ="Workspace Area"
Colorset.r[9] ="Current Table Border"
Colorset.r[10]="Reserved"
Colorset.r[11]="Reserved"
Colorset.r[12]="Help & System Forms Border"
Colorset.r[13]="Help & System Forms Text"
Colorset.r[14]="Highlighted Help Text"
Colorset.r[15]="Reverse Video Text"
Colorset.r[16]="PAL Debugger Line"
Colorset.r[17]="Field Values"
Colorset.r[18]="Negative Values"
Colorset.r[19]="Examples & Checkmarks"
Colorset.r[20]="Reserved"
Colorset.r[21]="Help Index Text"
Colorset.r[22]="Help Index Text"
Colorset.r[23]="Help Index Text"
Colorset.r[24]="Vertical Ruler"
Colorset.r[25]="Report Band Lines"
Colorset.r[26]="Reserved"
Colorset.r[27]="Reverse Video & High Intensity"
Colorset.r[28]="Reverse Video Text"
Colorset.r[29]="High Intensity"
Colorset.r[30]="Reserved"
Colorset.r[31]="Reserved"
Colorset.r[99]="Highlight"
Colorset.r[1000]="Desktop Area"
Colorset.r[1001]="Normal Menu Command"
Colorset.r[1002]="Disabled Menu Command"
Colorset.r[1003]="Hot Key For Menu Command"
Colorset.r[1004]="Highlighted Menu Command"
Colorset.r[1005]="Highlighted Disabled Menu Command"
Colorset.r[1006]="Hot Key - Highlighted Menu Command"
Colorset.r[1007]="Inactive Window Frame"
Colorset.r[1008]="Active Window Frame"
Colorset.r[1009]="Frame Icons"
Colorset.r[1010]="Scroll Bar"
Colorset.r[1011]="Scroll Bar Controls"
Colorset.r[1012]="Reserved"
Colorset.r[1013]="Reserved"
Colorset.r[1014]="Reserved"
Colorset.r[1015]="Reserved"
Colorset.r[1016]="Reserved"
Colorset.r[1017]="Reserved"
Colorset.r[1018]="Reserved"
Colorset.r[1019]="Reserved"
Colorset.r[1020]="Reserved"
Colorset.r[1021]="Reserved"
Colorset.r[1022]="Reserved"
Colorset.r[1023]="Reserved"
Colorset.r[1024]="Reserved"
Colorset.r[1025]="Reserved"
Colorset.r[1026]="Reserved"
Colorset.r[1027]="Reserved"
Colorset.r[1028]="Reserved"
Colorset.r[1029]="Reserved"
Colorset.r[1030]="Reserved"
Colorset.r[1031]="Inactive Dialog Box Frame"
Colorset.r[1032]="Active Dialog Box Frame"
Colorset.r[1033]="Selected Dialog Box Frame"
Colorset.r[1034]="Scroll Bar Page Area"
Colorset.r[1035]="Scroll Bar Controls"
Colorset.r[1036]="Default Background Text"
Colorset.r[1037]="Label - Linked Control Inactive"
Colorset.r[1038]="Label - Linked Control Active"
Colorset.r[1039]="Label Hot Key"
Colorset.r[1040]="Text - Normal Push Button"
Colorset.r[1041]="Text - Default Push Button"
Colorset.r[1042]="Text - Selected Push Button"
Colorset.r[1043]="Reserved"
Colorset.r[1044]="Hot Key - Push Buttton Label"
Colorset.r[1045]="Button Shadow"
Colorset.r[1046]="Normal Radio Button / Check Box"
Colorset.r[1047]="Highlighted Radio Button / Check Box"
Colorset.r[1048]="Hot Key - Radio Button / Check Box"
Colorset.r[1049]="Normal Type-In Box"
Colorset.r[1050]="Selected Type-In Box"
Colorset.r[1051]="Type-In Box Arrows"
Colorset.r[1052]="Reserved"
Colorset.r[1053]="Reserved"
Colorset.r[1054]="Reserved"
Colorset.r[1055]="Reserved"
Colorset.r[1056]="Normal Pick List Item"
Colorset.r[1057]="Highlighted Pick List Item"
Colorset.r[1058]="Selected Pick List Item"
Colorset.r[1059]="Column Dividers"
Colorset.r[1060]="Reserved"
Colorset.r[1061]="Reserved"
Colorset.r[1062]="Reserved"
Colorset.r[1063]="Speed Bar Text"
Colorset.r[1064]="Reserved"
Colorset.r[1065]="Speed Bar Hot Key"
Colorset.r[1066]="Speed Bar Selected Item"
Colorset.r[1067]="Reserved"
Colorset.r[1068]="Speed Bar Selected Hot Key"
Colorset.r[1069]="Speed Bar Vertical Separator Line"
Colorset.r[1070]="SQL Server Title"
Colorset.r[1071]="Menu Prompts & Status"
Colorset.r[1072]="PAW Menu Command"
Colorset.r[1073]="Reserved"
Colorset.r[1074]="PAW Hot Key"
Colorset.r[1075]="PAW Highlighted Menu Command"
Colorset.r[1076]="Reserved"
Colorset.r[1077]="PAW Hot Key - Highlighted  Menu Command"

Dynarray Win.r[]
Win.r["CanvasHeight"] = 18            ; Define the dynamic array of window
Win.r["CanvasWidth"]  = 52            ; attributes
Win.r["Style"]        = 31            ;
Win.r["CanResize"]    = False         ;
Win.r["CanMaximize"]  = False         ;
Win.r["CanMove"]      = True          ;
Win.r["Title"]        = "Inspector"   ;

Dynarray Handles.r[]

Endproc 			      ; Loader.u
;-------------------------------------------------------------------------------
;    PROCNAME: INSP.U
; DESCRIPTION: Main analyzer routine for INSPECT
;     EXAMPLE: Insp.u("Sys")
;  PARAMETERS: Pop.a - Menu selection made by the user
;     RETURNS: None
;SETS GLOBALS: Tbl.a   - Current table
;	       Fld.a   - Current field
;	       Error.n - Current error code
;USES GLOBALS:
;  PROCEDURES:
;      AUTHOR: Harry Goldman
;	       DataBase Designs, Inc.
;	       Copyright 1993, All Rights Reserved
;     CREATED: 6/93 - 10/93
; ATTRIBUTION: None
; MODIFIED BY: None
;    COMMENTS: Authorization is hearby granted for the reuse and or
;	  distribution of this application in part or in whole with
;	       proper attribution to the author.
;-------------------------------------------------------------------------------
Proc Insp.u(Pop.a)
   private Pop.a,	; Menu selection made by the user
      S.n,              ; Left margin
      H1.a, H2.a,       ; Colum headings
      W1.n, W2.n,       ; Colum Widths
      Elmnt,				; Dynamic array element
      Fld.r,				; Dynamic array of INSPECT results
      Count.n,          ; Display size of results on screen
      Order.r,          ; Array that will display results on screen
      Start.n, End.n,   ; Starting and ending point of items to display
      Temp.r,           ; Temporary Dynamic array with full table name
      L,                ; Dynarray Element in ForEach Loop
      F1.a, F2.a, F3.a, ; Formats for on screen display
      Win.h             ; Window handle

If Substr(Pop.a,1,3) <> "Sys" And         ; If this is not a system inspector
   Substr(Pop.a,1,3) <> "Win" Then        ; or a canvas inspector, grab the
   If Tbl.a = "" Then                     ; current table name and field
      Tbl.a = Table()
      Fld.a = Field()
      If Error.n = 20 Then          ; If there is no table on the workspace
         Return                     ; or it isn't current an ErrorProc will
      Endif                         ; trigger and set the Error trapper to 20
   Endif
Endif

; If the user wants to see the structure of the table get it to Fld.r, then
; parse the table name so that we can getthe family and optimal sizes for
; this table.
;
; Otherwise, based on the value of POP.A, execute the functions needed to
; run the INSPECTOR.

Switch
   Case Pop.a = "Struct": TableInfo Tbl.a To Fld.r
      ParseFileName FullFilename(Tbl.a) To Temp.r
      ForEach L in Temp.r
         Fld.r[L] = Temp.r[L]
      EndforEach
      Fld.r["Family Size"] = Format("W12,EC",FamilySize(Tbl.a))
      Fld.r["Optimal Size"] = Format("W12,EC",Fld.r["Blocks"] * Fld.r["BlockSize"] * 1024)
   Case Pop.a = "Flds"  : TableInfo Tbl.a Fields To Dynarray Fld.r
   Case Pop.a = "Forms" : TableInfo Tbl.a Forms  To Fld.r
   Case Pop.a = "Idx"   : TableInfo Tbl.a Indexes To Fld.r
   Case Pop.a = "Rpt"   : TableInfo Tbl.a Reports To Fld.r
   Case Pop.a = "Setf"  : TableInfo Tbl.a Settings Fld.a To Fld.r
   Case Pop.a = "Set"   : TableInfo Tbl.a Settings To Fld.r
   Case Pop.a = "Vlchk" : TableInfo Tbl.a Valchecks Fld.a To Fld.r
   Case Pop.a = "Sysc"  : Sysinfo Config To Fld.r
   Case Pop.a = "Sysd"  : Sysinfo Drives To Fld.r
   Case Pop.a = "Syse"  : Sysinfo Environment To Fld.r
   Case Pop.a = "Sysp"  : Sysinfo To Fld.r
   Case Pop.a = "Win"   : If Not WinAtt.u() Then Return False Endif
Endswitch

If DynarraySize(Fld.r) < 1 Then
   Beep
   Message "No Information To Report"
   Sleep 1000
   Return
Endif

BuildWindow.u(Min(DynarraySize(Fld.r),13))
                                                ; Get the window column
                                                ; formats, and the column
L = Match(Formats.r[Pop.a],"..,..,..",S.n,W1.n,W2.n) ; headings
L = Match(Line2.r[Pop.a],"..,..",H1.a,H2.a)

F1.a = "W"+W1.n+",AL"                           ; Set up the format specs
F2.a = "W"+W2.n+",AL"                           ;
F3.a = "W"+W2.n+",EC,AL"                        ;

Array Order.r[DynarraySize(Fld.r)]              ; Dynamic array to store
                                                ; items in a known order
Count.n = IIF(Pop.a="Flds",1,DynarraySize(Fld.r))
ForEach Elmnt in Fld.r                          ; Get the items in order
   Order.r[Count.n] = Elmnt                     ;
   Count.n = IIF(Pop.a="Flds",Count.n+1,Count.n-1)
EndForEach                                      ;

Count.n = DynarraySize(Fld.r)                   ; how many items are there ?
Start.n = 1                                     ; Allow only 15 at a time to
End.n = Min(14,DynarraySize(Fld.r))             ; be displayed

While True
   Canvas Off
   Switch                                       ; Special handling for:
      Case Pop.a = "Vlchk" :                    ;      Valchecks
         @ 0,0 ?? Format("W50,AC",Line1.r[Pop.a] + Fld.a)
      Case SubStr(Pop.a,1,3) = "Sys" :          ;      System
         @ 0,0 ?? Format("W50,AC",Line1.r[Pop.a])
      Case Pop.a = "Setf" :                     ;      Field settings
         @ 0,0 ?? Format("W50,AC",Line1.r[Pop.a] + Fld.a)
      Otherwise :                               ;      All others
         @ 0,0 ?? Format("W50,AC",Line1.r[Pop.a] + Tbl.a)
   EndSwitch

   S.n = Numval(S.n)                            ; Start at the top
   @ 2,S.n ?? H1.a
   @ 2,Numval(S.n) + Numval(W1.n) ?? H2.a
                                                ; Color the canvas
   Paintcanvas Attribute 31 0,0, Min(Count.n,13)+4,51
   Style Attribute 31                           ; Set the text color

   For I from Max(1,Start.n) To Min(End.n,Count.n) ; Display on the canvas
      ? Spaces(S.n), Format(F1.a,Order.r[I])
      If Numval(Fld.r[Order.r[I]]) <> "Error" Then
         ?? Format(F3.a,Fld.r[Order.r[I]])
      Else
         ?? Format(F2.a,Fld.r[Order.r[I]])
      Endif
   EndFor

   Prompt Spaces(80)                            ; Clear the prompt line
   Canvas on
   If Not ShowEngine.u(False,Count.n+1,1) Then	; Call the event handler
      Quitloop
   Endif

   Canvas Off
   Clear

EndWhile

ClearSpeedbar                                   ; Clear the speedbar
Prompt                                          ; Clear the prompt
If IsWindow(Handles.r["Display"]) Then          ; If the display window is
   Window Select Handles.r["Display"]           ; still on the workspace,
   Window Close                                 ; close it.
Endif
Echo Normal                                     ; reset the echo layer
Echo Off
Style
Endproc 					; Insp.u
;-------------------------------------------------------------------------------
;    PROCNAME: BUILDWINDOW.U
; DESCRIPTION: A generic routine to build a floating window on screen
;     EXAMPLE: BuidlWindow.u(10)
;  PARAMETERS: H.n - Height of the window to build
;     RETURNS: None
;SETS GLOBALS: Handles.r["Display"] - Window handle of the display window
;USES GLOBALS: None
;  PROCEDURES: None
;      AUTHOR: Harry Goldman
;              DataBase Designs, Inc.
;              Copyright 1993, All Rights Reserved
;     CREATED: 6/93 - 10/93
; ATTRIBUTION: None
; MODIFIED BY: None
;    COMMENTS: Authorization is hearby granted to the to reuse and or
;              distribute this application in part or in whole with
;              proper attribution to the author.
;-------------------------------------------------------------------------------
Proc BuildWindow.u(H.n)
   private H.n                           ; Window height

Win.r["CanvasHeight"] = H.n + 5          ; Define the dynamic array of window
                                         ;
Window Create Floating @ 2,15            ; Define the floating canvas
   Width 55 Height H.n + 7               ;
   Attributes Win.r                      ;
   To Handles.r["Display"]               ;

Endproc

;-------------------------------------------------------------------------------
;    PROCNAME: SHOWCOLORS.U
; DESCRIPTION: Display color choices
;     EXAMPLE:
;  PARAMETERS: Low.n  - Starting Color element
;	       High.n - Ending Color Element
;	       Line.a - Title Line
;     RETURNS: None
;SETS GLOBALS: None
;USES GLOBALS: Handles.r["Display"]
;  PROCEDURES: ShowEngine.u()
;      AUTHOR: Harry Goldman
;	       DataBase Designs, Inc.
;	       Copyright 1993, All Rights Reserved
;     CREATED:
; ATTRIBUTION: none
; MODIFIED BY:
;    COMMENTS: Authorization is hearby granted for the reuse and or
;	  distribution of this application in part or in whole with
;	       proper attribution to the author.
;-------------------------------------------------------------------------------
Proc ShowColors.u(Low.n,High.n,Line.a)
   private I, L,		       ; Counters
      Low.n,			       ; Starting Color Element
      High.n,			       ; Ending Color Element
      C.r,                  ; Dynarray of current colors
      Line.a,			       ; Title Line
      L.n,                  ; Count the line to display
      Start.n,			       ; Screen Coordinates
      End.n                 ; Screen Cooridnates

If Pop.a = "ColorsR" Then                       ; Reset Colors
   SetColors Default                            ;
   Return True                                  ;
Endif

GetColors To C.r                                ; Grab the current colors
Low.n  = Numval(Low.n)                          ; Set the screen coordinates
High.n = Numval(High.n)                         ; for the color window
						;
BuildWindow.u(Min(High.n-Low.n+2,13))
                                                ;
Start.n = 0                                     ;
End.n = Min(14,High.n - Low.n)                  ;

While True
   PaintCanvas Attribute 31 0,0, Min(High.n-Low.n+6,17),51
   @ 0,0 ?? Format("W55,AC","Color Settings")
   Setmargin 2

   L.n = 1                                      ;
   For I from Low.n + Start.n To Low.n + End.n	; Loop through the color
      If Isassigned(C.r[I]) Then                ; elements and find those
         L.n = L.n + 1                            ; that are in use in the
         ? Format("W4",Strval(I)) + " - " +       ; colors dynamic array.
         Format("W33,AR",Colorset.r[I]) + " - " + ; Those that are defined
         Format("W4,Ar",C.r[I])                   ; get displayed on screen
         Paintcanvas Attribute C.r[I] L.n,2,L.n,49;
      Endif                                     ;
   Endfor                                       ;
                                                ;
   If Not ShowEngine.u(True,High.n,Low.n) Then	; Call the 'engine' to
      Quitloop                                  ; control display of the
   Endif                                        ;   window
                                                ;
   Clear                                        ;
Endwhile                                        ;
Prompt                                          ; Cancel the prompt
If IsWindow(Handles.r["Display"]) Then          ; Cancel the display window
   Window Select Handles.r["Display"]           ;
   Window Close                                 ;
Endif                                           ;
ClearSpeedbar                                   ; Cancel the speedbar

EndProc
;-------------------------------------------------------------------------------
;    PROCNAME: ChangeColor.u
; DESCRIPTION: Allow the user to select new color choices
;     EXAMPLE: ChangeColor.u(Exp.a,C.n,C.r)
;  PARAMETERS: Exp.a - Window title message
;	       C.n   - Dynarray of current colors
;	       C.r   - Color element to change
;     RETURNS: None
;SETS GLOBALS: C.r   - Current color settings
;USES GLOBALS: None
;  PROCEDURES: None
;      AUTHOR: Harry Goldman
;	       DataBase Designs, Inc.
;	       Copyright 1993, All Rights Reserved
;     CREATED:
; ATTRIBUTION: none
; MODIFIED BY:
;    COMMENTS: Authorization is hearby granted for the reuse and or
;	  distribution of this application in part or in whole with
;	       proper attribution to the author.
;-------------------------------------------------------------------------------
Proc ChangeColor.u(Exp.a,C.n,C.r)
   private Win.r,				 ; Window Attributes
      I, J,                 ; Counters
      Exp.a,					 ; Window title message
      CCol.n,					 ; Column Pointer
      CRow.n,					 ; Row Pointer
      Select.n, 				 ; Selected'
      C.r,                  ; Dynarray of current colors
      C.n                   ; Color element to change

Dynarray Win.r[]                                ; Define window attributes
   Win.r["CanvasHeight"] = 8                    ; for the color selection
   Win.r["CanvasWidth"]  = 16                   ; window
   Win.r["CanResize"]    = False                ;
   Win.r["CanMaximize"]  = False                ;
   Win.r["CanClose"]     = False                ;
   Win.r["Title"]        = "Colors"             ;
		   ;
Window Create Floating @ -200,-200              ; Create a window off
   Width 18 Height 10                           ; screen
   Attributes Win.r                             ;
   To Handles.r["Colors"]                       ;
                                                ;
@ 0,0                                           ;
                                                ;
For I from 0 to 7                               ; Fill the window with
   For J from  0 to 15                          ; colored ASCII boxes
      Paintcanvas Fill "" Attribute I*16 + J I,J,I,J  ;
   Endfor                                       ;
Endfor                                          ;

Prompt Format("W80,AC",Exp.a + " Color Selection") ; Set the prompt line

Window Move Handles.r["Colors"] To 2,15         ; Place the window in the
                                                ; viewable area.
CCol.n = 0                                      ; Column Pointer
CRow.n = 0                                      ; Row Pointer
Select.n = -1                                   ;
                                                ;
While Select.n < 0                              ;
                                                ;
   @ CRow.n, CCol.n                             ; Position the cursor
   Cursor box                                   ; Change the cursor to a box
   Window Echo Handles.r["Colors"] True         ; Make sure that echo is on
   Style Blink                                  ; Make the cursor blink
   Getevent Key 27, -77, -72, -75, -80, 13      ; Get key and mouse events
       Mouse "DOWN"                             ;
       To E.r                                   ;
                                                ;
   Switch                                       ;
      Case E.r["Type"] = "KEY" :                ; Keystrokes

; If the user presses a cursor key or [Pgup]/[Pgdn] we want to move the cursor
; to a new location on the color selection window. To do this we need to
; track where the cursor currently is, and what key was pressed. We also
; have to allow for wrapping the cursor horizontally and vertically.

      Switch
         Case E.r["Keycode"] = 27 : Quitloop
         Case E.r["Keycode"] = 13 : Select.n = CRow.n*16 + CCol.n
         Case E.r["Keycode"] = -77 : CCol.n = IIF(CCol.n = 15,0,CCol.n + 1)
         Case E.r["Keycode"] = -75 : CCol.n = IIF(CCol.n = 0,15,CCol.n -1)
         Case E.r["Keycode"] = -72 : CRow.n = IIF(CRow.n = 0,7,CRow.n - 1)
         Case E.r["Keycode"] = -80 : CRow.n = IIF(CRow.n = 7,0,CRow.n + 1)
      EndSwitch
      Case E.r["Type"] = "MOUSE" :              ; Mouse events

; With mouse events we need to first make sure that the mouse event happened
; on the color selection window. If it didn't, just beep. If the mouse event
; was on the color selection window, we need to localize the event to
; determine where it happened, and then, what event occured. If the event was
; a double click we can determine the color attribute that was selected by
; looking at the row and column of the double click.

      Switch
         Case WindowAt(E.r["Row"],E.r["Col"]) <> Handles.r["Colors"] : Beep
         Case E.r["DoubleClick"] :
            LocalizeEvent E.r
            Select.n = (E.r["Row"]-1)*16+E.r["Col"]-1
         Otherwise :

; If the mouse event was simply a mouse DOWN, we want to reposistion the
; cursor at the new location.

            If E.r["Action"] = "DOWN" Then
               LocalizeEvent E.r
               If E.r["Col"] >= 1 And E.r["Col"] <= 16 Then
                  CCol.n = E.r["Col"]-1
               Endif
               If E.r["Row"] >= 1 And E.r["Row"] <= 8 Then
                  CRow.n = E.r["Row"]-1
               Endif
            Endif
      Endswitch
   Endswitch                                    ;
Endwhile                                        ;
                                                ;
Cursor Normal                                   ; Reset the cursor
Window Close                                    ; Close the color selections
                                                ;
If Isassigned(Inum.n) Then                      ; Either return the color
   Return Select.n                              ; attribute selected, or
Else                                            ;
   If Select.n >= 0 Then                        ; add it to the colors
      C.r[C.n] = Select.n                       ; Dynarray and set the new
      SetColors From C.r                        ; colors
   Endif                                        ;
Endif                                           ;
Return                                          ;
EndProc
;-------------------------------------------------------------------------------
;    PROCNAME: ShowEngine.u
; DESCRIPTION: Main processing engine for INSPECT display windows;
;     EXAMPLE: ShowEngine.u(
;  PARAMETERS: Mouse.l - Allow for mouse interaction (Logical True/False)
;	       High.n  - Highest parameter to display
;	       Low.n   - Lowest parameter to display
;     RETURNS: True  -
;	       False -
;SETS GLOBALS: None
;USES GLOBALS: None
;  PROCEDURES: ChangeColor.u()	- Allow for colors to be changed
;	       PromptEngine.u() - Prompt the user for new values
;      AUTHOR: Harry Goldman
;	       DataBase Designs, Inc.
;	       Copyright 1993, All Rights Reserved
;     CREATED:
; ATTRIBUTION: none
; MODIFIED BY:
;    COMMENTS: Authorization is hearby granted for the reuse and or
;	  distribution of this application in part or in whole with
;	       proper attribution to the author.
;-------------------------------------------------------------------------------
Proc ShowEngine.u(Mouse.l,High.n,Low.n)
   private Quit.l,				 ; Quit procedure indicator
      Exit.l,		       ; Return value
      High.n,		       ; High parameter to view
      Low.n,		       ; Low parameter to view
      Pos.n,		       ; Current window position
      ERec.r,		       ; GetEvnt Dynamic array
      Mouse.l,		       ; Call color routines
      Set.n              ; Current curcor position

Quit.l = False		       ;
Exit.l = True		       ;

If High.n - Low.n + 2 > 13 Then                 ; Set the sppedbar based on
   Speedbar                                     ; the number of parameters
      "~[Pgup]~ Previous"    : -73,             ; to display. If there are
      "~[PgDn]~ More"        : -81,             ; more then 15 parameters,
      "~[Esc]~ Quit"         : 27,              ; allow more pages
      "~Doubleclick~ select" : 32
Else
   Speedbar
      "~[Esc]~ Quit"         : 27,
      "~Doubleclick~ select" : 32
Endif

If Pop.a = "Sysd" Then                          ; Change the mouse parameter
   Mouse.l = true                               ; if we are on the drive
Endif                                           ; display.

Win.r["ScrollCol"]=0
Win.r["ScrollRow"]=0
Window SetAttributes Handles.r["Display"] From Win.r

@3,0                                            ;
While Not Quit.l                                ; Stay in a loop until the
   GetEvent                                     ; quit indicator is TRUE.
      Key -81, -73, 27, 32                      ; Trap [PgDn], [PgUp],
      Mouse "Down"                              ; [Esc] and [Spacebar]. Also
      Message "Close"                           ; trap closing the window
      To ERec.r                                 ; and mouse clicks
                                                ;
   Switch
      Case Erec.r["Type"] = "KEY" :             ; A keystroke was pressed
      Switch
         Case Erec.r["Keycode"] = 27 :          ; User pressed [Esc]
            Quit.l = True                       ; Drop out of the loop
            Exit.l = False                      ; Return a FALSE

         Case Erec.r["Keycode"] = 32 :          ; User pressed [Spacebar]
            Beep                                ; nothing to do.
            Message "Please DoubleClick With The Mouse On The Active Window"

         Case Erec.r["Keycode"] = -73 :         ; User pressed [PgUp]
            Start.n = Max(0,Start.n-15)         ; Show the previous 15 items
            End.n   = Min(Start.n+14,High.n - Low.n)
            Quit.l  = True                      ; Drop out of the loop

         Case Erec.r["KeyCode"] = -81 :         ; User pressed [PgDn]
            Start.n = Min(Start.n+15,Max(High.n-Low.n-13,0)) ; Show 15 items
            End.n   = Min(Start.n+14,High.n - Low.n)
            Quit.l = True                       ; Drop out of the loop
      EndSwitch

      Case Erec.r["Type"] = "MESSAGE" :         ; User CLOSED the window
         ExecEvent Erec.r                       ; Allow the event
         Quit.l = True                          ; Drop out of the loop
         Exit.l = False                         ; Return a FALSE

      Case Erec.r["Type"] = "MOUSE" :           ; Use used the mouse
         If Erec.r["DoubleClick"] Then          ; If the user doubleclicked
                                                ; but not on the diaply
                                                ; window, just beep
            If WindowAt(Erec.r["Row"],Erec.r["Col"]) <>
               Handles.r["Display"] Then
               Beep
            Else                                ; Otherwise figure out where
               LocalizeEvent Erec.r             ; the click was in the window

               Set.n = Start.n + Low.n + (Erec.r["Row"]-3)
                                                ; If the click was not on a
                                                ; valid item, beep
               If Set.n < Low.n Or Set.n > High.n+1 Then
                  Beep
               Else                             ; Otherwise, get the actual
                                                ; parameter that was clicked
                                                ; on.
                  @ Erec.r["Row"]-1,1
                  If (Set.n-Max(1,Start.n)-1) >= 1 And
                     (Set.n-Max(1,Start.n)-1) <= High.n Then
                        Pos.n =Start.n + Low.n + Erec.r["Row"]-5
                  Endif
; If the current display shows the drives available, and the user double
; clicked, show the available disk space.

                  If Pop.a = "Sysd"  Then
                     Message "Space Available On "+Order.r[Set.n-2]+" Drive = " +
                     Format ("W12,EC",Drivespace(Substr(Order.r[Set.n-2],1,1))) +
                     " Bytes"
                  Else

; Otherwise, if the mouse if being allowed, that means that some of the
; color parameters can be changed. Call the change routine. When the program
; returns from the change routine, make sure that the correct window and
; canavs are selected.

                     If Mouse.l Then
                        ChangeColor.u(Colorset.r[Set.n],Set.n,C.r)
                        Window Select Handles.r["Display"]
                        SetCanvas Handles.r["Display"]
                        Quit.l = True
                     Else

; If the mouse is not being allowed, then the color routine isn't relevant.
; In that case, check the dynamic array CHANGES.R[] to see if the current
; parameter is changeable. If it is, call the prompt routine that asks for
; a new value.

                        If Isassigned(Changes.r[Order.r[Pos.n]]) Then
                           PromptEngine.u(Changes.r[Order.r[Pos.n]],
                                 Fld.r[Order.r[Pos.n]])
                        Endif
                     Endif
                  Endif
               Endif
            Endif
         Else
            ExecEvent Erec.r
         Endif
   Endswitch
Endwhile

Return Exit.l
Endproc 				   ; ShowEngine.u()
;-------------------------------------------------------------------------------
;    PROCNAME: InspError.u
; DESCRIPTION: INSPECT internal error handler;
;     EXAMPLE: Autocall by PDOX
;  PARAMETERS: None
;     RETURNS: 1 - Continue with the next statement
;SETS GLOBALS: Error.n - GLobal error value
;USES GLOBALS: None
;  PROCEDURES: None
;      AUTHOR: Harry Goldman
;	       DataBase Designs, Inc.
;	       Copyright 1993, All Rights Reserved
;     CREATED:
; ATTRIBUTION: none
; MODIFIED BY:
;    COMMENTS: Authorization is hearby granted for the reuse and or
;	  distribution of this application in part or in whole with
;	       proper attribution to the author.
;-------------------------------------------------------------------------------
Proc InspError.u()
   private Err.r                                ; Dynamic array for error
                                                ; data
ErrorInfo To Err.r

Switch
   Case Err.r["Code"] = 20 :
      Beep
      Message "No Table On Workspace"
      Sleep 1000
      Error.n = 20
      Return 1
   Otherwise :
      Error.n = Err.r["Code"]
      Beep Sleep 100 Beep
      Message Err.r["Message"]
      Sleep 1000
      Return 1
Endswitch

Endproc 					; InspError.u()
;-------------------------------------------------------------------------------
;    PROCNAME: PromptEngine.u()
; DESCRIPTION: Prompt the user for a new value ;
;     EXAMPLE: PromptEngine.u(Parameter,Value)
;  PARAMETERS: Item.a  - Parameter information to modify (Long string)
;	       Val.a   - Current value of the parameter
;     RETURNS: None
;SETS GLOBALS: Fld.r  - Window attributes of the current workspace window
;USES GLOBALS: Win.h  - Window handle of the current workspace window
;	       Fld.r  - Window attributes of the current workspace window
;  PROCEDURES: None
;      AUTHOR: Harry Goldman
;	       DataBase Designs, Inc.
;	       Copyright 1993, All Rights Reserved
;     CREATED:
; ATTRIBUTION: none
; MODIFIED BY:
;    COMMENTS: Authorization is hearby granted for the reuse and or
;	  distribution of this application in part or in whole with
;	       proper attribution to the author.
;
;	       The varaible Item.a is a string that contains multiple
;	       comma seperated items. Before we can start this procedure
;	       we need to parse the values out if Item.a. The values passed
;	       in Item.s are stored in the Dynamic array Changes.r[] :
;
; Changes.r["HasShadow"]  = "Window Shadow: True / False,A6,8,TF,Array,F"
;
;	       Using this method allows for fast and easy passage of multiple
;	       values to a procedure in a generic fashion.
;-------------------------------------------------------------------------------
Proc PromptEngine.u(Item.a,Val.a)
   private Item.a,       ; Multi item parameter
	   Msg.a,				 ; Message to the user
	   Fmt.a,				 ; Format to accept data in
	   Width.n,				 ; Width of the prompt
	   Cmd.a,				 ; Command to execute
      Q.l,               ; True / False, execute
	   CPos.n,				 ; Middle of the dialog box
	   PPos.n,				 ; Prompt offset
	   Pic.a,				 ; Prompt picture
	   New.u,				 ; New value
	   Val.a,				 ; Current value
	   DFmt.a,				 ; Dialog Format
	   DWidth.n				 ; Dialog Width

                                                ; Start by parsing the
                                                ; Item.a varaible


If Not Match(Item.a,"..,..,..,..,..,..",Msg.a,Fmt.a,Width.n,Pic.a,Cmd.a,Q.l) Then
   Return
Endif

Q.l = (Q.l = "T")                               ;
DWidth.n = Max(Len(Msg.a)+5,Numval(Width.n)+5)	; Set the dialog width
DWidth.n = Max(35,DWidth.n)                     ;
DFmt.a = "W" + Strval(Dwidth.n) + ",AC"         ; Set the prompt format
PPos.n = Int(Dwidth.n / 4)                      ; Set the prompt offset

Switch                                          ; Set the Picture Clause
   Case Pic.a = "TF" : Pic.a = "True,False"     ; for special cases
   Case Pic.a = "P"  : Pic.a = ".,;,"           ;
Endswitch                                       ;

CPos.n = Int((80-Numval(Width.n)) / 2)          ; middle of the dialog
New.u = Val.a                                   ; Set the new value

ShowDialog "INSPECT!"                           ; Build a dialog box
   @ 15, CPos.n                                 ;
   Height 6 Width DWidth.n                      ;

   @ 0,0 ?? Format(DFmt.a,Msg.a)                ; Message to the user

   Accept @ 1,Int(Dwidth.n/2) - Int(Numval(Width.n)/2)
   Width Numval(Width.n) Fmt.a
   Picture Pic.a
   Tag "ACC"
   To New.u

   PushButton @ 2,PPos.n  -4
   Width 9 "  OK  " OK Default Value "OK" Tag "OKTAG" To Button.b

   PushButton @ 2,PPos.n *3 -4
   Width 9 "CANCEL" CANCEL Value "CAN" Tag "CANTAG" To Button.b
EndDialog

If Retval Then                                  ; If the DB was accepted
   Echo Off                                     ; then check the value that
   If Isassigned(New.u) Then                    ; was entered.

; If the user entry ended with a backslash character "\" then convert it
; to a double backlash, as the singgle backslash is a reserved charater in
; PDOX

      QChar.a = IIf(Substr(New.u,Len(New.u),1) = Chr(92),"\\","")

; If the new item is to be set imediately, then use and Execute.

      If Q.l Then
         Execute Cmd.a + " \"" + New.u + QChar.a + "\""
      Else

; If the prompt is for a PDOX configuration item, use the SYSINFO CONFIG
; Command to set the new parameter

         Switch
            Case Cmd.a = "Array" And Pop.a <> "Win" :
               Fld.r[Order.r[Pos.n]] = New.u
               Sysinfo Config From Fld.r

; If the prompt is for a window attribute, get the window attributes, of
; the original window, and change that item.

            Case Cmd.a = "Array" And Pop.a = "Win" :
               New.u=IIF(New.u="True",True,IIF(New.u="False",False,New.u))
               Fld.r[Order.r[Pos.n]] = New.u
               Window Echo Win.h True
               Window SetAttributes Win.h From Fld.r
               Window GetAttributes Win.h To Fld.r
               Window Select Handles.r["Display"]
               Style Attribute 31
               ?? Spaces(S.n-1), Format(F1.a,Order.r[Pos.n])
               If Numval(Fld.r[Order.r[Pos.n]]) <> "Error" Then
                  ?? Format(F3.a,Fld.r[Order.r[Pos.n]])
               Else
                  ?? Format(F2.a,Fld.r[Order.r[Pos.n]])
               Endif
               Style

; Otherwise the command can just be run, with no special consequences.

            Otherwise :
               Execute Cmd.a + " " + New.u + Qchar.a
         EndSwitch
      Endif
   Endif
   Echo Normal
Endif
Return
Endproc 					; PromptEngine.u()
;-------------------------------------------------------------------------------
;    PROCNAME: WinAtt.u()
; DESCRIPTION: Get the window attributes of the current workspace window;
;     EXAMPLE: WinAtt.u()
;  PARAMETERS: None
;     RETURNS: True  - If the current window is a non sysem window (Handle>0)
;	       False - System Window (Handle=0)
;SETS GLOBALS: Fld.r - Window attributes of the current workspace window
;USES GLOBALS: None
;  PROCEDURES: None
;      AUTHOR: Harry Goldman
;	       DataBase Designs, Inc.
;	       Copyright 1993, All Rights Reserved
;     CREATED:
; ATTRIBUTION: none
; MODIFIED BY:
;    COMMENTS: Authorization is hearby granted for the reuse and or
;	  distribution of this application in part or in whole with
;	       proper attribution to the author.
;-------------------------------------------------------------------------------
Proc WinAtt.u()

Win.h = WindowAt(Rec.r["Row"],Rec.r["Col"])     ; Get the Window at the
                                                ; current cursor position
If Win.h > 0 Then                               ; If this is not a system
   Window GetAttributes Win.h To Fld.r          ; window, get its attributes
Else                                            ;
   Beep                                         ; Otherwise tell the user
   Message "No Window At This Screen Location"  ; that there is no window
Endif                                           ;
Return Win.h > 0                                ; Return True/False
EndProc 					 ; Winatt.u()
;-------------------------------------------------------------------------------
;    PROCNAME: GetClick.u()
; DESCRIPTION: After a right click has occured we need to determine what
;              type of object inspector menu is needed. This is done by
;              looking at waht is under the mouse position that was clicked on.
;
;     EXAMPLE: GetClick.u()
;  PARAMETERS: None
;     RETURNS: ClickType.u - Type of inspector menu to display
;SETS GLOBALS: None
;USES GLOBALS: Rec.r - Dynarray of current event
;  PROCEDURES: None
;      AUTHOR: Harry Goldman
;              DataBase Designs, Inc.
;              Copyright 1994, All Rights Reserved
;     CREATED:
; ATTRIBUTION: none
; MODIFIED BY:
;    COMMENTS: Authorization is hearby granted for the reuse and or
;              distribution of this application in part or in whole with
;              proper attribution to the author.
;-------------------------------------------------------------------------------
Proc GetClick.u()
   private Win.h,        ; Window handle of click
	   Att.r,				 ; Attributes of the window
      FullFld.a,         ;
	   Type.a,				 ;
      zz, Ms.a,          ;
	   Frm.a,				 ;
	   Title.a				 ;

;  ClickType.u = 1 : SystemMenu.u()
;  Clicktype.u = 2 : WindowMenu.u()
;  Clicktype.u = 3 : TableMenu.u()
;  Clicktype.u = 4 : FieldMenu.u()

ClickType.u = 1

Win.h = Windowat(Rec.r["Row"],Rec.r["Col"])     ; Check for a window

If Win.h < 1 Then                               ; If there is no window at the
   Return                                       ; click, stop, use a system
Endif                                           ; menu

ExecEvent Rec.r                                 ; Allow the click
Tbl.a = ""                                      ; Default the table name

Window GetAttributes Win.h To Att.r             ; Get the attributes of the
Switch                                          ; window
   Case Upper(Att.r["Type"]) = "IMAGE" or       ; IMAGE or FORM Window
        Upper(Att.r["Type"]) = "FORM" :

; If the window that was clicked on is an image window (it has a table in it)
; then we need to determine where on the window the click occured. If the
; click is on the window frame, show a Window menu.

   Switch
      Case Rec.r["Row"] = Att.r["OriginRow"] : ClickType.u = 2  ; Top Frame
      Case Rec.r["Col"] = Att.r["OriginCol"] : ClickType.u = 2  ; Left Frame
      Case Rec.r["Row"] = Att.r["OriginRow"] + Att.r["Height"] - 1 :
                      ClickType.u = 2                           ; Bottom Frame
      Case Rec.r["Col"] = Att.r["OriginCol"] + Att.r["Width"] - 1 :
                      ClickType.u = 2                           ; Right Frame
      Case Rec.r["Row"] > Att.r["OriginRow"] + 1 And
      Rec.r["Col"] > Att.r["OriginCol"] + 7 : Clicktype.u = 4   ; Field Data
					       If Field() = "#" Then
                         ClickType.u = 3
					       Endif
      Otherwise : ClickType.u = 3
   Endswitch

; If the window is the form designer, we will have to handle things a bit
; differently. We don't want to make changes that will have an effect
; outside of the form designer. We do on the other hand want to be able to
; inspect fields placed on the form.
;
; We can tell if this is a form design window by checking the value of the
; TYPE attribute of the window. We will also grab the title of the window
; and parse from it the name table and the form.
;
; We then need to determine if the item to inspect is a field, and if so,
; is it a field in the current table, or an embeded table, or a calculated
; field. If this is a calculated field call the proc to display the contents
; of the field, and set the Clicktype to 0 to termiate the processing.
;
; NOTE: The FieldInfo() function will tell you that the field type is
; "Formula" on a calculated field in the form designer. In the report
; designer a calculated field doesn't report correctly.


   Case Upper(Att.r["Type"]) = "FORM DESIGN"  :
      FullFld.a = Fieldinfo()
      zz = Match(Att.r["Title"],"..: ..",Title.a,Tbl.a)
      zz = Match(Tbl.a,"...F..",Tbl.a,Frm.a)
      If FullFld.a = "" Then
         ClickType.u = 3
      Else
         If Not Match(FullFld.a,"[..->..]",Tbl.a,Fld.a) Then
            zz = Match(FullFld.a,".., ..",Type.a,Fld.a)
         Endif
         If IsAssigned(Type.a) and Type.a = "Formula" Then
            ShowCalc.u()
            ClickType.u = 0
            Return
         EndIf
         ClickType.u = 5
      Endif

; If the window is the report designer, we want to handle things like we
; did in the form designer. We don't want to make changes that will have
; an effect outside of the report designer. We do on the other hand want
; to be able to inspect fields placed on the report.
;
; We can tell if this is a report design window by checking the value of
; the TYPE attribute of the window. We will also grab the title of the
; window and parse from it the name table and the report.
;
; We then need to determine if the item to inspect is a field, and if so,
; is it a field in the current table, or an embeded table, or a calculated
; field. If this is a calculated field call the proc to display the contents
; of the field, and set the Clicktype to 0 to termiate the processing.

   Case Upper(Att.r["Type"]) = "REPORT DESIGN"  :
      FullFld.a = Fieldinfo()
      zz = Match(Att.r["Title"],"..: ..",Title.a,Tbl.a)
      zz = Match(Tbl.a,"...R..",Tbl.a,Frm.a)
      If FullFld.a = "" Then
         ClickType.u = 3
      Else
         If Not Match(FullFld.a,"[..->..]",Tbl.a,Fld.a) Then
            Fld.a = FullFld.a
         Endif
         Echo Off
         Menu {Field} {Calcedit} Enter Ms.a = Window()
         Menu Esc
         Echo Normal
         If Ms.a = "" Then
            ShowCalc.u()
            ClickType.u = 0
            Return
         Endif
         ClickType.u = 5
      Endif
   Otherwise : Clicktype.u = 2                  ; In all other cases, use
Endswitch                                       ; a system inspect menu
Return
Endproc 					; Getclick.u
;-------------------------------------------------------------------------------
;    PROCNAME: SystemMenu.u
; DESCRIPTION: Place the field inspection menu on the workspace;
;     EXAMPLE: SystemMenu.u()
;  PARAMETERS: None
;     RETURNS: None
;SETS GLOBALS: Pop.a
;USES GLOBALS: None
;  PROCEDURES: None
;      AUTHOR: Harry Goldman
;	       DataBase Designs, Inc.
;	       Copyright 1993, All Rights Reserved
;     CREATED:
; ATTRIBUTION: none
; MODIFIED BY:
;    COMMENTS: Authorization is hearby granted for the reuse and or
;	  distribution of this application in part or in whole with
;	       proper attribution to the author.
;-------------------------------------------------------------------------------
Proc SystemMenu.u()
ShowPopup "SYSTEM INSPECT!" @ Row.n,Col.n
   "Paradox"       : "Show Paradox Information"       : "Sysp",
   "Configuration" : "Show Configuration Level Info"  : "Sysc",
   "Colors"        : "Show System Colors"             : "Colors"
      Submenu
         "Menu"    : "Menu Color Settings"                 : "ColorsM",
         "Windows" : "Window Color Settings"               : "ColorsW",
         "Dialog"  : "Dialog Box Color Setup"              : "ColorsD",
         "Speedbar": "Speedbar Color Setup"                : "ColorsS",
         "PAW"     : "Application Workshop Color Setup"    : "ColorsA",
         "Reset"   : "Reset Color Settings To The Default" : "ColorsR"
      EndSubmenu,
   "Drives"        : "Show Drive Status"              : "Sysd",
   "Environment"   : "Show Environment Information"   : "Syse"
EndMenu
To Pop.a
EndProc 				; SystemMenu.u()
;-------------------------------------------------------------------------------
;    PROCNAME: WindowMenu.u
; DESCRIPTION: Place the field inspection menu on the workspace;
;     EXAMPLE: WindowMenu.u()
;  PARAMETERS: None
;     RETURNS: None
;SETS GLOBALS: Pop.a
;USES GLOBALS: None
;  PROCEDURES: None
;      AUTHOR: Harry Goldman
;	       DataBase Designs, Inc.
;	       Copyright 1993, All Rights Reserved
;     CREATED:
; ATTRIBUTION: none
; MODIFIED BY:
;    COMMENTS: Authorization is hearby granted for the reuse and or
;	  distribution of this application in part or in whole with
;	       proper attribution to the author.
;-------------------------------------------------------------------------------
Proc WindowMenu.u()
ShowPopup "WINDOW INSPECT!" @ Row.n,Col.n
   "Attributes" : "Show Window Attributes" : "Win",
   "Colors"     : "Show Window Colors"     : "ColorsW"
EndMenu
To Pop.a
EndProc 			       ; WindowMenu.u()
;-------------------------------------------------------------------------------
;    PROCNAME: TableMenu.u
; DESCRIPTION: Place the field inspection menu on the workspace;
;     EXAMPLE: TableMenu.u()
;  PARAMETERS: None
;     RETURNS: None
;SETS GLOBALS: Pop.a
;USES GLOBALS: None
;  PROCEDURES: None
;      AUTHOR: Harry Goldman
;	       DataBase Designs, Inc.
;	       Copyright 1993, All Rights Reserved
;     CREATED:
; ATTRIBUTION: none
; MODIFIED BY:
;    COMMENTS: Authorization is hearby granted for the reuse and or
;	  distribution of this application in part or in whole with
;	       proper attribution to the author.
;-------------------------------------------------------------------------------
Proc TableMenu.u()
ShowPopup "TABLE INSPECT!" @ Row.n,Col.n
   "Structure" : "Show Table Structure Information" : "Struct",
   "Fields" : "Show The Fields In This Table"  :  "Flds",
   "Highlight" : "Set Highlights"              : "Hg"
      SubMenu
         "Set Record" :  "Set A Record Highlight"   : "Hlrec",
         "Set Field"  :  "Set A Field Highlight"    : "HlFld",
         "Clear"      :  "Clear The Highlight"      : "HlClr"
      EndSubMenu,
   "Index"  : "Show Indexes Defined For This table" : "Idx",
   "Forms"  : "Show Forms Defined For This Table"   : "Forms",
   "Reports": "Show Reports Defined For This Table" : "Rpt"
EndMenu
To Pop.a
EndProc 			  ; TableMenu.u()
;-------------------------------------------------------------------------------
;    PROCNAME: FieldMenu.u
; DESCRIPTION: Place the field inspection menu on the workspace;
;     EXAMPLE: FieldMenu.u()
;  PARAMETERS: None
;     RETURNS: None
;SETS GLOBALS: Pop.a
;USES GLOBALS: None
;  PROCEDURES: None
;      AUTHOR: Harry Goldman
;	       DataBase Designs, Inc.
;	       Copyright 1993, All Rights Reserved
;     CREATED:
; ATTRIBUTION: none
; MODIFIED BY:
;    COMMENTS: Authorization is hearby granted for the reuse and or
;	  distribution of this application in part or in whole with
;	       proper attribution to the author.
;-------------------------------------------------------------------------------
Proc FieldMenu.u()
ShowPopup "Field INSPECT!" @ Row.n,Col.n
   "Highlight" : "Set Highlights"              : "Hg"
      SubMenu
         "Set Record" :  "Set A Record Highlight"   : "Hlrec",
         "Set Field"  :  "Set A Field Highlight"    : "HlFld",
         "Clear"      :  "Clear The Highlight"      : "HlClr"
      EndSubMenu,
   "Settings"  : "Show Image Settings Defined For This Table" : "Setf",
   "Valchecks" : "Show Valchecks For The Current Field"       : "Vlchk"
EndMenu
To Pop.a
EndProc 				    ; FieldMenu.u()
;-------------------------------------------------------------------------------
;    PROCNAME: DoHighlight.u
; DESCRIPTION: Procedure to set up a highlight bar.
;
;     EXAMPLE: DoHighlight.u()
;  PARAMETERS: None
;     RETURNS: None
;SETS GLOBALS: Highlight bar (field or record) on the current image
;USES GLOBALS: Pop.a
;  PROCEDURES: None
;      AUTHOR: Harry Goldman
;	       DataBase Designs, Inc.
;	       Copyright 1993, All Rights Reserved
;     CREATED:
; ATTRIBUTION: none
; MODIFIED BY:
;    COMMENTS: Authorization is hearby granted for the reuse and or
;	  distribution of this application in part or in whole with
;	       proper attribution to the author.
;
;	       This procedure does not save the highlight bar in the image
;	       settings ( {Image} {Keepset} .) If the user wishes to save the
;	       current settings, they must do so form the PDOX menus.
;-------------------------------------------------------------------------------
Proc DoHighlight.u()
   private INum.n,				 ; current image
      C.r                      ; Dynarray of colors

INum.n = Imageno()                              ; Get the current image #
						 ;
If Pop.a = "HlClr" Then                         ; Check the type of box
   Sethighlight Inum.n Record 0 Field 0         ; allow for a reset of the
   Return                                       ; highlight bar
Endif                                           ;
                                                ;
GetColors To C.r                                ; get the current colors
                                                ;
ChangeColor.u(Colorset.r["99"],99,C.r)          ; Call the color routine
						;
If Retval > 0 Then                              ; If highlight colors need
   Switch                                       ; to be changed

      Case Pop.a = "Hlrec" : SetHighlight Inum.n Record Retval  ; record
      Case Pop.a = "HlFld" : SetHighlight Inum.n Field Retval   ; field
   Endswitch
Endif                                           ; now that the highlight is
                                                ; set, we can return
Return                                          ;
Endproc 			; DoHighlight.u()
;-------------------------------------------------------------------------------
;    PROCNAME: ShowCalc.u
; DESCRIPTION: Procedure to set up a calculated field from a form or report.
;
;     EXAMPLE: ShowCalc.u()
;  PARAMETERS: None
;     RETURNS: None
;SETS GLOBALS: None
;USES GLOBALS: None
;  PROCEDURES: None
;      AUTHOR: Harry Goldman
;	       DataBase Designs, Inc.
;	       Copyright 1993, All Rights Reserved
;     CREATED:
; ATTRIBUTION: none
; MODIFIED BY:
;    COMMENTS: Authorization is hearby granted for the reuse and or
;	  distribution of this application in part or in whole with
;	       proper attribution to the author.
;
;-------------------------------------------------------------------------------
Proc ShowCalc.u()
   private Line.a,                              ; Calculated field contents
           H.n,                                 ; Width of calculated field
           I                                    ; Counter
                                                ;
Echo Off                                        ; Grab the contents of the
Menu {Field} {CalcEdit} Enter Line.a = MenuChoice() ; calculated field
Menu Esc                                        ;
                                                ;
H.n = 1 + Int(Len(Line.a)/50)                   ; Get the width
                                                ;
BuildWindow.u(H.n+3)                            ; Build a display window
                                                ;
@ 1,0 ?? Format("W55,AC","Calculated Field Contents") ; Header
@ 2,0                                           ;
                                                ;
For I From 1 To H.n                             ; Show the contents of the
   ? " " + Substr(Line.a,(I-1)*50+1,50)         ; calculated field 50 char's
Endfor                                          ; at a time
Echo Normal                                     ;
Return True                                     ;
EndProc
;-------------------------------------------------------------------------------


ErrorProc = "InspError.u"

Inspect.u()

Release Vars All
Release Procs All
