PROC dGetDate(dDfltDate)
;=============================================================================
;Copyright 1992 Professional Information Consulting, Inc., Leon Chalnick
;You are granted the right to use this code in your applications as long as 
;this notice is included.  You are not granted the right to resell this code.
;=============================================================================
;AUTHOR:		PIC-LMC
;DATE:		8/27/92
;FUNCTION:
;	Displays a calendar, allows user to scroll through days, weeks, months,
;years (using keyboard or mouse) and select a date.  It also provides an
;accept control into which user can directly enter the date (the calendar will
;be synchronized when user hits OK or leaves the accept control).  It provides
;a very friendly mechanism for onbtaining a date from the user. It will be most
;useful when the user would benefit from seeing an entire calendar.
;	This proc is primarily responsible for specifying the d-box. It is the
;d-box proc (dbpGetDate) that controls the processing of user input.
;	Note that the calendar is displayed through a single paintcanvas command.
;The calendaris actually a formatted text string produced in NewMonth().  It
;gets divided into rows which are stored in a dynamic array.  However, because
;the text string is formatted so that each line of the calendar is exactly the
;same length (and its < 255 chars), the paintcanvas command can paint it out
;over a rectangular area with one command.
;	Note the the 'mouse buttons' used for scrolling through months and years
;are not really buttons, but are produced w/canvas painting commands.  We
;check for mouse clicks on them by trapping the mouse "DOWN" event, and
;localizing it to screen coordinates wihtin the dbox window.
;	Here is the keyboard interface (mouse buttons follow these conventions too):
;	Key				Action
;-----------------------------------------
;	Left				previous day
;	Right				next day
;	Up					previous week
;	Down				next week
;	PgUp				same day, previous month
;	PgDn				same day, next month
;	Ctrl+PgUp		same day, previous year
;	Ctrl+PgDn		same day, next year
;
;	Enter				Ok
;	Esc				Cancel
;
;PARAMS:
;	dDfltDate - the date to which calendar will default.
;CALLS:
;	SetUpCal   - Sets up initial display of calander
;	dbpGetDate - d-box proc for this dialog box.
;RETURNS:
;	If user hit Ok, returns dCdate, the currently selected date.  If user
;hit cancel, returns a blankdate.
;=============================================================================
private
;Regional vars--will be used and/or initialized by one or more called procs:
hDBox,		;Handle to dBox window
lOk,			;True if user exited with Ok button, false if user exits w/Cancel
nCRow,		;Cursor row
nCCol,		;Cursor col
nOldCol,		;Last cursor column (used to paint over previous cursor location)
nOldRow,		;Last cursor row (used to paint over previous cursor location)
a,				;Text string containing entire formatted list of days
dyCal,		;Dynarray with formatted callendar (variable a, sliced up)
nCalRows,	;Number of rows in calander
nBtnClr,		;Pdox color for dbox buttons, used on Month/Year 'mouse buttons'
dCdate,		;Date that cursor is on-return value
aMth,			;Month of dCdate (for mouse button display)
aYr,			;Year of dCdate (for mouse button display)
;Local vars--Only used in this proc
aOkCancel	;Control var for OK Cancel buttons

lOk=false
SetUpCal()										;Set up initial calendar display
showdialog "Pick-A-Date"
	proc "dbpGetDate"
		trigger "OPEN", "ACCEPT", "DEPART"
		key "UP", "DOWN", "LEFT", "RIGHT", "PGUP", "PGDN", "CTRLPGUP", "CTRLPGDN"
		mouse "DOWN"
	@2,24 height 17 width 30
	@0,3 ?? " SU MO TU WE TH FR SA"

	;Paint the string of dates
	paintcanvas fill a attribute nDBoxClr 2,3,nCalRows+2,23

	;Paint two-byte 'cursor' over old cursor location
	paintcanvas attribute nDboxClr nOldRow,nOldCol,nOldRow,nOldCol+1

	;Paint new two-byte 'cursor'
	paintcanvas attribute nBtnClr nCRow,nCCol,nCRow,nCCol+1

	;Paint frame and highlight it
	frame single from 1,3 to 1+nCalRows+1,24
	paintcanvas attribute nDBoxHlt 1,24,1+nCalRows+1,24				;Vert hlt
	paintcanvas attribute nDBoxHlt 1+nCalRows+1,4,1+nCalRows+1,23	;Horz hlt

	;Paint "mouse buttons" for scrolling months/years
	@9,4 ?? chr(27)+" "+aMth+" "+chr(26)+"     "+chr(27)+" "+aYr+" "+chr(26)
	paintcanvas attribute nBtnClr 9,3,9,24
	paintcanvas attribute nDBoxClr 9,6,9,8
	paintcanvas attribute nDboxClr 9,12,9,14
	paintcanvas attribute nDBoxClr 9,18,9,21

	pushbutton @ 13,2 width 10
		" Ok"
		ok 
		default
		value "OK"
		tag "OK"
		to aOkCancel

	pushbutton @ 13,15 width 10
		"Cancel"
		cancel 
		value "CANCEL"
		tag "CANCEL"
		to aOkCancel

	accept @ 11,16 width 11
		"D"
		tag "ACCEPTDATE"
		To dAcptDate

	label @ 11,0
		"~S~elected date:"
		for "ACCEPTDATE"

enddialog
return iif(lOk,dCdate,blankdate())
ENDPROC

PROC dbpGetDate(aType,aTag,vEventVal,vElementVal)
;=============================================================================
;Copyright 1992 Professional Information Consulting, Inc., Leon Chalnick
;You are granted the right to use this code in your applications as long as 
;this notice is included.  You are not granted the right to resell this code.
;=============================================================================
;AUTHOR:		PIC-LMC
;DATE:		8/27/92
;FUNCTION:
;	Controls processing of events that we trapped for in d-box spec.  Calls
;appropriate procs for subsequent processing.
;	Keyboard events are easy to process.  To process mouse clicks, we must 
;localize the event to the dbox window and determine what action to take based
;on whcih row/column mouse was clicked in.  NOTE THAT ROW AND COLUMN
;COORDINATES RESULTING FROM THE "localize evnet" COMMAND ARE 1 GREATER THAN THE
;COORDINATES THAT WERE DEFINED IN THE DBOX!
;	Note also that Paradox can't properly proceses events once they have been
;localized to a window (and we want it to exec localized events that are not in
;the coordinates of either the calendar or the 'mouse buttons'; i.e., push
;button and d-box moving events.)  To work around this problem, we'll copy the
;original mouse event from vEventVal to dyEventRec and localize dyEventRec
;instead of vEventVal.  That way, Pdox can still process the original event
;(vEventVal) when this proc returns true.
;
;PARAMS:
;	The normal params for a d-box:  aType is the type of event that called
;the dbox proc.  aTag is the tag of the current control element. vEventVal
;is usually (always in this case) a dynamic array containg a record of the 
;event that resulted in call to d-box proc.  vElementVal will be blank in this
;d-box, but will contain value related to check box, if one is used in d-box.
;
;CALLS:
;	NewDay         - Determines new cursor location for movment of one day
;	IncreaseDate   - Increases date by one DAY, WEEK, MONTH or YEAR
;	DecreaseDate   - Decreases date by one DAY, WEEK, MONTH or YEAR
;
;CALLED BY:
;	dGetDate
;=============================================================================
private 
nKey,				;If call was for a key event, will contain keycode, otherwise, 0
lMouseClick,	;True if call was for a mouse down event
nRow,				;If call was for mouse down, row localized to dbox widnow
nCol,				;If call was for mouse down, column localized to dbox widnow
nDay,				;If call was for mouse down, day user clicked on (or "Error" if
					;user clicked on blank).
dyEventRec,		;We need to localize the mouse event to this d-box window.  The
					;problem is, Paradox can't execute events that have been 
					;localized, so we copy the original event (vEventVal) into this
					;dynarray and localize THIS dynarray.
aEventTag,		;Event tag identifier used in 'foreach loop'
lProcessEvent	;Return value

nKey=iif(aType="EVENT" and vEventVal["TYPE"]="KEY", vEventVal["KEYCODE"], 0)
lMouseClick=iif(aType="EVENT" and vEventVal["TYPE"]="MOUSE", vEventVal["ACTION"]="DOWN", false)
lProcessEvent=true
switch
case aType="ACCEPT":
	lOk=true
case aType="DEPART" and aTag="ACCEPTDATE":
	;The user is departing the accept control.  We'll synchronize the display
	;of the mouse buttons, calendar and the cursor to the new date.  This code
	;is not necessary if user didn't change date, but it's so fast that it is
	;simpler to do it anyway.
	dCdate=dAcptDate
	aMth=substr("JanFebMarAprMayJunJulAugSepOctNovDec",month(dCdate)*3-2,3)
	aYr=strval(year(dCdate))
	NewMonth(dCdate)
	NewDay(dCdate)
	refreshdialog
case nKey<>0 and aTag="ACCEPTDATE":
	;We'll return true--user is in accept control, but is hitting keys that 
	;we're trapping for.  We'll just let Pdox process them.
case lMouseClick:
	;Because localized events aren't properly processed when you return true
	;from the d-box proc, we will copy the event array to a dynarray and
	;localize the event in the copied array.
	dynarray dyEventRec[]
	foreach aEventTag in vEventVal
		dyEventRec[aEventTag]=vEventVal[aEventTag]
	endforeach
	localizeevent dyEventRec
	;Make sure click was in dbox window
	if dyEventRec["WINDOW"]=hDBox
		then nRow=dyEventRec["ROW"]
		nCol=dyEventRec["COL"]
		switch
		case (nRow>=3 and nRow<=2+nCalRows):
			;Check for mouse clicks in calendar itself (on col/row containing a number)
			if search(","+strval(nCol)+",", ",5,6,8,9,11,12,14,15,17,18,20,21,23,24,")<>0
				then nDay=numval(substr(dyCal[nRow-2],iif(mod(nCol,3)=0, nCol-1-3, nCol-3), 2))
				;If user clicked on a blank location (at beginning or ending of 
				;month), nDay will = Error.
				if nDay<>"Error"
					then dCdate=dateval(strval(month(dCdate))+"/"+strval(nDay)+"/"+strval(year(dCdate)))
					NewDay(dCdate)					;Determine row/col for new day
					;aMth is used in display of month in 'mouse buttons'
					aMth=substr("JanFebMarAprMayJunJulAugSepOctNovDec",month(dCdate)*3-2,3)
					dAcptDate=dCdate				;Resync value in accept control
					refreshdialog					;Executes d-box canvas painting commands
				endif ;Mouse click location wasn't blank
			endif ;User clicked on column/row that could have a date in it
			lProcessEvent=false
		case nRow=10:
			;Check for clicks on "mouse buttons" for changing months and years
			switch
			case nCol>=4 and nCol<=6:		;Decrease month
				DecreaseDate("MONTH")
			case nCol>=10 and nCol<=12:	;Increase month
				IncreaseDate("MONTH")
			case nCol>=16 and nCol<=18:	;Decrease year
				DecreaseDate("YEAR")
			case nCol>=23 and nCol<=25:	;Increase year
				IncreaseDate("YEAR")
			endswitch
			lProcessEvent=false
		otherwise:
			;User clicked mouse within d-box window boundry, but in an area in
			;which we will just let Paradox process the event.
		endswitch ;Row user clicked in
	endif ;User clicked mouse in dbox
;The rest of the events are keyboard events
case nKey=asc("UP"):
	DecreaseDate("WEEK")
case nKey=asc("DOWN"):
	IncreaseDate("WEEK")
case nKey=asc("LEFT"):
	DecreaseDate("DAY")
case nKey=asc("RIGHT"):
	IncreaseDate("DAY")
case nKey=asc("PGUP"):
	DecreaseDate("MONTH")
case nKey=asc("PGDN"):
	IncreaseDate("MONTH")
case nKey=asc("CTRLPGUP"):
	DecreaseDate("YEAR")
case nKey=asc("CTRLPGDN"):
	IncreaseDate("YEAR")
case aType="OPEN":
	;Save window handle; it's used to localize mouse events
	window handle dialog to hDbox
endswitch
return lProcessEvent
ENDPROC

PROC SetUpCal()
;=============================================================================
;Copyright 1992 Professional Information Consulting, Inc., Leon Chalnick
;You are granted the right to use this code in your applications as long as 
;this notice is included.  You are not granted the right to resell this code.
;=============================================================================
;AUTHOR:		PIC-LMC
;DATE:		8/27/92
;FUNCTION:
;	Sets up initial display of calendar, screen coordinates, other variables
;used throughout process.
;
;CALLED BY:
;	dGetDate
;CALLS:
;	NewMonth - Initializes vars used in producing calendar
;	NewDay   - Determines new cursor location for movment of one day
;=============================================================================
private
dy		;Temp dynarray for system colors

getcolors to dy
nBtnClr=dy["1040"]							;System color for dbox buttons
dynarray dyCal[]								;Dynarray used to hold calendar weeks
aMth=substr("JanFebMarAprMayJunJulAugSepOctNovDec",month(dDfltDate)*3-2,3)
aYr=strval(year(dDfltDate))
NewMonth(dDfltDate)
;Must set up "current" screen coordinates of cursor.  Will be changed in
;NewDay().
nCRow=2											;Row can't be less than 2
nCCol=4											;Column can't be less than 4
NewDay(dDfltDate)								;Determines actual scrn coord for cursor
dCdate=dDfltDate								;Set current cursor date to dflt date
dAcptDate=dDfltDate							;Resync value in accept control
ENDPROC

PROC NewMonth(dNewDate)
;=============================================================================
;Copyright 1992 Professional Information Consulting, Inc., Leon Chalnick
;You are granted the right to use this code in your applications as long as 
;this notice is included.  You are not granted the right to resell this code.
;=============================================================================
;AUTHOR:		PIC-LMC
;DATE:		8/27/92
;FUNCTION:
;	Sets up screen display of a new month.  This means re-initializing the text
;string , 'a', with the formatted list of days in the month.  It also must 
;split this string into a dynarray (dyCal[]) and determine the number of rows
;in the calendar.
;  The dynarray is used by NewDay() and dbpGetDate() in determining the row
;and column for dCdate.
;  The number of calendar rows is used in determining where certain things are
;to be painted and where the mouse was clicked.
;	The text string is produced so that it looks like this:
;
;"SPACES BEFORE 1ST DAY OF MTH"+" nn nn nn ..."+"SPACES AFTER LAST DAY OF MTH"
;
;PARAMS:
;	dNewDate  - This is the new date to which the cursor is being moved.
;CALLED BY:
;	SetUpCal
;	IncreaseDate
;	DecreaseDate
;=============================================================================
private
i,				;Counter in for loops
d1stOfMth,	;Date of first day in month
nOffset,		;Offset that the day in the week (of the 1st day in month) is from
				;Sunday
nSpacesB4,	;Number of spaces before 1st day of mth on row (there are 3 
				;spaces/day + 1 between each day)
nSpcsAft,	;Number of spaces from last day in month to end of cal
nDays			;Number of days in month

d1stOfMth=dateval(strval(month(dNewDate))+"/01/"+strval(year(dNewDate)))
nOffSet=(search(dow(d1stOfMth),"SunMonTueWedThuFriSat")+2)/3
nSpacesB4=(nOffset-1)*3
nDays=iif( month(dNewDate)=12, 31, 
		day(dateval(strval(month(dNewDate)+1)+"/1/"+strval(year(dNewDate)))-1) )
;Determine number of rows calendar will require
nCalRows=iif( mod(nSpacesB4+3*nDays,21)=0, (nSpacesB4+3*nDays)/21, int((nSpacesB4+3*nDays)/21)+1 )
;Build text string consisting of all days in month, padded in front and rear
;with spaces based on which days of the week the month starts and ends on.
a=spaces(nSpacesB4)
for i from 1 to nDays
	a=a+format("W3,AR",i)
endfor
;Determine, then add spaces after last day
nSpcsAft=0
nSpcsAft=(search(dow(d1stOfMth+nDays-1),"FriThuWedTueMonSun")+2)
a=a+spaces(nSpcsAft)
;Slice text string into formatted dynarray elements
for i from 1 to nCalRows
	dyCal[i]=substr(a,i*21-20,21)
endfor
ENDPROC

PROC NewDay(dNewDate)
;=============================================================================
;Copyright 1992 Professional Information Consulting, Inc., Leon Chalnick
;You are granted the right to use this code in your applications as long as 
;this notice is included.  You are not granted the right to resell this code.
;=============================================================================
;AUTHOR:		PIC-LMC
;DATE:		8/27/92
;FUNCTION:
;	Determines screen coordinates of new date on calendar in dbox.
;	Note that the var, nCRow is used first to identify the array element in 
;dyCal[] that contains the new day, then it's adjusted to represent the
;correct screen row for that day.
;
;PARAMS:
;	dNewDate  - The new calendar date for whcih cursor coords must be determined
;CALLED BY:
;	dbpGetDate
;	SetUpCal
;	DecreaseDate
;	IncreaseDate
;=============================================================================
private
i			;Counter used in determining current cursor row and column

nOldRow=nCRow									;Save old positions so they can be
nOldCol=nCCol									;painted over.
;Determine cursor row for new date
for i from 1 to nCalRows
	;Check day of last number in array element (at end of week)
	if day(dNewDate)<=numval(substr(dyCal[i],20,2))
	 or substr(dyCal[i],20,2)="  " 
		then nCRow=i
		quitloop
	endif
endfor
;At this point, nCrow represents dyCal[] array element containing day
;Determine cursor column for new date
for i from 1 to 7
	;Break row containing new day into 3 char chunks and compare to day
	if day(dNewDate)=numval(substr(dyCal[nCRow],i*3-2,3))
		then nCCol=3+(i*3-2)					;Days start in column 3
		quitloop
	endif
endfor
nCRow=nCRow+1									;Add 1 cause dates start on 2nd row
;Now nCRow represents d-box _window row_ containing day
ENDPROC 

PROC DecreaseDate(aPeriod)
;=============================================================================
;Copyright 1992 Professional Information Consulting, Inc., Leon Chalnick
;You are granted the right to use this code in your applications as long as 
;this notice is included.  You are not granted the right to resell this code.
;=============================================================================
;AUTHOR:		PIC-LMC
;DATE:		8/27/92
;FUNCTION:
;	Decreases the current date (dCdate) by the time period specifed in aPeriod.
;This is either a DAY, WEEK, MONTH or YEAR.  Days and weeks are easy, because
;the number of days by which date will be changing is absolute.  The
;functionality of changing months and years is a little differnt though.
;	If user increments by a month (or year), I wanted new date to be set to the
;same numeric day (e.g., the 15th day of the mth) but in the next (or previous)
;month (or year).  In most cases, this isn't a problem.  However, not all
;months have 31 days.  So if, for example, cursor was on January 31 and user 
;hits PgUp for next mth, cursor will move to the LAST in February, whatever it
;happens to be.  The same problem exists when moving from year to year in the
;case of Febrary--you don't know how many days it has.
;  The way we check this is to create the new date by creating the new date
;with the current day.  If the resulting date="Error", then you know that we
;need to grab the last day of the month instead.
;	Of course, there's also the problem of changing years as you move back and
;forth between December and January.  This is handled easily with an iif().
;
;PARAMS:
;	aPeriod  - The period by which dCdate is to be decreased.  This is either
;DAY, WEEK, MONTH or YEAR
;
;CALLS:
;	NewMonth - Initializes vars used in producing calendar
;	NewDay   - Determines new cursor location for movment of one day
;CALLED BY:
;	dbpGetDate
;=============================================================================
private
nNbrDays,		;Number of days by which date is to be changed
dOldDate,		;Original date.  Used to determine if month changed
nMth,				;New month; Makes date formula easier to read (a little <G>)
nYr				;New year; Makes date formula easier to read (a little <G>)

dOldDate=dCdate
switch
case aPeriod="DAY":
	dCdate=dCdate-1
case aPeriod="WEEK":
	dCdate=dCdate-7
case aPeriod="MONTH":
	nYr=iif(month(dCdate)=1, year(dCdate)-1, year(dCdate))
	nMth=iif(month(dCdate)=1, 12, month(dCdate)-1)
	;To determine last day of pervious month, will subtract 1 from 1st day of
	;current month.
	if "Error"=dateval(strval(nMth) +"/"+strval(day(dCdate))+"/"+ strval(nYr))
		then dCdate=dateval( strval(month(dCdate)) +"/1/"+ strval(year(dCdate)) )-1
	else
		dCdate=dateval(strval(nMth) +"/"+strval(day(dCdate))+"/"+ strval(nYr))
	endif
case aPeriod="YEAR":
	nYr=max(year(dCdate)-1,0)				;Don't let year go negative!
	;Logic below is same for MONTH, above
	if "Error"=dateval(strval(month(dCdate))+"/"+strval(day(dCdate))+"/"+strval(nYr))
		then nMth=iif(month(dCdate)=12, 1, month(dCdate)+1)
		dCdate=dateval( strval(nMth) +"/1/"+ strval(nYr) )-1
	else
		dCdate=dateval(strval(month(dCdate))+"/"+strval(day(dCdate))+"/"+strval(nYr))
	endif
endswitch
;Compare original date to new date to see if month changed.
if month(dOldDate)>month(dCdate)
 or month(dOldDate)=1 and month(dCdate)=12
 or year(dOldDate)<>year(dCdate)
	then NewMonth(dCdate)
endif
;Reproduce aMth and aYr for 'mouse button' display.  Not 100% efficient doing
;this on every call, but the code is quite fast...
aMth=substr("JanFebMarAprMayJunJulAugSepOctNovDec",month(dCdate)*3-2,3)
aYr=strval(year(dCdate))
NewDay(dCdate)
dAcptDate=dCdate								;Resync value in accept control
refreshdialog									;Executes d-box canvas painting commands
ENDPROC

PROC IncreaseDate(aPeriod)
;=============================================================================
;Copyright 1992 Professional Information Consulting, Inc., Leon Chalnick
;You are granted the right to use this code in your applications as long as 
;this notice is included.  You are not granted the right to resell this code.
;=============================================================================
;AUTHOR:		PIC-LMC
;DATE:		8/27/92
;FUNCTION:
;	Increases the current date (dCdate) by the time period specifed in aPeriod.
;This is either a DAY, WEEK, MONTH or YEAR.  It works just like DecreaseDate()
;but obviously, in the opposite direction, so to understand the issues that it
;has to deal with, read comments for DecreaseDate().  They were not combined
;into one proc because some of the number we have to deal with when increasing
;the date are actually different (e.g. look at the logic under MONTH type 
;procesing.)
;
;PARAMS:
;	aPeriod  - The period by which dCdate is to be increased.  This is either
;DAY, WEEK, MONTH or YEAR
;
;CALLS:
;	NewMonth - Initializes vars used in producing calendar
;	NewDay   - Determines new cursor location for movment of one day
;CALLED BY:
;	dbpGetDate
;=============================================================================
private
nNbrDays,		;Number of days by which date is to be changed
dOldDate,		;Original date.  Used to determine if month changed
nMth,				;New month; Makes date formula easier to read (a little <G>)
nYr				;New year; Makes date formula easier to read (a little <G>)

dOldDate=dCdate
switch
case aPeriod="DAY":
	dCdate=dCdate+1
case aPeriod="WEEK":
	dCdate=dCdate+7
case aPeriod="MONTH":
	nYr=iif(month(dCdate)=12, year(dCdate)+1, year(dCdate))
	nMth=iif(month(dCdate)=12, 1, month(dCdate)+1)
	;To determine last day of _Next_ month, must subtract 1 from 1st day of
	;two months from now.
	if "Error"=dateval(strval(nMth) +"/"+ strval(day(dCdate)) +"/"+ strval(nYr))
		then nMth=iif(month(dCdate)>=11, month(dCdate)-10, month(dCdate)+2)
		dCdate=dateval( strval(nMth) +"/1/"+ strval(year(dCdate)) )-1
	else
		dCdate=dateval( strval(nMth) +"/"+strval(day(dCdate))+"/"+ strval(nYr))
	endif
case aPeriod="YEAR":
	nYr=year(dCdate)+1
	if "Error"=
	 dateval(strval(month(dCdate)) +"/"+ strval(day(dCdate)) +"/"+ strval(nYr))
		then nMth=iif(month(dCdate)=12, 1, month(dCdate)+1)
		dCdate=dateval( strval(nMth) +"/1/"+ strval(nYr) )-1
	else
		dCdate=dateval( strval(month(dCdate)) 
		+"/"+ strval(day(dCdate))+"/"+ strval(nYr))
	endif
endswitch
;Compare original date to new date to see if month changed.
if month(dOldDate)<month(dCdate);Compare original date to new date
 or month(dOldDate)=12 and month(dCdate)=1
 or year(dOldDate)<>year(dCdate)
	then NewMonth(dCdate)
endif
;Reproduce aMth and aYr for 'mouse button' display.  Not 100% efficient doing
;this on every call, but the code is quite fast...
aMth=substr("JanFebMarAprMayJunJulAugSepOctNovDec",month(dCdate)*3-2,3)
aYr=strval(year(dCdate))
NewDay(dCdate)
dAcptDate=dCdate								;Resync value in accept control
refreshdialog									;Executes d-box canvas painting commands
ENDPROC

;=============================================================================
;Code below is an example of how to call dGetDate()
;
getcolors to x
nDboxClr=x["1036"]							;Dbox text
nDboxHlt=nDBoxClr-mod(nDBoxClr,16)+15	;Highlight for 3D dbox appearance

showpulldown									;Clear out menu
endmenu
clear												;clear PAL canvas
message dGetDate(dateval("2/2/92"))
retval=getchar()
release vars x, nDboxHlt, ndBoxClr
