; By: Dave Schlieder                                            10/12/90
;     Shared Logic, Inc.
;     1016 Second Street
;     Encinitas, Ca 92024
;     (619)943-1086
;     CompuServe: 72460,3561
;
;
; These procedures are part of a collection of utilities that I have developed
; and put together for the soul purpose of placing in the BORDB libs a single
; collection of utilities that perform the most often asked questions about
; date and time manipulation in Paradox.
;
; I do not use any of these functions as they stand at this time, but have
; used variations of them.  These procedures were built with extensive error
; checking as part of their structure.  This error checking may not be
; relevant in each and every case, nor may the defaults be acceptable
; for your particular requirement.
;
; Some of the procedures may alter the order in which the information is
; passed in order to compute a "positive" result, others default to a
; value if the parameter is blank.  It is up to you to determine if these
; conditions are proper for your situation.
;
; Some of these procedures would lend themselves to a scan loop of a table, for
; example.  The scan for conditions could handle the error checking, instead
; of the procedure being called.  Alternately in a scan loop, placing the code
; directly in the scan loop itself, instead of calling a procedure each time
; should increase the speed of the code.  Each case needs to be weighed on
; an individual basis
;
; I make no claims that these procedures are completely error free.  They have
; been tested, but that doesn't mean they were tested for every situation, or
; that they may have returned an incorrect value that I didn't catch.
;
; They are meant as a guide to be adapted as required for your particular
; situation.

;-------------------------------------------------------------------------------
; Seconds.to.Days.Time
; Purpose: Converts a number of seconds to a sting in the format of:
; "n Day(s) hh:mm:ss"
; Returns "Error" if the parameter passed is not a valid positive number
;
; Acceptable calls to this procedure:
;       Seconds.to.Days.Time(24*189)
;       Seconds.to.Days.Time(189325)
;
;-------------------------------------------------------------------------------
Proc Seconds.to.Days.Time(Seconds.to.Convert)   ; the number of seconds to
                                                ; to convert
Private Days,        ; number of computed days
        Hours,       ; number of computed hours
        Minutes,     ; number of computed minutes
        Seconds,     ; number of computed seconds
        Remainder    ; the remaining value of a math division function

; error checking
If Type(Seconds.to.Convert) <> "N" AND Type(Seconds.to.Convert) <> "S" Then
   Seconds.to.Convert = NumVal(Seconds.to.Convert)
   If Seconds.to.Convert = "Error" Then Return "Error" EndIf
EndIf

; insures a positive number
Seconds.to.Convert = Abs(Seconds.to.Convert)

; computes the number of days
Days = Int(Seconds.to.Convert/86400)
If Days = 1 Then Days = "1 Day " Else Days = StrVal(Days) + " Days " EndIf

; computes the hours, minutes and seconds
Remainder = Mod(Seconds.to.Convert,86400)     Hours = Int(Remainder/3600)
Remainder = Mod(Remainder,3600)               Minutes = Int(Remainder/60)
Seconds   = Mod(Remainder,60)

; return the result of the computations
Return Days + Fill("0",2-Len(Hours)) + StrVal(Hours) + ":" +
              Fill("0",2-Len(Minutes)) + StrVal(Minutes) + ":" +
              Fill("0",2-Len(Seconds)) + StrVal(Seconds)
EndProc

;-------------------------------------------------------------------------------
; Time.Math
; Purpose: Computes the time between two sets of dates and times.
; Returns the answer in seconds, which can be converted into days, hours,
; or minutes as required.
;
; Acceptable calls to this procedure:
;       Time.Math("1/31/90","","","13:46:24")
;       Time.Math(1/31/90,"23:59:59","1/1/1953","13:46:24")
;       Time.Math(1/31/2090,"","","")
;       Time.Math([Date In],[Time In],[Date Out],[Time Out])
;
;-------------------------------------------------------------------------------
Proc Time.Math(Date.1,   ; the date that matches the Time.1 parameter
               Time.1,   ; the time that matches the Date.1 parameter
               Date.2,   ; the date that matches the Time.2 parameter
               Time.2)   ; the time that matches the Date.2 parameter

Private Temp.Var,       ; a temporary holding variable used to switch the
                        ; values passed through the parameters of this procedure
                        ; as required
        Hours.1,        ; the hours of the Time.1 parameter
        Minutes.1,      ; the minutes of the Time.1 parameter
        Seconds.1,      ; the seconds of the Time.1 parameter
        Hours.2,        ; the hours of the Time.2 parameter
        Minutes.2,      ; the minutes of the Time.2 parameter
        Seconds.2,      ; the seconds of the Time.2 parameter
        Number.of.Days  ; the number of days between the Date.1 and Date.1
                        ; parameters

; testing for blank parameters
If IsBlank(Date.1) OR Date.1 = BlankDate() Then Date.1 = Today() EndIf
If IsBlank(Date.2) OR Date.2 = BlankDate() Then Date.2 = Today() EndIf

; testing for the proper data type
If Type(Date.1) <> "D" Then Date.1 = DateVal(Date.1) EndIf
If Type(Date.2) <> "D" Then Date.2 = DateVal(Date.2) EndIf

; error checking
If Date.1 = "Error" OR Date.2 = "Error" Then Return "Error" EndIf

; makes Date.1 and Time.1 the earliest date and time if it isn't
If Date.1 < Date.2 Then
   Temp.Var = Date.1      Date.1   = Date.2
   Date.2   = Temp.Var    Temp.Var = Time.1
   Time.1   = Time.2      Time.2   = Temp.Var
EndIf

; testing for blank parameters
If IsBlank(Time.1) Then Time.1 = Time() EndIf
If IsBlank(Time.2) Then Time.2 = Time() EndIf

; error checking
If Not Match(Time.1,"..:..:..",Hours.1,Minutes.1,Seconds.1) OR
   Not Match(Time.2,"..:..:..",Hours.2,Minutes.2,Seconds.2) Then
   Return "Error"
EndIf

; convert the matched variables to numbers
Hours.1   = NumVal(Hours.1  )   Minutes.1 = NumVal(Minutes.1)
Seconds.1 = NumVal(Seconds.1)   Hours.2   = NumVal(Hours.2  )
Minutes.2 = NumVal(Minutes.2)   Seconds.2 = NumVal(Seconds.2)

; more error checking
If Hours.1 = "Error" OR Minutes.1 = "Error" OR Seconds.1 = "Error" OR
   Hours.2 = "Error" OR Minutes.2 = "Error" OR Seconds.2 = "Error" Then
   Return "Error"
EndIf

; and some more error checking. note: the maximum time is 23:59:59
; one second more, and the time is 00:00:00
If Hours.1   > 23 OR Hours.1   < 0 OR Minutes.1 > 59 OR Minutes.1 < 0 OR
   Seconds.1 > 59 OR Seconds.1 < 0 OR Hours.2   > 23 OR Hours.2   < 0 OR
   Minutes.2 > 59 OR Minutes.2 < 0 OR Seconds.2 > 59 OR Seconds.2 < 0 Then
   Return "Error"
EndIf
If ( Hours.1 = 24 AND (Minutes.1 > 0 OR Seconds.1 > 0 ) ) OR
   ( Hours.2 = 24 AND (Minutes.2 > 0 OR Seconds.2 > 0 ) ) Then
   Return "Error"
Endif

; computes the seconds after midnight for both time parameters
Seconds.1 = Seconds.1 + Minutes.1 * 60 + Hours.1 * 3600
Seconds.2 = Seconds.2 + Minutes.2 * 60 + Hours.2 * 3600

; and the number of days
Number.of.Days = Date.1 - Date.2

If Number.of.Days = 0 Then
   ; the absolute value as the difference between two numbers is always
   ; a positive number and the absolute value of the difference between
   ; two numbers is always the same, no matter which number is greater
   Return Abs(Seconds.1-Seconds.2)
Else
   ; since there is more then one day, then we know that the time is the
   ; the number of full days between the dates plus the number of
   ; seconds till midnight of the first date plus the number of seconds
   ; after midnight of the last date
   Return (Number.of.Days -1) * 86400 + (Seconds.2 - 86400) + Seconds.1
EndIf

EndProc

;-------------------------------------------------------------------------------
; Time.Math.AM.PM
; Purpose: Computes the time between two sets of dates and times.
; This variation will handle times in the format of HH:MM AM and HH:MM PM
; Returns the answer in minutes, which can be converted into days or hours
; as required.
;
; Acceptable calls to this procedure:
;       Time.Math.AM.PM("1/31/90","12:34 AM","","10:46 pM")
;       Time.Math.AM.PM(1/31/90,"02:59 aM","1/1/1953","12:46 PM")
;
;-------------------------------------------------------------------------------
Proc Time.Math.AM.PM(Date.1,     ; the date that matches the Time.1 parameter
                     Time.1,     ; the time that matches the Date.1 parameter
                     Date.2,     ; the date that matches the Time.2 parameter
                     Time.2)     ; the time that matches the Date.2 parameter

Private Temp.Var,       ; a temporary holding variable used to switch the
                        ; values passed through the parameters of this procedure
                        ; as required
        Hours.1,        ; the hours of the Time.1 parameter
        Minutes.1,      ; the minutes of the Time.1 parameter
        PM.1,           ; used to determine if Time.1 is after 12:00 noom
        Hours.2,        ; the hours of the Time.2 parameter
        Minutes.2,      ; the minutes of the Time.2 parameter
        PM.2,           ; used to determine if Time.1 is after 12:00 noom
        Number.of.Days  ; the number of days between the Date.1 and Date.1
                        ; parameters

; testing for blank parameters
If IsBlank(Date.1) OR Date.1 = BlankDate() Then Date.1 = Today() EndIf
If IsBlank(Date.2) OR Date.2 = BlankDate() Then Date.2 = Today() EndIf

; testing for the proper data type
If Type(Date.1) <> "D" Then Date.1 = DateVal(Date.1) EndIf
If Type(Date.2) <> "D" Then Date.2 = DateVal(Date.2) EndIf

; error checking
If Date.1 = "Error" OR Date.2 = "Error" Then Return "Error" EndIf
If IsBlank(Time.1) Then Return "Error" EndIf
If IsBlank(Time.2) Then Return "Error" EndIf

; makes Date.1 and Time.1 the earliest date and time if it isn't
If Date.1 < Date.2 Then
   Temp.Var = Date.1      Date.1   = Date.2
   Date.2   = Temp.Var    Temp.Var = Time.1
   Time.1   = Time.2      Time.2   = Temp.Var
EndIf

; error checking
If Not Match(Time.1,"..:.. ..",Hours.1,Minutes.1,PM.1) OR
   Not Match(Time.2,"..:.. ..",Hours.2,Minutes.2,PM.2) Then
   Return "Error"
EndIf

; convert the matched variables to numbers
Hours.1   = NumVal(Hours.1  )   Minutes.1 = NumVal(Minutes.1)
Hours.2   = NumVal(Hours.2  )   Minutes.2 = NumVal(Minutes.2)

; converts these strings to upper case
PM.1 = Upper(PM.1)
PM.2 = Upper(PM.2)

; more error checking
If Hours.1 = "Error" OR Minutes.1 = "Error" OR
   Hours.2 = "Error" OR Minutes.2 = "Error" OR
   ( PM.1 <> "AM" AND PM.1 <> "PM" ) OR
   ( PM.2 <> "AM" AND PM.2 <> "PM" ) Then
   Return "Error"
EndIf

; set these variables to a logical value
PM.1 = ( PM.1 = "PM" )
PM.2 = ( PM.2 = "PM" )

; and some more error checking. note: the maximum time is 12:59
; one second more, and the time is 01:00
If Hours.1 > 12 OR Hours.1 < 0 OR Minutes.1 > 59 OR Minutes.1 < 0 OR
   Hours.2 > 12 OR Hours.2 < 0 OR Minutes.2 > 59 OR Minutes.2 < 0 Then
   Return "Error"
EndIf

; computes the minutes after midnight for both time parameters
Minutes.1 = Minutes.1 + Hours.1 * 60
Minutes.2 = Minutes.2 + Hours.2 * 60

; adjusts if it is a PM time
If PM.1 Then Minutes.1 = Minutes.1 + 720 EndIf
If PM.2 Then Minutes.2 = Minutes.2 + 720 EndIf

; computes the number of days
Number.of.Days = Date.1 - Date.2

If Number.of.Days = 0 Then
   ; the absolute value as the difference between two numbers is always
   ; a positive number and the absolute value of the difference between
   ; two numbers is always the same, no matter which number is greater
   Return Abs(Minutes.1-Minutes.2)
Else
   ; since there is more then one day, then we know that the time is the
   ; the number of full days between the dates plus the number of
   ; minutes till midnight of the first date plus the number of minutes
   ; after midnight of the last date
   Return (Number.of.Days -1) * 1440 + (Minutes.1 - 1440) + Minutes.2
EndIf

EndProc

;-------------------------------------------------------------------------------
; Proc Adjust.a.Date -   The procedure may return "Error", but should
;                        return a valid date if the date passed to this
;                        is a valid date
;
; Purpose: Adjusts a date based on the passed parameters.  The date to be
; adjusted needs to be a valid date.  Valid Paradox date formats are:
; 1) mm/dd/yy
; 2) dd.mm.yy
; 3) dd-Mon-yy
; The date can be adjusted by days, months or years, either forwards or
; backwards.  For example, a date can be adjusted forward a number of days,
; backwards a number of months, and forwards a number of years in one step.
; To adjust a date backwards by a number of days, months or years, simply
; pass in the argument list a negative number.
; The use or the Int() function is to ignore any decimals in any of the
; parameters.  This procedure will NOT adjust a date based on fractions of
; a month or year (or a day!).
;
; Order of precedence in this routine:
; First, months and years are added in, directly to the month and year values.
; If the resulting month is an illegal month, then the year value is adjusted
; up or down as required by the number of years, and the new month is then
; determined.  From this computed date, the procedure then assures that it
; is a valid date.  For example, adding 1 month to 1/31/90 results in the
; date of 2/31/90 which is invalid.  The procedure then adjusts the date
; down 1 day at a time until it arrives at a valid date.  It does not,
; however, add these days back into the date.
;
; The reasoning behind this is that it is often desired to bump a date by a
; given number of years or months, without changing the month.
;
; Once a valid date is computed after the month and year adjustment, then
; this procedure will add in the number of days to adjust.
;
; This procedure should not be used for adjusting a date by just a given number
; of days, as Paradox already handles that type of date math.
;
; Acceptable calls to this procedure:
;       Adjust.a.Date("1/31/90",0,3,0)
;       Adjust.a.Date(Today(),0,6,2)
;       Adjust.a.Date(Today()+89,-89,6,-20)
;       Adjust.a.Date([Date Field],0,1,5)
;
;-------------------------------------------------------------------------------
Proc Adjust.a.Date(Date.to.Adjust,      ; The date to be adjusted.
                                        ; Depending how this procedure is
                                        ; called, it may or not be a date.
                                        ; If this parameter is passed as the
                                        ; value of a date field or a
                                        ; variable that is a date, it will
                                        ; be a date.  If the parameter is
                                        ; passed as a literal surrounded by
                                        ; quotes, it will be a string
                    Days.to.Adjust,     ; Number of days to adjust by
                    Months.to.Adjust,   ; Number of months to adjust by
                    Years.to.Adjust)    ; Number of years to adjust by
                                        ; These values may be of type "N"
                                        ; or "S".  Any other type is ignored
                                        ; during the computation.
                                        ; If required, the procedure could
                                        ; be changed to attempt to convert
                                        ; these strings to a number using the
                                        ; NumVal(..) function, but I decided to
                                        ; leave that type of error checking out

Private Day.of.Date,     ; Day of the date to be adjusted
        Month.of.Date,   ; Month of the date to be adjusted
        Year.of.Date,    ; Year of the date to be adjusted
        Bad.Date         ; Logical - is the final date an invalid date

; initialize the variable
Bad.Date = False

If Type(Date.to.Adjust) <> "D" Then                       ; First, make sure
   Date.to.Adjust = DateVal(Date.to.Adjust)               ; the date to be
   If Date.to.Adjust = "Error" Then Return "Error" EndIf  ; adjusted is a
EndIf                                                     ; valid date

; Makes sure that the number of days to adjust is a number
If (Type(Days.to.Adjust) <> "N" AND Type(Days.to.Adjust) <> "S") Then
   Days.to.Adjust = 0
Else
   Days.to.Adjust = Int(Days.to.Adjust)
EndIf

Day.of.Date   = Day  (Date.to.Adjust)    ; Parse out the values of the day,
Month.of.Date = Month(Date.to.Adjust)    ; month and year of the date
Year.of.Date  = Year (Date.to.Adjust)

; Now we can adjust the number of years
If Type(Years.to.Adjust) = "N" OR Type(Years.to.Adjust) = "S" Then
   Year.of.Date = Int(Year.of.Date + Int(Years.to.Adjust))
EndIf
; and the number of months
If Type(Months.to.Adjust) = "N" OR Type(Months.to.Adjust) = "S" Then
   Month.of.Date = Int(Month.of.Date + Int(Months.to.Adjust))
EndIf

; This switch will make any adjustments to the year as required if the month
; adjustment puts us into another year, either forwards or backwards
Switch
   Case Month.of.Date > 12 :
      Year.of.Date  = Int(Year.of.Date + Int(Month.of.Date / 12))
      If Int(Mod(Month.of.Date,12)) <> 0 then
         Month.of.Date = Int(Mod(Month.of.Date,12))
      Else
          Month.of.Date = 12
      EndIf
   Case Month.of.Date < 1 :
      Month.of.Date = Int(Month.of.Date - 12)
      Year.of.Date  = Int(Year.of.Date - Int((Month.of.Date) / -12))
      If Int(Mod(Month.of.Date,-12)) <> 0 then
         Month.of.Date = Int(12 + Mod(Month.of.Date,-12))
      Else
          Month.of.Date = 1
      EndIf
Endswitch

Day.of.Date   = StrVal(Day.of.Date)   ; Convert the values to strings
Month.of.Date = StrVal(Month.of.Date) ; for the following while loop
Year.of.Date  = StrVal(Year.of.Date)

; This loop will adjust the day down as required if the current month
; can not have that day, ie. 9/31/90 is illegal and will be adjusted
; to 9/30/90
While DateVal(Month.of.Date+"/"+Day.of.Date+"/"+Year.of.Date) = "Error"
   Day.of.Date = StrVal(Int( NumVal(Day.of.Date) -1 ))
   ; This is a safety valve so there is a way out of the loop if we never
   ; trip a legal date.  Eventually, Day.of.Date will equal "0" or "-n"
   If Day.of.Date = "0" OR Search("-",Day.of.Date) <> 0 Then
      Bad.Date = True QuitLoop
   EndIf
Endwhile

If Bad.Date Then
   ; if the above loop somehow terminated and never reached a valid date,
   ; return "Error"
   Return "Error"
Else
   ; otherwise return the adjusted date, adding in the days to adjust
   Return DateVal(Month.of.Date+"/"+Day.of.Date+"/"+Year.of.Date)+Days.to.Adjust
EndIf

EndProc

;-------------------------------------------------------------------------------
; Years.Months.Days
; Purpose: Returns the number of years, months and days between any two given
; dates.  Any blank value or blankdate defaults to Today()
; Returns a string in the pattern of "n Year(s), n Month(s), n Day(s)" or
; the string "Error" if one of the dates is illegal.
; The dates can be supplied in either order, ie. oldest/latest or latest/newest
;
; Acceptable calls to this procedure:
;       Years.Months.Days("1/31/90","")
;       Years.Months.Days(Today(),1/1/1934)
;       Years.Months.Days(Today()+89,"1/31/90")
;       Years.Months.Days([Date Field],BlankDate())
;
;-------------------------------------------------------------------------------
Proc Years.Months.Days(Date.1,  ; One of two dates
                       Date.2)  ; The other of two dates
Private Temp.Var,    ; temporary holding variable to switch the dates if
                     ; it is determined that Date.2 is earlier then Date.1
        Years,       ; number of years between the dates
        Months,      ; number of months between the dates
        Days,        ; number of days between the dates
        Plus.Days    ; variable used to adjust the days variable based on the
                     ; number of possible days in the Date.2 parameter

; fill in the default values id the dates are blank
If IsBlank(Date.1) OR Date.1 = BlankDate() Then Date.1 = Today() EndIf
If IsBlank(Date.2) OR Date.2 = BlankDate() Then Date.2 = Today() EndIf

; attempt to convert string dates to the proper data type
If Type(Date.1) <> "D" Then Date.1 = DateVal(Date.1) EndIf
If Type(Date.2) <> "D" Then Date.2 = DateVal(Date.2) EndIf

; error checking
If Date.1 = "Error" OR Date.2 = "Error" Then Return "Error" EndIf

; make Date.1 the earliest date if it isn't already
If Date.1 < Date.2 Then
   Temp.Var = Date.1   Date.1 = Date.2   Date.2 = Temp.Var
EndIf

; compute the number of years between the two dates
If Month(Date.2) < Month(Date.1) OR
   (Month(Date.2) = Month(Date.1) AND Day(Date.2) <= Day(Date.1)) Then
   Years = Year(Date.1) - Year(Date.2)
Else
   Years = Year(Date.1) - Year(Date.2) - 1
EndIf
If Years = 1 Then
   Years = "1 Year,"
Else
   Years = StrVal(Years) + " Years, "
EndIf

; compute the number of months between the two dates
Months = Month(Date.1) - Month(Date.2)
If Day(Date.1) < Day(Date.2) Then Months = Months - 1 EndIf
If Months < 0 Then Months = Months + 12 EndIf
If Months = 1 Then
   Months = "1 Month,"
Else
   Months = StrVal(Months) + " Months, "
EndIf

; compute how many days may be in the month of the latest date
Plus.Days = 31
While DateVal(StrVal(Month(Date.2))+"/"+StrVal(Plus.Days)+"/"+
              StrVal(Year(Date.2))) = "Error"
   Plus.Days = Plus.Days - 1
EndWhile

; compute the number of days between the two dates
Days = Day(Date.1) - Day(Date.2)
If Days < 0 Then Days = Days + Plus.Days EndIf
If Days = 1 Then
   Days = "1 Day"
Else
   Days = StrVal(Days) + " Days"
EndIf

; return the computed answer
Return Years + Months + Days

EndProc

;-------------------------------------------------------------------------------
; Age - may return "Error", otherwise should return an integer
; Purpose: Computes the number of whole years between 2 dates
;
; Acceptable calls to this procedure:
;       Age("1/31/42","")
;       Age(Today(),1/1/99)
;
;-------------------------------------------------------------------------------
Proc Age(Birth.Date,   ; Birthdate to calculate the age of
         From.Date)    ; Date to calculate from - default is Today() if
                       ; left blank ("") or a BlankDate() value is passed

; error checking
If Type(Birth.Date) <> "D" Then
   Birth.Date = DateVal(Birth.Date)
   If Birth.Date = "Error" then Return "Error" EndIf
EndIf

; sets the default value of today() as required
If IsBlank(From.Date) Then From.Date = Today() EndIf

; error checking
If Type(From.Date) <> "D" Then
   From.Date = DateVal(From.Date)
   If From.Date = "Error" then Return "Error" EndIf
EndIf

; don't want to under compute the person who is born on a leap day!
If Month(Birth.Date) = 2 AND Day(Birth.Date) = 29 Then
   Birth.Date = Birth.Date - 1
EndIf

; error checking again
If From.Date < Birth.Date Then Return "Error" EndIf

; return the number of whole years between the two dates
If Month(Birth.Date) < Month(From.Date) OR
   (Month(Birth.Date) = Month(From.Date) AND
    Day(Birth.Date) <= Day(From.Date)) Then
   Return Year(From.Date) - Year(Birth.Date)
Else
   Return Year(From.Date) - Year(Birth.Date) - 1
EndIf

EndProc

;-------------------------------------------------------------------------------
; Date.to.Julian - procedure may return "Error" if the date passed is an
;                  illegal date
; Purpose: Converts a Date to the Julian date format of YYYYDDD
;          Could easily be converted to return the date in the YYDDD format
;          by taking the substr(Year(Date.to.Convert,3,2)
;          I decided on the YYYYDDD format as we are approaching the next
;          century
;
; Acceptable calls to this procedure:
;       Date.to.Julian("1/31/42")
;       Date.to.Julian(Today())
;       Date.to.Julian(BlankDate())
;       Date.to.Julian(12/25/83)
;
;-------------------------------------------------------------------------------
Proc Date.to.Julian(Date.to.Convert)  ; Date to convert to Julian format
                                      ; If blank ("") or a BlankDate(),
                                      ; default is Today()

; error checking
If IsBlank(Date.to.Convert) Then Date.to.Convert = Today() EndIf
If Type(Date.to.Convert) <> "D" Then
   Date.to.Convert = DateVal(Date.to.Convert)
   If Date.to.Convert = "Error" Then Return "Error" EndIf
EndIf

; returns a number in the format of YYYYDDD
Return NumVal(StrVal(Year(Date.to.Convert))+SubStr(
       Format("W4,EZ",Date.to.Convert-
       DateVal("1/1/"+StrVal(Year(Date.to.Convert)))+1),2,3))

EndProc

;-------------------------------------------------------------------------------
; Julian.to.Date - procedure may return "Error" if the Julian date passed is
;                  an illegal Julian Date
; Purpose: Converts a Julian Date in the format YYDDD or YYYYDDD to a date
; format.
;
; Acceptable calls to this procedure:
;       Julian.to.Date("42124")
;       Julian.to.Date(1990165)
;       Julian.to.Date(2000001)
;       Julian.to.Date("2001010")
;
;-------------------------------------------------------------------------------
Proc Julian.to.Date(Julian.to.Convert)   ; Julian Date to convert to
                                         ; a regular date format
Private Julian.Date,    ; The computed date from the Julian date
        Julian.Days,    ; Number of days in the Julian date
        Julian.Year     ; 4 digit value of the Julian Year

; error checking
If Type(Julian.to.Convert) <> "N" AND Type(Julian.to.Convert) <> "S" Then
   Julian.to.Convert = NumVal(Julian.to.Convert)
   If Julian.to.Convert = "Error" Then Return "Error" EndIf
EndIf
If Len(Julian.to.Convert) <> 5 AND Len(Julian.to.Convert) <> 7 Then
   Return "Error"
EndIf

; compute based on the length of the parameter
If Len(Julian.to.Convert) = 5 Then
   Julian.Date = DateVal("1/1/"+SubStr(Julian.to.Convert,1,2))
   Julian.Days  = NumVal(SubStr(Julian.to.Convert,3,3))
Else
   Julian.Date = DateVal("1/1/"+SubStr(Julian.to.Convert,1,4))
   Julian.Days  = NumVal(SubStr(Julian.to.Convert,5,3))
EndIf

; error checking
If Julian.Date = "Error" or Julian.Days = "Error" then Return "Error" EndIf

; compute the actual date
Julian.Year  = Year(Julian.Date)
Julian.Date = Julian.Date + Julian.Days - 1

If Julian.Year <> Year(Julian.Date) Then
   ; if the computed number of days took us into another year, then
   ; the julian date was an illegal date
   Return "Error"
Else
   Return Julian.Date
EndIf

EndProc

;-------------------------------------------------------------------------------
; Convert.String.Date - converts a delimited or non delimited date to a
;                       Paradox date or returns "Error" if it could not
;                       convert it.
;
; The string passed as the first parameter to this procedure must evaluate
; to a legal date and be in one of the following formats:
;
;   Single delimited with a slash, dash, space, period, etc
;      mm dd yy
;      dd mm yy                     Note: all yy values may also be in the
;      yy mm dd                           format of yyyy
;      yy dd mm
;
;   No delimiters - all numbers
;      mmddyy
;      ddmmyy
;      yymmdd
;      yyddmm
;
; The second string passed must be either "MDY", "DMY", "YMD" or "YDM" in any
; case to indicate if the date is in a month day year, day month year,
; year month day or year day month orientation
;
; The third parameter to this procedure needs to indicate the delimiter for
; the date.  Any delimiter may be used and it may be more then a single
; character.  If there is no delimiter, then the parameter should be
; null ("").
;
; Acceptable calls to this procedure:
;       Convert.String.Date("102589","mDy","")
;       Convert.String.Date("10|25|89","MDY","|")
;       Convert.String.Date("92aaa10aaa25","ymd","aaa")
;       Convert.String.Date("1992,25,12","ydm",",")
;       Convert.String.Date("1862:10:13","ymd",":")
;
;-------------------------------------------------------------------------------
Proc Convert.String.Date(Date.String,    ; The date string to be converted
                         Date.Order,     ; The order of the month day
                                         ; year orientation
                         Delimiter)      ; The delimiter for the date string

Private Var.1,          ; variable assigned in a match function
        Var.2,          ;    "        "     "  "   "      "
        Var.2,          ;    "        "     "  "   "      "
        String.to.Match ; String used in a match function

; error checking
If IsBlank(Delimiter) Then
   If Len(Date.String) <> 6 AND Len(Date.String) <> 8 Then
      Return "Error"
   EndIf
   ; adjust the Date.String parameter so that a generic match
   ; function call can be used
   Switch
      Case Upper(Date.Order) = "MDY" OR Upper(Date.Order) = "DMY" :
         Date.String = SubStr(Date.String,1,2) + " " +
                       SubStr(Date.String,3,2) + " " +
                       SubStr(Date.String,5,4)
      Case Upper(Date.Order) = "YMD" OR Upper(Date.Order) = "YDM" :
         If Len(Date.String) = 6 Then
            Date.String = SubStr(Date.String,1,2) + " " +
                          SubStr(Date.String,3,2) + " " +
                          SubStr(Date.String,5,2)
         Else
            Date.String = SubStr(Date.String,1,4) + " " +
                          SubStr(Date.String,5,2) + " " +
                          SubStr(Date.String,7,2)
         EndIf
      OtherWise : Return "Error"
   EndSwitch
   Delimiter = " "
EndIf

; set up the string.to.match variable
Switch
   ; takes care or dates seperated with a special character
   ; NOTE: There may be other delimiters that need to be handled
   ; on an exception basis other then ".", "@" and "\"
   Case Delimiter = "." OR Delimiter = "@":
      String.to.Match = "..\""+Delimiter+"\"..\""+Delimiter+"\".."
   Case Delimiter = "\\" :
      String.to.Match = "..\"\\"+Delimiter+"\"..\"\\"+Delimiter+"\".."
   OtherWise :
      String.to.Match = ".."+Delimiter+".."+Delimiter+".."
EndSwitch

; try to get a match
If Match(Date.String,String.to.Match,Var.1,Var.2,Var.3) Then
   ; if successful, then attempt to return the date based on the
   ; orientation given.  if the date is not valid, then the
   ; returned value will be "Error"
   Switch
      Case Upper(Date.Order) = "MDY" :
         Return DateVal(Var.1+"/"+Var.2+"/"+Var.3)
      Case Upper(Date.Order) = "DMY" :
         Return DateVal(Var.2+"/"+Var.1+"/"+Var.3)
      Case Upper(Date.Order) = "YMD" :
         Return DateVal(Var.2+"/"+Var.3+"/"+Var.1)
      Case Upper(Date.Order) = "YDM" :
         Return DateVal(Var.3+"/"+Var.2+"/"+Var.1)
   EndSwitch
EndIf

; if the above match failed, then return "Error"
Return "Error"

EndProc

;-------------------------------------------------------------------------------
; Convert.Alpha.Date - converts a Alpha date to a Paradox date or returns
;                      "Error" if it could not convert it.
;
; The string passed to this procedure must evaluate to a legal date and
; be in one of the following formats:
;
;      Mon[th] dd, yy     Note: all yy values may also be in the
;      dd Mon[th], yy           format of yyyy
;      yy, Mon[th] dd
;      yy, dd Mon[th]
;
; Acceptable calls to this procedure:
;       Convert.Alpha.Date("January 12, 1863")
;       Convert.Alpha.Date("90, Jul 23")
;       Convert.Alpha.Date("87, 12 Aug")
;
;-------------------------------------------------------------------------------
Proc Convert.Alpha.Date(Alpha.Date)  ; The Alpha date to be converted

Private The.Day,        ; Procedure determined day
        The.Month,      ; Procedure determined month
        The.Year,       ; Procedure determined year
        Hold.It.Var     ; Holding variable to hold a value during
                        ; variable re-assignment as required

; attempt a match based on the possible formats
Switch
   Case Match(Alpha.Date,".. .., ..",The.Month,The.Day,The.Year) :
      If NumVal(The.Month) <> "Error" Then
         Hold.It.Var = The.Month
         The.Month   = The.Day
         The.Day     = Hold.It.Var
      EndIf
   Case Match(Alpha.Date,".., .. ..",The.Year,The.Month,The.Day) :
      If NumVal(The.Month) <> "Error" Then
         Hold.It.Var = The.Month
         The.Month   = The.Day
         The.Day     = Hold.It.Var
      EndIf
   OtherWise : Return "Error"
EndSwitch

; determine the number of the month
The.Month =
   (Search(SubStr(The.Month,1,3),"JanFebMarAprMayJunJulAugSepOctNovDec")+2)/3

; returned the evaluated date (may be "Error")
Return DateVal(StrVal(The.Month)+"/"+The.Day+"/"+The.Year)

EndProc

;-------------------------------------------------------------------------------
; Convert.Comma.Date - converts a comma delimited date to a Paradox date or
;                      returns "Error" if it could not convert it.
;
; The string passed as the first parameter to this procedure must evaluate
; to a legal date and be in one of the following formats:
;
;      mm dd, yy           Note: all yy values may also be in the
;      dd mm, yy                 format of yyyy
;      yy, mm dd
;      yy, dd mm
;
; The second string passed must be either "MD" or "DM" in any case to indicate
; if the date is in a month day or day month orientation
;
; Acceptable calls to this procedure:
;       Convert.Comma.Date("12 12, 1863","md")
;       Convert.Comma.Date("90, 07 23","MD")
;       Convert.Comma.Date("87, 12 09","dm")
;
;-------------------------------------------------------------------------------
Proc Convert.Comma.Date(Comma.Date,      ; The date to be converted
                        Month.Day.Order) ; The order of the month day
                                         ; orientation

Private The.Day,     ; the computed day of the date
        The.Month,   ; the computed month of the date
        The.Year     ; the computed year of the date

; attempt a match on the acceptable formats
Switch
   Case Upper(Month.Day.Order) = "MD" :
      Switch
         Case Match(Comma.Date,".. .., ..",The.Month,The.Day,The.Year) :
         Case Match(Comma.Date,".., .. ..",The.Year,The.Month,The.Day) :
         OtherWise : Return "Error"
      EndSwitch
   Case Upper(Month.Day.Order) = "DM" :
      Switch
         Case Match(Comma.Date,".. .., ..",The.Day,The.Month,The.Year) :
         Case Match(Comma.Date,".., .. ..",The.Year,The.Day,The.Month) :
         OtherWise : Return "Error"
      EndSwitch
   OtherWise : Return "Error"
EndSwitch

; return the evaluated date (may be "Error")
Return DateVal(The.Month+"/"+The.Day+"/"+The.Year)

EndProc

;-------------------------------------------------------------------------------
; Remove.Blanks
; Purpose: Removes blanks from any where in a string.
; Will not alter non alpha data types.
;
; This is included in case the user stumbles across a problem where a match
; leaves trailing and leading blanks in a string after the match function.
; It will also remove blanks from within the boundaries of a string.
;
; NOTE: Version 3.5 running in real mode will crash when attempting to
;       determine the NumVal(..) of a string that only contains blanks,
;       such as " " or "  ".  "" does not pose a problem.
;       If in doubt, before using NumVal(..), this procedure will remove
;       all blanks from a string and prevent a crash.
;
; Acceptable calls to this procedure:
;       Remove.Blanks("12 12, 1863")
;       Remove.Blanks(" this  is  a  test  of  removing  blanks  ")
;
;-------------------------------------------------------------------------------
Proc Remove.Blanks(Value.to.Trim)   ; string to be trimmed of all blanks

Private Before.Blank,   ; value determined in a match that occurred before
                        ; the first found blank
        After.Blank     ; value determined in a match that occurred after
                        ; the first found blank

; as long as a blank is found in the string, this will remove those blanks
While Match(Value.to.Trim,".. ..",Before.Blank,After.Blank)
   Value.to.Trim = Before.Blank + After.Blank
EndWhile
Return Value.to.Trim

EndProc