; Author: Dave Schlieder
;         Director of Programming Services
;         Shared Logic, Inc.
;         10919 Technology Place  Suite A
;         San Diego, Calif.  92127
;         (619)487-4545
;
; This script contains three procedures that will allow a user to browse
; the directory trees of any accessible drive and accessible directory.
;
; These procedures have NOT been tested against network directories where
; the user might be restricted to access rights to directories.
;
; These procedures were created due to a query on the PDOXDOS forum of 
; CompuServe on 10-07-93 asking if there was a way to emulate the directory 
; tree browser in Paradox 4.5.
;
; You are free to use these procedures in your own applications.  My only 
; request is that you acknowledge me as the original author of these procedures.
;
; You can simply play this script to demonstrate it's capabilities.  If you 
; desire to add them to your libraries, you'll need to add the appropriate
; WRITELIB statements to add these to your libraries.
;
; There is one thing these procedures don't do that Paradox's browser does.
; You can not press keys to walk the tree, looking for directories based on 
; those key presses.  I suppose this could be done by trapping all of the key
; keys and evaluating those keys against the array, but this would take a fair
; amount of coding for this.

If Version() < 4.5 Then 
   Beep Sleep 250 Beep
   Return "Version 4.5 or higher of Paradox required!"
EndIf

;
;                                                      Dialog.DirectoryBrowser()
;
; Author: Dave Schlieder
;         Director of Programming Services
;         Shared Logic, Inc.
;         10919 Technology Place  Suite A
;         San Diego, Calif.  92127
;         (619)487-4545
; Purpose: A dialog to allow a user to walk the directory tree similar to
;          the browser in Paradox for changing directories or setting the
;          users' private directory.
;          Positioning could be altered to be variables to place
;          the dialog always centered based on the current video mode
Proc Dialog.DirectoryBrowser()
Private Tree.Array,
        ;Dir.Accept,
        Tree.Select,
        Button.Var,
        Color.Bag,
        BlackChisel,
        WhiteChisel

; The variable Dir.Accept is not declared as a private variable in this 
; procedure so that the calling procedure can evaluate the directory the
; user choose if this procedure returns true.

; Initialize the array that will show the tree.
Array Tree.Array[1]
Tree.Array[1] = ""
; Initialize the starting directory to the current directory.
; If desired, this could become a parameter to this procedure so that
; when it's called, it wouldn't be dependent on the current directory
; but whatever directory the programmer wanted the user to start at.
Dir.Accept = Directory()

; Set the colors for the frame so that the background color will match
; the background based on the current dialog box colors.
GetColors To Color.Bag
BlackChisel = INT(Color.Bag[1036]/16)*16
WhiteChisel = INT(Color.Bag[1036]/16)*16+15

; The directory tree browser dialog box
ShowDialog "Directory Tree Browser"
   Proc "Wait.Tree"
      TRIGGER "ACCEPT","SELECT","ARRIVE","UPDATE"
      KEY 13
   @2,14 Height 19 Width 53

   Frame Double From 0,1 To 3,49
   PaintCanvas Attribute BlackChisel 0,1,0,49
   PaintCanvas Attribute BlackChisel 0,1,3,1
   PaintCanvas Attribute WhiteChisel 3,2,3,49
   PaintCanvas Attribute WhiteChisel 0,49,3,49

   Frame Double From 4,1 To 16,39
   PaintCanvas Attribute BlackChisel 4,1,4,39
   PaintCanvas Attribute BlackChisel 4,1,16,1
   PaintCanvas Attribute WhiteChisel 16,2,16,39
   PaintCanvas Attribute WhiteChisel 4,39,16,39

   Accept @2,3 Width 45
      "A60"
      Picture "*!"
      Tag "Dir.Accept"
      To Dir.Accept

   Label @1,3
      "~D~irectory:"
      For "Dir.Accept"

   PickArray @6,3 Height 10 Width 34
      Columns 1
      Tree.Array
      Tag "Tree.Array"
      To Tree.Select

   Label @5,3
      "Directory ~t~ree"
      For "Tree.Array"

   PushButton @5,40 Width 10
      "~O~K"
      Ok
      Default
      Value True
      Tag "OK.Button"
      To Button.Var

   PushButton @7,40 Width 10
      "~S~elect"
      Value Wait.Tree("SELECT","","","")
      Tag "Select.Button"
      To Button.Var

   PushButton @9,40 Width 10
      "~C~ancel"
      Cancel
      Value False
      Tag "Cancel.Button"
      To Button.Var
EndDialog

Return RetVal

EndProc
;
;                                                                    Wait.Tree()
;
; Author: Dave Schlieder
;         Director of Programming Services
;         Shared Logic, Inc.
;         10919 Technology Place  Suite A
;         San Diego, Calif.  92127
;         (619)487-4545
; Purpose: Dialog procedure for a directory tree browser
Proc Wait.Tree(TriggerType,TagValue,EventValue,ElementValue)
Private Element,
        ReturnValue,
        SkipDir

Switch

   ; When updating the directory name in the accept box, the
   ; directory tree array is refreshed based on the new value
   ; We save the return value in a variable because RefreshControl
   ; alters the state of RetVal
   Case TriggerType = "UPDATE" And 
        TagValue = "Dir.Accept" And
        Dir.Accept <> ControlValue("Dir.Accept") :
      ReturnValue = Tree.Build(ControlValue("Dir.Accept"))
      RefreshControl "Tree.Array"
      Return ReturnValue

   ; If the user presses the enter key while in the tree, instead of
   ; accepting the dialog box, this is treated as if the user pressed
   ; the "Select" button.  This is an attempt to mimic the directory
   ; tree browser in native Paradox.  Paradox's browser changes the
   ; default button to the "Select" button whenever the user enters
   ; the tree area of the dialog box.  In a ShowDialog, you can't change 
   ; the default button on the fly without actually calling a whole new
   ; dialog procedure.  This seems like too much overhead just to change the
   ; default button.
   Case TriggerType = "EVENT" And
        EventValue["KEYCODE"] = 13 And
        TagValue = "Tree.Array" :
        SelectControl "Select.Button"
        PushKeys 13
        Return False

   ; Whenever the user attempts to accept the dialog box, we only allow
   ; it to be accepted if the current directory is a valid one.
   Case TriggerType = "ACCEPT" :
      Switch
         Case DirExists(ControlValue("Dir.Accept")) = 1 :
            Return True
         Case DirExists(ControlValue("Dir.Accept")) = 0 :
            Beep
            Message "This directory does not exist."
            SelectControl "Dir.Accept"
            Return False
         OtherWise :
            Beep
            Message "This is in invalid directory name."
            SelectControl "Dir.Accept"
            Return False
      EndSwitch

   ; Here the user is attempting to make a selection from the tree.
   ; (The select button also calls this procedure with a trigger type
   ;  of SELECT with all other values passed to this procedure as null)
   Case TriggerType = "SELECT" :
      ; Here we make sure the variable Tree.Select is current by 
      ; getting it's control value
      Tree.Select = ControlValue("Tree.Array")
      Switch
         ; Any time the user selects Drives or the tree has never been filled
         ; out, we blank out the Dir.Accept variable
         Case Tree.Array[Tree.Select] = "Drives" Or
              Tree.Array[Tree.Select] = "" :
            Dir.Accept = ""
         ; Here the user has selected a drive letter.  We trim out all
         ; of the garbage characters to set Dir.Accept to the drive
         ; letter selected.
         Case Search(":",Tree.Array[Tree.Select]) <> 0 :
            Dir.Accept = XTRIM(Tree.Array[Tree.Select]," \\") + "\\"
         ; Here the user has actually selected a directory and we need
         ; to rebuild the Dir.Accept variable
         OtherWise :
            Dir.Accept = ""
            ; we don't want to include in the list any directory that
            ; is at the same level as the one the user has elected and this
            ; we determine based on the first occurrence of the "" character
            SkipDir = Search("",Tree.Array[Tree.Select])
            ; Element 2 of the array is always the drive letter and we
            ; start with it and add in each directory out of the array
            For Element From 2 To Tree.Select 
               ; Here we skip directories that are at the same level as
               ; the one selected from the list
               If Search("",Tree.Array[Element]) = SkipDir And
                  Element <> Tree.Select Then
                  Loop
               EndIf
               Dir.Accept = 
               Dir.Accept + XTRIM(Tree.Array[Element]," \\") + "\\"
            EndFor
      EndSwitch
      ; Then we rebuild the tree
      Tree.Build(Dir.Accept)
      RefreshDialog
      If TagValue <> "Tree.Array" Then
         SelectControl "Tree.Array"
      EndIf

   ; In this case, the user has entered the tree but it hasn't been
   ; filled out at this point so we need to initialize the tree
   Case TriggerType = "ARRIVE" And
        TagValue = "Tree.Array" And
        Tree.Array[1] = "" :
      Tree.Build(Dir.Accept)
      RefreshControl TagValue

EndSwitch

Return True

EndProc
;
;                                                                   Tree.Build()
;
; Author: Dave Schlieder
;         Director of Programming Services
;         Shared Logic, Inc.
;         10919 Technology Place  Suite A
;         San Diego, Calif.  92127
;         (619)487-4545
; Purpose: Procedure to build an array that is used to show a directory tree.
Proc Tree.Build(Root.Dir)
Private Tree.DynArray,
        Drive.Array,
        ElementPointer,
        Tree.Array.Size,
        Current.Echo.State,
        Valid.Dir,
        LastChoice,
        BeginDir,
        EndDir

; The first element of the array will always be "Drives"
; We start off filling out a dynamic array then transfer that information 
; into a fixed array so it will be sorted correctly in the dialog box
; By using a temporary dynamic array, we can determine the size of the fixed
; array after we fill in the dynamic array
DynArray Tree.DynArray[]
Tree.DynArray[1] = "Drives"

If Root.Dir = "" Then
   ; If no directory is specified, then we create a list of drives for
   ; the user to select from using SysInfo Drives
   SysInfo Drives To Drive.Array
   ; Set a pointer for the elements as we are going to assign the
   ; elements in reverse.
   ElementPointer = DynArraySize(Drive.Array) + 1
   ; Define the size of the fixed array for the dialog box
   Array Tree.Array[ElementPointer]
   ; Keep it's size in memory
   Tree.Array.Size = ElementPointer
   ; Now, this seems to work, keeping the drive letters in alpha order,
   ; but this may not always work because stepping through a dynamic array
   ; in a foreach loop is not supposed to have a determinable order.
   ; But, it is working and if need be, this array could be sorted, but
   ; that would add additional overhead to this procedure.
   ForEach Element In Drive.Array
      Tree.DynArray[ElementPointer] = Element
      ElementPointer = ElementPointer - 1
   EndForEach
   ; Now assign the elements in the dynamic array to the fixed array 
   ; based on the numerical value of the index.  This only works because
   ; we created the dynamic array with incremental indices that we can
   ; convert to a numeric for the fixed array.
   ForEach Element In Tree.DynArray
      Tree.Array[NumVal(Element)] = Tree.DynArray[Element]
   EndForEach
   ; Now, exclude the "Drives" element 1 and the last drive letter in this
   ; loop but all others bet a " " appended to the drive letter
   For Element From 2 To Tree.Array.Size-1
      Tree.Array[Element] = " " + Tree.Array[Element]
   EndFor
   ; The last element finishes off the tree
   Tree.Array[Element] = " " + Tree.Array[Element]
   Tree.Select = 1
   Return True
Else
   ; Here we have a directory name and we need to build the tree based on 
   ; that name.  However, before we continue, we could have an invalid 
   ; name.  Let's validate it by calling the dialog procedure as if the
   ; user was attempting to accept the current value.
   Wait.Tree("ACCEPT","","","")
   If Not RetVal Then Return RetVal EndIf
   ; For some of the further processing, we want to insure that
   ; the current path ends in a backslash.
   If SubStr(Root.Dir,Len(Root.Dir),1) <> "\\" Then
      Root.Dir = Root.Dir + "\\"
   EndIf
   ; Save the current echo state so we can restore it as needed
   SysInfo To Current.Echo.State
   ; We want echo off if it's not
   If Current.Echo.State["ECHO"] <> "OFF" Then
      Echo Off
   EndIf
   ; Open up the MiniEdit menu and select {Open}.
   ; This is available in almost all modes so it is fairly safe to
   ; assume that we can do this.
   ; We put in a wild card of "*." to eliminate files that have extensions.
   ; (This will include directories with periods in them however which is
   ;  what we want.)
   MiniEdit {Open} TypeIn Root.Dir + "*." Tab
   ; Just in case we have an inaccessible directory on somehow an invalid 
   ; directory name got past the above checking.
   Valid.Dir = Window()
   If Valid.Dir = "" Then
      ; Let's make sure we have the full directory name for the tree
      ; We don't want to start off with "C:\\PDOX45\\..\\"
      ; FullFileName will return the directory name without the .. in it
      Root.Dir = FullFileName(Root.Dir)
      ; keep track of which level in the tree we're on and the indent level
      ; the drive letter is indented 1 space and each directory is indented
      ; an additional 2 spaces.
      ElementPointer = 1
      Indent = 1
      ; Break the directory up into it's components with this match loop.
      ; This is why we insured the trailing backslash before hand.
      While Match(Root.Dir,"..\\..",Element,Root.Dir)
         ; The drive letter gets a backslash
         If Search(":",Element) = 2 Then
            Element = Element + "\\"
         EndIf
         ; Up the element counter
         ElementPointer = ElementPointer + 1
         ; Fill in the array element with the proper indent level
         ; and graphics characters
         Tree.DynArray[ElementPointer] = Spaces(Indent) + "" + Upper(Element)
         ; Reset the indent level
         Indent = Indent + 2
      EndWhile
      ; Set the variable Tree.Select to the current directory level so the
      ; selected item in the tree is the current level
      Tree.Select = ElementPointer
      ; So we know when we have completed finding all of the sub-directories
      ; of the current directory.
      LastChoice = ""
      ; We don't need to start off with this directory level so we attempt
      ; to skip past it
      If MenuChoice() = "..\\" Then Down EndIf
      ; As long as we have a menu choice!
      While MenuChoice() <> ""
         ; If we pick the same one up again, we must have hit the last choice
         ; and we need to exit this loop.
         If MenuChoice() = LastChoice Then QuitLoop EndIf
         ; If the current choice is not a directory name (as it could be a file
         ; without an extension) or the current choice is "..\\" we attempt
         ; to go past it but we need to keep track of the last one found so
         ; we can exit this loop
         If Search("\\",MenuChoice()) = 0 Or
            MenuChoice() = "..\\" Then 
            Down
            LastChoice = MenuChoice()
            Loop 
         EndIf
         ; Up the pointer
         ElementPointer = ElementPointer + 1
         ; Add in the new directory adding in the graphics and trimming
         ; of the trailing backslash
         Tree.DynArray[ElementPointer] = 
            Spaces(Indent) + "" + 
            Upper(SubStr(MenuChoice(),1,Len(MenuChoice())-1))
         ; Keep track of the last one!
         LastChoice = MenuChoice()
         ; See if there's another one.
         Down
      EndWhile
      ; If the current dynamic array element has the "" character in it,
      ; we need to change it to the "" character to close off the bottom
      ; of the tree.
      If Match(Tree.DynArray[ElementPointer],"....",BeginDir,EndDir) Then
         Tree.DynArray[ElementPointer] = BeginDir + "" + EndDir
      EndIf
      ; Create our fixed array of the proper size
      Array Tree.Array[ElementPointer]
      ; Transfer the elements from the dynamic array to the fixed array
      ForEach Element In Tree.DynArray
         Tree.Array[NumVal(Element)] = Tree.DynArray[Element]
      EndForEach
   EndIf
   ; Clear the MiniEdit {Open} dialog box
   Menu Esc
EndIf

; If echo wasn't off, and we turned it off, set it back like it was
If Current.Echo.State["ECHO"] <> "OFF" Then
   Switch 
      Case Current.Echo.State["ECHO"] = "NORMAL" : Echo Normal
      Case Current.Echo.State["ECHO"] = "SLOW"   : Echo Slow  
      Case Current.Echo.State["ECHO"] = "FAST"   : Echo Fast
   EndSwitch
EndIf

; A little error check here based on the attempt to open the 
; MiniEdit {Open} dialog box
If Valid.Dir <> "" Then
   Beep
   Message Valid.Dir
   ; if the error wasn't a drive that wasn't ready,
   ; then refresh the tree because we don't have a valid
   ; directory to work with.
   If Valid.Dir <> "Drive not ready" Then
      Array Tree.Array[1]
      Tree.Array[1] = "Drives"
   EndIf
   Return False
EndIf

Return True

EndProc

; Call the procedure defined in memory above.
Dialog.DirectoryBrowser()
