/*
** dbstruct - Program to display the structure of a valid Paradox database
**            file and create a PAL query image for use in scripts
**
** Syntax: dbstruct <filename> [/f]
**
** .DB extension is optional, any others are ignored.
**
** Use optional /f switch to create a file named <filename>.STR
** listing the structure of the Paradox database and a PAL query
** image that can be edited.
**
** 4/25/90   James Louie
*/

#include <ctype.h>
#include <malloc.h>
#include <stdio.h>
#include <string.h>
#include "pxengine.h"

#define MAXFIELDS 255
#define NAMESIZE 26
#define TYPESIZE 6
#define PAGEWIDTH 72

void errorexit( int errcode );

int main( int argc, char *argv[] )
{
	TABLEHANDLE tblHandle;
	RECORDNUMBER nRecs;
	struct fieldinfo
	{
		char fldName[NAMESIZE];
		char fieldType[TYPESIZE];
	};
	struct fieldinfo *fldinfo, *firstfld;
	char drive[3], dir[30], fname[9], ext[4], linebuf[PAGEWIDTH + 1];
	int i, j, nFlds, nKeyFlds, pxErr, fout=0;
	FILE *fp;

	if (! ((argc == 2) || (argc == 3)) )
	{
		printf("\nDBSTRUCT.EXE: Program to display the structure of a valid Paradox database.\n\n");
		printf("Syntax: dbstruct <filename> [/f]\n\n");
		printf("        .DB extension is optional, any others are ignored.\n\n");
		printf("        Use optional /f switch to create a file named <filename>.STR\n");
		printf("        listing the structure of the Paradox database and a PAL query\n");
		printf("        image that can be edited.\n\n");
		printf("        Written by James Louie (CIS 70441,127) - April 1990\n");
		exit();
	}

	/* Open the output file if needed */
	if (strcmpi(argv[2],"/f\0") == 0)
	{
		fout = 1;
		_splitpath(argv[1], drive, dir, fname, ext);
		if ((fp = fopen(strcat(fname,".str"),"w")) == NULL)
		{
			printf("Error: Could not open output file.\n");
			exit();
		}
	}

	/* Open the Paradox table */
	PXInit();
	if ((pxErr = PXTblOpen(argv[1], &tblHandle, 0, 0)) != PXSUCCESS)
		errorexit(pxErr);
	
	/* Determine how many records are in table */
	if ((pxErr = PXTblNRecs(tblHandle, &nRecs)) != PXSUCCESS)
		errorexit(pxErr);

	/* Determine number of fields in table */
	if ((pxErr = PXRecNFlds(tblHandle, &nFlds)) != PXSUCCESS)
		errorexit(pxErr);

	/* Determine how many fields are in index */
	if ((pxErr = PXKeyNFlds(tblHandle, &nKeyFlds)) != PXSUCCESS)
		errorexit(pxErr);
	
	/* Allocate memory to store info for fields */
	if (! (fldinfo = (struct fieldinfo *) malloc(nFlds * sizeof(*fldinfo))))
	{
		printf("Error: Could not allocate memory.\n");
		exit();
	}
	firstfld = fldinfo;

	if (fout)
	{
		fprintf(fp,"; Database file: %s\n", argv[1]);
		fprintf(fp,"; Number of records: %ld\n;\n", nRecs);
		if (nKeyFlds)
			fprintf(fp,"; Field types ending with '*' indicate key fields.\n;\n");
		fprintf(fp,"; Field #    Type     Name\n");
		fprintf(fp,";          \n");
	}
	else
	{
		printf("\nDatabase file: %s\n", argv[1]);
		printf("Number of records: %ld\n\n", nRecs);
		if (nKeyFlds)
			printf("Field types ending with '*' indicate key fields.\n\n");
		printf("Field #    Type     Name\n");
		printf("         \n");
	}

	/* Get field names and field types, then print them */
	for (i=1; i <= nFlds; i++, fldinfo++)
		if (((pxErr = PXFldName(tblHandle, i, NAMESIZE, fldinfo->fldName)) == PXSUCCESS) &&
			 ((pxErr = PXFldType(tblHandle, i, TYPESIZE, fldinfo->fieldType)) == PXSUCCESS))
		{
  			if (i <= nKeyFlds)
				strcat(fldinfo->fieldType, "*");

			if (fout)
				fprintf(fp,"; %4d       %-5s    %s\n", i, fldinfo->fieldType, fldinfo->fldName);
			else
				printf("%4d       %-5s    %s\n", i, fldinfo->fieldType, fldinfo->fldName);
		}
		else
			errorexit(pxErr);

	/* Print a PAL query image */
	if (fout)
	{
		fprintf(fp,"\nQuery\n\n");
		_splitpath(argv[1], drive, dir, fname, ext);
		strlwr(fname);
		fname[0] = _toupper(fname[0]);

		i = 1;
		while (i <= nFlds)
		{
			strcpy(linebuf, "   ");
			strcat(linebuf, fname);
			strcat(linebuf, " | ");
			
			while ((i <= nFlds) && ((strlen(linebuf) + strlen(firstfld->fldName) + 7) <= PAGEWIDTH))
			{
				strcat(linebuf, firstfld->fldName);
				if (strlen(firstfld->fldName) < 5)
					for (j=1; j <= (5 - strlen(firstfld->fldName)); j++)
						strcat(linebuf, " ");
				strcat(linebuf, " | ");
				firstfld++;
				i++;
			}
			
			fprintf(fp,"%s\n",linebuf);
			for (j=0; j < strlen(linebuf) - 1; j++)
				if (linebuf[j] != '|')
					linebuf[j] = ' ';
	  		fprintf(fp,"%s\n%s\n%s\n\n",linebuf,linebuf,linebuf);
		}
		
		fprintf(fp,"Endquery");
		strupr(fname);
		printf("\nFile %s.STR containing Paradox structure information has been created.\n",fname);
	}

	/* Close down */
	PXTblClose(tblHandle);
	PXExit();
	if (fout)
		fclose(fp);
	exit();
}

/*
** errorexit - Function to print the Paradox Engine error message and exit
*/

void errorexit( int errcode )
{
	printf("%s\n", PXErrMsg(errcode));
	PXExit();
	exit(errcode);
}
