/****************************************************************************
*                                                                           
* FILE: TEST.CPP
*
* DESCRIPTION:  This is the main program file demonstrating the new classes
*               TCursorDialog and TFieldInputLine
*
****************************************************************************/
//
// what classes do we use from turbo vision
//
#define Uses_TApplication
#define Uses_TRect
#define Uses_TMenuBar
#define Uses_TMenu
#define Uses_TMenuItem
#define Uses_TKeys
#define Uses_TDeskTop
#define Uses_MsgBox
#define Uses_TLabel
#define Uses_TButton

#include  <tv.h>

//
// define commands for the buttons on the example Dialog Box
//
const cmFirstRec  = 200;
const cmNextRec   = 201;
const cmPrevRec   = 202;
const cmLastRec   = 203;
const cmDeleteRec = 204;
const cmNewRec    = 205;

//
// include our special classes
//
#include  "tfldipl.h" // TFieldInputLine
#include  "tcurdlg.h" // TCursorDialog

//
// include the database framework class .H files
//
#include  <bengine.h>
#include  <bdatabas.h>
#include  <bcursor.h>
#include  <brecord.h>

//
// we need to define the menu command for the open menu selection
//
#define cmOpenCursor  100

////////////////////////////////////////////////////////////////
//
// the constants for the address table
//
////////////////////////////////////////////////////////////////
//
// table name
//
const char        *ADDRESS_NAME       = "ADDRESS";

//
// num fields and num key fields
//
const int         ADDRESS_NUM_FIELDS  = 7;
const int         ADDRESS_NUM_KEYS    = 2;

//
// field numbers
//
const FIELDNUMBER ADDRESS_LAST_NAME   = 1;
const FIELDNUMBER ADDRESS_FIRST_NAME  = 2;
const FIELDNUMBER ADDRESS_STREET      = 3;
const FIELDNUMBER ADDRESS_CITY        = 4;
const FIELDNUMBER ADDRESS_STATE       = 5;
const FIELDNUMBER ADDRESS_ZIP         = 6;
const FIELDNUMBER ADDRESS_TELEPHONE   = 7;

//
// field lengths
//
const int         LAST_NAME_LEN       = 30;
const int         FIRST_NAME_LEN      = 30;
const int         STREET_LEN          = 30;
const int         CITY_LEN            = 30;
const int         STATE_LEN           = 2;
const int         ZIP_LEN             = 10;
const int         TELEPHONE_LEN       = 14;

//
// the array of field descriptors for the table
//
const FieldDesc ADDRESS_FIELD_DESCRIPTORS[ADDRESS_NUM_FIELDS] = 
{
  {ADDRESS_LAST_NAME,   "Last Name",  fldChar, fldstNone, LAST_NAME_LEN},
  {ADDRESS_FIRST_NAME,  "First Name", fldChar, fldstNone, FIRST_NAME_LEN},
  {ADDRESS_STREET,      "Street",     fldChar, fldstNone, STREET_LEN},
  {ADDRESS_CITY,        "City",       fldChar, fldstNone, CITY_LEN},
  {ADDRESS_STATE,       "State",      fldChar, fldstNone, STATE_LEN},
  {ADDRESS_ZIP,         "Zip",        fldChar, fldstNone, ZIP_LEN},
  {ADDRESS_TELEPHONE,   "Telephone",  fldChar, fldstNone, TELEPHONE_LEN}
};


/****************************************************************************
*
*        CLASS: TTestApp
*
*     INHERITS: TApplication
*
*  DESCRIPTION: This is out test application class.
*
*
*       THE MEMBER FUNCTIONS FOR THIS CLASS CAN BE FOUND IN THIS FILE.
*
*
****************************************************************************/
class TTestApp : public TApplication
{
private:

public:
  //
  // pointer to the BEngine and the BDatabase used in the program
  //
  static BDatabase *theDatabase;
  static BEngine   *theEngine;

  //
  // constructor
  //
  TTestApp();
  //
  // destructor
  //
  ~TTestApp();

  //
  // the function to create the applications menu
  //
  static TMenuBar *initMenuBar(TRect r);

  //
  // the event handler for the program
  //
  virtual void handleEvent(TEvent &event);

  //
  // the function that shows the cursor dialog box
  //
  void showCursorDialog();

  //
  // a function to open the address cursor
  //
  BCursor *openTable();
};

//
// here are the static members of TTestApp
//
BDatabase  *TTestApp::theDatabase;
BEngine    *TTestApp::theEngine;


/****************************************************************************
*
*        CLASS: TExampleDialog
*
*     INHERITS: TCursorDialog
*
*  DESCRIPTION: This class is a TCursorDialog type of dialog.  It 
*               demonstrates how to use a TCursorDialog.
*              
****************************************************************************/
class TExampleDialog : public TCursorDialog
{
public:
  //
  // constructor
  //
  TExampleDialog(BCursor *aCursor);

  //
  // event handler
  //
  virtual void handleEvent(TEvent &event);
};


/****************************************************************************
*          
* CLASS: TTestApp
*                                                                
* MEMBER FUNCTION: TTestApp (constructor)
*
* DESCRIPTION:  This function will create the applications menu, status line,
*               and desk top.  We have also added the construction of the
*               database frameworks BEngine and BDatabase objects.
*               
* RETURNS: NONE
*
****************************************************************************/
TTestApp::TTestApp(): 
  TProgInit(TTestApp::initStatusLine, TTestApp::initMenuBar, 
    TTestApp::initDeskTop)
{
  //
  // we need to construct the BEngine object for the program
  //
  theEngine = new BEngine(pxLocal); // this can be changed to pxNet

  //
  // construct the database also
  //
  theDatabase = new BDatabase(theEngine);
}

/****************************************************************************
*          
* CLASS: TTestApp
*                                                                
* MEMBER FUNCTION: ~TTestApp (destructor)
*
* DESCRIPTION:  This is the function that is called when the application 
*               ends. Since we constructed the engine and the database during
*               the construction of TTestApp, this is a good function to 
*               destroy the objects.
*               
* RETURNS: NONE
*
****************************************************************************/
TTestApp::~TTestApp()
{
  //
  // since the application is done,  we need to destroy the engine and the
  // database objects that were created in the classes constructor
  //
  delete theDatabase;
  delete theEngine;
}

/****************************************************************************
*          
* CLASS: TTestApp
*                                                                
* MEMBER FUNCTION:  initMenuBar
*
* DESCRIPTION:  This function creates the programs menu bar.  The menu
*               has a single item on it (Open) that allows the user to open
*               the dialog box
*               
* RETURNS: the programs menu bar
*
****************************************************************************/
TMenuBar *TTestApp::initMenuBar(TRect r)
{
  //
  // set the bottom line of the menu to 1 line below the top of the menu
  //
  r.b.y = r.a.y + 1;

  //
  // create a menu item to open the cursor (table)
  //
  return new TMenuBar(r, new TMenu( *new TMenuItem("~O~pen", cmOpenCursor, 
    kbNoKey, hcNoContext, NULL)));
}

/****************************************************************************
*          
* CLASS: TTestApp
*                                                                
* MEMBER FUNCTION: handleEvent
*
* DESCRIPTION:  this function responds to the events that happen in the 
*               application.  It will check to see if the user has pressed
*               the Open menu selection and it will call the 
*               showCursorDialog() function.
*               
* RETURNS: NONE
*
****************************************************************************/
void TTestApp::handleEvent(TEvent &event)
{
  //
  // call the base class TApplication, to handle the event first
  //
  TApplication::handleEvent(event);

  //
  // did the user press the Open menu selection
  //
  if(event.what == evCommand && event.message.command == cmOpenCursor)
  {
    //
    // call the function that puts up the tables dialog box
    //
    showCursorDialog();

    clearEvent(event);
  }
}

/****************************************************************************
*          
* CLASS: TTestApp
*                                                                
* MEMBER FUNCTION: showCursorDialog
*
* DESCRIPTION:  This function constructs and displays our example cursor
*               dialog.
*               
* RETURNS: NONE
*
****************************************************************************/
void TTestApp::showCursorDialog()
{
  TExampleDialog  *ourExampleDialog;
  BCursor         *addressCursor;

  //
  // open the address table
  //
  addressCursor = openTable();

  //
  // did we get a cursor
  //
  if(addressCursor == NULL)
  {
    return;
  }

  //
  // check the cursor's lastError member to see if it did not open
  //
  if(addressCursor->lastError != PXSUCCESS)
  {
    //
    // put up a message box saying why the table did not open
    //
    messageBox(mfError | mfOKButton, "Could not open table %s because: %s",
      ADDRESS_NAME, theEngine->getErrorMessage(addressCursor->lastError));
  }
  else
  {
    //
    // go to the first record in the address table
    //
    addressCursor->gotoRec(1);
    //
    // get the field values from the current record and place them into the
    // generic record buffer
    //
    addressCursor->getRecord();

    //
    // construct our example dialog
    //
    ourExampleDialog = new TExampleDialog(addressCursor);

    //
    // exec the dialog into view
    //
    deskTop->execView(ourExampleDialog);

    //
    // destroy the dialog
    //
    destroy(ourExampleDialog);
  }

  //
  // delete the cursor object
  //
  delete addressCursor;
}

/****************************************************************************
*          
* CLASS: TTestApp
*                                                                
* MEMBER FUNCTION: openTable
*
* DESCRIPTION:  This function will open or create and open the address
*               table.
*               
* RETURNS:  A pointer to the address table's cursor or NULL if it could
*           not be opened.
*
****************************************************************************/
BCursor *TTestApp::openTable()
{
  Retcode result;

  //
  // see if the address table exists
  //
  if(!theDatabase->tableExists(ADDRESS_NAME))
  {
    //
    // create the table
    //
    result = theDatabase->createTable(ADDRESS_NAME, ADDRESS_NUM_FIELDS, 
      ADDRESS_FIELD_DESCRIPTORS);

    //
    // did it get created
    //
    if(result != PXSUCCESS)
    {
      //
      // display an error box and return
      //
      messageBox(mfError | mfOKButton,
        "Could not create the %s table because: %s", ADDRESS_NAME, 
        theEngine->getErrorMessage(result));

      return NULL;
    }
    else
    {
      //
      // create the primary key
      //
      result = theDatabase->createPIndex(ADDRESS_NAME, ADDRESS_NUM_KEYS);

      //
      // did it get created
      //
      if(result != PXSUCCESS)
      {
        //
        // display an error box and return
        //
        messageBox(mfError | mfOKButton,
          "Could not create table's %s primary key because: %s", 
          ADDRESS_NAME, theEngine->getErrorMessage(result));

        return NULL;
      }
    }
  }

  //
  // open the table
  //
  return(new BCursor(theDatabase, ADDRESS_NAME, 0, FALSE));
}


/****************************************************************************
*          
* CLASS: TExampleDialog
*                                                                
* MEMBER FUNCTION: TExampleDialog (constructor)
*
* DESCRIPTION:  This is the constructor of our example TCursorDialog.  Since
*               this class is based on a TCursorDialog we must
*               call the constructor for this class.
*               
* RETURNS: NONE
*
****************************************************************************/
TExampleDialog::TExampleDialog(BCursor *aCursor):
  TCursorDialog(TRect(10, 2, 70, 20), "Example TCursorDialog", aCursor),
  TWindowInit(TExampleDialog::initFrame)
{
  TView *newLine;

  //
  // place a TFieldInputLine in for the Last Name field.  Also label it.
  //
  newLine = new TFieldInputLine(TRect(16, 2, 49, 3), LAST_NAME_LEN + 1,
    aCursor, ADDRESS_LAST_NAME);
  insert(newLine);
  insert(new TLabel(TRect(2, 2, 13, 3), "~L~ast Name:", newLine));

  //
  // place a TFieldInputLine in for the First Name field.  Also label it.
  //
  newLine = new TFieldInputLine(TRect(16, 3, 49, 4), FIRST_NAME_LEN + 1,
    aCursor, ADDRESS_FIRST_NAME);
  insert(newLine);
  insert(new TLabel(TRect(2, 3, 15, 4), "~F~ast Name:", newLine));

  //
  // place a TFieldInputLine in for the Street field.  Also label it.
  //
  newLine = new TFieldInputLine(TRect(16, 5, 49, 6), STREET_LEN + 1,
    aCursor, ADDRESS_STREET);
  insert(newLine);
  insert(new TLabel(TRect(2, 5, 15, 6), "~S~treet:", newLine));

  //
  // place a TFieldInputLine in for the City field.  Also label it.
  //
  newLine = new TFieldInputLine(TRect(16, 6, 49, 7), CITY_LEN + 1,
    aCursor, ADDRESS_CITY);
  insert(newLine);
  insert(new TLabel(TRect(2, 6, 15, 7), "~C~ity:", newLine));

  //
  // place a TFieldInputLine in for the State field.  Also label it.
  //
  newLine = new TFieldInputLine(TRect(16, 7, 21, 8), STATE_LEN + 1,
    aCursor, ADDRESS_STATE);
  insert(newLine);
  insert(new TLabel(TRect(2, 7, 15, 8), "S~t~ate:", newLine));

  //
  // place a TFieldInputLine in for the ZIP field.  Also label it.
  //
  newLine = new TFieldInputLine(TRect(16, 8, 30, 9), ZIP_LEN + 1,
    aCursor, ADDRESS_ZIP);
  insert(newLine);
  insert(new TLabel(TRect(2, 8, 15, 9), "~Z~ip:", newLine));

  //
  // place a TFieldInputLine in for the Telephone field.  Also label it.
  //
  newLine = new TFieldInputLine(TRect(16, 10, 35, 11), TELEPHONE_LEN + 1,
    aCursor, ADDRESS_TELEPHONE);
  insert(newLine);
  insert(new TLabel(TRect(2, 10, 15, 11), "Tele~p~hone:", newLine));

  //
  // place the movement buttons
  //
  insert(new TButton(TRect(2,  12, 14, 14), "First", cmFirstRec, bfNormal));
  insert(new TButton(TRect(16, 12, 28, 14), "Previous",  cmPrevRec, 
    bfNormal));
  insert(new TButton(TRect(30, 12, 42, 14), "Next",  cmNextRec, bfNormal));
  insert(new TButton(TRect(44, 12, 56, 14), "Last",  cmLastRec, bfNormal));

  //
  // place delete and new record buttons
  //
  insert(new TButton(TRect(9,  15, 21, 17), "New", cmNewRec, bfNormal));
  insert(new TButton(TRect(23, 15, 35, 17), "Delete",  cmDeleteRec, 
    bfNormal));

  //
  // put in a button to close the dialog
  //
  insert(new TButton(TRect(37,  15, 49, 17), "Close", cmOK, bfDefault));

  //
  // tell the field input lines on the dialog box to load the field values
  // from the generic record buffer
  //
  loadFieldValues();
}

/****************************************************************************
*          
* CLASS: TExampleDialog
*                                                                
* MEMBER FUNCTION: handleEvent
*
* DESCRIPTION:  this is the event handler for the example dialog.  It
*               responds to the button presses that occur in the dialog
*               
* RETURNS: NONE
*
****************************************************************************/
void TExampleDialog::handleEvent(TEvent &event)
{
  //
  // act like the base class
  //
  TCursorDialog::handleEvent(event);

  //
  // is this one of the buttons on the dialog
  //
  if(event.what == evCommand)
  {
    Retcode err;

    //
    // has the user changed any of the field values
    //
    if(hasRecordValuesChanged())
    {
      //
      // update the record
      //
      storeFieldValues();
      theCursor->updateRec(theCursor->genericRec);
    }


    switch(event.message.command)
    {
      case cmFirstRec:
        //
        // go to the first record
        //
        err = theCursor->gotoBegin();
        if(err)
        {
          break;
        }

      case cmNextRec:   
        //
        // go to the next record
        //
        err = theCursor->gotoNext();
        break;

      case cmLastRec:   
        //
        // go to the last record
        //
        err = theCursor->gotoEnd();
        if(err)
        {
          break;
        }

      case cmPrevRec:   
        //
        // go to the previous record
        //
        err = theCursor->gotoPrev();
        break;

      case cmDeleteRec: 
        //
        // delete the current record
        //
        err = theCursor->deleteRec();
        //
        // go to the next record if there was no error
        //
        if(!err)
        {
          theCursor->gotoNext();

          if(theCursor->getCurRecNum() == 0L)
          {
            //
            // if the current record number is 0 then we deleted the last 
            // record in the table, so clear out the record buffer
            //
            theCursor->genericRec->clear();
          }
        }
        break;

      case cmNewRec:    
        //
        // add a new (blank) record to the table
        //
        theCursor->genericRec->clear();
        //
        // append this record to the table
        //
        err = theCursor->appendRec(theCursor->genericRec);
        break;

      default:
        return;
    }

    //
    // if there was an error then show the error message
    //
    if(err)
    {
      messageBox(mfError | mfOKButton, "%s", 
        TTestApp::theEngine->getErrorMessage(err));
    }

    //
    // load the field values into the generic record buffer
    //
    theCursor->getRecord();
    //
    // tell the field input lines to reload the field values
    //
    loadFieldValues();

    clearEvent(event);
  }

}


/****************************************************************************
*          
* FUNCTION: main
*
* DESCRIPTION:  this is where the program starts
*               
* RETURNS: the status of the program
*
****************************************************************************/
int main()
{
  TTestApp  theTest;

  theTest.run();

  return(0);
}

