{$S-,R-,I-,V-,B-}

{XGRAF - V1.01 Scaling Graph and Curve drawing program}
{Copyright (c) 1988 Michael Day - all rights reserved}
{first release 22 October 1988} {first public release}

{This is a shareware program. Refer to the license agreement for further}
{information. If you do not have the license documentation you may}
{obtain it by writting to me at: }

{     Michael Day                                               }
{     C/O Day Research                                          }
{     P.O. Box 22902                                            }
{     Milwaukie, OR 97222                                       }
{                                                               }
{     CIS [73577,2225]                                          }
{     Mike Day  UUCP:...!tektronix!reed!qiclab!bakwatr!mikeday  }

{If you include $10.00 I will also send you a current copy of the full}
{shareware package.}

{Note: If you agree to the terms of the shareware license you may}
{use this program free of royalties, and you may use this program}
{in conjunction with any program you may develop, that XGRAF is a}
{part of, in private or commercial applications free of royalty payments.}
{The catch is that you must provide a copy of any enhancments to XGRAF}
{to be distributed among the other XGRAF developers.}
{See the license agreement for further details.}


{ This graph unit will draw a prescribed graph on }
{ the screen ready to have a curve drawn on it.}

unit gunit;

interface

uses graph;

const
  MaxgNum = 20;       { Maximum number of graphs allowed }
  MaxScale = 40;      { Maximum number of Scale divisions allowed }
  ScaleStrSize = 12;  { Maximum size of Scale string }

  gTopTitle   = 0;    { Title and scale identification }
  gBotTitle   = 1;
  gTopScale   = 4;
  gBotScale   = 5;
  gLeftScale  = 6;
  gRightScale = 7;

  gLeftWrite   = 0;    { title placement }
  gCenterWrite = 1;
  gRightWrite  = 2;

  gError : integer = 0;

type
  Grect = record Xmin,Ymin,Xmax,Ymax:integer; end;

  TitleRec = record
               TArea    : Grect;   { Title area }
               FColor   : integer; { Foreground color }
               BColor   : integer; { Background color }
               BPattern : integer; { Background pattern }
               WritePos : integer; { left, center, or right}
               Str : string[80];   { Title String }
             end;

  ScaleArray = array[1..MaxScale] of string[ScaleStrSize];
  ScaleRec = record
               SArea    : Grect;     { Scale area }
               FColor   : integer;   { Foreground color }
               BColor   : integer;   { Background color }
               BPattern : integer;   { Background pattern }
               MsSize   : integer;   { Max Scale str length }
               ScaleStr : ScaleArray;{ Array of scale strings }
               ScalePos : array[1..MaxScale] of integer; { scale tick pos }
               ScaleArea: array[1..MaxScale] of Grect;  { Scale text area }
             end;

  TitlePtr = ^TitleRec;
  ScalePtr = ^ScaleRec;


  {The following structure is the primary data structure used by the graph.}
  GraphRecType = record
       {-- These vars must be set by the calling program --}

                {-- controlled by gSetArea --}
                GArea    : Grect;      { Full graph area used (with labels) }

                {-- controlled by gSetTitle --}
                TopTitle   : TitlePtr; { Pointer to Top Title string }
                BotTitle   : TitlePtr; { Pointer to Bot Title string }

                {-- controlled by gSetScale --}
                Y1Scale   : ScalePtr; { Pointer to Y1 Scale strings }
                Y2Scale   : ScalePtr; { Pointer to Y2 Scale strings }
                X1Scale   : ScalePtr; { Pointer to X1 Scale strings }
                X2Scale   : ScalePtr; { Pointer to X2 Scale strings }

                {-- controlled by gSetTick --}
		XTickCnt  : byte;     { Number of X Ticks }
		YTickCnt  : byte;     { Number of Y Ticks }
                XGratCnt  : byte;     { Number of X Graticules}
                YGratCnt  : byte;     { Number of Y Graticules}
                XTickOff  : byte;     { First graticule tick from X start }
                YTickOff  : byte;     { First graticule tick from Y start }
                XGratTick : byte;     { X graticules per tick }
                YGratTick : byte;     { Y graticules per tick }

                {-- controlled by gSetGraph --}
                GratColor : byte;     { Color of graticules and ticks }
                GBColor   : byte;     { Graph Background color }
                GBPattern : byte;     { Graph Background pattern }
                GraphType : byte;     { bit     0      1    }
                  {GraphType defines} { on - Display  Dots  }
                  {graticule display} { off- Inhibit  Lines }

       {-- These vars are generated by gDefine --}
                XGOff : real;         { Pixel Offset between X graticules }
                YGOff : real;         { Pixel offset between Y graticules }

                GtArea : Grect;       { Scale tick mark outside area }
                GpArea : Grect;       { Active graph pixel area (screen area) }
	end;

type GraphPtr = ^GraphRecType;
var gNum : array [1..MaxgNum] of GraphPtr;


{-------------------------------------------------------------------------}
{ check to see if the requested graph number exists and if it is valid }
{ returns true if exists, false if not, with the error in gError }

function gExist(G:integer):boolean;


{----------------------------------------------------------------------}
{ allocate and clear graph info on heap returns nil on insufficient memory }

function gAlloc(G: integer): boolean;


{----------------------------------------------------------------------}
{ This procedure reclaims all dynamic memory used by the graph.      }
{ Note: if you wish to clear or restore the screen, you must do that }
{ on your own. }

function gRelease(G:integer):boolean;


{----------------------------------------------------------------------}
{this defines the overall pixel area on the screen that is to be used}

function gSetArea(G: integer; x1,y1,x2,y2:integer): boolean;


{-------------------------------------------------------------------------}
{Sets the title "S" for the specified graph "G" in the Position "Pos". }
{FC sets the Foreground color, BC the background color, and BP sets the }
{background pattern. WP determines if the title is to be left justified,}
{right justified, or centered.}

function gSetTitle(G: integer; Pos, WP: byte;
                               FC, BC, BP: byte;
                               S:string): boolean;


{-------------------------------------------------------------------------}
{Sets the Scale "Sa" for the specified graph "G" in the Position "Pos" }
{with a maximum string length of "Cnt". FC, BC, and BP define the scale }
{text foreground color, background color, and background pattern. }

function gSetScale(G: integer; Pos,Cnt: integer;
                   FC,BC,BP: byte;
                   Sa:ScaleArray): boolean;


{-------------------------------------------------------------------------}
{this defines the tick and graticule markings on the graph}
{XTc and YTc define the number of scale tick marks along the X and Y axis}
{XTo and YTo define the first position in which a tick mark is drawn}
{XTg and YTg define how many graticules to draw between tick marks}

function gSetTick(G:integer; XTc,YTc,XGc,YGc,XTo,YTo,XGt,YGt:integer):boolean;


{-------------------------------------------------------------------------}
{define graph screen area}

function gDefine(G:integer):boolean;



{-------------------------------------------------------------------------}
{this defines the type of graph and graph coloring }
{FC sets the graticule color, BC the background color, BP the background}
{pattern, and GT the graph type. Graph type is defined as follows: }
{ bit     0      1    }
{ on - Display  Dots  }
{ off- Inhibit  Lines }

function gSetGraph(G:integer; FC,BC,BP,GT:integer):boolean;



{------------------------------------------------------------------------}
{ Draw the graph labels.  G=graph number L=label }

function gDrawLabel(G:integer; L:integer): boolean;


{-------------------------------------------------------------------------}
{this draws the graph and graticules on the screen}

function gDrawGraph(G:integer):boolean;



{-------------------------------------------------------------------------}
{this clears the graph area and draws a full graph on the screen w/labels}

function gDrawFull(G:integer):boolean;


{-------------------------------------------------------------------------}
{ This returns the available plot area inside the graph in: x1,y1,x2,y2 }

function gPlotArea(G:integer; var x1,y1,x2,y2:integer): boolean;


{----------------------------------------------------------------------}
{ Returns the error msg associated with the indicated error number. }

function gErrorMsg(Error:integer):string;



{**********************************************************************}

implementation


const
  NoHeapSpace  = 1;
  GraphInUse   = 2;
  NoGraphAvail = 3;
  BadGraphNum  = 4;
  BadGraphLab  = 5;


{++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++}
{                      graph support routines                          }
{++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++}

{----------------------------------------------------------------------}
{generic memory allocation function - sets error if can't allocate}

function Alloc(var P; Size:integer):boolean;
begin
   Alloc := false;
   if MaxAvail < Size then
   begin
     gError := NoHeapSpace;    {Whoa! Bummer, not enough heap space}
     Exit;
   end;

   GetMem(pointer(P), Size);
   if pointer(P) = nil then
   begin
     gError := NoHeapSpace;    {Surprise! Not enough memory}
     Exit;
   end;
   fillchar(pointer(P)^, Size, 0);  {clear out the allocated memory}
   Alloc := true;
end;


{-------------------------------------------------------------------------}
{write the string starting at the left of the given rectangle}

procedure gBoxWrite(Item: string; {string to be written}
                    R: Grect;       {area to write it in}
                    FColor: byte;   {foreground color}
                    BColor: byte;   {background colors}
                    BPattern: word; {background pattern}
                    Pos: byte);     {string print position}
var x,y: integer;
      s: string[80];
begin
    s := copy(Item,1,succ((R.Xmax-R.Xmin) div TextWidth('X')));
    y := (succ(R.Ymax-R.Ymin) shr 1) + succ(R.Ymin);
    case Pos of
      gRightWrite :
      begin
        SetTextJustify(RightText,CenterText);
        x := pred(R.Xmax);
        if TextWidth(s) > (x - R.Xmin) then
        begin
          SetTextJustify(LeftText,CenterText);
          x := succ(R.Xmin);
        end;
      end;

      gCenterWrite :
      begin
        SetTextJustify(CenterText,CenterText);
        x := (succ(R.Xmax-R.Xmin) shr 1) + R.Xmin;
      end;

      else
      begin
        SetTextJustify(LeftText,CenterText);
        x := R.Xmin+3;
      end;
    end;

    SetFillStyle(BPattern,BColor);
    bar(R.Xmin,R.Ymin,R.Xmax,R.Ymax);
    moveto(x,y);
    setcolor(Fcolor);
    outtext(s);
end;


{-------------------------------------------------------------------------}
{- LabelDraw is used by DrawLabel and DrawFull -}

function LabelDraw(G:integer; L:integer):boolean;

var i:integer;
begin
  LabelDraw := false;
  with gNum[G]^ do
  begin
    case L of
      gTopTitle: begin
           if TopTitle = nil then Exit;
           with TopTitle^ do
             gBoxWrite(Str, TArea, FColor, BColor, BPattern, WritePos);
         end;
      gBotTitle: begin
           if BotTitle = nil then Exit;
           with BotTitle^ do
             gBoxWrite(Str, TArea, FColor, BColor, BPattern, WritePos);
         end;
      gTopScale: begin
           if X1Scale = nil then Exit;
           with X1Scale^ do
             for i := 1 to XTickCnt do
               gBoxWrite(ScaleStr[i], ScaleArea[i],
                         FColor, BColor, BPattern, gCenterWrite);
         end;
      gBotScale: begin
           if X2Scale = nil then Exit;
           with X2Scale^ do
             for i := 1 to XTickCnt do
               gBoxWrite(ScaleStr[i], ScaleArea[i],
                         FColor, BColor, BPattern, gCenterWrite);
         end;
      gLeftScale: begin
           if Y1Scale = nil then Exit;
           with Y1Scale^ do
             for i := 1 to YTickCnt do
               gBoxWrite(ScaleStr[i], ScaleArea[i],
                          FColor, BColor, BPattern, gRightWrite);
         end;
      gRightScale: begin
           if Y2Scale = nil then Exit;
           with Y2Scale^ do
             for i := 1 to YTickCnt do
               gBoxWrite(ScaleStr[i], ScaleArea[i],
                          FColor, BColor, BPattern, gLeftWrite);
         end;
      else
         Exit;
      end;
  end;
  LabelDraw := true;
end;


{-------------------------------------------------------------------------}
{Grat Draw is used by DrawFull and DrawGraph}

procedure GratDraw(G:integer);

var i,ii,gao:integer;

begin
   with gNum[G]^ do
   begin

     with GpArea do
     begin

       {- outline the graph plot area -}
       SetColor(GratColor);
       Rectangle(Xmin,Ymin,Xmax,Ymax);

       {- Now we draw the graticules, if any -}
       if (GraphType and $0003) = $0003 then  {dot graticules}
       begin
         for i := 1 to YGratCnt do
         begin
           for ii := 1 to XGratCnt do
           begin
             PutPixel(Xmin+round(ii*XGoff),Ymax-round(i*YGoff),GratColor);
           end;
         end;
       end;

       if (GraphType and $0003) = $0001 then  {line graticules}
       begin
         for i := 1 to YGratCnt do
         begin
           gao := Ymax - round(i*YGoff);
           line(Xmin, gao, Xmax, gao);
         end;
         for i := 1 to XGratCnt do
         begin
           gao := Xmin + round(i*XGoff);
           line(gao, Ymin, gao, Ymax);
         end;
       end;

     end;  {with GpArea}

     {- Now we draw the tick marks, if any -}
     if Y1Scale <> nil then
     begin
       for i := 1 to YTickCnt do
       begin
         line(GtArea.Xmin, Y1Scale^.ScalePos[i],
              GpArea.Xmin, Y1Scale^.ScalePos[i]);
       end;
     end;
     if Y2Scale <> nil then
     begin
       for i := 1 to YTickCnt do
       begin
         line(GpArea.Xmax, Y2Scale^.ScalePos[i],
              GtArea.Xmax, Y2Scale^.ScalePos[i]);
       end;
     end;
     if X1Scale <> nil then
     begin
       for i := 1 to XTickCnt do
       begin
         line(X1Scale^.ScalePos[i], GtArea.Ymin,
              X1Scale^.ScalePos[i], GpArea.Ymin);
       end;
     end;
     if X2Scale <> nil then
     begin
       for i := 1 to XTickCnt do
       begin
         line(X2Scale^.ScalePos[i], GpArea.Ymax,
              X2Scale^.ScalePos[i], GtArea.Ymax);
       end;
     end;

   end;
end;




{+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++}
{                     major graphing routines                             }
{+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++}


{========================================================================}
{ check to see if the requested graph number exists and if it is valid }
{ returns true if exists, false if not, with the error in gError }

function gExist(G:integer):boolean;
begin
   gExist := false;
   gError := 0;
   if G > MaxgNum then
   begin
     gError := BadGraphNum;             {Whoa! Ya gotta use valid numbers!}
     Exit;
   end;
   if gNum[G]=nil then
   begin
     gError := NoGraphAvail; {Whoa! This graph number was never allocated!}
     Exit;
   end;
   gExist := true;
end;


{========================================================================}
{ allocate and clear graph info on heap returns nil on insufficient memory }

function gAlloc(G: integer): boolean;
var Gp:GraphPtr;
begin
  gAlloc := false;
  gError := 0;

  if G > MaxgNum then
  begin
    gError := BadGraphNum;      {Whoa! Ya gotta use valid numbers!}
    Exit;
  end;
  if gNum[G]<>nil then       {check to see if graph already being used}
  begin
    gError := GraphInUse;
    Exit;
  end;
  if not(Alloc(Gp, sizeof(Gp^))) then Exit;   {allocate space on heap}

  gNum[G] := Gp;     {made it, so stick the pointer in the graph array}
  gAlloc := true;
end;


{========================================================================}
{ This removes a graph from the stack. You must remove it from the }
{ screen yourself if that is a requirement. }

function gRelease(G:integer):boolean;
begin
  gRelease := false;
  if not(gExist(G)) then Exit;             {check for valid graph number}
  with gNum[G]^ do                   {dump any labels that are allocated}
  begin
    if TopTitle <> nil then freemem(TopTitle,sizeof(TopTitle^));
    if BotTitle <> nil then freemem(BotTitle,sizeof(BotTitle^));
    if X1Scale <> nil then freemem(X1Scale,sizeof(X1Scale^));
    if X2Scale <> nil then freemem(X2Scale,sizeof(X2Scale^));
    if Y1Scale <> nil then freemem(Y1Scale,sizeof(Y1Scale^));
    if Y2Scale <> nil then freemem(Y2Scale,sizeof(Y2Scale^));
  end; 
  freemem(gNum[G],sizeof(gNum[G]^));       {finally dump the graph data}
  gNum[G] := nil;                          {and turn off the pointer}
  gRelease := true;
end;


{========================================================================}
{this defines the overall pixel area on the screen that is to be used}

function gSetArea(G: integer; x1,y1,x2,y2:integer): boolean;
begin
  gSetArea := false;
  if not(gExist(G)) then Exit;            {check for valid graph number}
  with gNum[G]^ do
  begin
    GArea.Xmin := x1;
    GArea.Ymin := y1;
    GArea.Xmax := x2;
    GArea.Ymax := y2;
  end;
  gSetArea := true;
end;


{========================================================================}
{Sets the title "S" for the specified graph "G" in the Position "Pos". }
{FC sets the Foreground color, BC the background color, and BP sets the }
{background pattern. WP determines if the title is to be left justified,}
{right justified, or centered.}
function gSetTitle(G: integer; Pos, WP: byte;
                               FC, BC, BP: byte;
                               S:string): boolean;

{----------------------------------}
function SetTit(var Tp:TitlePtr;
                    WP,FC,BC,BP:byte;
                    S:string):boolean;
begin
   SetTit := false;
   if length(S) <> 0 then
   begin
     if Tp = nil then
       if not(Alloc(Tp,sizeof(Tp^))) then Exit;
     with Tp^ do
     begin
       FColor := FC;
       BColor := BC;
       BPattern := BP;
       WritePos := WP;
       Str := S;
     end;
   end
   else
   begin
     if Tp <> nil then
     begin
       freemem(Tp,sizeof(Tp^));
       Tp := nil;
     end;
   end;
   SetTit := true;
end;

{----------------------------------------------}
{- gSetTitle -}
begin
   gSetTitle := false;
   if not(gExist(G)) then Exit;

   with gNum[G]^ do
   begin
     case Pos of
       gTopTitle : if not(SetTit(TopTitle, WP,FC,BC,BP,S)) then Exit;
       gBotTitle : if not(SetTit(BotTitle, WP,FC,BC,BP,S)) then Exit;
       else
       begin
         gError := BadGraphLab;
         Exit;
       end;
     end;
   end;
   gSetTitle := true;
end;


{========================================================================}
{Sets the Scale "Sa" for the specified graph "G" in the Position "Pos" }
{with a maximum string length of "Cnt". FC, BC, and BP define the scale }
{text foreground color, background color, and background pattern. }
{you must call gDefine after this function before you can use the graph}

function gSetScale(G: integer; Pos,Cnt: integer;
                    FC,BC,BP: byte;
                    Sa:ScaleArray): boolean;


{----------------------------------------}
function SetGsc(var Sp:ScalePtr;
                    Cnt,FC,BC,BP:byte;
                    Sa:ScaleArray):boolean;
begin
   SetGsc := false;
   if Cnt > 0 then
   begin
     if Sp = nil then
       if not(Alloc(Sp,sizeof(Sp^))) then Exit;
     with Sp^ do
     begin
        ScaleStr := Sa;
        MsSize := Cnt;
        FColor := FC;
        BColor := BC;
        BPattern := BP;
     end;
   end
   else
   begin
     if Sp <> nil then
     begin
       freemem(Sp,sizeof(Sp^));
       Sp := nil;
     end;
   end;
   SetGsc := true;
end;

{---------------------------------------------}
{- gSetScale -}

begin
   gSetScale := false;
   if not(gExist(G)) then Exit;

   with gNum[G]^ do
   begin
     case Pos of
       gTopScale  : if not(SetGsc(X1Scale,Cnt,FC,BC,BP,Sa)) then Exit;
       gBotScale  : if not(SetGsc(X2Scale,Cnt,FC,BC,BP,Sa)) then Exit;
       gLeftScale : if not(SetGsc(Y1Scale,Cnt,FC,BC,BP,Sa)) then Exit;
       gRightScale: if not(SetGsc(Y2Scale,Cnt,FC,BC,BP,Sa)) then Exit;
       else
       begin
         gError := BadGraphLab;
         Exit;
       end;
     end;
   end;
   gSetScale := true;
end;


{========================================================================}
{this defines the tick and graticule markings on the graph}
{XTc and YTc define the number of scale tick marks along the X and Y axis}
{XTo and YTo define the first position in which a tick mark is drawn}
{XTg and YTg define how many graticules to draw between tick marks}

function gSetTick(G:integer; XTc,YTc,XGc,YGc,XTo,YTo,XGt,YGt:integer):boolean;
begin
   gSetTick := false;
   if not(gExist(G)) then Exit;
   with gNum[G]^ do
   begin
     XTickCnt := XTc;
     YTickCnt := YTc;
     XGratCnt := XGc;
     YGratCnt := YGc;
     XTickOff := XTo;
     YTickOff := YTo;
     XGratTick := XGt;
     YGratTick := YGt;
   end;
   gSetTick := true;
end;


{========================================================================}
{define graph screen area}

function gDefine(G:integer):boolean;


{--------------------------------------------}
{returns a limited variable V between B and E}
function Limit(V,B,E:integer):integer;
begin
  Limit := V;
  if V < B then Limit := B;
  if V > E then Limit := E;
end;

function GetYGp(G,I:integer):integer;
begin
  with gNum[G]^ do
  begin
    GetYGp := GpArea.Ymin + round(XGOff*I);
  end;
end;

function GetXGp(G,I:integer):integer;
begin
  with gNum[G]^ do
  begin
    GetXGp := GpArea.Xmin + round(XGOff*I);
  end;
end;

{set area and tick address for Xscale}
procedure SetXsa(G:integer; Fs:integer; var Xsa:ScalePtr);
var i : integer;
begin
   with gNum[G]^,Xsa^ do
   begin
      for i := 1 to XTickCnt do
      begin
        ScalePos[i] := GpArea.Xmin+round(XGOff*((pred(i)*XGratTick)+XTickOff));
        ScaleArea[i].Xmin := ScalePos[i] - Fs;
        ScaleArea[i].Xmax := ScalePos[i] + Fs;
      end;
   end;
end;


procedure SetYsa(G:integer; Fs:integer; var Ysa:ScalePtr);
var i : integer;
begin
   with gNum[G]^,Ysa^ do
   begin
      for i := 1 to YTickCnt do
      begin
        ScalePos[i] := GpArea.Ymax-round(YGOff*((pred(i)*YGratTick)+YTickOff));
        ScaleArea[i].Ymin := ScalePos[i] - Fs;
        ScaleArea[i].Ymax := ScalePos[i] + Fs;
      end;
   end;
end;


{-----------------------------------}
{- gDefine -}

var GxArea: Grect;
    i,ix : integer;
    X1Fs,X2Fs,Y1Fs,Y2Fs:integer;
begin
   gDefine := false;
   if not(gExist(G)) then Exit;            {check for valid graph number}

   with gNum[G]^ do
   begin
     XTickCnt := Limit(XTickCnt,1,MaxScale);
     YTickCnt := Limit(YTickCnt,1,MaxScale);
     XGratTick := Limit(XGratTick,1,MaxScale);
     YGratTick := Limit(YGratTick,1,MaxScale);
     GxArea := GArea;

     {- set the Top Title area -}
     if TopTitle <> nil then
     begin
       with TopTitle^ do
       begin
         TArea := GArea;
         TArea.Ymax := GArea.Ymin + (TextHeight('X')+2);
         GxArea.Ymin := succ(TArea.Ymax);
       end;
     end;

     {- set the Bot Title area -}
     if BotTitle <> nil then
     begin
       with BotTitle^ do
       begin
         TArea := GArea;
         TArea.Ymin := GArea.Ymax - (TextHeight('X')+2);
         GxArea.Ymax := pred(TArea.Ymin);
       end;
     end;

     GtArea := GxArea; {GxArea now = area inside titles}
     GpArea := GtArea;

     {- set the X1Scale background area and tick area -}
     if X1Scale <> nil then
     begin
       with X1Scale^ do            { TopScale }
       begin
         MsSize := Limit(MsSize,1,MaxScale);
         SArea := GxArea;
         SArea.Ymax := GxArea.Ymin + (TextHeight('X'));     {text}
         GtArea.Ymin := succ(SArea.Ymax);
         for i := 1 to MaxScale do
           ScaleArea[i] := SArea;
         GpArea.Ymin := GtArea.Ymin + (TextHeight('X') shr 2); {tick}
         X1Fs := (MsSize*TextWidth('X')) shr 1;
       end;
     end;

     {- set the X2Scale background area and tick area -}
     if X2Scale <> nil then
     begin
       with X2Scale^ do           { BotScale }
       begin
         MsSize := Limit(MsSize,1,MaxScale);
         SArea := GxArea;
         SArea.Ymin := GxArea.Ymax - (TextHeight('X'));      {text}
         GtArea.Ymax := pred(SArea.Ymin);
         for i := 1 to MaxScale do
           ScaleArea[i] := SArea;
         GpArea.Ymax := GtArea.Ymax - (TextHeight('X') shr 2);  {tick}
         X2Fs := (MsSize*TextWidth('X')) shr 1;
       end;
     end;

     {- set the Y1Scale background area and tick area -}
     if Y1Scale <> nil then
     begin
       with Y1Scale^ do           { LeftScale }
       begin
         MsSize := Limit(MsSize,1,MaxScale);
         SArea := GxArea;
         SArea.Xmax := GxArea.Xmin + (TextWidth('X')*MsSize) + 4; {text}
         GtArea.Xmin := succ(SArea.Xmax);
         for i := 1 to MaxScale do
           ScaleArea[i] := SArea;
         GpArea.Xmin := GtArea.Xmin + (TextWidth('X') shr 2);  {tick}
         Y1Fs := (TextHeight('X')) shr 1;
       end;
     end;

     {- set the Y2Scale background area and tick area -}
     if Y2Scale <> nil then
     begin
       with Y2Scale^ do          { RightScale }
       begin
         MsSize := Limit(MsSize,1,MaxScale);
         SArea := GxArea;
         SArea.Xmin := GxArea.Xmax - (TextWidth('X')*MsSize) - 4;  {text}
         GtArea.Xmax := pred(SArea.Xmin);
         for i := 1 to MaxScale do
           ScaleArea[i] := SArea;
         GpArea.Xmax := GtArea.Xmax - (TextWidth('X') shr 2);   {tick}
         Y2Fs := (TextHeight('X')) shr 1;
       end;
     end;
     {GpArea now = active graph plot area, GtArea = scale tick area}

     XGoff := (GpArea.Xmax-GpArea.Xmin)/(XGratCnt);
     YGoff := (GpArea.Ymax-GpArea.Ymin)/(YGratCnt);

     {set the X1Scale text area and tick position addresses}
     if X1Scale <> nil then SetXsa(G,X1Fs,X1Scale);

     {set the X2Scale text area and tick position addresses}
     if X2Scale <> nil then SetXsa(G,X2Fs,X2Scale);

     {set the Y1Scale text area and tick position addresses}
     if Y1Scale <> nil then SetYsa(G,Y1Fs,Y1Scale);

     {set the Y2Scale text area and tick position addresses}
     if Y2Scale <> nil then SetYsa(G,Y2Fs,Y2Scale);

   end;
   gDefine := true;
end;


{========================================================================}
{this defines the type of graph and graph coloring }
{FC sets the graticule color, BC the background color, BP the background}
{pattern, and GT the graph type. Graph type is defined as follows: }
{ bit     0      1    }
{ on - Display  Dots  }
{ off- Inhibit  Lines }

function gSetGraph(G:integer; FC,BC,BP,GT:integer):boolean;
begin
   gSetGraph := false;
   if not(gExist(G)) then Exit;
   with gNum[G]^ do
   begin
     GratColor := FC;
     GBColor := BC;
     GBPattern := BP;
     GraphType := GT;
   end;
   gSetGraph := true;
end;


{========================================================================}
{ Draw the graph labels.  G=graph number L=label }

function gDrawLabel(G:integer; L:integer): boolean;

var  i : integer;
begin
  gDrawLabel := false;
  if not(gExist(G)) then Exit;            {check for valid graph number}

  if not(LabelDraw(G,L)) then
  begin
     gError := BadGraphLab;  {invalid label request}
     Exit;
  end;
  gDrawLabel := true;
end;


{========================================================================}
{this draws (or re-draws) the graph and graticules on the screen}
{ GraphType: }
{ bit     0      1    }
{ on - Display  Dots  }
{ off- Inhibit  Lines }

function gDrawGraph(G:integer):boolean;
begin
   gDrawGraph := false;
   if not(gExist(G)) then Exit;

   with gNum[G]^ do
   begin
     SetFillStyle(GBPattern,GBColor);       {- Erase the full graph area -}
     Bar(GtArea.Xmin,GtArea.Ymin,GtArea.Xmax,GtArea.Ymax);
     GratDraw(G);                                 {- now draw the graph -}
   end;

   gDrawGraph := true;
end;


{========================================================================}
{this clears the graph area and draws a full graph on the screen w/labels}

function gDrawFull(G:integer):boolean;


{---------------------------------------}
procedure DrawScaleBar(var Sp:ScalePtr);
begin
   if Sp <> nil then
   begin
     with Sp^,SArea do
     begin
       SetFillStyle(BPattern,BColor);       {- Erase the full scale area -}
       Bar(Xmin,Ymin,Xmax,Ymax);
     end;
   end;
end;

{-------------------------------------}
{- gDrawFull -}
var i:integer;

begin
   gDrawFull := false;
   if not(gExist(G)) then Exit;

   with gNum[G]^ do
   begin
     SetFillStyle(GBPattern,GBColor);       {- Erase the full graph area -}
     Bar(GArea.Xmin,GArea.Ymin,GArea.Xmax,GArea.Ymax);
     DrawScaleBar(X1Scale);
     DrawScaleBar(X2Scale);
     DrawScaleBar(Y1Scale);
     DrawScaleBar(Y2Scale);
     GratDraw(G);                                 {- now draw the graph -}
   end;
   for i := 0 to 7 do
     if LabelDraw(G,i) then;  {ignore any errors}

   gDrawFull := true;
end;

{========================================================================}
{ This returns the available plot area inside the graph in: x1,y1,x2,y2 }

function gPlotArea(G:integer; var x1,y1,x2,y2:integer): boolean;

begin
   gPlotArea := false;
   if not(gExist(G)) then Exit;
   with gNum[G]^ do
   begin
     x1 := GpArea.Xmin;
     y1 := GpArea.Ymin;
     x2 := GpArea.Xmax;
     y2 := GpArea.Ymax;
   end;
   gPlotArea := true;
end;


{========================================================================}
{ return error message relating to the error encountered }

function gErrorMsg(Error: integer): string;
var s : string[12];
begin
  case gError of
    NoHeapSpace  : gErrorMsg := 'Not enough memory ';
    GraphInUse   : gErrorMsg := 'Graph in use ';
    NoGraphAvail : gErrorMsg := 'Graph not initialized ';
    BadGraphNum  : gErrorMsg := 'Invalid graph number ';
    BadGraphLab  : gErrorMsg := 'Invalid graph label ';
    else
    begin
      str(Error,S);
      gErrorMsg := 'Unknown Error number: '+s+' ';
    end;
  end;
end;


{**********************************************************************}
{Initialization}

var i : integer;
begin
  for i := 1 to MaxgNum do
  begin
    gNum[i] := nil;
  end;
end.

