{ This Unit contains DIB manipulation routines for Loading and Saving etc. }
{ Written by Mike Scott CompuServe 100140,2420 }
unit DIBUnit ;

{$I-}

interface

uses WinTypes, WinProcs, Objects, OMemory ;

type
  PContrastTable = ^TContrastTable ;
  TContrastTable = array[ 0..63 ] of byte ;

  PContrastTable256 = ^TContrastTable256 ;
  TContrastTable256 = array[ 0..255 ] of byte ;

type
  PScanLine  = ^TScanLine ;
  TScanLine  = array[ 0..65520 ] of byte ;
  PScanLines = ^TScanLines ;
  TScanLines = array[ 0..16000 ] of PScanLine ;

type
  PDIB = ^TDIB ;
  TDIB = object( TObject )
    DIBBits        : THandle ;
    BitmapInfoSize : word ;
    BitmapInfo     : PBitmapInfo ;
    Width          : Word ;
    Height         : Word ;
    ScanWidth      : word ;
    FileName       : PChar ;
    ScanLines      : PScanLines ;
    LineBuffer     : PScanLine ;
    constructor Init( AFileName : PChar ) ;
    constructor Assemble( ADIBBits : THandle ;
                          ABitmapInfo : PBitmapInfo ;
                          ABitmapInfoSize : word ) ;
    destructor Done ; virtual ;
    procedure GetWidthAndHeight ;
    procedure Paint( DC       : HDC ;
                     x, y     : integer ;
                     APalette : HPalette ) ; virtual ;
    procedure PaintWithContrast( DC       : HDC ;
                                 x, y     : integer ;
                                 APalette : HPalette ;
                                 const AContrastTable : TContrastTable256 ) ; virtual ;
    function  SaveAs( AFileName : PChar ) : boolean ; virtual ;
    function  AccessScanLines : boolean ;
    procedure FreeScanLines ;
    function  GetScanLine( ALineNo : word ;
                           ALineBuffer : PScanLine ) : PScanLine ; virtual ;
    function  GetBufferedScanLine( ALineNo : word ) : PScanLine ; virtual ;
    function  GetScanColumn( AColumn : word ; ABuffer : PScanLine ) : PScanLine ; virtual ;
    function  GetAPixel( x, y : word ) : longint ; virtual ;
    function  CreateHBitmap( DC : HDC ) : HBitmap ; virtual ;
  end ;

function  LoadDIB( AFileName : PChar ;
                   var BitmapInfo : PBitmapInfo ;
                   var BitmapInfoSize : word ;
                   var DIBBits        : THandle ) : boolean ;

implementation

uses Strings, BMPUnit ;


{TDIB}

constructor TDIB.Init( AFileName : PChar ) ;

begin
  inherited Init ;
  FileName := nil ;
  BitmapInfo := nil ;
  Width := 0 ;
  Height := 0 ;
  ScanWidth := 0 ;
  ScanLines := nil ;
  LineBuffer := nil ;

  { load the DIB }
  LoadDIB( AFileName, BitmapInfo, BitmapInfoSize, DIBBits ) ;
  GetWidthAndHeight ;

  { create file name PChar if successful }
  if DIBBits <> 0 then FileName := StrNew( AFileName ) else Fail ;
end ;


constructor TDIB.Assemble( ADIBBits : THandle ;
                           ABitmapInfo : PBitmapInfo ;
                           ABitmapInfoSize : word ) ;

begin
  { check parameters }
  if ( ADIBBits = 0 ) or ( ABitmapInfo = nil ) or
     ( ABitmapInfoSize = 0 ) then Fail ;

  { set up from parameters }
  DIBBits := ADIBBits ;
  BitmapInfo := ABitmapInfo ;
  BitmapInfoSize := ABitmapInfoSize ;
  GetWidthAndHeight ;
  FileName := nil ;
  ScanLines := nil ;
  LineBuffer := nil ;
end ;


procedure TDIB.GetWidthAndHeight ;

begin
  if BitmapInfo <> nil then with BitmapInfo^.bmiHeader do begin
    Width := biWidth ;
    Height := biHeight ;
    ScanWidth := ( Width + 3 ) and not 3 ;
  end ;
end ;


destructor TDIB.Done ;

begin
  Inherited Done ;
  if DIBBits <> 0 then GlobalFree( DIBBits ) ;
  if BitmapInfo <> nil then FreeMem( BitmapInfo, BitmapInfoSize ) ;
  FreeScanLines ;
  StrDispose( FileName ) ;
end ;


procedure TDIB.Paint( DC   : HDC ;
                      x, y : integer ;
                      APalette : HPalette ) ;

var Bits       : pointer ;
    OldPalette : HPalette ;
    DoCreatePalette : boolean ;

begin
  { paint it }
  with BitmapInfo^.bmiHeader do begin
    DoCreatePalette := APalette = 0 ;
    if DoCreatePalette then APalette := CreatePalette64( PC_NOCOLLAPSE ) ;
    OldPalette := SelectPalette( DC, APalette, true ) ;
    RealizePalette( DC ) ;
    Bits := GlobalLock( DIBBits ) ;
    StretchDIBits( DC, 0, 0, biWidth, biHeight, 0, 0, biWidth, biHeight,
                   Bits, BitmapInfo^, DIB_RGB_COLORS, SRCCOPY ) ;
    GlobalUnlock( DIBBits ) ;
    SelectPalette( DC, OldPalette, true ) ;
    if DoCreatePalette then DeleteObject( APalette ) ;
    RealizePalette( DC ) ;
  end ;
end ;


procedure TDIB.PaintWithContrast( DC   : HDC ;
                                  x, y : integer ;
                                  APalette : HPalette ;
                                  const AContrastTable : TContrastTable256 ) ;

var Bits       : pointer ;
    OldPalette : HPalette ;
    DoCreatePalette : boolean ;
    ABitmapInfo     : PBitmapInfo ;
    n, i            : byte ;

begin
  { paint it }
  with BitmapInfo^.bmiHeader do begin
    DoCreatePalette := APalette = 0 ;
    if DoCreatePalette then APalette := CreatePalette64( PC_NOCOLLAPSE ) ;
    OldPalette := SelectPalette( DC, APalette, true ) ;
    RealizePalette( DC ) ;

    { create a new BitmapInfo with an adjusted contrast table }
    GetMem( ABitmapInfo, BitmapInfoSize ) ;
    Move( BitmapInfo^, ABitmapInfo^, sizeof( TBitmapInfo ) - sizeof( TRGBQuad ) ) ;
    for i := 0 to 255 do with ABitmapInfo^.bmiColors[ i ] do begin
      n := AContrastTable[ i ] ;
      rgbBlue := n ;
      rgbGreen := n ;
      rgbRed := n ;
    end ;

    Bits := GlobalLock( DIBBits ) ;
    StretchDIBits( DC, 0, 0, biWidth, biHeight, 0, 0, biWidth, biHeight,
                   Bits, ABitmapInfo^, DIB_RGB_COLORS, SRCCOPY ) ;
    GlobalUnlock( DIBBits ) ;
    FreeMem( ABitmapInfo, BitmapInfoSize ) ;

    SelectPalette( DC, OldPalette, true ) ;
    if DoCreatePalette then DeleteObject( APalette ) ;
    RealizePalette( DC ) ;
  end ;
end ;


function  TDIB.SaveAs( AFileName : PChar ) : boolean ;

var BitmapFileHeader : TBitmapFileHeader ;
    F                : File ;
    AllOK            : boolean ;
    ABytesWritten    : word ;
    DoIncSeg         : boolean ;
    ADIBSize         : longint ;
    ABytesLeft       : longint ;
    ABytesToWrite    : longint ;
    P                : pointer ;

begin
  SaveAs := false ;
  AllOK := false ;
  if ( DIBBits = 0 ) or ( BitmapInfo = nil ) then exit ;

  assign( F, AFileName ) ;
  rewrite( F, 1 ) ;
  if InOutRes = 0 then begin

    { create and save the file header }
    with BitmapFileHeader, BitmapInfo^.bmiHeader do begin
      bfType := $4D42 ;
      bfSize := sizeof( TBitmapFileHeader ) + BitmapInfoSize +
                ( ( biWidth + 3 ) and not 3 ) * longint( biHeight ) ;
      bfReserved1 := 0 ;
      bfReserved2 := 0 ;
      bfOffBits := sizeof( TBitmapFileHeader ) + BitmapInfoSize ;
      BlockWrite( F, BitmapFileHeader,
                  sizeof( TBitmapFileHeader ), ABytesWritten ) ;
      if ( InOutRes = 0 ) and ( ABytesWritten = sizeof( TBitmapFileHeader ) ) then
      begin

        { save the bitmap info header and colour table }
        BlockWrite( F, BitmapInfo^, BitmapInfoSize, ABytesWritten ) ;
        if ( InOutRes = 0 ) and ( ABytesWritten = BitmapInfoSize ) then begin

          { write out the DIB bits }
          ADIBSize := ( ( biWidth + 3 ) and not 3 ) * longint( biHeight ) ;
          ABytesLeft := ADIBSize ;
          DoIncSeg := false ;
          P := GlobalLock( DIBBits ) ;
          while ABytesLeft > 0 do begin
            { write out in up to 32K chunks 'cos we can't write out a full 64K in one go! }
            ABytesToWrite := ABytesLeft ;
            if ABytesToWrite > 32768 then ABytesToWrite := 32768 ;
            BlockWrite( F, P^, ABytesToWrite, ABytesWritten ) ;
            if ABytesWritten <> ABytesToWrite then break ;
            ABytesLeft := ABytesLeft - ABytesToWrite ;
            if DoIncSeg then P := Ptr( Seg( P^ ) + SelectorInc, Ofs( P^ ) - 32768 ) else
              P := Ptr( Seg( P^ ), Ofs( P^ ) + 32768 ) ;
            DoIncSeg := not DoIncSeg ;
          end ;
          if ABytesLeft = 0 then AllOK := true ;
          GlobalUnlock( DIBBits ) ;
        end ;
      end ;
    end ;
    close( F ) ;
  end ;
  if not AllOK then Erase( F ) ;
  SaveAs := AllOK ;
end ;


function  TDIB.AccessScanLines : boolean ;

var p : LongRec ;
    i : integer ;
    LastLo : word ;

begin
  { create an array of pointers to the start of each scan line but
    set the pointer to NIL for the scan lines that span a segment boundary }
  AccessScanLines := false ;
  GetMem( ScanLines, Height * sizeof( Pointer ) ) ;
  GetMem( LineBuffer, ScanWidth ) ;
  if ( ScanLines <> nil ) and ( LineBuffer <> nil ) then begin
    i := Height - 1 ;
    pointer( p ) := GlobalLock( DIBBits ) ;
    while i >= 0 do begin
      ScanLines^[ i ] := PScanLine( p ) ;
      LastLo := p.Lo ;
      inc( p.Lo, ScanWidth ) ;

      { check if wrapped around a segment boundary }
      if p.Lo < LastLo then begin
        if p.Lo <> 0 then ScanLines^[ i ] := nil ;  { flag the dodgy scan line }
        inc( p.Hi, SelectorInc ) ;  { increment the segment part }
      end ;
      dec( i ) ;
    end ;
    AccessScanLines := true ;
  end else FreeScanLines ;
end ;


procedure TDIB.FreeScanLines ;

begin
  GlobalUnlock( DIBBits ) ;
  if ScanLines <> nil then begin
    FreeMem( ScanLines, Height * sizeof( Pointer ) ) ;
    ScanLines := nil ;
  end ;
  if LineBuffer <> nil then begin
    FreeMem( LineBuffer, ScanWidth ) ;
    LineBuffer := nil ;
  end ;
end ;


function  TDIB.GetScanLine( ALineNo : word ;
                            ALineBuffer : PScanLine ) : PScanLine ;

var p : PScanLine ;
    Bytes : word ;

begin
  GetScanLine := ALineBuffer ;

  { following check could be removed for speed }
  if ALineNo < Height then begin

    { get a pointer to the scan line }
    p := ScanLines^[ ALineNo ] ;

    { check for line spanning segment boundary }
    if p = nil then begin

      { copy the offending line to the supplied buffer }
      if ALineBuffer <> nil then begin

        { calculate actual address from previous scan line by adding a scan line width }
        p := ScanLines^[ ALineNo + 1 ] ;  { '+' because DIB's are bottom up }
        inc( LongRec( p ).Lo, ScanWidth ) ;

        { move the number of bytes up to the segment boundary }
        Bytes := 65536 - ( LongRec( p ).Lo ) ;
        Move( p^, ALineBuffer^, Bytes ) ;

        { move the remainder from the start of the next segment }
        inc( LongRec( p ).Hi, SelectorInc ) ;
        LongRec( p ).Lo := 0 ;
        Move( p^, ALineBuffer^[ Bytes ], ( ScanWidth - Bytes ) ) ;
      end ;
    end else GetScanLine := p ;
  end ;
end ;


function  TDIB.GetBufferedScanLine( ALineNo : word ) : PScanLine ;

begin
  GetBufferedScanLine := GetScanLine( ALineNo, LineBuffer ) ;
end ;


function  TDIB.GetScanColumn( AColumn : word ;
                              ABuffer : PScanLine ) : PScanLine ;

var y : Integer ;

begin
  if ABuffer <> nil then
    for y := 0 to Height - 1 do ABuffer^[ y ] := GetAPixel( AColumn, y ) ;
  GetScanColumn := ABuffer ;
end ;


function  TDIB.GetAPixel( x, y : word ) : longint ;

var p : PScanLine ;
    BytesInFirstSeg : word ;

begin
  { return the value of the pixel at (x,y) }
  { check bounds }
  if ( x < Width ) and ( y < Height ) then begin

    { get a pointer to the start of the line }
    p := ScanLines^[ y ] ;

    { if the scan line does not span a segment boundary then PoP }
    if p <> nil then GetAPixel := p^[ x ] else begin

      { calculate the address of the start of the line }
      p := ScanLines^[ y + 1 ] ;  { '+' because DIB's are bottom-up }
      inc( LongRec( p ).Lo, ScanWidth ) ;

      { find out which segment the pixel is in }
      BytesInFirstSeg := 65536 - LongRec( p ).Lo ;

      { if in first segment then PoP }
      if x < BytesInFirstSeg then GetAPixel := p^[ x ] else begin

        { calculate pointer }
        inc( LongRec( p ).Hi, SelectorInc ) ;
        LongRec( p ).Lo := 0 ;

        { and return value }
        GetAPixel := p^[ x - BytesInFirstSeg ] ;
      end ;
    end ;
  end else GetAPixel := 0 ;
end ;


function  TDIB.CreateHBitmap( DC : HDC ) : HBitmap ;

begin
  { the palette should have been selected in already!! }
  CreateHBitmap := CreateDIBitmap( DC, BitmapInfo^.bmiHeader,
                                   CBM_INIT, GlobalLock( DIBBits ),
                                   BitmapInfo^, DIB_RGB_COLORS ) ;
  GlobalUnlock( DIBBits ) ;
end ;


{ DIB loading stuff }

procedure AHIncr; far; external 'KERNEL' index 114;


procedure GetBitmapData(var TheFile: File;
                            BitsHandle : THandle ;
                            BitsByteSize : Longint);
type
  LongType = record
    case Word of
      0 : ( Ptr : Pointer ) ;
      1 : ( Long: Longint ) ;
      2 : ( Lo : Word ;
	          Hi : Word ) ;
  end;

var
  Count : Longint ;
  Start, ToAddr, Bits : LongType ;

begin
  Start.Long := 0 ;
  Bits.Ptr := GlobalLock( BitsHandle ) ;
  Count := BitsByteSize - Start.Long ;
  while Count > 0 do
  begin
    ToAddr.Hi := Bits.Hi + ( Start.Hi * Ofs( AHIncr ) );
    ToAddr.Lo := Start.Lo ;
    if Count > $4000 then Count := $4000 ;
    BlockRead( TheFile, ToAddr.Ptr^, Count ) ;
    Start.Long := Start.Long + Count ;
    Count := BitsByteSize - Start.Long ;
  end;
  GlobalUnlock( BitsHandle ) ;
end;


function  LoadDIB( AFileName : PChar ;
                   var BitmapInfo : PBitmapInfo ;
                   var BitmapInfoSize : word ;
                   var DIBBits        : THandle ) : boolean ;

var F : file ;
    BitmapFileHeader : TBitmapFileHeader ;
    BitmapInfoHeader : TBitmapInfoHeader ;
    BytesRead        : word ;
    nColourEntries   : word ;
    Bytes            : longint ;

begin
  BitmapInfo := nil ;
  BitmapInfoSize := 0 ;
  DIBBits := 0 ;
  LoadDib := false ;

  { open file and bitmap file header }
  assign( F, AFileName ) ; reset( F, 1 ) ;
  if IOResult = 0 then begin
    BlockRead( F, BitmapFileHeader, sizeof( BitmapFileHeader ), BytesRead ) ;
    if BytesRead = sizeof( BitmapFileHeader ) then with BitmapFileHeader do begin

      { check format etc. }
      if bfType = $4D42 then begin  { 'BF' }

        { read bitmap info header }
        BlockRead( F, BitmapInfoHeader, sizeof( BitmapInfoHeader ), BytesRead ) ;
        if BytesRead = sizeof( BitmapInfoHeader ) then with BitmapInfoHeader do begin

          { check colours & compression to make sure we can handle it }
          if ( biBitCount in [ 1, 4, 8 ] ) and ( biCompression = bi_RGB ) then begin

            { update image size }
            biSizeImage := ( ( ( biWidth * biBitCount ) + 31 ) div 32 ) * 4 * biHeight ;

            { calculate size of BitmapInfo & create }
            nColourEntries := biClrUsed ;
            if nColourEntries = 0 then nColourEntries := 1 shl biBitCount ;
            BitmapInfoSize := sizeof( BitmapInfo^ ) - sizeof( BitmapInfo^.bmiColors ) +
                              ( 1 shl biBitCount ) * sizeof( TRGBQuad ) ;
            GetMem( BitmapInfo, BitmapInfoSize ) ;

            if BitmapInfo <> nil then with BitmapInfo^ do begin

              { copy bitmap info header and load colour table }
              bmiHeader := BitmapInfoHeader ;
              Bytes := BitmapInfoSize - sizeof( bmiHeader ) {- sizeof( bmiColors )} ;
              BlockRead( F, bmiColors, Bytes, BytesRead ) ;
              if BytesRead = Bytes then begin

                { allocate handle for bitmap bits & load from file }
                DIBBits := GlobalAlloc( gmem_Moveable or gmem_Zeroinit, biSizeImage ) ;
                GetBitmapData( F, DIBBits, biSizeImage ) ;
                LoadDIB := true ;
              end ;
            end ;
          end ;
        end ;
      end ;
    end ;
    close( F ) ;
  end ;
end ;


end.
