{$A-,B-,D+,E+,F+,I+,L+,N-,O+,R+,S+,V-}

(***********************************************************************
       Data Base Objects for use with B-Tree Filer from Turbo Power
                  New Communications Technology, Inc.
                             Version 1.0
                          by John Poindexter
************************************************************************)

(*  This is a unit under development that is designed to use B-Tree Filer
    and OOP Professional to create some objects that are useful for data
    base management within an application that must handle various data bases
    of differing structures.

    There are some restrictions on the layout of the data base.

    The maximum number of fields in the data base record is limited to 50,
    but may easily be changed with the MaxDBFields constant.

    The fields that are to be keys to the data base must begin just after
    the delete field and run consecutively.  For example:

                       ADbaseRec = record
                         del    : longint;
                         field1 : string;       {key1}
                         field2 : string;       {key2}
                         field3 : string;       {key3}
                         field4 : string;
                       end;

    These restrictions could be relaxed with some added complexity.

    A big shortcoming at this point is the display of error and other
    messages.  Work will be done on this area.

    Many other methods are to be added.

    CAUTION:  You must define a descendant of Dbase which adds a field
    referencing the actual data structure of the data base records.  You
    may not declare an instance of Dbase.  You must also initialize the
    array of instances of dbIndex that you declare.
*)

(************************************************************************)

Unit ULDbase;

Interface

Uses OpString, OpInline, OpCrt, OpRoot, OpDate, OpEntry, Filer,
     Reorg, Rebuild, NumKeys;

const
  MaxDBFields = 50;

type
             {  0         1        2        3        4    }
  DataType = (ByteTyp, WordTyp, IntTyp,  LongTyp, BoolTyp,
             {  5         6        7        8        9    }
              CharTyp, StrTyp,  EnumTyp, TimeTyp, DateTyp);

  String8 = string[8];

(************************************************************************
  The dbIndex object describes the structure of the data base records
  and provides methods for returning the field values.
************************************************************************)

  dbIndexPtr = ^dbIndex;
  dbIndex = object(Root)
    ofs : word;            {offset into data structure for field}
    typ : DataType;        {type of data e.g. byte, integer, etc}
    len : byte;            {length of field - does not include 0 byte of string}
    dup : boolean;         {if field is to be a key, are duplicates allowed}
    nam : String8;         {name of field}
    dsp : boolean;         {is field to be displayed in Browse}
    xmt : boolean;         {is field to be transmitted over network}
    res : record end;      {for extensibility}
    constructor Init(deofs:word; detyp:DataType; delen:byte; dedup:boolean;
                     denam:String8; dedsp:boolean; dexmt:boolean);
    destructor Done; virtual;
    function GetChar(var DatS): char;
    function GetStr(var DatS): string;
    function GetByte(var DatS): byte;
    function GetWord(var DatS): word;
    function GetInt(var DatS): integer;
    function GetLong(var DatS): longint;
  end;

  dbIndexArrayPtr = ^dbIndexArray;
  dbIndexArray = array[1..MaxDBFields] of dbIndex;

(************************************************************************
  Dbase is the ancestor of all data bases to be managed by the application.
  You should never use an instance of Dbase, but define a descendant that
  at least adds the actual data base record variable as a field.  The
  Entry Screen field must be initialized separately.
************************************************************************)

  DbasePtr = ^Dbase;
  Dbase = object(Root)
    dbOpen : boolean;
    dbName : IsamFileBlockName;
    dbTitle : string[20];
    dbIFBPtr : IsamFileBlockPtr;
    dbStruLen : word;
    dbNrFields : byte;
    dbNrKeys : byte;
    dbIndices : dbIndexArrayPtr;
    dbIID : IsamIndDescr;
    dbNrRecords : longint;
    dbRecordsAdded : longint;
    dbKeysToAdd : longint;
    dbX, dbY : byte;
    dbCurRecNr : longint;
    dbEntry : EntryScreen;
    dbCustomStru : record end;    {this field is used to ref the actual
                                   structure of descendant objects}
    constructor Init(Name, Title: string; StruLen: word; NrFields: byte;
                     NrIndices: byte; Indices: dbIndexArrayPtr);
    destructor Done; virtual;
    function BuildKey(var DatS; KeyNr : integer): IsamKeyStr; virtual;
    function ConvertRecord(var DatSOld; var DatSNew): boolean; virtual;
    procedure OpenDataBase;
    procedure MakeDataBase;
    procedure CloseDataBase;
    procedure GetRecord(RefNr: longint);
    procedure ReOrganize;
    procedure ShowRecord;
  end;

const
  CurrentDbasePtr : DbasePtr = nil;

function GlobalBuildKey(var DatS; KeyNr: integer): IsamKeyStr;
function GlobalConvertRecord(var DatSOld; var DatSNew): boolean;

(************************************************************************)

Implementation

(* global functions for interfacing with non-OOP routines *)

function GlobalBuildKey(var DatS; KeyNr: integer): IsamKeyStr;
begin
  GlobalBuildKey := CurrentDBasePtr^.BuildKey(DatS, KeyNr);
end;

function GlobalConvertRecord(var DatSOld; var DatSNew): boolean;
begin
  GlobalConvertRecord := CurrentDBasePtr^.ConvertRecord(DatSOld, DatSNew);
end;

(* dbIndex Object Methods *)

constructor dbIndex.Init(deofs:word;detyp:DataType;delen:byte;dedup:boolean;
                         denam:String8; dedsp:boolean; dexmt:boolean);
begin
  if not Root.Init then Fail;
  ofs := deofs;
  typ := detyp;
  len := delen;
  dup := dedup;
  nam := denam;
  dsp := dedsp;
  xmt := dexmt;
end;

function dbIndex.GetChar(var DatS): char;
var
  dsptr : pointer;
begin
  dsptr := AddWordToPtr(Normalized(@DatS), ofs);
  GetChar := Char(dsptr^);
end;

function dbIndex.GetStr(var DatS): string;
var
  dsptr : pointer;
begin
  dsptr := AddWordToPtr(Normalized(@DatS), ofs);
  GetStr := String(dsptr^);
end;

function dbIndex.GetByte(var DatS): byte;
var
  dsptr : pointer;
begin
  dsptr := AddWordToPtr(Normalized(@DatS), ofs);
  GetByte := Byte(dsptr^);
end;

function dbIndex.GetInt(var DatS): integer;
var
  dsptr : pointer;
begin
  dsptr := AddWordToPtr(Normalized(@DatS), ofs);
  GetInt := Integer(dsptr^);
end;

function dbIndex.GetWord(var DatS): word;
var
  dsptr : pointer;
begin
  dsptr := AddWordToPtr(Normalized(@DatS), ofs);
  GetWord := Word(dsptr^);
end;

function dbIndex.GetLong(var DatS): longint;
var
  dsptr : pointer;
begin
  dsptr := AddWordToPtr(Normalized(@DatS), ofs);
  GetLong := LongInt(dsptr^);
end;

destructor dbIndex.Done;
begin
  Root.Done;
end;

(* Dbase Object Methods *)

constructor Dbase.Init(Name, Title: string; StruLen: word; NrFields: byte;
                       NrIndices: byte; Indices: dbIndexArrayPtr);
var
  i : integer;
begin
  if not Root.Init then Fail;
  if GetPageStack(10000) = 0 then;
  if not IsamOk then
  begin
    WriteLn;
    WriteLn('Not enough memory available.');
    Done;
    Halt;
  end;
  dbOpen := False;
  dbCurRecNr := 0;
  dbName := Copy(Name, 1, 64);
  dbTitle := Copy(Title, 1, 20);
  dbStruLen := StruLen;
  dbNrFields := NrFields;
  dbNrKeys := NrIndices;
  dbIndices := Indices;
  for i := 1 to NrIndices do
  begin
    dbIID[i].KeyL := dbIndices^[i].len;
    dbIID[i].AllowDupK := dbIndices^[i].dup;
  end;
end;

function Dbase.BuildKey(var DatS; KeyNr : integer): IsamKeyStr;
begin
  if dbKeysToAdd = 0 then
  begin
    dbKeysToAdd := dbNrKeys * dbRecordsAdded;
    WriteLn;
    Write('Keys to Add:');
    dbX := WhereX;
    dbY := WhereY;
    WriteLn;
    WriteLn('Working...');
  end;
  if (KeyNr < 1) or (KeyNr > dbNrKeys) then Exit;
  case dbIndices^[KeyNr].typ of
    ByteTyp,
    BoolTyp,
    EnumTyp : BuildKey := Long2Str(dbIndices^[KeyNr].GetByte(DatS));
    WordTyp : BuildKey := WordToKey(dbIndices^[KeyNr].GetWord(DatS));
    IntTyp  : BuildKey := IntToKey(dbIndices^[KeyNr].GetInt(DatS));
    LongTyp : BuildKey := LongToKey(dbIndices^[KeyNr].GetLong(DatS));
    CharTyp : BuildKey := #1+dbIndices^[KeyNr].GetChar(DatS);
    StrTyp  : BuildKey := dbIndices^[KeyNr].GetStr(DatS);
    TimeTyp : BuildKey := TimeToSortString(Time(dbIndices^[KeyNr].GetLong(DatS)));
    DateTyp : BuildKey := DateToSortString(Date(dbIndices^[KeyNr].GetLong(DatS)));
  end;
  Dec(dbKeysToAdd);
  GoToXY(dbX,dbY);
  WriteLn(LeftPad(Long2Str(dbKeysToAdd), 7));
end;

function Dbase.ConvertRecord(var DatSOld; var DatSNew): boolean;
begin
  Move(DatSOld, DatSNew, dbStruLen);
  if dbRecordsAdded = 0 then
  begin
    Write('Records Added:');
    dbX := WhereX;
    dbY := WhereY;
  end;
  Inc(dbRecordsAdded);
  GoToXY(dbX,dbY);
  Write(LeftPad(Long2Str(dbRecordsAdded), 5));
  ConvertRecord := True;
end;

procedure Dbase.OpenDataBase;
begin
  if dbOpen then Exit;
  OpenFileBlock(dbIFBPtr, dbName);
  if not IsamOk then
  begin
    WriteLn(IsamErrorMessage(IsamError));
    Halt;
  end;
  dbOpen := True;
end;

procedure Dbase.MakeDataBase;
begin
  MakeFileBlock(dbIFBPtr, dbName, dbStruLen, dbNrKeys, dbIID);
  if not IsamOk then
  begin
    WriteLn(IsamErrorMessage(IsamError));
    Halt;
  end;
  dbOpen := True;
end;

procedure Dbase.CloseDataBase;
begin
  if not dbOpen then Exit;
  CloseFileBlock(dbIFBPtr);
  if not IsamOk then
  begin
    WriteLn(IsamErrorMessage(IsamError));
    Halt;
  end;
  dbOpen := False;
end;

procedure Dbase.GetRecord(RefNr: longint);
begin
  if not dbOpen then OpenDataBase;
  GetRec(dbIFBPtr, RefNr, dbCustomStru);
  if not IsamOk then WriteLn(IsamErrorMessage(IsamError))
  else dbCurRecNr := RefNr;
end;

procedure Dbase.ReOrganize;
begin
  dbRecordsAdded := 0;
  dbKeysToAdd := 0;
  WriteLn('Reorganizing '+dbName+' data base...');
  ReOrgFileBlock(dbName, dbStruLen, dbNrKeys, dbIID, dbStruLen,
                 @GlobalBuildKey, @GlobalConvertRecord);
  if not IsamOk then WriteLn(IsamErrorMessage(IsamError));
end;

procedure Dbase.ShowRecord;
{ should only be called after the first GetRecord }
var
  i : integer;
begin
  if dbCurRecNr = 0 then Exit;
  WriteLn(dbTitle+'  Record No.: ', dbCurRecNr);
  for i := 1 to dbNrFields do
  begin
    Write(Pad(dbIndices^[i].nam, 8)+': ');
    case dbIndices^[i].typ of
      ByteTyp,
      BoolTyp,
      EnumTyp : WriteLn(dbIndices^[i].GetByte(dbCustomStru));
      WordTyp : WriteLn(dbIndices^[i].GetWord(dbCustomStru));
      IntTyp  : WriteLn(dbIndices^[i].GetInt(dbCustomStru));
      LongTyp : WriteLn(dbIndices^[i].GetLong(dbCustomStru));
      CharTyp : WriteLn(dbIndices^[i].GetChar(dbCustomStru));
      StrTyp  : WriteLn(dbIndices^[i].GetStr(dbCustomStru));
      TimeTyp : WriteLn(TimeToTimeString('hh:mm:ss',Time(dbIndices^[i].GetLong(dbCustomStru))));
      DateTyp : WriteLn(DateToDateString('mm/dd/yy',Date(dbIndices^[i].GetLong(dbCustomStru))));
    end;
  end;
end;

destructor Dbase.Done;

begin
  if dbOpen then CloseDataBase;
  {dbEntry.Done;}
  ReleasePageStack;
  Root.Done;
end;

End.
