Program DMXAMORT;

{$V- }

(*
    This program uses the DMX unit to generate an amortization schedule.

    Two DMX objects are used:

        The SelectWindow object uses a single-record editor to input
        the financial parameters Year, Month, Amount, Rate, and Years
        to calculate the loan amortization sheet.
        SELECTOR uses this object-type.

        The AmortWindow object is a descendant of DMXviewer, so as to
        scroll through the table without allowing changes.  The DataAt
        virtual function is modified to calculate each record as it
        needs to be displayed.
        This is why the entire table is not kept in memory.
        AMORT uses this object-type.

    This program is meant only as a demonstration for various uses of
    the DMX objects.

    This program is designed for MONTHLY payments only.  BI-WEEKLY mortgages
    have a radically different schedule --although you may certainly modify
    this program to accommodate week numbers instead of month names.

 *)


uses   Dos, Crt, DMX2;


type   AmortRec     =  record
                         Year       : word;
                         MonthName  : string [12];
                         Principal  : real;
                         Interest   : real;
                         Balance    : real;
                       end;

       SelectRec    =  record
                         Year,Month : word;
                         Principal  : real;
                         Interest   : real;
                         Years      : word;
                       end;

       SelectWindow = object (DMXwindow)
                        Info        : SelectRec;
                        procedure PickIt;
                      end;

       AmortWindow  = object (DMXviewer)
                        linedata    :  AmortRec;
                        Total       :  real;
                        Rate        :  real;
                        Payment     :  real;
                        Years       :  word;
                        Periods     :  word;
                        FirstMonth  :  word;
                        FirstYear   :  word;
                        LastPayment :  word;
                        LastTotal   :  real;

                        function  DataAt (recnum : longint) : pointer;
                                  virtual;
                        procedure Amortize (prin,interest :real; ys,y,m :word);
                      end;


const  maintitle   =  ' year   month           principal        interest          balance    ';
       baseformat  =  ' WWWW  ____________ ($R,RRR,RRR.RR)($R,RRR,RRR.RR)($RR,RRR,RRR.RR)';

       spectitle   =  '  year   month       amount       rate    years ';
       specformat  =  '  WWWW |  WW   $rr,rrr,rrr.rr  %rr.rr |   WW  ';

       cr          =  #13;
       Esc         =  #27;
       F9          =  'C';

       Months      :  array [0..11] of string [12] =
                             ('January  ',
                              'February ',
                              'March    ',
                              'April    ',
                              'May      ',
                              'June     ',
                              'July     ',
                              'August   ',
                              'September',
                              'October  ',
                              'November ',
                              'December ');



var    Key,ext     :  char;
       DOScolors   :  word;          { original colors }

       SELECTOR    :  SelectWindow;  { selector object }
       AMORT       :  AmortWindow;   { viewer object }


  {  }


function  Radical (number,exponent : real) : real;
{ returns the value of number raised to the power of exponent }
begin
  Radical := exp (Ln (abs (number)) * abs (exponent))
end;


  {  }


function  AmortPMT (prin,interest,term : real) : real;
{ amount of the periodic payment on the loan }
begin
  AmortPMT := prin * (interest / (1 - (1 / (Radical (1 + interest,term)))));
end;


  {  }


function  AmortWindow.DataAt (recnum : longint) : pointer;
{ The original descendant of this method returns a pointer }
{ to where the record would be stored in memory.           }
{ This virtual method pretends to retrieve the next record }
{ by calculating the record contents in LINEDATA.          }
{ The function then returns a pointer to LINEDATA.         }
{ RECNUM is actually the payment number, minus zero.       }
var  i,j  : word;
begin
  With linedata do
    begin
    If LastPayment <> recnum then
      begin
      If LastPayment > recnum then
        begin
        LastPayment := 0;
        LastTotal   := Total;
        end
       else
        Inc (LastPayment);
      For i := LastPayment to recnum do
        begin
        Interest   :=  Rate * LastTotal;
        Principal  :=  Payment - Interest;
        LastTotal  :=  LastTotal - Principal;
        If recnum = pred (recordlimit) then
          Balance := 0.0
         else
          Balance :=  LastTotal;
        end;
      LastPayment := recnum;
      end;
    Year       := ((recnum + FirstMonth) div 12) + FirstYear;
    MonthName  :=  Months [(recnum + FirstMonth) mod 12];
    end;

  DataAt := addr (linedata);
end;


  {  }


procedure AmortWindow.Amortize (prin,interest : real;  ys,y,m : word);
{ This method initializes the amortization variables, and then runs EditData. }
var  void : byte;
begin
  FirstYear   :=  y;
  FirstMonth  :=  m;

  Total       :=  prin;
  Rate        :=  interest / 100 / 12;
  Years       :=  ys;
  Periods     :=  Years * 12;
  LastPayment :=  32000;

  Payment := AmortPMT (Total,Rate,Periods);

  Window (6,2,75,24);
  WindBorder (LightCyan);

  TextAttr := bordercolor;
  ClrScr;
  GotoXY (2,23);
  write ('Your monthly payment is:   $', Payment:1:2);
  ClrPosition;  { reset current record pointer }
  OpenBuffer (void, sizeof (AmortRec) * Periods);

  EditData (void, Key,ext, [Esc],[]);
end;


  {  }


procedure SelectWindow.PickIt;
begin
  showzeroes  := True;  { make any zero amounts visible }
  With Info do
    begin
    Year      := 1990;
    Month     :=  1;
    Principal := 10000.00;
    Interest  := 10;
    Years     := 30;
    end;

  Repeat
    Window (17,9,64,13);
    TextAttr  := bordercolor;
    WindBorder (Yellow);
    ClrScr;
    GotoXY (10,5);
    write ('press F9 to calculate mortgage');

    OpenBuffer (Info, sizeof (Info));
    EditData (Info, Key,ext, [Esc],[F9]);

    If (ext = F9) then With Info do
      begin
      If (Principal > 0) and (Interest > 0)
        and
         (Years > 1) and (Month > 0) and (Month <= 12)
       then
        begin
        AMORT.Amortize (Principal,Interest, Years,Year, Month - 1);
        Key := #0;
        end
       else
        begin
        Sound (1000);  Delay (100);  NoSound;
        end;
      end;
  Until Key = Esc;
end;  { PickIt }


  {  }


Begin
  DOScolors := TextAttr;  { save the original screen color }
  TextAttr  := LightGray;
  ClrScr;

  AMORT.Init (maintitle, baseformat, 2,2, $3B,$3F,$70);
  SELECTOR.Init (spectitle, specformat, 2,2, $3E,$3F,$70);

  SELECTOR.Pickit;

  TextAttr := DOScolors;  { close the program in the original colors }
  Window (1,1,80,25);
  ClrScr;
End.

