{$R-,S-,I-,V-,F-,B-}
{$M 12000 0 655360}

(*
Modification History:
  1.1 - Failed to replace 0000 topic numbers if xref was last on line
        (fix in AdjustXrefsOnS)
      - Increased MaxTopicName to 60
*)

program ModHelp;
  {-Compresses, renumbers topics, fills in missing topic xrefs}
uses
  Dos,
  OpString,
  OpRoot,
  OpDos;

const
  CRLF = ^D^A;
  MaxTopicName = 60;                       {Max len for topic name} {!!.01}
  MaxIncludeNest = 5;                      {Max include level}
  MaxTotalFiles = 100;                     {Max total files in help system}
  FileBuffSize = 4096;                     {Text buffer size}
  CommentMark = ';';                       {Comment mark}
  CommandMark = '!';                       {Command mark}
  TempFileName : String[80] = 'TMPHLP';    {Name part for all temp files}
  IndexMrk = ^D;                           {Index command mark}
  Digits = '0123456789';

  {Overriddable options}
  FirstNumber : Word = 1;                  {First new number assigned}
  TopicInc : Word = 1;                     {Increment between topics}
  IncludeInc : Word = 1;                   {Increment between include files}
  RenameToBak : Boolean = False;           {Flag for creating BAK files}
  DoList : Boolean = False;                {Flag for creating LST file}

type
  NameStr = String[MaxTopicName];
  String80 = String[80];
  String6 = String[6];

  {Topic object, created during scan}
  TopicObj = object(DoubleListNode)
    FNamePtr : Pointer;
    Number : Word;
    Name : String[MaxTopicName];
    NewNumber : Word;
    constructor Init(TopicNumber : Word; TopicName : NameStr);
    destructor Done; virtual;
  end;
  TopicObjPtr = ^TopicObj;

  {Keeps track of current input files}
  FileArray = array[0..MaxIncludeNest] of Text;
  FileBuff = array[1..FileBuffSize] of Byte;
  LineArray = array[0..MaxIncludeNest] of LongInt;

  {Keeps track of all input files processed}
  FileRecord = record
    InFName : String[80];
    OutFName : String[80];
    Renamed : Boolean;
  end;
  FileRecArray = array[1..MaxTotalFiles] of FileRecord;

var
  InF : FileArray;                   {Input files}
  OutF : FileArray;                  {Output files (mirror input)}
  InName : String80;                 {Current input file name}
  OutName : String80;                {Current output file name}
  InBuff : FileBuff;                 {Input text buffer}
  Files : FileRecArray;              {Tracks _all_ processed files}
  TotalFiles : Word;                 {File counter}
  CurrentFile : Word;                {Number of current file}

  TopicPtr : TopicObjPtr;            {Pointer to topic object node}
  TopicList : Doublelist;            {DoubleList of topic objects}

  TopicCnt : Word;                   {Count of all topics}
  IncLev : Word;                     {Current include nest level}
  LineNum : LineArray;               {Current input line number}
  TotLines : LongInt;                {Total number of lines}
  CurSect : Word;                    {Current topic section}
  HighestTopic : Word;               {Highest topic so far}
  NextNumber : Word;                 {Next new topic number to assign}

  S : String;                        {Misc.}
  Slen : Byte absolute S;
  Spos : Word;
  C : String;

  function NextNewNumber : Word;
    {-Returns next new topic number}
  begin
    Inc(NextNumber, TopicInc);
    NextNewNumber := NextNumber;
  end;

  constructor TopicObj.Init(TopicNumber : Word; TopicName : NameStr);
    {-Creates a topic node}
  begin
    DoubleListNode.Init;
    Number := TopicNumber;
    Name := TopicName;
    NewNumber := NextNewNumber;
    FnamePtr := nil;
  end;

  destructor TopicObj.Done;
    {-Removes a topic node}
  begin
    DoubleListNode.Done;
  end;

  function NewTopicNum(OldTopicNum : Word) : String6;
    {-Return (as a string) the replacement topic number}
  begin
    {Get the Head, then traverse the entire List}
    TopicPtr := TopicObjPtr(TopicList.Head);
    while TopicPtr <> nil do begin
      if TopicPtr^.Number = OldTopicNum then begin
        NewTopicNum := Long2Str(TopicPtr^.NewNumber);
        Exit;
      end else
        TopicPtr := TopicObjPtr(TopicList.Next(TopicPtr));
    end;
    NewTopicNum := '0';
  end;

  function FlushDosBuffers(var F) : Boolean;
    {-Flush DOS's buffers for the specified file}
  var
    Handle : Word absolute F;
    Regs : Registers;
  begin
    FlushDosBuffers := False;
    with Regs do begin
      {dupe the file handle}
      AH := $45;
      BX := Handle;
      MsDos(Regs);
      if Odd(Flags) then
        Exit;

      {close the duped file}
      BX := AX;
      AH := $3E;
      MsDos(Regs);
      if Odd(Flags) then
        Exit;
    end;
    FlushDosBuffers := True;
  end;

  function FileName(var F : Text) : String;
    {-Return name of file}
  var
    NLen : Byte;
  begin
    with TextRec(F) do begin
      NLen := 0;
      while Name[NLen] <> #0 do begin
        FileName[NLen+1] := Name[NLen];
        inc(NLen);
      end;
      FileName[0] := Char(NLen);
    end;
  end;

  procedure RemoveTmpFiles;
    {-Deletes all tmphelp.nnn files on error}
  var
    I : Integer;
    F : File;
  begin
    {Delete all tmp files}
    if TotalFiles > 0 then
      for I := 1 to TotalFiles do
        with Files[I] do begin
          Assign(F, OutFName);
          Erase(F);
          if IOResult <> 0 then
            {ignore errors}
        end;
  end;

  procedure Error(S : String);
    {-Displays error message and halts}
  begin
    WriteLn(^M'Error: '+ S);
    RemoveTmpFiles;
    Halt(1);
  end;

  procedure ErrorLine(Msg : string);
    {-Report error position and message}
  begin
    WriteLn(^M'ERROR: ', Msg);
    WriteLn('File: ', FileName(Inf[IncLev]));
    WriteLn('Line number: ', LineNum[IncLev]);
    WriteLn(S);
    RemoveTmpFiles;
    Halt(1);
  end;

  procedure WriteCopyright;
    {-Display copyright notice}
  begin
    WriteLn('TP 5.5 Help Modifier 1.0. Copyright (c) TurboPower Software 1989');
    WriteLn(#10#13);
  end;

  procedure FlushTextIn(var F : Text);
    {-Flush text file opened for reading}
  type
    LH = record L, H : Word; end;
  var
    Bytes : LongInt;
    Regs : Registers;
  begin
    with TextRec(F), Regs do begin
      Bytes := LongInt(BufPos)-BufEnd;
      if Bytes = 0 then
        Exit;

      {Position file pointer past last data used}
      AX := $4201;
      BX := Handle;
      CX := LH(Bytes).H;
      DX := LH(Bytes).L;
      MsDos(Regs);

      {Mark buffer empty}
      BufPos := 0;
      BufEnd := 0;
    end;
  end;

  procedure OpenInf(Name : String80);
    {-Open input file}
  begin
    if IncLev > 0 then
      FlushTextIn(InF[IncLev-1]);
    Assign(InF[IncLev], Name);
    Reset(InF[IncLev]);
    if IoResult <> 0 then
      Error(Name+' not found');
    LineNum[IncLev] := 0;
    Write(^M, CharStr(' ', 64), ^M, Pad(StUpcase(Name), 13), 0:5);
  end;

  procedure CloseInf;
    {-Close input file}
  begin
    WriteLn(^H^H^H^H^H, LineNum[IncLev]:5);
    Close(InF[IncLev]);
    inc(TotLines, LineNum[IncLev]);
    if IncLev > 0 then begin
      SetTextBuf(InF[IncLev-1], InBuff, FileBuffSize);
      Write(Pad(StUpcase(FileName(InF[IncLev-1])), 13), LineNum[IncLev-1]:5);
    end;
  end;

  function JustNamePart(PathName : PathStr) : String;
    {-Strips drive, path and extension}
  var
    S : String;
    HasDot : Byte;
  begin
    S := JustFileName(PathName);
    HasDot := Pos('.', S);
    if HasDot <> 0 then
      S := Copy(S, 1, HasDot-1);
    JustNamePart := S;
  end;

  procedure WriteHelp;
    {-Summarize usage}
  begin
    WriteLn('Usage: MODHELP [Options] InFile '^M^J);
    WriteLn('  An extension of .TXT is assumed for InFile.');
    WriteLn;
    WriteLn('Options:');
    WriteLn('  /T topicgap     amount to increment each topic (default = 1)');
    WriteLn('  /I includegap   amount to increment between include files (default = 1)');
    WriteLn('  /B   create backups (files TMPHELP.000 - TMPHELP.nnn)');
    WriteLn('  /F firstnumber  starting number for renumbering');
    WriteLn('  /L   write topic name/number report to InFile.LST');
    Halt(1);
  end;

  procedure UnknownCommand(Arg : String);
    {-Complain about bad command line}
  begin
    Error('Unknown command line argument '+StUpcase(Arg));
  end;

  procedure Initialize;
    {-Prepare for analysis of help file}
  var
    I : Word;
    Arg : string[127];
  begin
    TotalFiles := 0;
    InName := '';
    OutName := '';

    I := 0;
    while I <= ParamCount do begin
      Inc(I);
      Arg := ParamStr(I);
      if (Arg[1] = '/') or (Arg[1] = '-') then begin
        if Length(Arg) <> 2 then
          Error('Invalid command line option');
        case Upcase(Arg[2]) of
          'T' : if I >= ParamCount then
                  Error('topicgap number missing')
                else begin
                    Inc(I);
                    if not Str2Word(ParamStr(I), TopicInc) then
                      Error('topicgap not numeric');
                end;

          'I' : if I >= ParamCount then
                  Error('includegap number missing')
                else begin
                    Inc(I);
                    if not Str2Word(ParamStr(I), IncludeInc) then
                      Error('includegap not numeric');
                end;

          'F' : if I >= ParamCount then
                  Error('firstnumber number missing')
                else begin
                    Inc(I);
                    if not Str2Word(ParamStr(I), FirstNumber) then
                      Error('firstnumber not numeric');
                end;

          'B' : RenameToBak := True;

          'L' : DoList := True;
        else
          Error('Invalid command line option');
        end;
      end else if Length(InName) = 0 then
        InName := DefaultExtension(StUpcase(CleanPathName(Arg)), 'TXT')
      else if Length(OutName) = 0 then
        OutName := StUpcase(CleanPathName(Arg))
      else
        Error('Too many filenames specified');
    end;

    {Show help if we didn't understand the parms}
    if Length(InName) = 0 then
      WriteHelp;

    {Some more inits}
    WriteLn('Pass 1 ...........');
    IncLev := 0;
    OpenInf(InName);
    TopicList.Init;
    NextNumber := FirstNumber;
    Dec(NextNumber, TopicInc);
  end;

  function ForceExtension(PathName : PathStr; Ext : ExtStr) : String;
    {-Strips path and drive, forces Ext}
  var
    S : String;
    HasDot : Byte;
  begin
    S := JustFileName(PathName);
    HasDot := Pos('.', S);
    if HasDot <> 0 then
      S := Copy(S, 1, HasDot-1);
    ForceExtension := S + '.' + Ext;
  end;

  procedure SkipWhite;
    {-Advance Spos past white space}
  begin
    while (Spos <= Slen) and (S[Spos] <= ' ') do
      Inc(Spos);
  end;

  procedure ParseWord(var C : string; MaxLen : Byte);
    {-Parse next word from S, returning it in C}
  var
    Clen : Byte absolute C;
  begin
    SkipWhite;
    Clen := 0;
    while (Spos <= Slen) and (S[Spos] > ' ') and (S[Spos] <> CommentMark)
    do begin
      if Clen < MaxLen then begin
        Inc(Clen);
        C[Clen] := S[Spos];
      end;
      Inc(Spos);
    end;
  end;

  function ParseNumber(Name : string) : Word;
    {-Parse a word from the line}
  var
    C : string[8];
    N : Word;
  begin
    ParseWord(C, 8);
    if Length(C) = 0 then
      ErrorLine(Name+' expected');
    if not Str2Word(C, N) then
      ErrorLine('Invalid '+Name+' specified');
    ParseNumber := N;
  end;

  function ParseLongInt(Name : string) : LongInt;
    {-Parse a word from the line}
  var
    C : string[12];
    N : LongInt;
  begin
    ParseWord(C, 12);
    if Length(C) = 0 then
      ErrorLine(Name+' expected');
    if not Str2Long(C, N) then
      ErrorLine('Invalid '+Name+' specified');
    ParseLongInt := N;
  end;

  function ClassifyCommand(C : string) : Word;
    {-Classify valid help metacommands}
  const
    NumCommands = 12;
    CommandNames : array[1..NumCommands] of string[5] =
    ('TOPIC', 'LINE', 'PAGE', 'WIDTH', 'INDEX', 'NOIND',
     'INCLU', 'WRAP', 'NOWRA', 'SCROL', 'BIAS', 'NOSEA');
  var
    I : Integer;
  begin
    C := StUpcase(Copy(C, 1, 5));
    for I := 1 to NumCommands do
      if C = CommandNames[I] then begin
        ClassifyCommand := I;
        Exit;
      end;
    ClassifyCommand := 0;
  end;

  procedure ReadTextLine;
    {-Read next line from help text}
  begin
    Inc(LineNum[IncLev]);
    ReadLn(InF[IncLev], S);
    if IoResult <> 0 then
      ErrorLine('Error reading from '+InName);
    if Slen = 0 then
      S[1] := #0;
    if LineNum[IncLev] and $0F = 0 then
      Write(^H^H^H^H^H, LineNum[IncLev]:5);
  end;

  procedure WriteTextLine;
    {-Write the current text line}
  begin
    WriteLn(OutF[IncLev], S);
    if IoResult <> 0 then
      ErrorLine('Error writing to temporary output file');
  end;

  procedure BuildTopicList;
    {-Scan input file once to build topic list}
  var
    Cnt : Word;
    Ch : Char;
    IncName : String80;
  begin
    SetTextBuf(InF[IncLev], InBuff, FileBuffSize);

    while not eof(InF[IncLev]) do begin
      ReadTextLine;
      case S[1] of
        CommandMark :      {Line is a help metacommand}
          begin
            Spos := 2;
            ParseWord(C, 8);
            case ClassifyCommand(C) of
              1 :          {TOPIC}
                begin
                  {New section, get section number}
                  CurSect := ParseNumber('Topic number');
                  if (LongInt(CurSect) <= 0) or
                     (LongInt(CurSect) > 65521) then
                    ErrorLine('Invalid topic number');

                  {Update topic number maximums}
                  if CurSect > HighestTopic then
                    HighestTopic := CurSect;

                  {Get optional topic name}
                  SkipWhite;
                  C := Copy(S, Spos, 80);

                  {Add this guy to the list}
                  New(TopicPtr, Init(CurSect, C));
                  if TopicPtr = nil then
                    Error('Out of memory');
                  TopicList.Append(TopicPtr);
                end;

              7 :          {INCLUDE}
                if IncLev = MaxIncludeNest then
                  Error('Too many nested files')
                else begin
                  {Include file, get filename}
                  ParseWord(IncName, 79);
                  inc(IncLev);
                  OpenInf(IncName);
                  Inc(NextNumber, IncludeInc);
                  Dec(NextNumber, TopicInc);
                  BuildTopicList;
                end;

              {Ignore other metacommands this pass}
            end;
          end;
        CommentMark :
          {Ignore comment lines}
          ;
      end;
    end;

    CloseInf;
    dec(IncLev);
  end;

  procedure OpenOutF(Name : String80);
    {-Open a new output file}
  var
    TmpExt : ExtStr;
    TmpS : String;
  begin
    Inc(CurrentFile);
    Inc(TotalFiles);
    if CurrentFile > MaxTotalFiles then
      Error('Too many files');
    TmpExt := LeftPadCh(Long2Str(CurrentFile), '0', 3);
    TmpS := AddBackSlash(JustPathName(Name)) + TempFileName + '.' + TmpExt;
    if IncLev > 0 then
      if not FlushDosBuffers(OutF[IncLev-1]) then
        Error('Failed to flush DOS buffers while switching files');
    Assign(OutF[IncLev], TmpS);
    ReWrite(OutF[IncLev]);
    if IOResult <> 0 then
      Error('Failed to create file ' + TmpS);

    {Add this file to the list}
    with Files[CurrentFile] do begin
      InFName := Name;
      OutFName := TmpS;
      Renamed := False;
    end;
  end;

  procedure CloseOutF;
    {-Close current output file}
  begin
    Close(OutF[IncLev]);
    if IOResult <> 0 then ;
    if IncLev > 0 then
  end;

  procedure InitForRewrite;
    {-prepare to create new files}
  begin
    WriteLn('Pass 2 ...........');
    IncLev := 0;
    OpenInF(InName);
    CurrentFile := 0;
    OpenOutF(InName);
  end;

  function LookupTopic(TopicName : String) : Word;
  begin
    {Get the Head, then traverse the entire List}
    TopicPtr := TopicObjPtr(TopicList.Head);
    while TopicPtr <> nil do begin
      if CompUCString(TopicPtr^.Name, TopicName) = Equal then begin
        LookupTopic := TopicPtr^.NewNumber;
        Exit;
      end else
        TopicPtr := TopicObjPtr(TopicList.Next(TopicPtr));
    end;
    LookupTopic := 0;
  end;

  procedure AdjustXrefsOnS;
  var
    OldTopicStr, TopicStr, NewS : String;
    TopicName : String;
    Topic : Word;
    I, J, K : Word;
    OriginalLength : Word;
  begin
    J := 0;
    I := 0;
    while I < Length(S) do begin
      {Transfer chars one at a time}
      Inc(I);
      Inc(J);
      NewS[J] := S[I];
      if NewS[J] = IndexMrk then begin

        {Get the topic number, convert to Word}
        K := 1;
        Inc(I);
        while Pos(S[I], Digits) <> 0 do begin
          TopicStr[K] := S[I];
          Inc(I);
          Inc(K);
          if I >= Length(S) then
            Error('Cross-reference straddles line break');
        end;
        Dec(I);
        OriginalLength := Pred(K);
        TopicStr[0] := Char(OriginalLength);
        OldTopicStr := TopicStr;
        if not Str2Word(TopicStr, Topic) then
          Error('Non-numeric topic number: ' + TopicStr);

        {If the current topic number is zero, try to look the correct one}
        if Topic = 0 then begin
          TopicName := TrimLead(Copy(S, I+1, 255));         {!!.01}
          TopicName := Trim(Copy(TopicName, 1, Pred(Pos(^E, TopicName))));
          Topic := LookupTopic(TopicName);
          TopicStr := Long2Str(Topic);
        end else
          {Get the new topic number}
          TopicStr := NewTopicNum(Topic);

        {Prevent changing an existing number to zero}
        if TopicStr = '0' then
          TopicStr := OldTopicStr;

        {Stuff the new topic number into NewS}
        if Length(TopicStr) < OriginalLength then
          TopicStr := LeftPadCh(TopicStr, '0', OriginalLength);
        Move(TopicStr[1], NewS[J+1], Length(TopicStr));
        Inc(J, Length(TopicStr));
      end;
    end;

    {Finished, move new string back into S}
    NewS[0] := Char(J);
    S := NewS;
  end;

  procedure RewriteFiles;
    {-read old files/write new ones (with updated topic numbers)}
  var
    Cnt : Word;
    Ch : Char;
    IncName : String80;
    OldS : String;
  begin
    SetTextBuf(InF[IncLev], InBuff, FileBuffSize);

    while not eof(InF[IncLev]) do begin
      ReadTextLine;
      case S[1] of
        CommandMark :      {Line is a help metacommand}
          begin
            Spos := 2;
            ParseWord(C, 8);
            case ClassifyCommand(C) of
              1 :          {TOPIC}
                begin
                  {New section, get section number}
                  CurSect := ParseNumber('Topicnumber');
                  if (LongInt(CurSect) <= 0) or
                     (LongInt(CurSect) > 65521) then
                    ErrorLine('Invalid topic number');

                  {Change to the new topic number}
                  S := '!TOPIC ' + NewTopicNum(CurSect) + Copy(S, Spos, 255);
                end;

              7 :          {INCLUDE}
                if IncLev = MaxIncludeNest then
                  Error('Too many nested files')
                else begin
                  {Include file, get filename}
                  ParseWord(IncName, 79);
                  {Process this include file}
                  inc(IncLev);
                  OpenInf(IncName);
                  OpenOutf(IncName);
                  OldS := S;
                  ReWriteFiles;
                  S := OldS;
                end;
            end;
          end;
        CommentMark :
          {Ignore comment lines}
          ;
        else
          if Pos(IndexMrk, S) <> 0 then
            AdjustXrefsOnS;
      end;
      WriteTextLine;
    end;

    CloseInf;
    CloseOutF;
    dec(IncLev);
  end;

  procedure RenameFiles;
    {-delete (or rename to bak) original files, rename new to original}
  var
    I : Word;
    F : File;
    BakName : String;
  begin
    for I := 1 to TotalFiles do
      with Files[I] do begin
        {Rename the input file to help???.bak}
        if RenameToBak then begin
          BakName := AddBackSlash(JustPathName(InFName)) + 'HELP' +
                     LeftPadCh(Long2Str(I), '0', 3) + '.BAK';
          Assign(F, BakName);
          Erase(F);
          if IOResult <> 0 then;
          Assign(F, InFName);
          Rename(F, BakName);
          if IOResult <> 0 then
            Error('Failed to rename file from '+InFName);
        end;

        {Rename the tmphlp.??? file to the input name}
        Assign(F, InFName);
        Erase(F);
        if IOResult <> 0 then;
        Assign(F, OutFName);
        Rename(F, InFName);
        if IOResult <> 0 then
          Error('Failed to rename file to '+InFName);
      end;
  end;

  procedure CreateListFile;
  var
    T : Text;
    T1, T2 : TopicObjPtr;
  begin
    Assign(T, AddBackSlash(JustPathName(InName)) + JustNamePart(InName) + '.lst');
    ReWrite(T);
    TopicPtr := TopicObjPtr(TopicList.Head);
    while TopicPtr <> nil do
      with TopicPtr^ do begin
        WriteLn(T, Pad(Name, MaxTopicName+2), NewNumber:8);
        TopicPtr := TopicObjPtr(TopicList.Next(TopicPtr));
      end;
    Close(T);
  end;

begin
  WriteCopyright;

  {Pass 1, build doublelist of topics}
  Initialize;
  BuildTopicList;

  {Pass 2, create new help text files}
  InitForRewrite;
  ReWriteFiles;
  RenameFiles;

  {Create unsorted (sequential topic order) report}
  if DoList then
    CreateListFile;
end.
