{$R-,S-,I-,V-,F-,B-}

program PrnHelp;
  {-Print a proof version of help file from input text}
uses
  Dos,
  OpDos,
  OpRoot,
  OpString,
  OpInline,
  OpCrt,
  OpWindow,
  OpPick,
  OpHelp;

const
  FileBuffSize = 4096;       {Size of input and output file buffers}
  CommandMark = '!';         {Marks help metacommand in text file}
  CommentMark = ';';         {At start of line, marks comment in text file}
  MaxIncludeNest = 1;        {Maximum depth of include nesting}
  MaxFontStack = 10;         {Maximum nesting of fonts}
  DefaultFontChar = #0;      {On font stack for default}

type
  FileBuff = array[1..FileBuffSize] of Byte;
  String80 = string[80];
  StringPtr = ^string;
  FileArray = array[0..MaxIncludeNest] of Text;
  LineArray = array[0..MaxIncludeNest] of LongInt;
  FontArray = array[DefaultFontChar..XrefToggle] of string[31];
  FontStack = array[1..MaxFontStack] of Char;

const
  {Set up for 'G' cartridge on LaserJet+}
  FontsOn : FontArray =
  (#27'(0U'#27'(s0p12h10v0s0b8T'#27'&d@',   {Def PrestigeElite}
   #27'(8U'#27'(s0p10h12v0s03T'#27'&d@',    {^A  Courier}
   #27'(0U'#27'(s0p12h10v0s3b8T'#27'&d@',   {^B  PrestigeBold}
   #27'(0U'#27'(s0p12h10v1s0b8T'#27'&d@',   {^C  PrestigeItalic}
   #27'(0U'#27'(s0p16.66h7v0s0b8T',         {^D  PrestigeElite 16.66}
   #27'(0U'#27'(s0p12h10v0s0b8T'#27'&dD'    {^E  PrestigeElite Underscore}
  );
  FontsOff : FontArray =
  ('',          {None required for 'G' cartridge of LaserJet+}
   '',
   '',
   '',
   '',
   ''
  );
  PrnReset = #27'E';         {Clear all fonts}
  PrnFormFeed = ^L^M;        {Formfeed}

var
  InName : String80;         {Input file name}
  OutName : String80;        {Output file name}
  InF : FileArray;           {Input files (with include files)}
  OutF : file;               {Output file}
  InBuff : FileBuff;         {Buffer for input text}
  OutBuff : FileBuff;        {Buffer for binary output}
  OutPos : Word;             {Bytes used in output buffer}
  IncLev : Word;             {Include nesting level}
  LineNum : LineArray;       {Current input line number}
  TotLines : LongInt;        {Total number of lines}
  Hdr : HelpHeader;          {Header of help file}
  TextWid : Byte;            {Max characters in a line}

  C : String80;              {Command or command parameter}
  Clen : Byte absolute C;    {Length of parameter}
  S : string;                {Raw input line}
  Slen : Byte absolute S;    {Length of input line}
  Spos : Byte;               {Position in input line}

  TopicBias : LongInt;       {Added to topic number references}
  CurSect : Word;
  LineLen : Byte;
  SectLen : Word;
  InXref : Boolean;
  Wrapping : Boolean;
  FormFed : Boolean;
  ShowXrefs : Boolean;       {True to print cross-references}

  Stack : FontStack;
  StackP : Word;

  procedure Error(Msg : string);
    {-Write error message and halt}
  begin
    WriteLn(^M'ERROR: ', Msg);
    Halt(1);
  end;

  function FileName(var F : Text) : String;
    {-Return name of file}
  var
    NLen : Byte;
  begin
    with TextRec(F) do begin
      NLen := 0;
      while Name[NLen] <> #0 do begin
        FileName[NLen+1] := Name[NLen];
        inc(NLen);
      end;
      FileName[0] := Char(NLen);
    end;
  end;

  procedure ErrorLine(Msg : string);
    {-Report error position and message}
  begin
    WriteLn(^M'ERROR: ', Msg);
    WriteLn('File: ', FileName(Inf[IncLev]));
    WriteLn('Line number: ', LineNum[IncLev]);
    WriteLn(S);
    Halt(1);
  end;

  procedure FlushTextIn(var F : Text);
    {-Flush text file opened for reading}
  type
    LH = record L, H : Word; end;
  var
    Bytes : LongInt;
    Regs : Registers;
  begin
    with TextRec(F), Regs do begin
      Bytes := LongInt(BufPos)-BufEnd;
      if Bytes = 0 then
        Exit;

      {Position file pointer past last data used}
      AX := $4201;
      BX := Handle;
      CX := LH(Bytes).H;
      DX := LH(Bytes).L;
      MsDos(Regs);

      {Mark buffer empty}
      BufPos := 0;
      BufEnd := 0;
    end;
  end;

  procedure OpenInf(Name : String80);
    {-Open input file}
  begin
    if IncLev > 0 then
      FlushTextIn(InF[IncLev-1]);
    Assign(InF[IncLev], Name);
    Reset(InF[IncLev]);
    if IoResult <> 0 then
      Error(Name+' not found');
    LineNum[IncLev] := 0;
    Write(^M, CharStr(' ', 64), ^M, Pad(StUpcase(Name), 13), 0:5);
  end;

  procedure CloseInf;
    {-Close input file}
  begin
    WriteLn(^H^H^H^H^H, LineNum[IncLev]:5);
    Close(InF[IncLev]);
    inc(TotLines, LineNum[IncLev]);
    if IncLev > 0 then begin
      dec(IncLev);
      SetTextBuf(InF[IncLev], InBuff, FileBuffSize);
      Write(Pad(StUpcase(FileName(InF[IncLev])), 13), LineNum[IncLev]:5);
    end;
  end;

  procedure Initialize;
    {-Prepare}
  var
    I : Word;
    Arg : string[127];
  begin
    InName := '';
    OutName := '';
    ShowXrefs := False;

    for I := 1 to ParamCount do begin
      Arg := ParamStr(I);
      if (Arg[1] = '/') or (Arg[1] = '-') then begin
        if Length(Arg) <> 2 then
          Error('Invalid command line option');
        case Upcase(Arg[2]) of
          'X' : ShowXrefs := True;
        else
          Error('Invalid command line option');
        end;
      end else if Length(InName) = 0 then
        InName := DefaultExtension(StUpcase(CleanPathName(Arg)), 'TXT')
      else if Length(OutName) = 0 then
        OutName := StUpcase(CleanPathName(Arg))
      else
        Error('Too many filenames specified');
    end;

    if Length(InName) = 0 then begin
      WriteLn('Usage: PRNHELP [Options] InFile [OutFile]'^M^J);
      WriteLn('  An extension of .TXT is assumed for InFile.');
      WriteLn('  If OutFile is not specified, MAKEHELP writes to InFile.PRN.');
      WriteLn;
      WriteLn('Options:');
      WriteLn('  /X  print cross-reference topic numbers');
      Halt(1);
    end;

    if Length(OutName) = 0 then
      OutName := ForceExtension(InName, 'PRN');

    if InName = OutName then
      Error('Input and output filenames must differ');

    Assign(OutF, OutName);
    Rewrite(OutF, 1);
    if IoResult <> 0 then
      Error('Cannot create '+OutName);

    with Hdr do begin
      {Default help header}
      FillChar(Hdr, SizeOf(HelpHeader), 0);
      ID := LongInt(HelpId);
      WindowWidth := 40;

      TextWid := WindowWidth-2; {Correct text dimensions for default spacing}
      TopicBias := 0;
      TotLines := 0;

      {Initialize counters}
      CurSect := 0;
      LineLen := 0;
      SectLen := 0;
      OutPos := 0;
      InXref := False;
      Wrapping := True;
      FormFed := False;

      IncLev := 0;
      OpenInf(InName);
    end;
  end;

  procedure ReadTextLine;
    {-Read next line from help text}
  begin
    Inc(LineNum[IncLev]);
    ReadLn(InF[IncLev], S);
    if IoResult <> 0 then
      ErrorLine('Error reading from '+InName);
    if Slen = 0 then
      S[1] := #0;
    if LineNum[IncLev] and $0F = 0 then
      Write(^H^H^H^H^H, LineNum[IncLev]:5);
  end;

  procedure SkipWhite;
    {-Advance Spos past white space}
  begin
    while (Spos <= Slen) and (S[Spos] <= ' ') do
      Inc(Spos);
  end;

  procedure ParseWord(var C : string; MaxLen : Byte);
    {-Parse next word from S, returning it in C}
  var
    Clen : Byte absolute C;
  begin
    SkipWhite;
    Clen := 0;
    while (Spos <= Slen) and (S[Spos] > ' ') and (S[Spos] <> CommentMark)
    do begin
      if Clen < MaxLen then begin
        Inc(Clen);
        C[Clen] := S[Spos];
      end;
      Inc(Spos);
    end;
  end;

  function ParseNumber(Name : string) : Word;
    {-Parse a word from the line}
  var
    C : string[8];
    N : Word;
  begin
    ParseWord(C, 8);
    if Length(C) = 0 then
      ErrorLine(Name+' expected');
    if not Str2Word(C, N) then
      ErrorLine('Invalid '+Name+' specified');
    ParseNumber := N;
  end;

  function ParseLongInt(Name : string) : LongInt;
    {-Parse a word from the line}
  var
    C : string[12];
    N : LongInt;
  begin
    ParseWord(C, 12);
    if Length(C) = 0 then
      ErrorLine(Name+' expected');
    if not Str2Long(C, N) then
      ErrorLine('Invalid '+Name+' specified');
    ParseLongInt := N;
  end;

  function ClassifyCommand(C : string) : Word;
    {-Classify valid help metacommands}
  const
    NumCommands = 12;
    CommandNames : array[1..NumCommands] of string[5] =
    ('TOPIC', 'LINE', 'PAGE', 'WIDTH', 'INDEX', 'NOIND',
     'INCLU', 'WRAP', 'NOWRA', 'SCROL', 'BIAS', 'NOSEA');
  var
    I : Integer;
  begin
    C := StUpcase(Copy(C, 1, 5));
    for I := 1 to NumCommands do
      if C = CommandNames[I] then begin
        ClassifyCommand := I;
        Exit;
      end;
    ClassifyCommand := 0;
  end;

  procedure BlockWriteChk(var B; Bytes : Word);
    {-Write a block to output and error check}
  var
    BytesWritten : Word;
  begin
    BlockWrite(OutF, B, Bytes, BytesWritten);
    if (IoResult <> 0) or (BytesWritten <> Bytes) then
      Error('Error writing to '+OutName);
  end;

  procedure FlushBuffer;
    {-Write the output buffer to file}
  begin
    if OutPos > 0 then begin
      BlockWriteChk(OutBuff, OutPos);
      OutPos := 0;
    end;
  end;

  procedure CharOut(Ch : Char);
    {-Write a single character}
  begin
    Inc(OutPos);
    OutBuff[OutPos] := Byte(Ch);
    if OutPos = FileBuffSize then
      FlushBuffer;
    Inc(SectLen);
    FormFed := False;
  end;

  procedure StringOut(S : String);
    {-Write a string of characters}
  var
    I : Byte;
  begin
    for I := 1 to Length(S) do
      CharOut(S[I]);
  end;

  procedure ResetPrinter;
    {-Reset the fonts}
  begin
    StringOut(PrnReset);
  end;

  procedure PushFont(FChar : Char);
    {-Activate a new font, pushing it onto the font stack}
  begin
    if StackP >= MaxFontStack then
      ErrorLine('Font stack overflow');
    inc(StackP);
    Stack[StackP] := FChar;
    StringOut(FontsOn[FChar]);
  end;

  procedure PopFont;
    {-Remove top font from font stack and activate previous font}
  begin
    if StackP >= 1 then
      {Disable the current font}
      StringOut(FontsOff[Stack[StackP]]);
    dec(StackP);
    if StackP < 1 then
      ErrorLine('Font stack underflow');
    {Re-enable the font at top of stack}
    StringOut(FontsOn[Stack[StackP]]);
  end;

  procedure InitializePrinter;
    {-Clear font stack, push default font}
  begin
    StackP := 0;
    {Activate default font}
    PushFont(DefaultFontChar);
  end;

  procedure NewSection;
    {-End the current section and prepare for the new}
  begin
    if LineLen > 0 then
      StringOut(^M^J);
    StringOut(CharStr('-', TextWid));
    StringOut(^M^J);
    SectLen := 0;
    LineLen := 0;
  end;

  procedure NewPage;
    {-End the current page}
  begin
    StringOut(^M^J);
    LineLen := 0;
  end;

  procedure NewLine;
    {-End the current line}
  begin
    if InXref then
      {Line break in xref}
      ;

    {Keep track of longest line}
    if LineLen > Hdr.MaxLineLen then
      Hdr.MaxLineLen := LineLen;

    StringOut(^M^J);
    LineLen := 0;
  end;

  function LenCount(Ch : Char) : Byte;
    {-Return length to count for character}
  begin
    case Ch of
      Attr1Toggle..XrefToggle :
        LenCount := 0;
    else
      LenCount := 1;
    end;
  end;

  procedure WordOut(var Spos : Byte; Tpos : Byte);
    {-Write line starting at Spos and continuing to Tpos}
  var
    Topic : Word;
    Code : Word;
    Ch : Char;
    Finished : Boolean;
  begin
    while Spos < Tpos do begin
      Ch := S[Spos];
      case Ch of
        IndexMarker : {Convert cross-reference topic number to binary}
          begin
            {Collect the topic number}
            Clen := 0;
            repeat
              Inc(Spos);
              Inc(Clen);
              C[Clen] := S[Spos];
            until (Spos >= Tpos) or (S[Spos] = XrefToggle);
            Dec(Clen);
            Val(C, Topic, Code);

            if Code <> 0 then
              ErrorLine('Invalid cross-reference topic number');
            {Bias the topic number}
            inc(Topic, TopicBias);
            if Topic = CurSect then
              {Topic cross-references itself}
              ;

            if ShowXrefs then begin
              PushFont(IndexMarker);
              StringOut(C);
              PopFont;
            end;
          end;

        LiteralMarker : {Output literal character following}
          begin
            Clen := 0;
            Finished := False;
            repeat
              Inc(Spos);
              Inc(Clen);
              C[Clen] := S[Spos];
              if Spos >= Tpos then
                Finished := True
              else
                case S[Spos] of
                  '0'..'9' : ;
                else
                  Finished := True;
                end;
            until Finished;
            Dec(Clen);
            Val(C, Topic, Code);
            if (Code <> 0) or (Topic > 255) then
              ErrorLine('Invalid literal character');
            CharOut(Char(lo(Topic)));
            Inc(LineLen);
          end;

        Attr1Toggle,
        Attr2Toggle,
        Attr3Toggle,
        XrefToggle :
          begin
            if Stack[StackP] = Ch then
              PopFont
            else
              PushFont(Ch);
            if Ch = XrefToggle then
              InXref := not InXref;
            Inc(Spos);
          end;

      else
        CharOut(Ch);
        Inc(LineLen);
        Inc(Spos);
      end;
    end;
  end;

  procedure LineOut;
    {-Wrap and write text lines}
  var
    Tpos : Byte;
    Tlen : Byte;
  begin
    if not Wrapping then begin
      {Write entire line without wrapping}
      SPos := 1;
      WordOut(SPos, Slen+1);
      NewLine;
      Exit;
    end;

    if Slen = 0 then begin
      {Empty line, finish previous line}
      if LineLen > 0 then
        NewLine;
      {Insert blank line}
      NewLine;
      Exit;
    end;

    {Non-empty line}
    if (S[1] = ' ') then
      {Finish previous line}
      if LineLen > 0 then
        NewLine;

    Spos := 1;
    repeat

      {Write white space}
      while (Spos <= Slen) and (S[Spos] = ' ') do begin
        if LineLen < TextWid then begin
          CharOut(' ');
          Inc(LineLen);
        end;
        Inc(Spos);
      end;
      if Spos > Slen then
        Exit;

      {See if next word fits on line}
      Tpos := Spos;
      Tlen := 0;
      repeat
        case S[Tpos] of
          IndexMarker : {Skip over the cross-reference topic number}
            repeat
              Inc(Tpos);
            until (Tpos > Slen) or (S[Tpos] = XrefToggle);

          LiteralMarker : {Skip over the character number}
            begin
              repeat
                Inc(Tpos);
              until (Tpos > Slen) or (S[Tpos] < '0') or (S[Tpos] > '9');
              Inc(Tlen);
            end;

        else
          Inc(Tlen, LenCount(S[Tpos]));
          Inc(Tpos);
        end;
      until (Tpos > Slen) or (S[Tpos] = ' ');

      if LineLen+Tlen > TextWid then
        {Word won't fit on line, start a new one}
        NewLine;

      {Write the word}
      WordOut(Spos, TPos);

    until Spos > Slen;

    {End line with blank}
    if LineLen < TextWid then begin
      CharOut(' ');
      Inc(LineLen);
    end;
  end;

  procedure ScanFile;
    {-Scan input file to create help text}
  var
    IncName : String80;
  begin
    with Hdr do begin
      LineNum[IncLev] := 0;
      SetTextBuf(InF[IncLev], InBuff, FileBuffSize);

      while not eof(InF[IncLev]) do begin
        ReadTextLine;
        case S[1] of
          CommandMark :      {A help metacommand}
            begin
              Spos := 2;
              ParseWord(C, 8);
              case ClassifyCommand(C) of
                1 :          {TOPIC}
                  begin
                    if CurSect <> 0 then
                      {Complete previous section}
                      NewSection;
                    {Clear font stack, reset printer, set up default font}
                    InitializePrinter;

                    {Get section number}
                    CurSect := ParseNumber('Topic number');
                    {Bias the topic number}
                    inc(CurSect, TopicBias);
                    {Get optional topic name}
                    SkipWhite;
                    C := Copy(S, Spos, 80);

                    PushFont(IndexMarker);
                    StringOut(Long2Str(CurSect));
                    PopFont;
                    PushFont(Attr1Toggle);
                    StringOut(' '+C);
                    PopFont;
                    StringOut(^M^J+CharStr('-', TextWid)+^M^J);
                  end;

                2 :          {LINE}
                  NewLine;

                3 :          {PAGE}
                  NewPage;

                4 :          {WIDTH}
                  if CurSect <> 0 then
                    ErrorLine('WIDTH statement must precede first help topic')
                  else begin
                    {Parse width}
                    WindowWidth := ParseNumber('Width');
                    {Correct dimension for default spacing}
                    TextWid := WindowWidth-2;
                  end;

                5 :          {INDEX}
                  if CurSect = 0 then
                    ErrorLine('INDEX statement must follow TOPIC statement')
                  else if ParseNumber('Index number') = 0 then
                    ;

                6 :          {NOINDEX}
                  if CurSect = 0 then
                    ErrorLine('NOINDEX statement must follow TOPIC statement');

                7 :          {INCLUDE}
                  if IncLev = MaxIncludeNest then
                    Error('Too many nested files')
                  else begin
                    {Include file, get filename}
                    ParseWord(IncName, 79);
                    inc(IncLev);
                    OpenInf(IncName);
                    ScanFile;
                  end;

                 8 :         {WRAP}
                   Wrapping := True;

                 9 :         {NOWRAP}
                   Wrapping := False;

                10 :         {SCROLL}
                   Scrolling := True;

                11 :         {BIAS}
                  begin
                    TopicBias := ParseLongInt('Topic bias');
                    if (TopicBias < -65000) or (TopicBias > 65000) then
                      ErrorLine('Topic bias out of range');
                  end;

                12 :         {NOSEARCH}
                  if CurSect = 0 then
                    ErrorLine('NOSEARCH statement must follow TOPIC statement');
              else
                ErrorLine('Unrecognized directive');
              end;
            end;
          CommentMark :
            {Ignore line} ;
        else
          {A text line - wrap and output}
          LineOut;
        end;
      end;

      CloseInf;
      if not FormFed then begin
        StringOut(PrnFormFeed); {End file with form feed}
        FormFed := True;
      end;
    end;
  end;

  procedure ScanDone;
    {-Clean up when scan pass is done}
  begin
    with Hdr do begin
      WriteLn(^M, TotLines:6, ' total lines input');
      ResetPrinter;
      {Assure output goes to disk}
      FlushBuffer;
      Close(OutF);
    end;
  end;

begin
  {Initialize globals and parse command line}
  Initialize;

  {Initialize printer}
  ResetPrinter;
  InitializePrinter;

  ScanFile;
  ScanDone;
end.
