{$A+,B-,D+,E+,F-,G-,I-,L+,N-,O-,R-,S-,V+,X+}
{$M 4096,0,1024}
Unit TmsTimer;
INTERFACE

{--------------------------------------------------------------------------
  A timer can be used for 2 puposes:

  1) To measere the time it takes for a process. ( Interval function )
     It returns the time elapsed since the last time it was called.

  2) As an alarm clock. The alarm clock works by setting the time (offset
     and start time) and the delay at which it runs out. The alarm must be
     constantly polled (tested) to see if has run out yet. If the alarm has
     run out the test procces calls a virtual timeout procedure, which by
     default resets the alarm, starting the process all over again.

  All time values are specified in 1/10 millisecond (Tms).

  Misc Unit functions:
  Tms time valus can be formated with the TimeSplit and TimeStr functions.
  GetTmsTime retrives the current PC time to 1/10 of a millisecond.
--------------------------------------------------------------------------}

CONST T24h  = 864000000;  { Number of [mSec/10] Tics in a PcDay           }
			  { Rem this is 2.2ms  short of a real day        }
      tmNewOffset= True;  { Use it when calling the start procedures	  }

TYPE
  pTimer= ^tTimer;
  tTimer= Object
    UseTimeOutProc: Boolean;

    Constructor Init(Tms:LongInt);      { & set Delay in MSec/10          }
    Destructor  Done;			Virtual;
    { time measurement }
    Function  Interval: LongInt;	{ MSec/10 Current- last call time }

    { Functions to set alarm clock vars }
    Procedure SetOfs(Tms:LongInt);	{ Offset in MSec/10 PC Time       }
    Procedure SyncOfs(T: pTimer);	{ Synchronize 2 timer ofsets      }
    Procedure SetDelay(Tms: LongInt);   { Timer Delay in MSec/10	  }

    { Alarm clock start functions }
    Function  StartNow(NewOfset: Boolean): LongInt;  { Start immidiatly       }
    Function  StartIn(Tms:LongInt; NewOfset: Boolean):LongInt;  { Start at Ofset+ Tms, Return absolute PC start time }
    Function  StartAt(Tms:LongInt; NewOfset: Boolean):LongInt;  { Start at PC MSec/10, Return start Time- Offset     }
    Procedure StartAtEnd(NewOfset: Boolean);         { Periode= Delay 	      }
    Procedure StartAtTest(NewOfset:Boolean);         { Synch with last Test   }

    { Alarm clock timeout functions }
    Function  TimeOut:     Boolean;	{ Is current time>= TimeOut, Sets TestTime var }
    Procedure TimeOutProc; Virtual;	{ Default= StartAtEnd	}

    { Functions returning internal alarm clock vars }
    Function  TOfset:	LongInt;	{ MSec/10 Offset in absulute PC Time  }
    Function  Time:	LongInt;	{ MSec/10 Current time- Offset	      }
    Function  TStart:	LongInt;	{ MSec/10 Timer start - Offset	      }
    Function  TElapsed:	LongInt;	{ Msec/10 Current time- Timer start   }
    Function  TRemain:	LongInt;	{ MSec/10 Timeout- current time       }
    Function  TTest:	LongInt;	{ MSec/10 last Test call - Offset     }
    Function  TEnd: 	LongInt;	{ MSec/10 next Timeout- Offset        }
    Function  TLastEnd: LongInt;	{ MSec/10 Last Timeout- Offset        }

    Private
     NewDay  :  Boolean;	{ True if Timout is in next 24H period }
     OfsetT,			{ Timer Offset	}
     StartT,			{ Timer Start	}
     TestT,			{ Last TimeOut Test }
     EndT,			{ Next Timeout	}
     LastEndT,    		{ Last Timeout	}
     LastT,			{ last Interval }
     DelayT  : LongInt;

     StartN1, { Round correction vars for StartAtEnd,SetDelay }
     StartN2,
     Err_N1,
     Err_N2  : Word;
     Err_Val1,
     Err_Val2: ShortInt; { is set +/- 1 }
   END;

Function  TmsTime : LongInt;  { Get time in 1/10 Msec of PC clock }
Procedure TmsSplit(Tms : LongInt; VAR H,M,S,T: Integer);
Function  TmsStr(Tms: LongInt): String;
Function  TmsAdd(T1,T2: LongInt): LongInt;
Function  TmsSub(T1,T2: LongInt): LongInt;
Function  TmsCmp12H(T1,T2: LongInt): LongInt;



IMPLEMENTATION
USES DOS;                                 

CONST TFreq = 3.7286875;  { (1193180 MHz/10000 ms/s) SHR 5= Freq at bit 5 }
      PCDay = $C0058000;  { =$1800B0,0000 Tics that PC thinks is in 1 Day }
 TmsOverflow= 575935539.5;{ =maximum internal time value in Msec/10=~16H  }

VAR   Wrap24: Boolean;



Procedure TmsSplit(Tms: LongInt; VAR H,M,S,T: Integer);
BEGIN
Tms:= ABS(Tms);
H:= Trunc(Tms/36000000);
M:= Trunc(Tms/600000-H*60);
S:= Trunc(Tms/10000-60*(M+60*H));
T:= Tms- 10000*(S+60*(M+60*H));
END;


Function TmsStr(Tms: LongInt): String;
VAR   H,M,S,T: Integer;
      HH,MM,SS: String[2];
      TT: String[4];
BEGIN
TmsSplit(Tms,H,M,S,T);
Str(H:2,HH);  IF HH[1]=' ' THEN HH[1]:='0';
Str(M:2,MM);  IF MM[1]=' ' THEN MM[1]:='0';
Str(S:2,SS);  IF SS[1]=' ' THEN SS[1]:='0';
Str(T:4,TT);  IF TT[1]=' ' THEN TT[1]:='0';
	      IF TT[2]=' ' THEN TT[2]:='0';
	      IF TT[3]=' ' THEN TT[3]:='0';
IF Tms>=0
  THEN TmsStr:= ' '+HH+':'+MM+' '+SS+'.'+TT
  ELSE TmsStr:= '-'+HH+':'+MM+' '+SS+'.'+TT;
END;



{---------------------------------------------------------------------------}
{ Convert Internal time format into 1/10 Milisecond format		    }
{ The results range= [0..24H], the maximum internal range		    }
{---------------------------------------------------------------------------}
Function  TmSec(T: LongInt):Longint;
Type LongRec= Record Lo,Hi: Word END;
BEGIN
IF T>=0			      { Rem internal format is negative after 16H }
  THEN TmSec:= Round(T/TFreq)
  ELSE BEGIN
  LongRec(T).hi:= LongRec(T).Hi and $7FFF;
  TmSec:= Round(TmsOverFlow+T/TFreq);	{ Add 16H=2^31 to |T| }
  END;
END;

{---------------------------------------------------------------------------}
{ Convert Tms to internal time format which increments at 37.286875 tics/ms }
{ IF Tms is out of bounds [0..24Hours] then it is truncated to the boundary }
{ Ie 24Hours+ -> 24H and  if Tms <0 -> 0.                                   }
{ Rem a PC day is short by 2.2ms. So when truncating a value >24 hours the  }
{ the resulting time value will be short by 2.2 ms.                         }
{---------------------------------------------------------------------------}
Function  TRec(Tms: Longint): LongInt;
BEGIN
Tms:= Tms mod T24H;
IF Tms<= TmsOverflow
  THEN TRec:= Round(Tms*TFreq)
  ELSE TRec:= Round((Tms-TmsOverFlow)*TFreq)+$7FFFFFFF;
END;


{-----------------------------------------------------------------}
{ Turbo < 6.0 has no unsigned longint type. This function will    }
{ compare to longints as if they were unsinded.                   }
{ result 1: X>Y,  0: X=Y,  -1: X<Y                                }
{-----------------------------------------------------------------}
Function  DblCmp(x,y: LongInt): integer; Assembler;
ASM
MOV	AX,Word(x)+2
CMP     AX,Word(y)+2        { Check high order word first }
JA	@Above
JB	@Below
MOV	AX,Word(x)
CMP	AX,Word(y)          { Check Lo order word }
JA	@Above
JB      @Below
MOV     AX,0                { X = Y }
JMP	@END

@Above:                     { X > Y }
MOV     AX,1
JMP	@END

@Below:                     { X < Y }
MOV     AX,-1

@END:
END; { DblCmp }


{-----------------------------------------------------------------}
{ ADD unsigned longint internal time format MOD 24 hours.         }
{ Set overflow flag Wrap24 if a sum > 24 Hours. In addition If    }
{ the Sum overflows an unsigned longing >FFFF:FFFF then adjust it }
{   PcDay= $C0058000 (=$1800B0,0000 Tics that accur in one day)   }
{  -PcDay= $3FFA7FFF (=NOT(PcDay)=The amount missing to FFFF:FFFF }
{								  }
{                                                                 }
{             0          FFFF:FFFF   0           FFFF:FFFF   0    }
{         \ /                     \ /                     \ /     }
{          |----------------|------|----------------|------|      }
{          |    PcDay       |-PcDay|    PcDay       |-PcDay|      }
{          |----------------|------|----------------|------|      }
{                                                                 }
{ Case1: T1+T2 > PcDay                                            }
{          |----- T1 + T2 -----|                                  }
{          |    PcDay       |  |                                  }
{            Result of MOD= |--|                                  }
{                                                                 }
{                                                                 }
{ Case2: T1+T2 > FFFF:FFFF                                        }
{          |------ T1 + T2 --------------|                        }
{          |    PcDay       |-PCday|+Over|                        }
{            Result of MOD= |------------|                        }
{                                                                 }
{-----------------------------------------------------------------}
Function Add(T1,T2: Longint): LongInt; Assembler;
ASM
MOV	Wrap24,0		{ Reset overflow flag }
MOV	AX,Word(T1)
MOV	DX,Word(T2)+2
ADD	AX,Word(T2)		{ add low order word first	 }
ADC	DX,Word(T1)+2		{ add High order word with carry }
JC	@AddNDay
CMP	DX,Seg PCDay
JB	@END
JA	@SubDay
CMP	AX,Offset PCDay
JB	@END

@SubDay:
SUB	AX,Offset (PCDay)	{ T < PCDay < $FFFF:FFFF ->T mod PCDay }
SBB     DX,Seg (PCDay)
MOV	Wrap24,1		{ Set overflow flag }
JMP	@END

@AddNDay:               	{ $FFFF:FFFF>= T -> Add (-PCDay) }
ADD	AX,Offset(-PCDay)       { Rem -PCDay= FFFF;FFFF-PCDay    }
ADC	DX,Seg(-PCDay)
MOV	Wrap24,1		{ Set overflow flag }
JMP	@END

@END:
END;


{-----------------------------------------------------------------}
{ Add 2 Tenth Ms time values accounting for 24 wrap around        }
{-----------------------------------------------------------------}
Function TmsAdd(T1,T2: Longint): LongInt;  { SUB MOD 24 hours }
BEGIN
TmsAdd:= (T1+T2) mod T24H;
END;


{-----------------------------------------------------------------}
{ Subtract unsigned longint internal time format MOD 24 hours.    }
{-----------------------------------------------------------------}
Function SUB(T1,T2: Longint): LongInt;  { SUB MOD 24 hours }
BEGIN
IF DblCmp(T1,T2)>=0		{ Check 24 Hour wrap around }
  THEN SUB:= T1-T2
  ELSE SUB:= T1+(PcDay-T2);
END;


{-----------------------------------------------------------------}
{ Subtract Tenth Ms time values. This function is the inverse to  }
{ TmsAdd and simular to SUB but its domain is Tenth Ms time values}
{ wheras SUB's domain is the internal time format.                }
{ The returned value is always within [0..24H] and never negative.}
{ If T1 < T2, which would normaly result in a negative value, then}
{ 24H is added to T1 moving it into the next day, and the 24H time}
{ loss is compensate for I.e. 1H- 23H== 25H-23H= +2H 	          }
{                                                                 }
{ TIP: use TmsCmp12H instead of TmsSub when comparing time values }
{ within a timeout loop. If for example T1 is the current time    }
{ and T2 is the timeout value then repeated calls to TmsSub will  }
{ produce decreasing return values as time passes. But when T1    }
{ surpases T2 the result won't go negative, instead it will jump  }
{ by 24H!   							  }
{-----------------------------------------------------------------}
Function TmsSUB(T1,T2: Longint): LongInt;  { SUB MOD 24 hours }
BEGIN
IF T1>=T2			  { Check 24 Hour wrap around }
  THEN TmsSub:= T1-T2
  ELSE TmsSub:= T1-(T2-T24H);
END;


{-----------------------------------------------------------------}
{ This function is usefull for comparing time differances <=12H   }
{ accounting for 24H wrap around.  [Time]= Tenth Ms user values.  }
{ There are alwase two ways of looking at time differances a) the }
{ 2 times occured within the same day b) the times are from diff  }
{ erent days. As a result, given 2 times there is alwase a |smal- }
{ lest differance| that is smaller than 12H ie by thinking of the }
{ times as from the same day or by seeing them as 2 times from 	  }
{ different days. By doing this the function result always takes  }
{ on values from [-12..+12]. A negative value means that T2>T1 in }
{ the sense that T2 must have occured the next day and that the   }
{ 24H jump must be accounted for by adding 24H to to T2.  E.g.:   }
{ 23H- 1H== 23- 25= -2.                                           }
{ TIP: Use this function when comparing time values within a      }
{ timeout loop. If for example T1 is the current time and T2 is   }
{ the timeout value, then repeated calls to TmsCmp12H will produce}
{ decreasing return values as time passes. And when T1 surpases   }
{ T2 the result will go negative, which should be the loop termia-}
{ tion condition to watch for.                                    }
{ CAUTION: The function will make a 24H jump in results when going}
{ from -12H to +12H. Thats why only time comparisons 12H should   }
{ be used with this function.                                     }
{ eg. Implied day crossing: Cmp(23H, 1H)= -2H,  Cmp( 1H,23H)= +2H }
{ Normal time difference:   Cmp( 1H,12H)=-11H,  Cmp(13H, 1H)=+12H }
{-----------------------------------------------------------------}
Function  TmsCmp12H(T1,T2: LongInt): LongInt;
VAR Temp: LongInt;
BEGIN
Temp:= T1-T2;
IF ABS(Temp)<=(T24h DIV 2)
  THEN TmsCmp12H:= Temp { T1 and T2 are from same day	    }
  ELSE IF T1<T2       	{ T1 and T2 are from different days }
    THEN TmsCmp12H:=  TmsSub(T1,T2)     { T1<T2 -> TmsSub will ad 24H to T1 }
    ELSE TmsCmp12H:= -TmsSub(T2,T1)	{ T2<T1 -> TmsSub will ad 24H to T2 }
END;


{--------------------------------------------------------------------------}
{ Get time in internal format TRec= Tics SHL 11 + Not(Diveder Count) SHR 5 }
{ Function result is in AD:DX registers                                    }
{--------------------------------------------------------------------------}
Function GetTRec: LongInt; assembler;
Const   TChip= $40;   { IO address of 8253 or newer 8254-2 }
      BiosSeg= $40;
	TBios= $6C;
       PicAdr= $20;
       PicIrr= $0A;	{ Interupts pending for service }
       PicIsr= $0B;     { Interupts currently serviced  }
ASM
cli
mov    al,00
out    TChip+3,al
in     al,TChip
mov    bl,al
in     al,TChip
mov    bh,al            { Tchip in BX   }

{-----------------------------------------------------------------------
 The timer chip continues decrementing even atfer a CLI is issued. So
 handle case of Pit counter 0 -> FFFF wrap around atfer CLI was issued.
 The effect of not handling this case is a false reading of time going
 backwards by 55ms! Reseting the value read into the BX register to 0
 is equivalent to incrementing the Bios tic count by 1 thus making
 up for the 55ms time backdrop.
------------------------------------------------------------------------}
cmp    bx,$FFFA         { $10-$A=6 == test if 5us since reset      	}
jbe    @Tics
mov    al,PicIrr
out    PicAdr,al
in     al,PicAdr	{ Get 8259 ISR pending bits    			}
and    al,1
cmp    al,1		{ Is an IRQ0 = bit 1 pending   			}
jne    @Tics		{ If Yes: -> Clear counter back a bit as if IRQ	}
xor    bx,bx		{ did't happen. The next PicTic will retrigger	}
			{ the IRQ as soon as they are enabled           }
{-----------------------------------------------------------------------}

@Tics:
mov    di,BiosSeg
mov    es,di
mov    ax,es:[TBios]
mov    dx,es:[TBios+2]  { Tics in dx:ax }
sti

not    bx          { Timeleft: BX= FFFF-TChip count }
mov    cx,0005
shr    bx,cl       { only need 11 of the 16 bits    }
mov    dh,dl       { only need Last 5 bits of dx }
mov    dl,ah       { together with ah }
mov    cx,0003
shl    dx,cl       { shl DX:AH 3  }
xor    ah,ah
shl    ax,cl       { split al 3Bits:5bits }
mov    cl,ah       { ch already 0 }
or     dx,cx       { juxteposition hi bits next to each other in dx }
mov    ah,al
xor    al,al
or     ax,bx       { juxteposition lo bits next to each other in ax }
END; { GetTRec }


Function TmsTime: LongInt;    { Get time in 1/10 Msec of PC clock }
BEGIN
TmsTime:= TmSec(GetTRec);
END;



{-------------------- Timer object functions ---------------------}

Constructor TTimer.Init(Tms: LongInt);
Const   TChip= $40;  { IO address of 8253 or newer 8254-2 }
BEGIN
ASM CLI END;
Port[TChip+3]:= $34; { $34:= 0011 0110 = set port 0 to 2 byte read/write }
Port[TChip]:= 0;     { Reset so initial count is 64K                     }
Port[TChip]:= 0;
ASM STI END;
UseTimeOutProc:= True;
SetDelay(Tms);
EndT:= GetTRec;
StartAtEnd(True);
TestT:= StartT;
LastT:= StartT
END;

Destructor TTimer.Done;
BEGIN
END;

Function TTimer.Interval: LongInt;  { measures time since last call }
VAR   T: LongInt;
BEGIN
T:= GetTRec;
Interval:= TmSec(Sub(T,LastT));
LastT:= T;
END;


{ Start at Ofset+ Tms, Return absolute PC start time }
Function TTimer.StartIn(Tms: LongInt; NewOfset: Boolean): LongInt;
BEGIN
StartT:= ADD(TRec(Tms),OfsetT);
IF NewOfset THEN OfsetT:= StartT;
LastEndT:= EndT;
EndT:= ADD(StartT,DelayT);	{ Sets Wrap24 }
NewDay:= Wrap24;
StartIn:= TmSec(StartT);
StartN1:= 0;
StartN2:= 0;
END;


{ Start time is now, Return absolute Pc start time }
Function TTimer.StartNow(NewOfset: Boolean): LongInt;
BEGIN
StartT:= GetTRec;
IF NewOfset THEN OfsetT:= StartT;
LastEndT:= EndT;
EndT:= ADD(StartT,DelayT);    { Sets Wrap24 }
NewDay:= Wrap24;
StartNow:= TmSec(StartT);
StartN1:= 0;
StartN2:= 0;
END;


{ Start at Tms in absolute PC time, Return relative start time= Tms- Ofset }
Function TTimer.StartAt(Tms: LongInt; NewOfset: Boolean): LongInt;
BEGIN
StartT:= TRec(Tms);
IF NewOfset THEN OfsetT:= StartT;
LastEndT:= EndT;
EndT:= ADD(StartT,DelayT);	{ Sets Wrap24 }
NewDay:= Wrap24;
StartAt:= TStart;
StartN1:= 0;
StartN2:= 0;
END;


Procedure TTimer.StartAtEnd(NewOfset: Boolean);
Var Dt: LongInt;
BEGIN
StartT:= EndT;
IF NewOfset
  THEN BEGIN
  StartN1:=0;
  StartN2:= 0;
  OfsetT:= StartT;
  END;
LastEndT:= EndT;
Dt:=DelayT;

Inc(StartN1);
IF (StartN1>= Err_N1)                { First order rounding correction }
  THEN BEGIN
  Dt:= Dt+Err_Val1; { ok not to call ADD because Err_val=0 if DelayT <TmsOverFlow }
  StartN1:= 0;
  END;

Inc(StartN2);
IF (StartN2>= Err_N2)                { Second order rounding correction }
  THEN BEGIN
  Dt:= Dt+Err_Val2; { ok not to call ADD because Err_val=0 if DelayT <TmsOverFlow }
  StartN2:= 0;
  END;

EndT:= ADD(StartT,DT);		{ Sets Wrap24 }
NewDay:= Wrap24;
END;


Procedure TTimer.StartAtTest(NewOfset: Boolean);
BEGIN
StartT:= TestT;
IF NewOfset THEN OfsetT:= StartT;
LastEndT:= EndT;
EndT:= ADD(StartT,DelayT);	{ Wrap24 }
NewDay:= Wrap24;
StartN1:= 0;
StartN2:= 0;
END;


Function TTimer.TimeOut: Boolean;
LABEL Cmp_Dbl_AE,			{ Above or Equal address label }
      Cmp_Dbl_BL;			{ Below address label          }
BEGIN
{ Compare an unsigned double word. Turbo can only compare signed longints }
ASM
CALL	GetTRec				{ Get current time }
LES	DI,SELF				{ Get efective object address }
MOV	ES:[DI+Offset(TestT)],AX	{ and save result in TestT }
MOV	ES:[DI+Offset(TestT)+2],DX

CMP	DX,ES:[DI+Offset(EndT)+2]	{ Check high order word first }
JA	@CmpNewDay			{ Jump to Above or Equal }
JB	Cmp_Dbl_Bl			{ Jump to Below }
CMP	AX,ES:[DI+Offset(EndT)]		{ Check Lo order word }
JB	Cmp_Dbl_Bl			{ Jump to Below }

@CmpNewDay:				{ Check for 24hour crossover }
CMP	ES:[DI+NewDay],0
JZ	Cmp_Dbl_AE			{ Jump to Above or Equal }

CMP	DX,ES:[DI+Offset(StartT)+2]	{ If T>=Start then Below ! }
JB	Cmp_Dbl_AE			{ T< Start-> Above or Equal }
JA	Cmp_Dbl_Bl			{ T> Start-> Below }
CMP	AX,ES:[DI+Offset(StartT)]	{ Check Lo order word }
JB	Cmp_Dbl_AE			{ T< Start-> Above or Equal }
JMP	Cmp_Dbl_Bl			{ T>=Start-> Below }
END;

IF TestT > EndT     { The test is actually perfomed in above ASM }
  THEN Cmp_Dbl_BL: TimeOut:= False
  ELSE Cmp_Dbl_AE: BEGIN
  IF UseTimeOutProc THEN TimeOutProc;  { call virtual user function }
  TimeOut:= True;
  END;
END;


Procedure TTimer.TimeOutProc;		{ default time out function	}
BEGIN
StartAtEnd(False);			{ Restart Timer periodically	}
END;


Procedure Ttimer.SetDelay(Tms: LongInt); { maximum is 24 Hours in mSec/10 }
Type LongRec= Record Lo,Hi: Word END;
VAR Error: Real;
BEGIN
DelayT:= TRec(Tms);

{ New: Do to rounding, the internal time will not set to exactly what was
 requested. The error will show up after repeated calls to StartAtEnd.
 This calculation returns how many calls can be made to StartAtEnd without
 any effect do to rounding. If more calls are made then the StartAtEnd
 must delay an aditional +/- internal tic, depending on the sign. Thus
 allowing for a first order correction. Actually 2 orders are performed.
 Delays greater than TmsOverFlow=~ 16H are not corrected. Corrections
 are realy only needed for small delays where the relative error is more
 significant. The effective delay is then:
		DelayT+ Err_Val1/Err_N1+ Err_Val2/Err_N2
 Which should be closer to Tms* TFreq than juest DelayT. To see the effect
 of not correcting: edit the lines where Err_Val1,2 are set to +/- 1 to 0.
 Then set the delay to 10ms and call the TimeOut routine repeatedly which
 by default calls StartAtEnd at every timeout. Print the resulting timeout
 times by calling LastEnd and watch the drift.                           }
IF DelayT>=0		       { Rem internal format is negative after 16H }
  THEN Error:= TFreq*Tms-DelayT
  ELSE Error:= 0;

IF Error>0		{ First order correction, Temp in [0.1 ms] }
  THEN BEGIN
  Err_Val1:= 1;
  Err_N1:= Round(1/Error);
  END
  ELSE IF Error<0
    THEN BEGIN
    Err_Val1:= -1;
    Err_N1:= Round(-1/Error);
    END
    ELSE BEGIN
    Err_Val1:= 0;
    Err_N1:= $FFFF;  { Set very high -> Error less than a second order corr }
    END;
StartN1:= 0;

Error:= Error-Err_Val1/Err_N1;
IF Error>0		{ Second order correction }
  THEN BEGIN
  Err_Val2:= 1;
  Err_N2:= Round(1/Error);
  END
  ELSE IF Error<0
    THEN BEGIN
    Err_Val2:= -1;
    Err_N2:= Round(1/-Error);
    END
    ELSE BEGIN
    Err_Val2:= 0;
    Err_N2:= $FFFF;  { Set very high -> Error less than a second order corr }
    END;
StartN2:= 0;
END;

Procedure Ttimer.SetOfs(Tms: LongInt);	{ maximum is 24 Hours in mSec/10  }
BEGIN
OfsetT:= TRec(Tms);
END;

Procedure Ttimer.SyncOfs(T: pTimer);	{ Synchronize to timer ofsets     }
BEGIN
OfsetT:= T^.OfsetT;
END;

Function Ttimer.TOfset: LongInt;	{ OfsetT in mSec/10	}
BEGIN
TOfset:= TmSec(OfsetT);
END;

Function Ttimer.Time: LongInt;		{ PcTime - SetTime	}
BEGIN
Time:= TmSec(SUB(GetTRec,OfsetT));
END;

Function TTimer.TStart: LongInt;	{ timer Start - SetTime }
BEGIN
TStart:= TmSec(SUB(StartT,OfsetT));
END;

Function TTimer.TElapsed: LongInt;	{ Time since timer start}
BEGIN
TElapsed:= TmSec(SUB(GetTRec,StartT));
END;

Function TTimer.TRemain: LongInt;	{ Time till timer ends	}
Var Temp: longInt;
BEGIN
TRemain:= TmSec(SUB(EndT,GetTRec));
END;

Function TTimer.TEnd: LongInt;          { timer End - SetTime 	}
BEGIN
TEnd:= TmSec(SUB(EndT,OfsetT));
END;

Function TTimer.TLastEnd: LongInt;      { Last timer End - SetTime }
BEGIN
TLastEnd:= TmSec(SUB(LastEndT,OfsetT));
END;

Function TTimer.TTest: LongInt;		{ Last Test - SetTime 	}
BEGIN
TTest:= TmSec(SUB(TestT,OfsetT));
END;

END.
