        (* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *)
        (*                                                           *)
        (*                          SerialIO                         *)
        (*                                                           *)
        (*                    This unit works with                   *)
        (*                         - INS8250                         *)
        (*                         - NS16450                         *)
        (*                         - NS16550                         *)
        (*                         - NS16550A and                    *)
        (*                   compatible UART chips.                  *)
        (*                                                           *)
        (*      Copyright (c) 1994 by Gian Donato "Dado" Colussi     *)
        (*            E-mail: gcolussi@kruuna.helsinki.fi            *)
        (*                                                           *)
        (*             THE AUTHOR TAKES NO RESPONSIBILITY            *)
        (*          OF ANY KIND OF PHYSICAL OR MENTAL DAMAGE         *)
        (*                    CAUSED BY THIS CODE                    *)
        (*           OR PROGRAMS/UNITS DELIVERED WITH THIS.          *)
        (*                                                           *)
        (* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *)


UNIT SerialIO;


INTERFACE


USES

  DOS,                  (* SetIntVec and GetIntVec are needed to install interrupt handler *)
  Hex,                  (* We must be able to read and write hex numbers *)
  StrIO,                (* String I/O routines *)
  Keyboard,             (* Key codes *)
  Error,                (* Error handling routines *)
  Buffer,               (* Dynamic circular buffer *)
  Consts,               (* Miscellaneous constants *)
  Regs;                 (* UART and PIC registers *)


TYPE

  ProcedurePointer      = ^ProcedureType;
  ProcedureType         = PROCEDURE;

  SerialPointer         = ^SerialClass;
  SerialClass           =
    OBJECT              (* Public section *)

      input_buffer      : BufferClass;
      output_buffer     : BufferClass;

      CONSTRUCTOR       AllocatePort(base_address : Word;
                                     speed        : Byte;
                                     data_bits    : Byte;
                                     parity       : Byte;
                                     stop_bits    : Byte;
                                     fifo_size    : Byte;
                                     flow_ctrl    : Byte;
                                     int_types    : Byte;
                                     in_buf_size  : Word;
                                     out_buf_size : Word);
      DESTRUCTOR        ReleasePort; VIRTUAL;

      PROCEDURE         EnableUART; VIRTUAL;
      PROCEDURE         DisableUART; VIRTUAL;

                        (*
                          When you use these procedures remember to use
                          the constant values defined in the file
                          CONSTS.PAS as parametres.
                        *)
      PROCEDURE         SetSpeed(new_speed : Byte); VIRTUAL;
      PROCEDURE         SetDataBits(new_data_bits : Byte); VIRTUAL;
      PROCEDURE         SetParity(new_parity : Byte); VIRTUAL;
      PROCEDURE         SetStopBits(new_stop_bits : Byte); VIRTUAL;
      PROCEDURE         SetFIFO(new_mode : Byte); VIRTUAL;
      PROCEDURE         SetFlowControl(flow_mode : Byte); VIRTUAL;
      PROCEDURE         SetInterrupts(int_types : Byte); VIRTUAL;

      PROCEDURE         ResetRCVRFIFO; VIRTUAL;
      PROCEDURE         ResetXMITFIFO; VIRTUAL;
      PROCEDURE         SetDMAFIFO(status : Boolean); VIRTUAL;

      PROCEDURE         SendChar(ch : Char); VIRTUAL;
      PROCEDURE         SendStr(str : String); VIRTUAL;
      FUNCTION          ReceiveChar : Char; VIRTUAL;
      PROCEDURE         WriteChar(ch : Char); VIRTUAL;
      FUNCTION          ReadChar : Char; VIRTUAL;

      PROCEDURE         ReadModemStatus; VIRTUAL;
      FUNCTION          CTS : Boolean; VIRTUAL;
      FUNCTION          DSR : Boolean; VIRTUAL;
      FUNCTION          RI  : Boolean; VIRTUAL;
      FUNCTION          DCD : Boolean; VIRTUAL;

      PROCEDURE         SetDTR(status : Boolean); VIRTUAL;
      PROCEDURE         SetRTS(status : Boolean); VIRTUAL;
      FUNCTION          DTR : Boolean; VIRTUAL;
      FUNCTION          RTS : Boolean; VIRTUAL;

      PROCEDURE         ReadLineStatus; VIRTUAL;
      FUNCTION          DR : Boolean; VIRTUAL;
      FUNCTION          OE : Boolean; VIRTUAL;
      FUNCTION          PE : Boolean; VIRTUAL;
      FUNCTION          FE : Boolean; VIRTUAL;
      FUNCTION          BI : Boolean; VIRTUAL;
      FUNCTION          THRE : Boolean; VIRTUAL;
      FUNCTION          TEMT : Boolean; VIRTUAL;
      FUNCTION          ErrRCVRFIFO : Boolean; VIRTUAL;

      FUNCTION          InterruptIdentification : Byte; VIRTUAL;
      FUNCTION          InterruptPending : Boolean; VIRTUAL;

      PRIVATE           (* Private section *)

      base              : Word;
      uart_type         : Byte;
      fifo              : Byte;
      flow_control      : Byte;
      in_flow_status    : Boolean;
      out_flow_status   : Boolean;
      modem_status      : Byte;
      line_status       : Byte;

    END; (* SerialClass *)

  InterruptPointer      = ^InterruptClass;
  InterruptClass        =
    OBJECT              (* Public section *)

      CONSTRUCTOR       InstallInterruptHandler(irq : Byte);
      DESTRUCTOR        RemoveInterruptHandler; VIRTUAL;

      PROCEDURE         AddService(port_number : Byte); VIRTUAL;
      PROCEDURE         RemoveService(port_number : Byte); VIRTUAL;
      FUNCTION          Service(port_number : Byte) : Boolean; VIRTUAL;
      PROCEDURE         EnableIRQ; VIRTUAL;
      PROCEDURE         DisableIRQ; VIRTUAL;

      PRIVATE           (* Private section *)
    
      irq_line          : Byte;
      vector            : Pointer;
      old_vector        : Pointer;
      ports_to_serve    : Longint;

    END; (* InterruptClass *)


VAR

  sio                   : ARRAY [1..MAX_PORTS] OF SerialPointer;
  irqs                  : ARRAY [0..MAX_IRQS] OF InterruptPointer;


IMPLEMENTATION


VAR
  exit_save             : Pointer;      (* Previous ExitProc is saved in this variable *)
  xon_request           : Boolean;
  xoff_request          : Boolean;


(*
  Returns the interrupt number used by 'irq'
*)
FUNCTION InterruptByIRQ(irq : Byte) : Byte;
BEGIN
  CASE irq OF
    0  : InterruptByIRQ := $08;         (* Timer *)
    1  : InterruptByIRQ := $09;         (* Keyboard service required *)
    2  : InterruptByIRQ := $0A;         (* Slave 8259A or EGA/VGA vertical retrace *)
    3  : InterruptByIRQ := $0B;         (* COM2 or COM4 *)
    4  : InterruptByIRQ := $0C;         (* COM1 OR COM3 *)
    5  : InterruptByIRQ := $0D;         (* Fixed disk or data request from LPT2 *)
    6  : InterruptByIRQ := $0E;         (* Floppy diks service requided *)
    7  : InterruptByIRQ := $0F;         (* Data request from LPT1 (Unreliable on IBM mono) *)

(*
  The following irqs are not supported by this unit.
  Perhaps they will someday... who knows?
*)
    8  : InterruptByIRQ := $70;         (* Real Time Clock *)
    9  : InterruptByIRQ := $71;         (* Software redirected to IRQ2 *)
    10 : InterruptByIRQ := $72;         (* Reserved *)
    11 : InterruptByIRQ := $73;         (* Reserved *)
    12 : InterruptByIRQ := $74;         (* Mouse interrupt *)
    13 : InterruptByIRQ := $75;         (* Numeric coprosessor error *)
    14 : InterruptByIRQ := $76;         (* Fixed disk controller *)
    15 : InterruptByIRQ := $77;         (* Reserved *)
  ELSE
     InterruptByIRQ := 0;               (* Invalid IRQ number *)
  END; (* CASE *)
END; (* InterruptByIRQ *)

(*
  Disables all maskable interrupts
*)
PROCEDURE CLI;
  INLINE($FA);

(*
  Enables all maskable interrupts
*)
PROCEDURE STI;
  INLINE($FB);

(*
  Sends an End Of Interrupt message to the PIC
*)
PROCEDURE EOI(irq : Byte);
BEGIN
  Port[ISR] := SPECIFIC_EOI OR irq AND $F;
END; (* EOI *)

(*
  The following routine is copied from another serial I/O driver. I have
  not read any documentation on this function and the quality of it may
  be questionable.

  However, I believe that every bit in the return value represents an
  IRQ line so that if a specified bit is a logic 1 that IRQ line needs
  to be served and if it's a logic 0 that IRQ line need not to be
  served.

  7 6 5 4 3 2 1 0
  | | | | | | | \- Bit 0 => IRQ 0
  | | | | | | \-- Bit 1 => IRQ 1
  | | | | | \--- Bit 2 => IRQ 2
  | | | | \---- Bit 3 => IRQ 3
  | | | \----- Bit 4 => IRQ 4
  | | \------ Bit 5 => IRQ 5
  | \------- Bit 6 => IRQ 6
  \-------- Bit 7 => IRQ 7
*)
FUNCTION RIL : Byte;
BEGIN

  (* Report Interrupt Level Request *)
  Port[ISR] := $B;

  (* Return Interrupt Level *)
  RIL := Port[ISR];
END; (* RIL *)

(*
  Enables interrupts from a specified IRQ line.
  We must manipulate the Interrupt Mask Register so that the bit that
  represents the irq we want to use is set to logic 0. Let's say that
  irq 3 must be enabled and IMR looks like this:
      11001110
  After manipulation IMR would look like this:
      11000110
*)
PROCEDURE EnableIRQ(irq : Byte);
BEGIN
  Port[IMR] := Port[IMR] AND NOT (1 SHL irq);
END; (* EnableIRQ *)

(*
  Disables interrupts from a specified IRQ line
*)
PROCEDURE DisableIRQ(irq : Byte);
BEGIN
  Port[IMR] := Port[IMR] OR (1 SHL irq);
END; (* DisableIRQ *)

(*
  Returns TRUE if the specified irq line is available for use.
  In other words, if the bit in Interrupt Mask Register that
  represents the irq we want to use is logic 1 (disabled)
  IRQAvailable returns TRUE.
*)
FUNCTION IRQAvailable(irq : Byte) : Boolean;
BEGIN
  IRQAvailable := Port[IMR] AND (1 SHL irq) = 1 SHL irq;
END; (* IRQAvailable *)

(*
  This routine will be called when the program terminates
*)
{$F+}
PROCEDURE ExitSerialIO;
VAR
  i     : Byte;

BEGIN

  (* Release all ports *)
  FOR i := 1 TO MAX_PORTS DO
    IF sio[i] <> NIL THEN
      BEGIN
        Dispose(sio[i], ReleasePort);
        sio[i] := NIL;
      END; (* IF *)

  (* Remove all interrupt handlers *)
  FOR i := 0 TO MAX_IRQS DO
    IF irqs[i] <> NIL THEN
      BEGIN
        Dispose(irqs[i], RemoveInterruptHandler);
        irqs[i] := NIL;
      END; (* IF *)

  (* Return previous exit pointer *)
  ExitProc := exit_save;
END; (* ExitSerialIO *)
{$F-}

(*
  Snoops the type of UART from a specified base address.
*)
FUNCTION DetectUART(base : Word) : Byte;
CONST
  VAL_1 = $F;   (* These are pseudo-random values for testing UART registers *)
  VAL_2 = $A;

VAR
  uart_type     : Byte;
  original,
  check_1,
  check_2       : Byte;

BEGIN

  (*
    Let's test Interrupt Enable Register with two different values
    and return the original value after tests.
  *)
  uart_type := UART_NONE;
  original := Port[base + IER];
  Port[base + IER] := VAL_1;
  check_1 := Port[base + IER];
  Port[base + IER] := VAL_2;
  check_2 := Port[base + IER];
  Port[base + IER] := original;
  IF (check_1 = VAL_1) AND (check_2 = VAL_2) THEN

    (*
      The test was successful! It is now safe enough to suppose
      that there is an UART chip.
    *)
    BEGIN

      (*
        We cannot be sure of the type of the UART yet, but let's
        suppose it is an INS8250 or compatible. If the following tests
        fail, the value of uart_type is not changed.
      *)
      uart_type := UART_INS8250;

      (*
        Now we'll test Scratch Register with two different values and
        return the original value after tests as we did before.
      *)
      original := Port[base + SCR];
      Port[base + SCR] := VAL_1;
      check_1 := Port[base + SCR];
      Port[base + SCR] := VAL_2;
      check_2 := Port[base + SCR];
      Port[base + SCR] := original;
      IF (check_1 = VAL_1) AND (check_2 = VAL_2) THEN

      (*
        The test was successful again! This UART contains a
        Scratch Register, so it has to be an NS16450,
        an NS16550 or an NS16550A chip.
      *)
        BEGIN

          (*
            Let's suppose the UART chip is an NS16450 or compatible.
            If the following tests fail, this will be the last value set
            to the uart_type.
          *)
          uart_type := UART_NS16450;

          (*
            If the FIFO Control Register exists, it is safe enough to
            suppose that there is an NS16550, an NS16550A or compatible
            UART chip.

            !!! CAUTION !!! CAUTION !!! CAUTION !!! CAUTION !!!

                FIFO does NOT work on NS16550 UART chip. Be
                sure NOT to enable FIFOs on UARTs of that
                type. If you do, it may cause a disaster.

            !!! CAUTION !!! CAUTION !!! CAUTION !!! CAUTION !!!
          *)
          Port[base + FCR] := FCR_FIFO_ENABLE + FCR_RCVR_TRIGGER_LSB + FCR_RCVR_TRIGGER_MSB;

          (*
            In NS16550A chip _both_ FIFO Trigger Level Bits stay up while
            in NS16550 chip  only _one_ FIFO Trigger Level Bit stays up.
            I can't remember which one it is but that doesn't change
            the fact that in INS16550 just ONE bit stays up.
          *)
          CASE Port[base + IIR] AND (IIR_FIFOE0 + IIR_FIFOE1) OF
            IIR_FIFOE0              : uart_type := UART_NS16550;
            IIR_FIFOE1              : uart_type := UART_NS16550;
            IIR_FIFOE0 + IIR_FIFOE1 : uart_type := UART_NS16550A;
          END; (* CASE *)
        END; (* IF *)
    END; (* IF *)
  DetectUART := uart_type;
END; (* DetectUART *)

(*
  Returns the speed value that can be written directly into the
  Divisor Latch
*)
FUNCTION DivisorLatch(speed : Byte) : Word;
VAR
  bps   : Longint;

BEGIN
  CASE speed OF
    SPEED_1200          : bps := 1200;
    SPEED_2400          : bps := 2400;
    SPEED_9600          : bps := 9600;
    SPEED_19200         : bps := 19200;
    SPEED_38400         : bps := 38400;
    SPEED_57600         : bps := 57600;
    SPEED_115200        : bps := 115200;
  ELSE
    bps := 0;
  END; (* CASE *)
  IF bps = 0 THEN
    DivisorLatch := 0
  ELSE
    DivisorLatch := Round(CLOCK_FREQ / (bps * 16));
END; (* DivisorLatch *)

(*
  This routine will be called when an interrupt occures
*)
{$F+}
PROCEDURE SerialHandler; INTERRUPT;
VAR
  port_number   : Byte;
  ril_data      : Byte;
  irq           : Byte;
  tmp           : Byte;

BEGIN

  (*
    Disable all maskable interrupts
  *)
  CLI;

  (*
    Report Interrupt Level
  *)
  ril_data := RIL;

  (*
    Browse all interrupts
  *)
  FOR irq := 0 TO MAX_IRQS DO

    (*
      Check if the IRQ line needs to be served
    *)
    IF ril_data AND (1 SHL irq) > 0 THEN

      (*
        Check if the irq line is served by this driver
      *)
      IF irqs[irq] <> NIL THEN

        (*
          Poll all ports that are served by the IRQ that caused the interrupt
        *)
        FOR port_number := 1 TO MAX_PORTS DO
          IF irqs[irq]^.Service(port_number) THEN
            REPEAT
              CASE sio[port_number]^.InterruptIdentification OF

                (*
                  No interrupt; do nothing.
                *)
                IIR_NONE     : ;

                (*
                  Receiver Line Status.
                  Interrupt reset by reading Line Status Register.
                *)
                IIR_RLS      : BEGIN
                                 sio[port_number]^.ReadLineStatus;
                                END; (* IIR_RLS *)

                (*
                   Received Data Available.
                   Interrupt reset by reading Receiver Buffer Register or
                   the FIFO Drops Below the Trigger Level.
                *)
                IIR_RDA      : BEGIN
                                 WHILE sio[port_number]^.DR DO
                                   sio[port_number]^.input_buffer.WriteChar(sio[port_number]^.ReceiveChar);
                                 IF sio[port_number]^.input_buffer.OverflowDanger AND sio[port_number]^.RTS THEN
                                   sio[port_number]^.SetRTS(FALSE);
                               END; (* IIR_RDA *)

                (*
                  Character Timeout Indication.
                  Interrupt reset by reading the Receiver Buffer Register.
                *)
                IIR_CTI      : BEGIN
                                 WHILE sio[port_number]^.DR DO
                                   sio[port_number]^.input_buffer.WriteChar(sio[port_number]^.ReceiveChar);
                                 IF sio[port_number]^.input_buffer.OverflowDanger AND sio[port_number]^.RTS THEN
                                   sio[port_number]^.SetRTS(FALSE);
                               END; (* IIR_CTI *)

                (*
                  Transmitter Holding Register Empty.
                  Interrupt reset by reading the Interrupt Identification Register (if source of
                  interrupt) or Writing into the Transmitter Holding Register
                *)
                IIR_THRE     : BEGIN
                                 IF sio[port_number]^.output_buffer.CharAvailable AND sio[port_number]^.CTS THEN
                                   sio[port_number]^.SendChar(sio[port_number]^.output_buffer.ReadChar)
                                 ELSE
                                   tmp := sio[port_number]^.InterruptIdentification;
                               END; (* IIR_THRE *)

                (*
                   MODEM Status.
                   Interrupt reset by reading the MODEM Status Register.
                *)
                IIR_MS       : BEGIN
                                 sio[port_number]^.ReadModemStatus;
                               END; (* IIR_MS *)
              END; (* CASE *)

              (*
                Send End Of Interrupt message to the PIC
              *)
              EOI(irq);
            UNTIL NOT sio[port_number]^.InterruptPending;

  (*
    Enable all maskable interrupts
  *)
  STI;
END; (* SerialHandler *)
{$F-}


(* ------- *)
(* Methods *)
(* ------- *)

CONSTRUCTOR SerialClass.AllocatePort(base_address : Word;
                                     speed        : Byte;
                                     data_bits    : Byte;
                                     parity       : Byte;
                                     stop_bits    : Byte;
                                     fifo_size    : Byte;
                                     flow_ctrl    : Byte;
                                     int_types    : Byte;
                                     in_buf_size  : Word;
                                     out_buf_size : Word);
BEGIN

  (*
    The UART should be located at the address stored in 'base_address'
  *)
  base := base_address;

  (*
    Find out the type of UART to be used
  *)
  uart_type := DetectUART(base);
  CASE uart_type OF

    (*
      In this case there was no UART at the base address
    *)
    UART_NONE    : PrintError('No UART detected at address ' + HexW(base) + 'h', dead_end);

    (*
      Unknown UART type was detected
    *)
    UART_UNKNOWN : PrintError('Unknown UART detected at address ' + HexW(base) + 'h', ignore);
  END; (* CASE *)
  CLI;
  DisableUART;
  STI;

  (*
    Set line dicipline
  *)
  SetSpeed(speed);
  SetDataBits(data_bits);
  SetParity(parity);
  SetStopBits(stop_bits);
  SetFIFO(fifo_size);
  SetFlowControl(flow_ctrl);

  (*
    Enable/Disable different type of interrupts.
    Look 'Interrupt Types in Priority Order' from REGS.PAS file for more
    information.
  *)
  SetInterrupts(int_types);

  (*
    Open input and output buffers
  *)
  input_buffer.Open(in_buf_size);
  output_buffer.Open(out_buf_size);

  (*
    Input buffer will ignore the XON and XOFF characters. They only
    influence the flow control.

    Why don't we make also the output buffer ignore those characters?
    That's because the user must be able to send XON and XOFF to control
    the flow, for example to stop a massive flood.
  *)
  input_buffer.Ignore(XON);
  input_buffer.Ignore(XOFF);
  IF error_alarm THEN

    (*
      Some of the operations drove us to a dead end.
      We have to exit immediatelly!
    *)
    Exit;
  in_flow_status := TRUE;
  out_flow_status := TRUE;
  SetDTR(TRUE);
  SetRTS(TRUE);

  (*
    Interrupts from UART must be enabled.
    Interrupt from the PIC must be enabled too, but that should be done
    later.
  *)
  CLI;
  EnableUART;
  STI;
  ReadModemStatus;
  ReadLineStatus;
END; (* SerialClass.AllocatePort *)

DESTRUCTOR SerialClass.ReleasePort;
VAR
  junk  : Char;

BEGIN
  CLI;
  DisableUART;
  SetRTS(FALSE);
  SetDTR(FALSE);
  SetInterrupts(0);
  WHILE DR DO
    junk := ReceiveChar;
  STI;
  input_buffer.Close;
  output_buffer.Close;
END; (* SerialClass.ReleasePort *)

(*
  Sets Modem Control Register bit Out2 to logic 1
*)
PROCEDURE SerialClass.EnableUART;
BEGIN
  Port[base + MCR] := Port[base + MCR] OR MCR_OUT_2;
END; (* SerialClass.EnableUART *)

(*
  Sets Modem Control Register bit Out2 to logic 0
*)
PROCEDURE SerialClass.DisableUART;
BEGIN
  Port[base + MCR] := Port[base + MCR] AND NOT MCR_OUT_2;
END; (* SerialClass.DisableUART *)

(*
  Enables Divisor Latch Access
  Writes speed value
  Disables Divisor Latch Access
*)
PROCEDURE SerialClass.SetSpeed(new_speed : Byte);
BEGIN
  Port[base + LCR] := Port[base + LCR] OR LCR_DLAB;
  PortW[base + DLL] := DivisorLatch(new_speed);
  Port[base + LCR] := Port[base + LCR] AND NOT LCR_DLAB;
END; (* SerialClass.SetSpeed *)

PROCEDURE SerialClass.SetDataBits(new_data_bits : Byte);
BEGIN
  Port[base + LCR] := Port[base + LCR] AND NOT (LCR_WLS0 OR LCR_WLS1) OR new_data_bits AND (LCR_WLS0 OR LCR_WLS1);
END; (* SerialClass.SetDataBits *)

PROCEDURE SerialClass.SetParity(new_parity : Byte);
BEGIN
  Port[base + LCR] := Port[base + LCR] AND NOT (LCR_PEN OR LCR_EPS OR LCR_STICK_PAR)
                      OR new_parity AND (LCR_PEN OR LCR_EPS OR LCR_STICK_PAR);
END; (* SerialClass.SetParity *)

PROCEDURE SerialClass.SetStopBits(new_stop_bits : Byte);
BEGIN
  Port[base + LCR] := Port[base + LCR] AND NOT LCR_STB OR new_stop_bits AND LCR_STB;
END; (* SerialClass.SetStopBits *)

PROCEDURE SerialClass.SetFIFO(new_mode : Byte);
BEGIN
  Port[base + FCR] := new_mode;
END; (* SerialClass.SetFIFO *)

(*
  Set flow control mode (none, rts/cts or xon/xoff)
*)
PROCEDURE SerialClass.SetFlowControl(flow_mode : Byte);
BEGIN
  CASE flow_mode OF
    FLOW_RTS_CTS  : flow_control := FLOW_RTS_CTS;
    FLOW_XON_XOFF : flow_control := FLOW_XON_XOFF;
  ELSE
    flow_control := FLOW_XON_XOFF;
  END; (* CASE *)
END; (* SerialClass.SetFlowContol *)

(*
  Enables/Disables different type of interrupts
*)
PROCEDURE SerialClass.SetInterrupts(int_types : Byte);
BEGIN
  Port[base + IER] := int_types AND $F;
END; (* SerialClass.SetInterrupts *)

(*
  Resets Receiver FIFO
*)
PROCEDURE SerialClass.ResetRCVRFIFO;
BEGIN
  Port[base + FCR] := Port[base + FCR] OR FCR_RCVR_FIFO_RESET;
END; (* SerialClass.ResetRCVRFIFO *)

(*
  Resets Transmitter FIFO
*)
PROCEDURE SerialClass.ResetXMITFIFO;
BEGIN
  Port[base + FCR] := Port[base + FCR] OR FCR_XMIT_FIFO_RESET;
END; (* SerialClass.ResetXMITFIFO *)

(*
  Enables/Disables Direct Memory Access FIFO
*)
PROCEDURE SerialClass.SetDMAFIFO(status : Boolean);
BEGIN
  CASE status OF
    TRUE  : Port[base + FCR] := Port[base + FCR] OR FCR_DMA_MODE_SELECT;
    FALSE : Port[base + FCR] := Port[base + FCR] AND NOT FCR_DMA_MODE_SELECT;
  END; (* CASE *)
END; (* SerialClass.SetDMAFIFO *)

(*
  Writes a character into Transmitter Holding Register
*)
PROCEDURE SerialClass.SendChar(ch : Char);
BEGIN
  IF xon_request THEN
    BEGIN
      xon_request := FALSE;
      SendStr(XON + ch);
    END; (* IF *)
  IF xoff_request THEN
    BEGIN
      xoff_request := FALSE;
      SendStr(XOFF + ch);
    END; (* IF *)
  Port[base + THR] := Ord(ch);
END; (* SerialClass.SendChar *)

(*
  Writes a string into Transmitter Holding Register
*)
PROCEDURE SerialClass.SendStr(str : String);
VAR
  i     : Byte;

BEGIN
  FOR i := 1 TO Byte(str[0]) DO
    SendChar(str[i]);
END; (* SerialClass.SendStr *)

(*
  Reads a character from Receiver Buffer Register
*)
FUNCTION SerialClass.ReceiveChar : Char;
VAR
  ch    : Char;

BEGIN
  ch := Chr(Port[base + RBR]);
  CASE ch OF
    XON   : in_flow_status := TRUE;
    XOFF  : in_flow_status := FALSE;
  END; (*  CASE *)
  ReceiveChar := ch;
END; (* SerialClass.ReceiveChar *)

(*
  Writes a character into the THR or into the output buffer
*)
PROCEDURE SerialClass.WriteChar(ch : Char);
BEGIN
  CASE THRE OF
    TRUE  : CASE output_buffer.CharAvailable OF
              TRUE  : output_buffer.WriteChar(ch);
              FALSE : SendChar(ch);
            END; (* CASE *)
    FALSE : output_buffer.WriteChar(ch);
  END; (* CASE *)
END; (* SerialClass.WriteChar *)

(*
  Reads a character from input buffer.
  Sets RTS on if there is no overflow danger and the RTS is set off
*)
FUNCTION SerialClass.ReadChar : Char;
BEGIN
  CASE input_buffer.CharAvailable OF
    TRUE  : ReadChar := input_buffer.ReadChar;
    FALSE : ReadChar := NUL;
  END; (* CASE *)
  IF NOT input_buffer.OverflowDanger AND NOT RTS THEN
    SetRTS(TRUE);
END; (* SerialClass.ReadChar *)

PROCEDURE SerialClass.ReadModemStatus;
BEGIN
  modem_status := Port[base + MSR];
END; (* SerialClass.ReadModemStatus *)

(*
  Clear To Send
*)
FUNCTION SerialClass.CTS : Boolean;
BEGIN
  CASE flow_control OF
    FLOW_XON_XOFF   : CTS := out_flow_status;
    FLOW_RTS_CTS    : CTS := Port[base + MSR] AND MSR_CTS > 0;
  END; (* CASE *)
END; (* SerialClass.CTS *)

(*
  Data Set Ready
*)
FUNCTION SerialClass.DSR : Boolean;
BEGIN
  DSR := Port[base + MSR] AND MSR_DSR > 0;
END; (* SerialClass.DSR *)

(*
  Ring Indicator
*)
FUNCTION SerialClass.RI : Boolean;
BEGIN
  RI := Port[base + MSR] AND MSR_RI > 0;
END; (* SerialClass.RI *)

(*
  Data Carrier Detected
*)
FUNCTION SerialClass.DCD : Boolean;
BEGIN
  DCD := Port[base + MSR] AND MSR_DCD > 0;
END; (* SerialClass.DCD *)

(*
  Sets DTR (Data Terminal Ready) to on/off
*)
PROCEDURE SerialClass.SetDTR(status : Boolean);
BEGIN
  CASE status OF
    TRUE  : Port[base + MCR] := Port[base + MCR] OR MCR_DTR;
    FALSE : Port[base + MCR] := Port[base + MCR] AND NOT MCR_DTR;
  END; (* CASE *)
END; (* SerialClass.SetDTR *)

(*
  Sets RTS (Request To Send) to on/off
*)
PROCEDURE SerialClass.SetRTS(status : Boolean);
BEGIN
  in_flow_status := status;
  CASE flow_control OF
    FLOW_RTS_CTS    : CASE status OF
                        TRUE      : Port[base + MCR] := Port[base + MCR] OR MCR_RTS;
                        FALSE     : Port[base + MCR] := Port[base + MCR] AND NOT MCR_RTS;
                      END; (* CASE *)
    FLOW_XON_XOFF   : CASE status OF
                        TRUE      : xon_request := TRUE;
                        FALSE     : xoff_request := TRUE;
                      END; (* CASE *)
  END; (* CASE *)
END; (* SerialClass.SetRTS *)

(*
  Data Terminal Ready
*)
FUNCTION SerialClass.DTR : Boolean;
BEGIN
  DTR := Port[base + MCR] AND MCR_DTR > 0;
END; (* SerialClass.DTR *)

(*
  Request To Send
*)
FUNCTION SerialClass.RTS : Boolean;
BEGIN
  CASE flow_control OF
    FLOW_XON_XOFF   : RTS := in_flow_status;
    FLOW_RTS_CTS    : RTS := Port[base + MCR] AND MCR_RTS > 0;
  END; (* CASE *)
END; (* SerialClass.RTS *)

PROCEDURE SerialClass.ReadLineStatus;
BEGIN
  line_status := Port[base + LSR];
END; (* SerialClass.ReadLineStatus *)

(*
  Data Ready (Receive Buffer Register contains data)
*)
FUNCTION SerialClass.DR : Boolean;
BEGIN
  DR := Port[base + LSR] AND LSR_DR > 0;
END; (* SerialClass.DR *)

(*
  Overrun Error
*)
FUNCTION SerialClass.OE : Boolean;
BEGIN
  OE := Port[base + LSR] AND LSR_OE > 0;
END; (* SerialClass.OE *)

(*
  Parity Error
*)
FUNCTION SerialClass.PE : Boolean;
BEGIN
  PE := Port[base + LSR] AND LSR_PE > 0;
END; (* SerialClass.PE *)

(*
  Framing Error
*)
FUNCTION SerialClass.FE : Boolean;
BEGIN
  FE := Port[base + LSR] AND LSR_FE > 0;
END; (* SerialClass.FE *)

(*
  Break Interrupt
*)
FUNCTION SerialClass.BI : Boolean;
BEGIN
  BI := Port[base + LSR] AND LSR_BI > 0;
END; (* SerialClass.BI *)

(*
  Transmitter Holding Register Empty
*)
FUNCTION SerialClass.THRE : Boolean;
BEGIN
  THRE := Port[base + LSR] AND LSR_THRE > 0;
END; (* SerialClass.THRE *)

(*
  Transmitter Empty
*)
FUNCTION SerialClass.TEMT : Boolean;
BEGIN
  TEMT := Port[base + LSR] AND LSR_TEMT > 0;
END; (* SerialClass.TEMT *)

(*
  Error in Receiver FIFO
*)
FUNCTION SerialClass.ErrRCVRFIFO : Boolean;
BEGIN
  ErrRCVRFIFO := Port[base + LSR] AND LSR_ERCVR_FIFO > 0;
END; (* SerialClass.ErrRCVRFIFO *)

(*
  Returns the data in Interrupt Identification Register
*)
FUNCTION SerialClass.InterruptIdentification : Byte;
BEGIN
  InterruptIdentification := Port[base + IIR] AND $F;
END; (* SerialClass.InterruptIdentification *)

(*
  Returns TRUE if an interrupt is pending
*)
FUNCTION SerialClass.InterruptPending : Boolean;
BEGIN
  InterruptPending := Port[base + IIR] AND IIR_IP = 0;
END; (* SerialClass.InterruptPending *)

(*
  Installs the interrupt handler, but does NOT enable interrupts.
  To enable interrupts, use method EnableIRQ.
*)
CONSTRUCTOR InterruptClass.InstallInterruptHandler(irq : Byte);
VAR
  int_no        : Byte;
  i             : Byte;

BEGIN
  IF NOT IRQAvailable(irq) THEN
    BEGIN
      PrintError('IRQ #' + Int2Str(irq) + ' is already in use', dead_end);
      Exit;
    END; (* IF *)
  IF error_alarm THEN
    Exit;
  irq_line := irq;
  vector := @SerialHandler;
  int_no := InterruptByIRQ(irq);
  GetIntVec(int_no, old_vector);
  SetIntVec(int_no, vector);
  ports_to_serve := 0;
END; (* InterruptClass.InstallInterruptHandler *)

(*
  Removes interrupt handler and automaticly disables interrupts
*)
DESTRUCTOR InterruptClass.RemoveInterruptHandler;
VAR
  i     : Byte;

BEGIN
  CLI;
  DisableIRQ;
  SetIntVec(InterruptByIRQ(irq_line), old_vector);
  STI;
END; (* InterruptClass.RemoveInterruptHandler *)

(*
  Adds port to be served
*)
PROCEDURE InterruptClass.AddService(port_number : Byte);
BEGIN
  IF port_number <= MAX_PORTS THEN
    ports_to_serve := ports_to_serve OR (1 SHL (port_number - 1));
END; (* InterruptClass.AddService *)

(*
  Removes port from service list
*)
PROCEDURE InterruptClass.RemoveService(port_number : Byte);
BEGIN
  IF port_number <= MAX_PORTS THEN
    ports_to_serve := ports_to_serve AND NOT (1 SHL (port_number - 1));
END; (* InterruptClass.RemoveService *)

(*
  Returns TRUE if the specified port is in service list
*)
FUNCTION InterruptClass.Service(port_number : Byte) : Boolean;
BEGIN
  Service := ports_to_serve AND (1 SHL (port_number - 1)) > 0;
END; (* InterruptClass.Service *)

(*
  Enables interrupt from PIC
*)
PROCEDURE InterruptClass.EnableIRQ;
BEGIN
  CLI;
  Port[IMR] := Port[IMR] AND NOT (1 SHL irq_line);

  (*
    Just to be sure
  *)
  EOI(irq_line);
  STI;
END; (* InterruptClass.EnableIRQ *)

(*
  Disables interrupts from PIC
*)
PROCEDURE InterruptClass.DisableIRQ;
BEGIN
  Port[IMR] := Port[IMR] OR (1 SHL irq_line);
END; (* InterruptClass.DisableIRQ *)


VAR

  i     : Byte;

(*
  This code will be executed automaticly when this unit is being used
*)
BEGIN
  FOR i := 1 TO MAX_PORTS DO
    sio[i] := NIL;
  FOR i := 0 TO MAX_IRQS DO
    irqs[i] := NIL;
  xon_request := FALSE;
  xoff_request := FALSE;
  exit_save := ExitProc;
  ExitProc := @ExitSerialIO;
END. (* SerialIO *)
