(* ---------------------------------- *)
(* Buffer                             *)
(* Copyright (c) 1994 by Dado Colussi *)
(* All rights reserved                *)
(* ---------------------------------- *)


UNIT Buffer;


INTERFACE


CONST

  IGNORE_LIST_SIZE      = 255;


TYPE

  BufferEntryPointer    = ^Char;

  BufferPointer         = ^BufferClass;
  BufferClass           =
    OBJECT              (* Public section *)

      CONSTRUCTOR       Open(buffer_size : Word);               (* Open buffer *)
      DESTRUCTOR        Close; VIRTUAL;                         (* Close buffer *)

      FUNCTION          Load : Word; VIRTUAL;                   (* Returns the number of active characters in the bufer *)
      FUNCTION          OverflowDanger : Boolean; VIRTUAL;      (* Returns TRUE if the buffer is getting full *)
      FUNCTION          CharAvailable : Boolean; VIRTUAL;       (* Returns TRUE if the buffer contains characters *)
      PROCEDURE         WriteChar(ch : Char); VIRTUAL;          (* Writes a character into buffer *)
      PROCEDURE         WriteStr(str : String); VIRTUAL;        (* Writes a string into buffer *)
      FUNCTION          ReadChar : Char; VIRTUAL;               (* Reads a character from buffer *)

      PROCEDURE         FlushBuffer; VIRTUAL;                   (* Waits until the buffer is empty *)
      PROCEDURE         PurgeBuffer; VIRTUAL;                   (* Clears the buffer immediatelly. The data will be LOST! *)

      PROCEDURE         Ignore(ch : Char); VIRTUAL;             (* Set character to be ignored *)
      PROCEDURE         Notice(ch : Char); VIRTUAL;             (* Set character to be noticed (not ignored) *)

      PRIVATE           (* Private section *)

      start             : BufferEntryPointer;   (* Starting pointer *)
      head              : BufferEntryPointer;   (* Head pointer *)
      tail              : BufferEntryPointer;   (* Tail pointer *)
      size              : Word;                 (* The buffer size. One char resrves one byte from the buffer *)
      ignore_list       : ARRAY [0..IGNORE_LIST_SIZE] OF Boolean;

    END; (* BufferClass *)


IMPLEMENTATION


USES

  Error;


VAR

  overflow_danger_status        : Boolean;


CONSTRUCTOR BufferClass.Open(buffer_size : Word);
VAR
  i     : Byte;

BEGIN
  IF MaxAvail < buffer_size THEN
    BEGIN
      PrintError('Not enough memory for buffering', dead_end);
      Exit;
    END; (* IF *)
  size := buffer_size;
  GetMem(start, size);
  head := start;
  tail := start;
  overflow_danger_status := FALSE;
  FOR i := 0 TO IGNORE_LIST_SIZE DO
    ignore_list[i] := FALSE;
END; (* BufferClass.Open *)

DESTRUCTOR BufferClass.Close;
BEGIN
  FreeMem(start, size);
  head := NIL;
  tail := NIL;
  size := 0;
END; (* BufferClass.Close *)

FUNCTION BufferClass.Load : Word;
BEGIN
  IF CharAvailable THEN
    IF Ofs(head^) > Ofs(tail^) THEN
      Load := Ofs(head^) - Ofs(tail^)
    ELSE
      Load := size - (Ofs(tail^) - Ofs(head^))
  ELSE
    Load := 0;
END; (* BufferClass.Load *)

(*
  Returns TRUE when the load of the buffer is higher than 3/4 of the
  buffer size. Returns FALSE when the load is less than 3/4 of the
  buffer size or has lowered to under 1/2 of the buffer size.
*)
FUNCTION BufferClass.OverflowDanger : Boolean;
BEGIN
  CASE overflow_danger_status OF
    TRUE  : overflow_danger_status := Load > size DIV 2;
    FALSE : overflow_danger_status := Load > (3 * size) DIV 4;
  END; (* CASE *)
  OverflowDanger := overflow_danger_status;
END; (* BufferClass.OverflowDanger *)

FUNCTION BufferClass.CharAvailable : Boolean;
BEGIN
  CharAvailable := head <> tail;
END; (* BufferClass.CharAvailable *)

PROCEDURE BufferClass.WriteChar(ch : Char);
BEGIN
  IF ignore_list[Ord(ch)] THEN
    Exit;
  head^ := ch;
  IF Ofs(head^) = Ofs(start^) + size THEN
    head := start
  ELSE
    head := Ptr(Seg(head^), Ofs(head^) + 1);
END; (* BufferClass.WriteChar *)

PROCEDURE BufferClass.WriteStr(str : String);
VAR
  i     : Byte;

BEGIN
  FOR i := 1 TO Byte(str[0]) DO
    WriteChar(str[i]);
END; (* BufferClass.WriteStr *)

FUNCTION BufferClass.ReadChar : Char;
BEGIN
  ReadChar := tail^;
  IF Ofs(tail^) = Ofs(start^) + size THEN
    tail := start
  ELSE
    tail := Ptr(Seg(tail^), Ofs(tail^) + 1);
END; (* BufferClass.ReadChar *)

PROCEDURE BufferClass.FlushBuffer;
BEGIN
  REPEAT
  UNTIL NOT CharAvailable;
END; (* BufferClass.FlushBuffer *)

PROCEDURE BufferClass.PurgeBuffer;
BEGIN
  head := tail;
END; (* BufferClass.PurgeBuffer *)

PROCEDURE BufferClass.Ignore(ch : Char);
BEGIN
  ignore_list[Ord(ch)] := TRUE;
END; (* BufferClass.Ignore *)

PROCEDURE BufferClass.Notice(ch : Char);
BEGIN
  ignore_list[Ord(ch)] := FALSE;
END; (* BufferClass.Notice *)


END. (* Buffer *)

