(*$U-,R-,V-,S-*)
PROGRAM DumbTrm;

(*--------------------------------------------------------------------------*)
(*                                                                          *)
(*   Program:   DumbTrm                                                     *)
(*                                                                          *)
(*   Author:    Philip R. Burns                                             *)
(*                                                                          *)
(*   Date:      November 11, 1987.                                          *)
(*                                                                          *)
(*   Purpose:   Dumb terminal emulator to demonstrate PibAsync routines.    *)
(*                                                                          *)
(*   Usage:     Compile to .EXE file using Turbo Pascal and execute the     *)
(*              .EXE file by typing  DUMBTRM  at the DOS prompt.            *)
(*                                                                          *)
(*              You will be prompted for the required communications        *)
(*              parameters.  If the serial port can be opened successfully, *)
(*              then a dumb terminal mode is entered.                       *)
(*                                                                          *)
(*              To exit dumb terminal mode, type ^C (ascii 03).  To send    *)
(*              a break, type ^B (ascii 02).                                *)
(*                                                                          *)
(*   Needs:     See includes below for files included from PibAsync.Arc.    *)
(*                                                                          *)
(*   Remarks:   This program doesn't check for input mistakes when the      *)
(*              communications parameters are being entered.                *)
(*                                                                          *)
(*--------------------------------------------------------------------------*)

                                   (* Get global types *)
                                   (* Get declarations for async i/o *)
USES
   Crt, Dos, GlobType, PibTimer, PibAsync;

VAR
   Baud_Rate   : WORD              (* Baud rate for connection, e.g., 1200  *);
   Com_Port    : INTEGER           (* Which port, e.g., 1 for COM1:         *);
   Parity      : CHAR              (* Parity, e.g., E for even parity       *);
   Data_Bits   : INTEGER           (* How many bits per character, e.g., 8  *);
   Stop_Bits   : INTEGER           (* How many stop bits -- nearly always 1 *);

VAR
   InBufSize   : INTEGER           (* Size of input buffer                  *);
   OutBufSize  : INTEGER           (* Size of output buffer                 *);
   Do_XonXoff  : CHAR              (* 'Y' to do XON/XOFF flow control       *);
   Do_HardWired: CHAR              (* 'Y' to do XON/XOFF flow control       *);
   Do_CTS      : CHAR              (* 'Y' to do CTS checking                *);
   Do_DSR      : CHAR              (* 'Y' to do DSR checking                *);


(*--------------------------------------------------------------------------*)

PROCEDURE Get_Comm_Params;

VAR
   YesNo : CHAR;

BEGIN (* Get_Comm_Params *)

   WRITELN;

   WRITE('Enter com port (1,2,3,4):                             ');
   READLN( Com_Port );

   WRITE('Enter baud rate (300,1200,2400,4800,9600,19200,38400):');
   READLN( Baud_Rate );

   WRITE('Enter parity (E=even, N=one, O=odd, M=mark, S=space): ');
   READLN( Parity );

   Parity := UpCase( Parity );

   WRITE('Enter bits per character (7 or 8):                    ');
   READLN( Data_Bits );

   WRITE('Enter stop bits (usually 1):                          ');
   READLN( Stop_Bits );

   WRITE('Enter size in bytes of async receive buffer:          ');
   READLN( InBufSize );

   WRITE('Enter size in bytes of async output buffer:           ');
   READLN( OutBufSize );

   WRITE('Use XON/XOFF flow control (Y/N)?                      ');
   READLN( Do_XonXoff );

   WRITE('Do CTS checking (Y/N)?                                ');
   READLN( Do_CTS );

   WRITE('Do DSR checking (Y/N)?                                ');
   READLN( Do_DSR );

   WRITE('Is connection hard-wired (Y/N)?                       ');
   READLN( Do_HardWired );

END   (* Get_Comm_Params *);

(*--------------------------------------------------------------------------*)

FUNCTION Initialize_Communications : BOOLEAN;

BEGIN (* Initialize_Communications *)

                                   (* Set CTS checking *)

   Async_Do_CTS         := ( UpCase( Do_CTS ) = 'Y' );

                                   (* Set DSR checking *)

   Async_Do_DSR         := ( UpCase( Do_DSR ) = 'Y' );

                                   (* Set XON/XOFF to user request *)

   Async_Do_XonXoff     := ( UpCase( Do_XonXoff ) = 'Y' );

                                   (* Set hard-wired as user requests *)

   Async_Hard_Wired_On  := ( UpCase( Do_HardWired ) = 'Y' );

                                   (* Set half-second break duration *)
   Async_Break_Length   := 500;

                                   (* Let XON/XOFF break points default. *)

   Async_Init( InBufSize, OutBufSize, 0, 0, 0);

                                   (* If com port 3 or 4, make sure     *)
                                   (* port address specified in memory. *)
   IF ( Com_Port > 2 ) THEN
      IF ( NOT Async_Port_Address_Given( Com_Port ) ) THEN
         Async_Setup_Port( Com_Port, -1, -1 );

                                   (* Try opening the serial port.   *)

   IF ( NOT Async_Open( Com_Port, Baud_Rate, Parity, Data_Bits, Stop_Bits ) ) THEN
      BEGIN
         WRITELN('Cannot open serial port.');
         Initialize_Communications := FALSE;
      END
   ELSE
      BEGIN
         WRITELN('Serial port opened, DumbTrm ready.');
         Initialize_Communications := TRUE;
      END;

END   (* Initialize_Communications *);

(*--------------------------------------------------------------------------*)

PROCEDURE Emulate_Dumb_Terminal;

VAR
   Kch: CHAR;
   Ch : CHAR;

BEGIN (* Emulate_Dumb_Terminal *)

                                   (* Begin loop over serial port   *)
                                   (* and keyboard.                 *)
   REPEAT
                                   (* Pick up and display character *)
                                   (* from port, if any.            *)

      IF Async_Receive( Ch ) THEN
         IF ( Ch <> CHR( 0 ) ) THEN
            WRITE( Ch );
                                   (* Read character from keyboard  *)
                                   (* and send out port, unless it  *)
                                   (* is ^B or ^C.                  *)
                                   (*                               *)
                                   (* ^B -- send a break            *)
                                   (* ^C -- quit                    *)
      IF KeyPressed THEN
         BEGIN
            KCh := ReadKey;
            CASE KCh OF
               ^B: Async_Send_Break;
               ^C: ;
               ELSE
                  Async_Send( Kch );
            END (* CASE *);
         END;

   UNTIL ( Kch = ^C );

END   (* Emulate_Dumb_Terminal *);

(*--------------------------------------------------------------------------*)

PROCEDURE Finish_Communications;

BEGIN (* Finish_Communications *)
                                   (* Close port and drop DTR *)
   Async_Close( FALSE );
                                   (* Release space allocated for buffers *)
   Async_Release_Buffers;

END   (* Finish_Communications *);

(*--------------------------------------------------------------------------*)

BEGIN (* DumbTrm *)

                                   (* Request serial port parameters     *)
   Get_Comm_Params;
                                   (* Initialize port                    *)

   IF Initialize_Communications THEN
      BEGIN
                                   (* Emulate dumb terminal until ^C hit *)
         Emulate_Dumb_Terminal;
                                   (* Close down port                    *)
         Finish_Communications;

      END;

END   (* DumbTrm *).