unit PCX256;

(* This is a unit to display 320 x 200 x 256-color .PCX files. The actual
   work of decoding the file and moving the data into memory is done in
   assembler and is quite fast. (You don't need TASM or any knowledge of
   assembly language. As long as PCX256.OBJ is present, it will be linked
   into the unit when you compile.)

   It's assumed that the image is the width of the screen and that you will
   set the correct display mode. The unit doesn't check to see what format
   the .PCX file is in, although it will flag an error if the file was
   produced by an early version of Paintbrush that doesn't have palette
   information.

   See SHOW256.PAS for a sample implementation.

   References:
   ~~~~~~~~~~
   Richard F. Ferraro, "Programmer's Guide to the EGA and VGA Cards"
   (Addison-Wesley, 1988).

   "Technical Reference Manual [for Paintbrush]" (Zsoft, 1988). The
   information in this slim booklet is also found in PCX.DOC, a file
   distributed with at least some versions of Microsoft/PC Paintbrush.

   Software:
   ~~~~~~~~
   Besides the various incarnations of Paintbrush (ZSoft and Microsoft),
   the excellent Deluxe Paint II Enhanced (Electronic Arts) can also create
   files in .PCX format. Other graphics programs have conversion utilities.
   *)

(* --------------------------------------------------------------------- *)

INTERFACE

uses DOS;

type    RGBrec = record
                   redval, greenval, blueval: byte;
                 end;

var     pcxfilename: pathstr;
        file_error: boolean;
        RGBpal: array[0..255] of RGBrec;
        page_addr: word;

const   page0 = $A000;

procedure READ_PCX256(pfilename: pathstr);

(* ------------------------------------------------------------------ *)

IMPLEMENTATION

var     datalength: word;
        scratch: pointer;
        repeatcount: byte;
        video_index: word;
        palette_start, total_read: longint;
        palette_flag: byte;
        version: word;

const   buffsize = 65521;              { Largest possible }

procedure DECODE_PCX256; external;
{$L pcx256}

procedure READ_PCX256(pfilename: pathstr);

var     x, gun, pcxcode: byte;
        pcxfile: file;

procedure CLEANUP;

begin
close(pcxfile);
freemem(scratch,buffsize);
end;

begin    { READ_PCX256 }

(* To minimize disk access and speed things up, we read the file into a
   scratchpad in dynamic memory. Large files have to be done in two or more
   chunks because of the 64K limit on dynamic memory variables. *)
getmem(scratch, buffsize);                 { Allocate scratchpad }

assign(pcxfile, pfilename);
{$I-} reset(pcxfile, 1);  {$I+}
file_error:= (IOresult <> 0);
if file_error then
begin
  cleanup; exit;
end;
blockread(pcxfile, version, 1);             { Read first two bytes }
file_error:= (hi(version) < 5);             { No palette info. }
if file_error then
begin
  cleanup; exit;
end;
palette_start:= filesize(pcxfile) - 769;

seek(pcxfile, 128);                        { Scrap file header }
total_read:= 128;

repeatcount:= 0;                           { Initialize assembler vars. }
video_index:= 0;

repeat
  blockread(pcxfile, scratch^, buffsize, datalength);
  inc(total_read, datalength);
  if (total_read > palette_start) then
      dec(datalength, total_read - palette_start);
  decode_pcx256;                                 { Decode and display }
until (eof(pcxfile)) or (total_read>= palette_start);

(* The last 769 btes of the file are palette information, starting with a
   one-byte flag. Each group of three bytes represents the RGB values of
   one of the color registers. The values have to be divided by 4 to be
   brought within the range 0-63 expected by the registers. *)

seek(pcxfile, palette_start);
blockread(pcxfile, palette_flag, 1);
file_error:= (palette_flag <> 12);
if file_error then
begin
  cleanup; exit;
end;
blockread(pcxfile, RGBpal, 768);         { Get palette info. }
for x:= 0 to 255 do
with RGBpal[x] do
begin
  redval:= redval div 4;
  greenval:= greenval div 4;
  blueval:= blueval div 4;
end;
cleanup;
end;  { READ_PCX_FILE }

(* -------------------------- Initialization ----------------------------- *)

BEGIN
page_addr:= page0;                          { Destination for data }
END.

