Unit Pages;
{$R-,S-,O+}

{Project Completion Date : April 7, 1990}

{No modifications to date}

{Virtual Page Object}

{Pages are essentially dynamically allocated arrays of bytes, which can}
{be associated with a particular "chunk" of bytes in a given (non-text)}
{File. When a page receives the Message LOAD(F,addr), it will load the }
{PAGESIZE(ElSize) or less if EOF, bytes starting from ADDR in File F.  }
{Pages store their current virtual address, so FLUSH needs only the    }
{File as input.  A Page can ACCEPT or RETRIEVE individual data elements}
{using the given element's true index. Function LOWESTEL and HIGHESTEL }
{allow the Page to report which "chunk" of elements it is currently    }
{holding, while CONTAINS reports whether or not the given Index is     }
{within the Page's current range (simple shorthand for a query to BOTH }
{HighestEl and LowestEl).                                              }

{Note that Pages can be tricky to work with, thus motivating the design}
{of the much less error-prone PageManager objects!                     }

{Possible error messages, and associated DosExitCodes:

            0 Insufficient Memory to allocate Page.
            1 File Not Open! Attempted LOAD or FLUSH.
            2 Attempted to FLUSH an unloaded Page.
            3 Disk Full Error.
            4 Attempted RETRIEVE with wrong Data Size.
            5 Attempted ACCEPT with wrong Data Size.
            6 Page Indexing Error - Use CONTAINS before ACCEPT or RETRIEVE.
            7 Attempted to allocate Page with DataSize > <MaxPageSize>.

 All these errors are easy to guard against, and if you are using either of}
{the PageManager objects, you should never see any of them, with the       }
{possible exception of 0 and 7.  Encountering error 0 (with the PageManager}
{object) can happen if the Heap is internally fragmented, but MemAvail     }
{indicates sufficient HeapSpace for the desired number of pages -- not a   }
{very likely eventuality, but certainly possible. Error 7 will only be     }
{encountered when the user erroneously attempts to use a data element that }
{is larger then MaxPageSize bytes, or somehow a negative (or zero) size is }
{reported to INIT.                                                         }
{Of course, numerous TP errors are possible if you attempt to use ANY of   }
{the Page methods (other then PageSize) prior to calling INIT, or after    }
{calling DESTROY!                                                          }


INTERFACE

Const
  MaxPageSize = 1536;  {512 = 1 DOS Sector}

Type
  Space   = Array[0..0] of Byte;
  Flex    = ^Space;

  Page  = Object
            {Page overhead = 15 bytes}

            VirtAddr : LongInt; {Zero-based}
            Data     : Flex;
            Changed  : Boolean; {Was Data Changed?}
            PSize    : Word;    {Actual Page Size}
            DSize    : Word;    {Actual Data Size}
            NumBytes : Word;    {Usually = PSize}

            Procedure Init (NumElems,ElementSize : Word);
              {NumElems should ALWAYS be MaxPageSize Div ElementSize}
              {with ONE exception: when the File will be SMALLER }
              {then MaxPageSize, which is admittedly silly, but  }
              {completeness demands the possibility.             }

            Procedure Destroy; {Return all memory to Heap}

            Procedure Load (Var F : File; Addr : LongInt);
              {Load PageSize bytes from F into Data, Starting at Addr}
              {Addr is the Element Index, NOT the Byte Address}

            Procedure Flush (Var F : File);
              {If Changed, Write PSize bytes from Data onto F}
              {otherwise do nothing}

            Function LowestEl  : LongInt; {Actual Index of Lowest El in Page}
            Function HighestEl : LongInt; {Actual Index of Highest "     "  }

            Function Contains(Addr : LongInt) : Boolean;
              {Is Addr Indexed El in this page?}

            Function PageSize (ElSize : Word) : Word;
              {Actual PageSize, assuming ElSize is the element size}
              {CAN be called without calling INIT.}

            Procedure Retrieve (Addr : LongInt; Size : Word; Var El);
              {Retrieve the Element indexed by ADDR. If Not present}
              {in this page, generate Error 6}

            Procedure Accept   (Addr : LongInt; Size : Word; Var El);
              {Update the Element indexed by ADDR. If Not present}
              {in this page, generate Error 6}

          End;

IMPLEMENTATION

Procedure Error (Num : Byte);
Begin
  WriteLn;
  Write ('Page Error [',Num,'] : ');
  Case Num of
           0 : WriteLn (' Insufficient Memory to allocate Page.');
           1 : WriteLn (' File Not Open! Attempted LOAD or FLUSH.');
           2 : WriteLn (' Attempted to FLUSH an unloaded Page.');
           3 : WriteLn (' Disk Full Error. ');
           4 : WriteLn (' Attempted RETRIEVE with wrong Data Size.');
           5 : WriteLn (' Attempted ACCEPT with wrong Data Size.');
           6 : WriteLn (' Page Indexing Error - Use CONTAINS before ACCEPT or RETRIEVE.');
           7 : WriteLn (' Attempted to allocate Page with DataSize > ',MaxPageSize,'.');
           8 : WriteLn (' Attempted ACCEPT or RETRIEVE on Unused Page.');
         End;
  WriteLn;
  WriteLn ('***** PROGRAM TERMINATED *****');
  HALT (Num)
End;

Procedure Page.Init;
Begin
  VirtAddr := -1;  {Flag unused}
  Changed  := False;
  DSize    := ElementSize;
  If DSize > MaxPageSize Then Error (7);
  If DSize <= 0 Then Error (7);
  PSize := MaxPageSize - (MaxPageSize Mod DSize); {round off pagesize to}
  If PSize > NumElems*ElementSize                    {nearest whole element}
    Then
      PSize := NumElems*ElementSize; {Special Case: 1-Page File}
  GetMem (Data,PSize);
  NumBytes := PSize;
  If Data = Nil Then Error (0) {Out of memory error}
End;

Procedure Page.Destroy;
Begin
  FreeMem (Data,PSize)
End;

Procedure Page.Load;
Var
  BytesRead : Word;
Begin
  {$I-}
  Seek (F,Addr*DSize); {ADDR is the Index of the base element in the}
  {$I+}                {requested page - relative to virtual Index}
  If IOResult <> 0 Then Error (1);

  BlockRead (F,Data^,PSize,BytesRead); {Specifying BytesRead allows for}
                                       {the last page to be only a partial}
  NumBytes := BytesRead;
  VirtAddr := Addr;
  Changed := False
End;

Procedure Page.Flush (Var F : File);
Var
  BytesWritten : Word;
Begin
  If Changed Then
    Begin
      If VirtAddr < 0 Then Error (2);
      {$I-}
      Seek (F,VirtAddr*DSize);
      {$I+}
      If IOResult <> 0 Then Error (1);

      BlockWrite (F,Data^,NumBytes,BytesWritten);
      If BytesWritten <> NumBytes Then Error (3);

      Changed := False
    End
End;

Function Page.LowestEl  : LongInt;
Begin
  LowestEl := VirtAddr
End;

Function Page.HighestEl : LongInt;
Begin
  HighestEl := VirtAddr + (NumBytes Div DSize) - 1
End;

Function Page.Contains(Addr : LongInt) : Boolean;
Begin
  Contains := ((Addr >= LowestEl) and (Addr <= HighestEl))
End;

Procedure Page.Retrieve;
Var
  ByteAddr : Word;
Begin
  If Size <> DSize Then Error (4);
  ByteAddr := (Addr - VirtAddr) * DSize;
  If ByteAddr > (NumBytes - DSize) Then Error (6);
  Move (Data^[ByteAddr],El,DSize)
End;

Procedure Page.Accept;
Var
  ByteAddr : Word;
Begin
  If Size <> DSize Then Error (5);
  ByteAddr := (Addr - VirtAddr) * DSize;
  If ByteAddr > (NumBytes - DSize) Then Error (6);
  Move (El,Data^[ByteAddr],DSize);
  Changed := True
End;

Function Page.PageSize (ElSize : Word) : Word;
Begin
  PageSize := MaxPageSize - (MaxPageSize Mod ElSize)
End;

{$F+}
Function HeapErrorTrap (Size : Word) : Integer;
Begin
  HeapErrorTrap := 1  { New and GetMem return Nil if out_of_memory }
End;
{$F-}

BEGIN
  HeapError := @HeapErrorTrap;
END.
