(****************************************************)
(*   Low Level Novell Netware Calls.                *)
(*   (C) CSM Limited. All Rights Reserved.          *)
(*   parts (c) Rex. K. Perkins.                     *)
(*                                                  *)
(*   Unit contains DOS level calls to the           *)
(*   Novell API.                                    *)
(*                                                  *)
(*   Written by                                     *)
(*          G.T. Swindell   January 1993            *)
(*                                                  *)
(*   DPMI Version 1.1                               *)
(****************************************************)

Unit NvllDPMI;
Interface
Uses
  DOS, WinAPI, Strings, Objects, SimRMI, NvllVars;

Type
  NameArray = Array[0..47] of Char;

Var
  ConnectionIDTable : pConnectionIDTable;
  FileServerNames : pFileServerNames;

Function IS_IPX_Loaded : Boolean;
Function Open_Semaphore(Semaphore_Name : PChar; Var SemaphoreHandle : LongInt) : Word;
Procedure Close_Semaphore(SemaphoreHandle : LongInt);
Procedure InitConnectionID;
Procedure SetPreferredConnectionID(IDNo : Byte);
Procedure GetPreferredConnectionID(Var IDNo : Byte);

Procedure GetConnectionID;
Procedure GetFileServerNames;

Procedure GetPrinterList;
Procedure GetUserLists;
Procedure GetObjectConnectionNumbers(var Rec; Name : NameArray; ObjectType : Word);

Procedure SetDefaultLocalPrinter(PrnNum : Byte);
Function CaptureStatus : Boolean;

Procedure CapturePrinter(S : PChar; BannerEnable : Word; BannerStr : String);
Procedure EndCapture;

Procedure Clear_Printer_Queue;
Procedure Clear_User_List;

{ Novell Broadcast Message Routines }
Type
  Str55 = String[55];

Var
  Entry_BroadCast_Mode : Byte;
  MessagePending : Boolean;

Procedure SetMessageMode(Mode : Byte);
Function GetMessageMode : Byte;

Function GetBroadCastMessage : String;
Procedure SendBroadCastMessage(Var msg : OpenString; Server : Byte; Var ConnectionList);

Implementation
var
  DPRegs : tRealModeRecord;
  DosAlloc : tDosAlloc;
  TempArray : Array[0..3] of Byte;

(*********************************************)
(* FUNCTION : Is_IPX_Loaded                  *)
(*                                           *)
(* PURPOSE  : Test wether the IPX shell is   *)
(*            loaded at the workstation      *)
(*                                           *)
(* INPUT    : -                              *)
(*                                           *)
(* OUTPUT   : Boolean                        *)
(*********************************************)

Function Is_IPX_Loaded : Boolean;
begin
  FillChar(DPRegs, SizeOf(tRealModeRecord), 0);
  DPRegs.AX := $7A00;
  SimRealModeInt($2F, DPRegs);
  NetWareDriversPresent := DPRegs.AL = $FF;
  Is_IPX_Loaded := NetWareDriversPresent;
end;

(*********************************************)
(* FUNCTION : Open_Semaphore                 *)
(*                                           *)
(* PURPOSE  : Opens a semaphore on the       *)
(*            network server currently       *)
(*            mapped.                        *)
(*                                           *)
(* INPUT    : Semaphore name                 *)
(*                                           *)
(* OUTPUT   : 0 - Call unsuccessfull.        *)
(*            1+ - Count for that Semaphore  *)
(*            string.                        *)
(*********************************************)

Function Open_Semaphore(Semaphore_Name : PChar; Var SemaphoreHandle : LongInt) : Word;
Var
  TempSemaphore : tSemaphore;
  SemaphorePtr : Pointer;

begin
  FillChar(DPRegs, SizeOf(tRealModeRecord), 0);
  SetPreferredConnectionID(1);

  DosAlloc.Addresses := GlobalDOSAlloc(SizeOf(tSemaphore));
  SemaphorePtr := PTR(DosAlloc.Selector, 0);
  StrCopy(SemaphorePtr, Semaphore_Name);

  With DPRegs Do
  begin
    AH := $C5;
    AL := 0;
    DS := DosAlloc.Segment;
    DX := 0;
    CL := 1;
  end;

  SimRealModeInt($21, DPRegs);

  If DPRegs.AL = 0 then
  begin
    Open_Semaphore := DPRegs.BL;
    LongRec(SemaphoreHandle).Hi := DPRegs.CX;
    LongRec(SemaphoreHandle).Lo := DPRegs.DX;
  end
  else
    Open_Semaphore := 0;
  SetPreferredConnectionID(0);
  GlobalDOSFree(DosAlloc.Selector);
end;

(*********************************************)
(* FUNCTION : Close_Semaphore                *)
(*                                           *)
(* PURPOSE  : Opens a semaphore on the       *)
(*            network server currently       *)
(*            mapped.                        *)
(*                                           *)
(* INPUT    : Semaphore name                 *)
(*                                           *)
(* OUTPUT   : -                              *)
(*********************************************)

Procedure Close_Semaphore(SemaphoreHandle : LongInt);
begin
  FillChar(DPRegs, SizeOf(tRealModeRecord), 0);
  SetPreferredConnectionID(1);
  With DPRegs Do
  begin
    AH := $C5;
    AL := 4;
    CX := LongRec(SemaphoreHandle).Hi;
    DX := LongRec(SemaphoreHandle).Lo;
  end;
  SimRealModeInt($21, DPRegs);
  SetPreferredConnectionID(0);
end;

(*********************************************)
(* FUNCTION : InitConnectionID               *)
(*                                           *)
(* PURPOSE  : Initialises the connection ID  *)
(*            to it's base value.            *)
(*                                           *)
(* INPUT    : -                              *)
(*                                           *)
(* OUTPUT   : -                              *)
(*********************************************)

Procedure InitConnectionID;
begin
  ConnectionID := $FFFFFFFF;
end;

(*********************************************)
(* FUNCTION : GetConnectionID                *)
(*                                           *)
(* PURPOSE  : Sets up the connection ID      *)
(*            table for the current file-    *)
(*            server.                        *)
(*                                           *)
(* INPUT    : -                              *)
(*                                           *)
(* OUTPUT   : Filled ConnectionIDTable       *)
(*            record.                        *)
(*********************************************)

Procedure GetConnectionID;
Var
  RetPtr : pConnectionIDTable;
  RetSel : Word;

begin
  FillChar(DPRegs, SizeOf(tRealModeRecord), 0);
  GetMem(ConnectionIDTable, Sizeof(tConnectionIDTable));

  With DPRegs do
  begin
    AX := $EF03; { Novell Call }

    SimRealModeInt($21, DPRegs); { Call the ISR }

    RetSel := AllocSelector(0);
    SetSelectorBase(RetSel, ES shl 4);
    SetSelectorLimit(RetSel, SI + Sizeof(tConnectionIDTable));
    RetPtr := Ptr(RetSel, SI);

    Move(RetPtr^, ConnectionIDTable^, Sizeof(tConnectionIDTable));

    FreeSelector(RetSel);
  end;
end;

(*********************************************)
(* FUNCTION : GetFileServerNames             *)
(*                                           *)
(* PURPOSE  : Gets the file server names     *)
(*            of those that are attached.    *)
(*                                           *)
(* INPUT    : -                              *)
(*                                           *)
(* OUTPUT   : Filled FileServerNames         *)
(*            record.                        *)
(*********************************************)

Procedure GetFileServerNames;
Var
  RetPtr : pFileServerNames;
  RetSel : Word;

begin
  GetMem(FileServerNames, SizeOf(tFileServerNames));
  FillChar(DPRegs, SizeOf(tRealModeRecord), 0);

  With DPRegs do
  begin
    AL := $04;
    AH := $EF; { Novell Call }

    SimRealModeInt($21, DPRegs); { Call the ISR }

    RetSel := AllocSelector(0);
    SetSelectorBase(RetSel, ES shl 4);
    SetSelectorLimit(RetSel, SI + Sizeof(tFileServerNames));
    RetPtr := Ptr(RetSel, SI);

    Move(RetPtr^, FileServerNames^, Sizeof(tFileServerNames));
    FreeSelector(RetSel);
  end;
end;

(*********************************************)
(* FUNCTION : SetPreferredConnectionID       *)
(*                                           *)
(* PURPOSE  : Sets the system to look at     *)
(*            a particular server.           *)
(*                                           *)
(* INPUT    : File Server number (1..8)      *)
(*                                           *)
(* OUTPUT   : -                              *)
(*********************************************)

Procedure SetPreferredConnectionID(IDNo : Byte);
begin
  FillChar(DPRegs, SizeOf(tRealModeRecord), 0);
  With DPRegs do
  begin
    AH := $F0;
    AL := $00;
    DL := IDNo;
  end;
  SimRealModeInt($21, DPRegs);
end;

(*********************************************)
(* FUNCTION : GetPreferredConnectionID       *)
(*                                           *)
(* PURPOSE  : Gets the current preferred     *)
(*            server.                        *)
(*                                           *)
(* INPUT    : -                              *)
(*                                           *)
(* OUTPUT   : File Server number (1..8)      *)
(*********************************************)

Procedure GetPreferredConnectionID(Var IDNo : Byte);
begin
  FillChar(DPRegs, SizeOf(tRealModeRecord), 0);
  With DPRegs do
  begin
    AH := $F0;
    AL := $01;
  end;
  SimRealModeInt($21, DPRegs);
  IDNo := DPRegs.AL;
end;

Procedure SetBindaryBuffers;
begin
  DOSAlloc.Addresses := GlobalDOSAlloc(SizeOf(tBufferArray));
  RequestPtr := PTR(DosAlloc.Selector, 0);
  RequestSelector := DosAlloc.Selector;
  RequestSeg := DosAlloc.Segment;

  DOSAlloc.Addresses := GlobalDOSAlloc(SizeOf(tBufferArray));
  ReplyPtr := PTR(DosAlloc.Selector, 0);
  ReplySelector := DosAlloc.Selector;
  ReplySeg := DosAlloc.Segment;

  FillChar(RequestPtr^, Sizeof(tBufferArray), 0);
  FillChar(ReplyPtr^, Sizeof(tBufferArray), 0);
end;

Procedure FreeBindaryBuffers;
begin
  GlobalDOSFree(ReplySelector);
  GlobalDOSFree(RequestSelector);
end;

(*********************************************)
(* FUNCTION : ScanBindaryObject              *)
(*                                           *)
(* PURPOSE  : Scans for a particular         *)
(*            bindary object within the      *)
(*            current file server.           *)
(*                                           *)
(* INPUT    : Bindary object required.       *)
(*            One of the byXXXX variables    *)
(*            above.                         *)
(*                                           *)
(* OUTPUT   : Name of object found.          *)
(*            Nul string if none             *)
(*********************************************)

Procedure ScanBindaryObject(var Name : NameArray; BindaryOb : Word);
Var
  i : Byte;
  ObjectName : Array[0..99] of Char;
begin
  FillChar(DPRegs, SizeOf(tRealModeRecord), 0);
  SetBindaryBuffers;
  StrCopy(ObjectName, '*');
  RequestPtr^[2] := $37;
  Move(ConnectionID, TempArray, SizeOf(LongInt));

  RequestPtr^[3] := TempArray[3];
  RequestPtr^[4] := TempArray[2];
  RequestPtr^[5] := TempArray[1];
  RequestPtr^[6] := TempArray[0];

  RequestPtr^[7] := Hi(BindaryOb);
  RequestPtr^[8] := Lo(BindaryOb);

  RequestPtr^[9] := StrLen(ObjectName); { Length Object Name }
  For i := 0 to StrLen(Objectname) do
    RequestPtr^[10+i] := Byte(ObjectName[i]);

  RequestPtr^[0] := 10 + StrLen(objectName) - 2;
  RequestPtr^[1] := 0;

  ReplyPtr^[0] := 200;
  ReplyPtr^[1] := 0;

  With DPRegs Do
  begin
    AH := $E3;
    DS := RequestSeg;
    SI := 0;
    ES := ReplySeg;
    SI := 0;
  end;
  SimRealModeInt($21, DPRegs);
  If DPRegs.AL = 0 then
  begin
    TempArray[3] := Byte(ReplyPtr^[2]);
    TempArray[2] := Byte(ReplyPtr^[3]);
    TempArray[1] := Byte(ReplyPtr^[4]);
    TempArray[0] := Byte(ReplyPtr^[5]);
    Move(TempArray, ConnectionID, SizeOf(LongInt));
    Move(ReplyPtr^[8], Name, Sizeof(NameArray));
  end
  else
    StrCopy(Name, '');
  FreeBindaryBuffers;
end;

(*********************************************)
(* FUNCTION : GetPrinterList                 *)
(*                                           *)
(* PURPOSE  : Sets up a chain of attached    *)
(*            printers. Adds those from all  *)
(*            servers in one chain           *)
(*                                           *)
(* INPUT    : -                              *)
(*                                           *)
(* OUTPUT   : TopPrinter is set to the first *)
(*            printer in the chain.          *)
(*********************************************)

Procedure GetPrinterList;
var
  ServerConnection : Byte;
  Name : NameArray;
  PastPrinter,
  NewPrinter : PPrinterQueue;

begin
  PastPrinter := Nil;
  NewPrinter := Nil;
  FirstPrinter := Nil;
  For ServerConnection := 1 to 8 do
  begin
    InitConnectionID;
    If (ConnectionIDTable^[ServerConnection].SlotInUse<>0) then
    begin
      SetPreferredConnectionID(ServerConnection);
      StrCopy(Name, '*');
      While Name[0] <> #0 do
      begin
        ScanBindaryObject(Name, byPrinter);
        If Name[0]<>#0 then
        begin
          PastPrinter := NewPrinter;
          GetMem(NewPrinter, Sizeof(TPrinterQueue));
          StrCopy(NewPrinter^.Name, FileServerNames^[ServerConnection]);
          StrCat(NewPrinter^.Name, '/');
          StrCat(NewPrinter^.Name, Name);
          NewPrinter^.ServerConnection := Serverconnection;
          NewPrinter^.ConnectionID := ConnectionID;
          NewPrinter^.Prev := PastPrinter;
          NewPrinter^.Next := Nil;
          If FirstPrinter=Nil then
            FirstPrinter := NewPrinter;
          If PastPrinter<>Nil then PastPrinter^.Next := NewPrinter;
        end;
      end;
    end;
  end;
end;

(*********************************************)
(* FUNCTION : GetUserList                    *)
(*                                           *)
(* PURPOSE  : Sets up a chain of attached    *)
(*            users.                         *)
(*                                           *)
(* INPUT    : -                              *)
(*                                           *)
(* OUTPUT   : FirstUser is set for each      *)
(*            fileserver attached.           *)
(*********************************************)

Procedure GetUserLists;
var
  ServerConnection : Byte;
  Name : NameArray;
  PastUser,
  NewUser : PUserList;

begin
  For ServerConnection := 1 to 8 do
  begin
    PastUser := Nil;
    NewUser := Nil;
    FirstUser[ServerConnection] := Nil;
    InitConnectionID;
    If (ConnectionIDTable^[ServerConnection].SlotInUse<>0) then
    begin
      SetPreferredConnectionID(ServerConnection);
      StrCopy(Name, '*');
      While name[0]<>#0 do
      begin
        ScanBindaryObject(Name, byUser);
        If name[0]<>#0 then
        begin
          PastUser := NewUser;
          GetMem(NewUser, Sizeof(TUserList));
          StrCopy(NewUser^.Server, FileServerNames^[ServerConnection]);
          StrCopy(NewUser^.Name, Name);
          NewUser^.ServerConnection := Serverconnection;
          NewUser^.ConnectionID := ConnectionID;
          NewUser^.Prev := PastUser;
          NewUser^.Next := Nil;
          If FirstUser[ServerConnection]=Nil then
            FirstUser[ServerConnection] := NewUser;
          If PastUser<>Nil then PastUser^.Next := NewUser;
        end;
      end;
    end;
  end;
end;

(*********************************************)
(* FUNCTION : GetObjectConnectionNumbers     *)
(*                                           *)
(* PURPOSE  : Gets the connection numbers    *)
(*            for the given object.          *)
(*                                           *)
(* INPUT    : name - Object name             *)
(*            tpe - type of the object       *)
(*            One of the byXXXX above.       *)
(*                                           *)
(* OUTPUT   : Record contaion the list of    *)
(*            connection numbers.            *)
(*********************************************)

Procedure GetObjectConnectionNumbers(var Rec; Name : NameArray; ObjectType : Word);
begin
  FillChar(DPRegs, SizeOf(tRealModeRecord), 0);
  SetBindaryBuffers;
  ReplyPtr^[0] := 102;

  RequestPtr^[2] := $15;
  RequestPtr^[3] := Hi(ObjectType);
  RequestPtr^[4] := Lo(ObjectType);
  RequestPtr^[5] := StrLen(Name);
  Move(Name[1], RequestPtr^[6], SizeOf(NameArray));
  RequestPtr^[0] := 6 + StrLen(Name) - 2;

  With DPRegs Do
  begin
    AH := $E3;
    DS := RequestSeg;
    SI := 0;
    ES := ReplySeg;
    DI := 0;
  end;
  SimRealModeInt($21, DPRegs);

  Move(ReplyPtr^[2], Rec, ReplyPtr^[2]+1);

  FreeBindaryBuffers;
end;

(*********************************************)
(* FUNCTION : CapturePrinter                 *)
(*                                           *)
(* PURPOSE  : Captures a given printer.      *)
(*                                           *)
(* INPUT    : S - Printer name.              *)
(*            BannerEnable - enables the     *)
(*                           banner          *)
(*            BannerStr - the banner string  *)
(*                                           *)
(* OUTPUT   : -                              *)
(*********************************************)

Procedure CapturePrinter(S : PChar; BannerEnable : Word; BannerStr : String);
Type
  pCapture_Request = ^tCapture_Request;
  tCapture_Request = Record
    Status : Byte;
    Flags : Byte;
    TabSize : Byte;
    ServerPrinter : Byte;
    NumCopies : Byte;
    FormType : Byte;
    Reserved : Byte;
    BannerText : Array[0..13] of Char;
  end;

Var
  Printer : PPrinterQueue;
  CapturePtr : PCapture_Request;
  CaptureSel,
  CaptureSeg : Word;
  Banner : Array[0..13] of Char;

begin
  StrPCopy(Banner, BannerStr);
  Printer := FirstPrinter;
  EndCapture;
  While Printer<>Nil do
  begin
    If StrIComp(Printer^.Name, S)=0 then
    begin
      SetPreferredConnectionID(Printer^.ServerConnection);
      FillChar(DPRegs, SizeOf(tRealModeRecord), 0);
      With DPRegs do
      begin
        AH := $B8;
        AL := $06;
        DH := $00;
        Move(Printer^.ConnectionID, TempArray, SizeOf(LongInt));
        BL := TempArray[3];
        BH := TempArray[2];
        CL := TempArray[1];
        CH := TempArray[0];
      end;
      SimRealModeInt($21, DPRegs); { Set Capture Queue }
      FillChar(DPRegs, SizeOf(tRealModeRecord), 0);
      With DPRegs DO
      begin
        AH := $DF;
        AL := $04;
        DL := $00;
      end;
      SimRealModeInt($21, DPRegs); { Start Capture }
      DosAlloc.Addresses := GlobalDOSAlloc(sizeof(tCapture_Request));
      CapturePtr := PTR(DosAlloc.Selector, 0);
      CaptureSel := DosAlloc.Selector;
      CaptureSeg := DosAlloc.Segment;
      FillChar(DPRegs, SizeOf(tRealModeRecord), 0);
      With DPRegs DO
      begin
        AH := $B8;
        AL := $00;
        CX := SizeOf(tCapture_Request);
        ES := CaptureSeg;
        BX := 0;
      end;
      SimRealModeInt($21, DPRegs); { Get Default Flags }
      With CapturePtr^ do
      begin
        StrCopy(BannerText, Banner);
        If (BannerEnable AND prBanner)=prBanner then
          Flags := Flags OR $80 { Set Banner flag }
        else
          Flags := Flags AND $7F; { Clear Banner Flag }
      end;
      FillChar(DPRegs, SizeOf(tRealModeRecord), 0);
      With DPRegs DO
      begin
        AH := $B8;
        AL := $01;
        CX := SizeOf(tCapture_Request);
        ES := CaptureSeg;
        BX := 0;
      end;
      SimRealModeInt($21, DPRegs); { Get Default Flags }
      Printer := Nil;
    end
    else
      Printer := Printer^.Next;
  end;
end;

(*********************************************)
(* FUNCTION : EndCapture                     *)
(*                                           *)
(* PURPOSE  : Sets the printer back to local *)
(*            mode.                          *)
(*                                           *)
(* INPUT    : -                              *)
(*                                           *)
(* OUTPUT   : -                              *)
(*********************************************)

Procedure EndCapture;
begin
  FillChar(DPRegs, SizeOf(tRealModeRecord), 0);
  With DPRegs do
  begin
    AH := $DF;
    DL := $05;
    DH := $00;
  end;
  SimRealModeInt($21, DPRegs);
end;

(*********************************************)
(* FUNCTION : Clear_Printer_Queue            *)
(*                                           *)
(* PURPOSE  : Sets the printer back to local *)
(*            mode and frees the printer     *)
(*            queue memory.                  *)
(*                                           *)
(* INPUT    : -                              *)
(*                                           *)
(* OUTPUT   : -                              *)
(*********************************************)

Procedure Clear_Printer_Queue;
var
  NextQueue,
  Queue : PPrinterQueue;
begin
  EndCapture;
  Queue := FirstPrinter;
  While Queue<>Nil do
  begin
    NextQueue := Queue;
    FreeMem(Queue, Sizeof(TPrinterQueue));
    Queue := NextQueue^.Next;
  end;
end;

(*********************************************)
(* FUNCTION : Clear_User_List                *)
(*                                           *)
(* PURPOSE  : Frees the user list chain      *)
(*            memory.                        *)
(*                                           *)
(* INPUT    : -                              *)
(*                                           *)
(* OUTPUT   : -                              *)
(*********************************************)

Procedure Clear_User_List;
var
  i : Byte;
  NextUser,
  User : PUserList;
begin
  For i := 1 to 8 do
  begin
    User := FirstUser[i];
    While User<>Nil do
    begin
      NextUser := User;
      FreeMem(User, Sizeof(TUserList));
      User := NextUser^.Next;
    end;
  end;
end;

(*********************************************)
(* FUNCTION : GetMessageMode                 *)
(*                                           *)
(* PURPOSE  : Gets the current mode for      *)
(*            receiving messages.            *)
(*                                           *)
(* INPUT    : -                              *)
(*                                           *)
(* OUTPUT   : The mode value.                *)
(*********************************************)

Function GetMessageMode : Byte;
begin
  FillChar(DPRegs, SizeOf(tRealModeRecord), 0);
  With DPRegs do
  begin
    AH := $DE;
    DL := $04;
  end;
  SimRealModeInt($21, DPRegs);
  GetMessageMode := DPRegs.AL;
end;

(*********************************************)
(* FUNCTION : SetMessageMode                 *)
(*                                           *)
(* PURPOSE  : Sets the mode for              *)
(*            receiving messages.            *)
(*                                           *)
(* INPUT    : The mode value (0 - 3)         *)
(*                                           *)
(* OUTPUT   : -                              *)
(*********************************************)

Procedure SetMessageMode(Mode : Byte);
begin
  FillChar(DPRegs, SizeOf(tRealModeRecord), 0);
  DPRegs.AH := $DE;
  DPRegs.DL := Mode;
  SimRealModeInt($21, DPRegs);
end;

(*********************************************)
(* FUNCTION : GetBroadcastMessage            *)
(*                                           *)
(* PURPOSE  : Gets a message from the        *)
(*            message queue.                 *)
(*                                           *)
(* INPUT    :                                *)
(*                                           *)
(* OUTPUT   : A message                      *)
(*********************************************)

Function GetBroadCastMessage : String;
Var
  Msg : String;
  ServerConnection : Byte;

begin
  Msg:='';
  If Is_IPX_Loaded then
  begin
    For ServerConnection := 1 to 8 do
    begin
      InitConnectionID;
      FillChar(DPRegs, SizeOf(tRealModeRecord), 0);
      SetBindaryBuffers;
      If (ConnectionIDTable^[ServerConnection].SlotInUse<>0) AND (Msg='') then
      begin
        SetPreferredConnectionID(ServerConnection);
        RequestPtr^[0] := 1;
        RequestPtr^[2] := $01;
        ReplyPtr^[0] := 120;
        With DPRegs do
        begin
          AH := $E1;
          DS := RequestSeg;
          SI := 0;
          ES := ReplySeg;
          DI := 0;
        end;
        SimRealModeInt($21, DPRegs);
        Move(ReplyPtr^[3], Msg[1], ReplyPtr^[2]);
        Msg[0] := CHAR(ReplyPtr^[2]);
      end;
      FreeBindaryBuffers;
    end;
  end;
  GetBroadCastMessage := Msg;
end;

(*********************************************)
(* FUNCTION : SendBroadcastMessage           *)
(*                                           *)
(* PURPOSE  : Sends a message to the         *)
(*            message queue.                 *)
(*            FUNCTION NOT YET IMPLEMENTED   *)
(*            PROPERLY. (No return values)   *)
(*                                           *)
(* INPUT    :                                *)
(*                                           *)
(* OUTPUT   :                                *)
(*********************************************)
Procedure SendBroadCastMessage(var Msg : OpenString; Server : Byte; var ConnectionList);
var
  ConnectionListLength : Byte absolute ConnectionList;
  i : Byte;

begin
  FillChar(DPRegs, SizeOf(tRealModeRecord), 0);
  SetBindaryBuffers;
  SetPreferredConnectionID(Server);
  ReplyPtr^[0] := 101;
  Move(ConnectionList, RequestPtr^[3], ConnectionListLength+1);
  RequestPtr^[2] := $00;
  FillChar(RequestPtr^[5 + ConnectionListLength], 55, 32);
  RequestPtr^[4 + ConnectionListLength] := 55;
  For i := 1 to Length(Msg) do
    RequestPtr^[4 + i + ConnectionListLength] := Byte(Msg[i]);
  RequestPtr^[0] := 58 + ConnectionListLength;
  With DPRegs do
  begin
    AH := $E1;
    DS := RequestSeg;
    SI := 0;
    ES := ReplySeg;
    DI := 0;
  end;
  SimRealModeInt($21, DPRegs);
  FreeBindaryBuffers;
end;


Procedure SetDefaultLocalPrinter(PrnNum : Byte);
begin
  FillChar(DPRegs, SizeOf(tRealModeRecord), 0);
  With DPRegs do
  begin
    AH := $B8;
    AL := $05;
    DH := PrnNum;
  end;
  SimRealModeInt($21, DPRegs);
end;


Function CaptureStatus : Boolean;
begin
  FillChar(DPRegs, SizeOf(tRealModeRecord), 0);
  With DPRegs do
  begin
    AH := $F0;
    AL := $03;
  end;
  SimRealModeInt($21, DPRegs);
  CaptureStatus := DPRegs.AH = $FF;
end;

begin
  { Initialisation Code }

  If Is_IPX_Loaded then { Check whether Netware is there }
  begin
    Writeln('Initialising Netware Parameters. Please wait');
    InitConnectionID;
    SetDefaultLocalPrinter(0); { Set to LPT1 }
    GetConnectionID;
    GetFileServerNames;
    GetUserLists;
    GetPrinterList;
    Entry_BroadCast_Mode := GetMessageMode;
    SetMessageMode(3); { Complete Machine Control }
    MessagePending := FALSE;
  end;
end.