%title "Mouse routines"

        IDEAL

CrtWidth        equ     (Byte Ptr 4AH)

segment         data    word    public
        extrn  MousePresent:word
        extrn  EventFlag:word
        extrn  Xcoord:byte
        extrn  Ycoord:byte
        extrn  Buttons:word
        extrn  MCursorAddr:word
        extrn  MouseOn:word
ends


segment         code    word    public
        assume  cs:code,ds:data

PUBLIC  MouseRegister
;====================================================================
; The procedure 'MouseRegister' is called to test for the presense
; of a mouse as well as register the mouse handler with MOUSE.SYS
;
; On entry:
;       BP + 6 = Event mask to use in registeration.
;       DS = Turbo Pascal data segment
;
; On exit:
;       AX, BX, CX, DX affected
;       Mouse status variables set accordingly.
;         If mouse driver not present:
;               MousePresent = 0
;               EventFlag    = 0
;               Xcoord       = -1
;               Ycoord       = -1
;               Buttons      = 0
;               MCursorAddr  = -1
;               MouseOn      = 0
;         If mouse driver is present:
;               MousePresent = 1
;               EventFlag    = 0   Until 1st mouse event occurs
;               Xcoord       = -1  Until 1st mouse event occurs
;               Ycoord       = -1  Until 1st mouse event occurs
;               Buttons      = 0   Until 1st mouse event occurs
;               MCursorAddr  = -1  Until 1st mouse event occurs
;               MouseOn      = 0
;
;
proc    MouseRegister   far
        push    bp
        mov     bp,sp
        push    es                      ;Save ES register

;Initialize external mouse status variables

        mov     [MousePresent],0        ;Start with mouse not present
        mov     [EventFlag],0           ;Initialize event flag
        mov     [Xcoord],-1             ;Initial X off screen
        mov     [Ycoord],-1             ;Initial Y off screen
        mov     [MouseOn],0             ;Show mouse is off
        mov     [Buttons],0             ;Initial button status idle
        mov     [MCursorAddr],-1        ;Initial Addr off screen

;Check to see if mouse driver present

        mov     ah,35h                  ;Get Int vector
        mov     al,33h                  ;For mouse
        int     21h                     ;Call DOS
        mov     ax,es                   ;Move segment to AX
        or      ax,bx                   ;Is vector 0
        jz      @@Exit                  ;Yes, so exit
        mov     al,[es:bx]              ;Get first byte pointed to
        cmp     al,0CFh                 ;Does it point to an IRET
        je      @@Exit                  ;Yes, so no mouse

;Driver is present

        mov     ax,0                    ;Reset mouse
        int     33h                     ;Call mouse driver
        mov     [MousePresent],1        ;Show mouse status
        push    cs
        pop     es                      ;ES:DX = handler address
        mov     dx,offset CS:MouseHandler
        mov     cx,[BP+6]               ;CX = event mask
        mov     ax,0Ch                  ;Register handler
        int     33h                     ;Call mouse driver

;Set mouse cursor in software

        mov     ax,10                   ;Set text cursor
        mov     bx,0                    ;Cursor in software
        mov     cx,009FFh               ;Makes text cursor red background
        mov     dx,4000h
        int     33h
@@Exit:
        pop     es                      ;Restore ES register
        pop     bp
        ret     2
endp    MouseRegister



PUBLIC  MouseUnregister
;======================================================================
; Unregister mouse if present
;
; On entry:
;       DS = Turbo Pascal data segment
;
; On exit:
;       AX affected
;
proc    MouseUnregister far
        test    [MousePresent],1        ;Get mouse status
        jz      @@Exit                  ;Mouse not present
        mov     ax,0                    ;Function 0 reset mouse driver
        int     33h                     ;Call mouse driver
        mov     [MousePresent],0
@@Exit:
        ret
endp    MouseUnregister




;=======================================================================
; This procedure is called only by the mouse driver software and so is
; not decalared public. The procedure 'MouseRegister' should be called
; to register this routine with the mouse driver software. When a mouse
; event occurs corresponding to the event mask passed in the call to the
; 'MouseRegister' procedure, this routine will be entered with a far call
; from the mouse driver.  At entry to the handler, the registers are set
; up as follows:
;
;       AX = mouse event flags (bits set signal a corresponding event):
;              BIT      SIGNIFICANCE
;               0       mouse movement
;               1       left button pressed
;               2       left button released
;               3       right button pressed
;               4       right button released
;               5-15    reserved
;       BX = Button state
;               bit     significance
;               0       left button is down
;               1       right button is down
;               2-15    reserved
;       CX = X coordinate (0 to 632)
;       DX = Y coordinate (0 to 192)
;       SI = raw vertical mickey count
;       DI = raw horizontal mickey count
;       DS = mouse driver data segment
;
; On exit:
;       Mouse status variables are updated to reflect the event.
;
proc    MouseHandler    far
        push    ds                      ;Save mouse data segment

        mov     di,data                 ;Make our data segment addressable
        mov     ds,di
        assume  ds:data

        shr     cx,1                    ;Put X coord in the range (0-79)
        shr     cx,1                    ; by dividing by 8
        shr     cx,1
        mov     [Xcoord],cl             ;Store X coord
        shr     dx,1                    ;Put Y coord in the range (0-24)
        shr     dx,1                    ; by dividing by 8
        shr     dx,1
        mov     [Ycoord],dl             ;Store Y coord
        mov     [Buttons],bx            ;Store button data
        mov     [EventFlag],ax          ;Save type of event

;Compute screen offset address of mouse position

        mov     ax,40h
        mov     es,ax                   ;ES = Segment of BIOD data area
        mov     al,dl                   ;AX = Y coordinate
        mul     [es:CrtWidth]           ; times the screen width
        add     ax,cx                   ; add in X position offset
        shl     ax,1                    ; adjust for character attribute
        mov     [MCursorAddr],ax        ;Save it

        pop     ds                      ;Restore mouse data segment
        ret
endp    MouseHandler




PUBLIC  HideMouse
;==================================================================
; Hide the mouse cursor
;
; On entry:
;       Nothing.
;
; On exit:
;       All registers preserved.
;
proc    HideMouse       far
        push    ax                      ;Save registers
        push    ds

        mov     ax,data
        mov     ds,ax                   ;DS = data segment
        test    [MousePresent],1        ;Mouse present?
        jz      @@Exit                  ;No
        test    [MouseOn],1             ;Is mouse cursor displayed?
        jz      @@Exit                  ;No
        mov     ax,2                    ;Hide mouse function
        int     33h
        mov     [MouseOn],0             ;Set mouse on status OFF
@@Exit:
        pop     ds                      ;Restore registers
        pop     ax
        ret
endp    HideMouse




PUBLIC  ShowMouse
;==================================================================
; Show the mouse cursor
;
; On entry:
;       Nothing.
;
; On exit:
;       All registers preserved.
;
proc    ShowMouse       far
        push    ax                      ;Save registers
        push    ds

        mov     ax,data
        mov     ds,ax                   ;DS = data segment
        test    [MousePresent],1        ;Mouse present?
        jz      @@Exit                  ;No
        test    [MouseOn],1             ;Is mouse cursor displayed
        jnz     @@Exit                  ;Yes
        mov     ax,1                    ;Show mouse function
        int     33h
        mov     [MouseOn],1             ;Set mouse on status ON
@@Exit:
        pop     ds                      ;Restore registers
        pop     ax
        ret
endp    ShowMouse
ends
end

