(*--------------------------------------------------------------------------*)
(*                 Finv -- Inverse Central F Distribution                   *)
(*--------------------------------------------------------------------------*)

FUNCTION Finv( Alpha, Dfn, Dfe: REAL ) : REAL;

(*--------------------------------------------------------------------------*)
(*                                                                          *)
(*       Function:  Finv                                                    *)
(*                                                                          *)
(*       Purpose:   Calculates central inverse F                            *)
(*                                                                          *)
(*       Calling Sequence:                                                  *)
(*                                                                          *)
(*            Fval   := Finv( Alpha, Dfn, Dfe : REAL ) : REAL;              *)
(*                                                                          *)
(*                 Alpha  --- Probability value                             *)
(*                 Dfn    --- Numerator Df                                  *)
(*                 Dfe    --- Denominator Df                                *)
(*                                                                          *)
(*                 Fval   --- returned F-value corresponding to 'Alpha'.    *)
(*                                                                          *)
(*       Calls:                                                             *)
(*                                                                          *)
(*            BetaInv (Inverse Beta Distribution)                           *)
(*                                                                          *)
(*       Remarks:                                                           *)
(*                                                                          *)
(*            Any error results in '-1.0' as returned F value.              *)
(*                                                                          *)
(*--------------------------------------------------------------------------*)

CONST
   MaxIter  = 100;
   Dprec    = 10;

VAR
   Fin:   REAL;
   Iter:  INTEGER;
   Cprec: REAL;
   Ierr:  INTEGER;

BEGIN (* Finv *)

   Fin   := -1.0;

   IF( ( Dfn > 0.0 ) AND ( Dfe > 0.0 ) ) THEN

      IF( ( Alpha >= 0.0 ) AND ( Alpha <= 1.0 ) ) THEN

         BEGIN

            Fin := BetaInv( 1.0 - Alpha, Dfn/2.0, Dfe/2.0, MaxIter, Dprec,
                            Iter,  Cprec, Ierr  );

            IF( ( Fin >= 0.0 ) AND ( Fin < 1.0 ) AND ( Ierr = 0 ) ) THEN
                  Fin  := Fin * Dfe / ( Dfn * ( 1.0 - Fin ) );

         END;

   Finv := Fin;

END   (* Finv *);