(*****************************************************)
(* UNIT NAME :  InfoBox.PAS                          *)
(*        BY :  KARL GUDMUNDS                        *)
(*                                                   *)
(*   PURPOSE :  Display a box of text on a graphics  *)
(*              screen at xlo, ylo, returning the    *)
(*              character pressed.                   *)
(*****************************************************)

{$R-,I-,V-,S-,B-,N-,E-,F+,O+,D-}

Unit InfoBox;
interface
Uses Graph;

type
  TextList = array[1..12] of string[40];

Function ShowInfoBox(xlo, ylo:word; InfoText : TextList; NumListed:word) : char;
{show a box of text on a graphics screen, return the character pressed}

function SaveGraphWindow(ThisViewPort:ViewPortType) : pointer;

procedure RestoreGraphWindow(ThisViewPort : ViewPortType;
                             WindowPointer : pointer);

{===========================================================================}
implementation
Uses TPCrt;

  {make GetMem return NIL instead of causing a runtime heap error}
  {if there is not enough heap memory to save the underlying window}
{$F+}
function HeapFunc(Size:word):integer;
begin
  HeapFunc:=1;
end;


function WaitForAKey : char;
{this clears any waiting keypresses, then waits for a key and
 clears any extended scan codes}
var ExtraKeys:char;
begin
  While keypressed do ExtraKeys:=Readkey; {clear keyboard buffer}
  ExtraKeys:=Readkey;
  WaitForAKey:=ExtraKeys;
end;

function SaveGraphWindow(ThisViewPort:ViewPortType) : pointer;
var
  WindowPointer : pointer;
  SaveHeapError : Pointer;
  OrigVP : ViewPortType;
begin
  SaveHeapError := HeapError;
  HeapError := @HeapFunc; {so if getmem fails, returns nil}

  GetViewSettings(OrigVP);
  SetViewPort(0, 0, GetMaxX, GetMaxY, ClipOn); {so GetImage get correct part}

  With ThisViewPort do
    GetMem(WindowPointer, ImageSize(x1, y1, x2, y2));

  If WindowPointer = NIL then
  begin
    SaveGraphWindow:=NIL;
    With OrigVP do SetViewPort(x1,y1,x2,y2,Clip);
    exit;
  end; {not enough memory}

  With ThisViewPort do
    GetImage(x1, y1, x2, y2, WindowPointer^);  {save window area underneath}
  SaveGraphWindow:=WindowPointer; {should point to area of memory holding the image}

  With OrigVP do SetViewPort(x1,y1,x2,y2,Clip);
end;


procedure RestoreGraphWindow(ThisViewPort : ViewPortType; WindowPointer : pointer);
var OrigVP : ViewPortType;
begin
  GetViewSettings(OrigVP);
  With ThisViewPort do  SetViewPort(x1, y1, x2, y2, Clip);

  If WindowPointer <> NIL then
  begin
    With ThisViewPort do
      PutImage(0,0,WindowPointer^,NormalPut); {upper left of the viewport}
    FreeMem(WindowPointer, SizeOf(WindowPointer^));
  end;

  With OrigVP do SetViewPort(x1, y1, x2, y2, Clip);
end;


(********************************)
Function ShowInfoBox(xlo, ylo:word; InfoText : TextList; NumListed:word) : char;
var
  WindowPointer:pointer;
  LineSpace, BoxHeight, BoxWidth : Word;
  OriginalVP, ThisVP:ViewPortType;
  ReturnCharacter : char;

  procedure SetUpBoxSize;
  var Count:byte;
  begin
    SetTextStyle(DefaultFont,HorizDir,1);
    SetTextJustify(LeftText,CenterText);

    BoxWidth:=0;
    For Count := 1 to NumListed do
      If Length(InfoText[Count]) > BoxWidth then
        BoxWidth := Length(InfoText[Count]);

    LineSpace:= TextHeight('M') + 2;

    BoxWidth:=(BoxWidth+2) * TextWidth('M');
    BoxHeight:=(NumListed * LineSpace) + TextHeight('M');
    SetColor(White);
  end; {box size and characteristics}

  procedure ListBoxText(NumToList, Spacing:word);
  var LineCount:byte;
  begin
    MoveTo(0, 0);
    If NumToList > 0 then
    For LineCount:=1 to NumToList do
    begin
      MoveTo(0, GetY + Spacing);
      OutText(' '+ InfoText[LineCount]);
    end;
  end; {list box text}

begin {show info box}
  GetViewSettings(OriginalVP); {so can go back to a previous setting}
  SetViewPort(0,0,GetMaxX,GetMaxY,ClipOn); {so coordinates are absolute}
  SetUpBoxSize;

  With ThisVP do
  begin
    x1:=xlo; y1:=ylo;
    x2:=x1+BoxWidth;
    y2:=y1+BoxHeight;
    Clip:=ClipOff;

    WindowPointer:=SaveGraphWindow(ThisVP);
    SetViewPort(x1,y1,x2,y2,Clip);
    ClearViewPort;
    Rectangle(0,0,x2-x1,y2-y1);
  end;

  ListBoxText(NumListed, LineSpace);

  ReturnCharacter:=WaitForAKey;

  RestoreGraphWindow(ThisVP, WindowPointer);
  With OriginalVP do SetViewPort(x1,x2,y1,y2,Clip);

  ShowInfoBox:=ReturnCharacter;
end; {ShowInfoBox}


end. {unit}
